﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows.Forms;

namespace LfsPoints {
    public class ListViewBase : ListView {
        private IColumnSorter _columnSorter;

        public event EventHandler ItemSelected;
        public event EventHandler ItemDeselected;

        public ListViewItem SelectedItem { get; private set; }
        public new ContextMenuStrip ContextMenuStrip { get; set; }

        public IColumnSorter ColumnSorter {
            get { return _columnSorter; }
            set {
                _columnSorter = value;
                ListViewItemSorter = _columnSorter;
            }
        }

        public ListViewBase()
            : base() {
            InitializeComponent();

            // Set to double-buffered.
            SetStyle(ControlStyles.OptimizedDoubleBuffer | ControlStyles.AllPaintingInWmPaint, true);
            SetStyle(ControlStyles.EnableNotifyMessage, true);
        }

        [DebuggerStepThrough]
        protected override void OnNotifyMessage(Message m) {
            const int WM_ERASEBKGND = 0x14;

            // Eat message to reduce flicker.
            if (m.Msg != WM_ERASEBKGND) {
                base.OnNotifyMessage(m);
            }
        }

        protected override void OnSelectedIndexChanged(EventArgs e) {
            if (SelectedItems.Count == 1) {
                if (SelectedItem != null) {
                    SelectedItem = null;
                    OnItemDeselected(e);
                }

                SelectedItem = SelectedItems[0];
                OnItemSelected(e);
            }
            else if (SelectedItems.Count == 0 && SelectedItem != null) {
                SelectedItem = null;
                OnItemDeselected(e);
            }

            base.OnSelectedIndexChanged(e);
        }

        protected override void OnMouseDown(MouseEventArgs e) {
            if (ContextMenuStrip != null && e.Button == MouseButtons.Right) {
                ListViewItem item = GetItemAt(e.X, e.Y);
                if (item != null) {

                    // MouseDown occurs before IndexChanged, so make sure correct item is selected.
                    // This is because some ContextMenus need to access the currently selected item.
                    if (SelectedItem != null) {
                        SelectedItem = null;
                        OnItemDeselected(e);
                    }
                    SelectedItem = item;
                    OnItemSelected(e);

                    ContextMenuStrip.Show(Cursor.Position);
                }
            }

            base.OnMouseDown(e);
        }

        protected override void OnColumnClick(ColumnClickEventArgs e) {
            if (e.Column == ColumnSorter.Column) {
                if (ColumnSorter.Order == SortOrder.Ascending) {
                    ColumnSorter.Order = SortOrder.Descending;
                }
                else {
                    ColumnSorter.Order = SortOrder.Ascending;
                }
            }
            else {
                ColumnSorter.Column = e.Column;
                ColumnSorter.Order = SortOrder.Ascending;
            }

            Sort();

            base.OnColumnClick(e);
        }

        public void ClearItems() {
            Items.Clear();

            //if (ColumnSorter != null) {
            //    ColumnSorter.Reset();
            //}

            //if (SelectedItem != null) {
            //    SelectedItem = null;
            //    OnItemDeselected(EventArgs.Empty);
            //}
        }

        protected override void OnKeyDown(KeyEventArgs e) {
            if (SelectedItem != null) {
                if (e.Control && e.KeyCode == Keys.A) {
                    SelectAll();
                }
                else if (e.Control && e.KeyCode == Keys.C) {
                    CopyToClipboard();
                }
            }

            base.OnKeyDown(e);
        }

        private void SelectAll() {
            BeginUpdate();
            foreach (ListViewItem item in Items) {
                item.Selected = true;
            }
            EndUpdate();
        }

        private void CopyToClipboard() {
            // Get column widths for padding.
            int[] columnWidths = new int[Columns.Count];
            foreach (ListViewItem item in SelectedItems) {
                for (int i = 0; i < columnWidths.Length; i++) {
                    if (item.SubItems[i].Text.Length > columnWidths[i]) {
                        columnWidths[i] = item.SubItems[i].Text.Length;
                    }
                }
            }

            // Get lines.
            List<string> lines = new List<string>();
            for (int i = 0; i < SelectedItems.Count; i++) {
                string[] items = new string[SelectedItems[i].SubItems.Count];
                for (int j = 0; j < SelectedItems[i].SubItems.Count; j++) {
                    items[j] = SelectedItems[i].SubItems[j].Text.PadRight(columnWidths[j]);
                }
                lines.Add(String.Join(" ", items));
            }

            string text = String.Join((string)Environment.NewLine, lines);
            Clipboard.SetText(text, TextDataFormat.UnicodeText);
        }

        protected virtual void OnItemSelected(EventArgs e) {
            EventHandler temp = ItemSelected;
            if (temp != null) {
                temp(this, e);
            }
        }

        protected virtual void OnItemDeselected(EventArgs e) {
            EventHandler temp = ItemDeselected;
            if (temp != null) {
                temp(this, e);
            }
        }

        private void InitializeComponent() {
            this.SuspendLayout();
            // 
            // ListViewBase
            // 
            this.Dock = System.Windows.Forms.DockStyle.Fill;
            this.FullRowSelect = true;
            this.GridLines = true;
            this.HideSelection = false;
            this.Margin = new System.Windows.Forms.Padding(0);
            this.View = System.Windows.Forms.View.Details;
            this.ResumeLayout(false);

        }
    }
}
