#include "WheelInterface.h"
#include "globals.h"
#include <dxerr.h>

#define G27ID 0xC29B046D
#define DFPID 0xC298046D

#define ESCAPE_COMMAND_LEDS 0
#define LEDS_VERSION_NUMBER 0x00000001;

const std::string diInitErr = "DirectInput error";

WheelInterface::WheelInterface()
{
	deviceFound = false;
}

WheelInterface::~WheelInterface(void)
{
	if(deviceFound)
	{
		SendLeds(0, 1, 2, FALSE);
		diDevice->Unacquire();
	}
}

/** Initializes DirectInput and
	tried to find a G27 wheel
	*/
HRESULT WheelInterface::InitDirectInput(void)
{
	HRESULT hr;
	//Initialize DirectInput
	if(FAILED(hr = DirectInput8Create(GetModuleHandle(NULL), DIRECTINPUT_VERSION, IID_IDirectInput8, (VOID**)&di, NULL)))
	{
		//Error initializing DirectInput, tell the user and report back
		MessageBox(hMWin, "Cannot initialize DirectInput.", diInitErr.c_str(), MB_ICONERROR | MB_OK);
		return E_FAIL;
	}

	//Enumerate all connected gaming devices and find a G27 wheel.
	if(FAILED(hr = di->EnumDevices(DI8DEVCLASS_GAMECTRL, DevEnumCallBack, this, DIEDFL_ATTACHEDONLY)))
	{
		//Error enumerating devices
		return E_FAIL;
	}

	//Check if a G27 was found
	if(!deviceFound)
	{
		//It wasn't, tell the user and report back
		MessageBox(hMWin, "G27 not found, is the wheel connected and switched to the native mode?", diInitErr.c_str(), MB_ICONERROR | MB_OK);
		return E_FAIL;
	}

	//Set expected data format
	if (FAILED(hr = diDevice->SetDataFormat(&c_dfDIJoystick2)))
	{
		MessageBox(hMWin, "Error setting data format", diInitErr.c_str(), MB_ICONERROR | MB_OK);
		return E_FAIL;
	}

	/*Set the cooperative level to let DirectInput know how this device should
	interact with the system and with other DirectInput applications.*/
	if (FAILED(hr = diDevice->SetCooperativeLevel(hMWin, DISCL_EXCLUSIVE | DISCL_BACKGROUND)))
	{
		MessageBox(hMWin, "Error setting cooperative level, application might not work correctly.", diInitErr.c_str(), MB_ICONWARNING | MB_OK);
	}

	return S_OK;
}

/** Sends a LED command to the wheel*/
void WheelInterface::SendLeds(int RPM, int firstLeds, int redline, bool dbg)
{
	//Prepare LEDs command
	WheelData wheelData;
	ZeroMemory(&wheelData, sizeof(wheelData));

	wheelData.size = sizeof(WheelData);
	wheelData.versionNbr = LEDS_VERSION_NUMBER;
	wheelData.rpmData.currentRPM = (float)RPM;
	wheelData.rpmData.rpmFirstLedTurnsOn = (float)firstLeds;
	wheelData.rpmData.rpmRedLine = (float)redline;

	//Prepare effect command
	DIEFFESCAPE data;
	ZeroMemory(&data, sizeof(data));

	data.dwSize = sizeof(DIEFFESCAPE);
	data.dwCommand = ESCAPE_COMMAND_LEDS;
	data.lpvInBuffer = &wheelData;
	data.cbInBuffer = sizeof(wheelData);

	HRESULT hrA = diDevice->Acquire();
	HRESULT hrE = diDevice->Escape(&data);

	//DEBUG
	if(dbg) {
		std::ostringstream oss;
		oss << "First LEDs: " << wheelData.rpmData.rpmFirstLedTurnsOn << " Redline: " << wheelData.rpmData.rpmRedLine << " Current: " << RPM;
		std::string temp(oss.str());

		std::string debugInfo;
		debugInfo += TranslateError(hrA, std::string("Acquire: ")) + " ";
		debugInfo += TranslateError(hrE, std::string("Send cmd: ")) + " ";
		debugInfo += temp;

		SendMessage(hEditBox, WM_SETTEXT, 0, (LPARAM)debugInfo.c_str());
	}
}

/** Translates HRESULT error code
	to a human-readable text output.*/
std::string WheelInterface::TranslateError(HRESULT hr, std::string& errOp)
{
	std::string errType;		//A type of known error the application has encountered

	std::ostringstream oss;
	oss << (int*)hr;
	std::string errCode(oss.str());
	
	//Known errors
	switch(hr)
	{
		case S_OK:
			errType = "OK";
			break;
		case DI_NOEFFECT:
			errType = "No effect";
			break;
		case DIERR_INPUTLOST:
			errType = "Input lost";
			break;
		case DIERR_NOTACQUIRED:
			errType = "Not acquired";
			break;
		case DIERR_ACQUIRED:
			errType = "Aquired";
			break;
		case E_ACCESSDENIED:
			errType = "Access denied";
			break;
		case E_FAIL:
			errType = "Failed";
			break;
		default:
			errType = DXGetErrorString(hr);
			break;
	}

	return errOp + " " + errType + " (" + errCode + ")";
}

/** Callback for enumarating devices,
	checks device's VID/PID and if a
	match is found, initializes diDevice with
	the found device and ends.*/
BOOL CALLBACK DevEnumCallBack(const DIDEVICEINSTANCE* instance, VOID* context)
{
	WheelInterface* parent = (WheelInterface*)context;
	HRESULT hr;
	if(FAILED(hr = parent->di->CreateDevice(instance->guidInstance, &parent->diDevice, NULL)))
	{
		//Error when enumerating device
		return DIENUM_CONTINUE;
	}

	//Check the device's PID and VID
	if((instance->guidProduct.Data1 == G27ID) || (instance->guidProduct.Data1 == DFPID))
	{
		//G27 found, stop enumerating
		parent->deviceFound = true;
		return DIENUM_STOP;
	}
	
	//Invalid PID/VID, do not store the device and continue enumarating
	return DIENUM_CONTINUE;
}