#include "confighandler.h"

ConfigHandler::ConfigHandler(void)
{
	ipAddress = L"127.0.0.1";
	inSimPort = 29999;
	outGaugePort = 30000;
	adminPass = L"";
	updateSpeed = 200;
	mode = 0;
	units = 0;
	xOffset = 1;
	yOffset = 1;
	dispInLFS = true;
	dispInExt = true;

	lineNum = 0;
}

/** Opens and reads configuration file.*/
void ConfigHandler::loadConfigFile(void)
{
	//Open ecmeter.cfg for reading
	std::wifstream fRdr("./ecmeter.cfg", std::wifstream::in);
	
	//Were we able to open the file?
	if(!fRdr)
	{
		std::wcerr << L"(ConfigHandler)Error: Unable to open config file, using defaults." << std::endl;
		return;
	}

	//Read the file line by line
	std::wstring tempStr;

	while(std::getline(fRdr, tempStr))
	{
		parseLine(&tempStr);
		lineNum++;
	}
	
	fRdr.close();
}

/** Separates given line to three parts,
    keyword, value and extra. Keyword is everything from the
    beginning to the first occurence of "=". Value is everything between
    "=" and first occurence of "#" or the end of the line.
*/
void ConfigHandler::parseLine(std::wstring* inStr)
{
	size_t commentCharPos;
	size_t valueCharPos;
	std::wstring* data = new std::wstring[3];
	
	//Init data
	for(char i = 0; i < 3; i++)
	{
		data[i].assign(L"");
	}
	
	//Is there a commentary in the line
	commentCharPos = inStr->find(L"#");

	//It there a keyword/value separator in the line?
	valueCharPos = inStr->find(L"=");
	
	//std::wcout << valueCharPos << " " << commentCharPos << std::endl; //DEBUG
	
	if(valueCharPos == std::wstring::npos || valueCharPos > commentCharPos)	//No keyword/value separator found, line is invalid
	{
		data[2].assign(*inStr);
		storeParsedLine(data);
	}
	else
	{
	  //Separate keyword, value and a possible trailing commentary
	  data[0].assign(inStr->substr(0, valueCharPos+1));
	  data[1].assign(inStr->substr(valueCharPos+1, commentCharPos - valueCharPos - 1));
	  if(commentCharPos != std::wstring::npos)
	  {
		  data[2].assign(inStr->substr(commentCharPos, std::wstring::npos));
	  }
	  
	  storeParsedLine(data);

	  //Convert keyword to lower case
	  transform(data[0].begin(), data[0].end(), data[0].begin(), std::ptr_fun(::tolower));

	  //Remove white spaces from keyword
	  removeWhiteSpaces(&data[0]);

	  processReadData(data);
	}

	delete[] data;
}

/** Set application according
*   contents of the configuration file.
*/
void ConfigHandler::processReadData(std::wstring* inData)
{
	std::wstringstream ws(inData[1]);
	inData[0].assign(inData[0].substr(0, inData[0].length()-1));    //Remove trailing "=";
	
	//Character values
	if(inData[0].compare(L"ip") == 0)
	{
		removeWhiteSpaces(&inData[1]);
		ipAddress.assign(inData[1]);
		//std::wcout << "IP address is " << ipAddress << std::endl; //DEBUG
	}
	else if(inData[0].compare(L"adminpass") == 0)
	{
		//removeWhiteSpaces(&inData[1]);
		adminPass.assign(inData[1]);
		//std::wcout << inData[1].length() << std::endl; //DEBUG
		//std::wcout << "Admin pass is " << adminPass << std::endl; //DEBUG
	}

	//Numeric values
	else if(inData[0].compare(L"insimport") == 0)
	{
		ws >> inSimPort;
		//std::wcout << "InSim port is " << inSimPort << std::endl; //DEBUG
	}
	else if(inData[0].compare(L"outgaugeport") == 0)
	{
		ws >> outGaugePort;
		//std::wcout << "OutGauge port is " << outGaugePort << std::endl; //DEBUG
	}
	else if(inData[0].compare(L"updatespeed") == 0)
	{
		ws >> updateSpeed;
		if (updateSpeed < 1)
			updateSpeed = 1;
		//std::wcout << "Update speed is: " << updateSpeed << std::endl; //DEBUG
	}
	else if(inData[0].compare(L"outputtolfs") == 0)
	{
		short temp;
		ws >> temp;
		if(temp > 0)
		{
			dispInLFS = true;
		}
		else
		{
		  dispInLFS = false;
		}
	}
	else if(inData[0].compare(L"outputtowin") == 0)
	{
		short temp;
		ws >> temp;
		if(temp > 0)
		{
			dispInExt = true;
		}
		else
		{
		  dispInExt = false;
		}
	}

	//Settings user might change in runtime
	else if(inData[0].compare(L"mode") == 0)
	{
		short temp;
		ws >> temp;
		modeAt = lineNum;
		if(temp >= 0 && temp<= 3)
		{
			mode = temp;
		}
		//std::wcout << "Mode is " << mode << std::endl; //DEBUG
	}
	else if(inData[0].compare(L"units") == 0)
	{
		short temp;
		ws >> temp;
		unitsAt = lineNum;
		if(temp >= 0 && temp <= 2)
		{
			units = temp;
		}
		//std::wcout << "Units are set to: " << units << std::endl; //DEBUG
	}
	else if(inData[0].compare(L"xoffset") == 0)
	{
		short temp;
		ws >> temp;
		xOffsetAt = lineNum;
		if(temp >= 0 && temp <= 255)
		{
			xOffset = temp;
		}
		//std::wcout << "X offset is: " << xOffset << std::endl; //DEBUG
	}
	else if(inData[0].compare(L"yoffset") == 0)
	{
		short temp;
		ws >> temp;
		yOffsetAt = lineNum;
		if(temp >= 0 && temp <= 255)
		{
			yOffset = temp;
		}
		//std::wcout << "Y offset is: " << yOffset << std::endl; //DEBUG
	}
}

void ConfigHandler::removeWhiteSpaces(std::wstring* inStr)
{
  for(unsigned int idx = 0; idx < inStr->length(); idx++)
	{
		if(inStr->compare(idx, 1, L" ") == 0)
		{
			inStr->replace(idx, 1, L"");
		}
	}
}

bool ConfigHandler::saveConfigFile(void)
{ 
  //std::wcout << modeAt+1 << " " << unitsAt+1 << " " << xOffsetAt+1 << " " << yOffsetAt+1 << std::endl; //DEBUG
  
  //Update config table
  parsedConfig.at(modeAt).at(1).assign(L" " + to_string<short>(mode, std::dec) + L"   ");
  parsedConfig.at(unitsAt).at(1).assign(L" " + to_string<short>(units, std::dec) + L"   ");
  parsedConfig.at(xOffsetAt).at(1).assign(L" " + to_string<short>(xOffset, std::dec) + L"   ");
  parsedConfig.at(yOffsetAt).at(1).assign(L" " + to_string<short>(yOffset, std::dec) + L"   ");
  
  //Delete original file
  if(remove("./ecmeter.cfg") != 0)
  {
    std::wcout << "(ConfigHandler)Error: Cannot remove old config file!" << std::endl;
    return false;
  }
  
  //Create stream for writing
  std::wofstream fWrtr;
  fWrtr.open("./ecmeter.cfg", std::wofstream::out);
  
  //Can we write to the file?
  if(!fWrtr.is_open())
  {
    std::wcout << "(ConfigHandler)Error: Cannot open config file for writing" << std::endl;
    return false;
  }
  
  //Write data
  for(size_t idx = 0; idx < parsedConfig.size(); idx++)
  {
    fWrtr << parsedConfig.at(idx).at(0) << parsedConfig.at(idx).at(1) << parsedConfig.at(idx).at(2) << std::endl;
  }
  
  fWrtr.flush();
  fWrtr.close();
  
  return true;  
}
  
void ConfigHandler::storeParsedLine(std::wstring* inData)
{
  //std::wcout << inData[0] << "|" << inData[1] << "|" << inData[2] << std::endl; //DEBUG
  
  std::vector<std::wstring> tempVector;
  tempVector.push_back(inData[0]);
  tempVector.push_back(inData[1]);
  tempVector.push_back(inData[2]);
  parsedConfig.push_back(tempVector);
}

ConfigHandler::~ConfigHandler(void)
{
}

template <class T>
std::wstring to_string(T t, std::ios_base & (*f)(std::ios_base&))
{
  std::wostringstream oss;
  oss << f << t;
  return oss.str();
}

