<?php
require_once('enc.php');

class CP932 implements Encoding
{
	public static function encode($utf8string)
	{
		$cp932encoded = '';
		$end = strlen($utf8string);

		for($i = 0; $i < $end;)
		{
			$ordv = ord($utf8string[$i]);

			// ASCII range, single byte
			if($ordv < 0x80)
			{
				$cp932encoded .= self::$encode_map[$utf8string[$i]];
				++$i;
			}

			// Multibyte
			else
			{
				$bytes = 2; // UTF-8 multibyte sequences are at least 2 byte wide
				for($j = 32; $ordv & $j; $j = $j >> 1)
				{
					++$bytes;
				}
				$cp932encoded .= self::$encode_map[substr($utf8string, $i, $bytes)];
				$i += $bytes;
			}
		}

		return $cp932encoded;
	}
	public static function decode($cp932string)
	{
		$utf8encoded = '';
		$end = strlen($cp932string);

		for($i = 0; $i < $end;)
		{
			$ordv = ord($cp932string[$i]);

			// ASCII range + -symbol, single byte
			if($ordv <= 0x80)
			{
				$utf8encoded .= self::$decode_map[$cp932string[$i]];
				++$i;
			}
			else
			{
				$utf8encoded .= self::$decode_map[substr($cp932string, $i, 2)];
				$i += 2;
			}
		}

		return $utf8encoded;
	}

	public static function contains($chr, $utf8=false)
	{
		return ($utf8 ? isset(self::$encode_map[$chr]) : isset(self::$decode_map[$chr]));
	}

	private static $encode_map = array(
		"\x00"		=> "\x00",	//NULL
		"\x01"		=> "\x01",	//START OF HEADING
		"\x02"		=> "\x02",	//START OF TEXT
		"\x03"		=> "\x03",	//END OF TEXT
		"\x04"		=> "\x04",	//END OF TRANSMISSION
		"\x05"		=> "\x05",	//ENQUIRY
		"\x06"		=> "\x06",	//ACKNOWLEDGE
		"\x07"		=> "\x07",	//BELL
		"\x08"		=> "\x08",	//BACKSPACE
		"\x09"		=> "\x09",	//HORIZONTAL TABULATION
		"\x0a"		=> "\x0a",	//LINE FEED
		"\x0b"		=> "\x0b",	//VERTICAL TABULATION
		"\x0c"		=> "\x0c",	//FORM FEED
		"\x0d"		=> "\x0d",	//CARRIAGE RETURN
		"\x0e"		=> "\x0e",	//SHIFT OUT
		"\x0f"		=> "\x0f",	//SHIFT IN
		"\x10"		=> "\x10",	//DATA LINK ESCAPE
		"\x11"		=> "\x11",	//DEVICE CONTROL ONE
		"\x12"		=> "\x12",	//DEVICE CONTROL TWO
		"\x13"		=> "\x13",	//DEVICE CONTROL THREE
		"\x14"		=> "\x14",	//DEVICE CONTROL FOUR
		"\x15"		=> "\x15",	//NEGATIVE ACKNOWLEDGE
		"\x16"		=> "\x16",	//SYNCHRONOUS IDLE
		"\x17"		=> "\x17",	//END OF TRANSMISSION BLOCK
		"\x18"		=> "\x18",	//CANCEL
		"\x19"		=> "\x19",	//END OF MEDIUM
		"\x1a"		=> "\x1a",	//SUBSTITUTE
		"\x1b"		=> "\x1b",	//ESCAPE
		"\x1c"		=> "\x1c",	//FILE SEPARATOR
		"\x1d"		=> "\x1d",	//GROUP SEPARATOR
		"\x1e"		=> "\x1e",	//RECORD SEPARATOR
		"\x1f"		=> "\x1f",	//UNIT SEPARATOR
		"\x20"		=> "\x20",	//SPACE
		"\x21"		=> "\x21",	//EXCLAMATION MARK
		"\x22"		=> "\x22",	//QUOTATION MARK
		"\x23"		=> "\x23",	//NUMBER SIGN
		"\x24"		=> "\x24",	//DOLLAR SIGN
		"\x25"		=> "\x25",	//PERCENT SIGN
		"\x26"		=> "\x26",	//AMPERSAND
		"\x27"		=> "\x27",	//APOSTROPHE
		"\x28"		=> "\x28",	//LEFT PARENTHESIS
		"\x29"		=> "\x29",	//RIGHT PARENTHESIS
		"\x2a"		=> "\x2a",	//ASTERISK
		"\x2b"		=> "\x2b",	//PLUS SIGN
		"\x2c"		=> "\x2c",	//COMMA
		"\x2d"		=> "\x2d",	//HYPHEN-MINUS
		"\x2e"		=> "\x2e",	//FULL STOP
		"\x2f"		=> "\x2f",	//SOLIDUS
		"\x30"		=> "\x30",	//DIGIT ZERO
		"\x31"		=> "\x31",	//DIGIT ONE
		"\x32"		=> "\x32",	//DIGIT TWO
		"\x33"		=> "\x33",	//DIGIT THREE
		"\x34"		=> "\x34",	//DIGIT FOUR
		"\x35"		=> "\x35",	//DIGIT FIVE
		"\x36"		=> "\x36",	//DIGIT SIX
		"\x37"		=> "\x37",	//DIGIT SEVEN
		"\x38"		=> "\x38",	//DIGIT EIGHT
		"\x39"		=> "\x39",	//DIGIT NINE
		"\x3a"		=> "\x3a",	//COLON
		"\x3b"		=> "\x3b",	//SEMICOLON
		"\x3c"		=> "\x3c",	//LESS-THAN SIGN
		"\x3d"		=> "\x3d",	//EQUALS SIGN
		"\x3e"		=> "\x3e",	//GREATER-THAN SIGN
		"\x3f"		=> "\x3f",	//QUESTION MARK
		"\x40"		=> "\x40",	//COMMERCIAL AT
		"\x41"		=> "\x41",	//LATIN CAPITAL LETTER A
		"\x42"		=> "\x42",	//LATIN CAPITAL LETTER B
		"\x43"		=> "\x43",	//LATIN CAPITAL LETTER C
		"\x44"		=> "\x44",	//LATIN CAPITAL LETTER D
		"\x45"		=> "\x45",	//LATIN CAPITAL LETTER E
		"\x46"		=> "\x46",	//LATIN CAPITAL LETTER F
		"\x47"		=> "\x47",	//LATIN CAPITAL LETTER G
		"\x48"		=> "\x48",	//LATIN CAPITAL LETTER H
		"\x49"		=> "\x49",	//LATIN CAPITAL LETTER I
		"\x4a"		=> "\x4a",	//LATIN CAPITAL LETTER J
		"\x4b"		=> "\x4b",	//LATIN CAPITAL LETTER K
		"\x4c"		=> "\x4c",	//LATIN CAPITAL LETTER L
		"\x4d"		=> "\x4d",	//LATIN CAPITAL LETTER M
		"\x4e"		=> "\x4e",	//LATIN CAPITAL LETTER N
		"\x4f"		=> "\x4f",	//LATIN CAPITAL LETTER O
		"\x50"		=> "\x50",	//LATIN CAPITAL LETTER P
		"\x51"		=> "\x51",	//LATIN CAPITAL LETTER Q
		"\x52"		=> "\x52",	//LATIN CAPITAL LETTER R
		"\x53"		=> "\x53",	//LATIN CAPITAL LETTER S
		"\x54"		=> "\x54",	//LATIN CAPITAL LETTER T
		"\x55"		=> "\x55",	//LATIN CAPITAL LETTER U
		"\x56"		=> "\x56",	//LATIN CAPITAL LETTER V
		"\x57"		=> "\x57",	//LATIN CAPITAL LETTER W
		"\x58"		=> "\x58",	//LATIN CAPITAL LETTER X
		"\x59"		=> "\x59",	//LATIN CAPITAL LETTER Y
		"\x5a"		=> "\x5a",	//LATIN CAPITAL LETTER Z
		"\x5b"		=> "\x5b",	//LEFT SQUARE BRACKET
		"\x5c"		=> "\x5c",	//REVERSE SOLIDUS
		"\x5d"		=> "\x5d",	//RIGHT SQUARE BRACKET
		"\x5e"		=> "\x5e",	//CIRCUMFLEX ACCENT
		"\x5f"		=> "\x5f",	//LOW LINE
		"\x60"		=> "\x60",	//GRAVE ACCENT
		"\x61"		=> "\x61",	//LATIN SMALL LETTER A
		"\x62"		=> "\x62",	//LATIN SMALL LETTER B
		"\x63"		=> "\x63",	//LATIN SMALL LETTER C
		"\x64"		=> "\x64",	//LATIN SMALL LETTER D
		"\x65"		=> "\x65",	//LATIN SMALL LETTER E
		"\x66"		=> "\x66",	//LATIN SMALL LETTER F
		"\x67"		=> "\x67",	//LATIN SMALL LETTER G
		"\x68"		=> "\x68",	//LATIN SMALL LETTER H
		"\x69"		=> "\x69",	//LATIN SMALL LETTER I
		"\x6a"		=> "\x6a",	//LATIN SMALL LETTER J
		"\x6b"		=> "\x6b",	//LATIN SMALL LETTER K
		"\x6c"		=> "\x6c",	//LATIN SMALL LETTER L
		"\x6d"		=> "\x6d",	//LATIN SMALL LETTER M
		"\x6e"		=> "\x6e",	//LATIN SMALL LETTER N
		"\x6f"		=> "\x6f",	//LATIN SMALL LETTER O
		"\x70"		=> "\x70",	//LATIN SMALL LETTER P
		"\x71"		=> "\x71",	//LATIN SMALL LETTER Q
		"\x72"		=> "\x72",	//LATIN SMALL LETTER R
		"\x73"		=> "\x73",	//LATIN SMALL LETTER S
		"\x74"		=> "\x74",	//LATIN SMALL LETTER T
		"\x75"		=> "\x75",	//LATIN SMALL LETTER U
		"\x76"		=> "\x76",	//LATIN SMALL LETTER V
		"\x77"		=> "\x77",	//LATIN SMALL LETTER W
		"\x78"		=> "\x78",	//LATIN SMALL LETTER X
		"\x79"		=> "\x79",	//LATIN SMALL LETTER Y
		"\x7a"		=> "\x7a",	//LATIN SMALL LETTER Z
		"\x7b"		=> "\x7b",	//LEFT CURLY BRACKET
		"\x7c"		=> "\x7c",	//VERTICAL LINE
		"\x7d"		=> "\x7d",	//RIGHT CURLY BRACKET
		"\x7e"		=> "\x7e",	//TILDE
		"\x7f"		=> "\x7f",	//DELETE
		"\xef\xbd\xa1"		=> "\xa1",	//HALFWIDTH IDEOGRAPHIC FULL STOP
		"\xef\xbd\xa2"		=> "\xa2",	//HALFWIDTH LEFT CORNER BRACKET
		"\xef\xbd\xa3"		=> "\xa3",	//HALFWIDTH RIGHT CORNER BRACKET
		"\xef\xbd\xa4"		=> "\xa4",	//HALFWIDTH IDEOGRAPHIC COMMA
		"\xef\xbd\xa5"		=> "\xa5",	//HALFWIDTH KATAKANA MIDDLE DOT
		"\xef\xbd\xa6"		=> "\xa6",	//HALFWIDTH KATAKANA LETTER WO
		"\xef\xbd\xa7"		=> "\xa7",	//HALFWIDTH KATAKANA LETTER SMALL A
		"\xef\xbd\xa8"		=> "\xa8",	//HALFWIDTH KATAKANA LETTER SMALL I
		"\xef\xbd\xa9"		=> "\xa9",	//HALFWIDTH KATAKANA LETTER SMALL U
		"\xef\xbd\xaa"		=> "\xaa",	//HALFWIDTH KATAKANA LETTER SMALL E
		"\xef\xbd\xab"		=> "\xab",	//HALFWIDTH KATAKANA LETTER SMALL O
		"\xef\xbd\xac"		=> "\xac",	//HALFWIDTH KATAKANA LETTER SMALL YA
		"\xef\xbd\xad"		=> "\xad",	//HALFWIDTH KATAKANA LETTER SMALL YU
		"\xef\xbd\xae"		=> "\xae",	//HALFWIDTH KATAKANA LETTER SMALL YO
		"\xef\xbd\xaf"		=> "\xaf",	//HALFWIDTH KATAKANA LETTER SMALL TU
		"\xef\xbd\xb0"		=> "\xb0",	//HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
		"\xef\xbd\xb1"		=> "\xb1",	//HALFWIDTH KATAKANA LETTER A
		"\xef\xbd\xb2"		=> "\xb2",	//HALFWIDTH KATAKANA LETTER I
		"\xef\xbd\xb3"		=> "\xb3",	//HALFWIDTH KATAKANA LETTER U
		"\xef\xbd\xb4"		=> "\xb4",	//HALFWIDTH KATAKANA LETTER E
		"\xef\xbd\xb5"		=> "\xb5",	//HALFWIDTH KATAKANA LETTER O
		"\xef\xbd\xb6"		=> "\xb6",	//HALFWIDTH KATAKANA LETTER KA
		"\xef\xbd\xb7"		=> "\xb7",	//HALFWIDTH KATAKANA LETTER KI
		"\xef\xbd\xb8"		=> "\xb8",	//HALFWIDTH KATAKANA LETTER KU
		"\xef\xbd\xb9"		=> "\xb9",	//HALFWIDTH KATAKANA LETTER KE
		"\xef\xbd\xba"		=> "\xba",	//HALFWIDTH KATAKANA LETTER KO
		"\xef\xbd\xbb"		=> "\xbb",	//HALFWIDTH KATAKANA LETTER SA
		"\xef\xbd\xbc"		=> "\xbc",	//HALFWIDTH KATAKANA LETTER SI
		"\xef\xbd\xbd"		=> "\xbd",	//HALFWIDTH KATAKANA LETTER SU
		"\xef\xbd\xbe"		=> "\xbe",	//HALFWIDTH KATAKANA LETTER SE
		"\xef\xbd\xbf"		=> "\xbf",	//HALFWIDTH KATAKANA LETTER SO
		"\xef\xbe\x80"		=> "\xc0",	//HALFWIDTH KATAKANA LETTER TA
		"\xef\xbe\x81"		=> "\xc1",	//HALFWIDTH KATAKANA LETTER TI
		"\xef\xbe\x82"		=> "\xc2",	//HALFWIDTH KATAKANA LETTER TU
		"\xef\xbe\x83"		=> "\xc3",	//HALFWIDTH KATAKANA LETTER TE
		"\xef\xbe\x84"		=> "\xc4",	//HALFWIDTH KATAKANA LETTER TO
		"\xef\xbe\x85"		=> "\xc5",	//HALFWIDTH KATAKANA LETTER NA
		"\xef\xbe\x86"		=> "\xc6",	//HALFWIDTH KATAKANA LETTER NI
		"\xef\xbe\x87"		=> "\xc7",	//HALFWIDTH KATAKANA LETTER NU
		"\xef\xbe\x88"		=> "\xc8",	//HALFWIDTH KATAKANA LETTER NE
		"\xef\xbe\x89"		=> "\xc9",	//HALFWIDTH KATAKANA LETTER NO
		"\xef\xbe\x8a"		=> "\xca",	//HALFWIDTH KATAKANA LETTER HA
		"\xef\xbe\x8b"		=> "\xcb",	//HALFWIDTH KATAKANA LETTER HI
		"\xef\xbe\x8c"		=> "\xcc",	//HALFWIDTH KATAKANA LETTER HU
		"\xef\xbe\x8d"		=> "\xcd",	//HALFWIDTH KATAKANA LETTER HE
		"\xef\xbe\x8e"		=> "\xce",	//HALFWIDTH KATAKANA LETTER HO
		"\xef\xbe\x8f"		=> "\xcf",	//HALFWIDTH KATAKANA LETTER MA
		"\xef\xbe\x90"		=> "\xd0",	//HALFWIDTH KATAKANA LETTER MI
		"\xef\xbe\x91"		=> "\xd1",	//HALFWIDTH KATAKANA LETTER MU
		"\xef\xbe\x92"		=> "\xd2",	//HALFWIDTH KATAKANA LETTER ME
		"\xef\xbe\x93"		=> "\xd3",	//HALFWIDTH KATAKANA LETTER MO
		"\xef\xbe\x94"		=> "\xd4",	//HALFWIDTH KATAKANA LETTER YA
		"\xef\xbe\x95"		=> "\xd5",	//HALFWIDTH KATAKANA LETTER YU
		"\xef\xbe\x96"		=> "\xd6",	//HALFWIDTH KATAKANA LETTER YO
		"\xef\xbe\x97"		=> "\xd7",	//HALFWIDTH KATAKANA LETTER RA
		"\xef\xbe\x98"		=> "\xd8",	//HALFWIDTH KATAKANA LETTER RI
		"\xef\xbe\x99"		=> "\xd9",	//HALFWIDTH KATAKANA LETTER RU
		"\xef\xbe\x9a"		=> "\xda",	//HALFWIDTH KATAKANA LETTER RE
		"\xef\xbe\x9b"		=> "\xdb",	//HALFWIDTH KATAKANA LETTER RO
		"\xef\xbe\x9c"		=> "\xdc",	//HALFWIDTH KATAKANA LETTER WA
		"\xef\xbe\x9d"		=> "\xdd",	//HALFWIDTH KATAKANA LETTER N
		"\xef\xbe\x9e"		=> "\xde",	//HALFWIDTH KATAKANA VOICED SOUND MARK
		"\xef\xbe\x9f"		=> "\xdf",	//HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
		"\xe3\x80\x80"		=> "\x81\x40",	//IDEOGRAPHIC SPACE
		"\xe3\x80\x81"		=> "\x81\x41",	//IDEOGRAPHIC COMMA
		"\xe3\x80\x82"		=> "\x81\x42",	//IDEOGRAPHIC FULL STOP
		"\xef\xbc\x8c"		=> "\x81\x43",	//FULLWIDTH COMMA
		"\xef\xbc\x8e"		=> "\x81\x44",	//FULLWIDTH FULL STOP
		"\xe3\x83\xbb"		=> "\x81\x45",	//KATAKANA MIDDLE DOT
		"\xef\xbc\x9a"		=> "\x81\x46",	//FULLWIDTH COLON
		"\xef\xbc\x9b"		=> "\x81\x47",	//FULLWIDTH SEMICOLON
		"\xef\xbc\x9f"		=> "\x81\x48",	//FULLWIDTH QUESTION MARK
		"\xef\xbc\x81"		=> "\x81\x49",	//FULLWIDTH EXCLAMATION MARK
		"\xe3\x82\x9b"		=> "\x81\x4a",	//KATAKANA-HIRAGANA VOICED SOUND MARK
		"\xe3\x82\x9c"		=> "\x81\x4b",	//KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
		"\xc2\xb4"		=> "\x81\x4c",	//ACUTE ACCENT
		"\xef\xbd\x80"		=> "\x81\x4d",	//FULLWIDTH GRAVE ACCENT
		"\xc2\xa8"		=> "\x81\x4e",	//DIAERESIS
		"\xef\xbc\xbe"		=> "\x81\x4f",	//FULLWIDTH CIRCUMFLEX ACCENT
		"\xef\xbf\xa3"		=> "\x81\x50",	//FULLWIDTH MACRON
		"\xef\xbc\xbf"		=> "\x81\x51",	//FULLWIDTH LOW LINE
		"\xe3\x83\xbd"		=> "\x81\x52",	//KATAKANA ITERATION MARK
		"\xe3\x83\xbe"		=> "\x81\x53",	//KATAKANA VOICED ITERATION MARK
		"\xe3\x82\x9d"		=> "\x81\x54",	//HIRAGANA ITERATION MARK
		"\xe3\x82\x9e"		=> "\x81\x55",	//HIRAGANA VOICED ITERATION MARK
		"\xe3\x80\x83"		=> "\x81\x56",	//DITTO MARK
		"\xe4\xbb\x9d"		=> "\x81\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x80\x85"		=> "\x81\x58",	//IDEOGRAPHIC ITERATION MARK
		"\xe3\x80\x86"		=> "\x81\x59",	//IDEOGRAPHIC CLOSING MARK
		"\xe3\x80\x87"		=> "\x81\x5a",	//IDEOGRAPHIC NUMBER ZERO
		"\xe3\x83\xbc"		=> "\x81\x5b",	//KATAKANA-HIRAGANA PROLONGED SOUND MARK
		"\xe2\x80\x95"		=> "\x81\x5c",	//HORIZONTAL BAR
		"\xe2\x80\x90"		=> "\x81\x5d",	//HYPHEN
		"\xef\xbc\x8f"		=> "\x81\x5e",	//FULLWIDTH SOLIDUS
		"\xef\xbc\xbc"		=> "\x81\x5f",	//FULLWIDTH REVERSE SOLIDUS
		"\xef\xbd\x9e"		=> "\x81\x60",	//FULLWIDTH TILDE
		"\xe2\x88\xa5"		=> "\x81\x61",	//PARALLEL TO
		"\xef\xbd\x9c"		=> "\x81\x62",	//FULLWIDTH VERTICAL LINE
		"\xe2\x80\xa6"		=> "\x81\x63",	//HORIZONTAL ELLIPSIS
		"\xe2\x80\xa5"		=> "\x81\x64",	//TWO DOT LEADER
		"\xe2\x80\x98"		=> "\x81\x65",	//LEFT SINGLE QUOTATION MARK
		"\xe2\x80\x99"		=> "\x81\x66",	//RIGHT SINGLE QUOTATION MARK
		"\xe2\x80\x9c"		=> "\x81\x67",	//LEFT DOUBLE QUOTATION MARK
		"\xe2\x80\x9d"		=> "\x81\x68",	//RIGHT DOUBLE QUOTATION MARK
		"\xef\xbc\x88"		=> "\x81\x69",	//FULLWIDTH LEFT PARENTHESIS
		"\xef\xbc\x89"		=> "\x81\x6a",	//FULLWIDTH RIGHT PARENTHESIS
		"\xe3\x80\x94"		=> "\x81\x6b",	//LEFT TORTOISE SHELL BRACKET
		"\xe3\x80\x95"		=> "\x81\x6c",	//RIGHT TORTOISE SHELL BRACKET
		"\xef\xbc\xbb"		=> "\x81\x6d",	//FULLWIDTH LEFT SQUARE BRACKET
		"\xef\xbc\xbd"		=> "\x81\x6e",	//FULLWIDTH RIGHT SQUARE BRACKET
		"\xef\xbd\x9b"		=> "\x81\x6f",	//FULLWIDTH LEFT CURLY BRACKET
		"\xef\xbd\x9d"		=> "\x81\x70",	//FULLWIDTH RIGHT CURLY BRACKET
		"\xe3\x80\x88"		=> "\x81\x71",	//LEFT ANGLE BRACKET
		"\xe3\x80\x89"		=> "\x81\x72",	//RIGHT ANGLE BRACKET
		"\xe3\x80\x8a"		=> "\x81\x73",	//LEFT DOUBLE ANGLE BRACKET
		"\xe3\x80\x8b"		=> "\x81\x74",	//RIGHT DOUBLE ANGLE BRACKET
		"\xe3\x80\x8c"		=> "\x81\x75",	//LEFT CORNER BRACKET
		"\xe3\x80\x8d"		=> "\x81\x76",	//RIGHT CORNER BRACKET
		"\xe3\x80\x8e"		=> "\x81\x77",	//LEFT WHITE CORNER BRACKET
		"\xe3\x80\x8f"		=> "\x81\x78",	//RIGHT WHITE CORNER BRACKET
		"\xe3\x80\x90"		=> "\x81\x79",	//LEFT BLACK LENTICULAR BRACKET
		"\xe3\x80\x91"		=> "\x81\x7a",	//RIGHT BLACK LENTICULAR BRACKET
		"\xef\xbc\x8b"		=> "\x81\x7b",	//FULLWIDTH PLUS SIGN
		"\xef\xbc\x8d"		=> "\x81\x7c",	//FULLWIDTH HYPHEN-MINUS
		"\xc2\xb1"		=> "\x81\x7d",	//PLUS-MINUS SIGN
		"\xc3\x97"		=> "\x81\x7e",	//MULTIPLICATION SIGN
		"\xc3\xb7"		=> "\x81\x80",	//DIVISION SIGN
		"\xef\xbc\x9d"		=> "\x81\x81",	//FULLWIDTH EQUALS SIGN
		"\xe2\x89\xa0"		=> "\x81\x82",	//NOT EQUAL TO
		"\xef\xbc\x9c"		=> "\x81\x83",	//FULLWIDTH LESS-THAN SIGN
		"\xef\xbc\x9e"		=> "\x81\x84",	//FULLWIDTH GREATER-THAN SIGN
		"\xe2\x89\xa6"		=> "\x81\x85",	//LESS-THAN OVER EQUAL TO
		"\xe2\x89\xa7"		=> "\x81\x86",	//GREATER-THAN OVER EQUAL TO
		"\xe2\x88\x9e"		=> "\x81\x87",	//INFINITY
		"\xe2\x88\xb4"		=> "\x81\x88",	//THEREFORE
		"\xe2\x99\x82"		=> "\x81\x89",	//MALE SIGN
		"\xe2\x99\x80"		=> "\x81\x8a",	//FEMALE SIGN
		"\xc2\xb0"		=> "\x81\x8b",	//DEGREE SIGN
		"\xe2\x80\xb2"		=> "\x81\x8c",	//PRIME
		"\xe2\x80\xb3"		=> "\x81\x8d",	//DOUBLE PRIME
		"\xe2\x84\x83"		=> "\x81\x8e",	//DEGREE CELSIUS
		"\xef\xbf\xa5"		=> "\x81\x8f",	//FULLWIDTH YEN SIGN
		"\xef\xbc\x84"		=> "\x81\x90",	//FULLWIDTH DOLLAR SIGN
		"\xef\xbf\xa0"		=> "\x81\x91",	//FULLWIDTH CENT SIGN
		"\xef\xbf\xa1"		=> "\x81\x92",	//FULLWIDTH POUND SIGN
		"\xef\xbc\x85"		=> "\x81\x93",	//FULLWIDTH PERCENT SIGN
		"\xef\xbc\x83"		=> "\x81\x94",	//FULLWIDTH NUMBER SIGN
		"\xef\xbc\x86"		=> "\x81\x95",	//FULLWIDTH AMPERSAND
		"\xef\xbc\x8a"		=> "\x81\x96",	//FULLWIDTH ASTERISK
		"\xef\xbc\xa0"		=> "\x81\x97",	//FULLWIDTH COMMERCIAL AT
		"\xc2\xa7"		=> "\x81\x98",	//SECTION SIGN
		"\xe2\x98\x86"		=> "\x81\x99",	//WHITE STAR
		"\xe2\x98\x85"		=> "\x81\x9a",	//BLACK STAR
		"\xe2\x97\x8b"		=> "\x81\x9b",	//WHITE CIRCLE
		"\xe2\x97\x8f"		=> "\x81\x9c",	//BLACK CIRCLE
		"\xe2\x97\x8e"		=> "\x81\x9d",	//BULLSEYE
		"\xe2\x97\x87"		=> "\x81\x9e",	//WHITE DIAMOND
		"\xe2\x97\x86"		=> "\x81\x9f",	//BLACK DIAMOND
		"\xe2\x96\xa1"		=> "\x81\xa0",	//WHITE SQUARE
		"\xe2\x96\xa0"		=> "\x81\xa1",	//BLACK SQUARE
		"\xe2\x96\xb3"		=> "\x81\xa2",	//WHITE UP-POINTING TRIANGLE
		"\xe2\x96\xb2"		=> "\x81\xa3",	//BLACK UP-POINTING TRIANGLE
		"\xe2\x96\xbd"		=> "\x81\xa4",	//WHITE DOWN-POINTING TRIANGLE
		"\xe2\x96\xbc"		=> "\x81\xa5",	//BLACK DOWN-POINTING TRIANGLE
		"\xe2\x80\xbb"		=> "\x81\xa6",	//REFERENCE MARK
		"\xe3\x80\x92"		=> "\x81\xa7",	//POSTAL MARK
		"\xe2\x86\x92"		=> "\x81\xa8",	//RIGHTWARDS ARROW
		"\xe2\x86\x90"		=> "\x81\xa9",	//LEFTWARDS ARROW
		"\xe2\x86\x91"		=> "\x81\xaa",	//UPWARDS ARROW
		"\xe2\x86\x93"		=> "\x81\xab",	//DOWNWARDS ARROW
		"\xe3\x80\x93"		=> "\x81\xac",	//GETA MARK
		"\xe2\x88\x88"		=> "\x81\xb8",	//ELEMENT OF
		"\xe2\x88\x8b"		=> "\x81\xb9",	//CONTAINS AS MEMBER
		"\xe2\x8a\x86"		=> "\x81\xba",	//SUBSET OF OR EQUAL TO
		"\xe2\x8a\x87"		=> "\x81\xbb",	//SUPERSET OF OR EQUAL TO
		"\xe2\x8a\x82"		=> "\x81\xbc",	//SUBSET OF
		"\xe2\x8a\x83"		=> "\x81\xbd",	//SUPERSET OF
		"\xe2\x88\xaa"		=> "\x81\xbe",	//UNION
		"\xe2\x88\xa9"		=> "\x81\xbf",	//INTERSECTION
		"\xe2\x88\xa7"		=> "\x81\xc8",	//LOGICAL AND
		"\xe2\x88\xa8"		=> "\x81\xc9",	//LOGICAL OR
		"\xef\xbf\xa2"		=> "\x81\xca",	//FULLWIDTH NOT SIGN
		"\xe2\x87\x92"		=> "\x81\xcb",	//RIGHTWARDS DOUBLE ARROW
		"\xe2\x87\x94"		=> "\x81\xcc",	//LEFT RIGHT DOUBLE ARROW
		"\xe2\x88\x80"		=> "\x81\xcd",	//FOR ALL
		"\xe2\x88\x83"		=> "\x81\xce",	//THERE EXISTS
		"\xe2\x88\xa0"		=> "\x81\xda",	//ANGLE
		"\xe2\x8a\xa5"		=> "\x81\xdb",	//UP TACK
		"\xe2\x8c\x92"		=> "\x81\xdc",	//ARC
		"\xe2\x88\x82"		=> "\x81\xdd",	//PARTIAL DIFFERENTIAL
		"\xe2\x88\x87"		=> "\x81\xde",	//NABLA
		"\xe2\x89\xa1"		=> "\x81\xdf",	//IDENTICAL TO
		"\xe2\x89\x92"		=> "\x81\xe0",	//APPROXIMATELY EQUAL TO OR THE IMAGE OF
		"\xe2\x89\xaa"		=> "\x81\xe1",	//MUCH LESS-THAN
		"\xe2\x89\xab"		=> "\x81\xe2",	//MUCH GREATER-THAN
		"\xe2\x88\x9a"		=> "\x81\xe3",	//SQUARE ROOT
		"\xe2\x88\xbd"		=> "\x81\xe4",	//REVERSED TILDE
		"\xe2\x88\x9d"		=> "\x81\xe5",	//PROPORTIONAL TO
		"\xe2\x88\xb5"		=> "\x81\xe6",	//BECAUSE
		"\xe2\x88\xab"		=> "\x81\xe7",	//INTEGRAL
		"\xe2\x88\xac"		=> "\x81\xe8",	//DOUBLE INTEGRAL
		"\xe2\x84\xab"		=> "\x81\xf0",	//ANGSTROM SIGN
		"\xe2\x80\xb0"		=> "\x81\xf1",	//PER MILLE SIGN
		"\xe2\x99\xaf"		=> "\x81\xf2",	//MUSIC SHARP SIGN
		"\xe2\x99\xad"		=> "\x81\xf3",	//MUSIC FLAT SIGN
		"\xe2\x99\xaa"		=> "\x81\xf4",	//EIGHTH NOTE
		"\xe2\x80\xa0"		=> "\x81\xf5",	//DAGGER
		"\xe2\x80\xa1"		=> "\x81\xf6",	//DOUBLE DAGGER
		"\xc2\xb6"		=> "\x81\xf7",	//PILCROW SIGN
		"\xe2\x97\xaf"		=> "\x81\xfc",	//LARGE CIRCLE
		"\xef\xbc\x90"		=> "\x82\x4f",	//FULLWIDTH DIGIT ZERO
		"\xef\xbc\x91"		=> "\x82\x50",	//FULLWIDTH DIGIT ONE
		"\xef\xbc\x92"		=> "\x82\x51",	//FULLWIDTH DIGIT TWO
		"\xef\xbc\x93"		=> "\x82\x52",	//FULLWIDTH DIGIT THREE
		"\xef\xbc\x94"		=> "\x82\x53",	//FULLWIDTH DIGIT FOUR
		"\xef\xbc\x95"		=> "\x82\x54",	//FULLWIDTH DIGIT FIVE
		"\xef\xbc\x96"		=> "\x82\x55",	//FULLWIDTH DIGIT SIX
		"\xef\xbc\x97"		=> "\x82\x56",	//FULLWIDTH DIGIT SEVEN
		"\xef\xbc\x98"		=> "\x82\x57",	//FULLWIDTH DIGIT EIGHT
		"\xef\xbc\x99"		=> "\x82\x58",	//FULLWIDTH DIGIT NINE
		"\xef\xbc\xa1"		=> "\x82\x60",	//FULLWIDTH LATIN CAPITAL LETTER A
		"\xef\xbc\xa2"		=> "\x82\x61",	//FULLWIDTH LATIN CAPITAL LETTER B
		"\xef\xbc\xa3"		=> "\x82\x62",	//FULLWIDTH LATIN CAPITAL LETTER C
		"\xef\xbc\xa4"		=> "\x82\x63",	//FULLWIDTH LATIN CAPITAL LETTER D
		"\xef\xbc\xa5"		=> "\x82\x64",	//FULLWIDTH LATIN CAPITAL LETTER E
		"\xef\xbc\xa6"		=> "\x82\x65",	//FULLWIDTH LATIN CAPITAL LETTER F
		"\xef\xbc\xa7"		=> "\x82\x66",	//FULLWIDTH LATIN CAPITAL LETTER G
		"\xef\xbc\xa8"		=> "\x82\x67",	//FULLWIDTH LATIN CAPITAL LETTER H
		"\xef\xbc\xa9"		=> "\x82\x68",	//FULLWIDTH LATIN CAPITAL LETTER I
		"\xef\xbc\xaa"		=> "\x82\x69",	//FULLWIDTH LATIN CAPITAL LETTER J
		"\xef\xbc\xab"		=> "\x82\x6a",	//FULLWIDTH LATIN CAPITAL LETTER K
		"\xef\xbc\xac"		=> "\x82\x6b",	//FULLWIDTH LATIN CAPITAL LETTER L
		"\xef\xbc\xad"		=> "\x82\x6c",	//FULLWIDTH LATIN CAPITAL LETTER M
		"\xef\xbc\xae"		=> "\x82\x6d",	//FULLWIDTH LATIN CAPITAL LETTER N
		"\xef\xbc\xaf"		=> "\x82\x6e",	//FULLWIDTH LATIN CAPITAL LETTER O
		"\xef\xbc\xb0"		=> "\x82\x6f",	//FULLWIDTH LATIN CAPITAL LETTER P
		"\xef\xbc\xb1"		=> "\x82\x70",	//FULLWIDTH LATIN CAPITAL LETTER Q
		"\xef\xbc\xb2"		=> "\x82\x71",	//FULLWIDTH LATIN CAPITAL LETTER R
		"\xef\xbc\xb3"		=> "\x82\x72",	//FULLWIDTH LATIN CAPITAL LETTER S
		"\xef\xbc\xb4"		=> "\x82\x73",	//FULLWIDTH LATIN CAPITAL LETTER T
		"\xef\xbc\xb5"		=> "\x82\x74",	//FULLWIDTH LATIN CAPITAL LETTER U
		"\xef\xbc\xb6"		=> "\x82\x75",	//FULLWIDTH LATIN CAPITAL LETTER V
		"\xef\xbc\xb7"		=> "\x82\x76",	//FULLWIDTH LATIN CAPITAL LETTER W
		"\xef\xbc\xb8"		=> "\x82\x77",	//FULLWIDTH LATIN CAPITAL LETTER X
		"\xef\xbc\xb9"		=> "\x82\x78",	//FULLWIDTH LATIN CAPITAL LETTER Y
		"\xef\xbc\xba"		=> "\x82\x79",	//FULLWIDTH LATIN CAPITAL LETTER Z
		"\xef\xbd\x81"		=> "\x82\x81",	//FULLWIDTH LATIN SMALL LETTER A
		"\xef\xbd\x82"		=> "\x82\x82",	//FULLWIDTH LATIN SMALL LETTER B
		"\xef\xbd\x83"		=> "\x82\x83",	//FULLWIDTH LATIN SMALL LETTER C
		"\xef\xbd\x84"		=> "\x82\x84",	//FULLWIDTH LATIN SMALL LETTER D
		"\xef\xbd\x85"		=> "\x82\x85",	//FULLWIDTH LATIN SMALL LETTER E
		"\xef\xbd\x86"		=> "\x82\x86",	//FULLWIDTH LATIN SMALL LETTER F
		"\xef\xbd\x87"		=> "\x82\x87",	//FULLWIDTH LATIN SMALL LETTER G
		"\xef\xbd\x88"		=> "\x82\x88",	//FULLWIDTH LATIN SMALL LETTER H
		"\xef\xbd\x89"		=> "\x82\x89",	//FULLWIDTH LATIN SMALL LETTER I
		"\xef\xbd\x8a"		=> "\x82\x8a",	//FULLWIDTH LATIN SMALL LETTER J
		"\xef\xbd\x8b"		=> "\x82\x8b",	//FULLWIDTH LATIN SMALL LETTER K
		"\xef\xbd\x8c"		=> "\x82\x8c",	//FULLWIDTH LATIN SMALL LETTER L
		"\xef\xbd\x8d"		=> "\x82\x8d",	//FULLWIDTH LATIN SMALL LETTER M
		"\xef\xbd\x8e"		=> "\x82\x8e",	//FULLWIDTH LATIN SMALL LETTER N
		"\xef\xbd\x8f"		=> "\x82\x8f",	//FULLWIDTH LATIN SMALL LETTER O
		"\xef\xbd\x90"		=> "\x82\x90",	//FULLWIDTH LATIN SMALL LETTER P
		"\xef\xbd\x91"		=> "\x82\x91",	//FULLWIDTH LATIN SMALL LETTER Q
		"\xef\xbd\x92"		=> "\x82\x92",	//FULLWIDTH LATIN SMALL LETTER R
		"\xef\xbd\x93"		=> "\x82\x93",	//FULLWIDTH LATIN SMALL LETTER S
		"\xef\xbd\x94"		=> "\x82\x94",	//FULLWIDTH LATIN SMALL LETTER T
		"\xef\xbd\x95"		=> "\x82\x95",	//FULLWIDTH LATIN SMALL LETTER U
		"\xef\xbd\x96"		=> "\x82\x96",	//FULLWIDTH LATIN SMALL LETTER V
		"\xef\xbd\x97"		=> "\x82\x97",	//FULLWIDTH LATIN SMALL LETTER W
		"\xef\xbd\x98"		=> "\x82\x98",	//FULLWIDTH LATIN SMALL LETTER X
		"\xef\xbd\x99"		=> "\x82\x99",	//FULLWIDTH LATIN SMALL LETTER Y
		"\xef\xbd\x9a"		=> "\x82\x9a",	//FULLWIDTH LATIN SMALL LETTER Z
		"\xe3\x81\x81"		=> "\x82\x9f",	//HIRAGANA LETTER SMALL A
		"\xe3\x81\x82"		=> "\x82\xa0",	//HIRAGANA LETTER A
		"\xe3\x81\x83"		=> "\x82\xa1",	//HIRAGANA LETTER SMALL I
		"\xe3\x81\x84"		=> "\x82\xa2",	//HIRAGANA LETTER I
		"\xe3\x81\x85"		=> "\x82\xa3",	//HIRAGANA LETTER SMALL U
		"\xe3\x81\x86"		=> "\x82\xa4",	//HIRAGANA LETTER U
		"\xe3\x81\x87"		=> "\x82\xa5",	//HIRAGANA LETTER SMALL E
		"\xe3\x81\x88"		=> "\x82\xa6",	//HIRAGANA LETTER E
		"\xe3\x81\x89"		=> "\x82\xa7",	//HIRAGANA LETTER SMALL O
		"\xe3\x81\x8a"		=> "\x82\xa8",	//HIRAGANA LETTER O
		"\xe3\x81\x8b"		=> "\x82\xa9",	//HIRAGANA LETTER KA
		"\xe3\x81\x8c"		=> "\x82\xaa",	//HIRAGANA LETTER GA
		"\xe3\x81\x8d"		=> "\x82\xab",	//HIRAGANA LETTER KI
		"\xe3\x81\x8e"		=> "\x82\xac",	//HIRAGANA LETTER GI
		"\xe3\x81\x8f"		=> "\x82\xad",	//HIRAGANA LETTER KU
		"\xe3\x81\x90"		=> "\x82\xae",	//HIRAGANA LETTER GU
		"\xe3\x81\x91"		=> "\x82\xaf",	//HIRAGANA LETTER KE
		"\xe3\x81\x92"		=> "\x82\xb0",	//HIRAGANA LETTER GE
		"\xe3\x81\x93"		=> "\x82\xb1",	//HIRAGANA LETTER KO
		"\xe3\x81\x94"		=> "\x82\xb2",	//HIRAGANA LETTER GO
		"\xe3\x81\x95"		=> "\x82\xb3",	//HIRAGANA LETTER SA
		"\xe3\x81\x96"		=> "\x82\xb4",	//HIRAGANA LETTER ZA
		"\xe3\x81\x97"		=> "\x82\xb5",	//HIRAGANA LETTER SI
		"\xe3\x81\x98"		=> "\x82\xb6",	//HIRAGANA LETTER ZI
		"\xe3\x81\x99"		=> "\x82\xb7",	//HIRAGANA LETTER SU
		"\xe3\x81\x9a"		=> "\x82\xb8",	//HIRAGANA LETTER ZU
		"\xe3\x81\x9b"		=> "\x82\xb9",	//HIRAGANA LETTER SE
		"\xe3\x81\x9c"		=> "\x82\xba",	//HIRAGANA LETTER ZE
		"\xe3\x81\x9d"		=> "\x82\xbb",	//HIRAGANA LETTER SO
		"\xe3\x81\x9e"		=> "\x82\xbc",	//HIRAGANA LETTER ZO
		"\xe3\x81\x9f"		=> "\x82\xbd",	//HIRAGANA LETTER TA
		"\xe3\x81\xa0"		=> "\x82\xbe",	//HIRAGANA LETTER DA
		"\xe3\x81\xa1"		=> "\x82\xbf",	//HIRAGANA LETTER TI
		"\xe3\x81\xa2"		=> "\x82\xc0",	//HIRAGANA LETTER DI
		"\xe3\x81\xa3"		=> "\x82\xc1",	//HIRAGANA LETTER SMALL TU
		"\xe3\x81\xa4"		=> "\x82\xc2",	//HIRAGANA LETTER TU
		"\xe3\x81\xa5"		=> "\x82\xc3",	//HIRAGANA LETTER DU
		"\xe3\x81\xa6"		=> "\x82\xc4",	//HIRAGANA LETTER TE
		"\xe3\x81\xa7"		=> "\x82\xc5",	//HIRAGANA LETTER DE
		"\xe3\x81\xa8"		=> "\x82\xc6",	//HIRAGANA LETTER TO
		"\xe3\x81\xa9"		=> "\x82\xc7",	//HIRAGANA LETTER DO
		"\xe3\x81\xaa"		=> "\x82\xc8",	//HIRAGANA LETTER NA
		"\xe3\x81\xab"		=> "\x82\xc9",	//HIRAGANA LETTER NI
		"\xe3\x81\xac"		=> "\x82\xca",	//HIRAGANA LETTER NU
		"\xe3\x81\xad"		=> "\x82\xcb",	//HIRAGANA LETTER NE
		"\xe3\x81\xae"		=> "\x82\xcc",	//HIRAGANA LETTER NO
		"\xe3\x81\xaf"		=> "\x82\xcd",	//HIRAGANA LETTER HA
		"\xe3\x81\xb0"		=> "\x82\xce",	//HIRAGANA LETTER BA
		"\xe3\x81\xb1"		=> "\x82\xcf",	//HIRAGANA LETTER PA
		"\xe3\x81\xb2"		=> "\x82\xd0",	//HIRAGANA LETTER HI
		"\xe3\x81\xb3"		=> "\x82\xd1",	//HIRAGANA LETTER BI
		"\xe3\x81\xb4"		=> "\x82\xd2",	//HIRAGANA LETTER PI
		"\xe3\x81\xb5"		=> "\x82\xd3",	//HIRAGANA LETTER HU
		"\xe3\x81\xb6"		=> "\x82\xd4",	//HIRAGANA LETTER BU
		"\xe3\x81\xb7"		=> "\x82\xd5",	//HIRAGANA LETTER PU
		"\xe3\x81\xb8"		=> "\x82\xd6",	//HIRAGANA LETTER HE
		"\xe3\x81\xb9"		=> "\x82\xd7",	//HIRAGANA LETTER BE
		"\xe3\x81\xba"		=> "\x82\xd8",	//HIRAGANA LETTER PE
		"\xe3\x81\xbb"		=> "\x82\xd9",	//HIRAGANA LETTER HO
		"\xe3\x81\xbc"		=> "\x82\xda",	//HIRAGANA LETTER BO
		"\xe3\x81\xbd"		=> "\x82\xdb",	//HIRAGANA LETTER PO
		"\xe3\x81\xbe"		=> "\x82\xdc",	//HIRAGANA LETTER MA
		"\xe3\x81\xbf"		=> "\x82\xdd",	//HIRAGANA LETTER MI
		"\xe3\x82\x80"		=> "\x82\xde",	//HIRAGANA LETTER MU
		"\xe3\x82\x81"		=> "\x82\xdf",	//HIRAGANA LETTER ME
		"\xe3\x82\x82"		=> "\x82\xe0",	//HIRAGANA LETTER MO
		"\xe3\x82\x83"		=> "\x82\xe1",	//HIRAGANA LETTER SMALL YA
		"\xe3\x82\x84"		=> "\x82\xe2",	//HIRAGANA LETTER YA
		"\xe3\x82\x85"		=> "\x82\xe3",	//HIRAGANA LETTER SMALL YU
		"\xe3\x82\x86"		=> "\x82\xe4",	//HIRAGANA LETTER YU
		"\xe3\x82\x87"		=> "\x82\xe5",	//HIRAGANA LETTER SMALL YO
		"\xe3\x82\x88"		=> "\x82\xe6",	//HIRAGANA LETTER YO
		"\xe3\x82\x89"		=> "\x82\xe7",	//HIRAGANA LETTER RA
		"\xe3\x82\x8a"		=> "\x82\xe8",	//HIRAGANA LETTER RI
		"\xe3\x82\x8b"		=> "\x82\xe9",	//HIRAGANA LETTER RU
		"\xe3\x82\x8c"		=> "\x82\xea",	//HIRAGANA LETTER RE
		"\xe3\x82\x8d"		=> "\x82\xeb",	//HIRAGANA LETTER RO
		"\xe3\x82\x8e"		=> "\x82\xec",	//HIRAGANA LETTER SMALL WA
		"\xe3\x82\x8f"		=> "\x82\xed",	//HIRAGANA LETTER WA
		"\xe3\x82\x90"		=> "\x82\xee",	//HIRAGANA LETTER WI
		"\xe3\x82\x91"		=> "\x82\xef",	//HIRAGANA LETTER WE
		"\xe3\x82\x92"		=> "\x82\xf0",	//HIRAGANA LETTER WO
		"\xe3\x82\x93"		=> "\x82\xf1",	//HIRAGANA LETTER N
		"\xe3\x82\xa1"		=> "\x83\x40",	//KATAKANA LETTER SMALL A
		"\xe3\x82\xa2"		=> "\x83\x41",	//KATAKANA LETTER A
		"\xe3\x82\xa3"		=> "\x83\x42",	//KATAKANA LETTER SMALL I
		"\xe3\x82\xa4"		=> "\x83\x43",	//KATAKANA LETTER I
		"\xe3\x82\xa5"		=> "\x83\x44",	//KATAKANA LETTER SMALL U
		"\xe3\x82\xa6"		=> "\x83\x45",	//KATAKANA LETTER U
		"\xe3\x82\xa7"		=> "\x83\x46",	//KATAKANA LETTER SMALL E
		"\xe3\x82\xa8"		=> "\x83\x47",	//KATAKANA LETTER E
		"\xe3\x82\xa9"		=> "\x83\x48",	//KATAKANA LETTER SMALL O
		"\xe3\x82\xaa"		=> "\x83\x49",	//KATAKANA LETTER O
		"\xe3\x82\xab"		=> "\x83\x4a",	//KATAKANA LETTER KA
		"\xe3\x82\xac"		=> "\x83\x4b",	//KATAKANA LETTER GA
		"\xe3\x82\xad"		=> "\x83\x4c",	//KATAKANA LETTER KI
		"\xe3\x82\xae"		=> "\x83\x4d",	//KATAKANA LETTER GI
		"\xe3\x82\xaf"		=> "\x83\x4e",	//KATAKANA LETTER KU
		"\xe3\x82\xb0"		=> "\x83\x4f",	//KATAKANA LETTER GU
		"\xe3\x82\xb1"		=> "\x83\x50",	//KATAKANA LETTER KE
		"\xe3\x82\xb2"		=> "\x83\x51",	//KATAKANA LETTER GE
		"\xe3\x82\xb3"		=> "\x83\x52",	//KATAKANA LETTER KO
		"\xe3\x82\xb4"		=> "\x83\x53",	//KATAKANA LETTER GO
		"\xe3\x82\xb5"		=> "\x83\x54",	//KATAKANA LETTER SA
		"\xe3\x82\xb6"		=> "\x83\x55",	//KATAKANA LETTER ZA
		"\xe3\x82\xb7"		=> "\x83\x56",	//KATAKANA LETTER SI
		"\xe3\x82\xb8"		=> "\x83\x57",	//KATAKANA LETTER ZI
		"\xe3\x82\xb9"		=> "\x83\x58",	//KATAKANA LETTER SU
		"\xe3\x82\xba"		=> "\x83\x59",	//KATAKANA LETTER ZU
		"\xe3\x82\xbb"		=> "\x83\x5a",	//KATAKANA LETTER SE
		"\xe3\x82\xbc"		=> "\x83\x5b",	//KATAKANA LETTER ZE
		"\xe3\x82\xbd"		=> "\x83\x5c",	//KATAKANA LETTER SO
		"\xe3\x82\xbe"		=> "\x83\x5d",	//KATAKANA LETTER ZO
		"\xe3\x82\xbf"		=> "\x83\x5e",	//KATAKANA LETTER TA
		"\xe3\x83\x80"		=> "\x83\x5f",	//KATAKANA LETTER DA
		"\xe3\x83\x81"		=> "\x83\x60",	//KATAKANA LETTER TI
		"\xe3\x83\x82"		=> "\x83\x61",	//KATAKANA LETTER DI
		"\xe3\x83\x83"		=> "\x83\x62",	//KATAKANA LETTER SMALL TU
		"\xe3\x83\x84"		=> "\x83\x63",	//KATAKANA LETTER TU
		"\xe3\x83\x85"		=> "\x83\x64",	//KATAKANA LETTER DU
		"\xe3\x83\x86"		=> "\x83\x65",	//KATAKANA LETTER TE
		"\xe3\x83\x87"		=> "\x83\x66",	//KATAKANA LETTER DE
		"\xe3\x83\x88"		=> "\x83\x67",	//KATAKANA LETTER TO
		"\xe3\x83\x89"		=> "\x83\x68",	//KATAKANA LETTER DO
		"\xe3\x83\x8a"		=> "\x83\x69",	//KATAKANA LETTER NA
		"\xe3\x83\x8b"		=> "\x83\x6a",	//KATAKANA LETTER NI
		"\xe3\x83\x8c"		=> "\x83\x6b",	//KATAKANA LETTER NU
		"\xe3\x83\x8d"		=> "\x83\x6c",	//KATAKANA LETTER NE
		"\xe3\x83\x8e"		=> "\x83\x6d",	//KATAKANA LETTER NO
		"\xe3\x83\x8f"		=> "\x83\x6e",	//KATAKANA LETTER HA
		"\xe3\x83\x90"		=> "\x83\x6f",	//KATAKANA LETTER BA
		"\xe3\x83\x91"		=> "\x83\x70",	//KATAKANA LETTER PA
		"\xe3\x83\x92"		=> "\x83\x71",	//KATAKANA LETTER HI
		"\xe3\x83\x93"		=> "\x83\x72",	//KATAKANA LETTER BI
		"\xe3\x83\x94"		=> "\x83\x73",	//KATAKANA LETTER PI
		"\xe3\x83\x95"		=> "\x83\x74",	//KATAKANA LETTER HU
		"\xe3\x83\x96"		=> "\x83\x75",	//KATAKANA LETTER BU
		"\xe3\x83\x97"		=> "\x83\x76",	//KATAKANA LETTER PU
		"\xe3\x83\x98"		=> "\x83\x77",	//KATAKANA LETTER HE
		"\xe3\x83\x99"		=> "\x83\x78",	//KATAKANA LETTER BE
		"\xe3\x83\x9a"		=> "\x83\x79",	//KATAKANA LETTER PE
		"\xe3\x83\x9b"		=> "\x83\x7a",	//KATAKANA LETTER HO
		"\xe3\x83\x9c"		=> "\x83\x7b",	//KATAKANA LETTER BO
		"\xe3\x83\x9d"		=> "\x83\x7c",	//KATAKANA LETTER PO
		"\xe3\x83\x9e"		=> "\x83\x7d",	//KATAKANA LETTER MA
		"\xe3\x83\x9f"		=> "\x83\x7e",	//KATAKANA LETTER MI
		"\xe3\x83\xa0"		=> "\x83\x80",	//KATAKANA LETTER MU
		"\xe3\x83\xa1"		=> "\x83\x81",	//KATAKANA LETTER ME
		"\xe3\x83\xa2"		=> "\x83\x82",	//KATAKANA LETTER MO
		"\xe3\x83\xa3"		=> "\x83\x83",	//KATAKANA LETTER SMALL YA
		"\xe3\x83\xa4"		=> "\x83\x84",	//KATAKANA LETTER YA
		"\xe3\x83\xa5"		=> "\x83\x85",	//KATAKANA LETTER SMALL YU
		"\xe3\x83\xa6"		=> "\x83\x86",	//KATAKANA LETTER YU
		"\xe3\x83\xa7"		=> "\x83\x87",	//KATAKANA LETTER SMALL YO
		"\xe3\x83\xa8"		=> "\x83\x88",	//KATAKANA LETTER YO
		"\xe3\x83\xa9"		=> "\x83\x89",	//KATAKANA LETTER RA
		"\xe3\x83\xaa"		=> "\x83\x8a",	//KATAKANA LETTER RI
		"\xe3\x83\xab"		=> "\x83\x8b",	//KATAKANA LETTER RU
		"\xe3\x83\xac"		=> "\x83\x8c",	//KATAKANA LETTER RE
		"\xe3\x83\xad"		=> "\x83\x8d",	//KATAKANA LETTER RO
		"\xe3\x83\xae"		=> "\x83\x8e",	//KATAKANA LETTER SMALL WA
		"\xe3\x83\xaf"		=> "\x83\x8f",	//KATAKANA LETTER WA
		"\xe3\x83\xb0"		=> "\x83\x90",	//KATAKANA LETTER WI
		"\xe3\x83\xb1"		=> "\x83\x91",	//KATAKANA LETTER WE
		"\xe3\x83\xb2"		=> "\x83\x92",	//KATAKANA LETTER WO
		"\xe3\x83\xb3"		=> "\x83\x93",	//KATAKANA LETTER N
		"\xe3\x83\xb4"		=> "\x83\x94",	//KATAKANA LETTER VU
		"\xe3\x83\xb5"		=> "\x83\x95",	//KATAKANA LETTER SMALL KA
		"\xe3\x83\xb6"		=> "\x83\x96",	//KATAKANA LETTER SMALL KE
		"\xce\x91"		=> "\x83\x9f",	//GREEK CAPITAL LETTER ALPHA
		"\xce\x92"		=> "\x83\xa0",	//GREEK CAPITAL LETTER BETA
		"\xce\x93"		=> "\x83\xa1",	//GREEK CAPITAL LETTER GAMMA
		"\xce\x94"		=> "\x83\xa2",	//GREEK CAPITAL LETTER DELTA
		"\xce\x95"		=> "\x83\xa3",	//GREEK CAPITAL LETTER EPSILON
		"\xce\x96"		=> "\x83\xa4",	//GREEK CAPITAL LETTER ZETA
		"\xce\x97"		=> "\x83\xa5",	//GREEK CAPITAL LETTER ETA
		"\xce\x98"		=> "\x83\xa6",	//GREEK CAPITAL LETTER THETA
		"\xce\x99"		=> "\x83\xa7",	//GREEK CAPITAL LETTER IOTA
		"\xce\x9a"		=> "\x83\xa8",	//GREEK CAPITAL LETTER KAPPA
		"\xce\x9b"		=> "\x83\xa9",	//GREEK CAPITAL LETTER LAMDA
		"\xce\x9c"		=> "\x83\xaa",	//GREEK CAPITAL LETTER MU
		"\xce\x9d"		=> "\x83\xab",	//GREEK CAPITAL LETTER NU
		"\xce\x9e"		=> "\x83\xac",	//GREEK CAPITAL LETTER XI
		"\xce\x9f"		=> "\x83\xad",	//GREEK CAPITAL LETTER OMICRON
		"\xce\xa0"		=> "\x83\xae",	//GREEK CAPITAL LETTER PI
		"\xce\xa1"		=> "\x83\xaf",	//GREEK CAPITAL LETTER RHO
		"\xce\xa3"		=> "\x83\xb0",	//GREEK CAPITAL LETTER SIGMA
		"\xce\xa4"		=> "\x83\xb1",	//GREEK CAPITAL LETTER TAU
		"\xce\xa5"		=> "\x83\xb2",	//GREEK CAPITAL LETTER UPSILON
		"\xce\xa6"		=> "\x83\xb3",	//GREEK CAPITAL LETTER PHI
		"\xce\xa7"		=> "\x83\xb4",	//GREEK CAPITAL LETTER CHI
		"\xce\xa8"		=> "\x83\xb5",	//GREEK CAPITAL LETTER PSI
		"\xce\xa9"		=> "\x83\xb6",	//GREEK CAPITAL LETTER OMEGA
		"\xce\xb1"		=> "\x83\xbf",	//GREEK SMALL LETTER ALPHA
		"\xce\xb2"		=> "\x83\xc0",	//GREEK SMALL LETTER BETA
		"\xce\xb3"		=> "\x83\xc1",	//GREEK SMALL LETTER GAMMA
		"\xce\xb4"		=> "\x83\xc2",	//GREEK SMALL LETTER DELTA
		"\xce\xb5"		=> "\x83\xc3",	//GREEK SMALL LETTER EPSILON
		"\xce\xb6"		=> "\x83\xc4",	//GREEK SMALL LETTER ZETA
		"\xce\xb7"		=> "\x83\xc5",	//GREEK SMALL LETTER ETA
		"\xce\xb8"		=> "\x83\xc6",	//GREEK SMALL LETTER THETA
		"\xce\xb9"		=> "\x83\xc7",	//GREEK SMALL LETTER IOTA
		"\xce\xba"		=> "\x83\xc8",	//GREEK SMALL LETTER KAPPA
		"\xce\xbb"		=> "\x83\xc9",	//GREEK SMALL LETTER LAMDA
		"\xce\xbc"		=> "\x83\xca",	//GREEK SMALL LETTER MU
		"\xce\xbd"		=> "\x83\xcb",	//GREEK SMALL LETTER NU
		"\xce\xbe"		=> "\x83\xcc",	//GREEK SMALL LETTER XI
		"\xce\xbf"		=> "\x83\xcd",	//GREEK SMALL LETTER OMICRON
		"\xcf\x80"		=> "\x83\xce",	//GREEK SMALL LETTER PI
		"\xcf\x81"		=> "\x83\xcf",	//GREEK SMALL LETTER RHO
		"\xcf\x83"		=> "\x83\xd0",	//GREEK SMALL LETTER SIGMA
		"\xcf\x84"		=> "\x83\xd1",	//GREEK SMALL LETTER TAU
		"\xcf\x85"		=> "\x83\xd2",	//GREEK SMALL LETTER UPSILON
		"\xcf\x86"		=> "\x83\xd3",	//GREEK SMALL LETTER PHI
		"\xcf\x87"		=> "\x83\xd4",	//GREEK SMALL LETTER CHI
		"\xcf\x88"		=> "\x83\xd5",	//GREEK SMALL LETTER PSI
		"\xcf\x89"		=> "\x83\xd6",	//GREEK SMALL LETTER OMEGA
		"\xd0\x90"		=> "\x84\x40",	//CYRILLIC CAPITAL LETTER A
		"\xd0\x91"		=> "\x84\x41",	//CYRILLIC CAPITAL LETTER BE
		"\xd0\x92"		=> "\x84\x42",	//CYRILLIC CAPITAL LETTER VE
		"\xd0\x93"		=> "\x84\x43",	//CYRILLIC CAPITAL LETTER GHE
		"\xd0\x94"		=> "\x84\x44",	//CYRILLIC CAPITAL LETTER DE
		"\xd0\x95"		=> "\x84\x45",	//CYRILLIC CAPITAL LETTER IE
		"\xd0\x81"		=> "\x84\x46",	//CYRILLIC CAPITAL LETTER IO
		"\xd0\x96"		=> "\x84\x47",	//CYRILLIC CAPITAL LETTER ZHE
		"\xd0\x97"		=> "\x84\x48",	//CYRILLIC CAPITAL LETTER ZE
		"\xd0\x98"		=> "\x84\x49",	//CYRILLIC CAPITAL LETTER I
		"\xd0\x99"		=> "\x84\x4a",	//CYRILLIC CAPITAL LETTER SHORT I
		"\xd0\x9a"		=> "\x84\x4b",	//CYRILLIC CAPITAL LETTER KA
		"\xd0\x9b"		=> "\x84\x4c",	//CYRILLIC CAPITAL LETTER EL
		"\xd0\x9c"		=> "\x84\x4d",	//CYRILLIC CAPITAL LETTER EM
		"\xd0\x9d"		=> "\x84\x4e",	//CYRILLIC CAPITAL LETTER EN
		"\xd0\x9e"		=> "\x84\x4f",	//CYRILLIC CAPITAL LETTER O
		"\xd0\x9f"		=> "\x84\x50",	//CYRILLIC CAPITAL LETTER PE
		"\xd0\xa0"		=> "\x84\x51",	//CYRILLIC CAPITAL LETTER ER
		"\xd0\xa1"		=> "\x84\x52",	//CYRILLIC CAPITAL LETTER ES
		"\xd0\xa2"		=> "\x84\x53",	//CYRILLIC CAPITAL LETTER TE
		"\xd0\xa3"		=> "\x84\x54",	//CYRILLIC CAPITAL LETTER U
		"\xd0\xa4"		=> "\x84\x55",	//CYRILLIC CAPITAL LETTER EF
		"\xd0\xa5"		=> "\x84\x56",	//CYRILLIC CAPITAL LETTER HA
		"\xd0\xa6"		=> "\x84\x57",	//CYRILLIC CAPITAL LETTER TSE
		"\xd0\xa7"		=> "\x84\x58",	//CYRILLIC CAPITAL LETTER CHE
		"\xd0\xa8"		=> "\x84\x59",	//CYRILLIC CAPITAL LETTER SHA
		"\xd0\xa9"		=> "\x84\x5a",	//CYRILLIC CAPITAL LETTER SHCHA
		"\xd0\xaa"		=> "\x84\x5b",	//CYRILLIC CAPITAL LETTER HARD SIGN
		"\xd0\xab"		=> "\x84\x5c",	//CYRILLIC CAPITAL LETTER YERU
		"\xd0\xac"		=> "\x84\x5d",	//CYRILLIC CAPITAL LETTER SOFT SIGN
		"\xd0\xad"		=> "\x84\x5e",	//CYRILLIC CAPITAL LETTER E
		"\xd0\xae"		=> "\x84\x5f",	//CYRILLIC CAPITAL LETTER YU
		"\xd0\xaf"		=> "\x84\x60",	//CYRILLIC CAPITAL LETTER YA
		"\xd0\xb0"		=> "\x84\x70",	//CYRILLIC SMALL LETTER A
		"\xd0\xb1"		=> "\x84\x71",	//CYRILLIC SMALL LETTER BE
		"\xd0\xb2"		=> "\x84\x72",	//CYRILLIC SMALL LETTER VE
		"\xd0\xb3"		=> "\x84\x73",	//CYRILLIC SMALL LETTER GHE
		"\xd0\xb4"		=> "\x84\x74",	//CYRILLIC SMALL LETTER DE
		"\xd0\xb5"		=> "\x84\x75",	//CYRILLIC SMALL LETTER IE
		"\xd1\x91"		=> "\x84\x76",	//CYRILLIC SMALL LETTER IO
		"\xd0\xb6"		=> "\x84\x77",	//CYRILLIC SMALL LETTER ZHE
		"\xd0\xb7"		=> "\x84\x78",	//CYRILLIC SMALL LETTER ZE
		"\xd0\xb8"		=> "\x84\x79",	//CYRILLIC SMALL LETTER I
		"\xd0\xb9"		=> "\x84\x7a",	//CYRILLIC SMALL LETTER SHORT I
		"\xd0\xba"		=> "\x84\x7b",	//CYRILLIC SMALL LETTER KA
		"\xd0\xbb"		=> "\x84\x7c",	//CYRILLIC SMALL LETTER EL
		"\xd0\xbc"		=> "\x84\x7d",	//CYRILLIC SMALL LETTER EM
		"\xd0\xbd"		=> "\x84\x7e",	//CYRILLIC SMALL LETTER EN
		"\xd0\xbe"		=> "\x84\x80",	//CYRILLIC SMALL LETTER O
		"\xd0\xbf"		=> "\x84\x81",	//CYRILLIC SMALL LETTER PE
		"\xd1\x80"		=> "\x84\x82",	//CYRILLIC SMALL LETTER ER
		"\xd1\x81"		=> "\x84\x83",	//CYRILLIC SMALL LETTER ES
		"\xd1\x82"		=> "\x84\x84",	//CYRILLIC SMALL LETTER TE
		"\xd1\x83"		=> "\x84\x85",	//CYRILLIC SMALL LETTER U
		"\xd1\x84"		=> "\x84\x86",	//CYRILLIC SMALL LETTER EF
		"\xd1\x85"		=> "\x84\x87",	//CYRILLIC SMALL LETTER HA
		"\xd1\x86"		=> "\x84\x88",	//CYRILLIC SMALL LETTER TSE
		"\xd1\x87"		=> "\x84\x89",	//CYRILLIC SMALL LETTER CHE
		"\xd1\x88"		=> "\x84\x8a",	//CYRILLIC SMALL LETTER SHA
		"\xd1\x89"		=> "\x84\x8b",	//CYRILLIC SMALL LETTER SHCHA
		"\xd1\x8a"		=> "\x84\x8c",	//CYRILLIC SMALL LETTER HARD SIGN
		"\xd1\x8b"		=> "\x84\x8d",	//CYRILLIC SMALL LETTER YERU
		"\xd1\x8c"		=> "\x84\x8e",	//CYRILLIC SMALL LETTER SOFT SIGN
		"\xd1\x8d"		=> "\x84\x8f",	//CYRILLIC SMALL LETTER E
		"\xd1\x8e"		=> "\x84\x90",	//CYRILLIC SMALL LETTER YU
		"\xd1\x8f"		=> "\x84\x91",	//CYRILLIC SMALL LETTER YA
		"\xe2\x94\x80"		=> "\x84\x9f",	//BOX DRAWINGS LIGHT HORIZONTAL
		"\xe2\x94\x82"		=> "\x84\xa0",	//BOX DRAWINGS LIGHT VERTICAL
		"\xe2\x94\x8c"		=> "\x84\xa1",	//BOX DRAWINGS LIGHT DOWN AND RIGHT
		"\xe2\x94\x90"		=> "\x84\xa2",	//BOX DRAWINGS LIGHT DOWN AND LEFT
		"\xe2\x94\x98"		=> "\x84\xa3",	//BOX DRAWINGS LIGHT UP AND LEFT
		"\xe2\x94\x94"		=> "\x84\xa4",	//BOX DRAWINGS LIGHT UP AND RIGHT
		"\xe2\x94\x9c"		=> "\x84\xa5",	//BOX DRAWINGS LIGHT VERTICAL AND RIGHT
		"\xe2\x94\xac"		=> "\x84\xa6",	//BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
		"\xe2\x94\xa4"		=> "\x84\xa7",	//BOX DRAWINGS LIGHT VERTICAL AND LEFT
		"\xe2\x94\xb4"		=> "\x84\xa8",	//BOX DRAWINGS LIGHT UP AND HORIZONTAL
		"\xe2\x94\xbc"		=> "\x84\xa9",	//BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
		"\xe2\x94\x81"		=> "\x84\xaa",	//BOX DRAWINGS HEAVY HORIZONTAL
		"\xe2\x94\x83"		=> "\x84\xab",	//BOX DRAWINGS HEAVY VERTICAL
		"\xe2\x94\x8f"		=> "\x84\xac",	//BOX DRAWINGS HEAVY DOWN AND RIGHT
		"\xe2\x94\x93"		=> "\x84\xad",	//BOX DRAWINGS HEAVY DOWN AND LEFT
		"\xe2\x94\x9b"		=> "\x84\xae",	//BOX DRAWINGS HEAVY UP AND LEFT
		"\xe2\x94\x97"		=> "\x84\xaf",	//BOX DRAWINGS HEAVY UP AND RIGHT
		"\xe2\x94\xa3"		=> "\x84\xb0",	//BOX DRAWINGS HEAVY VERTICAL AND RIGHT
		"\xe2\x94\xb3"		=> "\x84\xb1",	//BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
		"\xe2\x94\xab"		=> "\x84\xb2",	//BOX DRAWINGS HEAVY VERTICAL AND LEFT
		"\xe2\x94\xbb"		=> "\x84\xb3",	//BOX DRAWINGS HEAVY UP AND HORIZONTAL
		"\xe2\x95\x8b"		=> "\x84\xb4",	//BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
		"\xe2\x94\xa0"		=> "\x84\xb5",	//BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
		"\xe2\x94\xaf"		=> "\x84\xb6",	//BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
		"\xe2\x94\xa8"		=> "\x84\xb7",	//BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
		"\xe2\x94\xb7"		=> "\x84\xb8",	//BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
		"\xe2\x94\xbf"		=> "\x84\xb9",	//BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
		"\xe2\x94\x9d"		=> "\x84\xba",	//BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
		"\xe2\x94\xb0"		=> "\x84\xbb",	//BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
		"\xe2\x94\xa5"		=> "\x84\xbc",	//BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
		"\xe2\x94\xb8"		=> "\x84\xbd",	//BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
		"\xe2\x95\x82"		=> "\x84\xbe",	//BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
		"\xe2\x91\xa0"		=> "\x87\x40",	//CIRCLED DIGIT ONE
		"\xe2\x91\xa1"		=> "\x87\x41",	//CIRCLED DIGIT TWO
		"\xe2\x91\xa2"		=> "\x87\x42",	//CIRCLED DIGIT THREE
		"\xe2\x91\xa3"		=> "\x87\x43",	//CIRCLED DIGIT FOUR
		"\xe2\x91\xa4"		=> "\x87\x44",	//CIRCLED DIGIT FIVE
		"\xe2\x91\xa5"		=> "\x87\x45",	//CIRCLED DIGIT SIX
		"\xe2\x91\xa6"		=> "\x87\x46",	//CIRCLED DIGIT SEVEN
		"\xe2\x91\xa7"		=> "\x87\x47",	//CIRCLED DIGIT EIGHT
		"\xe2\x91\xa8"		=> "\x87\x48",	//CIRCLED DIGIT NINE
		"\xe2\x91\xa9"		=> "\x87\x49",	//CIRCLED NUMBER TEN
		"\xe2\x91\xaa"		=> "\x87\x4a",	//CIRCLED NUMBER ELEVEN
		"\xe2\x91\xab"		=> "\x87\x4b",	//CIRCLED NUMBER TWELVE
		"\xe2\x91\xac"		=> "\x87\x4c",	//CIRCLED NUMBER THIRTEEN
		"\xe2\x91\xad"		=> "\x87\x4d",	//CIRCLED NUMBER FOURTEEN
		"\xe2\x91\xae"		=> "\x87\x4e",	//CIRCLED NUMBER FIFTEEN
		"\xe2\x91\xaf"		=> "\x87\x4f",	//CIRCLED NUMBER SIXTEEN
		"\xe2\x91\xb0"		=> "\x87\x50",	//CIRCLED NUMBER SEVENTEEN
		"\xe2\x91\xb1"		=> "\x87\x51",	//CIRCLED NUMBER EIGHTEEN
		"\xe2\x91\xb2"		=> "\x87\x52",	//CIRCLED NUMBER NINETEEN
		"\xe2\x91\xb3"		=> "\x87\x53",	//CIRCLED NUMBER TWENTY
		"\xe2\x85\xa0"		=> "\x87\x54",	//ROMAN NUMERAL ONE
		"\xe2\x85\xa1"		=> "\x87\x55",	//ROMAN NUMERAL TWO
		"\xe2\x85\xa2"		=> "\x87\x56",	//ROMAN NUMERAL THREE
		"\xe2\x85\xa3"		=> "\x87\x57",	//ROMAN NUMERAL FOUR
		"\xe2\x85\xa4"		=> "\x87\x58",	//ROMAN NUMERAL FIVE
		"\xe2\x85\xa5"		=> "\x87\x59",	//ROMAN NUMERAL SIX
		"\xe2\x85\xa6"		=> "\x87\x5a",	//ROMAN NUMERAL SEVEN
		"\xe2\x85\xa7"		=> "\x87\x5b",	//ROMAN NUMERAL EIGHT
		"\xe2\x85\xa8"		=> "\x87\x5c",	//ROMAN NUMERAL NINE
		"\xe2\x85\xa9"		=> "\x87\x5d",	//ROMAN NUMERAL TEN
		"\xe3\x8d\x89"		=> "\x87\x5f",	//SQUARE MIRI
		"\xe3\x8c\x94"		=> "\x87\x60",	//SQUARE KIRO
		"\xe3\x8c\xa2"		=> "\x87\x61",	//SQUARE SENTI
		"\xe3\x8d\x8d"		=> "\x87\x62",	//SQUARE MEETORU
		"\xe3\x8c\x98"		=> "\x87\x63",	//SQUARE GURAMU
		"\xe3\x8c\xa7"		=> "\x87\x64",	//SQUARE TON
		"\xe3\x8c\x83"		=> "\x87\x65",	//SQUARE AARU
		"\xe3\x8c\xb6"		=> "\x87\x66",	//SQUARE HEKUTAARU
		"\xe3\x8d\x91"		=> "\x87\x67",	//SQUARE RITTORU
		"\xe3\x8d\x97"		=> "\x87\x68",	//SQUARE WATTO
		"\xe3\x8c\x8d"		=> "\x87\x69",	//SQUARE KARORII
		"\xe3\x8c\xa6"		=> "\x87\x6a",	//SQUARE DORU
		"\xe3\x8c\xa3"		=> "\x87\x6b",	//SQUARE SENTO
		"\xe3\x8c\xab"		=> "\x87\x6c",	//SQUARE PAASENTO
		"\xe3\x8d\x8a"		=> "\x87\x6d",	//SQUARE MIRIBAARU
		"\xe3\x8c\xbb"		=> "\x87\x6e",	//SQUARE PEEZI
		"\xe3\x8e\x9c"		=> "\x87\x6f",	//SQUARE MM
		"\xe3\x8e\x9d"		=> "\x87\x70",	//SQUARE CM
		"\xe3\x8e\x9e"		=> "\x87\x71",	//SQUARE KM
		"\xe3\x8e\x8e"		=> "\x87\x72",	//SQUARE MG
		"\xe3\x8e\x8f"		=> "\x87\x73",	//SQUARE KG
		"\xe3\x8f\x84"		=> "\x87\x74",	//SQUARE CC
		"\xe3\x8e\xa1"		=> "\x87\x75",	//SQUARE M SQUARED
		"\xe3\x8d\xbb"		=> "\x87\x7e",	//SQUARE ERA NAME HEISEI
		"\xe3\x80\x9d"		=> "\x87\x80",	//REVERSED DOUBLE PRIME QUOTATION MARK
		"\xe3\x80\x9f"		=> "\x87\x81",	//LOW DOUBLE PRIME QUOTATION MARK
		"\xe2\x84\x96"		=> "\x87\x82",	//NUMERO SIGN
		"\xe3\x8f\x8d"		=> "\x87\x83",	//SQUARE KK
		"\xe2\x84\xa1"		=> "\x87\x84",	//TELEPHONE SIGN
		"\xe3\x8a\xa4"		=> "\x87\x85",	//CIRCLED IDEOGRAPH HIGH
		"\xe3\x8a\xa5"		=> "\x87\x86",	//CIRCLED IDEOGRAPH CENTRE
		"\xe3\x8a\xa6"		=> "\x87\x87",	//CIRCLED IDEOGRAPH LOW
		"\xe3\x8a\xa7"		=> "\x87\x88",	//CIRCLED IDEOGRAPH LEFT
		"\xe3\x8a\xa8"		=> "\x87\x89",	//CIRCLED IDEOGRAPH RIGHT
		"\xe3\x88\xb1"		=> "\x87\x8a",	//PARENTHESIZED IDEOGRAPH STOCK
		"\xe3\x88\xb2"		=> "\x87\x8b",	//PARENTHESIZED IDEOGRAPH HAVE
		"\xe3\x88\xb9"		=> "\x87\x8c",	//PARENTHESIZED IDEOGRAPH REPRESENT
		"\xe3\x8d\xbe"		=> "\x87\x8d",	//SQUARE ERA NAME MEIZI
		"\xe3\x8d\xbd"		=> "\x87\x8e",	//SQUARE ERA NAME TAISYOU
		"\xe3\x8d\xbc"		=> "\x87\x8f",	//SQUARE ERA NAME SYOUWA
		"\xe2\x89\x92"		=> "\x87\x90",	//APPROXIMATELY EQUAL TO OR THE IMAGE OF
		"\xe2\x89\xa1"		=> "\x87\x91",	//IDENTICAL TO
		"\xe2\x88\xab"		=> "\x87\x92",	//INTEGRAL
		"\xe2\x88\xae"		=> "\x87\x93",	//CONTOUR INTEGRAL
		"\xe2\x88\x91"		=> "\x87\x94",	//N-ARY SUMMATION
		"\xe2\x88\x9a"		=> "\x87\x95",	//SQUARE ROOT
		"\xe2\x8a\xa5"		=> "\x87\x96",	//UP TACK
		"\xe2\x88\xa0"		=> "\x87\x97",	//ANGLE
		"\xe2\x88\x9f"		=> "\x87\x98",	//RIGHT ANGLE
		"\xe2\x8a\xbf"		=> "\x87\x99",	//RIGHT TRIANGLE
		"\xe2\x88\xb5"		=> "\x87\x9a",	//BECAUSE
		"\xe2\x88\xa9"		=> "\x87\x9b",	//INTERSECTION
		"\xe2\x88\xaa"		=> "\x87\x9c",	//UNION
		"\xe4\xba\x9c"		=> "\x88\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\x96"		=> "\x88\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\x83"		=> "\x88\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\xbf"		=> "\x88\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\x80"		=> "\x88\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x9b"		=> "\x88\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\xa8"		=> "\x88\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\xb6"		=> "\x88\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xa2"		=> "\x88\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xb5"		=> "\x88\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x9c"		=> "\x88\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x90"		=> "\x88\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xaa"		=> "\x88\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xa1"		=> "\x88\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa5"		=> "\x88\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xad"		=> "\x88\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xa6"		=> "\x88\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xa6"		=> "\x88\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xb5"		=> "\x88\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x93"		=> "\x88\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\xa7"		=> "\x88\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xa1"		=> "\x88\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xb1"		=> "\x88\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x9b"		=> "\x88\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x90"		=> "\x88\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\xbb"		=> "\x88\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xb4"		=> "\x88\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xa2"		=> "\x88\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xbe"		=> "\x88\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x8e"		=> "\x88\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x96"		=> "\x88\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x9f"		=> "\x88\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xb7"		=> "\x88\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x89"		=> "\x88\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xb5"		=> "\x88\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x89"		=> "\x88\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x97"		=> "\x88\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x88"		=> "\x88\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x87"		=> "\x88\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x8d"		=> "\x88\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x8f"		=> "\x88\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xa5"		=> "\x88\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x8a"		=> "\x88\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x8d"		=> "\x88\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x9d"		=> "\x88\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x89"		=> "\x88\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xb2"		=> "\x88\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xb7"		=> "\x88\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x94"		=> "\x88\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\x81"		=> "\x88\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x89"		=> "\x88\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x9f"		=> "\x88\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x8f"		=> "\x88\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xb0"		=> "\x88\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x93"		=> "\x88\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x85"		=> "\x88\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xba"		=> "\x88\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x8f"		=> "\x88\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xb0"		=> "\x88\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xbb"		=> "\x88\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xad"		=> "\x88\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xaf"		=> "\x88\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x83"		=> "\x88\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\x8e"		=> "\x88\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xa3"		=> "\x88\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x82"		=> "\x88\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x95"		=> "\x88\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xba"		=> "\x88\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xbb"		=> "\x88\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x95"		=> "\x88\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xa5"		=> "\x88\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x9f"		=> "\x88\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xb2"		=> "\x88\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\x81"		=> "\x88\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\xaf"		=> "\x88\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x80"		=> "\x88\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xb1"		=> "\x88\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xa2"		=> "\x88\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xb8"		=> "\x88\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xb2"		=> "\x88\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xa8"		=> "\x88\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\x8b"		=> "\x88\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xaf"		=> "\x88\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x81"		=> "\x88\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb0"		=> "\x88\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xbd"		=> "\x88\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xa1"		=> "\x88\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xa0"		=> "\x88\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\xbb"		=> "\x88\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x95"		=> "\x88\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xb2"		=> "\x88\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xab"		=> "\x88\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xa4"		=> "\x88\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\xad"		=> "\x88\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xa2"		=> "\x89\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xb0"		=> "\x89\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xa0"		=> "\x89\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\xbb"		=> "\x89\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x8b"		=> "\x89\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xb3"		=> "\x89\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x87"		=> "\x89\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\x8f"		=> "\x89\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xbd"		=> "\x89\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\x82"		=> "\x89\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xa8"		=> "\x89\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xaf"		=> "\x89\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x9c"		=> "\x89\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\xba"		=> "\x89\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x91"		=> "\x89\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x93"		=> "\x89\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xbc"		=> "\x89\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa6"		=> "\x89\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\x98"		=> "\x89\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\x84"		=> "\x89\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\x9d"		=> "\x89\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x9a"		=> "\x89\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xbb"		=> "\x89\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\xa5"		=> "\x89\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xa9"		=> "\x89\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xa6"		=> "\x89\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\x9c"		=> "\x89\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x8f"		=> "\x89\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\x82"		=> "\x89\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x91"		=> "\x89\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x8b"		=> "\x89\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xb2"		=> "\x89\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\x8f"		=> "\x89\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x8c"		=> "\x89\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xa1"		=> "\x89\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xb6"		=> "\x89\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\xb0"		=> "\x89\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xb1"		=> "\x89\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xa0"		=> "\x89\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xb3"		=> "\x89\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x84"		=> "\x89\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\xb8"		=> "\x89\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xb3"		=> "\x89\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xa9"		=> "\x89\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\x9b"		=> "\x89\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x88"		=> "\x89\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x8e"		=> "\x89\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xb4"		=> "\x89\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xb1"		=> "\x89\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x9b"		=> "\x89\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xa0"		=> "\x89\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xad"		=> "\x89\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\xb2"		=> "\x89\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xab"		=> "\x89\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x8a"		=> "\x89\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x85"		=> "\x89\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xa6"		=> "\x89\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x81"		=> "\x89\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\x8a"		=> "\x89\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xb2"		=> "\x89\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x8e"		=> "\x89\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xad"		=> "\x89\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x86"		=> "\x89\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x92"		=> "\x89\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xb0"		=> "\x89\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x84"		=> "\x89\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xb4"		=> "\x89\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xb6"		=> "\x89\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xa8"		=> "\x89\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xa9"		=> "\x89\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xb4"		=> "\x89\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xbf"		=> "\x89\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x94"		=> "\x89\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\x8e"		=> "\x89\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x94"		=> "\x89\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x99"		=> "\x89\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x95"		=> "\x89\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xbf"		=> "\x89\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x81"		=> "\x89\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xb6"		=> "\x89\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x91"		=> "\x89\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x97"		=> "\x89\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xa0"		=> "\x89\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x9b"		=> "\x89\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x9b"		=> "\x89\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\xa9"		=> "\x89\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xbc"		=> "\x89\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x9a"		=> "\x89\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xa5"		=> "\x89\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xb9"		=> "\x89\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xae"		=> "\x89\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xa5"		=> "\x89\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x80"		=> "\x89\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x9c"		=> "\x89\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xbc"		=> "\x89\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xba"		=> "\x89\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xaa"		=> "\x89\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xa7"		=> "\x89\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xb4"		=> "\x89\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\x8b"		=> "\x89\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\x81"		=> "\x89\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\x96"		=> "\x89\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xac"		=> "\x89\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x8e"		=> "\x89\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x84"		=> "\x89\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xa1"		=> "\x89\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x96"		=> "\x89\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xbb"		=> "\x89\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x84"		=> "\x89\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x8b"		=> "\x89\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xb6"		=> "\x89\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x86"		=> "\x89\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\xb6"		=> "\x89\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xa1"		=> "\x89\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x99"		=> "\x89\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xba"		=> "\x89\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb8"		=> "\x89\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xa9"		=> "\x89\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa9"		=> "\x89\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x8f"		=> "\x89\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\xb3"		=> "\x89\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x8b"		=> "\x89\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x96"		=> "\x89\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xae"		=> "\x89\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x95"		=> "\x89\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xbd"		=> "\x89\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xa1"		=> "\x89\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xb3"		=> "\x89\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xa0"		=> "\x89\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xaf"		=> "\x89\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\x89"		=> "\x89\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x8f"		=> "\x89\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\x81"		=> "\x89\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xb6"		=> "\x89\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xa1"		=> "\x89\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x91"		=> "\x89\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x87"		=> "\x89\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x9c"		=> "\x89\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xb6"		=> "\x89\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x8c"		=> "\x89\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xb3"		=> "\x89\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\xab"		=> "\x89\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x82"		=> "\x89\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x8d"		=> "\x89\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xbe"		=> "\x89\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xbc"		=> "\x89\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x87"		=> "\x89\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xb1"		=> "\x89\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x9b"		=> "\x89\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x84"		=> "\x89\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xb7"		=> "\x89\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xaf"		=> "\x89\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x93"		=> "\x89\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xa6"		=> "\x89\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xb2"		=> "\x89\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\xa9"		=> "\x89\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xa8"		=> "\x89\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xa6"		=> "\x89\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x8e"		=> "\x89\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x9e"		=> "\x89\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\x8a"		=> "\x89\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x84"		=> "\x89\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xa8"		=> "\x89\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x91"		=> "\x89\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x99"		=> "\x89\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xbb"		=> "\x89\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xa5"		=> "\x89\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xbd"		=> "\x89\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xbe"		=> "\x89\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x80"		=> "\x89\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x85"		=> "\x89\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x93"		=> "\x89\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x95"		=> "\x89\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x8b"		=> "\x89\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x9a"		=> "\x89\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\xa3"		=> "\x89\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x9e"		=> "\x89\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x8a"		=> "\x89\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x8a"		=> "\x89\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xbb"		=> "\x89\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xab"		=> "\x89\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xaa"		=> "\x89\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x94"		=> "\x89\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xa2"		=> "\x89\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x90"		=> "\x89\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x92"		=> "\x89\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x90"		=> "\x89\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xb9"		=> "\x89\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x81"		=> "\x8a\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xa6"		=> "\x8a\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xb0"		=> "\x8a\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xb7"		=> "\x8a\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\xb0"		=> "\x8a\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x8c"		=> "\x8a\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x86"		=> "\x8a\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xb5"		=> "\x8a\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xa5"		=> "\x8a\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\xb9"		=> "\x8a\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x8b"		=> "\x8a\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x8e"		=> "\x8a\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x9d"		=> "\x8a\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xb1"		=> "\x8a\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xbe"		=> "\x8a\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x96"		=> "\x8a\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xb3"		=> "\x8a\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xb3"		=> "\x8a\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x96"		=> "\x8a\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xa8"		=> "\x8a\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x82"		=> "\x8a\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\xaf"		=> "\x8a\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x8d"		=> "\x8a\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x8b"		=> "\x8a\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x97"		=> "\x8a\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xb2"		=> "\x8a\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\xa7"		=> "\x8a\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaa\xb8"		=> "\x8a\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xac"		=> "\x8a\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\xa8"		=> "\x8a\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x99"		=> "\x8a\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xa3"		=> "\x8a\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xbf"		=> "\x8a\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x8e"		=> "\x8a\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x8e"		=> "\x8a\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x83"		=> "\x8a\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\x87"		=> "\x8a\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x84"		=> "\x8a\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x93"		=> "\x8a\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xa1"		=> "\x8a\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xb9"		=> "\x8a\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xbc"		=> "\x8a\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xb8"		=> "\x8a\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xbb"		=> "\x8a\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xb2"		=> "\x8a\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xba"		=> "\x8a\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xab"		=> "\x8a\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x9a"		=> "\x8a\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\x92"		=> "\x8a\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xab"		=> "\x8a\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x83"		=> "\x8a\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\xad"		=> "\x8a\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xa3"		=> "\x8a\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x94"		=> "\x8a\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xa9"		=> "\x8a\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xa6"		=> "\x8a\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xb3"		=> "\x8a\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xbd"		=> "\x8a\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x8d"		=> "\x8a\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x8e"		=> "\x8a\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x9b"		=> "\x8a\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xa0"		=> "\x8a\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xab"		=> "\x8a\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xbf"		=> "\x8a\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xb6"		=> "\x8a\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x8d"		=> "\x8a\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\x9f"		=> "\x8a\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xb2"		=> "\x8a\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x9d"		=> "\x8a\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xb0"		=> "\x8a\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xac"		=> "\x8a\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xbb"		=> "\x8a\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x87"		=> "\x8a\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x91"		=> "\x8a\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\x9b"		=> "\x8a\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x90"		=> "\x8a\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x84"		=> "\x8a\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x94"		=> "\x8a\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xb9"		=> "\x8a\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xb6"		=> "\x8a\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x9b"		=> "\x8a\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xba"		=> "\x8a\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x84"		=> "\x8a\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xaa"		=> "\x8a\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x9c"		=> "\x8a\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x83"		=> "\x8a\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xb2"		=> "\x8a\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x9c"		=> "\x8a\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\x8c"		=> "\x8a\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\x9b"		=> "\x8a\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xa8"		=> "\x8a\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xa2"		=> "\x8a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x85"		=> "\x8a\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xb1"		=> "\x8a\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xa5"		=> "\x8a\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x88"		=> "\x8a\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x85"		=> "\x8a\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xa6"		=> "\x8a\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\xbe"		=> "\x8a\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x83"		=> "\x8a\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xa0"		=> "\x8a\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x92"		=> "\x8a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x8a"		=> "\x8a\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x98"		=> "\x8a\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xa7"		=> "\x8a\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xbb"		=> "\x8a\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x9a"		=> "\x8a\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xaa"		=> "\x8a\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\xa6"		=> "\x8a\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x8c"		=> "\x8a\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x98"		=> "\x8a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x9b"		=> "\x8a\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xb2"		=> "\x8a\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xb9"		=> "\x8a\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xa3"		=> "\x8a\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x9f"		=> "\x8a\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xa3"		=> "\x8a\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xbe"		=> "\x8a\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x9b"		=> "\x8a\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xa2"		=> "\x8a\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x91"		=> "\x8a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x93"		=> "\x8a\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xba"		=> "\x8a\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xbe"		=> "\x8a\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x93"		=> "\x8a\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x97"		=> "\x8a\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xa2"		=> "\x8a\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x97"		=> "\x8a\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\x85"		=> "\x8a\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\xb0"		=> "\x8a\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x98"		=> "\x8a\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xa3"		=> "\x8a\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x8b"		=> "\x8a\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xbf"		=> "\x8a\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\xa1"		=> "\x8a\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xa1"		=> "\x8a\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xa9"		=> "\x8a\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbc\xb6"		=> "\x8a\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xb0"		=> "\x8a\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x9d"		=> "\x8a\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xa6"		=> "\x8a\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x9e"		=> "\x8a\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xb3"		=> "\x8a\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x8c"		=> "\x8a\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xab"		=> "\x8a\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x84"		=> "\x8a\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x91"		=> "\x8a\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x93"		=> "\x8a\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x91"		=> "\x8a\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xa2"		=> "\x8a\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xa5"		=> "\x8a\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\x93"		=> "\x8a\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\xa8"		=> "\x8a\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x98"		=> "\x8a\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xb8"		=> "\x8a\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xab"		=> "\x8a\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xb8"		=> "\x8a\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x8c"		=> "\x8a\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\xa9"		=> "\x8a\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x8c"		=> "\x8a\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xbc"		=> "\x8a\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xa9"		=> "\x8a\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xab"		=> "\x8a\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x8b"		=> "\x8a\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x81"		=> "\x8a\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x91"		=> "\x8a\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x94"		=> "\x8a\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x98"		=> "\x8a\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x81"		=> "\x8a\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x8e"		=> "\x8a\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb1"		=> "\x8a\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x9c"		=> "\x8a\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\xa8"		=> "\x8a\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\xba"		=> "\x8a\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x87"		=> "\x8a\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\x89"		=> "\x8a\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x84"		=> "\x8a\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\x90"		=> "\x8a\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x8c"		=> "\x8a\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xbe"		=> "\x8a\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x8c"		=> "\x8a\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xae"		=> "\x8a\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xba"		=> "\x8a\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x97"		=> "\x8a\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xa2"		=> "\x8a\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x9f"		=> "\x8a\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x8b"		=> "\x8a\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x84"		=> "\x8a\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x9f"		=> "\x8b\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xb0"		=> "\x8b\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x85"		=> "\x8b\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\x97"		=> "\x8b\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xbd"		=> "\x8b\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xbf"		=> "\x8b\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x88"		=> "\x8b\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xa3"		=> "\x8b\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x80"		=> "\x8b\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x80"		=> "\x8b\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xbd"		=> "\x8b\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x8f"		=> "\x8b\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x98"		=> "\x8b\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xb4"		=> "\x8b\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xb7"		=> "\x8b\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x8c"		=> "\x8b\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x9d"		=> "\x8b\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xa2"		=> "\x8b\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\x8e"		=> "\x8b\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xbc"		=> "\x8b\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x80"		=> "\x8b\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xbd"		=> "\x8b\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x80"		=> "\x8b\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x93"		=> "\x8b\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x9c"		=> "\x8b\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xaf"		=> "\x8b\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x80"		=> "\x8b\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xac"		=> "\x8b\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xba"		=> "\x8b\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xa0"		=> "\x8b\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x91"		=> "\x8b\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x87"		=> "\x8b\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xa9"		=> "\x8b\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\xbb"		=> "\x8b\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xbc"		=> "\x8b\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xb0"		=> "\x8b\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xac"		=> "\x8b\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x8a"		=> "\x8b\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\xa0"		=> "\x8b\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x89"		=> "\x8b\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x83"		=> "\x8b\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xab"		=> "\x8b\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x94"		=> "\x8b\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x98"		=> "\x8b\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xb0"		=> "\x8b\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xa7"		=> "\x8b\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xb5"		=> "\x8b\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x8d"		=> "\x8b\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb4"		=> "\x8b\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xa2"		=> "\x8b\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x9a"		=> "\x8b\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x90"		=> "\x8b\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x86"		=> "\x8b\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x98"		=> "\x8b\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x85"		=> "\x8b\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x87"		=> "\x8b\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x91"		=> "\x8b\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x8a"		=> "\x8b\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xb8"		=> "\x8b\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xae"		=> "\x8b\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x93"		=> "\x8b\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xa5"		=> "\x8b\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x91"		=> "\x8b\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xbd"		=> "\x8b\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x82"		=> "\x8b\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xb2"		=> "\x8b\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xa3"		=> "\x8b\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\xb8"		=> "\x8b\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\x83"		=> "\x8b\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xb6"		=> "\x8b\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\xae"		=> "\x8b\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x88"		=> "\x8b\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x9a"		=> "\x8b\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xbe"		=> "\x8b\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xa6"		=> "\x8b\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xa7"		=> "\x8b\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x9b"		=> "\x8b\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xbb"		=> "\x8b\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x85"		=> "\x8b\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xa8"		=> "\x8b\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x92"		=> "\x8b\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xa0"		=> "\x8b\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x99"		=> "\x8b\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa0"		=> "\x8b\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x9a"		=> "\x8b\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xb1"		=> "\x8b\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x9d"		=> "\x8b\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xb8"		=> "\x8b\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x81"		=> "\x8b\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xa6"		=> "\x8b\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x9a"		=> "\x8b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xa8"		=> "\x8b\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xab"		=> "\x8b\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xac"		=> "\x8b\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x9b"		=> "\x8b\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xa0"		=> "\x8b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x91"		=> "\x8b\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x87"		=> "\x8b\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xb6"		=> "\x8b\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xb1"		=> "\x8b\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xb6"		=> "\x8b\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x94"		=> "\x8b\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xa1"		=> "\x8b\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xbf"		=> "\x8b\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xab"		=> "\x8b\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xac"		=> "\x8b\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\x83"		=> "\x8b\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xa1"		=> "\x8b\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xb7"		=> "\x8b\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x8a"		=> "\x8b\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xaf"		=> "\x8b\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x90"		=> "\x8b\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xad"		=> "\x8b\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x9f"		=> "\x8b\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x99"		=> "\x8b\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x8b"		=> "\x8b\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x81"		=> "\x8b\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x82"		=> "\x8b\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xad"		=> "\x8b\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xaf"		=> "\x8b\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xb8"		=> "\x8b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x85"		=> "\x8b\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x88"		=> "\x8b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x8e"		=> "\x8b\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\xb7"		=> "\x8b\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\xa1"		=> "\x8b\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\xbf"		=> "\x8b\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x97"		=> "\x8b\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x9a"		=> "\x8b\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xb0"		=> "\x8b\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x9d"		=> "\x8b\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xad"		=> "\x8b\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x81"		=> "\x8b\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xad"		=> "\x8b\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x80"		=> "\x8b\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xb2"		=> "\x8b\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xb5"		=> "\x8b\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\x89"		=> "\x8b\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x90"		=> "\x8b\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x81"		=> "\x8b\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x85"		=> "\x8b\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xa4"		=> "\x8b\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x87"		=> "\x8b\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xbe"		=> "\x8b\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa6"		=> "\x8b\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xa4"		=> "\x8b\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xa3"		=> "\x8b\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xbd"		=> "\x8b\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xb4"		=> "\x8b\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x81"		=> "\x8b\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xbd"		=> "\x8b\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x8b"		=> "\x8b\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x8a"		=> "\x8b\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xb9"		=> "\x8b\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x8c"		=> "\x8b\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xbf"		=> "\x8b\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\x9f"		=> "\x8b\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xb9"		=> "\x8b\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\x91"		=> "\x8b\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x91"		=> "\x8b\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x9f"		=> "\x8b\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x80"		=> "\x8b\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x9d"		=> "\x8b\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xb6"		=> "\x8b\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xa5"		=> "\x8b\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xba"		=> "\x8b\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x97"		=> "\x8b\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\x96"		=> "\x8b\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xa9"		=> "\x8b\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xa6"		=> "\x8b\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\xaf"		=> "\x8b\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x86"		=> "\x8b\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x88"		=> "\x8b\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x92"		=> "\x8b\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xb7"		=> "\x8b\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x9a"		=> "\x8b\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x9e"		=> "\x8b\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xb0"		=> "\x8b\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xba"		=> "\x8b\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xb6"		=> "\x8b\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x93"		=> "\x8b\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x87"		=> "\x8b\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x85"		=> "\x8b\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xb2"		=> "\x8b\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x9b"		=> "\x8b\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xa7"		=> "\x8b\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x91"		=> "\x8b\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x88"		=> "\x8b\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x98"		=> "\x8c\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x9f"		=> "\x8c\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x93"		=> "\x8c\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xb4"		=> "\x8c\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\xa1"		=> "\x8c\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\xaa"		=> "\x8c\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\x8a"		=> "\x8c\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x88"		=> "\x8c\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x82"		=> "\x8c\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x97"		=> "\x8c\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xb0"		=> "\x8c\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x91"		=> "\x8c\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\xac"		=> "\x8c\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xb2"		=> "\x8c\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x9b"		=> "\x8c\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xab"		=> "\x8c\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x93"		=> "\x8c\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xa4"		=> "\x8c\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x8d"		=> "\x8c\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\xa1"		=> "\x8c\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xa6"		=> "\x8c\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x88"		=> "\x8c\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x81"		=> "\x8c\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x82"		=> "\x8c\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\xbe"		=> "\x8c\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x91"		=> "\x8c\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x84"		=> "\x8c\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x93"		=> "\x8c\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\xad"		=> "\x8c\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xaa"		=> "\x8c\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\x8b"		=> "\x8c\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x91"		=> "\x8c\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xa2"		=> "\x8c\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x84"		=> "\x8c\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xb5"		=> "\x8c\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xb6"		=> "\x8c\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xa7"		=> "\x8c\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xa9"		=> "\x8c\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xb2"		=> "\x8c\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\xba"		=> "\x8c\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xac"		=> "\x8c\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xaf"		=> "\x8c\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x82"		=> "\x8c\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x93"		=> "\x8c\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xa6"		=> "\x8c\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xbd"		=> "\x8c\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xbb"		=> "\x8c\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x8c"		=> "\x8c\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\x99"		=> "\x8c\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x8b"		=> "\x8c\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xab"		=> "\x8c\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x8e"		=> "\x8c\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\x8a"		=> "\x8c\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x8d"		=> "\x8c\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x88"		=> "\x8c\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xa3"		=> "\x8c\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xa6"		=> "\x8c\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\xbd"		=> "\x8c\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x9a"		=> "\x8c\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\x8f"		=> "\x8c\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xb8"		=> "\x8c\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\x8e"		=> "\x8c\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xa8"		=> "\x8c\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x87"		=> "\x8c\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x9f"		=> "\x8c\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x83"		=> "\x8c\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x80"		=> "\x8c\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x99"		=> "\x8c\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x81"		=> "\x8c\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x91"		=> "\x8c\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xa0"		=> "\x8c\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xba"		=> "\x8c\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\x94"		=> "\x8c\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xb4"		=> "\x8c\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x90"		=> "\x8c\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x80"		=> "\x8c\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xa3"		=> "\x8c\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x88"		=> "\x8c\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xb6"		=> "\x8c\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xb9"		=> "\x8c\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa6"		=> "\x8c\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xa5"		=> "\x8c\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xbc"		=> "\x8c\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xb8"		=> "\x8c\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xa3"		=> "\x8c\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xa7"		=> "\x8c\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x8f"		=> "\x8c\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x85"		=> "\x8c\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\x8c"		=> "\x8c\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xba"		=> "\x8c\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xb2"		=> "\x8c\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xb8"		=> "\x8c\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xb3"		=> "\x8c\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xb2"		=> "\x8c\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x9c"		=> "\x8c\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xa9"		=> "\x8c\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xbd"		=> "\x8c\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xac"		=> "\x8c\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xae"		=> "\x8c\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\x94"		=> "\x8c\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xaf"		=> "\x8c\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xb9"		=> "\x8c\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x8c"		=> "\x8c\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xa9"		=> "\x8c\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x8b"		=> "\x8c\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x99"		=> "\x8c\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xa2"		=> "\x8c\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x92"		=> "\x8c\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xa3"		=> "\x8c\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\xb5"		=> "\x8c\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xba"		=> "\x8c\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x95"		=> "\x8c\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\x93"		=> "\x8c\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb9\xb8"		=> "\x8c\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x83"		=> "\x8c\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\x9f"		=> "\x8c\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xb3"		=> "\x8c\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xbb"		=> "\x8c\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xa6"		=> "\x8c\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x9b"		=> "\x8c\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x90"		=> "\x8c\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\x84"		=> "\x8c\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xbe"		=> "\x8c\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x83"		=> "\x8c\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xb7"		=> "\x8c\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x80"		=> "\x8c\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xba"		=> "\x8c\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x90"		=> "\x8c\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x8e"		=> "\x8c\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x8b"		=> "\x8c\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xa4"		=> "\x8c\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xbc"		=> "\x8c\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xba"		=> "\x8c\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x91"		=> "\x8c\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xa4"		=> "\x8c\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xb1"		=> "\x8c\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xab"		=> "\x8c\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xa7"		=> "\x8c\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xb8"		=> "\x8c\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x85"		=> "\x8c\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xaf"		=> "\x8c\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x96"		=> "\x8c\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x90"		=> "\x8c\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x8a"		=> "\x8c\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xb4"		=> "\x8c\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xa1"		=> "\x8c\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xa1"		=> "\x8c\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xb0"		=> "\x8c\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x8e"		=> "\x8c\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x87"		=> "\x8c\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xa8"		=> "\x8c\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xb7"		=> "\x8c\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x87"		=> "\x8c\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xa7"		=> "\x8c\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\x93"		=> "\x8c\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x94"		=> "\x8c\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x92"		=> "\x8c\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x8d"		=> "\x8c\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x88"		=> "\x8c\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x89"		=> "\x8c\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xbe"		=> "\x8c\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\xaf"		=> "\x8c\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x8c"		=> "\x8c\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xa1"		=> "\x8c\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x9f"		=> "\x8c\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xa7"		=> "\x8c\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\x8e"		=> "\x8c\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\x9a"		=> "\x8c\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x81"		=> "\x8c\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x9e"		=> "\x8c\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa4"		=> "\x8c\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xb7"		=> "\x8c\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x90"		=> "\x8c\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x9e"		=> "\x8c\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x89"		=> "\x8c\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xa4"		=> "\x8c\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xbc"		=> "\x8c\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xaf"		=> "\x8c\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x99"		=> "\x8c\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x96"		=> "\x8c\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x89"		=> "\x8c\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xac"		=> "\x8c\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x9f"		=> "\x8c\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xb9"		=> "\x8c\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xbe"		=> "\x8c\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\x9a"		=> "\x8c\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xa3"		=> "\x8c\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x91"		=> "\x8c\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x8e"		=> "\x8d\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x89"		=> "\x8d\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x91"		=> "\x8d\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xa2"		=> "\x8d\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xbd"		=> "\x8d\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x94"		=> "\x8d\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x9d"		=> "\x8d\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x8f"		=> "\x8d\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xa5"		=> "\x8d\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xa7"		=> "\x8d\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xb7"		=> "\x8d\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xb8"		=> "\x8d\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x83"		=> "\x8d\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x9a"		=> "\x8d\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xb7"		=> "\x8d\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x98"		=> "\x8d\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x92"		=> "\x8d\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x8c"		=> "\x8d\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x97"		=> "\x8d\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x98"		=> "\x8d\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xa7"		=> "\x8d\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xbb"		=> "\x8d\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x82"		=> "\x8d\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x83"		=> "\x8d\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xb4"		=> "\x8d\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xad"		=> "\x8d\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xa1"		=> "\x8d\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x97"		=> "\x8d\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x8b"		=> "\x8d\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x9f"		=> "\x8d\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xaa"		=> "\x8d\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xa9"		=> "\x8d\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xaf"		=> "\x8d\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x9d"		=> "\x8d\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xb2"		=> "\x8d\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x87"		=> "\x8d\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xac"		=> "\x8d\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xbf"		=> "\x8d\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xa0"		=> "\x8d\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x85"		=> "\x8d\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x98"		=> "\x8d\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x9e"		=> "\x8d\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xb1"		=> "\x8d\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x95"		=> "\x8d\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x83"		=> "\x8d\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xaf"		=> "\x8d\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xb1"		=> "\x8d\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x94"		=> "\x8d\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x8f"		=> "\x8d\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xaa"		=> "\x8d\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\x92"		=> "\x8d\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x8c"		=> "\x8d\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xa1"		=> "\x8d\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x9b"		=> "\x8d\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xa2"		=> "\x8d\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xbc"		=> "\x8d\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\x8a"		=> "\x8d\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xb5"		=> "\x8d\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xb1"		=> "\x8d\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xbf"		=> "\x8d\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xbc"		=> "\x8d\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xa4"		=> "\x8d\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x8d"		=> "\x8d\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x85"		=> "\x8d\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\x99"		=> "\x8d\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x98"		=> "\x8d\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xbb"		=> "\x8d\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x9b"		=> "\x8d\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xab"		=> "\x8d\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xb7"		=> "\x8d\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x88"		=> "\x8d\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x95"		=> "\x8d\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xb7"		=> "\x8d\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xa0"		=> "\x8d\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xaa"		=> "\x8d\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x9f"		=> "\x8d\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xb9"		=> "\x8d\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x8b"		=> "\x8d\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xbb"		=> "\x8d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x8a"		=> "\x8d\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xbd"		=> "\x8d\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x80"		=> "\x8d\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xb7"		=> "\x8d\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xa0"		=> "\x8d\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x92"		=> "\x8d\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\x84"		=> "\x8d\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x89"		=> "\x8d\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xb0"		=> "\x8d\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x91"		=> "\x8d\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xbd"		=> "\x8d\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x9a"		=> "\x8d\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaa\xa8"		=> "\x8d\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x9b"		=> "\x8d\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xbc"		=> "\x8d\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xa4"		=> "\x8d\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x83"		=> "\x8d\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x8a"		=> "\x8d\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xb0"		=> "\x8d\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\xa4"		=> "\x8d\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xbe"		=> "\x8d\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\x9a"		=> "\x8d\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xa8"		=> "\x8d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x87"		=> "\x8d\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x8f"		=> "\x8d\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x86"		=> "\x8d\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xb9"		=> "\x8d\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xb1"		=> "\x8d\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb7"		=> "\x8d\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x95"		=> "\x8d\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xba"		=> "\x8d\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xae"		=> "\x8d\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x82"		=> "\x8d\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x9b"		=> "\x8d\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x90"		=> "\x8d\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x89"		=> "\x8d\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\x86"		=> "\x8d\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\xaf"		=> "\x8d\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xa6"		=> "\x8d\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xae"		=> "\x8d\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xbb"		=> "\x8d\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x99"		=> "\x8d\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xb3"		=> "\x8d\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\x82"		=> "\x8d\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x90"		=> "\x8d\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\x96"		=> "\x8d\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x9f"		=> "\x8d\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x90"		=> "\x8d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xa7"		=> "\x8d\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\xab"		=> "\x8d\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\xb5"		=> "\x8d\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\xac"		=> "\x8d\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x8d"		=> "\x8d\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x80"		=> "\x8d\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\x89"		=> "\x8d\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x9e"		=> "\x8d\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xbb"		=> "\x8d\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xb0"		=> "\x8d\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xa9"		=> "\x8d\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x8d"		=> "\x8d\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xa1"		=> "\x8d\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xbd"		=> "\x8d\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xb3"		=> "\x8d\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x88"		=> "\x8d\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\xbd"		=> "\x8d\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x87"		=> "\x8d\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\x80"		=> "\x8d\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\x95"		=> "\x8d\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xa6"		=> "\x8d\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\xad"		=> "\x8d\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x8e"		=> "\x8d\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xb0"		=> "\x8d\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x9c"		=> "\x8d\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x81"		=> "\x8d\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x89"		=> "\x8d\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x9b"		=> "\x8d\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xa4"		=> "\x8d\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\xa8"		=> "\x8d\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x90"		=> "\x8d\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xaa"		=> "\x8d\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xa1"		=> "\x8d\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xb4"		=> "\x8d\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x82"		=> "\x8d\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\xaa"		=> "\x8d\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xba"		=> "\x8d\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x8a"		=> "\x8d\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xb4"		=> "\x8d\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xb2"		=> "\x8d\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x8e"		=> "\x8d\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\xbc"		=> "\x8d\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x95"		=> "\x8d\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\xba"		=> "\x8d\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x9c"		=> "\x8d\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x8a"		=> "\x8d\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x8b"		=> "\x8d\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\xbe"		=> "\x8d\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xa8"		=> "\x8d\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x94"		=> "\x8d\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xb5"		=> "\x8d\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x84"		=> "\x8d\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x96"		=> "\x8d\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xa2"		=> "\x8d\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xaf"		=> "\x8d\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x9c"		=> "\x8d\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xad"		=> "\x8d\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xb9"		=> "\x8d\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x99"		=> "\x8d\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x8a"		=> "\x8d\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xb7"		=> "\x8d\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x9f"		=> "\x8e\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xb6"		=> "\x8e\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xae"		=> "\x8e\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xa6"		=> "\x8e\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xad"		=> "\x8e\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xba"		=> "\x8e\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xa9"		=> "\x8e\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x91"		=> "\x8e\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x90"		=> "\x8e\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x96"		=> "\x8e\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\x8c"		=> "\x8e\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x86"		=> "\x8e\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xab"		=> "\x8e\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xbf"		=> "\x8e\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x92"		=> "\x8e\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x89"		=> "\x8e\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x98"		=> "\x8e\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x82"		=> "\x8e\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xb1"		=> "\x8e\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xa8"		=> "\x8e\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x92"		=> "\x8e\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xa3"		=> "\x8e\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x9f"		=> "\x8e\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xa6"		=> "\x8e\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x8a"		=> "\x8e\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xa3"		=> "\x8e\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x97"		=> "\x8e\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x82"		=> "\x8e\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\x95"		=> "\x8e\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x83"		=> "\x8e\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x9b"		=> "\x8e\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xb8"		=> "\x8e\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x90"		=> "\x8e\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xac"		=> "\x8e\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xab"		=> "\x8e\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x8b"		=> "\x8e\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x95"		=> "\x8e\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x94"		=> "\x8e\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xba"		=> "\x8e\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xbf"		=> "\x8e\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xba"		=> "\x8e\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xb8"		=> "\x8e\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xb2"		=> "\x8e\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\xa3"		=> "\x8e\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x9b"		=> "\x8e\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xab"		=> "\x8e\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x8b"		=> "\x8e\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x89"		=> "\x8e\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\xbf"		=> "\x8e\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x90"		=> "\x8e\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x8d"		=> "\x8e\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x82"		=> "\x8e\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xab"		=> "\x8e\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x97"		=> "\x8e\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x9d"		=> "\x8e\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x87"		=> "\x8e\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xaf"		=> "\x8e\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x9c"		=> "\x8e\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xaf"		=> "\x8e\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xbd"		=> "\x8e\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xa8"		=> "\x8e\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x9d"		=> "\x8e\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xa2"		=> "\x8e\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xbb"		=> "\x8e\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\x8f"		=> "\x8e\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\x85"		=> "\x8e\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x89"		=> "\x8e\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x81"		=> "\x8e\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xb8"		=> "\x8e\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x99"		=> "\x8e\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xab"		=> "\x8e\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xa2"		=> "\x8e\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x82"		=> "\x8e\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xb3"		=> "\x8e\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x96"		=> "\x8e\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x9e"		=> "\x8e\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xa9"		=> "\x8e\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xa6"		=> "\x8e\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x8c"		=> "\x8e\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xae"		=> "\x8e\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x87"		=> "\x8e\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x9c"		=> "\x8e\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x8c"		=> "\x8e\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xbc"		=> "\x8e\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xaf"		=> "\x8e\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x8b"		=> "\x8e\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xbc"		=> "\x8e\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x8d"		=> "\x8e\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x90"		=> "\x8e\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x97"		=> "\x8e\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xba"		=> "\x8e\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x88"		=> "\x8e\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x81"		=> "\x8e\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x82"		=> "\x8e\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xa1"		=> "\x8e\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x8b"		=> "\x8e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xbb"		=> "\x8e\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xbe"		=> "\x8e\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\xbd"		=> "\x8e\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x94"		=> "\x8e\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x81"		=> "\x8e\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xba"		=> "\x8e\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x8c"		=> "\x8e\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\xb3"		=> "\x8e\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xaa"		=> "\x8e\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\x94"		=> "\x8e\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\x9e"		=> "\x8e\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x90"		=> "\x8e\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb9\xbf"		=> "\x8e\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x8f"		=> "\x8e\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x98"		=> "\x8e\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xab"		=> "\x8e\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xba"		=> "\x8e\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\xb8"		=> "\x8e\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x8d"		=> "\x8e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xab"		=> "\x8e\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x83"		=> "\x8e\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xb1"		=> "\x8e\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\xb7"		=> "\x8e\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xb1"		=> "\x8e\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\x89"		=> "\x8e\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xa4"		=> "\x8e\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x89"		=> "\x8e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\xbf"		=> "\x8e\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x86"		=> "\x8e\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xbe"		=> "\x8e\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xaa"		=> "\x8e\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x9f"		=> "\x8e\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x80"		=> "\x8e\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xa0"		=> "\x8e\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xb2"		=> "\x8e\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xb4"		=> "\x8e\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\x9d"		=> "\x8e\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xa1"		=> "\x8e\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x8a"		=> "\x8e\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x9e"		=> "\x8e\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x8e"		=> "\x8e\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x99"		=> "\x8e\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x84"		=> "\x8e\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xa8"		=> "\x8e\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xa6"		=> "\x8e\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x9c"		=> "\x8e\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xae"		=> "\x8e\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xbe"		=> "\x8e\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x97"		=> "\x8e\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x85"		=> "\x8e\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x9d"		=> "\x8e\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x8a"		=> "\x8e\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xae"		=> "\x8e\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x87"		=> "\x8e\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\xaa"		=> "\x8e\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x9f"		=> "\x8e\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xba"		=> "\x8e\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xba"		=> "\x8e\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x93"		=> "\x8e\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\xbc"		=> "\x8e\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xb5"		=> "\x8e\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x8c"		=> "\x8e\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x88"		=> "\x8e\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xab"		=> "\x8e\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xa5"		=> "\x8e\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x82"		=> "\x8e\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xb1"		=> "\x8e\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb9"		=> "\x8e\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xbb"		=> "\x8e\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x96"		=> "\x8e\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x88"		=> "\x8e\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x8b"		=> "\x8e\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xb1"		=> "\x8e\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x8a"		=> "\x8e\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xa9"		=> "\x8e\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xa0"		=> "\x8e\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xae"		=> "\x8e\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xab"		=> "\x8e\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\xa3"		=> "\x8e\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x92"		=> "\x8e\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\x96"		=> "\x8e\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x92"		=> "\x8e\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x97"		=> "\x8e\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xaa"		=> "\x8e\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xbf"		=> "\x8e\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x88"		=> "\x8e\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xb9"		=> "\x8e\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xac"		=> "\x8e\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x80"		=> "\x8e\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x9a"		=> "\x8e\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x8e"		=> "\x8e\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xa8"		=> "\x8e\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x97"		=> "\x8f\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xb1"		=> "\x8f\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x9e"		=> "\x8f\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xae"		=> "\x8f\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x81"		=> "\x8f\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xbe"		=> "\x8f\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xb2"		=> "\x8f\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x80"		=> "\x8f\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x8b"		=> "\x8f\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x82"		=> "\x8f\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x8d"		=> "\x8f\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\x92"		=> "\x8f\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xad"		=> "\x8f\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x9f"		=> "\x8f\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\x90"		=> "\x8f\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x86"		=> "\x8f\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xb2"		=> "\x8f\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x90"		=> "\x8f\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\xb4"		=> "\x8f\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xaf"		=> "\x8f\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xb1"		=> "\x8f\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x8b"		=> "\x8f\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xac"		=> "\x8f\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x86"		=> "\x8f\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x9c"		=> "\x8f\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x80"		=> "\x8f\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x8f"		=> "\x8f\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x85"		=> "\x8f\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x81"		=> "\x8f\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x93"		=> "\x8f\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x8e"		=> "\x8f\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x94"		=> "\x8f\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x81"		=> "\x8f\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x8b"		=> "\x8f\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xa3"		=> "\x8f\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xa6"		=> "\x8f\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x8d"		=> "\x8f\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x83"		=> "\x8f\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x94"		=> "\x8f\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x99"		=> "\x8f\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xbf"		=> "\x8f\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x91"		=> "\x8f\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x9d"		=> "\x8f\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xae"		=> "\x8f\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x9b"		=> "\x8f\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\xbe"		=> "\x8f\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\x9f"		=> "\x8f\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xba"		=> "\x8f\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x93"		=> "\x8f\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xb0"		=> "\x8f\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x8a"		=> "\x8f\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xbb"		=> "\x8f\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xa5"		=> "\x8f\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xac"		=> "\x8f\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xa3"		=> "\x8f\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x9c"		=> "\x8f\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\xbf"		=> "\x8f\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x86"		=> "\x8f\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xaa"		=> "\x8f\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xac"		=> "\x8f\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xaf"		=> "\x8f\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x89"		=> "\x8f\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb3"		=> "\x8f\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x96"		=> "\x8f\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xa4"		=> "\x8f\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xbe"		=> "\x8f\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x94"		=> "\x8f\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xa1"		=> "\x8f\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xb5"		=> "\x8f\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x87"		=> "\x8f\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x86"		=> "\x8f\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xa6"		=> "\x8f\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x9d"		=> "\x8f\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x80"		=> "\x8f\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x91"		=> "\x8f\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x99"		=> "\x8f\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x9a"		=> "\x8f\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xb6"		=> "\x8f\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x92"		=> "\x8f\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xb2"		=> "\x8f\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xb8"		=> "\x8f\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xaf"		=> "\x8f\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xb7"		=> "\x8f\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xb8"		=> "\x8f\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xa9"		=> "\x8f\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x99"		=> "\x8f\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xb3"		=> "\x8f\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x8f"		=> "\x8f\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x90"		=> "\x8f\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x95"		=> "\x8f\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xa4"		=> "\x8f\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xa4"		=> "\x8f\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\xb7"		=> "\x8f\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x9f"		=> "\x8f\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x9d"		=> "\x8f\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xa0"		=> "\x8f\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x87"		=> "\x8f\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xac"		=> "\x8f\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xa8"		=> "\x8f\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x86"		=> "\x8f\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\xb1"		=> "\x8f\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\x97"		=> "\x8f\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xa8"		=> "\x8f\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xbe"		=> "\x8f\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\xbc"		=> "\x8f\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xb5"		=> "\x8f\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x86"		=> "\x8f\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x8f"		=> "\x8f\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x91"		=> "\x8f\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x9a"		=> "\x8f\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x84"		=> "\x8f\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x8a"		=> "\x8f\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xa0"		=> "\x8f\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xb0"		=> "\x8f\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xbf"		=> "\x8f\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x84"		=> "\x8f\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x9b"		=> "\x8f\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x8c"		=> "\x8f\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xb7"		=> "\x8f\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x87"		=> "\x8f\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x8c"		=> "\x8f\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xad"		=> "\x8f\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xb6"		=> "\x8f\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xbe"		=> "\x8f\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xa2"		=> "\x8f\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x9f"		=> "\x8f\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xb5"		=> "\x8f\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xbc"		=> "\x8f\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x88"		=> "\x8f\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x89"		=> "\x8f\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x98"		=> "\x8f\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\xbc"		=> "\x8f\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\xa6"		=> "\x8f\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xa7"		=> "\x8f\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x87"		=> "\x8f\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x81"		=> "\x8f\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\x9d"		=> "\x8f\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\x81"		=> "\x8f\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\xa5"		=> "\x8f\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xb0"		=> "\x8f\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xa0"		=> "\x8f\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x91"		=> "\x8f\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xa7"		=> "\x8f\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xb9"		=> "\x8f\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x96"		=> "\x8f\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x96"		=> "\x8f\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\x8b"		=> "\x8f\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x89"		=> "\x8f\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x9d"		=> "\x8f\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xb3"		=> "\x8f\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x9f"		=> "\x8f\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xbc"		=> "\x8f\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x94"		=> "\x8f\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xb3"		=> "\x8f\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xa1"		=> "\x8f\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x9e"		=> "\x8f\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xa4"		=> "\x8f\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xa6"		=> "\x8f\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\xbe"		=> "\x8f\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\x98"		=> "\x8f\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x9c"		=> "\x8f\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x98"		=> "\x8f\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x8a"		=> "\x8f\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x88"		=> "\x8f\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x9e"		=> "\x8f\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x97"		=> "\x8f\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x97"		=> "\x8f\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xb0"		=> "\x8f\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x8e"		=> "\x8f\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xb4"		=> "\x8f\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x8c"		=> "\x8f\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\xa2"		=> "\x8f\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xb8"		=> "\x8f\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x85"		=> "\x8f\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xbe"		=> "\x8f\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xa1"		=> "\x8f\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x96"		=> "\x8f\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\x84"		=> "\x8f\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xb6"		=> "\x8f\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xb3"		=> "\x8f\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xa3"		=> "\x8f\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xb8"		=> "\x8f\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xb2"		=> "\x8f\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xb8"		=> "\x8f\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa0"		=> "\x8f\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\xb1"		=> "\x8f\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\xb4"		=> "\x8f\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xbe"		=> "\x8f\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xad"		=> "\x90\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x8d"		=> "\x90\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x96"		=> "\x90\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xad"		=> "\x90\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x94"		=> "\x90\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xb7"		=> "\x90\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xb2"		=> "\x90\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\xa6"		=> "\x90\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\x9f"		=> "\x90\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x95"		=> "\x90\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xb1"		=> "\x90\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xbb"		=> "\x90\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xb8"		=> "\x90\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xa1"		=> "\x90\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xb5"		=> "\x90\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\x87"		=> "\x90\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\xa0"		=> "\x90\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x9d"		=> "\x90\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xa9"		=> "\x90\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x83"		=> "\x90\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x8e"		=> "\x90\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\xaf"		=> "\x90\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xb0"		=> "\x90\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x8b"		=> "\x90\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xae"		=> "\x90\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x9b"		=> "\x90\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xb8"		=> "\x90\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb1"		=> "\x90\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xb3"		=> "\x90\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xb9"		=> "\x90\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x9f"		=> "\x90\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x9e"		=> "\x90\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xa6"		=> "\x90\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xb3"		=> "\x90\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xa3"		=> "\x90\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xaf"		=> "\x90\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xaa"		=> "\x90\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xaa"		=> "\x90\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xba"		=> "\x90\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\xab"		=> "\x90\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\x9b"		=> "\x90\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xb2"		=> "\x90\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x9d"		=> "\x90\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x87"		=> "\x90\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xba"		=> "\x90\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x81"		=> "\x90\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x83"		=> "\x90\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\xb5"		=> "\x90\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xac"		=> "\x90\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x8b"		=> "\x90\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x9a"		=> "\x90\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xbd"		=> "\x90\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x8e"		=> "\x90\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x8a"		=> "\x90\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\x85"		=> "\x90\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xa3"		=> "\x90\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xad"		=> "\x90\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xa5"		=> "\x90\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x8f"		=> "\x90\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x88"		=> "\x90\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xa2"		=> "\x90\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xb3"		=> "\x90\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xa8"		=> "\x90\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x97"		=> "\x90\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xb9"		=> "\x90\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\x82"		=> "\x90\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xa5"		=> "\x90\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xa8"		=> "\x90\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\xb4"		=> "\x90\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\x8a"		=> "\x90\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xa1"		=> "\x90\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x8b"		=> "\x90\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xa0"		=> "\x90\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xb0"		=> "\x90\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x82"		=> "\x90\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x94"		=> "\x90\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x90"		=> "\x90\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x98"		=> "\x90\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x8f"		=> "\x90\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\x9e"		=> "\x90\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x84"		=> "\x90\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x87"		=> "\x90\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\xa9"		=> "\x90\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xb0"		=> "\x90\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xa2"		=> "\x90\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\xa8"		=> "\x90\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x9b"		=> "\x90\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xae"		=> "\x90\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x89"		=> "\x90\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x99"		=> "\x90\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x85"		=> "\x90\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x97"		=> "\x90\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x80"		=> "\x90\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xbe"		=> "\x90\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x84"		=> "\x90\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\xba"		=> "\x90\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xb8"		=> "\x90\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x96"		=> "\x90\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\xac"		=> "\x90\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x9d"		=> "\x90\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xaf"		=> "\x90\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x84"		=> "\x90\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xb6"		=> "\x90\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xa2"		=> "\x90\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x93"		=> "\x90\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x81"		=> "\x90\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xa7"		=> "\x90\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x90"		=> "\x90\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xbf"		=> "\x90\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xb4"		=> "\x90\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x9f"		=> "\x90\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xb4"		=> "\x90\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xb2"		=> "\x90\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x96"		=> "\x90\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xa3"		=> "\x90\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x85"		=> "\x90\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xb2"		=> "\x90\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x9f"		=> "\x90\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x9b"		=> "\x90\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xbe"		=> "\x90\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\x96"		=> "\x90\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xb0"		=> "\x90\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xbd"		=> "\x90\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xbf"		=> "\x90\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa0"		=> "\x90\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x93"		=> "\x90\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x8b"		=> "\x90\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x9d"		=> "\x90\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x92"		=> "\x90\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x92"		=> "\x90\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x99"		=> "\x90\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x89"		=> "\x90\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x8e"		=> "\x90\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x86"		=> "\x90\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xbb"		=> "\x90\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xad"		=> "\x90\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x9c"		=> "\x90\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x9a"		=> "\x90\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xa5"		=> "\x90\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x94"		=> "\x90\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x90"		=> "\x90\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xb3"		=> "\x90\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x8d"		=> "\x90\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x8d"		=> "\x90\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xbe"		=> "\x90\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x8a"		=> "\x90\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xac"		=> "\x90\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xa4"		=> "\x90\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xa1"		=> "\x90\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x9f"		=> "\x90\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xa9"		=> "\x90\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x87"		=> "\x90\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x99"		=> "\x90\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xa5"		=> "\x90\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\x82"		=> "\x90\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x98"		=> "\x90\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xad"		=> "\x90\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x83"		=> "\x90\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x80"		=> "\x90\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xac"		=> "\x90\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xaa"		=> "\x90\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xb6"		=> "\x90\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x8c"		=> "\x90\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x89"		=> "\x90\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x99"		=> "\x90\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x88"		=> "\x90\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x83"		=> "\x90\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xa0"		=> "\x90\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xa3"		=> "\x90\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x82"		=> "\x90\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x96"		=> "\x90\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x9d"		=> "\x90\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xa6"		=> "\x90\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x87"		=> "\x90\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xb0"		=> "\x90\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x93"		=> "\x90\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xb4"		=> "\x90\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x89"		=> "\x90\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\x85"		=> "\x90\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x97"		=> "\x90\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x93"		=> "\x90\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\x9c"		=> "\x90\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x8e"		=> "\x90\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xbd"		=> "\x90\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x8b"		=> "\x90\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xbf"		=> "\x90\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\xad"		=> "\x90\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x9a"		=> "\x90\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x8a"		=> "\x91\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xa8"		=> "\x91\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xba"		=> "\x91\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x9b"		=> "\x91\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xb9"		=> "\x91\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xa6"		=> "\x91\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xae"		=> "\x91\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x8e"		=> "\x91\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xb5"		=> "\x91\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xb8"		=> "\x91\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xb7"		=> "\x91\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\xad"		=> "\x91\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x91"		=> "\x91\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x83"		=> "\x91\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xae"		=> "\x91\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x8d"		=> "\x91\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x84"		=> "\x91\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xb8"		=> "\x91\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\xb6"		=> "\x91\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xa8"		=> "\x91\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x85"		=> "\x91\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x95"		=> "\x91\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xb3"		=> "\x91\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x8e"		=> "\x91\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\x8c"		=> "\x91\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x91"		=> "\x91\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xa8"		=> "\x91\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xaa"		=> "\x91\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xbe"		=> "\x91\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xbd"		=> "\x91\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x9a"		=> "\x91\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x99"		=> "\x91\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x8f"		=> "\x91\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x8e"		=> "\x91\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\x8e"		=> "\x91\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x96"		=> "\x91\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x9f"		=> "\x91\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x97"		=> "\x91\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xa0"		=> "\x91\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x84"		=> "\x91\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x87"		=> "\x91\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xb4"		=> "\x91\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\xbb"		=> "\x91\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xa1"		=> "\x91\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\xa0"		=> "\x91\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xa7"		=> "\x91\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xb5"		=> "\x91\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x8c"		=> "\x91\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xa2"		=> "\x91\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x89"		=> "\x91\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xaa"		=> "\x91\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xae"		=> "\x91\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x8f"		=> "\x91\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xbd"		=> "\x91\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x8b"		=> "\x91\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xa4"		=> "\x91\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x9d"		=> "\x91\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xa3"		=> "\x91\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb3"		=> "\x91\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\x9c"		=> "\x91\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x83"		=> "\x91\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\xbf"		=> "\x91\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xbb"		=> "\x91\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x8d"		=> "\x91\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xa9"		=> "\x91\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xb9"		=> "\x91\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xa3"		=> "\x91\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x8d"		=> "\x91\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xbd"		=> "\x91\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x95"		=> "\x91\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xa5"		=> "\x91\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x89"		=> "\x91\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xa9"		=> "\x91\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xb8"		=> "\x91\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x93"		=> "\x91\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x9f"		=> "\x91\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x8f"		=> "\x91\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\x9c"		=> "\x91\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xa1"		=> "\x91\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\x89"		=> "\x91\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\x98"		=> "\x91\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xac"		=> "\x91\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xbc"		=> "\x91\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xbb"		=> "\x91\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x85"		=> "\x91\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xb0"		=> "\x91\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x81"		=> "\x91\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xad"		=> "\x91\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\x97"		=> "\x91\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x9c"		=> "\x91\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\x92"		=> "\x91\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x8f"		=> "\x91\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\x97"		=> "\x91\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x8e"		=> "\x91\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x93"		=> "\x91\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\xb5"		=> "\x91\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x88"		=> "\x91\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xa0"		=> "\x91\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x83"		=> "\x91\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xb4"		=> "\x91\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x87"		=> "\x91\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb3"		=> "\x91\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xaf"		=> "\x91\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\x89"		=> "\x91\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x9f"		=> "\x91\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xac"		=> "\x91\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\xb3"		=> "\x91\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x9f"		=> "\x91\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x97"		=> "\x91\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x9e"		=> "\x91\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x8a"		=> "\x91\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x8f"		=> "\x91\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\x9a"		=> "\x91\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x92"		=> "\x91\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x96"		=> "\x91\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xb6"		=> "\x91\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x83"		=> "\x91\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x98"		=> "\x91\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xab"		=> "\x91\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x8a"		=> "\x91\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\x8d"		=> "\x91\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x91"		=> "\x91\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x9c"		=> "\x91\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x96"		=> "\x91\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x9a"		=> "\x91\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xaa"		=> "\x91\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xb0"		=> "\x91\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x91"		=> "\x91\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\xbe"		=> "\x91\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x95"		=> "\x91\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xa5"		=> "\x91\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb0"		=> "\x91\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x93"		=> "\x91\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x81"		=> "\x91\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xb5"		=> "\x91\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x95"		=> "\x91\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x80"		=> "\x91\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x84"		=> "\x91\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\xa8"		=> "\x91\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x93"		=> "\x91\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x86"		=> "\x91\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xbe"		=> "\x91\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x90"		=> "\x91\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xb1"		=> "\x91\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xaf"		=> "\x91\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x85"		=> "\x91\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xa0"		=> "\x91\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x8b"		=> "\x91\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xb4"		=> "\x91\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xbf"		=> "\x91\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xb0"		=> "\x91\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x9e"		=> "\x91\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x8e"		=> "\x91\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xbf"		=> "\x91\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x94"		=> "\x91\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x8b"		=> "\x91\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xb8"		=> "\x91\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x80"		=> "\x91\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xae"		=> "\x91\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x8a"		=> "\x91\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x9b"		=> "\x91\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x9b"		=> "\x91\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xa3"		=> "\x91\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xb0"		=> "\x91\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xa7"		=> "\x91\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xac"		=> "\x91\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x8d"		=> "\x91\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x8c"		=> "\x91\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\xb9"		=> "\x91\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x9d"		=> "\x91\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\xa7"		=> "\x91\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x93"		=> "\x91\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x84"		=> "\x91\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x85"		=> "\x91\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x98"		=> "\x91\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x9e"		=> "\x91\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x93"		=> "\x91\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xa2"		=> "\x91\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xaf"		=> "\x91\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xa2"		=> "\x91\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x97"		=> "\x91\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\xb8"		=> "\x91\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x81"		=> "\x91\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xbe"		=> "\x91\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xb8"		=> "\x91\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xa7"		=> "\x91\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xb8"		=> "\x91\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xaa"		=> "\x91\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xa9"		=> "\x92\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x86"		=> "\x92\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x94"		=> "\x92\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xb0"		=> "\x92\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xaa"		=> "\x92\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\xb1"		=> "\x92\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xbd"		=> "\x92\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xaa"		=> "\x92\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xbf"		=> "\x92\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x9a"		=> "\x92\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb0\xb7"		=> "\x92\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xb8"		=> "\x92\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\x88"		=> "\x92\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xbd"		=> "\x92\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xb0"		=> "\x92\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xb9"		=> "\x92\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x98"		=> "\x92\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\x86"		=> "\x92\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\xa6"		=> "\x92\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x85"		=> "\x92\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xa2"		=> "\x92\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xa6"		=> "\x92\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x8e"		=> "\x92\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xa1"		=> "\x92\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x9b"		=> "\x92\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xad"		=> "\x92\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xad"		=> "\x92\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xaf"		=> "\x92\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\xaa"		=> "\x92\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xbb"		=> "\x92\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\xbd"		=> "\x92\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x86"		=> "\x92\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x8b"		=> "\x92\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x95"		=> "\x92\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x9b"		=> "\x92\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xa3"		=> "\x92\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x87"		=> "\x92\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xbe"		=> "\x92\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xad"		=> "\x92\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x96"		=> "\x92\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\x80"		=> "\x92\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xb5"		=> "\x92\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xb7"		=> "\x92\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x87"		=> "\x92\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa4"		=> "\x92\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xa5"		=> "\x92\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\xb0"		=> "\x92\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x9b"		=> "\x92\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xa5"		=> "\x92\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xba"		=> "\x92\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xa0"		=> "\x92\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xb4"		=> "\x92\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x9a"		=> "\x92\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xae"		=> "\x92\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xb4"		=> "\x92\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x98"		=> "\x92\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x85"		=> "\x92\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\xb3"		=> "\x92\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x89"		=> "\x92\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x9c"		=> "\x92\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xb9"		=> "\x92\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x91"		=> "\x92\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x84"		=> "\x92\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x90"		=> "\x92\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xa9"		=> "\x92\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x92"		=> "\x92\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xb6"		=> "\x92\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\xa1"		=> "\x92\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\x80"		=> "\x92\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xad"		=> "\x92\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xb2"		=> "\x92\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x99"		=> "\x92\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xa0"		=> "\x92\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xbd"		=> "\x92\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xbc"		=> "\x92\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xb1"		=> "\x92\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xa8"		=> "\x92\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\xab"		=> "\x92\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xb7"		=> "\x92\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xbb"		=> "\x92\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x8e"		=> "\x92\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xb3"		=> "\x92\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x90"		=> "\x92\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x97"		=> "\x92\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\xa6"		=> "\x92\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xaa"		=> "\x92\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xa7"		=> "\x92\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\x97"		=> "\x92\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xaf"		=> "\x92\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x81"		=> "\x92\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x86"		=> "\x92\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x8b"		=> "\x92\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x8b"		=> "\x92\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xb5"		=> "\x92\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x96"		=> "\x92\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xb3"		=> "\x92\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x81"		=> "\x92\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x94"		=> "\x92\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xb5"		=> "\x92\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xab"		=> "\x92\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xb4"		=> "\x92\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xb2"		=> "\x92\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x91"		=> "\x92\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xa2"		=> "\x92\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x9d"		=> "\x92\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xae"		=> "\x92\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x92"		=> "\x92\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xba"		=> "\x92\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xba"		=> "\x92\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xb4"		=> "\x92\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\xb9"		=> "\x92\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xb8"		=> "\x92\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xb6"		=> "\x92\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xbf"		=> "\x92\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x9c"		=> "\x92\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\x85"		=> "\x92\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xb3"		=> "\x92\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x9a"		=> "\x92\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x95\xb7"		=> "\x92\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x82"		=> "\x92\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xa5"		=> "\x92\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x85"		=> "\x92\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\x97"		=> "\x92\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xb4"		=> "\x92\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x95"		=> "\x92\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x88"		=> "\x92\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x8d"		=> "\x92\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x83"		=> "\x92\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\xae"		=> "\x92\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xb3"		=> "\x92\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xa5"		=> "\x92\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\x9c"		=> "\x92\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x8e"		=> "\x92\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x8c"		=> "\x92\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xbd"		=> "\x92\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\x9a"		=> "\x92\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x9b"		=> "\x92\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x9a"		=> "\x92\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x9a"		=> "\x92\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x82"		=> "\x92\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xb4"		=> "\x92\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xbb"		=> "\x92\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x83"		=> "\x92\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xac"		=> "\x92\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x98"		=> "\x92\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xbb"		=> "\x92\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\xa6"		=> "\x92\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xb4"		=> "\x92\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x94"		=> "\x92\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xbf"		=> "\x92\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xb0"		=> "\x92\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\xaa"		=> "\x92\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xb7"		=> "\x92\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\xac"		=> "\x92\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xac"		=> "\x92\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xaa"		=> "\x92\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x8a"		=> "\x92\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xa3"		=> "\x92\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xb4"		=> "\x92\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xad"		=> "\x92\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x8e"		=> "\x92\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x9c"		=> "\x92\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xb5"		=> "\x92\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x83"		=> "\x92\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x9e"		=> "\x92\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x88"		=> "\x92\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xa4"		=> "\x92\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x9a"		=> "\x92\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x9d"		=> "\x92\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x95"		=> "\x92\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xad"		=> "\x92\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xb7"		=> "\x92\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x9f"		=> "\x92\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x8c"		=> "\x92\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xb5"		=> "\x92\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\xba"		=> "\x92\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x90"		=> "\x92\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xaf"		=> "\x92\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x80"		=> "\x92\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x87"		=> "\x92\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x8e"		=> "\x92\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x8b"		=> "\x92\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xa0"		=> "\x92\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\x87"		=> "\x92\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x82"		=> "\x92\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xa6"		=> "\x92\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x84"		=> "\x92\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x93"		=> "\x92\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\xb8"		=> "\x93\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84\xad"		=> "\x93\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x98"		=> "\x93\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\x8e"		=> "\x93\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xa5"		=> "\x93\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\x98"		=> "\x93\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xa2"		=> "\x93\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xb5"		=> "\x93\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\xb4"		=> "\x93\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x84"		=> "\x93\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x9b"		=> "\x93\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xa9"		=> "\x93\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x91"		=> "\x93\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xba"		=> "\x93\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xb2"		=> "\x93\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xb9"		=> "\x93\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xa4"		=> "\x93\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x8d"		=> "\x93\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xad"		=> "\x93\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x84"		=> "\x93\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xb8"		=> "\x93\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\xab"		=> "\x93\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xa9"		=> "\x93\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x95"		=> "\x93\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x97"		=> "\x93\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xbb"		=> "\x93\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x8f"		=> "\x93\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x9c"		=> "\x93\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xbc"		=> "\x93\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\xa2"		=> "\x93\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x9b"		=> "\x93\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xb9"		=> "\x93\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x9d"		=> "\x93\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xbf"		=> "\x93\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xb1"		=> "\x93\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xb0"		=> "\x93\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xbb"		=> "\x93\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x8e"		=> "\x93\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x90"		=> "\x93\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xb5"		=> "\x93\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x97"		=> "\x93\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xac"		=> "\x93\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xa0"		=> "\x93\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x92"		=> "\x93\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x97"		=> "\x93\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x9c"		=> "\x93\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa1"		=> "\x93\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xbb"		=> "\x93\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x9f"		=> "\x93\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xad"		=> "\x93\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x94"		=> "\x93\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\xbd"		=> "\x93\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x8d"		=> "\x93\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xa5"		=> "\x93\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xba"		=> "\x93\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xaa"		=> "\x93\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xa6"		=> "\x93\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x9f"		=> "\x93\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xb4"		=> "\x93\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x92"		=> "\x93\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x92"		=> "\x93\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x9a"		=> "\x93\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xac"		=> "\x93\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x8d"		=> "\x93\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x80"		=> "\x93\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\x90"		=> "\x93\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x94"		=> "\x93\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x98"		=> "\x93\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x97"		=> "\x93\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x95"		=> "\x93\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xb6"		=> "\x93\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\x8b"		=> "\x93\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xbc"		=> "\x93\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x95"		=> "\x93\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\xad"		=> "\x93\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xb1"		=> "\x93\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x83"		=> "\x93\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xbc"		=> "\x93\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x9f"		=> "\x93\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x97"		=> "\x93\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x98"		=> "\x93\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\xaf"		=> "\x93\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x9b"		=> "\x93\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\xaf"		=> "\x93\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x88"		=> "\x93\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x93"		=> "\x93\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x98"		=> "\x93\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\xb7"		=> "\x93\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x89"		=> "\x93\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x94"		=> "\x93\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x92"		=> "\x93\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x96"		=> "\x93\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xb1"		=> "\x93\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xb0"		=> "\x93\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xa3"		=> "\x93\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xa9"		=> "\x93\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xa4"		=> "\x93\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x8e"		=> "\x93\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x84"		=> "\x93\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x86"		=> "\x93\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\x8f"		=> "\x93\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x83"		=> "\x93\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x8f"		=> "\x93\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\x99"		=> "\x93\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xb6"		=> "\x93\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xad"		=> "\x93\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\xb0"		=> "\x93\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x98"		=> "\x93\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x8d"		=> "\x93\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x95"		=> "\x93\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x8c"		=> "\x93\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x82"		=> "\x93\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x8e"		=> "\x93\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xa7"		=> "\x93\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x9e"		=> "\x93\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x9e"		=> "\x93\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xb3"		=> "\x93\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xa5"		=> "\x93\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xb4"		=> "\x93\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\x84"		=> "\x93\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x93"		=> "\x93\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x85"		=> "\x93\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xa0"		=> "\x93\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x87"		=> "\x93\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xbf"		=> "\x93\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x97"		=> "\x93\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xb3"		=> "\x93\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x9c"		=> "\x93\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xb9"		=> "\x93\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xa3"		=> "\x93\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xbf"		=> "\x93\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xa4"		=> "\x93\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x92"		=> "\x93\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xac"		=> "\x93\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xad"		=> "\x93\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x83"		=> "\x93\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xa1"		=> "\x93\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xb8"		=> "\x93\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x81"		=> "\x93\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xb4"		=> "\x93\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x8a"		=> "\x93\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xb6"		=> "\x93\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xab"		=> "\x93\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x85"		=> "\x93\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x89"		=> "\x93\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x9e"		=> "\x93\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\xb8"		=> "\x93\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xaf"		=> "\x93\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x87"		=> "\x93\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xa6"		=> "\x93\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x8c"		=> "\x93\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x9a"		=> "\x93\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x81"		=> "\x93\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x93"		=> "\x93\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x91"		=> "\x93\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x87"		=> "\x93\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x8d"		=> "\x93\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x88"		=> "\x93\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\xa3"		=> "\x93\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x85"		=> "\x93\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x8d"		=> "\x93\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xaa"		=> "\x93\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x99"		=> "\x93\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x8e"		=> "\x93\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\x98"		=> "\x93\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xba"		=> "\x93\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x8b"		=> "\x93\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xa2"		=> "\x93\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\xb4"		=> "\x93\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x84"		=> "\x93\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xb7"		=> "\x93\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x97"		=> "\x93\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xa0"		=> "\x93\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x9f"		=> "\x93\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xa3"		=> "\x93\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x9d"		=> "\x93\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x8c"		=> "\x93\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xbc"		=> "\x93\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x90"		=> "\x93\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xa9"		=> "\x93\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x82"		=> "\x93\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x91"		=> "\x93\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x89"		=> "\x93\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\xb9"		=> "\x93\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xbf"		=> "\x93\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xa5"		=> "\x93\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\xb3"		=> "\x93\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xa5"		=> "\x93\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x82"		=> "\x94\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xbf"		=> "\x94\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\xae"		=> "\x94\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xbb"		=> "\x94\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x8a"		=> "\x94\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x8d"		=> "\x94\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x8d"		=> "\x94\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xa1"		=> "\x94\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xb0"		=> "\x94\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\xa2"		=> "\x94\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xa7"		=> "\x94\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xb1"		=> "\x94\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xab"		=> "\x94\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\xb1"		=> "\x94\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xb4"		=> "\x94\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xb5"		=> "\x94\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xbb"		=> "\x94\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x9a"		=> "\x94\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x83"		=> "\x94\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x98"		=> "\x94\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x83"		=> "\x94\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xbc"		=> "\x94\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x8b"		=> "\x94\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x9c"		=> "\x94\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\xa2"		=> "\x94\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xa9"		=> "\x94\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x83"		=> "\x94\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x8d"		=> "\x94\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xbd"		=> "\x94\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\xb3"		=> "\x94\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xbf"		=> "\x94\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xb2"		=> "\x94\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x97"		=> "\x94\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\xa4"		=> "\x94\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xb4"		=> "\x94\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x8a"		=> "\x94\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xad"		=> "\x94\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x87"		=> "\x94\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xb7"		=> "\x94\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xa2"		=> "\x94\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xbe"		=> "\x94\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xb6"		=> "\x94\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xb4"		=> "\x94\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\x86"		=> "\x94\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xb5"		=> "\x94\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xad"		=> "\x94\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\xac"		=> "\x94\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xb3"		=> "\x94\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x83"		=> "\x94\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x9d"		=> "\x94\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x92"		=> "\x94\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x97"		=> "\x94\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xaf"		=> "\x94\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x83"		=> "\x94\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x8c"		=> "\x94\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x8c"		=> "\x94\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xba"		=> "\x94\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xa9"		=> "\x94\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x8d"		=> "\x94\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x8d"		=> "\x94\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\xb9"		=> "\x94\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaa\x92"		=> "\x94\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x85"		=> "\x94\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xb3"		=> "\x94\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xa4"		=> "\x94\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xbd"		=> "\x94\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xb7"		=> "\x94\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xb2"		=> "\x94\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xa0"		=> "\x94\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xaa"		=> "\x94\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x99"		=> "\x94\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xbf"		=> "\x94\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xa4"		=> "\x94\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xa7"		=> "\x94\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xa9"		=> "\x94\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xaf"		=> "\x94\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xa5"		=> "\x94\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x9a"		=> "\x94\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x8d"		=> "\x94\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x8f"		=> "\x94\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x8a"		=> "\x94\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xbd"		=> "\x94\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x94"		=> "\x94\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x95"		=> "\x94\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xb6"		=> "\x94\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x84"		=> "\x94\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xab"		=> "\x94\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x9d"		=> "\x94\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xa0"		=> "\x94\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\x86"		=> "\x94\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x9b"		=> "\x94\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\xab"		=> "\x94\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x81"		=> "\x94\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xa6"		=> "\x94\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xbd"		=> "\x94\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\xb1"		=> "\x94\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xb2"		=> "\x94\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\xb8"		=> "\x94\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x87"		=> "\x94\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x88"		=> "\x94\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\xa8"		=> "\x94\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xa1"		=> "\x94\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x8c"		=> "\x94\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x91"		=> "\x94\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xa0"		=> "\x94\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xab"		=> "\x94\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xa2"		=> "\x94\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x8c"		=> "\x94\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xba"		=> "\x94\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x97"		=> "\x94\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xaa"		=> "\x94\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x90"		=> "\x94\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xb0"		=> "\x94\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x9c"		=> "\x94\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x8f"		=> "\x94\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xa5"		=> "\x94\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xa9"		=> "\x94\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\xba"		=> "\x94\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x99"		=> "\x94\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xa4"		=> "\x94\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xbc"		=> "\x94\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xb4"		=> "\x94\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xa4"		=> "\x94\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x8a"		=> "\x94\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x8d"		=> "\x94\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x9b"		=> "\x94\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x86"		=> "\x94\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\xac"		=> "\x94\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x91"		=> "\x94\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xbf"		=> "\x94\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\xbe"		=> "\x94\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x8e"		=> "\x94\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x88"		=> "\x94\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xaf"		=> "\x94\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xad"		=> "\x94\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x94"		=> "\x94\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x81"		=> "\x94\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xac"		=> "\x94\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xa9"		=> "\x94\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xa9"		=> "\x94\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x84"		=> "\x94\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x86"		=> "\x94\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xa9"		=> "\x94\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x92"		=> "\x94\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xaf"		=> "\x94\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\xbd"		=> "\x94\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xa9"		=> "\x94\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xaa"		=> "\x94\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xa4"		=> "\x94\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x90"		=> "\x94\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x83"		=> "\x94\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xae"		=> "\x94\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xaa"		=> "\x94\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x91"		=> "\x94\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xa6"		=> "\x94\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x83"		=> "\x94\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x87"		=> "\x94\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xbc"		=> "\x94\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xb2"		=> "\x94\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x89"		=> "\x94\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xb9"		=> "\x94\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xab"		=> "\x94\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x90"		=> "\x94\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x94"		=> "\x94\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x8c"		=> "\x94\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xb2"		=> "\x94\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xae"		=> "\x94\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x91"		=> "\x94\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x98"		=> "\x94\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x8b"		=> "\x94\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xb7"		=> "\x94\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xa5"		=> "\x94\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xab"		=> "\x94\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xb9"		=> "\x94\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xbb"		=> "\x94\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xbf"		=> "\x94\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x9e"		=> "\x94\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\x9b"		=> "\x94\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x8b"		=> "\x94\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xb8"		=> "\x94\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x99"		=> "\x94\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xbe"		=> "\x94\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xae"		=> "\x94\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x87"		=> "\x94\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x98"		=> "\x94\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xb5"		=> "\x94\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x89"		=> "\x94\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x8e"		=> "\x94\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\xbb"		=> "\x95\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x8a"		=> "\x95\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x97"		=> "\x95\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xb9"		=> "\x95\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x8b"		=> "\x95\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xad"		=> "\x95\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xa6"		=> "\x95\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x9d"		=> "\x95\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xb1"		=> "\x95\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x98"		=> "\x95\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xbc"		=> "\x95\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x85"		=> "\x95\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xa2"		=> "\x95\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x86"		=> "\x95\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xbc"		=> "\x95\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\xa7"		=> "\x95\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\xab"		=> "\x95\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaa\x9b"		=> "\x95\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x90"		=> "\x95\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xbe"		=> "\x95\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xac"		=> "\x95\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xb5"		=> "\x95\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xaa"		=> "\x95\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x99"		=> "\x95\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\xb7"		=> "\x95\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x82"		=> "\x95\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xa2"		=> "\x95\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\xa8"		=> "\x95\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xa8"		=> "\x95\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x95"		=> "\x95\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xb9"		=> "\x95\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x9f"		=> "\x95\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x8f"		=> "\x95\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x85"		=> "\x95\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x92"		=> "\x95\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x97"		=> "\x95\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa8"		=> "\x95\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xb2"		=> "\x95\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\x9c"		=> "\x95\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xad"		=> "\x95\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xad"		=> "\x95\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\x81"		=> "\x95\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xac"		=> "\x95\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x8c"		=> "\x95\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\x9c"		=> "\x95\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x95"		=> "\x95\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xa7"		=> "\x95\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x93"		=> "\x95\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xbb"		=> "\x95\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x8f"		=> "\x95\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xb6"		=> "\x95\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x8d"		=> "\x95\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x98"		=> "\x95\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\xa0"		=> "\x95\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xab"		=> "\x95\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\xa6"		=> "\x95\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x8c"		=> "\x95\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xa8"		=> "\x95\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x83"		=> "\x95\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x9c"		=> "\x95\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x96"		=> "\x95\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xb6"		=> "\x95\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xb7"		=> "\x95\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xa7"		=> "\x95\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xae"		=> "\x95\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xae"		=> "\x95\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xb6"		=> "\x95\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xa6"		=> "\x95\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x90"		=> "\x95\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x9a"		=> "\x95\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\x99"		=> "\x95\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x9c"		=> "\x95\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xa0"		=> "\x95\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xa6"		=> "\x95\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xb4"		=> "\x95\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\x9c"		=> "\x95\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x84"		=> "\x95\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xae"		=> "\x95\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xab"		=> "\x95\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xa6"		=> "\x95\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x9e"		=> "\x95\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xa1"		=> "\x95\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xaa"		=> "\x95\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\xa8"		=> "\x95\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x81"		=> "\x95\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x93"		=> "\x95\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa2\xa8"		=> "\x95\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xba"		=> "\x95\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x97"		=> "\x95\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x8f"		=> "\x95\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xaf"		=> "\x95\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xa9"		=> "\x95\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x85"		=> "\x95\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x8d"		=> "\x95\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x8f"		=> "\x95\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xb9"		=> "\x95\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x87"		=> "\x95\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x86"		=> "\x95\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb5"		=> "\x95\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x97"		=> "\x95\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x95"		=> "\x95\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xb8"		=> "\x95\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x8f"		=> "\x95\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xa9"		=> "\x95\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x92"		=> "\x95\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x86"		=> "\x95\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xbb"		=> "\x95\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\xb4"		=> "\x95\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xb3"		=> "\x95\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xa4"		=> "\x95\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xae"		=> "\x95\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x9a"		=> "\x95\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xae"		=> "\x95\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x89"		=> "\x95\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x9e"		=> "\x95\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x9b"		=> "\x95\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xb0"		=> "\x95\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x87"		=> "\x95\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\x9e"		=> "\x95\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x99"		=> "\x95\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xb5"		=> "\x95\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xb5"		=> "\x95\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x80"		=> "\x95\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xa3"		=> "\x95\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xb3"		=> "\x95\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x8a"		=> "\x95\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x84"		=> "\x95\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xa6"		=> "\x95\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\xbd"		=> "\x95\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x89"		=> "\x95\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x9b"		=> "\x95\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\xb3"		=> "\x95\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x81"		=> "\x95\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xbb"		=> "\x95\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x81"		=> "\x95\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x96"		=> "\x95\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xa7"		=> "\x95\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xa5"		=> "\x95\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xa5"		=> "\x95\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x91"		=> "\x95\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x86"		=> "\x95\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x8f"		=> "\x95\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x89"		=> "\x95\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x87"		=> "\x95\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x87"		=> "\x95\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xa8"		=> "\x95\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xba"		=> "\x95\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\x94"		=> "\x95\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x8d"		=> "\x95\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xbf"		=> "\x95\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x89"		=> "\x95\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\xa9"		=> "\x95\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x81"		=> "\x95\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\xad"		=> "\x95\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x9d"		=> "\x95\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x97"		=> "\x95\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xaa"		=> "\x95\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x83"		=> "\x95\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\x95"		=> "\x95\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xa9"		=> "\x95\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xab"		=> "\x95\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x9c"		=> "\x95\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x94"		=> "\x95\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x82"		=> "\x95\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x9f"		=> "\x95\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\x93"		=> "\x95\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x95"		=> "\x95\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x8a"		=> "\x95\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xae"		=> "\x95\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x8d"		=> "\x95\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xbf"		=> "\x95\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xa9"		=> "\x95\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa3"		=> "\x95\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xb8"		=> "\x95\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x85"		=> "\x95\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x86"		=> "\x95\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xb1"		=> "\x95\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x89"		=> "\x95\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x9d"		=> "\x95\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xb0"		=> "\x95\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xaf"		=> "\x95\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\xa9"		=> "\x95\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\x96"		=> "\x95\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xb1"		=> "\x95\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xa7"		=> "\x95\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xbe"		=> "\x95\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xb9"		=> "\x95\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x8b"		=> "\x95\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x95"		=> "\x96\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xa1"		=> "\x96\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\xb9"		=> "\x96\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xb2"		=> "\x96\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xab"		=> "\x96\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x9e"		=> "\x96\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xb3"		=> "\x96\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\x8c"		=> "\x96\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\xac"		=> "\x96\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x82"		=> "\x96\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x92"		=> "\x96\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xaa"		=> "\x96\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x8a"		=> "\x96\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\xa6"		=> "\x96\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x92"		=> "\x96\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xbd"		=> "\x96\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xb3"		=> "\x96\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xac"		=> "\x96\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x8f"		=> "\x96\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xa1"		=> "\x96\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x8d"		=> "\x96\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x96"		=> "\x96\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x8a"		=> "\x96\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xa8"		=> "\x96\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xbd"		=> "\x96\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x98"		=> "\x96\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x99"		=> "\x96\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xbf"		=> "\x96\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xb4"		=> "\x96\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x9b"		=> "\x96\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x90"		=> "\x96\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x92"		=> "\x96\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x92"		=> "\x96\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xa1"		=> "\x96\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xaa"		=> "\x96\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xa8"		=> "\x96\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x80"		=> "\x96\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x8c"		=> "\x96\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xbf"		=> "\x96\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xbe"		=> "\x96\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\xb2"		=> "\x96\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xa0"		=> "\x96\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xac"		=> "\x96\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x97"		=> "\x96\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x95"		=> "\x96\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x9c"		=> "\x96\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xa8"		=> "\x96\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xb2"		=> "\x96\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xb4"		=> "\x96\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xa7"		=> "\x96\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xa6"		=> "\x96\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x86"		=> "\x96\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xa6"		=> "\x96\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x83"		=> "\x96\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xa1"		=> "\x96\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x86"		=> "\x96\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x80"		=> "\x96\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x8c"		=> "\x96\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x94"		=> "\x96\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xac"		=> "\x96\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xbb"		=> "\x96\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xa1"		=> "\x96\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x86"		=> "\x96\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\xa9"		=> "\x96\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\xa8"		=> "\x96\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x94"		=> "\x96\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xbb"		=> "\x96\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x8b"		=> "\x96\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xb9"		=> "\x96\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xa7"		=> "\x96\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x9a"		=> "\x96\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x8e"		=> "\x96\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xa9"		=> "\x96\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x99"		=> "\x96\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x95"		=> "\x96\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x9c"		=> "\x96\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x95"		=> "\x96\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xaa"		=> "\x96\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xbe"		=> "\x96\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\x92"		=> "\x96\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x9d"		=> "\x96\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xa6"		=> "\x96\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xa3"		=> "\x96\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x88"		=> "\x96\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xb9"		=> "\x96\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xab"		=> "\x96\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xab"		=> "\x96\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\x84"		=> "\x96\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xad"		=> "\x96\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xad"		=> "\x96\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xbf"		=> "\x96\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x87"		=> "\x96\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xa2"		=> "\x96\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x80"		=> "\x96\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xab"		=> "\x96\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x93"		=> "\x96\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xb3"		=> "\x96\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xaa"		=> "\x96\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x85"		=> "\x96\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xb3"		=> "\x96\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x95"		=> "\x96\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xac"		=> "\x96\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x86"		=> "\x96\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x9c"		=> "\x96\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x8a"		=> "\x96\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x91"		=> "\x96\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x94"		=> "\x96\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x88"		=> "\x96\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x99"		=> "\x96\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x8d"		=> "\x96\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\x91"		=> "\x96\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xa0"		=> "\x96\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x99"		=> "\x96\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xa2"		=> "\x96\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\xa1"		=> "\x96\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x9f"		=> "\x96\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\x9b"		=> "\x96\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xa7"		=> "\x96\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xa1"		=> "\x96\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x8b"		=> "\x96\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\xbf"		=> "\x96\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\x98"		=> "\x96\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xa5"		=> "\x96\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x8d"		=> "\x96\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xbd"		=> "\x96\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x8e"		=> "\x96\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x9f"		=> "\x96\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xb7"		=> "\x96\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x98"		=> "\x96\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xb4"		=> "\x96\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\xaa"		=> "\x96\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x9d"		=> "\x96\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x85"		=> "\x96\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x8d"		=> "\x96\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x89"		=> "\x96\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xbf"		=> "\x96\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xac"		=> "\x96\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xa2"		=> "\x96\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xba"		=> "\x96\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\xb8"		=> "\x96\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xa1"		=> "\x96\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x82"		=> "\x96\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x84"		=> "\x96\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x9f"		=> "\x96\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x9b"		=> "\x96\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\x9b"		=> "\x96\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xb2"		=> "\x96\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xb2"		=> "\x96\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x97"		=> "\x96\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\x99"		=> "\x96\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\xb2"		=> "\x96\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xa8"		=> "\x96\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x99"		=> "\x96\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xae"		=> "\x96\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xa2"		=> "\x96\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xbf"		=> "\x96\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x85"		=> "\x96\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xa4"		=> "\x96\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xbb"		=> "\x96\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\xbe"		=> "\x96\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xb0"		=> "\x96\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x8f"		=> "\x96\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xb6"		=> "\x96\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x8b"		=> "\x96\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x80"		=> "\x96\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x81"		=> "\x96\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x9f"		=> "\x96\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xb6"		=> "\x96\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x9c"		=> "\x96\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xba"		=> "\x96\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\xb6"		=> "\x96\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x8e"		=> "\x96\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xa5"		=> "\x96\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xa2"		=> "\x96\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\x84"		=> "\x96\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xb9"		=> "\x96\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x84"		=> "\x96\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xac"		=> "\x96\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xb3"		=> "\x96\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x8d"		=> "\x96\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x96"		=> "\x96\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xb3"		=> "\x96\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xae"		=> "\x96\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x93"		=> "\x96\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x89"		=> "\x96\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x88"		=> "\x96\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xb9"		=> "\x96\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x92"		=> "\x96\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xad"		=> "\x97\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xb8"		=> "\x97\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\xaf"		=> "\x97\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x91"		=> "\x97\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\xaa"		=> "\x97\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x87"		=> "\x97\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x8b"		=> "\x97\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xa5"		=> "\x97\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xbd"		=> "\x97\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xa0"		=> "\x97\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x82"		=> "\x97\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x96"		=> "\x97\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x89"		=> "\x97\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x9a"		=> "\x97\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\xa7"		=> "\x97\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x8c"		=> "\x97\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xb6"		=> "\x97\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xb7"		=> "\x97\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xb1"		=> "\x97\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x90"		=> "\x97\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x95"		=> "\x97\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x98"		=> "\x97\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x8a"		=> "\x97\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x91"		=> "\x97\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\xb5"		=> "\x97\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x84"		=> "\x97\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\x8d"		=> "\x97\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x95"		=> "\x97\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x88"		=> "\x97\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x99"		=> "\x97\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x8e"		=> "\x97\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x89"		=> "\x97\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xbf"		=> "\x97\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x90"		=> "\x97\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\xad"		=> "\x97\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xbc"		=> "\x97\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x96"		=> "\x97\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xb9"		=> "\x97\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xb8"		=> "\x97\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x9a"		=> "\x97\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xba"		=> "\x97\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x81"		=> "\x97\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x9c"		=> "\x97\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x8a"		=> "\x97\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x98"		=> "\x97\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x8b"		=> "\x97\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xb6"		=> "\x97\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\x94"		=> "\x97\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xa8"		=> "\x97\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\xaf"		=> "\x97\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x8a"		=> "\x97\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x80"		=> "\x97\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\x89"		=> "\x97\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x89"		=> "\x97\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x81"		=> "\x97\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xa1"		=> "\x97\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\x8a"		=> "\x97\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xa5"		=> "\x97\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xbd"		=> "\x97\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x8a"		=> "\x97\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xbe"		=> "\x97\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x91"		=> "\x97\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xb2"		=> "\x97\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x83"		=> "\x97\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xb4"		=> "\x97\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\x8c"		=> "\x97\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xbc"		=> "\x97\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x80"		=> "\x97\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x85"		=> "\x97\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\xba"		=> "\x97\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xb8"		=> "\x97\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xa5"		=> "\x97\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\xb1"		=> "\x97\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xbc"		=> "\x97\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xb7"		=> "\x97\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x9b"		=> "\x97\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xa1"		=> "\x97\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xbd"		=> "\x97\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xaa"		=> "\x97\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\xb1"		=> "\x97\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb5"		=> "\x97\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\x90"		=> "\x97\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\x84"		=> "\x97\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xab"		=> "\x97\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\x8d"		=> "\x97\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\xad"		=> "\x97\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xa7"		=> "\x97\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xa9"		=> "\x97\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x8f"		=> "\x97\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xa5"		=> "\x97\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x8e"		=> "\x97\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xa8"		=> "\x97\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\x86"		=> "\x97\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\x83"		=> "\x97\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xa2"		=> "\x97\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x8f"		=> "\x97\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xa1"		=> "\x97\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x8c"		=> "\x97\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xa2"		=> "\x97\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xb8"		=> "\x97\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x8b"		=> "\x97\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\x87"		=> "\x97\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x8b"		=> "\x97\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\x8e"		=> "\x97\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xa0"		=> "\x97\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xa5"		=> "\x97\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x89"		=> "\x97\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\x81"		=> "\x97\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x9c"		=> "\x97\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\x89"		=> "\x97\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x99"		=> "\x97\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xab"		=> "\x97\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x92"		=> "\x97\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x86"		=> "\x97\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x9c"		=> "\x97\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbe\x8d"		=> "\x97\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xb6"		=> "\x97\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xae"		=> "\x97\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x85"		=> "\x97\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x9c"		=> "\x97\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x86"		=> "\x97\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xae"		=> "\x97\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x9a"		=> "\x97\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xa1"		=> "\x97\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x8c"		=> "\x97\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xae"		=> "\x97\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x99"		=> "\x97\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x81"		=> "\x97\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\xbc"		=> "\x97\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\x9f"		=> "\x97\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x82"		=> "\x97\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xad"		=> "\x97\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x9c"		=> "\x97\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xa7"		=> "\x97\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xaf"		=> "\x97\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x92"		=> "\x97\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xbc"		=> "\x97\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x8f"		=> "\x97\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xb5"		=> "\x97\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x98"		=> "\x97\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x9b"		=> "\x97\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x91"		=> "\x97\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xab"		=> "\x97\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\x98"		=> "\x97\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x97"		=> "\x97\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x8b"		=> "\x97\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x90"		=> "\x97\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xb3"		=> "\x97\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xa8"		=> "\x97\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xaa"		=> "\x97\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xa3"		=> "\x97\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\x97"		=> "\x97\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x9f"		=> "\x97\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xa0"		=> "\x97\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x81"		=> "\x97\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x99"		=> "\x97\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xaf"		=> "\x97\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x9e"		=> "\x97\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xa4"		=> "\x97\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xb6"		=> "\x97\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x8b"		=> "\x97\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xb7"		=> "\x97\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xb1"		=> "\x97\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xba"		=> "\x97\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x9c"		=> "\x97\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\xb2"		=> "\x97\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xbc"		=> "\x97\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x93"		=> "\x97\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xb4"		=> "\x97\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xb7"		=> "\x97\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xb6"		=> "\x97\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x8a"		=> "\x97\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x97"		=> "\x97\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xa2"		=> "\x97\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xa6"		=> "\x97\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xb4"		=> "\x97\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x97"		=> "\x97\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xa3"		=> "\x97\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\x88"		=> "\x97\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x82"		=> "\x97\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x89"		=> "\x97\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x8b"		=> "\x97\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x90"		=> "\x97\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xa3"		=> "\x97\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x89"		=> "\x97\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xbe"		=> "\x97\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xb4"		=> "\x97\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xaf"		=> "\x97\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\xae"		=> "\x98\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xa3"		=> "\x98\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xac"		=> "\x98\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x82"		=> "\x98\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\xaf"		=> "\x98\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x93"		=> "\x98\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\x89"		=> "\x98\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x82"		=> "\x98\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xaf"		=> "\x98\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xb2"		=> "\x98\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xb4"		=> "\x98\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\x81"		=> "\x98\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x8a"		=> "\x98\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x84"		=> "\x98\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x97"		=> "\x98\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xbc"		=> "\x98\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x94"		=> "\x98\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xaa"		=> "\x98\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x8f"		=> "\x98\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xa2"		=> "\x98\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xbc"		=> "\x98\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xad"		=> "\x98\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x81"		=> "\x98\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xbe"		=> "\x98\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x8b"		=> "\x98\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\x8e"		=> "\x98\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xad"		=> "\x98\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x93"		=> "\x98\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x84"		=> "\x98\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x8b"		=> "\x98\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xb2"		=> "\x98\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x96"		=> "\x98\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xad"		=> "\x98\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x8c"		=> "\x98\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xb1"		=> "\x98\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xaa"		=> "\x98\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x84"		=> "\x98\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x87"		=> "\x98\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x91"		=> "\x98\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xa0"		=> "\x98\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\xb2"		=> "\x98\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x99"		=> "\x98\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x98"		=> "\x98\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x90"		=> "\x98\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xab"		=> "\x98\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\x81"		=> "\x98\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xa8"		=> "\x98\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x80"		=> "\x98\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\xbe"		=> "\x98\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x97"		=> "\x98\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x95"		=> "\x98\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x8c"		=> "\x98\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x90"		=> "\x98\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x95"		=> "\x98\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xaa"		=> "\x98\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xb1"		=> "\x98\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xb6"		=> "\x98\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xbc"		=> "\x98\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xbf"		=> "\x98\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x82"		=> "\x98\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x96"		=> "\x98\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x98"		=> "\x98\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x82"		=> "\x98\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x85"		=> "\x98\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xab"		=> "\x98\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x8a"		=> "\x98\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x92"		=> "\x98\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x8d"		=> "\x98\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x8e"		=> "\x98\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x9e"		=> "\x98\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\x9f"		=> "\x98\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xa0"		=> "\x98\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xa2"		=> "\x98\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xb0"		=> "\x98\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xb3"		=> "\x98\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba\xb6"		=> "\x98\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x8e"		=> "\x98\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x8d"		=> "\x98\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x84"		=> "\x98\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x86"		=> "\x98\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x82"		=> "\x98\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x97"		=> "\x98\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x9e"		=> "\x98\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xad"		=> "\x98\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\x9f"		=> "\x98\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xb7"		=> "\x98\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x89"		=> "\x98\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x9a"		=> "\x98\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xb0"		=> "\x98\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x9b"		=> "\x98\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x9d"		=> "\x98\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x97"		=> "\x98\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x87"		=> "\x98\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xb6"		=> "\x98\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x88"		=> "\x98\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x8f"		=> "\x98\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x98"		=> "\x98\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xbb"		=> "\x98\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xa9"		=> "\x98\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xb0"		=> "\x98\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x91"		=> "\x98\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\xaf"		=> "\x98\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x86"		=> "\x98\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x96"		=> "\x98\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x98"		=> "\x98\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x94"		=> "\x98\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x9f"		=> "\x98\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x8e"		=> "\x98\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x98"		=> "\x98\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x9b"		=> "\x98\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x91"		=> "\x98\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x9a"		=> "\x98\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x90"		=> "\x98\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xa4"		=> "\x98\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xa5"		=> "\x98\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x9a"		=> "\x98\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa8"		=> "\x98\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x94"		=> "\x98\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xaa"		=> "\x98\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa5"		=> "\x98\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x85"		=> "\x98\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x9c"		=> "\x98\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xb6"		=> "\x98\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa1"		=> "\x98\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa9"		=> "\x98\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xac"		=> "\x98\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xbe"		=> "\x98\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xaf"		=> "\x98\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x91"		=> "\x98\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x86"		=> "\x98\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x83"		=> "\x98\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x87"		=> "\x98\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x83"		=> "\x98\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x95"		=> "\x98\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x90"		=> "\x98\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x88"		=> "\x98\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x9a"		=> "\x98\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x96"		=> "\x98\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xac"		=> "\x98\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xb8"		=> "\x98\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x80"		=> "\x98\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x9a"		=> "\x98\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x85"		=> "\x98\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\xb4"		=> "\x98\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\xb2"		=> "\x98\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x89"		=> "\x99\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x8a"		=> "\x99\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\xb3"		=> "\x99\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x82"		=> "\x99\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x96"		=> "\x99\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x9e"		=> "\x99\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xa5"		=> "\x99\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xad"		=> "\x99\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xa3"		=> "\x99\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xae"		=> "\x99\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xb9"		=> "\x99\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xb5"		=> "\x99\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x89"		=> "\x99\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x81"		=> "\x99\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x82"		=> "\x99\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x96"		=> "\x99\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x95"		=> "\x99\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x94"		=> "\x99\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\x9a"		=> "\x99\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\xa1"		=> "\x99\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\xba"		=> "\x99\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\xb7"		=> "\x99\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\xbc"		=> "\x99\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\xbb"		=> "\x99\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84\xbf"		=> "\x99\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x80"		=> "\x99\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x92"		=> "\x99\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x8c"		=> "\x99\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x94"		=> "\x99\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xa2"		=> "\x99\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xb8"		=> "\x99\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xa9"		=> "\x99\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xaa"		=> "\x99\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xae"		=> "\x99\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x80"		=> "\x99\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x82"		=> "\x99\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x98"		=> "\x99\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x8c"		=> "\x99\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x89"		=> "\x99\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x8f"		=> "\x99\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x91"		=> "\x99\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x93"		=> "\x99\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x95"		=> "\x99\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x96"		=> "\x99\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xa4"		=> "\x99\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xa6"		=> "\x99\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xa2"		=> "\x99\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xa9"		=> "\x99\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xaa"		=> "\x99\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xab"		=> "\x99\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xb3"		=> "\x99\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xb1"		=> "\x99\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xb2"		=> "\x99\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xb0"		=> "\x99\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xb5"		=> "\x99\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xbd"		=> "\x99\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x85"		=> "\x99\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x89"		=> "\x99\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x9b"		=> "\x99\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xa0"		=> "\x99\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x95"		=> "\x99\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xa9"		=> "\x99\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xad"		=> "\x99\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xb0"		=> "\x99\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xb5"		=> "\x99\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xbe"		=> "\x99\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x84"		=> "\x99\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x8b"		=> "\x99\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x94"		=> "\x99\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x8e"		=> "\x99\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xa7"		=> "\x99\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xaa"		=> "\x99\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xae"		=> "\x99\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xb3"		=> "\x99\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\xb9"		=> "\x99\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x8f"		=> "\x99\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x84"		=> "\x99\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x8b"		=> "\x99\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x8c"		=> "\x99\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x9e"		=> "\x99\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\x94"		=> "\x99\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xaa"		=> "\x99\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xb4"		=> "\x99\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xa9"		=> "\x99\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xb3"		=> "\x99\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xbf"		=> "\x99\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xbd"		=> "\x99\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x8d"		=> "\x99\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x94"		=> "\x99\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x92"		=> "\x99\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89\xb1"		=> "\x99\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x88"		=> "\x99\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x91"		=> "\x99\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xa8"		=> "\x99\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xa7"		=> "\x99\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xac"		=> "\x99\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xad"		=> "\x99\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xbc"		=> "\x99\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xb5"		=> "\x99\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x81"		=> "\x99\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x8d"		=> "\x99\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x97"		=> "\x99\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x9e"		=> "\x99\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xa3"		=> "\x99\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xa6"		=> "\x99\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xad"		=> "\x99\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xa0"		=> "\x99\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xb3"		=> "\x99\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xb5"		=> "\x99\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xb8"		=> "\x99\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\xb9"		=> "\x99\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x86"		=> "\x99\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x88"		=> "\x99\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xb8"		=> "\x99\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x8d"		=> "\x99\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x90"		=> "\x99\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x8f"		=> "\x99\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x95"		=> "\x99\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x9a"		=> "\x99\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xa3"		=> "\x99\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xaf"		=> "\x99\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xb1"		=> "\x99\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xb3"		=> "\x99\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xb8"		=> "\x99\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x80"		=> "\x99\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x86"		=> "\x99\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x85"		=> "\x99\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\x97"		=> "\x99\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x89"		=> "\x99\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x8d"		=> "\x99\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x96"		=> "\x99\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\x9e"		=> "\x99\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xa9"		=> "\x99\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xae"		=> "\x99\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x98"		=> "\x99\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xbb"		=> "\x99\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb7"		=> "\x99\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\x82"		=> "\x99\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\x96"		=> "\x99\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xa0"		=> "\x99\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xa6"		=> "\x99\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xa5"		=> "\x99\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xae"		=> "\x99\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xb0"		=> "\x99\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xb6"		=> "\x99\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x83"		=> "\x99\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x92"		=> "\x99\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x99"		=> "\x99\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x9f"		=> "\x99\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xbc"		=> "\x99\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xae"		=> "\x99\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xae"		=> "\x99\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xa8"		=> "\x99\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xad"		=> "\x99\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\xba"		=> "\x99\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x81"		=> "\x99\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xbd"		=> "\x99\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x80"		=> "\x99\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xac"		=> "\x99\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xad"		=> "\x99\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xbc"		=> "\x99\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xae"		=> "\x99\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xb6"		=> "\x99\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\xa9"		=> "\x99\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90\x9d"		=> "\x99\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x8e"		=> "\x99\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x8f"		=> "\x99\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xb5"		=> "\x99\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x8e"		=> "\x99\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\x9f"		=> "\x99\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xb1"		=> "\x99\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xb7"		=> "\x99\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xb0"		=> "\x99\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x92"		=> "\x99\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xbb"		=> "\x99\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x80"		=> "\x99\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91\xb6"		=> "\x99\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x84"		=> "\x99\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x90"		=> "\x99\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x86"		=> "\x99\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\x87"		=> "\x99\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xa2"		=> "\x99\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xb8"		=> "\x99\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xa5"		=> "\x99\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xac"		=> "\x99\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\x84"		=> "\x99\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\x88"		=> "\x99\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xa8"		=> "\x99\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xab"		=> "\x9a\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\x82"		=> "\x9a\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xa4"		=> "\x9a\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xbe"		=> "\x9a\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xbc"		=> "\x9a\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\x98"		=> "\x9a\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xa5"		=> "\x9a\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xa6"		=> "\x9a\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\x8f"		=> "\x9a\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\x94"		=> "\x9a\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xbd"		=> "\x9a\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xae"		=> "\x9a\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xad"		=> "\x9a\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xba"		=> "\x9a\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xa2"		=> "\x9a\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\xb9"		=> "\x9a\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x80"		=> "\x9a\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\xa3"		=> "\x9a\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x8c"		=> "\x9a\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\xae"		=> "\x9a\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x9c"		=> "\x9a\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x85"		=> "\x9a\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x96"		=> "\x9a\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x97"		=> "\x9a\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\xb8"		=> "\x9a\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94\xb3"		=> "\x9a\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\x9d"		=> "\x9a\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x99"		=> "\x9a\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x80"		=> "\x9a\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xaf"		=> "\x9a\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x8a"		=> "\x9a\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x9f"		=> "\x9a\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\xbb"		=> "\x9a\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\xbe"		=> "\x9a\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x98"		=> "\x9a\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x9e"		=> "\x9a\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xae"		=> "\x9a\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95\xbc"		=> "\x9a\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x83"		=> "\x9a\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xa9"		=> "\x9a\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x87"		=> "\x9a\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\xa8"		=> "\x9a\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\x9a"		=> "\x9a\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\x85"		=> "\x9a\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\x9f"		=> "\x9a\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\x84"		=> "\x9a\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\x9c"		=> "\x9a\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\xa4"		=> "\x9a\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\x94"		=> "\x9a\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\x94"		=> "\x9a\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\xb7"		=> "\x9a\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\x96"		=> "\x9a\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\xbe"		=> "\x9a\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\xbd"		=> "\x9a\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\x9b"		=> "\x9a\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\xb9"		=> "\x9a\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\x8e"		=> "\x9a\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\x90"		=> "\x9a\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x9f"		=> "\x9a\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\xb4"		=> "\x9a\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\xb6"		=> "\x9a\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\xb2"		=> "\x9a\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\xb8"		=> "\x9a\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\xab"		=> "\x9a\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\xa4"		=> "\x9a\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98\xaf"		=> "\x9a\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\xac"		=> "\x9a\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99\xaa"		=> "\x9a\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\x86"		=> "\x9a\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\x80"		=> "\x9a\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\x8a"		=> "\x9a\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\xa0"		=> "\x9a\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\x94"		=> "\x9a\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\x8f"		=> "\x9a\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\xa5"		=> "\x9a\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\xae"		=> "\x9a\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\xb6"		=> "\x9a\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\xb4"		=> "\x9a\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x82"		=> "\x9a\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a\xbc"		=> "\x9a\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x81"		=> "\x9a\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x83"		=> "\x9a\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x80"		=> "\x9a\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x88"		=> "\x9a\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x8e"		=> "\x9a\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x91"		=> "\x9a\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x93"		=> "\x9a\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\x97"		=> "\x9a\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xae"		=> "\x9a\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xb9"		=> "\x9a\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x80"		=> "\x9a\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b\xbf"		=> "\x9a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x84"		=> "\x9a\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x89"		=> "\x9a\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x88"		=> "\x9a\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x8b"		=> "\x9a\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x8d"		=> "\x9a\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x93"		=> "\x9a\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x98"		=> "\x9a\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x96"		=> "\x9a\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97\x87"		=> "\x9a\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\x9c"		=> "\x9a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\xa6"		=> "\x9a\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\xb7"		=> "\x9a\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\xb8"		=> "\x9a\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x8e"		=> "\x9a\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c\xbb"		=> "\x9a\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x80"		=> "\x9a\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x8f"		=> "\x9a\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\xa9"		=> "\x9a\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x80"		=> "\x9a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\x88"		=> "\x9a\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\xa1"		=> "\x9a\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\xbf"		=> "\x9a\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\x89"		=> "\x9a\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\x93"		=> "\x9a\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xa0"		=> "\x9a\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xb3"		=> "\x9a\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xa4"		=> "\x9a\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xaa"		=> "\x9a\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xb0"		=> "\x9a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x83"		=> "\x9a\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x86"		=> "\x9a\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x94"		=> "\x9a\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x92"		=> "\x9a\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x93"		=> "\x9a\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x8a"		=> "\x9a\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x96"		=> "\x9a\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\xa3"		=> "\x9a\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x8b"		=> "\x9a\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x99"		=> "\x9a\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\x9d"		=> "\x9a\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\xb2"		=> "\x9a\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xa1"		=> "\x9a\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\xa2"		=> "\x9a\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x8b"		=> "\x9a\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\xb0"		=> "\x9a\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x80"		=> "\x9a\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\x92"		=> "\x9a\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xbd"		=> "\x9a\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1\xb9"		=> "\x9a\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\x85"		=> "\x9a\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xb9"		=> "\x9a\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\x9f"		=> "\x9a\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xab"		=> "\x9a\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xba"		=> "\x9a\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x9e"		=> "\x9a\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xbb"		=> "\x9a\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xb8"		=> "\x9a\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xae"		=> "\x9a\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x85"		=> "\x9a\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x93"		=> "\x9a\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x91"		=> "\x9a\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x97"		=> "\x9a\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x99"		=> "\x9a\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x98"		=> "\x9a\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xa5"		=> "\x9a\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x9c"		=> "\x9a\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xa4"		=> "\x9a\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\x9f"		=> "\x9a\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xaf"		=> "\x9a\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xba"		=> "\x9a\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xb9"		=> "\x9a\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xbb"		=> "\x9a\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xbc"		=> "\x9a\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3\xbd"		=> "\x9a\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x82"		=> "\x9a\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x8a"		=> "\x9a\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x90"		=> "\x9a\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x9b"		=> "\x9a\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xa6"		=> "\x9a\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xa5"		=> "\x9a\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xac"		=> "\x9a\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xad"		=> "\x9a\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xb2"		=> "\x9a\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xb8"		=> "\x9a\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\xbe"		=> "\x9a\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x92"		=> "\x9a\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x95"		=> "\x9a\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x90"		=> "\x9a\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x8e"		=> "\x9a\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x9a"		=> "\x9a\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x98"		=> "\x9a\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xa2"		=> "\x9a\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xa0"		=> "\x9a\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xa7"		=> "\x9a\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xac"		=> "\x9a\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xa9"		=> "\x9a\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xb8"		=> "\x9b\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x81"		=> "\x9b\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x9d"		=> "\x9b\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x9e"		=> "\x9b\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\xab"		=> "\x9b\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xa3"		=> "\x9b\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xb2"		=> "\x9b\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x86"		=> "\x9b\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\xa8"		=> "\x9b\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x9c"		=> "\x9b\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x8d"		=> "\x9b\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x99"		=> "\x9b\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7\x9a"		=> "\x9b\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\xa5"		=> "\x9b\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\x9f"		=> "\x9b\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\x91"		=> "\x9b\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\x9c"		=> "\x9b\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\x89"		=> "\x9b\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\x9a"		=> "\x9b\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\x80"		=> "\x9b\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\xac"		=> "\x9b\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\x89"		=> "\x9b\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\xb5"		=> "\x9b\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8\xb6"		=> "\x9b\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\xa2"		=> "\x9b\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9\xaa"		=> "\x9b\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaa\x9a"		=> "\x9b\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaa\xbc"		=> "\x9b\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaa\xbe"		=> "\x9b\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\x8b"		=> "\x9b\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\x82"		=> "\x9b\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaa\xbd"		=> "\x9b\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\xa3"		=> "\x9b\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\x97"		=> "\x9b\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\xa6"		=> "\x9b\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\xa9"		=> "\x9b\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\x96"		=> "\x9b\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\xba"		=> "\x9b\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\xbb"		=> "\x9b\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\x8c"		=> "\x9b\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\x8b"		=> "\x9b\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\x96"		=> "\x9b\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\xb2"		=> "\x9b\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab\x90"		=> "\x9b\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\xaa"		=> "\x9b\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\xb6"		=> "\x9b\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac\xbe"		=> "\x9b\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x83"		=> "\x9b\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x85"		=> "\x9b\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x80"		=> "\x9b\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x91"		=> "\x9b\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x95"		=> "\x9b\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x9a"		=> "\x9b\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x9b"		=> "\x9b\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xa5"		=> "\x9b\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xa9"		=> "\x9b\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xb0"		=> "\x9b\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xb3"		=> "\x9b\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xb5"		=> "\x9b\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xb8"		=> "\x9b\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x88"		=> "\x9b\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xba"		=> "\x9b\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x80"		=> "\x9b\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\x83"		=> "\x9b\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xa6"		=> "\x9b\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae\xb8"		=> "\x9b\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x83"		=> "\x9b\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x87"		=> "\x9b\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x89"		=> "\x9b\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x94"		=> "\x9b\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x90"		=> "\x9b\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xa4"		=> "\x9b\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xa6"		=> "\x9b\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xa2"		=> "\x9b\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x9e"		=> "\x9b\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xa5"		=> "\x9b\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xab"		=> "\x9b\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xb0"		=> "\x9b\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xb6"		=> "\x9b\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xb3"		=> "\x9b\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x85"		=> "\x9b\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x87"		=> "\x9b\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x88"		=> "\x9b\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x8d"		=> "\x9b\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x93"		=> "\x9b\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xa0"		=> "\x9b\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xa2"		=> "\x9b\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xa8"		=> "\x9b\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xb8"		=> "\x9b\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\xb9"		=> "\x9b\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x81"		=> "\x9b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x86"		=> "\x9b\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x8e"		=> "\x9b\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x93"		=> "\x9b\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x90"		=> "\x9b\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\x8f"		=> "\x9b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\xb1"		=> "\x9b\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xac"		=> "\x9b\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xae"		=> "\x9b\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\xa2"		=> "\x9b\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xb6"		=> "\x9b\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1\xb9"		=> "\x9b\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\x8c"		=> "\x9b\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\x91"		=> "\x9b\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\x94"		=> "\x9b\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\x9b"		=> "\x9b\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xab"		=> "\x9b\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xbb"		=> "\x9b\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xb6"		=> "\x9b\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xbc"		=> "\x9b\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xb7"		=> "\x9b\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\x85"		=> "\x9b\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xbe"		=> "\x9b\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\x87"		=> "\x9b\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\x99"		=> "\x9b\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xa9"		=> "\x9b\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xbd"		=> "\x9b\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xba"		=> "\x9b\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xad"		=> "\x9b\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\x8c"		=> "\x9b\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xaa"		=> "\x9b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x8b"		=> "\x9b\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x95"		=> "\x9b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x97"		=> "\x9b\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\x9c"		=> "\x9b\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x9f"		=> "\x9b\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x9b"		=> "\x9b\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x91"		=> "\x9b\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x94"		=> "\x9b\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\xa2"		=> "\x9b\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x9a"		=> "\x9b\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x99"		=> "\x9b\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\x98"		=> "\x9b\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\x8c"		=> "\x9b\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\x92"		=> "\x9b\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\x8e"		=> "\x9b\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\x8b"		=> "\x9b\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\xac"		=> "\x9b\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\xb3"		=> "\x9b\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\xb6"		=> "\x9b\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\x87"		=> "\x9b\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\x84"		=> "\x9b\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\x82"		=> "\x9b\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xa2"		=> "\x9b\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\x9d"		=> "\x9b\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xac"		=> "\x9b\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xae"		=> "\x9b\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xbd"		=> "\x9b\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\x90"		=> "\x9b\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xb7"		=> "\x9b\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xbc"		=> "\x9b\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x89"		=> "\x9b\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x8d"		=> "\x9b\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x93"		=> "\x9b\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x92"		=> "\x9b\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x96"		=> "\x9b\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x9b"		=> "\x9b\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xab"		=> "\x9b\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xb2"		=> "\x9b\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\xb5"		=> "\x9b\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x8b"		=> "\x9b\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x9a"		=> "\x9b\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x99"		=> "\x9b\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x91"		=> "\x9b\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\x9b"		=> "\x9b\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xb6"		=> "\x9b\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8\xb7"		=> "\x9b\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x84"		=> "\x9b\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x83"		=> "\x9b\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x80"		=> "\x9b\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x8e"		=> "\x9b\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x97"		=> "\x9b\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x94"		=> "\x9b\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x9f"		=> "\x9b\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xa2"		=> "\x9b\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xa4"		=> "\x9b\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\x87"		=> "\x9b\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xb5"		=> "\x9b\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xb6"		=> "\x9b\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xba"		=> "\x9b\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xbc"		=> "\x9b\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9\xbf"		=> "\x9b\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba\xa0"		=> "\x9b\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x81"		=> "\x9b\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x82"		=> "\x9b\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x88"		=> "\x9b\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x90"		=> "\x9b\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x8f"		=> "\x9b\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x96"		=> "\x9c\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xa3"		=> "\x9c\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x9d"		=> "\x9c\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x9a"		=> "\x9c\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\x9b"		=> "\x9c\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xa2"		=> "\x9c\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xa1"		=> "\x9c\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xa8"		=> "\x9c\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xa9"		=> "\x9c\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xac"		=> "\x9c\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xb1"		=> "\x9c\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xb3"		=> "\x9c\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xb0"		=> "\x9c\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xb4"		=> "\x9c\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xb8"		=> "\x9c\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb\xbe"		=> "\x9c\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x83"		=> "\x9c\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x89"		=> "\x9c\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x9d"		=> "\x9c\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x9c"		=> "\x9c\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x8b"		=> "\x9c\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x91"		=> "\x9c\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\x96"		=> "\x9c\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xa9"		=> "\x9c\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xad"		=> "\x9c\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xb8"		=> "\x9c\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x81"		=> "\x9c\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x88"		=> "\x9c\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x8c"		=> "\x9c\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x8e"		=> "\x9c\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xaf"		=> "\x9c\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x91"		=> "\x9c\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x96"		=> "\x9c\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x97"		=> "\x9c\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x99"		=> "\x9c\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xa1"		=> "\x9c\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xad"		=> "\x9c\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xb3"		=> "\x9c\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xb7"		=> "\x9c\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x83"		=> "\x9c\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x82"		=> "\x9c\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xbf"		=> "\x9c\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x8a"		=> "\x9c\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x88"		=> "\x9c\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x91"		=> "\x9c\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x87"		=> "\x9c\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x9e"		=> "\x9c\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x99"		=> "\x9c\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\x98"		=> "\x9c\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xa0"		=> "\x9c\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xa8"		=> "\x9c\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xad"		=> "\x9c\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xbc"		=> "\x9c\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x96"		=> "\x9c\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xbb"		=> "\x9c\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xa4"		=> "\x9c\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xb8"		=> "\x9c\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xb1"		=> "\x9c\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x9d"		=> "\x9c\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xb3"		=> "\x9c\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xbf"		=> "\x9c\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xa1"		=> "\x9c\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xa0"		=> "\x9c\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x99"		=> "\x9c\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x90"		=> "\x9c\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xa9"		=> "\x9c\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x8e"		=> "\x9c\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xb1"		=> "\x9c\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x9b"		=> "\x9c\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x95"		=> "\x9c\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xab"		=> "\x9c\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xa6"		=> "\x9c\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\x8f"		=> "\x9c\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80\xba"		=> "\x9c\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x9a"		=> "\x9c\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x81"		=> "\x9c\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xaa"		=> "\x9c\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xb7"		=> "\x9c\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x9f"		=> "\x9c\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x8a"		=> "\x9c\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x86"		=> "\x9c\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x8d"		=> "\x9c\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xa3"		=> "\x9c\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x83"		=> "\x9c\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xa4"		=> "\x9c\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x82"		=> "\x9c\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xac"		=> "\x9c\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\xab"		=> "\x9c\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x99"		=> "\x9c\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x81"		=> "\x9c\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x8d"		=> "\x9c\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xa7"		=> "\x9c\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x83"		=> "\x9c\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x9a"		=> "\x9c\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x84"		=> "\x9c\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x9b"		=> "\x9c\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x96"		=> "\x9c\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x97"		=> "\x9c\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x92"		=> "\x9c\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xa7"		=> "\x9c\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x8b"		=> "\x9c\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xa1"		=> "\x9c\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xb8"		=> "\x9c\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xa0"		=> "\x9c\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x93"		=> "\x9c\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xb4"		=> "\x9c\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\xb0"		=> "\x9c\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xbd"		=> "\x9c\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x86"		=> "\x9c\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\xb5"		=> "\x9c\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x98"		=> "\x9c\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x8d"		=> "\x9c\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x95"		=> "\x9c\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x86"		=> "\x9c\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb6"		=> "\x9c\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb7"		=> "\x9c\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x80"		=> "\x9c\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb4"		=> "\x9c\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xba"		=> "\x9c\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x83"		=> "\x9c\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xa1"		=> "\x9c\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xbb"		=> "\x9c\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb1"		=> "\x9c\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x8d"		=> "\x9c\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x8e"		=> "\x9c\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x87"		=> "\x9c\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xbe"		=> "\x9c\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xa8"		=> "\x9c\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xa7"		=> "\x9c\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x8a"		=> "\x9c\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xbf"		=> "\x9c\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xbc"		=> "\x9c\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xac"		=> "\x9c\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xb4"		=> "\x9c\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xbd"		=> "\x9c\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x82"		=> "\x9c\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x84"		=> "\x9c\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xb3"		=> "\x9c\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xb7"		=> "\x9c\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x98"		=> "\x9c\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x99"		=> "\x9c\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x9a"		=> "\x9c\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xab"		=> "\x9c\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xb4"		=> "\x9c\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xaf"		=> "\x9c\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xa5"		=> "\x9c\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xb1"		=> "\x9c\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x9f"		=> "\x9c\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x9d"		=> "\x9c\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\x93"		=> "\x9c\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85\xb5"		=> "\x9c\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x99"		=> "\x9c\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x96"		=> "\x9c\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x87"		=> "\x9c\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xac"		=> "\x9c\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x94"		=> "\x9c\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x9a"		=> "\x9c\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x8a"		=> "\x9c\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x91"		=> "\x9c\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xab"		=> "\x9c\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xae"		=> "\x9c\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x8c"		=> "\x9c\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x8a"		=> "\x9c\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x89"		=> "\x9c\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xb7"		=> "\x9c\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x88"		=> "\x9c\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x83"		=> "\x9c\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x86"		=> "\x9c\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\xba"		=> "\x9c\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x8b"		=> "\x9c\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xb9"		=> "\x9c\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\x8d"		=> "\x9c\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xa6"		=> "\x9c\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xa3"		=> "\x9c\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xb6"		=> "\x9c\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xba"		=> "\x9c\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xb4"		=> "\x9c\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xbf"		=> "\x9c\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xbd"		=> "\x9c\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xbc"		=> "\x9c\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87\xbe"		=> "\x9c\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x80"		=> "\x9c\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x88"		=> "\x9c\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x89"		=> "\x9c\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x8d"		=> "\x9c\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x8c"		=> "\x9c\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x94"		=> "\x9c\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x9b"		=> "\x9c\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x9e"		=> "\x9d\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xa1"		=> "\x9d\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xaa"		=> "\x9d\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xae"		=> "\x9d\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xb0"		=> "\x9d\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xb2"		=> "\x9d\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\xb3"		=> "\x9d\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x81"		=> "\x9d\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x8e"		=> "\x9d\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x9e"		=> "\x9d\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xa3"		=> "\x9d\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x9b"		=> "\x9d\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xa0"		=> "\x9d\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xa8"		=> "\x9d\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xbc"		=> "\x9d\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x82"		=> "\x9d\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x89"		=> "\x9d\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\xbe"		=> "\x9d\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x92"		=> "\x9d\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x93"		=> "\x9d\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x96"		=> "\x9d\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x94"		=> "\x9d\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x83"		=> "\x9d\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x94"		=> "\x9d\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x97"		=> "\x9d\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x91"		=> "\x9d\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xbb"		=> "\x9d\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x8f"		=> "\x9d\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xbf"		=> "\x9d\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x86"		=> "\x9d\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x94"		=> "\x9d\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x88"		=> "\x9d\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x9c"		=> "\x9d\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x8c"		=> "\x9d\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x8a"		=> "\x9d\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x82"		=> "\x9d\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x87"		=> "\x9d\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\x9b"		=> "\x9d\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\x89"		=> "\x9d\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x8c"		=> "\x9d\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xae"		=> "\x9d\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xb1"		=> "\x9d\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\xa7"		=> "\x9d\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x82"		=> "\x9d\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\x88"		=> "\x9d\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xaf"		=> "\x9d\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b\xb5"		=> "\x9d\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\x90"		=> "\x9d\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c\xbe"		=> "\x9d\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\x8d"		=> "\x9d\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\x9c"		=> "\x9d\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\x8f"		=> "\x9d\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x96"		=> "\x9d\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x8e"		=> "\x9d\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x80"		=> "\x9d\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xab"		=> "\x9d\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xb6"		=> "\x9d\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xa3"		=> "\x9d\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x8f"		=> "\x9d\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x89"		=> "\x9d\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\x9f"		=> "\x9d\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xb5"		=> "\x9d\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xab"		=> "\x9d\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d\xa9"		=> "\x9d\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e\xbe"		=> "\x9d\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xa9"		=> "\x9d\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x80"		=> "\x9d\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x86"		=> "\x9d\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xa3"		=> "\x9d\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x89"		=> "\x9d\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x92"		=> "\x9d\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xb6"		=> "\x9d\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\x84"		=> "\x9d\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\x96"		=> "\x9d\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\xb4"		=> "\x9d\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\x86"		=> "\x9d\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\x93"		=> "\x9d\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\xa6"		=> "\x9d\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\xb6"		=> "\x9d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\x9d"		=> "\x9d\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\x97"		=> "\x9d\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\xa8"		=> "\x9d\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90\x8f"		=> "\x9d\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\xa7"		=> "\x9d\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\xaf"		=> "\x9d\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\xb6"		=> "\x9d\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\x8e"		=> "\x9d\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xaa"		=> "\x9d\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x95"		=> "\x9d\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x93"		=> "\x9d\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xa5"		=> "\x9d\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xa9"		=> "\x9d\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x88"		=> "\x9d\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xbc"		=> "\x9d\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x9a"		=> "\x9d\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x92"		=> "\x9d\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x85"		=> "\x9d\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x87"		=> "\x9d\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\xbb"		=> "\x9d\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x98"		=> "\x9d\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x82"		=> "\x9d\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xb1"		=> "\x9d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xa7"		=> "\x9d\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x89"		=> "\x9d\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xa0"		=> "\x9d\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xa1"		=> "\x9d\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xac"		=> "\x9d\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xa3"		=> "\x9d\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xaf"		=> "\x9d\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xac"		=> "\x9d\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xb6"		=> "\x9d\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xb4"		=> "\x9d\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xb2"		=> "\x9d\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xba"		=> "\x9d\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\x80"		=> "\x9d\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\xbd"		=> "\x9d\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\x98"		=> "\x9d\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\x9c"		=> "\x9d\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\x85"		=> "\x9d\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xa4"		=> "\x9d\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xa3"		=> "\x9d\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xab"		=> "\x9d\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xb4"		=> "\x9d\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xb5"		=> "\x9d\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xb7"		=> "\x9d\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xb6"		=> "\x9d\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94\xb8"		=> "\x9d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x8b"		=> "\x9d\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x88"		=> "\x9d\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x96"		=> "\x9d\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x95"		=> "\x9d\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x8d"		=> "\x9d\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x98"		=> "\x9d\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x9e"		=> "\x9d\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x9d"		=> "\x9d\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xb2"		=> "\x9d\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\xb8"		=> "\x9d\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x82"		=> "\x9d\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x83"		=> "\x9d\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x8a"		=> "\x9d\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x9b"		=> "\x9d\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\x9f"		=> "\x9d\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xab"		=> "\x9d\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96\xb7"		=> "\x9d\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x83"		=> "\x9d\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x86"		=> "\x9d\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x81"		=> "\x9d\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x84"		=> "\x9d\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x8c"		=> "\x9d\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x92"		=> "\x9d\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x9b"		=> "\x9d\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\x99"		=> "\x9d\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xa0"		=> "\x9d\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xa1"		=> "\x9d\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xb1"		=> "\x9d\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xb2"		=> "\x9d\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x8a"		=> "\x9d\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x83"		=> "\x9d\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97\xbb"		=> "\x9d\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xb3"		=> "\x9d\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xb5"		=> "\x9d\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xb6"		=> "\x9d\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xb4"		=> "\x9d\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x9c"		=> "\x9d\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x8f"		=> "\x9d\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x84"		=> "\x9d\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x89"		=> "\x9d\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x81"		=> "\x9d\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x9e"		=> "\x9d\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x9d"		=> "\x9d\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xa4"		=> "\x9d\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xa7"		=> "\x9d\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xa8"		=> "\x9d\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x9f"		=> "\x9d\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xa2"		=> "\x9d\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xb0"		=> "\x9d\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x83"		=> "\x9d\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x88"		=> "\x9d\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x8e"		=> "\x9d\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x89"		=> "\x9d\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x84"		=> "\x9d\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x98"		=> "\x9d\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x9d"		=> "\x9d\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x81"		=> "\x9d\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xb9"		=> "\x9d\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x89"		=> "\x9d\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xbe"		=> "\x9d\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xbc"		=> "\x9d\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x84"		=> "\x9e\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xb8"		=> "\x9e\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x96"		=> "\x9e\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\x9a"		=> "\x9e\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xa0"		=> "\x9e\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xbf"		=> "\x9e\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xa6"		=> "\x9e\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xa9"		=> "\x9e\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xb0"		=> "\x9e\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xb5"		=> "\x9e\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xb7"		=> "\x9e\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x8f"		=> "\x9e\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x96"		=> "\x9e\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x9e"		=> "\x9e\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xa6"		=> "\x9e\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xa7"		=> "\x9e\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xb8"		=> "\x9e\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xae"		=> "\x9e\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xbf"		=> "\x9e\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xb6"		=> "\x9e\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x81"		=> "\x9e\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xb8"		=> "\x9e\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\xb7"		=> "\x9e\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x86"		=> "\x9e\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x9e"		=> "\x9e\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xa0"		=> "\x9e\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\x99"		=> "\x9e\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xa3"		=> "\x9e\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xa4"		=> "\x9e\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x89"		=> "\x9e\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xb0"		=> "\x9e\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xa9"		=> "\x9e\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xbc"		=> "\x9e\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d\xaa"		=> "\x9e\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x8c"		=> "\x9e\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x8b"		=> "\x9e\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xa6"		=> "\x9e\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xa1"		=> "\x9e\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\x85"		=> "\x9e\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xb7"		=> "\x9e\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xaf"		=> "\x9e\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xb4"		=> "\x9e\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xac"		=> "\x9e\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xb3"		=> "\x9e\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xa9"		=> "\x9e\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xb8"		=> "\x9e\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xa4"		=> "\x9e\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x9e"		=> "\x9e\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x9d"		=> "\x9e\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xa2"		=> "\x9e\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xae"		=> "\x9e\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xb9"		=> "\x9e\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x8e"		=> "\x9e\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x86"		=> "\x9e\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\xa7"		=> "\x9e\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\x9c"		=> "\x9e\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x9e"		=> "\x9e\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x86"		=> "\x9e\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xa9"		=> "\x9e\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x80"		=> "\x9e\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x8d"		=> "\x9e\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xb2"		=> "\x9e\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x8e"		=> "\x9e\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xb3"		=> "\x9e\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\xab"		=> "\x9e\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x99"		=> "\x9e\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\xa3"		=> "\x9e\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\xb7"		=> "\x9e\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\xbf"		=> "\x9e\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x9f"		=> "\x9e\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x8f"		=> "\x9e\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xad"		=> "\x9e\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x94"		=> "\x9e\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x9d"		=> "\x9e\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x9b"		=> "\x9e\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x83"		=> "\x9e\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xae"		=> "\x9e\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xb9"		=> "\x9e\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\xb4"		=> "\x9e\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xb5"		=> "\x9e\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xa0"		=> "\x9e\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\xba"		=> "\x9e\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x8f"		=> "\x9e\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2\x8d"		=> "\x9e\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\xbe"		=> "\x9e\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x81"		=> "\x9e\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x8a"		=> "\x9e\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x88"		=> "\x9e\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x98"		=> "\x9e\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xa2"		=> "\x9e\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xa6"		=> "\x9e\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xa1"		=> "\x9e\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x8c"		=> "\x9e\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x8d"		=> "\x9e\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x94"		=> "\x9e\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xa7"		=> "\x9e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x95"		=> "\x9e\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xb6"		=> "\x9e\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x92"		=> "\x9e\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x84"		=> "\x9e\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x97"		=> "\x9e\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xa3"		=> "\x9e\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xa5"		=> "\x9e\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xb9"		=> "\x9e\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xa0"		=> "\x9e\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\xaf"		=> "\x9e\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xa8"		=> "\x9e\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xaa"		=> "\x9e\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\x9a"		=> "\x9e\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xa3"		=> "\x9e\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xa1"		=> "\x9e\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x86"		=> "\x9e\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xb9"		=> "\x9e\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xb7"		=> "\x9e\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x9c"		=> "\x9e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xb8"		=> "\x9e\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xab"		=> "\x9e\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x94"		=> "\x9e\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xbe"		=> "\x9e\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xae"		=> "\x9e\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xb9"		=> "\x9e\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xb4"		=> "\x9e\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xbd"		=> "\x9e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x99"		=> "\x9e\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4\xb0"		=> "\x9e\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xa1"		=> "\x9e\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x9e"		=> "\x9e\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\x9d"		=> "\x9e\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x81"		=> "\x9e\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5\xaa"		=> "\x9e\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xb2"		=> "\x9e\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xae"		=> "\x9e\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x90"		=> "\x9e\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xbf"		=> "\x9e\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x81"		=> "\x9e\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x93"		=> "\x9e\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xbe"		=> "\x9e\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x8e"		=> "\x9e\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xa8"		=> "\x9e\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x8a"		=> "\x9e\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x9d"		=> "\x9e\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xbb"		=> "\x9e\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x83"		=> "\x9e\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xa7"		=> "\x9e\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xae"		=> "\x9e\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x91"		=> "\x9e\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xa0"		=> "\x9e\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x9c"		=> "\x9e\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\x95"		=> "\x9e\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xb4"		=> "\x9e\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x9e"		=> "\x9e\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xa8"		=> "\x9e\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x82"		=> "\x9e\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x9b"		=> "\x9e\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xbf"		=> "\x9e\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\x8a"		=> "\x9e\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xb9"		=> "\x9e\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xb2"		=> "\x9e\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xa7"		=> "\x9e\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x85"		=> "\x9e\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6\xb1"		=> "\x9e\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x9e"		=> "\x9e\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xad"		=> "\x9e\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x94"		=> "\x9e\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xab"		=> "\x9e\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x8a"		=> "\x9e\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x92"		=> "\x9e\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x81"		=> "\x9e\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xa3"		=> "\x9e\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x93"		=> "\x9e\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x84"		=> "\x9e\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\x8c"		=> "\x9e\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xb2"		=> "\x9e\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xb6"		=> "\x9e\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xb8"		=> "\x9e\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x87"		=> "\x9e\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xa2"		=> "\x9e\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x99"		=> "\x9e\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xa6"		=> "\x9e\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x88"		=> "\x9e\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xb8"		=> "\x9e\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xa2"		=> "\x9e\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\x90"		=> "\x9e\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\x8d"		=> "\x9e\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xa0"		=> "\x9e\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\x84"		=> "\x9e\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xa2"		=> "\x9e\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xa3"		=> "\x9e\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\x97"		=> "\x9f\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x97"		=> "\x9f\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xbb"		=> "\x9f\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x83"		=> "\x9f\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x82"		=> "\x9f\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xb8"		=> "\x9f\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xb3"		=> "\x9f\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xac"		=> "\x9f\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x9e"		=> "\x9f\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x91"		=> "\x9f\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x9f"		=> "\x9f\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa\xaa"		=> "\x9f\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\x9a"		=> "\x9f\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\xaa"		=> "\x9f\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\xbb"		=> "\x9f\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\x85"		=> "\x9f\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x96"		=> "\x9f\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\xba"		=> "\x9f\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\x92"		=> "\x9f\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\x96"		=> "\x9f\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xb1"		=> "\x9f\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\x9f"		=> "\x9f\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xb8"		=> "\x9f\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xb7"		=> "\x9f\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x9c"		=> "\x9f\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac\xb9"		=> "\x9f\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xae"		=> "\x9f\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x87"		=> "\x9f\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x83"		=> "\x9f\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x89"		=> "\x9f\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x90"		=> "\x9f\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x99"		=> "\x9f\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x94"		=> "\x9f\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x9b"		=> "\x9f\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\x9f"		=> "\x9f\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xa1"		=> "\x9f\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xb8"		=> "\x9f\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xb9"		=> "\x9f\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad\xbf"		=> "\x9f\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x80"		=> "\x9f\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x84"		=> "\x9f\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x83"		=> "\x9f\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x8d"		=> "\x9f\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x98"		=> "\x9f\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x95"		=> "\x9f\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\x9e"		=> "\x9f\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xa4"		=> "\x9f\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xaa"		=> "\x9f\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xab"		=> "\x9f\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xaf"		=> "\x9f\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xb2"		=> "\x9f\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xb1"		=> "\x9f\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xb3"		=> "\x9f\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xb7"		=> "\x9f\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae\xbc"		=> "\x9f\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x86"		=> "\x9f\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x8b"		=> "\x9f\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x93"		=> "\x9f\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x9f"		=> "\x9f\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\xac"		=> "\x9f\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\xab"		=> "\x9f\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\xb3"		=> "\x9f\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\xaf"		=> "\x9f\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xbe"		=> "\x9f\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\x88"		=> "\x9f\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\x93"		=> "\x9f\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\x94"		=> "\x9f\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\x9b"		=> "\x9f\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\xa4"		=> "\x9f\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\xa3"		=> "\x9f\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x9e"		=> "\x9f\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x95"		=> "\x9f\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xa2"		=> "\x9f\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xaa"		=> "\x9f\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x82"		=> "\x9f\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x8d"		=> "\x9f\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x9a"		=> "\x9f\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x81"		=> "\x9f\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x9b"		=> "\x9f\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xbe"		=> "\x9f\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xa8"		=> "\x9f\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xb3"		=> "\x9f\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x92"		=> "\x9f\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x90"		=> "\x9f\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x84"		=> "\x9f\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xb1"		=> "\x9f\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x93"		=> "\x9f\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xbd"		=> "\x9f\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x97"		=> "\x9f\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x85"		=> "\x9f\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x9d"		=> "\x9f\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xae"		=> "\x9f\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xb1"		=> "\x9f\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xbe"		=> "\x9f\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\xba"		=> "\x9f\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x9b"		=> "\x9f\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xaf"		=> "\x9f\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x99"		=> "\x9f\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\xaa"		=> "\x9f\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x9f"		=> "\x9f\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x8d"		=> "\x9f\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xb6"		=> "\x9f\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xab"		=> "\x9f\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xbd"		=> "\x9f\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xb8"		=> "\x9f\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x99"		=> "\x9f\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xb5"		=> "\x9f\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\xb3"		=> "\x9f\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x92"		=> "\x9f\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x8c"		=> "\x9f\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xa3"		=> "\x9f\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x93"		=> "\x9f\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xa4"		=> "\x9f\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\x9a"		=> "\x9f\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xb9"		=> "\x9f\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\x99"		=> "\x9f\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x8e"		=> "\x9f\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x95"		=> "\x9f\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xa4"		=> "\x9f\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x85"		=> "\x9f\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb9"		=> "\x9f\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x95"		=> "\x9f\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x8a"		=> "\x9f\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\xb5"		=> "\x9f\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x87"		=> "\x9f\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xa6"		=> "\x9f\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\xb8"		=> "\x9f\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x86"		=> "\x9f\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xac"		=> "\x9f\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x9e"		=> "\x9f\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x8c"		=> "\x9f\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xa8"		=> "\x9f\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x92"		=> "\x9f\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x85"		=> "\x9f\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xba"		=> "\x9f\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x99"		=> "\x9f\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xa4"		=> "\x9f\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x95"		=> "\x9f\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xaa"		=> "\x9f\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xae"		=> "\x9f\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xad"		=> "\x9f\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\xae"		=> "\x9f\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xae"		=> "\x9f\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x99"		=> "\x9f\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\xb2"		=> "\x9f\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x9f"		=> "\x9f\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xbe"		=> "\x9f\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa3"		=> "\x9f\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\xab"		=> "\x9f\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xab"		=> "\x9f\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\xb6"		=> "\x9f\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x8d"		=> "\x9f\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x9f"		=> "\x9f\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x83"		=> "\x9f\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xba"		=> "\x9f\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x8e"		=> "\x9f\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa4"		=> "\x9f\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\xbf"		=> "\x9f\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\x9d"		=> "\x9f\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xb8"		=> "\x9f\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x82"		=> "\x9f\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xaa"		=> "\x9f\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x98"		=> "\x9f\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x89"		=> "\x9f\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xb7"		=> "\x9f\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x93"		=> "\x9f\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xbd"		=> "\x9f\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xaf"		=> "\x9f\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x84"		=> "\x9f\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xb2"		=> "\x9f\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x94"		=> "\x9f\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x95"		=> "\x9f\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x8f"		=> "\x9f\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xa5"		=> "\x9f\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x82"		=> "\x9f\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\x9f"		=> "\x9f\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\x81"		=> "\x9f\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x91"		=> "\x9f\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\x8c"		=> "\x9f\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\xac"		=> "\x9f\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\xb8"		=> "\x9f\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\xbe"		=> "\x9f\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xbf"		=> "\x9f\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\xb2"		=> "\x9f\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xb1"		=> "\x9f\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\xaf"		=> "\x9f\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xb2"		=> "\x9f\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\x8c"		=> "\x9f\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\xbe"		=> "\xe0\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc\x93"		=> "\xe0\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb\xb7"		=> "\xe0\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x86"		=> "\xe0\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xba"		=> "\xe0\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xb8"		=> "\xe0\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x81"		=> "\xe0\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x80"		=> "\xe0\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xaf"		=> "\xe0\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\x9b"		=> "\xe0\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xb3"		=> "\xe0\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xad"		=> "\xe0\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x82"		=> "\xe0\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xbc"		=> "\xe0\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\x98"		=> "\xe0\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x8e"		=> "\xe0\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x91"		=> "\xe0\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x82"		=> "\xe0\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xa6"		=> "\xe0\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xb3"		=> "\xe0\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xa3"		=> "\xe0\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xa1"		=> "\xe0\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xa4"		=> "\xe0\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xb9"		=> "\xe0\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x86"		=> "\xe0\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xaa"		=> "\xe0\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x9f"		=> "\xe0\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x95"		=> "\xe0\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xac"		=> "\xe0\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x94"		=> "\xe0\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x98"		=> "\xe0\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xb1"		=> "\xe0\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xae"		=> "\xe0\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\x9b"		=> "\xe0\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x89"		=> "\xe0\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x8b"		=> "\xe0\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xba"		=> "\xe0\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x91"		=> "\xe0\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x81"		=> "\xe0\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x8f"		=> "\xe0\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xbe"		=> "\xe0\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x9b"		=> "\xe0\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x9a"		=> "\xe0\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd\xb4"		=> "\xe0\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x9d"		=> "\xe0\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x98"		=> "\xe0\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x9f"		=> "\xe0\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\xb0"		=> "\xe0\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\xbe"		=> "\xe0\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\xb2"		=> "\xe0\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\x91"		=> "\xe0\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81\xa3"		=> "\xe0\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\x99"		=> "\xe0\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\x92"		=> "\xe0\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xaf"		=> "\xe0\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\xb1"		=> "\xe0\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xac"		=> "\xe0\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xb8"		=> "\xe0\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xb3"		=> "\xe0\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xae"		=> "\xe0\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\x9f"		=> "\xe0\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\x8b"		=> "\xe0\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\x9d"		=> "\xe0\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\x99"		=> "\xe0\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x89"		=> "\xe0\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83\xbd"		=> "\xe0\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x9c"		=> "\xe0\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x99"		=> "\xe0\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xa5"		=> "\xe0\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x95"		=> "\xe0\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\x88"		=> "\xe0\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xa6"		=> "\xe0\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xa2"		=> "\xe0\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x8c"		=> "\xe0\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x96"		=> "\xe0\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\xac"		=> "\xe0\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\x8f"		=> "\xe0\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xbb"		=> "\xe0\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\x84"		=> "\xe0\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\x95"		=> "\xe0\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\xa8"		=> "\xe0\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\xac"		=> "\xe0\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x97"		=> "\xe0\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\xb9"		=> "\xe0\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\xbe"		=> "\xe0\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x92"		=> "\xe0\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x89"		=> "\xe0\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x94"		=> "\xe0\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\x8e"		=> "\xe0\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xa0"		=> "\xe0\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xac"		=> "\xe0\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xa7"		=> "\xe0\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xb5"		=> "\xe0\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xbc"		=> "\xe0\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xb9"		=> "\xe0\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xbf"		=> "\xe0\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\x8d"		=> "\xe0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\x90"		=> "\xe0\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\x9b"		=> "\xe0\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xa8"		=> "\xe0\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xad"		=> "\xe0\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xac"		=> "\xe0\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xb0"		=> "\xe0\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xb2"		=> "\xe0\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xbb"		=> "\xe0\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xbc"		=> "\xe0\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88\xbf"		=> "\xe0\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x80"		=> "\xe0\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x86"		=> "\xe0\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x8b"		=> "\xe0\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\x98"		=> "\xe0\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xb4"		=> "\xe0\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89\xbe"		=> "\xe0\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\x82"		=> "\xe0\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\x81"		=> "\xe0\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\x87"		=> "\xe0\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\x92"		=> "\xe0\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\x96"		=> "\xe0\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xa2"		=> "\xe0\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xa7"		=> "\xe0\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xb9"		=> "\xe0\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xb2"		=> "\xe0\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x83"		=> "\xe0\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x86"		=> "\xe0\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x84"		=> "\xe0\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x8e"		=> "\xe0\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\x92"		=> "\xe0\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xa2"		=> "\xe0\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xa0"		=> "\xe0\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xa1"		=> "\xe0\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xb9"		=> "\xe0\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b\xb7"		=> "\xe0\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x8f"		=> "\xe0\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\x97"		=> "\xe0\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\x8a"		=> "\xe0\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\x9c"		=> "\xe0\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\x96"		=> "\xe0\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\x9d"		=> "\xe0\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xb4"		=> "\xe0\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xaf"		=> "\xe0\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xa9"		=> "\xe0\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xa5"		=> "\xe0\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xbe"		=> "\xe0\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\x8e"		=> "\xe0\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\x8f"		=> "\xe0\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x98"		=> "\xe0\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\x97"		=> "\xe0\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xaa"		=> "\xe0\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xa8"		=> "\xe0\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xb0"		=> "\xe0\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xb8"		=> "\xe0\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xb5"		=> "\xe0\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xbb"		=> "\xe0\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d\xba"		=> "\xe0\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x88"		=> "\xe0\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\xb3"		=> "\xe0\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x8e"		=> "\xe0\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\xbb"		=> "\xe0\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x80"		=> "\xe0\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xa5"		=> "\xe0\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xae"		=> "\xe0\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x9e"		=> "\xe0\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\xa2"		=> "\xe0\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\x85"		=> "\xe0\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xaf"		=> "\xe0\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xa5"		=> "\xe0\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xb8"		=> "\xe0\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xb2"		=> "\xe0\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xba"		=> "\xe0\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\x95"		=> "\xe0\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xbf"		=> "\xe0\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\x9f"		=> "\xe0\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\x99"		=> "\xe0\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\x81"		=> "\xe0\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\x9c"		=> "\xe0\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xa9"		=> "\xe0\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xb0"		=> "\xe0\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xa3"		=> "\xe0\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xaa"		=> "\xe0\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xb6"		=> "\xe0\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xbe"		=> "\xe0\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\x8b"		=> "\xe0\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\x9e"		=> "\xe0\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\xa7"		=> "\xe0\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\x8a"		=> "\xe0\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\x8f"		=> "\xe0\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\x94"		=> "\xe0\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xb1"		=> "\xe0\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xa0"		=> "\xe1\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xa3"		=> "\xe1\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xa7"		=> "\xe1\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xa9"		=> "\xe1\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xae"		=> "\xe1\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xb2"		=> "\xe1\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xb0"		=> "\xe1\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xb1"		=> "\xe1\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xb8"		=> "\xe1\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93\xb7"		=> "\xe1\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x84"		=> "\xe1\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x83"		=> "\xe1\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x85"		=> "\xe1\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x8c"		=> "\xe1\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x8e"		=> "\xe1\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x8d"		=> "\xe1\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x95"		=> "\xe1\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x93"		=> "\xe1\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x9e"		=> "\xe1\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xa6"		=> "\xe1\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xac"		=> "\xe1\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xbc"		=> "\xe1\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x84"		=> "\xe1\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x8d"		=> "\xe1\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x8a"		=> "\xe1\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x89"		=> "\xe1\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x9b"		=> "\xe1\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x86"		=> "\xe1\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\x9a"		=> "\xe1\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xa9"		=> "\xe1\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xa4"		=> "\xe1\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xa7"		=> "\xe1\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xab"		=> "\xe1\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xad"		=> "\xe1\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xb8"		=> "\xe1\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xb6"		=> "\xe1\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x86"		=> "\xe1\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x87"		=> "\xe1\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xb4"		=> "\xe1\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x8a"		=> "\xe1\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x89"		=> "\xe1\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x82"		=> "\xe1\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x94"		=> "\xe1\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x9a"		=> "\xe1\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\x9d"		=> "\xe1\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xa5"		=> "\xe1\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xa3"		=> "\xe1\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x82"		=> "\xe1\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xb3"		=> "\xe1\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x83"		=> "\xe1\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xb5"		=> "\xe1\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xbd"		=> "\xe1\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xb8"		=> "\xe1\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xbc"		=> "\xe1\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96\xb1"		=> "\xe1\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x8d"		=> "\xe1\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x8a"		=> "\xe1\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x92"		=> "\xe1\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x99"		=> "\xe1\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xa3"		=> "\xe1\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\x9e"		=> "\xe1\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xbe"		=> "\xe1\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xbf"		=> "\xe1\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xbc"		=> "\xe1\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\x81"		=> "\xe1\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xb0"		=> "\xe1\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xba"		=> "\xe1\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xb2"		=> "\xe1\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97\xb3"		=> "\xe1\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\x8b"		=> "\xe1\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\x8d"		=> "\xe1\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\x89"		=> "\xe1\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\x9f"		=> "\xe1\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\xa7"		=> "\xe1\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\xa0"		=> "\xe1\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\xa1"		=> "\xe1\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\xa2"		=> "\xe1\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\xa4"		=> "\xe1\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\xb4"		=> "\xe1\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\xb0"		=> "\xe1\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98\xbb"		=> "\xe1\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x87"		=> "\xe1\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x88"		=> "\xe1\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x86"		=> "\xe1\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x9c"		=> "\xe1\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\x98"		=> "\xe1\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xa1"		=> "\xe1\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xa2"		=> "\xe1\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xa8"		=> "\xe1\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xa9"		=> "\xe1\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xaa"		=> "\xe1\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xa7"		=> "\xe1\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xac"		=> "\xe1\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xb0"		=> "\xe1\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xb2"		=> "\xe1\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xb6"		=> "\xe1\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xb8"		=> "\xe1\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99\xbc"		=> "\xe1\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x80"		=> "\xe1\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x83"		=> "\xe1\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x88"		=> "\xe1\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x8b"		=> "\xe1\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x8e"		=> "\xe1\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x96"		=> "\xe1\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x93"		=> "\xe1\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x99"		=> "\xe1\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x9a"		=> "\xe1\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xb0"		=> "\xe1\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xb4"		=> "\xe1\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xb8"		=> "\xe1\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xb9"		=> "\xe1\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xba"		=> "\xe1\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x82"		=> "\xe1\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x8d"		=> "\xe1\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x96"		=> "\xe1\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x92"		=> "\xe1\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\x9e"		=> "\xe1\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xa1"		=> "\xe1\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xa5"		=> "\xe1\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xa7"		=> "\xe1\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xaa"		=> "\xe1\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\xaf"		=> "\xe1\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b\xbb"		=> "\xe1\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x88"		=> "\xe1\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x87"		=> "\xe1\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x84"		=> "\xe1\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xa9"		=> "\xe1\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xa4"		=> "\xe1\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x9e"		=> "\xe1\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xa5"		=> "\xe1\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xa6"		=> "\xe1\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\x9b"		=> "\xe1\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xb7"		=> "\xe1\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c\xb8"		=> "\xe1\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\x87"		=> "\xe1\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\x9a"		=> "\xe1\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xa8"		=> "\xe1\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xab"		=> "\xe1\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\x9b"		=> "\xe1\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xa5"		=> "\xe1\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xbf"		=> "\xe1\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xbe"		=> "\xe1\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d\xb9"		=> "\xe1\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\x8e"		=> "\xe1\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\x8b"		=> "\xe1\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\x91"		=> "\xe1\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xa0"		=> "\xe1\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\x9e"		=> "\xe1\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xb0"		=> "\xe1\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xb6"		=> "\xe1\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xb9"		=> "\xe1\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xbf"		=> "\xe1\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xbc"		=> "\xe1\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xbd"		=> "\xe1\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e\xbb"		=> "\xe1\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\x87"		=> "\xe1\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\x8d"		=> "\xe1\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\x97"		=> "\xe1\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\x9a"		=> "\xe1\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\x9c"		=> "\xe1\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xa3"		=> "\xe1\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xae"		=> "\xe1\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f\xbc"		=> "\xe1\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\x8c"		=> "\xe1\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\x92"		=> "\xe1\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xa6"		=> "\xe1\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xa0"		=> "\xe1\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xaa"		=> "\xe1\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\x85"		=> "\xe1\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x8e"		=> "\xe1\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xb4"		=> "\xe1\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x86"		=> "\xe1\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xbc"		=> "\xe1\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x9a"		=> "\xe1\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\x8c"		=> "\xe1\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xa3"		=> "\xe1\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xb5"		=> "\xe1\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xaa"		=> "\xe1\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xaf"		=> "\xe1\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x91"		=> "\xe1\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x86"		=> "\xe1\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x8b"		=> "\xe1\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x94"		=> "\xe1\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xbe"		=> "\xe1\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2\xbc"		=> "\xe1\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x85"		=> "\xe1\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x8a"		=> "\xe1\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\xac"		=> "\xe1\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\xa7"		=> "\xe2\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\x9a"		=> "\xe2\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\xbd"		=> "\xe2\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3\xb4"		=> "\xe2\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\x87"		=> "\xe2\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\x92"		=> "\xe2\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\x91"		=> "\xe2\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\x99"		=> "\xe2\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xac"		=> "\xe2\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xab"		=> "\xe2\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x80"		=> "\xe2\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\xa0"		=> "\xe2\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x97"		=> "\xe2\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x9f"		=> "\xe2\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x9a"		=> "\xe2\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x95"		=> "\xe2\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\x93"		=> "\xe2\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\xba"		=> "\xe2\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5\xbf"		=> "\xe2\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x8a"		=> "\xe2\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x9d"		=> "\xe2\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xa7"		=> "\xe2\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\x8b"		=> "\xe2\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xaa"		=> "\xe2\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xae"		=> "\xe2\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xb3"		=> "\xe2\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xb9"		=> "\xe2\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\xba"		=> "\xe2\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x89"		=> "\xe2\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\x95"		=> "\xe2\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xa7"		=> "\xe2\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xac"		=> "\xe2\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xa1"		=> "\xe2\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7\xa3"		=> "\xe2\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x88"		=> "\xe2\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x8d"		=> "\xe2\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x98"		=> "\xe2\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x99"		=> "\xe2\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xa0"		=> "\xe2\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\x9f"		=> "\xe2\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6\x80"		=> "\xe2\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xb1"		=> "\xe2\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xbb"		=> "\xe2\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xbe"		=> "\xe2\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8\xb7"		=> "\xe2\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x83"		=> "\xe2\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x97"		=> "\xe2\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\x89"		=> "\xe2\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xa1"		=> "\xe2\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xa2"		=> "\xe2\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xa9"		=> "\xe2\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbe\x9d"		=> "\xe2\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xb0"		=> "\xe2\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xb9"		=> "\xe2\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9\xbd"		=> "\xe2\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x88"		=> "\xe2\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x97"		=> "\xe2\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x95"		=> "\xe2\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x98"		=> "\xe2\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\x96"		=> "\xe2\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\xa9"		=> "\xe2\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x88"		=> "\xe2\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\xb0"		=> "\xe2\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\xb6"		=> "\xe2\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x85"		=> "\xe2\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x84"		=> "\xe2\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa\xbf"		=> "\xe2\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x83"		=> "\xe2\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x87"		=> "\xe2\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x8a"		=> "\xe2\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x8d"		=> "\xe2\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x8f"		=> "\xe2\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x95"		=> "\xe2\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x93"		=> "\xe2\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x99"		=> "\xe2\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x9a"		=> "\xe2\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x9d"		=> "\xe2\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xa1"		=> "\xe2\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xa2"		=> "\xe2\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xa6"		=> "\xe2\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xad"		=> "\xe2\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xb0"		=> "\xe2\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x82"		=> "\xe2\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x8f"		=> "\xe2\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x8a"		=> "\xe2\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x86"		=> "\xe2\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xb3"		=> "\xe2\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x98"		=> "\xe2\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x99"		=> "\xe2\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x9e"		=> "\xe2\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xb5"		=> "\xe2\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xa8"		=> "\xe2\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\xb6"		=> "\xe2\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x90"		=> "\xe2\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xba"		=> "\xe2\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x84"		=> "\xe2\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x8d"		=> "\xe2\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac\x8b"		=> "\xe2\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x8c"		=> "\xe2\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x85"		=> "\xe2\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xb5"		=> "\xe2\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xa5"		=> "\xe2\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xb4"		=> "\xe2\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xa7"		=> "\xe2\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xb0"		=> "\xe2\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xb1"		=> "\xe2\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xac"		=> "\xe2\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\xae"		=> "\xe2\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x9d"		=> "\xe2\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x98"		=> "\xe2\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x9f"		=> "\xe2\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x8d"		=> "\xe2\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x9c"		=> "\xe2\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x9a"		=> "\xe2\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x8b"		=> "\xe2\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x92"		=> "\xe2\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x8f"		=> "\xe2\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad\x9d"		=> "\xe2\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x99"		=> "\xe2\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x8b"		=> "\xe2\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x81"		=> "\xe2\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x8c"		=> "\xe2\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x8f"		=> "\xe2\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\xb4"		=> "\xe2\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x86"		=> "\xe2\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\x9d"		=> "\xe2\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xa9"		=> "\xe2\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x91"		=> "\xe2\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x94"		=> "\xe2\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xa6"		=> "\xe2\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xa5"		=> "\xe2\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\xa0"		=> "\xe2\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x80"		=> "\xe2\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x87"		=> "\xe2\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x93"		=> "\xe2\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xb3"		=> "\xe2\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xb7"		=> "\xe2\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x97"		=> "\xe2\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x8d"		=> "\xe2\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf\xb6"		=> "\xe2\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xa3"		=> "\xe2\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xa7"		=> "\xe2\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xaa"		=> "\xe2\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\x9f"		=> "\xe2\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xb7"		=> "\xe2\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xab"		=> "\xe2\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0\xbd"		=> "\xe2\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x8c"		=> "\xe2\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x83"		=> "\xe2\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x94"		=> "\xe2\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x8f"		=> "\xe2\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x80"		=> "\xe2\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x90"		=> "\xe2\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x98"		=> "\xe2\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x9f"		=> "\xe2\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\xa4"		=> "\xe2\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\x96"		=> "\xe2\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\xa5"		=> "\xe2\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\xac"		=> "\xe2\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1\xb5"		=> "\xe2\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x83"		=> "\xe2\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\x90"		=> "\xe2\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xa4"		=> "\xe2\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xad"		=> "\xe2\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xa2"		=> "\xe2\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xab"		=> "\xe2\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xa1"		=> "\xe2\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xa8"		=> "\xe2\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xb3"		=> "\xe2\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xb2"		=> "\xe2\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xb1"		=> "\xe2\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xae"		=> "\xe2\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xb9"		=> "\xe2\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2\xbd"		=> "\xe2\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x80"		=> "\xe2\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x85"		=> "\xe2\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x82"		=> "\xe2\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x98"		=> "\xe2\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x92"		=> "\xe2\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\x9c"		=> "\xe2\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xa2"		=> "\xe2\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xbb"		=> "\xe2\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xaf"		=> "\xe2\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xb2"		=> "\xe2\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xb4"		=> "\xe2\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xb6"		=> "\xe2\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3\xba"		=> "\xe2\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x86"		=> "\xe2\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x82"		=> "\xe3\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x9c"		=> "\xe3\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x95"		=> "\xe3\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\x8a"		=> "\xe3\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x85"		=> "\xe3\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x8b"		=> "\xe3\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xae"		=> "\xe3\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xb2"		=> "\xe3\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xbf"		=> "\xe3\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4\xb5"		=> "\xe3\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x86"		=> "\xe3\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xb3"		=> "\xe3\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x96"		=> "\xe3\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x8e"		=> "\xe3\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xb2"		=> "\xe3\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xa8"		=> "\xe3\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xae"		=> "\xe3\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x8f"		=> "\xe3\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xa3"		=> "\xe3\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\x93"		=> "\xe3\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\x89"		=> "\xe3\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x9b"		=> "\xe3\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\x8f"		=> "\xe3\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\xbd"		=> "\xe3\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\x9b"		=> "\xe3\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xba"		=> "\xe3\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xae"		=> "\xe3\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xa3"		=> "\xe3\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xb5"		=> "\xe3\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x87"		=> "\xe3\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xbd"		=> "\xe3\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xab"		=> "\xe3\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xbd"		=> "\xe3\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xa2"		=> "\xe3\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xaf"		=> "\xe3\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x9c"		=> "\xe3\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xb8"		=> "\xe3\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\x9f"		=> "\xe3\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xb0"		=> "\xe3\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x98"		=> "\xe3\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x9d"		=> "\xe3\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xa4"		=> "\xe3\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x9e"		=> "\xe3\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xbb"		=> "\xe3\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xb2"		=> "\xe3\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\xa1"		=> "\xe3\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x85"		=> "\xe3\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x8a"		=> "\xe3\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xa3"		=> "\xe3\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xa1"		=> "\xe3\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x92"		=> "\xe3\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xb1"		=> "\xe3\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x9f"		=> "\xe3\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x89"		=> "\xe3\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\x8b"		=> "\xe3\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xa2"		=> "\xe3\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x86"		=> "\xe3\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xa6"		=> "\xe3\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xbb"		=> "\xe3\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xb5"		=> "\xe3\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xb9"		=> "\xe3\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x83"		=> "\xe3\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xb7"		=> "\xe3\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xb2"		=> "\xe3\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8\xba"		=> "\xe3\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xa7"		=> "\xe3\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x9d"		=> "\xe3\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x96"		=> "\xe3\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x9e"		=> "\xe3\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x99"		=> "\xe3\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x9a"		=> "\xe3\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xb9"		=> "\xe3\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xaa"		=> "\xe3\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xa9"		=> "\xe3\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xbc"		=> "\xe3\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xbb"		=> "\xe3\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x83"		=> "\xe3\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x95"		=> "\xe3\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xbd"		=> "\xe3\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xae"		=> "\xe3\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\xbf"		=> "\xe3\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x88"		=> "\xe3\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x89"		=> "\xe3\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x8c"		=> "\xe3\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x92"		=> "\xe3\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x90"		=> "\xe3\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x93"		=> "\xe3\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x94"		=> "\xe3\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x96"		=> "\xe3\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x8e"		=> "\xe3\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x9b"		=> "\xe3\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x9c"		=> "\xe3\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbc\xb8"		=> "\xe3\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbc\xba"		=> "\xe3\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x85"		=> "\xe3\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x8c"		=> "\xe3\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x8d"		=> "\xe3\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x8e"		=> "\xe3\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x90"		=> "\xe3\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x91"		=> "\xe3\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x95"		=> "\xe3\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x94"		=> "\xe3\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x98"		=> "\xe3\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x9f"		=> "\xe3\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xa0"		=> "\xe3\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xa8"		=> "\xe3\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xa9"		=> "\xe3\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xa7"		=> "\xe3\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\xb8"		=> "\xe3\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x82"		=> "\xe3\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x86"		=> "\xe3\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x83"		=> "\xe3\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x88"		=> "\xe3\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x87"		=> "\xe3\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x8c"		=> "\xe3\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x94"		=> "\xe3\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x9e"		=> "\xe3\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x9d"		=> "\xe3\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\x9a"		=> "\xe3\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xa3"		=> "\xe3\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xaf"		=> "\xe3\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xb2"		=> "\xe3\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xb9"		=> "\xe3\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xae"		=> "\xe3\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xb6"		=> "\xe3\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xb8"		=> "\xe3\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xb1"		=> "\xe3\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\x85"		=> "\xe3\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\x86"		=> "\xe3\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\x8a"		=> "\xe3\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\x95"		=> "\xe3\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\x94"		=> "\xe3\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xa1"		=> "\xe3\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xa6"		=> "\xe3\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xa9"		=> "\xe3\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xb3"		=> "\xe3\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf\xb9"		=> "\xe3\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\x9c"		=> "\xe3\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x86"		=> "\xe3\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x84"		=> "\xe3\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x8b"		=> "\xe3\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x92"		=> "\xe3\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x98"		=> "\xe3\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x99"		=> "\xe3\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\x9c"		=> "\xe3\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\xa1"		=> "\xe3\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\xa8"		=> "\xe3\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\xbf"		=> "\xe3\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80\xbb"		=> "\xe3\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\x8a"		=> "\xe3\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\x86"		=> "\xe3\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\x92"		=> "\xe3\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\x98"		=> "\xe3\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\x9a"		=> "\xe3\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\x9f"		=> "\xe3\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xa2"		=> "\xe3\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xa8"		=> "\xe3\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xb3"		=> "\xe3\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xb2"		=> "\xe3\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xb0"		=> "\xe3\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xb6"		=> "\xe3\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xb9"		=> "\xe3\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xbd"		=> "\xe3\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81\xbf"		=> "\xe3\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x84"		=> "\xe3\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x86"		=> "\xe3\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x85"		=> "\xe3\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x9b"		=> "\xe3\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x93"		=> "\xe3\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\x9a"		=> "\xe3\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xad"		=> "\xe3\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x90"		=> "\xe3\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82\xac"		=> "\xe3\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x9b"		=> "\xe3\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xa5"		=> "\xe3\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x99"		=> "\xe3\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x9d"		=> "\xe3\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x84"		=> "\xe3\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x9a"		=> "\xe3\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\x96"		=> "\xe3\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x89"		=> "\xe3\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xaf"		=> "\xe3\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xb1"		=> "\xe3\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\x9b"		=> "\xe3\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\xa9"		=> "\xe3\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\xa3"		=> "\xe3\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\xaf"		=> "\xe3\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x8b"		=> "\xe3\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x8b"		=> "\xe4\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x86"		=> "\xe4\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84\xbe"		=> "\xe4\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x93"		=> "\xe4\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x91"		=> "\xe4\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83\xbc"		=> "\xe4\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xb1"		=> "\xe4\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xae"		=> "\xe4\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xa5"		=> "\xe4\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xa6"		=> "\xe4\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\xb4"		=> "\xe4\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x83"		=> "\xe4\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x88"		=> "\xe4\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x8a"		=> "\xe4\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x80"		=> "\xe4\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x82"		=> "\xe4\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xa0"		=> "\xe4\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x95"		=> "\xe4\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xa4"		=> "\xe4\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xa3"		=> "\xe4\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85\x9f"		=> "\xe4\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\x93"		=> "\xe4\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xa9"		=> "\xe4\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xb0"		=> "\xe4\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xb5"		=> "\xe4\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xbe"		=> "\xe4\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xb8"		=> "\xe4\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xbd"		=> "\xe4\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x80"		=> "\xe4\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x82"		=> "\xe4\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86\xba"		=> "\xe4\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x89"		=> "\xe4\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x8d"		=> "\xe4\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x91"		=> "\xe4\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x99"		=> "\xe4\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x98"		=> "\xe4\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x88"		=> "\xe4\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x9a"		=> "\xe4\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\x9f"		=> "\xe4\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xa0"		=> "\xe4\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xa7"		=> "\xe4\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xba"		=> "\xe4\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xbb"		=> "\xe4\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87\xbe"		=> "\xe4\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x81"		=> "\xe4\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x82"		=> "\xe4\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x85"		=> "\xe4\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x87"		=> "\xe4\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x8a"		=> "\xe4\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x8d"		=> "\xe4\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x90"		=> "\xe4\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\x96"		=> "\xe4\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xa9"		=> "\xe4\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xab"		=> "\xe4\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xb8"		=> "\xe4\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xb3"		=> "\xe4\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\x80"		=> "\xe4\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\x99"		=> "\xe4\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\x98"		=> "\xe4\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\x9d"		=> "\xe4\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\x9a"		=> "\xe4\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\x9f"		=> "\xe4\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xa4"		=> "\xe4\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xa2"		=> "\xe4\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xa8"		=> "\xe4\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xaa"		=> "\xe4\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xab"		=> "\xe4\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88\xae"		=> "\xe4\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xb1"		=> "\xe4\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xb7"		=> "\xe4\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xb8"		=> "\xe4\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89\xbe"		=> "\xe4\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\x8d"		=> "\xe4\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\x92"		=> "\xe4\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xab"		=> "\xe4\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\x9f"		=> "\xe4\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xbb"		=> "\xe4\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a\xac"		=> "\xe4\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xa1"		=> "\xe4\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xa3"		=> "\xe4\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x9f"		=> "\xe4\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x92"		=> "\xe4\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xb4"		=> "\xe4\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xb3"		=> "\xe4\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xba"		=> "\xe4\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x93"		=> "\xe4\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x83"		=> "\xe4\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xbb"		=> "\xe4\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\xb9"		=> "\xe4\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x9e"		=> "\xe4\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x86"		=> "\xe4\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x9c"		=> "\xe4\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x89"		=> "\xe4\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b\x99"		=> "\xe4\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xb5"		=> "\xe4\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xb4"		=> "\xe4\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x96"		=> "\xe4\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xb2"		=> "\xe4\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xb1"		=> "\xe4\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\x80"		=> "\xe4\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xb9"		=> "\xe4\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\x90"		=> "\xe4\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\x85"		=> "\xe4\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xaf"		=> "\xe4\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xab"		=> "\xe4\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x97"		=> "\xe4\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\x98"		=> "\xe4\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x85"		=> "\xe4\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x9a"		=> "\xe4\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\xaa"		=> "\xe4\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x9f"		=> "\xe4\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\xa2"		=> "\xe4\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x96"		=> "\xe4\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c\xa3"		=> "\xe4\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x8e"		=> "\xe4\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x87"		=> "\xe4\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x8a"		=> "\xe4\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xbc"		=> "\xe4\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\xb5"		=> "\xe4\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xb3"		=> "\xe4\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xb5"		=> "\xe4\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\xa0"		=> "\xe4\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\x89"		=> "\xe4\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\xa8"		=> "\xe4\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xb4"		=> "\xe4\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\x93"		=> "\xe4\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xab"		=> "\xe4\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x8e"		=> "\xe4\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xbd"		=> "\xe4\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\x83"		=> "\xe4\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x98"		=> "\xe4\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\x8b"		=> "\xe4\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x81"		=> "\xe4\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xb7"		=> "\xe4\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\x87"		=> "\xe4\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xa0"		=> "\xe4\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xb2"		=> "\xe4\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\x8d"		=> "\xe4\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xa2"		=> "\xe4\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xa0"		=> "\xe4\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e\xbd"		=> "\xe4\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xb8"		=> "\xe4\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x86"		=> "\xe4\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xbb"		=> "\xe4\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xad"		=> "\xe4\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xaa"		=> "\xe4\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xbc"		=> "\xe4\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x9a"		=> "\xe4\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\x84"		=> "\xe4\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xb7"		=> "\xe4\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xab"		=> "\xe4\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xad"		=> "\xe4\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xae"		=> "\xe4\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\x82"		=> "\xe4\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xa9"		=> "\xe4\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\x86"		=> "\xe4\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xac"		=> "\xe4\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xaf"		=> "\xe4\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xb9"		=> "\xe4\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90\xb5"		=> "\xe4\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x8a"		=> "\xe4\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\xa2"		=> "\xe4\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xb9"		=> "\xe4\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xbf"		=> "\xe4\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\x9f"		=> "\xe4\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x99"		=> "\xe4\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x8d"		=> "\xe4\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xbb"		=> "\xe4\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x9a"		=> "\xe4\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x90"		=> "\xe4\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x81"		=> "\xe4\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x86"		=> "\xe4\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x96"		=> "\xe4\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xa1"		=> "\xe4\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\xa1"		=> "\xe4\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\xbf"		=> "\xe4\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\xb4"		=> "\xe4\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x97"		=> "\xe4\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x98"		=> "\xe4\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\xac"		=> "\xe4\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x9f"		=> "\xe4\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x95"		=> "\xe4\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94\x94"		=> "\xe4\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\xbc"		=> "\xe4\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x80"		=> "\xe4\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xa3"		=> "\xe4\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x98"		=> "\xe4\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x88"		=> "\xe4\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x81"		=> "\xe5\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x82"		=> "\xe5\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x8b"		=> "\xe5\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x95"		=> "\xe5\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x80"		=> "\xe5\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xa4"		=> "\xe5\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x88"		=> "\xe5\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x91"		=> "\xe5\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x8a"		=> "\xe5\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xa8"		=> "\xe5\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xad"		=> "\xe5\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x94"		=> "\xe5\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x9b"		=> "\xe5\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xaa"		=> "\xe5\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x87"		=> "\xe5\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x9c"		=> "\xe5\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xb7"		=> "\xe5\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xbe"		=> "\xe5\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\x90"		=> "\xe5\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\x89"		=> "\xe5\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xba"		=> "\xe5\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\x8f"		=> "\xe5\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96\xb9"		=> "\xe5\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\x90"		=> "\xe5\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\x95"		=> "\xe5\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\x9d"		=> "\xe5\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xa5"		=> "\xe5\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\x9c"		=> "\xe5\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xb9"		=> "\xe5\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x8a"		=> "\xe5\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x93"		=> "\xe5\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x8b"		=> "\xe5\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xbe"		=> "\xe5\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97\xba"		=> "\xe5\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x86"		=> "\xe5\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\xa2"		=> "\xe5\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\x9a"		=> "\xe5\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\xb0"		=> "\xe5\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98\xbf"		=> "\xe5\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x8d"		=> "\xe5\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9\x95"		=> "\xe5\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x94"		=> "\xe5\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\x9f"		=> "\xe5\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\xa7"		=> "\xe5\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99\xb1"		=> "\xe5\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\x93"		=> "\xe5\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\xa3"		=> "\xe5\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\xa9"		=> "\xe5\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\xaa"		=> "\xe5\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\x8b"		=> "\xe5\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\x8c"		=> "\xe5\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\xb6"		=> "\xe5\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\xaf"		=> "\xe5\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x84"		=> "\xe5\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x86"		=> "\xe5\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\xb0"		=> "\xe5\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x89"		=> "\xe5\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\xa3"		=> "\xe5\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a\xab"		=> "\xe5\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x94"		=> "\xe5\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x9e"		=> "\xe5\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xa9"		=> "\xe5\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xac"		=> "\xe5\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x9f"		=> "\xe5\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\x9b"		=> "\xe5\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xaf"		=> "\xe5\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x92"		=> "\xe5\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x86"		=> "\xe5\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x88"		=> "\xe5\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x80"		=> "\xe5\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x83"		=> "\xe5\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xbb"		=> "\xe5\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x91"		=> "\xe5\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x89"		=> "\xe5\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x8d"		=> "\xe5\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b\xb9"		=> "\xe5\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x8a"		=> "\xe5\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\xb4"		=> "\xe5\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\xbf"		=> "\xe5\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\xb7"		=> "\xe5\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\xbb"		=> "\xe5\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\xa5"		=> "\xe5\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\xa9"		=> "\xe5\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c\x9a"		=> "\xe5\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xa0"		=> "\xe5\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x9f"		=> "\xe5\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xb8"		=> "\xe5\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x8c"		=> "\xe5\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x8e"		=> "\xe5\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xb4"		=> "\xe5\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x97"		=> "\xe5\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xa8"		=> "\xe5\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xae"		=> "\xe5\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x99"		=> "\xe5\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\x93"		=> "\xe5\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xa3"		=> "\xe5\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d\xaa"		=> "\xe5\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\x85"		=> "\xe5\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\xa2"		=> "\xe5\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\x9f"		=> "\xe5\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\x82"		=> "\xe5\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\xaf"		=> "\xe5\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\x8b"		=> "\xe5\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\xbd"		=> "\xe5\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\x80"		=> "\xe5\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\x90"		=> "\xe5\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x96"		=> "\xe5\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\xab"		=> "\xe5\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\x84"		=> "\xe5\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\xb3"		=> "\xe5\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\x87"		=> "\xe5\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\x86"		=> "\xe5\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e\xbb"		=> "\xe5\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\xaf"		=> "\xe5\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\xb2"		=> "\xe5\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\xa0"		=> "\xe5\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\x8f"		=> "\xe5\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\x8d"		=> "\xe5\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\xbe"		=> "\xe5\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\xb6"		=> "\xe5\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\xb7"		=> "\xe5\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\x8e"		=> "\xe5\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f\x92"		=> "\xe5\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\x91"		=> "\xe5\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\x96"		=> "\xe5\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\x95"		=> "\xe5\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\xa2"		=> "\xe5\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\xa1"		=> "\xe5\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\xb1"		=> "\xe5\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\xb6"		=> "\xe5\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\xb9"		=> "\xe5\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\xa7"		=> "\xe5\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0\xbb"		=> "\xe5\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x84"		=> "\xe5\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x82"		=> "\xe5\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x92"		=> "\xe5\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x99"		=> "\xe5\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\x9e"		=> "\xe5\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xa2"		=> "\xe5\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xab"		=> "\xe5\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x81"		=> "\xe5\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xbe"		=> "\xe5\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x9e"		=> "\xe5\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xb5"		=> "\xe5\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xbd"		=> "\xe5\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xb5"		=> "\xe5\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1\xb2"		=> "\xe5\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x82"		=> "\xe5\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x97"		=> "\xe5\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x92"		=> "\xe5\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xae"		=> "\xe5\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x99"		=> "\xe5\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xa2"		=> "\xe5\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\x8d"		=> "\xe5\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xa4"		=> "\xe5\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xb0"		=> "\xe5\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xbf"		=> "\xe5\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2\xb1"		=> "\xe5\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x83"		=> "\xe5\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x84"		=> "\xe5\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x94"		=> "\xe5\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x98"		=> "\xe5\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x99"		=> "\xe5\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\x9d"		=> "\xe5\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xb9"		=> "\xe5\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x82"		=> "\xe5\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xbc"		=> "\xe5\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xb4"		=> "\xe5\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xa8"		=> "\xe5\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xb2"		=> "\xe5\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x84"		=> "\xe5\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x8c"		=> "\xe5\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x8a"		=> "\xe5\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x93"		=> "\xe5\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\x83"		=> "\xe5\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x9e"		=> "\xe5\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\xa5"		=> "\xe5\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\xaa"		=> "\xe5\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\xab"		=> "\xe5\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\x81"		=> "\xe5\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\x84"		=> "\xe5\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\xbb"		=> "\xe5\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\xb6"		=> "\xe5\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\xb8"		=> "\xe5\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\x8c"		=> "\xe5\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x9d"		=> "\xe5\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xa0"		=> "\xe5\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\x9e"		=> "\xe5\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xa6"		=> "\xe6\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xa4"		=> "\xe6\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xad"		=> "\xe6\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xaa"		=> "\xe6\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xaf"		=> "\xe6\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xb4"		=> "\xe6\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xb7"		=> "\xe6\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\xbe"		=> "\xe6\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x83"		=> "\xe6\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x88"		=> "\xe6\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x8a"		=> "\xe6\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x93"		=> "\xe6\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\x98"		=> "\xe6\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xa1"		=> "\xe6\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xa9"		=> "\xe6\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xa6"		=> "\xe6\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xac"		=> "\xe6\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xaf"		=> "\xe6\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xb2"		=> "\xe6\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xba"		=> "\xe6\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xbd"		=> "\xe6\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6\xbf"		=> "\xe6\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\x80"		=> "\xe6\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\x9a"		=> "\xe6\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\x9c"		=> "\xe6\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\x9d"		=> "\xe6\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\xa7"		=> "\xe6\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\xb4"		=> "\xe6\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7\xb8"		=> "\xe6\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x83"		=> "\xe6\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x96"		=> "\xe6\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x90"		=> "\xe6\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x8c"		=> "\xe6\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x9b"		=> "\xe6\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x9d"		=> "\xe6\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xa5"		=> "\xe6\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xb6"		=> "\xe6\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x81"		=> "\xe6\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x9b"		=> "\xe6\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x92"		=> "\xe6\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x86"		=> "\xe6\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\x88"		=> "\xe6\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xbc"		=> "\xe6\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xad"		=> "\xe6\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xac"		=> "\xe6\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xa2"		=> "\xe6\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x85"		=> "\xe6\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x82"		=> "\xe6\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x84"		=> "\xe6\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa8"		=> "\xe6\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa1"		=> "\xe6\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x91"		=> "\xe6\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa5"		=> "\xe6\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa6"		=> "\xe6\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\x9a"		=> "\xe6\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa3"		=> "\xe6\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x84"		=> "\xe6\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x8d"		=> "\xe6\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x82"		=> "\xe6\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x9a"		=> "\xe6\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xab"		=> "\xe6\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xb3"		=> "\xe6\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xa7"		=> "\xe6\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xa4"		=> "\xe6\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xb1"		=> "\xe6\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x94"		=> "\xe6\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xa0"		=> "\xe6\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xa2"		=> "\xe6\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xb7"		=> "\xe6\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x9e"		=> "\xe6\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x9b"		=> "\xe6\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x8c"		=> "\xe6\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x87"		=> "\xe6\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x9a"		=> "\xe6\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\xa1"		=> "\xe6\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x96"		=> "\xe6\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x90"		=> "\xe6\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\x97"		=> "\xe6\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xa0"		=> "\xe6\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xb3"		=> "\xe6\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\xab"		=> "\xe6\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xa6"		=> "\xe6\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xab"		=> "\xe6\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xbe"		=> "\xe6\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac\xa8"		=> "\xe6\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x81"		=> "\xe6\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x8c"		=> "\xe6\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x8f"		=> "\xe6\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x8e"		=> "\xe6\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x89"		=> "\xe6\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x96"		=> "\xe6\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x9b"		=> "\xe6\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x9a"		=> "\xe6\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xab"		=> "\xe6\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x9f"		=> "\xe6\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xac"		=> "\xe6\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xaf"		=> "\xe6\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xb4"		=> "\xe6\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xbd"		=> "\xe6\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x80"		=> "\xe6\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x8c"		=> "\xe6\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x8e"		=> "\xe6\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x92"		=> "\xe6\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x93"		=> "\xe6\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x96"		=> "\xe6\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x99"		=> "\xe6\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae\x9a"		=> "\xe6\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb0\xba"		=> "\xe6\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x81"		=> "\xe6\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb0\xbf"		=> "\xe6\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x88"		=> "\xe6\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x8c"		=> "\xe6\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x8e"		=> "\xe6\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x90"		=> "\xe6\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\x95"		=> "\xe6\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xa2"		=> "\xe6\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xac"		=> "\xe6\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xb8"		=> "\xe6\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xba"		=> "\xe6\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x82"		=> "\xe6\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x89"		=> "\xe6\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x85"		=> "\xe6\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x8a"		=> "\xe6\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x8d"		=> "\xe6\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x8e"		=> "\xe6\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x94"		=> "\xe6\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1\xbc"		=> "\xe6\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\x98"		=> "\xe6\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x9d"		=> "\xe6\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xad"		=> "\xe6\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xaa"		=> "\xe6\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xbd"		=> "\xe6\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xb2"		=> "\xe6\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xb3"		=> "\xe6\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xae"		=> "\xe6\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2\xb6"		=> "\xe6\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x88"		=> "\xe6\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x81"		=> "\xe6\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xa4"		=> "\xe6\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xa3"		=> "\xe6\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x9a"		=> "\xe6\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xbd"		=> "\xe6\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xba"		=> "\xe6\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xbb"		=> "\xe6\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x84"		=> "\xe6\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x85"		=> "\xe6\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x8a"		=> "\xe6\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x87"		=> "\xe6\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x8f"		=> "\xe6\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x8d"		=> "\xe6\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x90"		=> "\xe6\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\x8e"		=> "\xe6\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x93"		=> "\xe6\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\x8d"		=> "\xe6\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x94"		=> "\xe6\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x96"		=> "\xe6\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xa7"		=> "\xe6\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xad"		=> "\xe6\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xb1"		=> "\xe6\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xb3"		=> "\xe6\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\x81"		=> "\xe6\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\x99"		=> "\xe6\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x82"		=> "\xe6\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\xbe"		=> "\xe6\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6\xba"		=> "\xe6\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x8f"		=> "\xe6\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x9a"		=> "\xe6\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x96"		=> "\xe6\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x8c"		=> "\xe6\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x9b"		=> "\xe6\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x8b"		=> "\xe6\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xaa"		=> "\xe6\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xab"		=> "\xe6\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\x9f"		=> "\xe6\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xa3"		=> "\xe6\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xbc"		=> "\xe6\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\x88"		=> "\xe6\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\x89"		=> "\xe6\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7\xbf"		=> "\xe6\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\x9d"		=> "\xe6\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\x9e"		=> "\xe6\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\x90"		=> "\xe6\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\x9f"		=> "\xe6\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x82"		=> "\xe6\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\xb5"		=> "\xe6\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\xb0"		=> "\xe6\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\xb4"		=> "\xe6\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x8a"		=> "\xe6\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x87"		=> "\xe7\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x89"		=> "\xe7\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x8c"		=> "\xe7\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x90"		=> "\xe7\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x88"		=> "\xe7\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x99"		=> "\xe7\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\xa4"		=> "\xe7\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\xa0"		=> "\xe7\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8\xaa"		=> "\xe7\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\xa3"		=> "\xe7\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\x95"		=> "\xe7\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\xb6"		=> "\xe7\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\xb2"		=> "\xe7\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9\xbc"		=> "\xe7\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x81"		=> "\xe7\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x87"		=> "\xe7\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x85"		=> "\xe7\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x84"		=> "\xe7\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x8b"		=> "\xe7\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x8a"		=> "\xe7\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x93"		=> "\xe7\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x91"		=> "\xe7\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x94"		=> "\xe7\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\x99"		=> "\xe7\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\xaa"		=> "\xe7\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\xa1"		=> "\xe7\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\xac"		=> "\xe7\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\xb0"		=> "\xe7\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x86"		=> "\xe7\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\xb1"		=> "\xe7\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba\xbe"		=> "\xe7\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x85"		=> "\xe7\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x88"		=> "\xe7\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x8b"		=> "\xe7\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\x9b"		=> "\xe7\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\xa3"		=> "\xe7\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\xbc"		=> "\xe7\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\xbb"		=> "\xe7\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\xab"		=> "\xe7\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb\xbe"		=> "\xe7\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x8a"		=> "\xe7\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x85"		=> "\xe7\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x95"		=> "\xe7\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x92"		=> "\xe7\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x99"		=> "\xe7\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x93"		=> "\xe7\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x9c"		=> "\xe7\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x9f"		=> "\xe7\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x9b"		=> "\xe7\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\x8c"		=> "\xe7\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xa6"		=> "\xe7\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xb3"		=> "\xe7\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xbb"		=> "\xe7\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xb9"		=> "\xe7\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x85"		=> "\xe7\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x82"		=> "\xe7\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc\xbe"		=> "\xe7\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x8c"		=> "\xe7\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x89"		=> "\xe7\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x86"		=> "\xe7\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x8e"		=> "\xe7\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x97"		=> "\xe7\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\x9c"		=> "\xe7\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\xa2"		=> "\xe7\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\xa3"		=> "\xe7\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd\xa4"		=> "\xe7\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\x9c"		=> "\xe7\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\x9f"		=> "\xe7\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xa3"		=> "\xe7\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xad"		=> "\xe7\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xaf"		=> "\xe7\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe\xb7"		=> "\xe7\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\x9a"		=> "\xe7\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xa5"		=> "\xe7\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xa2"		=> "\xe7\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xaa"		=> "\xe7\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xaf"		=> "\xe7\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x87"		=> "\xe7\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xb4"		=> "\xe7\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x85"		=> "\xe7\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xb9"		=> "\xe7\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xba"		=> "\xe7\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x91"		=> "\xe7\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x95"		=> "\xe7\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xa1"		=> "\xe7\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x8d"		=> "\xe7\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x9e"		=> "\xe7\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x96"		=> "\xe7\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x8b"		=> "\xe7\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xa7"		=> "\xe7\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xb6"		=> "\xe7\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xb5"		=> "\xe7\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xb9"		=> "\xe7\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf\xb8"		=> "\xe7\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x8f"		=> "\xe7\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x90"		=> "\xe7\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x91"		=> "\xe7\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x92"		=> "\xe7\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\x8e"		=> "\xe7\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x89"		=> "\xe7\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80\xbe"		=> "\xe7\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x96"		=> "\xe7\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x98"		=> "\xe7\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x9e"		=> "\xe7\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xa8"		=> "\xe7\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xaf"		=> "\xe7\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xb6"		=> "\xe7\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xa8"		=> "\xe7\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xb2"		=> "\xe7\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x82"		=> "\xe7\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\xbd"		=> "\xe7\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x81"		=> "\xe7\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x80"		=> "\xe7\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x8a"		=> "\xe7\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x89"		=> "\xe7\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\x8f"		=> "\xe7\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\xa8"		=> "\xe7\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\xaf"		=> "\xe7\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\xb1"		=> "\xe7\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82\xb5"		=> "\xe7\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\xa2"		=> "\xe7\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\xa4"		=> "\xe7\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89\x88"		=> "\xe7\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\x9b"		=> "\xe7\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84\x82"		=> "\xe7\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84\x92"		=> "\xe7\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84\x99"		=> "\xe7\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84\xb2"		=> "\xe7\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84\xb0"		=> "\xe7\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x8a"		=> "\xe7\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x96"		=> "\xe7\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\x98"		=> "\xe7\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xa3"		=> "\xe7\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xa5"		=> "\xe7\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xa9"		=> "\xe7\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xb3"		=> "\xe7\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85\xb2"		=> "\xe7\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x8b"		=> "\xe7\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x89"		=> "\xe7\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\x82"		=> "\xe7\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xa2"		=> "\xe7\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xab"		=> "\xe7\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xaf"		=> "\xe7\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xaa"		=> "\xe7\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xb5"		=> "\xe7\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xb4"		=> "\xe7\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86\xba"		=> "\xe7\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x80"		=> "\xe7\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x81"		=> "\xe7\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x89"		=> "\xe7\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x8b"		=> "\xe7\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x90"		=> "\xe7\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x96"		=> "\xe7\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x9f"		=> "\xe7\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xa1"		=> "\xe7\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x9b"		=> "\xe7\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xbc"		=> "\xe7\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xb5"		=> "\xe7\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xb6"		=> "\xe7\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x9e"		=> "\xe7\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xbf"		=> "\xe7\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x94"		=> "\xe7\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xac"		=> "\xe7\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x95"		=> "\xe7\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x91"		=> "\xe7\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x9e"		=> "\xe7\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x97"		=> "\xe7\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x85"		=> "\xe7\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x89"		=> "\xe7\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xa4"		=> "\xe7\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x88"		=> "\xe7\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x95"		=> "\xe7\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xbf"		=> "\xe7\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x8b"		=> "\xe7\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x90"		=> "\xe7\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x9c"		=> "\xe7\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x96"		=> "\xe7\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x93"		=> "\xe7\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x9b"		=> "\xe7\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x9a"		=> "\xe7\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x8f"		=> "\xe7\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\xb9"		=> "\xe7\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\xb7"		=> "\xe7\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xa9"		=> "\xe7\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x8f"		=> "\xe7\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xba"		=> "\xe7\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x84"		=> "\xe7\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xae"		=> "\xe7\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x99"		=> "\xe8\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa2"		=> "\xe8\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x9a"		=> "\xe8\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa3"		=> "\xe8\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xba"		=> "\xe8\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xb5"		=> "\xe8\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xbb"		=> "\xe8\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x9c"		=> "\xe8\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\xa0"		=> "\xe8\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\xbc"		=> "\xe8\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\xae"		=> "\xe8\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x96"		=> "\xe8\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\xb0"		=> "\xe8\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\xac"		=> "\xe8\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\xad"		=> "\xe8\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\x94"		=> "\xe8\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\xb9"		=> "\xe8\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x96"		=> "\xe8\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x97"		=> "\xe8\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\xa8"		=> "\xe8\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\xa5"		=> "\xe8\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x98"		=> "\xe8\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x83"		=> "\xe8\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x9d"		=> "\xe8\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x90"		=> "\xe8\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x88"		=> "\xe8\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\xa4"		=> "\xe8\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\x9a"		=> "\xe8\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\x94"		=> "\xe8\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\x93"		=> "\xe8\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\x83"		=> "\xe8\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\x87"		=> "\xe8\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\x90"		=> "\xe8\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\xb6"		=> "\xe8\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\xab"		=> "\xe8\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\xb5"		=> "\xe8\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\xa1"		=> "\xe8\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\xba"		=> "\xe8\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x81"		=> "\xe8\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x92"		=> "\xe8\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x84"		=> "\xe8\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x9b"		=> "\xe8\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xa0"		=> "\xe8\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xa2"		=> "\xe8\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x9e"		=> "\xe8\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xaa"		=> "\xe8\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xa9"		=> "\xe8\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xb0"		=> "\xe8\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xb5"		=> "\xe8\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xb7"		=> "\xe8\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xbd"		=> "\xe8\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x9a"		=> "\xe8\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xbc"		=> "\xe8\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xbe"		=> "\xe8\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x92\x81"		=> "\xe8\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\xbf"		=> "\xe8\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x82"		=> "\xe8\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x87"		=> "\xe8\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x8a"		=> "\xe8\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x94"		=> "\xe8\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x96"		=> "\xe8\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x98"		=> "\xe8\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x99"		=> "\xe8\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xa0"		=> "\xe8\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xa8"		=> "\xe8\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xa7"		=> "\xe8\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xad"		=> "\xe8\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xbc"		=> "\xe8\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xbb"		=> "\xe8\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xb9"		=> "\xe8\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\xbe"		=> "\xe8\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x8a"		=> "\xe8\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xb6"		=> "\xe8\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x83"		=> "\xe8\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x8d"		=> "\xe8\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x8c"		=> "\xe8\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x95"		=> "\xe8\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x94"		=> "\xe8\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x96"		=> "\xe8\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\x9c"		=> "\xe8\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\xa1"		=> "\xe8\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\xa5"		=> "\xe8\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97\xa2"		=> "\xe8\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\xa1"		=> "\xe8\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\xa8"		=> "\xe8\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\xae"		=> "\xe8\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98\xaf"		=> "\xe8\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x82"		=> "\xe8\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x8c"		=> "\xe8\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x8f"		=> "\xe8\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x8b"		=> "\xe8\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xb7"		=> "\xe8\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x9c"		=> "\xe8\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x9e"		=> "\xe8\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x9d"		=> "\xe8\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\x9f"		=> "\xe8\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xa6"		=> "\xe8\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xb2"		=> "\xe8\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99\xac"		=> "\xe8\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x8d"		=> "\xe8\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x98"		=> "\xe8\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x95"		=> "\xe8\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\x97"		=> "\xe8\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xaa"		=> "\xe8\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xa7"		=> "\xe8\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xb1"		=> "\xe8\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xb2"		=> "\xe8\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xb0"		=> "\xe8\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xb4"		=> "\xe8\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xb6"		=> "\xe8\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xb8"		=> "\xe8\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xb9"		=> "\xe8\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x8e"		=> "\xe8\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x8b"		=> "\xe8\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x89"		=> "\xe8\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x8d"		=> "\xe8\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5\x8d"		=> "\xe8\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x9c"		=> "\xe8\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x8d"		=> "\xe8\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\x95"		=> "\xe8\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b\xb9"		=> "\xe8\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x84"		=> "\xe8\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x86"		=> "\xe8\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x88"		=> "\xe8\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x93"		=> "\xe8\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x8e"		=> "\xe8\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x91"		=> "\xe8\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x8f"		=> "\xe8\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x96"		=> "\xe8\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\x99"		=> "\xe8\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xa4"		=> "\xe8\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xaa"		=> "\xe8\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xb0"		=> "\xe8\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xb9"		=> "\xe8\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xbd"		=> "\xe8\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xbe"		=> "\xe8\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x84"		=> "\xe8\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x86"		=> "\xe8\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x88"		=> "\xe8\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x82"		=> "\xe8\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x89"		=> "\xe8\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x9c"		=> "\xe8\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xa0"		=> "\xe8\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xa4"		=> "\xe8\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xa6"		=> "\xe8\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xa8"		=> "\xe8\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x92"		=> "\xe8\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xab"		=> "\xe8\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xb1"		=> "\xe8\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xb9"		=> "\xe8\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x85"		=> "\xe8\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xbc"		=> "\xe8\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x81"		=> "\xe8\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xba"		=> "\xe8\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x86"		=> "\xe8\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x8b"		=> "\xe8\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x8f"		=> "\xe8\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x90"		=> "\xe8\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\x9c"		=> "\xe8\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\xa8"		=> "\xe8\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\xa6"		=> "\xe8\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\xa3"		=> "\xe8\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\xb3"		=> "\xe8\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e\xb4"		=> "\xe8\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\x83"		=> "\xe8\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\x86"		=> "\xe8\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\x88"		=> "\xe8\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\x8b"		=> "\xe8\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\x9c"		=> "\xe8\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\xad"		=> "\xe8\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\x8f"		=> "\xe8\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\xb2"		=> "\xe8\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x9f"		=> "\xe8\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\xb6"		=> "\xe8\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f\xb5"		=> "\xe8\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x8f"		=> "\xe8\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\x8c"		=> "\xe8\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xb8"		=> "\xe8\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xa4"		=> "\xe8\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xa1"		=> "\xe8\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xb7"		=> "\xe8\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0\xbd"		=> "\xe8\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x86"		=> "\xe8\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x8f"		=> "\xe8\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x8b"		=> "\xe8\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xab"		=> "\xe8\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xaf"		=> "\xe8\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xb0"		=> "\xe8\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xb1"		=> "\xe9\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xb4"		=> "\xe9\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xb3"		=> "\xe9\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa2\xaa"		=> "\xe9\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa2\xaf"		=> "\xe9\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa2\xb1"		=> "\xe9\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa2\xb6"		=> "\xe9\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\x84"		=> "\xe9\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\x83"		=> "\xe9\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\x86"		=> "\xe9\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xa9"		=> "\xe9\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3\xab"		=> "\xe9\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x83"		=> "\xe9\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x89"		=> "\xe9\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x92"		=> "\xe9\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x94"		=> "\xe9\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x98"		=> "\xe9\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\xa1"		=> "\xe9\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x9d"		=> "\xe9\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\x9e"		=> "\xe9\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\xa4"		=> "\xe9\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\xa0"		=> "\xe9\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\xac"		=> "\xe9\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\xae"		=> "\xe9\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\xbd"		=> "\xe9\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4\xbe"		=> "\xe9\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x82"		=> "\xe9\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x89"		=> "\xe9\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x85"		=> "\xe9\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x90"		=> "\xe9\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x8b"		=> "\xe9\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x91"		=> "\xe9\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x92"		=> "\xe9\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x8c"		=> "\xe9\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5\x95"		=> "\xe9\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\x97"		=> "\xe9\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\x98"		=> "\xe9\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\xa5"		=> "\xe9\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\xad"		=> "\xe9\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\xae"		=> "\xe9\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6\xbc"		=> "\xe9\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x9f"		=> "\xe9\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x9b"		=> "\xe9\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x9d"		=> "\xe9\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x98"		=> "\xe9\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\x91"		=> "\xe9\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\xad"		=> "\xe9\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\xae"		=> "\xe9\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\xb1"		=> "\xe9\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\xb2"		=> "\xe9\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\xbb"		=> "\xe9\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\xb8"		=> "\xe9\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\x81"		=> "\xe9\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\x8f"		=> "\xe9\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\x85"		=> "\xe9\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7\xa2"		=> "\xe9\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\x99"		=> "\xe9\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\xab"		=> "\xe9\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\xb7"		=> "\xe9\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x85"		=> "\xe9\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x82"		=> "\xe9\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x80"		=> "\xe9\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x83"		=> "\xe9\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8\xbe"		=> "\xe9\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x95"		=> "\xe9\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x8d"		=> "\xe9\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x9b"		=> "\xe9\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x97"		=> "\xe9\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x9f"		=> "\xe9\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\xa2"		=> "\xe9\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\xa5"		=> "\xe9\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\xa4"		=> "\xe9\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\xa9"		=> "\xe9\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\xab"		=> "\xe9\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\xaa"		=> "\xe9\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaa\xad"		=> "\xe9\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaa\xb0"		=> "\xe9\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaa\xbc"		=> "\xe9\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x80"		=> "\xe9\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x8f"		=> "\xe9\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x91"		=> "\xe9\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x93"		=> "\xe9\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x94"		=> "\xe9\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x9e"		=> "\xe9\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x9f"		=> "\xe9\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xa2"		=> "\xe9\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xa3"		=> "\xe9\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xa6"		=> "\xe9\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xaf"		=> "\xe9\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xab"		=> "\xe9\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xae"		=> "\xe9\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xb4"		=> "\xe9\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xb1"		=> "\xe9\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xb7"		=> "\xe9\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\xbb"		=> "\xe9\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\x86"		=> "\xe9\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\x98"		=> "\xe9\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\x9a"		=> "\xe9\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\x9f"		=> "\xe9\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xa2"		=> "\xe9\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xa3"		=> "\xe9\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xa5"		=> "\xe9\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xa7"		=> "\xe9\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xa8"		=> "\xe9\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xa9"		=> "\xe9\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xaa"		=> "\xe9\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xae"		=> "\xe9\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xaf"		=> "\xe9\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac\xb2"		=> "\xe9\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x84"		=> "\xe9\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x83"		=> "\xe9\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x8f"		=> "\xe9\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x8d"		=> "\xe9\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x8e"		=> "\xe9\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x91"		=> "\xe9\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\x98"		=> "\xe9\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\xb4"		=> "\xe9\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x93"		=> "\xe9\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x83"		=> "\xe9\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x91"		=> "\xe9\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x96"		=> "\xe9\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x97"		=> "\xe9\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x9f"		=> "\xe9\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xa0"		=> "\xe9\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xa8"		=> "\xe9\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xb4"		=> "\xe9\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x80"		=> "\xe9\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x8a"		=> "\xe9\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xb9"		=> "\xe9\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x86"		=> "\xe9\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x8f"		=> "\xe9\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x91"		=> "\xe9\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x92"		=> "\xe9\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xa3"		=> "\xe9\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xa2"		=> "\xe9\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xa4"		=> "\xe9\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\x94"		=> "\xe9\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xa1"		=> "\xe9\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xba"		=> "\xe9\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xb2"		=> "\xe9\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xb1"		=> "\xe9\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf\xb0"		=> "\xe9\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x95"		=> "\xe9\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x94"		=> "\xe9\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x89"		=> "\xe9\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x93"		=> "\xe9\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x8c"		=> "\xe9\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x86"		=> "\xe9\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x88"		=> "\xe9\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x92"		=> "\xe9\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x8a"		=> "\xe9\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x84"		=> "\xe9\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xae"		=> "\xe9\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x9b"		=> "\xe9\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xa5"		=> "\xe9\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xa4"		=> "\xe9\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xa1"		=> "\xe9\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xb0"		=> "\xe9\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\x87"		=> "\xe9\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xb2"		=> "\xe9\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\x86"		=> "\xe9\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\xbe"		=> "\xe9\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\x9a"		=> "\xe9\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\xa0"		=> "\xe9\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\xa7"		=> "\xe9\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\xb6"		=> "\xe9\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1\xb8"		=> "\xe9\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xa7"		=> "\xe9\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xac"		=> "\xe9\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xb0"		=> "\xe9\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x89"		=> "\xe9\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x88"		=> "\xe9\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3\xab"		=> "\xe9\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x83"		=> "\xe9\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x86"		=> "\xe9\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xaa"		=> "\xe9\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xa6"		=> "\xe9\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xaf"		=> "\xe9\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xa3"		=> "\xe9\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x9f"		=> "\xe9\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x84"		=> "\xe9\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x95"		=> "\xe9\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\x92"		=> "\xe9\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x81"		=> "\xe9\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xbf"		=> "\xe9\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4\xbe"		=> "\xe9\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x86"		=> "\xe9\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x88"		=> "\xe9\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x9d"		=> "\xea\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x9e"		=> "\xea\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xa4"		=> "\xea\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x91"		=> "\xea\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x90"		=> "\xea\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\x99"		=> "\xea\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xb2"		=> "\xea\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\x89"		=> "\xea\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\x87"		=> "\xea\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xab"		=> "\xea\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xaf"		=> "\xea\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xba"		=> "\xea\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\x9a"		=> "\xea\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xa4"		=> "\xea\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xa9"		=> "\xea\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xb2"		=> "\xea\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\x84"		=> "\xea\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\x81"		=> "\xea\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xbb"		=> "\xea\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xb8"		=> "\xea\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6\xba"		=> "\xea\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\x86"		=> "\xea\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\x8f"		=> "\xea\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\x82"		=> "\xea\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\x99"		=> "\xea\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\x93"		=> "\xea\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\xb8"		=> "\xea\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\xa6"		=> "\xea\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\xad"		=> "\xea\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\xaf"		=> "\xea\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7\xbd"		=> "\xea\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb8\x9a"		=> "\xea\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb8\x9b"		=> "\xea\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb8\x9e"		=> "\xea\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb9\xb5"		=> "\xea\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb9\xb9"		=> "\xea\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb9\xbd"		=> "\xea\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x81"		=> "\xea\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x88"		=> "\xea\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x8b"		=> "\xea\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x8c"		=> "\xea\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x92"		=> "\xea\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x95"		=> "\xea\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x91"		=> "\xea\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\x9d"		=> "\xea\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xa5"		=> "\xea\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xa9"		=> "\xea\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xb8"		=> "\xea\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xaa"		=> "\xea\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba\xad"		=> "\xea\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\xa1"		=> "\xea\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x8c"		=> "\xea\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x8e"		=> "\xea\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x8f"		=> "\xea\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x90"		=> "\xea\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x94"		=> "\xea\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x9c"		=> "\xea\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x9e"		=> "\xea\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x9d"		=> "\xea\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xa0"		=> "\xea\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xa5"		=> "\xea\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xa8"		=> "\xea\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xaf"		=> "\xea\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xb4"		=> "\xea\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xb6"		=> "\xea\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xb7"		=> "\xea\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xb9"		=> "\xea\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xbb"		=> "\xea\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xbc"		=> "\xea\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\xbd"		=> "\xea\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\x87"		=> "\xea\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\x88"		=> "\xea\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xb7"		=> "\xea\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\x95"		=> "\xea\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\xa1"		=> "\xea\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\xac"		=> "\xea\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc\xbe"		=> "\xea\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\x8a"		=> "\xea\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\x92"		=> "\xea\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\x94"		=> "\xea\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xa3"		=> "\xea\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\x9f"		=> "\xea\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xa0"		=> "\xea\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xa1"		=> "\xea\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xa6"		=> "\xea\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xa7"		=> "\xea\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xac"		=> "\xea\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xaa"		=> "\xea\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xb7"		=> "\xea\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xb2"		=> "\xea\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd\xb6"		=> "\xea\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbe\x95"		=> "\xea\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbe\x9c"		=> "\xea\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbe\xa0"		=> "\xea\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0\xaf"		=> "\xea\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\x87"		=> "\xea\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81\x99"		=> "\xea\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xa4"		=> "\xea\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\x9c"		=> "\xea\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86\x99"		=> "\xea\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba\x8a"		=> "\xed\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x9c"		=> "\xed\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x88"		=> "\xed\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x88"		=> "\xed\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x9c"		=> "\xed\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x89"		=> "\xed\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xbb"		=> "\xed\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xb1"		=> "\xed\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x88"		=> "\xed\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xb9"		=> "\xed\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xbb"		=> "\xed\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x85"		=> "\xed\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xa8"		=> "\xed\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xa1"		=> "\xed\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xbc"		=> "\xed\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x80"		=> "\xed\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x83"		=> "\xed\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xb9"		=> "\xed\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x96"		=> "\xed\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x92"		=> "\xed\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x8a"		=> "\xed\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x9a"		=> "\xed\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x94"		=> "\xed\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x8d"		=> "\xed\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x80"		=> "\xed\x58",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa2"		=> "\xed\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xbf"		=> "\xed\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x9e"		=> "\xed\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x86"		=> "\xed\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xb0"		=> "\xed\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x82"		=> "\xed\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x94"		=> "\xed\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xb4"		=> "\xed\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x98"		=> "\xed\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x8a"		=> "\xed\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xa4"		=> "\xed\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x9d"		=> "\xed\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xbe"		=> "\xed\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xac"		=> "\xed\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x95"		=> "\xed\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x9c"		=> "\xed\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xa6"		=> "\xed\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x80"		=> "\xed\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x9b"		=> "\xed\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x80"		=> "\xed\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x87"		=> "\xed\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xa4"		=> "\xed\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb2"		=> "\xed\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\x93"		=> "\xed\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xb2"		=> "\xed\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x9d"		=> "\xed\x72",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x8e"		=> "\xed\x73",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe5\x92\x9c"		=> "\xed\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x8a"		=> "\xed\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xa9"		=> "\xed\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xbf"		=> "\xed\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x86"		=> "\xed\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x99"		=> "\xed\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\xa5"		=> "\xed\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xac"		=> "\xed\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x88"		=> "\xed\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x87"		=> "\xed\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x8f"		=> "\xed\x7e",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\x90"		=> "\xed\x80",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe5\xa2\x9e"		=> "\xed\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xb2"		=> "\xed\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x8b"		=> "\xed\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x93"		=> "\xed\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x9b"		=> "\xed\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x9d"		=> "\xed\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xa3"		=> "\xed\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xa4"		=> "\xed\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xba"		=> "\xed\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x96"		=> "\xed\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x80"		=> "\xed\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xaf"		=> "\xed\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x98"		=> "\xed\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xac"		=> "\xed\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x9e"		=> "\xed\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xa6"		=> "\xed\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xba"		=> "\xed\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xb5"		=> "\xed\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\xa7"		=> "\xed\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\x93"		=> "\xed\x94",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x91"		=> "\xed\x95",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe5\xb5\x82"		=> "\xed\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\xad"		=> "\xed\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xb8"		=> "\xed\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xb9"		=> "\xed\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x90"		=> "\xed\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xa1"		=> "\xed\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xb4"		=> "\xed\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xa7"		=> "\xed\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xb7"		=> "\xed\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x9e"		=> "\xed\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x9d"		=> "\xed\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x85"		=> "\xed\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x8a"		=> "\xed\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x9e"		=> "\xed\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x95"		=> "\xed\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xa0"		=> "\xed\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb2"		=> "\xed\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x91"		=> "\xed\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xb7"		=> "\xed\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xb0"		=> "\xed\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x98"		=> "\xed\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x93"		=> "\xed\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xa6"		=> "\xed\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xb5"		=> "\xed\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\xa0"		=> "\xed\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x9d"		=> "\xed\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x8e"		=> "\xed\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x8e"		=> "\xed\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x80"		=> "\xed\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x95"		=> "\xed\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xbb"		=> "\xed\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x89"		=> "\xed\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xae"		=> "\xed\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x9e"		=> "\xed\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xa4"		=> "\xed\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xa5"		=> "\xed\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x97"		=> "\xed\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x99"		=> "\xed\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x92"		=> "\xed\xbc",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe6\x99\xb3"		=> "\xed\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x99"		=> "\xed\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xa0"		=> "\xed\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xb2"		=> "\xed\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xbf"		=> "\xed\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xba"		=> "\xed\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x8e"		=> "\xed\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa4\xa9"		=> "\xed\xc4",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe6\x9d\xa6"		=> "\xed\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xbb"		=> "\xed\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x92"		=> "\xed\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x80"		=> "\xed\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x81"		=> "\xed\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x84"		=> "\xed\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x8f"		=> "\xed\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x93"		=> "\xed\xcc",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe6\xa5\xa8"		=> "\xed\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x94"		=> "\xed\xce",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe6\xa6\x98"		=> "\xed\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xa2"		=> "\xed\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xb0"		=> "\xed\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xab"		=> "\xed\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x86"		=> "\xed\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xb3"		=> "\xed\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xbe"		=> "\xed\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\xa2"		=> "\xed\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\xa4"		=> "\xed\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x96"		=> "\xed\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\xbf"		=> "\xed\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x9c"		=> "\xed\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x86"		=> "\xed\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xaf"		=> "\xed\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x9a"		=> "\xed\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x84"		=> "\xed\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x87"		=> "\xed\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xaf"		=> "\xed\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x96"		=> "\xed\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\xac"		=> "\xed\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x8f"		=> "\xed\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb8"		=> "\xed\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb2"		=> "\xed\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xbc"		=> "\xed\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xb9"		=> "\xed\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x9c"		=> "\xed\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa7"		=> "\xed\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xbc"		=> "\xed\xea",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xbf"		=> "\xed\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x88"		=> "\xed\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xb5"		=> "\xed\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xb5"		=> "\xed\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x85"		=> "\xed\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x87"		=> "\xed\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\xa8"		=> "\xed\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\x85"		=> "\xed\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xab"		=> "\xed\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x8f"		=> "\xed\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x84"		=> "\xed\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x9c"		=> "\xed\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x86"		=> "\xed\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x87"		=> "\xed\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x95"		=> "\xed\xf9",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\x87\x81"		=> "\xed\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xbe"		=> "\xed\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xb1"		=> "\xed\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xbe"		=> "\xee\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xa4"		=> "\xee\x41",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x96"		=> "\xee\x42",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\x8d\xb7"		=> "\xee\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\xbd"		=> "\xee\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x89"		=> "\xee\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x96"		=> "\xee\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xa3"		=> "\xee\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x92"		=> "\xee\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\x87"		=> "\xee\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xb5"		=> "\xee\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xa6"		=> "\xee\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xaa"		=> "\xee\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xa9"		=> "\xee\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xae"		=> "\xee\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xa2"		=> "\xee\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\x89"		=> "\xee\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\x9f"		=> "\xee\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x81"		=> "\xee\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xaf"		=> "\xee\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x82"		=> "\xee\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x9c"		=> "\xee\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x9e"		=> "\xee\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x9b"		=> "\xee\x57",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xa6"		=> "\xee\x58",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x97"		=> "\xee\x59",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\x9d\x86"		=> "\xee\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xaf"		=> "\xee\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xa1"		=> "\xee\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\x8e"		=> "\xee\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xa4"		=> "\xee\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xba"		=> "\xee\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xb0"		=> "\xee\x60",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x98"		=> "\xee\x61",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\x99"		=> "\xee\x62",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\x9a"		=> "\xee\x63",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\xa6\x94"		=> "\xee\x64",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9b"		=> "\xee\x65",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\xa6\x9b"		=> "\xee\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x91"		=> "\xee\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xa7"		=> "\xee\x68",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9c"		=> "\xee\x69",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\xab\xab"		=> "\xee\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x9e"		=> "\xee\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9d"		=> "\xee\x6c",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\xb5\x88"		=> "\xee\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x9c"		=> "\xee\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xb7"		=> "\xee\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xa0"		=> "\xee\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x96"		=> "\xee\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x92"		=> "\xee\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x87"		=> "\xee\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xa1"		=> "\xee\x74",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9e"		=> "\xee\x75",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\x8c\x81"		=> "\xee\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xa2"		=> "\xee\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xbf"		=> "\xee\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x87"		=> "\xee\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xb6"		=> "\xee\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\x88"		=> "\xee\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xb4"		=> "\xee\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x93"		=> "\xee\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x99"		=> "\xee\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xab"		=> "\xee\x80",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9f"		=> "\xee\x81",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\x96\xb0"		=> "\xee\x82",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa0"		=> "\xee\x83",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\xa1"		=> "\xee\x84",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\xa0\x87"		=> "\xee\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xb5"		=> "\xee\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x92"		=> "\xee\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xb7"		=> "\xee\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xb9"		=> "\xee\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa7"		=> "\xee\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xbe"		=> "\xee\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x9f"		=> "\xee\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa2"		=> "\xee\x8d",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\xab\xb6"		=> "\xee\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x93"		=> "\xee\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xbf"		=> "\xee\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xb0"		=> "\xee\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xb4"		=> "\xee\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x92"		=> "\xee\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xb6"		=> "\xee\x94",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa3"		=> "\xee\x95",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\xbb\x8f"		=> "\xee\x96",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa4"		=> "\xee\x97",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\xa5"		=> "\xee\x98",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x81\xa7"		=> "\xee\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\x9e"		=> "\xee\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa6"		=> "\xee\x9b",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x84\x95"		=> "\xee\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84\xa7"		=> "\xee\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x9a"		=> "\xee\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x97"		=> "\xee\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x9e"		=> "\xee\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xad"		=> "\xee\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xae"		=> "\xee\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xa4"		=> "\xee\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xa5"		=> "\xee\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x86"		=> "\xee\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x90"		=> "\xee\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x8a"		=> "\xee\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xba"		=> "\xee\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x80"		=> "\xee\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xbc"		=> "\xee\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x8e"		=> "\xee\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x99"		=> "\xee\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x91"		=> "\xee\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xb9"		=> "\xee\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xa7"		=> "\xee\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\xa7"		=> "\xee\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xb7"		=> "\xee\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xb8"		=> "\xee\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xa7"		=> "\xee\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x97"		=> "\xee\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x99"		=> "\xee\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x90"		=> "\xee\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa7"		=> "\xee\xb7",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x8b\x95"		=> "\xee\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xa0"		=> "\xee\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x93"		=> "\xee\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa5"		=> "\xee\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa1"		=> "\xee\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xbb"		=> "\xee\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa8"		=> "\xee\xbe",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x8c\x9e"		=> "\xee\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xbf"		=> "\xee\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x9d"		=> "\xee\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x82"		=> "\xee\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\xb0"		=> "\xee\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x97"		=> "\xee\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\xa4"		=> "\xee\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x86"		=> "\xee\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x9e"		=> "\xee\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\xb8"		=> "\xee\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\xb1"		=> "\xee\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x85"		=> "\xee\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x88"		=> "\xee\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x92"		=> "\xee\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa7\x9c"		=> "\xee\xcd",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\xa9"		=> "\xee\xce",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x9a\x9d"		=> "\xee\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xaf"		=> "\xee\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xb3"		=> "\xee\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xbb"		=> "\xee\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x83"		=> "\xee\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x8d"		=> "\xee\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x8f"		=> "\xee\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x91"		=> "\xee\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x95"		=> "\xee\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x97"		=> "\xee\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xa5"		=> "\xee\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xaa"		=> "\xee\xda",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\xab"		=> "\xee\xdb",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\xa4\xa7"		=> "\xee\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xac"		=> "\xee\xdd",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\xa6\x9e"		=> "\xee\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x8e"		=> "\xee\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x99"		=> "\xee\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x9c"		=> "\xee\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\xb5"		=> "\xee\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\xb2"		=> "\xee\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x8f"		=> "\xee\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xb1"		=> "\xee\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xbb"		=> "\xee\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x80"		=> "\xee\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xb0"		=> "\xee\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xab"		=> "\xee\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xad"		=> "\xee\xea",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\xb8\x99"		=> "\xee\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x91"		=> "\xee\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x85\xb0"		=> "\xee\xef",	//SMALL ROMAN NUMERAL ONE
		"\xe2\x85\xb1"		=> "\xee\xf0",	//SMALL ROMAN NUMERAL TWO
		"\xe2\x85\xb2"		=> "\xee\xf1",	//SMALL ROMAN NUMERAL THREE
		"\xe2\x85\xb3"		=> "\xee\xf2",	//SMALL ROMAN NUMERAL FOUR
		"\xe2\x85\xb4"		=> "\xee\xf3",	//SMALL ROMAN NUMERAL FIVE
		"\xe2\x85\xb5"		=> "\xee\xf4",	//SMALL ROMAN NUMERAL SIX
		"\xe2\x85\xb6"		=> "\xee\xf5",	//SMALL ROMAN NUMERAL SEVEN
		"\xe2\x85\xb7"		=> "\xee\xf6",	//SMALL ROMAN NUMERAL EIGHT
		"\xe2\x85\xb8"		=> "\xee\xf7",	//SMALL ROMAN NUMERAL NINE
		"\xe2\x85\xb9"		=> "\xee\xf8",	//SMALL ROMAN NUMERAL TEN
		"\xef\xbf\xa2"		=> "\xee\xf9",	//FULLWIDTH NOT SIGN
		"\xef\xbf\xa4"		=> "\xee\xfa",	//FULLWIDTH BROKEN BAR
		"\xef\xbc\x87"		=> "\xee\xfb",	//FULLWIDTH APOSTROPHE
		"\xef\xbc\x82"		=> "\xee\xfc",	//FULLWIDTH QUOTATION MARK
		"\xe2\x85\xb0"		=> "\xfa\x40",	//SMALL ROMAN NUMERAL ONE
		"\xe2\x85\xb1"		=> "\xfa\x41",	//SMALL ROMAN NUMERAL TWO
		"\xe2\x85\xb2"		=> "\xfa\x42",	//SMALL ROMAN NUMERAL THREE
		"\xe2\x85\xb3"		=> "\xfa\x43",	//SMALL ROMAN NUMERAL FOUR
		"\xe2\x85\xb4"		=> "\xfa\x44",	//SMALL ROMAN NUMERAL FIVE
		"\xe2\x85\xb5"		=> "\xfa\x45",	//SMALL ROMAN NUMERAL SIX
		"\xe2\x85\xb6"		=> "\xfa\x46",	//SMALL ROMAN NUMERAL SEVEN
		"\xe2\x85\xb7"		=> "\xfa\x47",	//SMALL ROMAN NUMERAL EIGHT
		"\xe2\x85\xb8"		=> "\xfa\x48",	//SMALL ROMAN NUMERAL NINE
		"\xe2\x85\xb9"		=> "\xfa\x49",	//SMALL ROMAN NUMERAL TEN
		"\xe2\x85\xa0"		=> "\xfa\x4a",	//ROMAN NUMERAL ONE
		"\xe2\x85\xa1"		=> "\xfa\x4b",	//ROMAN NUMERAL TWO
		"\xe2\x85\xa2"		=> "\xfa\x4c",	//ROMAN NUMERAL THREE
		"\xe2\x85\xa3"		=> "\xfa\x4d",	//ROMAN NUMERAL FOUR
		"\xe2\x85\xa4"		=> "\xfa\x4e",	//ROMAN NUMERAL FIVE
		"\xe2\x85\xa5"		=> "\xfa\x4f",	//ROMAN NUMERAL SIX
		"\xe2\x85\xa6"		=> "\xfa\x50",	//ROMAN NUMERAL SEVEN
		"\xe2\x85\xa7"		=> "\xfa\x51",	//ROMAN NUMERAL EIGHT
		"\xe2\x85\xa8"		=> "\xfa\x52",	//ROMAN NUMERAL NINE
		"\xe2\x85\xa9"		=> "\xfa\x53",	//ROMAN NUMERAL TEN
		"\xef\xbf\xa2"		=> "\xfa\x54",	//FULLWIDTH NOT SIGN
		"\xef\xbf\xa4"		=> "\xfa\x55",	//FULLWIDTH BROKEN BAR
		"\xef\xbc\x87"		=> "\xfa\x56",	//FULLWIDTH APOSTROPHE
		"\xef\xbc\x82"		=> "\xfa\x57",	//FULLWIDTH QUOTATION MARK
		"\xe3\x88\xb1"		=> "\xfa\x58",	//PARENTHESIZED IDEOGRAPH STOCK
		"\xe2\x84\x96"		=> "\xfa\x59",	//NUMERO SIGN
		"\xe2\x84\xa1"		=> "\xfa\x5a",	//TELEPHONE SIGN
		"\xe2\x88\xb5"		=> "\xfa\x5b",	//BECAUSE
		"\xe7\xba\x8a"		=> "\xfa\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4\x9c"		=> "\xfa\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x88"		=> "\xfa\x5e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\x88"		=> "\xfa\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93\x9c"		=> "\xfa\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x89"		=> "\xfa\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xbb"		=> "\xfa\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xb1"		=> "\xfa\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x88"		=> "\xfa\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xb9"		=> "\xfa\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xbb"		=> "\xfa\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\x85"		=> "\xfa\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8\xa8"		=> "\xfa\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xa1"		=> "\xfa\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb\xbc"		=> "\xfa\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x80"		=> "\xfa\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\x83"		=> "\xfa\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc\xb9"		=> "\xfa\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd\x96"		=> "\xfa\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x92"		=> "\xfa\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x8a"		=> "\xfa\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x9a"		=> "\xfa\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe\x94"		=> "\xfa\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\x8d"		=> "\xfa\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x80"		=> "\xfa\x74",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\xa2"		=> "\xfa\x75",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf\xbf"		=> "\xfa\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80\x9e"		=> "\xfa\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x86"		=> "\xfa\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\xb0"		=> "\xfa\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81\x82"		=> "\xfa\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82\x94"		=> "\xfa\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\xb4"		=> "\xfa\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83\x98"		=> "\xfa\x7d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\x8a"		=> "\xfa\x7e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85\xa4"		=> "\xfa\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\x9d"		=> "\xfa\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86\xbe"		=> "\xfa\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87\xac"		=> "\xfa\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88\x95"		=> "\xfa\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\x9c"		=> "\xfa\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xa6"		=> "\xfa\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x80"		=> "\xfa\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b\x9b"		=> "\xfa\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x80"		=> "\xfa\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\x87"		=> "\xfa\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c\xa4"		=> "\xfa\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d\xb2"		=> "\xfa\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\x93"		=> "\xfa\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e\xb2"		=> "\xfa\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f\x9d"		=> "\xfa\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x8e"		=> "\xfa\x90",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe5\x92\x9c"		=> "\xfa\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\x8a"		=> "\xfa\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92\xa9"		=> "\xfa\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93\xbf"		=> "\xfa\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96\x86"		=> "\xfa\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\x99"		=> "\xfa\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d\xa5"		=> "\xfa\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e\xac"		=> "\xfa\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x88"		=> "\xfa\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f\x87"		=> "\xfa\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x8f"		=> "\xfa\x9b",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\x90"		=> "\xfa\x9c",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe5\xa2\x9e"		=> "\xfa\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2\xb2"		=> "\xfa\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4\x8b"		=> "\xfa\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x93"		=> "\xfa\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x9b"		=> "\xfa\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\x9d"		=> "\xfa\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5\xa3"		=> "\xfa\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xa4"		=> "\xfa\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6\xba"		=> "\xfa\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad\x96"		=> "\xfa\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x80"		=> "\xfa\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\xaf"		=> "\xfa\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\x98"		=> "\xfa\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf\xac"		=> "\xfa\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0\x9e"		=> "\xfa\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xa6"		=> "\xfa\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2\xba"		=> "\xfa\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3\xb5"		=> "\xfa\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4\xa7"		=> "\xfa\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\x93"		=> "\xfa\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x91"		=> "\xfa\xb1",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe5\xb5\x82"		=> "\xfa\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5\xad"		=> "\xfa\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xb8"		=> "\xfa\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6\xb9"		=> "\xfa\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7\x90"		=> "\xfa\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xa1"		=> "\xfa\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc\xb4"		=> "\xfa\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd\xa7"		=> "\xfa\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe\xb7"		=> "\xfa\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf\x9e"		=> "\xfa\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81\x9d"		=> "\xfa\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x85"		=> "\xfa\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82\x8a"		=> "\xfa\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x9e"		=> "\xfa\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\x95"		=> "\xfa\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xa0"		=> "\xfa\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83\xb2"		=> "\xfa\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\x91"		=> "\xfa\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xb7"		=> "\xfa\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84\xb0"		=> "\xfa\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86\x98"		=> "\xfa\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88\x93"		=> "\xfa\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a\xa6"		=> "\xfa\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f\xb5"		=> "\xfa\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91\xa0"		=> "\xfa\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92\x9d"		=> "\xfa\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93\x8e"		=> "\xfa\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95\x8e"		=> "\xfa\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x80"		=> "\xfa\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x95"		=> "\xfa\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xbb"		=> "\xfa\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x89"		=> "\xfa\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xae"		=> "\xfa\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\x9e"		=> "\xfa\xd3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98\xa4"		=> "\xfa\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\xa5"		=> "\xfa\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x97"		=> "\xfa\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99\x99"		=> "\xfa\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x92"		=> "\xfa\xd8",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe6\x99\xb3"		=> "\xfa\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\x99"		=> "\xfa\xda",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xa0"		=> "\xfa\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xb2"		=> "\xfa\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a\xbf"		=> "\xfa\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b\xba"		=> "\xfa\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c\x8e"		=> "\xfa\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa4\xa9"		=> "\xfa\xe0",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe6\x9d\xa6"		=> "\xfa\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e\xbb"		=> "\xfa\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x92"		=> "\xfa\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f\x80"		=> "\xfa\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0\x81"		=> "\xfa\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1\x84"		=> "\xfa\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3\x8f"		=> "\xfa\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x93"		=> "\xfa\xe8",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe6\xa5\xa8"		=> "\xfa\xe9",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x94"		=> "\xfa\xea",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe6\xa6\x98"		=> "\xfa\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7\xa2"		=> "\xfa\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8\xb0"		=> "\xfa\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xab"		=> "\xfa\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\x86"		=> "\xfa\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xb3"		=> "\xfa\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9\xbe"		=> "\xfa\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\xa2"		=> "\xfa\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab\xa4"		=> "\xfa\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf\x96"		=> "\xfa\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0\xbf"		=> "\xfa\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\x9c"		=> "\xfa\xf6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2\x86"		=> "\xfa\xf7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1\xaf"		=> "\xfa\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3\x9a"		=> "\xfa\xf9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4\x84"		=> "\xfa\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x87"		=> "\xfa\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5\xaf"		=> "\xfa\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\x96"		=> "\xfb\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6\xac"		=> "\xfb\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\x8f"		=> "\xfb\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb8"		=> "\xfb\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xb2"		=> "\xfb\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7\xbc"		=> "\xfb\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xb9"		=> "\xfb\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9\x9c"		=> "\xfb\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xa7"		=> "\xfb\x48",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8\xbc"		=> "\xfb\x49",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba\xbf"		=> "\xfb\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\x88"		=> "\xfb\x4b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe\xb5"		=> "\xfb\x4c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf\xb5"		=> "\xfb\x4d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x85"		=> "\xfb\x4e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\x87"		=> "\xfb\x4f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80\xa8"		=> "\xfb\x50",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\x85"		=> "\xfb\x51",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82\xab"		=> "\xfb\x52",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x8f"		=> "\xfb\x53",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84\x84"		=> "\xfb\x54",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x9c"		=> "\xfb\x55",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x86"		=> "\xfb\x56",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85\x87"		=> "\xfb\x57",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x95"		=> "\xfb\x58",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\x87\x81"		=> "\xfb\x59",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87\xbe"		=> "\xfb\x5a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xb1"		=> "\xfb\x5b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a\xbe"		=> "\xfb\x5c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c\xa4"		=> "\xfb\x5d",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x96"		=> "\xfb\x5e",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\x8d\xb7"		=> "\xfb\x5f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e\xbd"		=> "\xfb\x60",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x89"		=> "\xfb\x61",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x96"		=> "\xfb\x62",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xa3"		=> "\xfb\x63",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\x92"		=> "\xfb\x64",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\x87"		=> "\xfb\x65",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f\xb5"		=> "\xfb\x66",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xa6"		=> "\xfb\x67",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xaa"		=> "\xfb\x68",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xa9"		=> "\xfb\x69",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90\xae"		=> "\xfb\x6a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91\xa2"		=> "\xfb\x6b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\x89"		=> "\xfb\x6c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92\x9f"		=> "\xfb\x6d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94\x81"		=> "\xfb\x6e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95\xaf"		=> "\xfb\x6f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x82"		=> "\xfb\x70",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x9c"		=> "\xfb\x71",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x9e"		=> "\xfb\x72",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\x9b"		=> "\xfb\x73",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a\xa6"		=> "\xfb\x74",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x97"		=> "\xfb\x75",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\x9d\x86"		=> "\xfb\x76",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a\xaf"		=> "\xfb\x77",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0\xa1"		=> "\xfb\x78",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\x8e"		=> "\xfb\x79",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xa4"		=> "\xfb\x7a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1\xba"		=> "\xfb\x7b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4\xb0"		=> "\xfb\x7c",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x98"		=> "\xfb\x7d",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\x99"		=> "\xfb\x7e",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\x9a"		=> "\xfb\x80",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\xa6\x94"		=> "\xfb\x81",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9b"		=> "\xfb\x82",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\xa6\x9b"		=> "\xfb\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\x91"		=> "\xfb\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab\xa7"		=> "\xfb\x85",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9c"		=> "\xfb\x86",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\xab\xab"		=> "\xfb\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae\x9e"		=> "\xfb\x88",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9d"		=> "\xfb\x89",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe7\xb5\x88"		=> "\xfb\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5\x9c"		=> "\xfb\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xb7"		=> "\xfb\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6\xa0"		=> "\xfb\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7\x96"		=> "\xfb\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9\x92"		=> "\xfb\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd\x87"		=> "\xfb\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe\xa1"		=> "\xfb\x91",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9e"		=> "\xfb\x92",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\x8c\x81"		=> "\xfb\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xa2"		=> "\xfb\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d\xbf"		=> "\xfb\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\x87"		=> "\xfb\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f\xb6"		=> "\xfb\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91\x88"		=> "\xfb\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92\xb4"		=> "\xfb\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x93"		=> "\xfb\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\x99"		=> "\xfb\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95\xab"		=> "\xfb\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\x9f"		=> "\xfb\x9d",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\x96\xb0"		=> "\xfb\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa0"		=> "\xfb\x9f",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\xa1"		=> "\xfb\xa0",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\xa0\x87"		=> "\xfb\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3\xb5"		=> "\xfb\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\x92"		=> "\xfb\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8\xb7"		=> "\xfb\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9\xb9"		=> "\xfb\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xa7"		=> "\xfb\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa\xbe"		=> "\xfb\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab\x9f"		=> "\xfb\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa2"		=> "\xfb\xa9",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\xab\xb6"		=> "\xfb\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\x93"		=> "\xfb\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad\xbf"		=> "\xfb\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xb0"		=> "\xfb\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3\xb4"		=> "\xfb\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4\x92"		=> "\xfb\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5\xb6"		=> "\xfb\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa3"		=> "\xfb\xb1",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe8\xbb\x8f"		=> "\xfb\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa4"		=> "\xfb\xb3",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\xa5"		=> "\xfb\xb4",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x81\xa7"		=> "\xfb\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83\x9e"		=> "\xfb\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa6"		=> "\xfb\xb7",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x84\x95"		=> "\xfb\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84\xa7"		=> "\xfb\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x9a"		=> "\xfb\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x97"		=> "\xfb\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\x9e"		=> "\xfb\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xad"		=> "\xfb\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xae"		=> "\xfb\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xa4"		=> "\xfb\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87\xa5"		=> "\xfb\xc0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x86"		=> "\xfb\xc1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x90"		=> "\xfb\xc2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\x8a"		=> "\xfb\xc3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xba"		=> "\xfb\xc4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x80"		=> "\xfb\xc5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xbc"		=> "\xfb\xc6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x8e"		=> "\xfb\xc7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x99"		=> "\xfb\xc8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\x91"		=> "\xfb\xc9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88\xb9"		=> "\xfb\xca",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xa7"		=> "\xfb\xcb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a\xa7"		=> "\xfb\xcc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xb7"		=> "\xfb\xcd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89\xb8"		=> "\xfb\xce",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xa7"		=> "\xfb\xcf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x97"		=> "\xfb\xd0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x99"		=> "\xfb\xd1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x90"		=> "\xfb\xd2",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa7"		=> "\xfb\xd3",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x8b\x95"		=> "\xfb\xd4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xa0"		=> "\xfb\xd5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\x93"		=> "\xfb\xd6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa5"		=> "\xfb\xd7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\xa1"		=> "\xfb\xd8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xbb"		=> "\xfb\xd9",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xa8"		=> "\xfb\xda",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x8c\x9e"		=> "\xfb\xdb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b\xbf"		=> "\xfb\xdc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x9d"		=> "\xfb\xdd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c\x82"		=> "\xfb\xde",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\xb0"		=> "\xfb\xdf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d\x97"		=> "\xfb\xe0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e\xa4"		=> "\xfb\xe1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x86"		=> "\xfb\xe2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\x9e"		=> "\xfb\xe3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f\xb8"		=> "\xfb\xe4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90\xb1"		=> "\xfb\xe5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x85"		=> "\xfb\xe6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91\x88"		=> "\xfb\xe7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96\x92"		=> "\xfb\xe8",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa7\x9c"		=> "\xfb\xe9",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\xa9"		=> "\xfb\xea",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\x9a\x9d"		=> "\xfb\xeb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a\xaf"		=> "\xfb\xec",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xb3"		=> "\xfb\xed",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c\xbb"		=> "\xfb\xee",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x83"		=> "\xfb\xef",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x8d"		=> "\xfb\xf0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x8f"		=> "\xfb\xf1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x91"		=> "\xfb\xf2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d\x95"		=> "\xfb\xf3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\x97"		=> "\xfb\xf4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1\xa5"		=> "\xfb\xf5",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xaa"		=> "\xfb\xf6",	//CJK COMPATIBILITY IDEOGRAPH
		"\xef\xa8\xab"		=> "\xfb\xf7",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\xa4\xa7"		=> "\xfb\xf8",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xac"		=> "\xfb\xf9",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\xa6\x9e"		=> "\xfb\xfa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9\x8e"		=> "\xfb\xfb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x99"		=> "\xfb\xfc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab\x9c"		=> "\xfc\x40",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\xb5"		=> "\xfc\x41",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad\xb2"		=> "\xfc\x42",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\x8f"		=> "\xfc\x43",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xb1"		=> "\xfc\x44",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae\xbb"		=> "\xfc\x45",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0\x80"		=> "\xfc\x46",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xb0"		=> "\xfc\x47",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5\xab"		=> "\xfc\x48",	//CJK UNIFIED IDEOGRAPH
		"\xef\xa8\xad"		=> "\xfc\x49",	//CJK COMPATIBILITY IDEOGRAPH
		"\xe9\xb8\x99"		=> "\xfc\x4a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb\x91"		=> "\xfc\x4b",	//CJK UNIFIED IDEOGRAPH
	);

	private static $decode_map = array(
		"\x00"		=> "\x00",	//NULL
		"\x01"		=> "\x01",	//START OF HEADING
		"\x02"		=> "\x02",	//START OF TEXT
		"\x03"		=> "\x03",	//END OF TEXT
		"\x04"		=> "\x04",	//END OF TRANSMISSION
		"\x05"		=> "\x05",	//ENQUIRY
		"\x06"		=> "\x06",	//ACKNOWLEDGE
		"\x07"		=> "\x07",	//BELL
		"\x08"		=> "\x08",	//BACKSPACE
		"\x09"		=> "\x09",	//HORIZONTAL TABULATION
		"\x0a"		=> "\x0a",	//LINE FEED
		"\x0b"		=> "\x0b",	//VERTICAL TABULATION
		"\x0c"		=> "\x0c",	//FORM FEED
		"\x0d"		=> "\x0d",	//CARRIAGE RETURN
		"\x0e"		=> "\x0e",	//SHIFT OUT
		"\x0f"		=> "\x0f",	//SHIFT IN
		"\x10"		=> "\x10",	//DATA LINK ESCAPE
		"\x11"		=> "\x11",	//DEVICE CONTROL ONE
		"\x12"		=> "\x12",	//DEVICE CONTROL TWO
		"\x13"		=> "\x13",	//DEVICE CONTROL THREE
		"\x14"		=> "\x14",	//DEVICE CONTROL FOUR
		"\x15"		=> "\x15",	//NEGATIVE ACKNOWLEDGE
		"\x16"		=> "\x16",	//SYNCHRONOUS IDLE
		"\x17"		=> "\x17",	//END OF TRANSMISSION BLOCK
		"\x18"		=> "\x18",	//CANCEL
		"\x19"		=> "\x19",	//END OF MEDIUM
		"\x1a"		=> "\x1a",	//SUBSTITUTE
		"\x1b"		=> "\x1b",	//ESCAPE
		"\x1c"		=> "\x1c",	//FILE SEPARATOR
		"\x1d"		=> "\x1d",	//GROUP SEPARATOR
		"\x1e"		=> "\x1e",	//RECORD SEPARATOR
		"\x1f"		=> "\x1f",	//UNIT SEPARATOR
		"\x20"		=> "\x20",	//SPACE
		"\x21"		=> "\x21",	//EXCLAMATION MARK
		"\x22"		=> "\x22",	//QUOTATION MARK
		"\x23"		=> "\x23",	//NUMBER SIGN
		"\x24"		=> "\x24",	//DOLLAR SIGN
		"\x25"		=> "\x25",	//PERCENT SIGN
		"\x26"		=> "\x26",	//AMPERSAND
		"\x27"		=> "\x27",	//APOSTROPHE
		"\x28"		=> "\x28",	//LEFT PARENTHESIS
		"\x29"		=> "\x29",	//RIGHT PARENTHESIS
		"\x2a"		=> "\x2a",	//ASTERISK
		"\x2b"		=> "\x2b",	//PLUS SIGN
		"\x2c"		=> "\x2c",	//COMMA
		"\x2d"		=> "\x2d",	//HYPHEN-MINUS
		"\x2e"		=> "\x2e",	//FULL STOP
		"\x2f"		=> "\x2f",	//SOLIDUS
		"\x30"		=> "\x30",	//DIGIT ZERO
		"\x31"		=> "\x31",	//DIGIT ONE
		"\x32"		=> "\x32",	//DIGIT TWO
		"\x33"		=> "\x33",	//DIGIT THREE
		"\x34"		=> "\x34",	//DIGIT FOUR
		"\x35"		=> "\x35",	//DIGIT FIVE
		"\x36"		=> "\x36",	//DIGIT SIX
		"\x37"		=> "\x37",	//DIGIT SEVEN
		"\x38"		=> "\x38",	//DIGIT EIGHT
		"\x39"		=> "\x39",	//DIGIT NINE
		"\x3a"		=> "\x3a",	//COLON
		"\x3b"		=> "\x3b",	//SEMICOLON
		"\x3c"		=> "\x3c",	//LESS-THAN SIGN
		"\x3d"		=> "\x3d",	//EQUALS SIGN
		"\x3e"		=> "\x3e",	//GREATER-THAN SIGN
		"\x3f"		=> "\x3f",	//QUESTION MARK
		"\x40"		=> "\x40",	//COMMERCIAL AT
		"\x41"		=> "\x41",	//LATIN CAPITAL LETTER A
		"\x42"		=> "\x42",	//LATIN CAPITAL LETTER B
		"\x43"		=> "\x43",	//LATIN CAPITAL LETTER C
		"\x44"		=> "\x44",	//LATIN CAPITAL LETTER D
		"\x45"		=> "\x45",	//LATIN CAPITAL LETTER E
		"\x46"		=> "\x46",	//LATIN CAPITAL LETTER F
		"\x47"		=> "\x47",	//LATIN CAPITAL LETTER G
		"\x48"		=> "\x48",	//LATIN CAPITAL LETTER H
		"\x49"		=> "\x49",	//LATIN CAPITAL LETTER I
		"\x4a"		=> "\x4a",	//LATIN CAPITAL LETTER J
		"\x4b"		=> "\x4b",	//LATIN CAPITAL LETTER K
		"\x4c"		=> "\x4c",	//LATIN CAPITAL LETTER L
		"\x4d"		=> "\x4d",	//LATIN CAPITAL LETTER M
		"\x4e"		=> "\x4e",	//LATIN CAPITAL LETTER N
		"\x4f"		=> "\x4f",	//LATIN CAPITAL LETTER O
		"\x50"		=> "\x50",	//LATIN CAPITAL LETTER P
		"\x51"		=> "\x51",	//LATIN CAPITAL LETTER Q
		"\x52"		=> "\x52",	//LATIN CAPITAL LETTER R
		"\x53"		=> "\x53",	//LATIN CAPITAL LETTER S
		"\x54"		=> "\x54",	//LATIN CAPITAL LETTER T
		"\x55"		=> "\x55",	//LATIN CAPITAL LETTER U
		"\x56"		=> "\x56",	//LATIN CAPITAL LETTER V
		"\x57"		=> "\x57",	//LATIN CAPITAL LETTER W
		"\x58"		=> "\x58",	//LATIN CAPITAL LETTER X
		"\x59"		=> "\x59",	//LATIN CAPITAL LETTER Y
		"\x5a"		=> "\x5a",	//LATIN CAPITAL LETTER Z
		"\x5b"		=> "\x5b",	//LEFT SQUARE BRACKET
		"\x5c"		=> "\x5c",	//REVERSE SOLIDUS
		"\x5d"		=> "\x5d",	//RIGHT SQUARE BRACKET
		"\x5e"		=> "\x5e",	//CIRCUMFLEX ACCENT
		"\x5f"		=> "\x5f",	//LOW LINE
		"\x60"		=> "\x60",	//GRAVE ACCENT
		"\x61"		=> "\x61",	//LATIN SMALL LETTER A
		"\x62"		=> "\x62",	//LATIN SMALL LETTER B
		"\x63"		=> "\x63",	//LATIN SMALL LETTER C
		"\x64"		=> "\x64",	//LATIN SMALL LETTER D
		"\x65"		=> "\x65",	//LATIN SMALL LETTER E
		"\x66"		=> "\x66",	//LATIN SMALL LETTER F
		"\x67"		=> "\x67",	//LATIN SMALL LETTER G
		"\x68"		=> "\x68",	//LATIN SMALL LETTER H
		"\x69"		=> "\x69",	//LATIN SMALL LETTER I
		"\x6a"		=> "\x6a",	//LATIN SMALL LETTER J
		"\x6b"		=> "\x6b",	//LATIN SMALL LETTER K
		"\x6c"		=> "\x6c",	//LATIN SMALL LETTER L
		"\x6d"		=> "\x6d",	//LATIN SMALL LETTER M
		"\x6e"		=> "\x6e",	//LATIN SMALL LETTER N
		"\x6f"		=> "\x6f",	//LATIN SMALL LETTER O
		"\x70"		=> "\x70",	//LATIN SMALL LETTER P
		"\x71"		=> "\x71",	//LATIN SMALL LETTER Q
		"\x72"		=> "\x72",	//LATIN SMALL LETTER R
		"\x73"		=> "\x73",	//LATIN SMALL LETTER S
		"\x74"		=> "\x74",	//LATIN SMALL LETTER T
		"\x75"		=> "\x75",	//LATIN SMALL LETTER U
		"\x76"		=> "\x76",	//LATIN SMALL LETTER V
		"\x77"		=> "\x77",	//LATIN SMALL LETTER W
		"\x78"		=> "\x78",	//LATIN SMALL LETTER X
		"\x79"		=> "\x79",	//LATIN SMALL LETTER Y
		"\x7a"		=> "\x7a",	//LATIN SMALL LETTER Z
		"\x7b"		=> "\x7b",	//LEFT CURLY BRACKET
		"\x7c"		=> "\x7c",	//VERTICAL LINE
		"\x7d"		=> "\x7d",	//RIGHT CURLY BRACKET
		"\x7e"		=> "\x7e",	//TILDE
		"\x7f"		=> "\x7f",	//DELETE
		"\xa1"		=> "\xef\xbd\xa1",	//HALFWIDTH IDEOGRAPHIC FULL STOP
		"\xa2"		=> "\xef\xbd\xa2",	//HALFWIDTH LEFT CORNER BRACKET
		"\xa3"		=> "\xef\xbd\xa3",	//HALFWIDTH RIGHT CORNER BRACKET
		"\xa4"		=> "\xef\xbd\xa4",	//HALFWIDTH IDEOGRAPHIC COMMA
		"\xa5"		=> "\xef\xbd\xa5",	//HALFWIDTH KATAKANA MIDDLE DOT
		"\xa6"		=> "\xef\xbd\xa6",	//HALFWIDTH KATAKANA LETTER WO
		"\xa7"		=> "\xef\xbd\xa7",	//HALFWIDTH KATAKANA LETTER SMALL A
		"\xa8"		=> "\xef\xbd\xa8",	//HALFWIDTH KATAKANA LETTER SMALL I
		"\xa9"		=> "\xef\xbd\xa9",	//HALFWIDTH KATAKANA LETTER SMALL U
		"\xaa"		=> "\xef\xbd\xaa",	//HALFWIDTH KATAKANA LETTER SMALL E
		"\xab"		=> "\xef\xbd\xab",	//HALFWIDTH KATAKANA LETTER SMALL O
		"\xac"		=> "\xef\xbd\xac",	//HALFWIDTH KATAKANA LETTER SMALL YA
		"\xad"		=> "\xef\xbd\xad",	//HALFWIDTH KATAKANA LETTER SMALL YU
		"\xae"		=> "\xef\xbd\xae",	//HALFWIDTH KATAKANA LETTER SMALL YO
		"\xaf"		=> "\xef\xbd\xaf",	//HALFWIDTH KATAKANA LETTER SMALL TU
		"\xb0"		=> "\xef\xbd\xb0",	//HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
		"\xb1"		=> "\xef\xbd\xb1",	//HALFWIDTH KATAKANA LETTER A
		"\xb2"		=> "\xef\xbd\xb2",	//HALFWIDTH KATAKANA LETTER I
		"\xb3"		=> "\xef\xbd\xb3",	//HALFWIDTH KATAKANA LETTER U
		"\xb4"		=> "\xef\xbd\xb4",	//HALFWIDTH KATAKANA LETTER E
		"\xb5"		=> "\xef\xbd\xb5",	//HALFWIDTH KATAKANA LETTER O
		"\xb6"		=> "\xef\xbd\xb6",	//HALFWIDTH KATAKANA LETTER KA
		"\xb7"		=> "\xef\xbd\xb7",	//HALFWIDTH KATAKANA LETTER KI
		"\xb8"		=> "\xef\xbd\xb8",	//HALFWIDTH KATAKANA LETTER KU
		"\xb9"		=> "\xef\xbd\xb9",	//HALFWIDTH KATAKANA LETTER KE
		"\xba"		=> "\xef\xbd\xba",	//HALFWIDTH KATAKANA LETTER KO
		"\xbb"		=> "\xef\xbd\xbb",	//HALFWIDTH KATAKANA LETTER SA
		"\xbc"		=> "\xef\xbd\xbc",	//HALFWIDTH KATAKANA LETTER SI
		"\xbd"		=> "\xef\xbd\xbd",	//HALFWIDTH KATAKANA LETTER SU
		"\xbe"		=> "\xef\xbd\xbe",	//HALFWIDTH KATAKANA LETTER SE
		"\xbf"		=> "\xef\xbd\xbf",	//HALFWIDTH KATAKANA LETTER SO
		"\xc0"		=> "\xef\xbe\x80",	//HALFWIDTH KATAKANA LETTER TA
		"\xc1"		=> "\xef\xbe\x81",	//HALFWIDTH KATAKANA LETTER TI
		"\xc2"		=> "\xef\xbe\x82",	//HALFWIDTH KATAKANA LETTER TU
		"\xc3"		=> "\xef\xbe\x83",	//HALFWIDTH KATAKANA LETTER TE
		"\xc4"		=> "\xef\xbe\x84",	//HALFWIDTH KATAKANA LETTER TO
		"\xc5"		=> "\xef\xbe\x85",	//HALFWIDTH KATAKANA LETTER NA
		"\xc6"		=> "\xef\xbe\x86",	//HALFWIDTH KATAKANA LETTER NI
		"\xc7"		=> "\xef\xbe\x87",	//HALFWIDTH KATAKANA LETTER NU
		"\xc8"		=> "\xef\xbe\x88",	//HALFWIDTH KATAKANA LETTER NE
		"\xc9"		=> "\xef\xbe\x89",	//HALFWIDTH KATAKANA LETTER NO
		"\xca"		=> "\xef\xbe\x8a",	//HALFWIDTH KATAKANA LETTER HA
		"\xcb"		=> "\xef\xbe\x8b",	//HALFWIDTH KATAKANA LETTER HI
		"\xcc"		=> "\xef\xbe\x8c",	//HALFWIDTH KATAKANA LETTER HU
		"\xcd"		=> "\xef\xbe\x8d",	//HALFWIDTH KATAKANA LETTER HE
		"\xce"		=> "\xef\xbe\x8e",	//HALFWIDTH KATAKANA LETTER HO
		"\xcf"		=> "\xef\xbe\x8f",	//HALFWIDTH KATAKANA LETTER MA
		"\xd0"		=> "\xef\xbe\x90",	//HALFWIDTH KATAKANA LETTER MI
		"\xd1"		=> "\xef\xbe\x91",	//HALFWIDTH KATAKANA LETTER MU
		"\xd2"		=> "\xef\xbe\x92",	//HALFWIDTH KATAKANA LETTER ME
		"\xd3"		=> "\xef\xbe\x93",	//HALFWIDTH KATAKANA LETTER MO
		"\xd4"		=> "\xef\xbe\x94",	//HALFWIDTH KATAKANA LETTER YA
		"\xd5"		=> "\xef\xbe\x95",	//HALFWIDTH KATAKANA LETTER YU
		"\xd6"		=> "\xef\xbe\x96",	//HALFWIDTH KATAKANA LETTER YO
		"\xd7"		=> "\xef\xbe\x97",	//HALFWIDTH KATAKANA LETTER RA
		"\xd8"		=> "\xef\xbe\x98",	//HALFWIDTH KATAKANA LETTER RI
		"\xd9"		=> "\xef\xbe\x99",	//HALFWIDTH KATAKANA LETTER RU
		"\xda"		=> "\xef\xbe\x9a",	//HALFWIDTH KATAKANA LETTER RE
		"\xdb"		=> "\xef\xbe\x9b",	//HALFWIDTH KATAKANA LETTER RO
		"\xdc"		=> "\xef\xbe\x9c",	//HALFWIDTH KATAKANA LETTER WA
		"\xdd"		=> "\xef\xbe\x9d",	//HALFWIDTH KATAKANA LETTER N
		"\xde"		=> "\xef\xbe\x9e",	//HALFWIDTH KATAKANA VOICED SOUND MARK
		"\xdf"		=> "\xef\xbe\x9f",	//HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
		"\x81\x40"		=> "\xe3\x80\x80",	//IDEOGRAPHIC SPACE
		"\x81\x41"		=> "\xe3\x80\x81",	//IDEOGRAPHIC COMMA
		"\x81\x42"		=> "\xe3\x80\x82",	//IDEOGRAPHIC FULL STOP
		"\x81\x43"		=> "\xef\xbc\x8c",	//FULLWIDTH COMMA
		"\x81\x44"		=> "\xef\xbc\x8e",	//FULLWIDTH FULL STOP
		"\x81\x45"		=> "\xe3\x83\xbb",	//KATAKANA MIDDLE DOT
		"\x81\x46"		=> "\xef\xbc\x9a",	//FULLWIDTH COLON
		"\x81\x47"		=> "\xef\xbc\x9b",	//FULLWIDTH SEMICOLON
		"\x81\x48"		=> "\xef\xbc\x9f",	//FULLWIDTH QUESTION MARK
		"\x81\x49"		=> "\xef\xbc\x81",	//FULLWIDTH EXCLAMATION MARK
		"\x81\x4a"		=> "\xe3\x82\x9b",	//KATAKANA-HIRAGANA VOICED SOUND MARK
		"\x81\x4b"		=> "\xe3\x82\x9c",	//KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
		"\x81\x4c"		=> "\xc2\xb4",	//ACUTE ACCENT
		"\x81\x4d"		=> "\xef\xbd\x80",	//FULLWIDTH GRAVE ACCENT
		"\x81\x4e"		=> "\xc2\xa8",	//DIAERESIS
		"\x81\x4f"		=> "\xef\xbc\xbe",	//FULLWIDTH CIRCUMFLEX ACCENT
		"\x81\x50"		=> "\xef\xbf\xa3",	//FULLWIDTH MACRON
		"\x81\x51"		=> "\xef\xbc\xbf",	//FULLWIDTH LOW LINE
		"\x81\x52"		=> "\xe3\x83\xbd",	//KATAKANA ITERATION MARK
		"\x81\x53"		=> "\xe3\x83\xbe",	//KATAKANA VOICED ITERATION MARK
		"\x81\x54"		=> "\xe3\x82\x9d",	//HIRAGANA ITERATION MARK
		"\x81\x55"		=> "\xe3\x82\x9e",	//HIRAGANA VOICED ITERATION MARK
		"\x81\x56"		=> "\xe3\x80\x83",	//DITTO MARK
		"\x81\x57"		=> "\xe4\xbb\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x81\x58"		=> "\xe3\x80\x85",	//IDEOGRAPHIC ITERATION MARK
		"\x81\x59"		=> "\xe3\x80\x86",	//IDEOGRAPHIC CLOSING MARK
		"\x81\x5a"		=> "\xe3\x80\x87",	//IDEOGRAPHIC NUMBER ZERO
		"\x81\x5b"		=> "\xe3\x83\xbc",	//KATAKANA-HIRAGANA PROLONGED SOUND MARK
		"\x81\x5c"		=> "\xe2\x80\x95",	//HORIZONTAL BAR
		"\x81\x5d"		=> "\xe2\x80\x90",	//HYPHEN
		"\x81\x5e"		=> "\xef\xbc\x8f",	//FULLWIDTH SOLIDUS
		"\x81\x5f"		=> "\xef\xbc\xbc",	//FULLWIDTH REVERSE SOLIDUS
		"\x81\x60"		=> "\xef\xbd\x9e",	//FULLWIDTH TILDE
		"\x81\x61"		=> "\xe2\x88\xa5",	//PARALLEL TO
		"\x81\x62"		=> "\xef\xbd\x9c",	//FULLWIDTH VERTICAL LINE
		"\x81\x63"		=> "\xe2\x80\xa6",	//HORIZONTAL ELLIPSIS
		"\x81\x64"		=> "\xe2\x80\xa5",	//TWO DOT LEADER
		"\x81\x65"		=> "\xe2\x80\x98",	//LEFT SINGLE QUOTATION MARK
		"\x81\x66"		=> "\xe2\x80\x99",	//RIGHT SINGLE QUOTATION MARK
		"\x81\x67"		=> "\xe2\x80\x9c",	//LEFT DOUBLE QUOTATION MARK
		"\x81\x68"		=> "\xe2\x80\x9d",	//RIGHT DOUBLE QUOTATION MARK
		"\x81\x69"		=> "\xef\xbc\x88",	//FULLWIDTH LEFT PARENTHESIS
		"\x81\x6a"		=> "\xef\xbc\x89",	//FULLWIDTH RIGHT PARENTHESIS
		"\x81\x6b"		=> "\xe3\x80\x94",	//LEFT TORTOISE SHELL BRACKET
		"\x81\x6c"		=> "\xe3\x80\x95",	//RIGHT TORTOISE SHELL BRACKET
		"\x81\x6d"		=> "\xef\xbc\xbb",	//FULLWIDTH LEFT SQUARE BRACKET
		"\x81\x6e"		=> "\xef\xbc\xbd",	//FULLWIDTH RIGHT SQUARE BRACKET
		"\x81\x6f"		=> "\xef\xbd\x9b",	//FULLWIDTH LEFT CURLY BRACKET
		"\x81\x70"		=> "\xef\xbd\x9d",	//FULLWIDTH RIGHT CURLY BRACKET
		"\x81\x71"		=> "\xe3\x80\x88",	//LEFT ANGLE BRACKET
		"\x81\x72"		=> "\xe3\x80\x89",	//RIGHT ANGLE BRACKET
		"\x81\x73"		=> "\xe3\x80\x8a",	//LEFT DOUBLE ANGLE BRACKET
		"\x81\x74"		=> "\xe3\x80\x8b",	//RIGHT DOUBLE ANGLE BRACKET
		"\x81\x75"		=> "\xe3\x80\x8c",	//LEFT CORNER BRACKET
		"\x81\x76"		=> "\xe3\x80\x8d",	//RIGHT CORNER BRACKET
		"\x81\x77"		=> "\xe3\x80\x8e",	//LEFT WHITE CORNER BRACKET
		"\x81\x78"		=> "\xe3\x80\x8f",	//RIGHT WHITE CORNER BRACKET
		"\x81\x79"		=> "\xe3\x80\x90",	//LEFT BLACK LENTICULAR BRACKET
		"\x81\x7a"		=> "\xe3\x80\x91",	//RIGHT BLACK LENTICULAR BRACKET
		"\x81\x7b"		=> "\xef\xbc\x8b",	//FULLWIDTH PLUS SIGN
		"\x81\x7c"		=> "\xef\xbc\x8d",	//FULLWIDTH HYPHEN-MINUS
		"\x81\x7d"		=> "\xc2\xb1",	//PLUS-MINUS SIGN
		"\x81\x7e"		=> "\xc3\x97",	//MULTIPLICATION SIGN
		"\x81\x80"		=> "\xc3\xb7",	//DIVISION SIGN
		"\x81\x81"		=> "\xef\xbc\x9d",	//FULLWIDTH EQUALS SIGN
		"\x81\x82"		=> "\xe2\x89\xa0",	//NOT EQUAL TO
		"\x81\x83"		=> "\xef\xbc\x9c",	//FULLWIDTH LESS-THAN SIGN
		"\x81\x84"		=> "\xef\xbc\x9e",	//FULLWIDTH GREATER-THAN SIGN
		"\x81\x85"		=> "\xe2\x89\xa6",	//LESS-THAN OVER EQUAL TO
		"\x81\x86"		=> "\xe2\x89\xa7",	//GREATER-THAN OVER EQUAL TO
		"\x81\x87"		=> "\xe2\x88\x9e",	//INFINITY
		"\x81\x88"		=> "\xe2\x88\xb4",	//THEREFORE
		"\x81\x89"		=> "\xe2\x99\x82",	//MALE SIGN
		"\x81\x8a"		=> "\xe2\x99\x80",	//FEMALE SIGN
		"\x81\x8b"		=> "\xc2\xb0",	//DEGREE SIGN
		"\x81\x8c"		=> "\xe2\x80\xb2",	//PRIME
		"\x81\x8d"		=> "\xe2\x80\xb3",	//DOUBLE PRIME
		"\x81\x8e"		=> "\xe2\x84\x83",	//DEGREE CELSIUS
		"\x81\x8f"		=> "\xef\xbf\xa5",	//FULLWIDTH YEN SIGN
		"\x81\x90"		=> "\xef\xbc\x84",	//FULLWIDTH DOLLAR SIGN
		"\x81\x91"		=> "\xef\xbf\xa0",	//FULLWIDTH CENT SIGN
		"\x81\x92"		=> "\xef\xbf\xa1",	//FULLWIDTH POUND SIGN
		"\x81\x93"		=> "\xef\xbc\x85",	//FULLWIDTH PERCENT SIGN
		"\x81\x94"		=> "\xef\xbc\x83",	//FULLWIDTH NUMBER SIGN
		"\x81\x95"		=> "\xef\xbc\x86",	//FULLWIDTH AMPERSAND
		"\x81\x96"		=> "\xef\xbc\x8a",	//FULLWIDTH ASTERISK
		"\x81\x97"		=> "\xef\xbc\xa0",	//FULLWIDTH COMMERCIAL AT
		"\x81\x98"		=> "\xc2\xa7",	//SECTION SIGN
		"\x81\x99"		=> "\xe2\x98\x86",	//WHITE STAR
		"\x81\x9a"		=> "\xe2\x98\x85",	//BLACK STAR
		"\x81\x9b"		=> "\xe2\x97\x8b",	//WHITE CIRCLE
		"\x81\x9c"		=> "\xe2\x97\x8f",	//BLACK CIRCLE
		"\x81\x9d"		=> "\xe2\x97\x8e",	//BULLSEYE
		"\x81\x9e"		=> "\xe2\x97\x87",	//WHITE DIAMOND
		"\x81\x9f"		=> "\xe2\x97\x86",	//BLACK DIAMOND
		"\x81\xa0"		=> "\xe2\x96\xa1",	//WHITE SQUARE
		"\x81\xa1"		=> "\xe2\x96\xa0",	//BLACK SQUARE
		"\x81\xa2"		=> "\xe2\x96\xb3",	//WHITE UP-POINTING TRIANGLE
		"\x81\xa3"		=> "\xe2\x96\xb2",	//BLACK UP-POINTING TRIANGLE
		"\x81\xa4"		=> "\xe2\x96\xbd",	//WHITE DOWN-POINTING TRIANGLE
		"\x81\xa5"		=> "\xe2\x96\xbc",	//BLACK DOWN-POINTING TRIANGLE
		"\x81\xa6"		=> "\xe2\x80\xbb",	//REFERENCE MARK
		"\x81\xa7"		=> "\xe3\x80\x92",	//POSTAL MARK
		"\x81\xa8"		=> "\xe2\x86\x92",	//RIGHTWARDS ARROW
		"\x81\xa9"		=> "\xe2\x86\x90",	//LEFTWARDS ARROW
		"\x81\xaa"		=> "\xe2\x86\x91",	//UPWARDS ARROW
		"\x81\xab"		=> "\xe2\x86\x93",	//DOWNWARDS ARROW
		"\x81\xac"		=> "\xe3\x80\x93",	//GETA MARK
		"\x81\xb8"		=> "\xe2\x88\x88",	//ELEMENT OF
		"\x81\xb9"		=> "\xe2\x88\x8b",	//CONTAINS AS MEMBER
		"\x81\xba"		=> "\xe2\x8a\x86",	//SUBSET OF OR EQUAL TO
		"\x81\xbb"		=> "\xe2\x8a\x87",	//SUPERSET OF OR EQUAL TO
		"\x81\xbc"		=> "\xe2\x8a\x82",	//SUBSET OF
		"\x81\xbd"		=> "\xe2\x8a\x83",	//SUPERSET OF
		"\x81\xbe"		=> "\xe2\x88\xaa",	//UNION
		"\x81\xbf"		=> "\xe2\x88\xa9",	//INTERSECTION
		"\x81\xc8"		=> "\xe2\x88\xa7",	//LOGICAL AND
		"\x81\xc9"		=> "\xe2\x88\xa8",	//LOGICAL OR
		"\x81\xca"		=> "\xef\xbf\xa2",	//FULLWIDTH NOT SIGN
		"\x81\xcb"		=> "\xe2\x87\x92",	//RIGHTWARDS DOUBLE ARROW
		"\x81\xcc"		=> "\xe2\x87\x94",	//LEFT RIGHT DOUBLE ARROW
		"\x81\xcd"		=> "\xe2\x88\x80",	//FOR ALL
		"\x81\xce"		=> "\xe2\x88\x83",	//THERE EXISTS
		"\x81\xda"		=> "\xe2\x88\xa0",	//ANGLE
		"\x81\xdb"		=> "\xe2\x8a\xa5",	//UP TACK
		"\x81\xdc"		=> "\xe2\x8c\x92",	//ARC
		"\x81\xdd"		=> "\xe2\x88\x82",	//PARTIAL DIFFERENTIAL
		"\x81\xde"		=> "\xe2\x88\x87",	//NABLA
		"\x81\xdf"		=> "\xe2\x89\xa1",	//IDENTICAL TO
		"\x81\xe0"		=> "\xe2\x89\x92",	//APPROXIMATELY EQUAL TO OR THE IMAGE OF
		"\x81\xe1"		=> "\xe2\x89\xaa",	//MUCH LESS-THAN
		"\x81\xe2"		=> "\xe2\x89\xab",	//MUCH GREATER-THAN
		"\x81\xe3"		=> "\xe2\x88\x9a",	//SQUARE ROOT
		"\x81\xe4"		=> "\xe2\x88\xbd",	//REVERSED TILDE
		"\x81\xe5"		=> "\xe2\x88\x9d",	//PROPORTIONAL TO
		"\x81\xe6"		=> "\xe2\x88\xb5",	//BECAUSE
		"\x81\xe7"		=> "\xe2\x88\xab",	//INTEGRAL
		"\x81\xe8"		=> "\xe2\x88\xac",	//DOUBLE INTEGRAL
		"\x81\xf0"		=> "\xe2\x84\xab",	//ANGSTROM SIGN
		"\x81\xf1"		=> "\xe2\x80\xb0",	//PER MILLE SIGN
		"\x81\xf2"		=> "\xe2\x99\xaf",	//MUSIC SHARP SIGN
		"\x81\xf3"		=> "\xe2\x99\xad",	//MUSIC FLAT SIGN
		"\x81\xf4"		=> "\xe2\x99\xaa",	//EIGHTH NOTE
		"\x81\xf5"		=> "\xe2\x80\xa0",	//DAGGER
		"\x81\xf6"		=> "\xe2\x80\xa1",	//DOUBLE DAGGER
		"\x81\xf7"		=> "\xc2\xb6",	//PILCROW SIGN
		"\x81\xfc"		=> "\xe2\x97\xaf",	//LARGE CIRCLE
		"\x82\x4f"		=> "\xef\xbc\x90",	//FULLWIDTH DIGIT ZERO
		"\x82\x50"		=> "\xef\xbc\x91",	//FULLWIDTH DIGIT ONE
		"\x82\x51"		=> "\xef\xbc\x92",	//FULLWIDTH DIGIT TWO
		"\x82\x52"		=> "\xef\xbc\x93",	//FULLWIDTH DIGIT THREE
		"\x82\x53"		=> "\xef\xbc\x94",	//FULLWIDTH DIGIT FOUR
		"\x82\x54"		=> "\xef\xbc\x95",	//FULLWIDTH DIGIT FIVE
		"\x82\x55"		=> "\xef\xbc\x96",	//FULLWIDTH DIGIT SIX
		"\x82\x56"		=> "\xef\xbc\x97",	//FULLWIDTH DIGIT SEVEN
		"\x82\x57"		=> "\xef\xbc\x98",	//FULLWIDTH DIGIT EIGHT
		"\x82\x58"		=> "\xef\xbc\x99",	//FULLWIDTH DIGIT NINE
		"\x82\x60"		=> "\xef\xbc\xa1",	//FULLWIDTH LATIN CAPITAL LETTER A
		"\x82\x61"		=> "\xef\xbc\xa2",	//FULLWIDTH LATIN CAPITAL LETTER B
		"\x82\x62"		=> "\xef\xbc\xa3",	//FULLWIDTH LATIN CAPITAL LETTER C
		"\x82\x63"		=> "\xef\xbc\xa4",	//FULLWIDTH LATIN CAPITAL LETTER D
		"\x82\x64"		=> "\xef\xbc\xa5",	//FULLWIDTH LATIN CAPITAL LETTER E
		"\x82\x65"		=> "\xef\xbc\xa6",	//FULLWIDTH LATIN CAPITAL LETTER F
		"\x82\x66"		=> "\xef\xbc\xa7",	//FULLWIDTH LATIN CAPITAL LETTER G
		"\x82\x67"		=> "\xef\xbc\xa8",	//FULLWIDTH LATIN CAPITAL LETTER H
		"\x82\x68"		=> "\xef\xbc\xa9",	//FULLWIDTH LATIN CAPITAL LETTER I
		"\x82\x69"		=> "\xef\xbc\xaa",	//FULLWIDTH LATIN CAPITAL LETTER J
		"\x82\x6a"		=> "\xef\xbc\xab",	//FULLWIDTH LATIN CAPITAL LETTER K
		"\x82\x6b"		=> "\xef\xbc\xac",	//FULLWIDTH LATIN CAPITAL LETTER L
		"\x82\x6c"		=> "\xef\xbc\xad",	//FULLWIDTH LATIN CAPITAL LETTER M
		"\x82\x6d"		=> "\xef\xbc\xae",	//FULLWIDTH LATIN CAPITAL LETTER N
		"\x82\x6e"		=> "\xef\xbc\xaf",	//FULLWIDTH LATIN CAPITAL LETTER O
		"\x82\x6f"		=> "\xef\xbc\xb0",	//FULLWIDTH LATIN CAPITAL LETTER P
		"\x82\x70"		=> "\xef\xbc\xb1",	//FULLWIDTH LATIN CAPITAL LETTER Q
		"\x82\x71"		=> "\xef\xbc\xb2",	//FULLWIDTH LATIN CAPITAL LETTER R
		"\x82\x72"		=> "\xef\xbc\xb3",	//FULLWIDTH LATIN CAPITAL LETTER S
		"\x82\x73"		=> "\xef\xbc\xb4",	//FULLWIDTH LATIN CAPITAL LETTER T
		"\x82\x74"		=> "\xef\xbc\xb5",	//FULLWIDTH LATIN CAPITAL LETTER U
		"\x82\x75"		=> "\xef\xbc\xb6",	//FULLWIDTH LATIN CAPITAL LETTER V
		"\x82\x76"		=> "\xef\xbc\xb7",	//FULLWIDTH LATIN CAPITAL LETTER W
		"\x82\x77"		=> "\xef\xbc\xb8",	//FULLWIDTH LATIN CAPITAL LETTER X
		"\x82\x78"		=> "\xef\xbc\xb9",	//FULLWIDTH LATIN CAPITAL LETTER Y
		"\x82\x79"		=> "\xef\xbc\xba",	//FULLWIDTH LATIN CAPITAL LETTER Z
		"\x82\x81"		=> "\xef\xbd\x81",	//FULLWIDTH LATIN SMALL LETTER A
		"\x82\x82"		=> "\xef\xbd\x82",	//FULLWIDTH LATIN SMALL LETTER B
		"\x82\x83"		=> "\xef\xbd\x83",	//FULLWIDTH LATIN SMALL LETTER C
		"\x82\x84"		=> "\xef\xbd\x84",	//FULLWIDTH LATIN SMALL LETTER D
		"\x82\x85"		=> "\xef\xbd\x85",	//FULLWIDTH LATIN SMALL LETTER E
		"\x82\x86"		=> "\xef\xbd\x86",	//FULLWIDTH LATIN SMALL LETTER F
		"\x82\x87"		=> "\xef\xbd\x87",	//FULLWIDTH LATIN SMALL LETTER G
		"\x82\x88"		=> "\xef\xbd\x88",	//FULLWIDTH LATIN SMALL LETTER H
		"\x82\x89"		=> "\xef\xbd\x89",	//FULLWIDTH LATIN SMALL LETTER I
		"\x82\x8a"		=> "\xef\xbd\x8a",	//FULLWIDTH LATIN SMALL LETTER J
		"\x82\x8b"		=> "\xef\xbd\x8b",	//FULLWIDTH LATIN SMALL LETTER K
		"\x82\x8c"		=> "\xef\xbd\x8c",	//FULLWIDTH LATIN SMALL LETTER L
		"\x82\x8d"		=> "\xef\xbd\x8d",	//FULLWIDTH LATIN SMALL LETTER M
		"\x82\x8e"		=> "\xef\xbd\x8e",	//FULLWIDTH LATIN SMALL LETTER N
		"\x82\x8f"		=> "\xef\xbd\x8f",	//FULLWIDTH LATIN SMALL LETTER O
		"\x82\x90"		=> "\xef\xbd\x90",	//FULLWIDTH LATIN SMALL LETTER P
		"\x82\x91"		=> "\xef\xbd\x91",	//FULLWIDTH LATIN SMALL LETTER Q
		"\x82\x92"		=> "\xef\xbd\x92",	//FULLWIDTH LATIN SMALL LETTER R
		"\x82\x93"		=> "\xef\xbd\x93",	//FULLWIDTH LATIN SMALL LETTER S
		"\x82\x94"		=> "\xef\xbd\x94",	//FULLWIDTH LATIN SMALL LETTER T
		"\x82\x95"		=> "\xef\xbd\x95",	//FULLWIDTH LATIN SMALL LETTER U
		"\x82\x96"		=> "\xef\xbd\x96",	//FULLWIDTH LATIN SMALL LETTER V
		"\x82\x97"		=> "\xef\xbd\x97",	//FULLWIDTH LATIN SMALL LETTER W
		"\x82\x98"		=> "\xef\xbd\x98",	//FULLWIDTH LATIN SMALL LETTER X
		"\x82\x99"		=> "\xef\xbd\x99",	//FULLWIDTH LATIN SMALL LETTER Y
		"\x82\x9a"		=> "\xef\xbd\x9a",	//FULLWIDTH LATIN SMALL LETTER Z
		"\x82\x9f"		=> "\xe3\x81\x81",	//HIRAGANA LETTER SMALL A
		"\x82\xa0"		=> "\xe3\x81\x82",	//HIRAGANA LETTER A
		"\x82\xa1"		=> "\xe3\x81\x83",	//HIRAGANA LETTER SMALL I
		"\x82\xa2"		=> "\xe3\x81\x84",	//HIRAGANA LETTER I
		"\x82\xa3"		=> "\xe3\x81\x85",	//HIRAGANA LETTER SMALL U
		"\x82\xa4"		=> "\xe3\x81\x86",	//HIRAGANA LETTER U
		"\x82\xa5"		=> "\xe3\x81\x87",	//HIRAGANA LETTER SMALL E
		"\x82\xa6"		=> "\xe3\x81\x88",	//HIRAGANA LETTER E
		"\x82\xa7"		=> "\xe3\x81\x89",	//HIRAGANA LETTER SMALL O
		"\x82\xa8"		=> "\xe3\x81\x8a",	//HIRAGANA LETTER O
		"\x82\xa9"		=> "\xe3\x81\x8b",	//HIRAGANA LETTER KA
		"\x82\xaa"		=> "\xe3\x81\x8c",	//HIRAGANA LETTER GA
		"\x82\xab"		=> "\xe3\x81\x8d",	//HIRAGANA LETTER KI
		"\x82\xac"		=> "\xe3\x81\x8e",	//HIRAGANA LETTER GI
		"\x82\xad"		=> "\xe3\x81\x8f",	//HIRAGANA LETTER KU
		"\x82\xae"		=> "\xe3\x81\x90",	//HIRAGANA LETTER GU
		"\x82\xaf"		=> "\xe3\x81\x91",	//HIRAGANA LETTER KE
		"\x82\xb0"		=> "\xe3\x81\x92",	//HIRAGANA LETTER GE
		"\x82\xb1"		=> "\xe3\x81\x93",	//HIRAGANA LETTER KO
		"\x82\xb2"		=> "\xe3\x81\x94",	//HIRAGANA LETTER GO
		"\x82\xb3"		=> "\xe3\x81\x95",	//HIRAGANA LETTER SA
		"\x82\xb4"		=> "\xe3\x81\x96",	//HIRAGANA LETTER ZA
		"\x82\xb5"		=> "\xe3\x81\x97",	//HIRAGANA LETTER SI
		"\x82\xb6"		=> "\xe3\x81\x98",	//HIRAGANA LETTER ZI
		"\x82\xb7"		=> "\xe3\x81\x99",	//HIRAGANA LETTER SU
		"\x82\xb8"		=> "\xe3\x81\x9a",	//HIRAGANA LETTER ZU
		"\x82\xb9"		=> "\xe3\x81\x9b",	//HIRAGANA LETTER SE
		"\x82\xba"		=> "\xe3\x81\x9c",	//HIRAGANA LETTER ZE
		"\x82\xbb"		=> "\xe3\x81\x9d",	//HIRAGANA LETTER SO
		"\x82\xbc"		=> "\xe3\x81\x9e",	//HIRAGANA LETTER ZO
		"\x82\xbd"		=> "\xe3\x81\x9f",	//HIRAGANA LETTER TA
		"\x82\xbe"		=> "\xe3\x81\xa0",	//HIRAGANA LETTER DA
		"\x82\xbf"		=> "\xe3\x81\xa1",	//HIRAGANA LETTER TI
		"\x82\xc0"		=> "\xe3\x81\xa2",	//HIRAGANA LETTER DI
		"\x82\xc1"		=> "\xe3\x81\xa3",	//HIRAGANA LETTER SMALL TU
		"\x82\xc2"		=> "\xe3\x81\xa4",	//HIRAGANA LETTER TU
		"\x82\xc3"		=> "\xe3\x81\xa5",	//HIRAGANA LETTER DU
		"\x82\xc4"		=> "\xe3\x81\xa6",	//HIRAGANA LETTER TE
		"\x82\xc5"		=> "\xe3\x81\xa7",	//HIRAGANA LETTER DE
		"\x82\xc6"		=> "\xe3\x81\xa8",	//HIRAGANA LETTER TO
		"\x82\xc7"		=> "\xe3\x81\xa9",	//HIRAGANA LETTER DO
		"\x82\xc8"		=> "\xe3\x81\xaa",	//HIRAGANA LETTER NA
		"\x82\xc9"		=> "\xe3\x81\xab",	//HIRAGANA LETTER NI
		"\x82\xca"		=> "\xe3\x81\xac",	//HIRAGANA LETTER NU
		"\x82\xcb"		=> "\xe3\x81\xad",	//HIRAGANA LETTER NE
		"\x82\xcc"		=> "\xe3\x81\xae",	//HIRAGANA LETTER NO
		"\x82\xcd"		=> "\xe3\x81\xaf",	//HIRAGANA LETTER HA
		"\x82\xce"		=> "\xe3\x81\xb0",	//HIRAGANA LETTER BA
		"\x82\xcf"		=> "\xe3\x81\xb1",	//HIRAGANA LETTER PA
		"\x82\xd0"		=> "\xe3\x81\xb2",	//HIRAGANA LETTER HI
		"\x82\xd1"		=> "\xe3\x81\xb3",	//HIRAGANA LETTER BI
		"\x82\xd2"		=> "\xe3\x81\xb4",	//HIRAGANA LETTER PI
		"\x82\xd3"		=> "\xe3\x81\xb5",	//HIRAGANA LETTER HU
		"\x82\xd4"		=> "\xe3\x81\xb6",	//HIRAGANA LETTER BU
		"\x82\xd5"		=> "\xe3\x81\xb7",	//HIRAGANA LETTER PU
		"\x82\xd6"		=> "\xe3\x81\xb8",	//HIRAGANA LETTER HE
		"\x82\xd7"		=> "\xe3\x81\xb9",	//HIRAGANA LETTER BE
		"\x82\xd8"		=> "\xe3\x81\xba",	//HIRAGANA LETTER PE
		"\x82\xd9"		=> "\xe3\x81\xbb",	//HIRAGANA LETTER HO
		"\x82\xda"		=> "\xe3\x81\xbc",	//HIRAGANA LETTER BO
		"\x82\xdb"		=> "\xe3\x81\xbd",	//HIRAGANA LETTER PO
		"\x82\xdc"		=> "\xe3\x81\xbe",	//HIRAGANA LETTER MA
		"\x82\xdd"		=> "\xe3\x81\xbf",	//HIRAGANA LETTER MI
		"\x82\xde"		=> "\xe3\x82\x80",	//HIRAGANA LETTER MU
		"\x82\xdf"		=> "\xe3\x82\x81",	//HIRAGANA LETTER ME
		"\x82\xe0"		=> "\xe3\x82\x82",	//HIRAGANA LETTER MO
		"\x82\xe1"		=> "\xe3\x82\x83",	//HIRAGANA LETTER SMALL YA
		"\x82\xe2"		=> "\xe3\x82\x84",	//HIRAGANA LETTER YA
		"\x82\xe3"		=> "\xe3\x82\x85",	//HIRAGANA LETTER SMALL YU
		"\x82\xe4"		=> "\xe3\x82\x86",	//HIRAGANA LETTER YU
		"\x82\xe5"		=> "\xe3\x82\x87",	//HIRAGANA LETTER SMALL YO
		"\x82\xe6"		=> "\xe3\x82\x88",	//HIRAGANA LETTER YO
		"\x82\xe7"		=> "\xe3\x82\x89",	//HIRAGANA LETTER RA
		"\x82\xe8"		=> "\xe3\x82\x8a",	//HIRAGANA LETTER RI
		"\x82\xe9"		=> "\xe3\x82\x8b",	//HIRAGANA LETTER RU
		"\x82\xea"		=> "\xe3\x82\x8c",	//HIRAGANA LETTER RE
		"\x82\xeb"		=> "\xe3\x82\x8d",	//HIRAGANA LETTER RO
		"\x82\xec"		=> "\xe3\x82\x8e",	//HIRAGANA LETTER SMALL WA
		"\x82\xed"		=> "\xe3\x82\x8f",	//HIRAGANA LETTER WA
		"\x82\xee"		=> "\xe3\x82\x90",	//HIRAGANA LETTER WI
		"\x82\xef"		=> "\xe3\x82\x91",	//HIRAGANA LETTER WE
		"\x82\xf0"		=> "\xe3\x82\x92",	//HIRAGANA LETTER WO
		"\x82\xf1"		=> "\xe3\x82\x93",	//HIRAGANA LETTER N
		"\x83\x40"		=> "\xe3\x82\xa1",	//KATAKANA LETTER SMALL A
		"\x83\x41"		=> "\xe3\x82\xa2",	//KATAKANA LETTER A
		"\x83\x42"		=> "\xe3\x82\xa3",	//KATAKANA LETTER SMALL I
		"\x83\x43"		=> "\xe3\x82\xa4",	//KATAKANA LETTER I
		"\x83\x44"		=> "\xe3\x82\xa5",	//KATAKANA LETTER SMALL U
		"\x83\x45"		=> "\xe3\x82\xa6",	//KATAKANA LETTER U
		"\x83\x46"		=> "\xe3\x82\xa7",	//KATAKANA LETTER SMALL E
		"\x83\x47"		=> "\xe3\x82\xa8",	//KATAKANA LETTER E
		"\x83\x48"		=> "\xe3\x82\xa9",	//KATAKANA LETTER SMALL O
		"\x83\x49"		=> "\xe3\x82\xaa",	//KATAKANA LETTER O
		"\x83\x4a"		=> "\xe3\x82\xab",	//KATAKANA LETTER KA
		"\x83\x4b"		=> "\xe3\x82\xac",	//KATAKANA LETTER GA
		"\x83\x4c"		=> "\xe3\x82\xad",	//KATAKANA LETTER KI
		"\x83\x4d"		=> "\xe3\x82\xae",	//KATAKANA LETTER GI
		"\x83\x4e"		=> "\xe3\x82\xaf",	//KATAKANA LETTER KU
		"\x83\x4f"		=> "\xe3\x82\xb0",	//KATAKANA LETTER GU
		"\x83\x50"		=> "\xe3\x82\xb1",	//KATAKANA LETTER KE
		"\x83\x51"		=> "\xe3\x82\xb2",	//KATAKANA LETTER GE
		"\x83\x52"		=> "\xe3\x82\xb3",	//KATAKANA LETTER KO
		"\x83\x53"		=> "\xe3\x82\xb4",	//KATAKANA LETTER GO
		"\x83\x54"		=> "\xe3\x82\xb5",	//KATAKANA LETTER SA
		"\x83\x55"		=> "\xe3\x82\xb6",	//KATAKANA LETTER ZA
		"\x83\x56"		=> "\xe3\x82\xb7",	//KATAKANA LETTER SI
		"\x83\x57"		=> "\xe3\x82\xb8",	//KATAKANA LETTER ZI
		"\x83\x58"		=> "\xe3\x82\xb9",	//KATAKANA LETTER SU
		"\x83\x59"		=> "\xe3\x82\xba",	//KATAKANA LETTER ZU
		"\x83\x5a"		=> "\xe3\x82\xbb",	//KATAKANA LETTER SE
		"\x83\x5b"		=> "\xe3\x82\xbc",	//KATAKANA LETTER ZE
		"\x83\x5c"		=> "\xe3\x82\xbd",	//KATAKANA LETTER SO
		"\x83\x5d"		=> "\xe3\x82\xbe",	//KATAKANA LETTER ZO
		"\x83\x5e"		=> "\xe3\x82\xbf",	//KATAKANA LETTER TA
		"\x83\x5f"		=> "\xe3\x83\x80",	//KATAKANA LETTER DA
		"\x83\x60"		=> "\xe3\x83\x81",	//KATAKANA LETTER TI
		"\x83\x61"		=> "\xe3\x83\x82",	//KATAKANA LETTER DI
		"\x83\x62"		=> "\xe3\x83\x83",	//KATAKANA LETTER SMALL TU
		"\x83\x63"		=> "\xe3\x83\x84",	//KATAKANA LETTER TU
		"\x83\x64"		=> "\xe3\x83\x85",	//KATAKANA LETTER DU
		"\x83\x65"		=> "\xe3\x83\x86",	//KATAKANA LETTER TE
		"\x83\x66"		=> "\xe3\x83\x87",	//KATAKANA LETTER DE
		"\x83\x67"		=> "\xe3\x83\x88",	//KATAKANA LETTER TO
		"\x83\x68"		=> "\xe3\x83\x89",	//KATAKANA LETTER DO
		"\x83\x69"		=> "\xe3\x83\x8a",	//KATAKANA LETTER NA
		"\x83\x6a"		=> "\xe3\x83\x8b",	//KATAKANA LETTER NI
		"\x83\x6b"		=> "\xe3\x83\x8c",	//KATAKANA LETTER NU
		"\x83\x6c"		=> "\xe3\x83\x8d",	//KATAKANA LETTER NE
		"\x83\x6d"		=> "\xe3\x83\x8e",	//KATAKANA LETTER NO
		"\x83\x6e"		=> "\xe3\x83\x8f",	//KATAKANA LETTER HA
		"\x83\x6f"		=> "\xe3\x83\x90",	//KATAKANA LETTER BA
		"\x83\x70"		=> "\xe3\x83\x91",	//KATAKANA LETTER PA
		"\x83\x71"		=> "\xe3\x83\x92",	//KATAKANA LETTER HI
		"\x83\x72"		=> "\xe3\x83\x93",	//KATAKANA LETTER BI
		"\x83\x73"		=> "\xe3\x83\x94",	//KATAKANA LETTER PI
		"\x83\x74"		=> "\xe3\x83\x95",	//KATAKANA LETTER HU
		"\x83\x75"		=> "\xe3\x83\x96",	//KATAKANA LETTER BU
		"\x83\x76"		=> "\xe3\x83\x97",	//KATAKANA LETTER PU
		"\x83\x77"		=> "\xe3\x83\x98",	//KATAKANA LETTER HE
		"\x83\x78"		=> "\xe3\x83\x99",	//KATAKANA LETTER BE
		"\x83\x79"		=> "\xe3\x83\x9a",	//KATAKANA LETTER PE
		"\x83\x7a"		=> "\xe3\x83\x9b",	//KATAKANA LETTER HO
		"\x83\x7b"		=> "\xe3\x83\x9c",	//KATAKANA LETTER BO
		"\x83\x7c"		=> "\xe3\x83\x9d",	//KATAKANA LETTER PO
		"\x83\x7d"		=> "\xe3\x83\x9e",	//KATAKANA LETTER MA
		"\x83\x7e"		=> "\xe3\x83\x9f",	//KATAKANA LETTER MI
		"\x83\x80"		=> "\xe3\x83\xa0",	//KATAKANA LETTER MU
		"\x83\x81"		=> "\xe3\x83\xa1",	//KATAKANA LETTER ME
		"\x83\x82"		=> "\xe3\x83\xa2",	//KATAKANA LETTER MO
		"\x83\x83"		=> "\xe3\x83\xa3",	//KATAKANA LETTER SMALL YA
		"\x83\x84"		=> "\xe3\x83\xa4",	//KATAKANA LETTER YA
		"\x83\x85"		=> "\xe3\x83\xa5",	//KATAKANA LETTER SMALL YU
		"\x83\x86"		=> "\xe3\x83\xa6",	//KATAKANA LETTER YU
		"\x83\x87"		=> "\xe3\x83\xa7",	//KATAKANA LETTER SMALL YO
		"\x83\x88"		=> "\xe3\x83\xa8",	//KATAKANA LETTER YO
		"\x83\x89"		=> "\xe3\x83\xa9",	//KATAKANA LETTER RA
		"\x83\x8a"		=> "\xe3\x83\xaa",	//KATAKANA LETTER RI
		"\x83\x8b"		=> "\xe3\x83\xab",	//KATAKANA LETTER RU
		"\x83\x8c"		=> "\xe3\x83\xac",	//KATAKANA LETTER RE
		"\x83\x8d"		=> "\xe3\x83\xad",	//KATAKANA LETTER RO
		"\x83\x8e"		=> "\xe3\x83\xae",	//KATAKANA LETTER SMALL WA
		"\x83\x8f"		=> "\xe3\x83\xaf",	//KATAKANA LETTER WA
		"\x83\x90"		=> "\xe3\x83\xb0",	//KATAKANA LETTER WI
		"\x83\x91"		=> "\xe3\x83\xb1",	//KATAKANA LETTER WE
		"\x83\x92"		=> "\xe3\x83\xb2",	//KATAKANA LETTER WO
		"\x83\x93"		=> "\xe3\x83\xb3",	//KATAKANA LETTER N
		"\x83\x94"		=> "\xe3\x83\xb4",	//KATAKANA LETTER VU
		"\x83\x95"		=> "\xe3\x83\xb5",	//KATAKANA LETTER SMALL KA
		"\x83\x96"		=> "\xe3\x83\xb6",	//KATAKANA LETTER SMALL KE
		"\x83\x9f"		=> "\xce\x91",	//GREEK CAPITAL LETTER ALPHA
		"\x83\xa0"		=> "\xce\x92",	//GREEK CAPITAL LETTER BETA
		"\x83\xa1"		=> "\xce\x93",	//GREEK CAPITAL LETTER GAMMA
		"\x83\xa2"		=> "\xce\x94",	//GREEK CAPITAL LETTER DELTA
		"\x83\xa3"		=> "\xce\x95",	//GREEK CAPITAL LETTER EPSILON
		"\x83\xa4"		=> "\xce\x96",	//GREEK CAPITAL LETTER ZETA
		"\x83\xa5"		=> "\xce\x97",	//GREEK CAPITAL LETTER ETA
		"\x83\xa6"		=> "\xce\x98",	//GREEK CAPITAL LETTER THETA
		"\x83\xa7"		=> "\xce\x99",	//GREEK CAPITAL LETTER IOTA
		"\x83\xa8"		=> "\xce\x9a",	//GREEK CAPITAL LETTER KAPPA
		"\x83\xa9"		=> "\xce\x9b",	//GREEK CAPITAL LETTER LAMDA
		"\x83\xaa"		=> "\xce\x9c",	//GREEK CAPITAL LETTER MU
		"\x83\xab"		=> "\xce\x9d",	//GREEK CAPITAL LETTER NU
		"\x83\xac"		=> "\xce\x9e",	//GREEK CAPITAL LETTER XI
		"\x83\xad"		=> "\xce\x9f",	//GREEK CAPITAL LETTER OMICRON
		"\x83\xae"		=> "\xce\xa0",	//GREEK CAPITAL LETTER PI
		"\x83\xaf"		=> "\xce\xa1",	//GREEK CAPITAL LETTER RHO
		"\x83\xb0"		=> "\xce\xa3",	//GREEK CAPITAL LETTER SIGMA
		"\x83\xb1"		=> "\xce\xa4",	//GREEK CAPITAL LETTER TAU
		"\x83\xb2"		=> "\xce\xa5",	//GREEK CAPITAL LETTER UPSILON
		"\x83\xb3"		=> "\xce\xa6",	//GREEK CAPITAL LETTER PHI
		"\x83\xb4"		=> "\xce\xa7",	//GREEK CAPITAL LETTER CHI
		"\x83\xb5"		=> "\xce\xa8",	//GREEK CAPITAL LETTER PSI
		"\x83\xb6"		=> "\xce\xa9",	//GREEK CAPITAL LETTER OMEGA
		"\x83\xbf"		=> "\xce\xb1",	//GREEK SMALL LETTER ALPHA
		"\x83\xc0"		=> "\xce\xb2",	//GREEK SMALL LETTER BETA
		"\x83\xc1"		=> "\xce\xb3",	//GREEK SMALL LETTER GAMMA
		"\x83\xc2"		=> "\xce\xb4",	//GREEK SMALL LETTER DELTA
		"\x83\xc3"		=> "\xce\xb5",	//GREEK SMALL LETTER EPSILON
		"\x83\xc4"		=> "\xce\xb6",	//GREEK SMALL LETTER ZETA
		"\x83\xc5"		=> "\xce\xb7",	//GREEK SMALL LETTER ETA
		"\x83\xc6"		=> "\xce\xb8",	//GREEK SMALL LETTER THETA
		"\x83\xc7"		=> "\xce\xb9",	//GREEK SMALL LETTER IOTA
		"\x83\xc8"		=> "\xce\xba",	//GREEK SMALL LETTER KAPPA
		"\x83\xc9"		=> "\xce\xbb",	//GREEK SMALL LETTER LAMDA
		"\x83\xca"		=> "\xce\xbc",	//GREEK SMALL LETTER MU
		"\x83\xcb"		=> "\xce\xbd",	//GREEK SMALL LETTER NU
		"\x83\xcc"		=> "\xce\xbe",	//GREEK SMALL LETTER XI
		"\x83\xcd"		=> "\xce\xbf",	//GREEK SMALL LETTER OMICRON
		"\x83\xce"		=> "\xcf\x80",	//GREEK SMALL LETTER PI
		"\x83\xcf"		=> "\xcf\x81",	//GREEK SMALL LETTER RHO
		"\x83\xd0"		=> "\xcf\x83",	//GREEK SMALL LETTER SIGMA
		"\x83\xd1"		=> "\xcf\x84",	//GREEK SMALL LETTER TAU
		"\x83\xd2"		=> "\xcf\x85",	//GREEK SMALL LETTER UPSILON
		"\x83\xd3"		=> "\xcf\x86",	//GREEK SMALL LETTER PHI
		"\x83\xd4"		=> "\xcf\x87",	//GREEK SMALL LETTER CHI
		"\x83\xd5"		=> "\xcf\x88",	//GREEK SMALL LETTER PSI
		"\x83\xd6"		=> "\xcf\x89",	//GREEK SMALL LETTER OMEGA
		"\x84\x40"		=> "\xd0\x90",	//CYRILLIC CAPITAL LETTER A
		"\x84\x41"		=> "\xd0\x91",	//CYRILLIC CAPITAL LETTER BE
		"\x84\x42"		=> "\xd0\x92",	//CYRILLIC CAPITAL LETTER VE
		"\x84\x43"		=> "\xd0\x93",	//CYRILLIC CAPITAL LETTER GHE
		"\x84\x44"		=> "\xd0\x94",	//CYRILLIC CAPITAL LETTER DE
		"\x84\x45"		=> "\xd0\x95",	//CYRILLIC CAPITAL LETTER IE
		"\x84\x46"		=> "\xd0\x81",	//CYRILLIC CAPITAL LETTER IO
		"\x84\x47"		=> "\xd0\x96",	//CYRILLIC CAPITAL LETTER ZHE
		"\x84\x48"		=> "\xd0\x97",	//CYRILLIC CAPITAL LETTER ZE
		"\x84\x49"		=> "\xd0\x98",	//CYRILLIC CAPITAL LETTER I
		"\x84\x4a"		=> "\xd0\x99",	//CYRILLIC CAPITAL LETTER SHORT I
		"\x84\x4b"		=> "\xd0\x9a",	//CYRILLIC CAPITAL LETTER KA
		"\x84\x4c"		=> "\xd0\x9b",	//CYRILLIC CAPITAL LETTER EL
		"\x84\x4d"		=> "\xd0\x9c",	//CYRILLIC CAPITAL LETTER EM
		"\x84\x4e"		=> "\xd0\x9d",	//CYRILLIC CAPITAL LETTER EN
		"\x84\x4f"		=> "\xd0\x9e",	//CYRILLIC CAPITAL LETTER O
		"\x84\x50"		=> "\xd0\x9f",	//CYRILLIC CAPITAL LETTER PE
		"\x84\x51"		=> "\xd0\xa0",	//CYRILLIC CAPITAL LETTER ER
		"\x84\x52"		=> "\xd0\xa1",	//CYRILLIC CAPITAL LETTER ES
		"\x84\x53"		=> "\xd0\xa2",	//CYRILLIC CAPITAL LETTER TE
		"\x84\x54"		=> "\xd0\xa3",	//CYRILLIC CAPITAL LETTER U
		"\x84\x55"		=> "\xd0\xa4",	//CYRILLIC CAPITAL LETTER EF
		"\x84\x56"		=> "\xd0\xa5",	//CYRILLIC CAPITAL LETTER HA
		"\x84\x57"		=> "\xd0\xa6",	//CYRILLIC CAPITAL LETTER TSE
		"\x84\x58"		=> "\xd0\xa7",	//CYRILLIC CAPITAL LETTER CHE
		"\x84\x59"		=> "\xd0\xa8",	//CYRILLIC CAPITAL LETTER SHA
		"\x84\x5a"		=> "\xd0\xa9",	//CYRILLIC CAPITAL LETTER SHCHA
		"\x84\x5b"		=> "\xd0\xaa",	//CYRILLIC CAPITAL LETTER HARD SIGN
		"\x84\x5c"		=> "\xd0\xab",	//CYRILLIC CAPITAL LETTER YERU
		"\x84\x5d"		=> "\xd0\xac",	//CYRILLIC CAPITAL LETTER SOFT SIGN
		"\x84\x5e"		=> "\xd0\xad",	//CYRILLIC CAPITAL LETTER E
		"\x84\x5f"		=> "\xd0\xae",	//CYRILLIC CAPITAL LETTER YU
		"\x84\x60"		=> "\xd0\xaf",	//CYRILLIC CAPITAL LETTER YA
		"\x84\x70"		=> "\xd0\xb0",	//CYRILLIC SMALL LETTER A
		"\x84\x71"		=> "\xd0\xb1",	//CYRILLIC SMALL LETTER BE
		"\x84\x72"		=> "\xd0\xb2",	//CYRILLIC SMALL LETTER VE
		"\x84\x73"		=> "\xd0\xb3",	//CYRILLIC SMALL LETTER GHE
		"\x84\x74"		=> "\xd0\xb4",	//CYRILLIC SMALL LETTER DE
		"\x84\x75"		=> "\xd0\xb5",	//CYRILLIC SMALL LETTER IE
		"\x84\x76"		=> "\xd1\x91",	//CYRILLIC SMALL LETTER IO
		"\x84\x77"		=> "\xd0\xb6",	//CYRILLIC SMALL LETTER ZHE
		"\x84\x78"		=> "\xd0\xb7",	//CYRILLIC SMALL LETTER ZE
		"\x84\x79"		=> "\xd0\xb8",	//CYRILLIC SMALL LETTER I
		"\x84\x7a"		=> "\xd0\xb9",	//CYRILLIC SMALL LETTER SHORT I
		"\x84\x7b"		=> "\xd0\xba",	//CYRILLIC SMALL LETTER KA
		"\x84\x7c"		=> "\xd0\xbb",	//CYRILLIC SMALL LETTER EL
		"\x84\x7d"		=> "\xd0\xbc",	//CYRILLIC SMALL LETTER EM
		"\x84\x7e"		=> "\xd0\xbd",	//CYRILLIC SMALL LETTER EN
		"\x84\x80"		=> "\xd0\xbe",	//CYRILLIC SMALL LETTER O
		"\x84\x81"		=> "\xd0\xbf",	//CYRILLIC SMALL LETTER PE
		"\x84\x82"		=> "\xd1\x80",	//CYRILLIC SMALL LETTER ER
		"\x84\x83"		=> "\xd1\x81",	//CYRILLIC SMALL LETTER ES
		"\x84\x84"		=> "\xd1\x82",	//CYRILLIC SMALL LETTER TE
		"\x84\x85"		=> "\xd1\x83",	//CYRILLIC SMALL LETTER U
		"\x84\x86"		=> "\xd1\x84",	//CYRILLIC SMALL LETTER EF
		"\x84\x87"		=> "\xd1\x85",	//CYRILLIC SMALL LETTER HA
		"\x84\x88"		=> "\xd1\x86",	//CYRILLIC SMALL LETTER TSE
		"\x84\x89"		=> "\xd1\x87",	//CYRILLIC SMALL LETTER CHE
		"\x84\x8a"		=> "\xd1\x88",	//CYRILLIC SMALL LETTER SHA
		"\x84\x8b"		=> "\xd1\x89",	//CYRILLIC SMALL LETTER SHCHA
		"\x84\x8c"		=> "\xd1\x8a",	//CYRILLIC SMALL LETTER HARD SIGN
		"\x84\x8d"		=> "\xd1\x8b",	//CYRILLIC SMALL LETTER YERU
		"\x84\x8e"		=> "\xd1\x8c",	//CYRILLIC SMALL LETTER SOFT SIGN
		"\x84\x8f"		=> "\xd1\x8d",	//CYRILLIC SMALL LETTER E
		"\x84\x90"		=> "\xd1\x8e",	//CYRILLIC SMALL LETTER YU
		"\x84\x91"		=> "\xd1\x8f",	//CYRILLIC SMALL LETTER YA
		"\x84\x9f"		=> "\xe2\x94\x80",	//BOX DRAWINGS LIGHT HORIZONTAL
		"\x84\xa0"		=> "\xe2\x94\x82",	//BOX DRAWINGS LIGHT VERTICAL
		"\x84\xa1"		=> "\xe2\x94\x8c",	//BOX DRAWINGS LIGHT DOWN AND RIGHT
		"\x84\xa2"		=> "\xe2\x94\x90",	//BOX DRAWINGS LIGHT DOWN AND LEFT
		"\x84\xa3"		=> "\xe2\x94\x98",	//BOX DRAWINGS LIGHT UP AND LEFT
		"\x84\xa4"		=> "\xe2\x94\x94",	//BOX DRAWINGS LIGHT UP AND RIGHT
		"\x84\xa5"		=> "\xe2\x94\x9c",	//BOX DRAWINGS LIGHT VERTICAL AND RIGHT
		"\x84\xa6"		=> "\xe2\x94\xac",	//BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
		"\x84\xa7"		=> "\xe2\x94\xa4",	//BOX DRAWINGS LIGHT VERTICAL AND LEFT
		"\x84\xa8"		=> "\xe2\x94\xb4",	//BOX DRAWINGS LIGHT UP AND HORIZONTAL
		"\x84\xa9"		=> "\xe2\x94\xbc",	//BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
		"\x84\xaa"		=> "\xe2\x94\x81",	//BOX DRAWINGS HEAVY HORIZONTAL
		"\x84\xab"		=> "\xe2\x94\x83",	//BOX DRAWINGS HEAVY VERTICAL
		"\x84\xac"		=> "\xe2\x94\x8f",	//BOX DRAWINGS HEAVY DOWN AND RIGHT
		"\x84\xad"		=> "\xe2\x94\x93",	//BOX DRAWINGS HEAVY DOWN AND LEFT
		"\x84\xae"		=> "\xe2\x94\x9b",	//BOX DRAWINGS HEAVY UP AND LEFT
		"\x84\xaf"		=> "\xe2\x94\x97",	//BOX DRAWINGS HEAVY UP AND RIGHT
		"\x84\xb0"		=> "\xe2\x94\xa3",	//BOX DRAWINGS HEAVY VERTICAL AND RIGHT
		"\x84\xb1"		=> "\xe2\x94\xb3",	//BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
		"\x84\xb2"		=> "\xe2\x94\xab",	//BOX DRAWINGS HEAVY VERTICAL AND LEFT
		"\x84\xb3"		=> "\xe2\x94\xbb",	//BOX DRAWINGS HEAVY UP AND HORIZONTAL
		"\x84\xb4"		=> "\xe2\x95\x8b",	//BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
		"\x84\xb5"		=> "\xe2\x94\xa0",	//BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
		"\x84\xb6"		=> "\xe2\x94\xaf",	//BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
		"\x84\xb7"		=> "\xe2\x94\xa8",	//BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
		"\x84\xb8"		=> "\xe2\x94\xb7",	//BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
		"\x84\xb9"		=> "\xe2\x94\xbf",	//BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
		"\x84\xba"		=> "\xe2\x94\x9d",	//BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
		"\x84\xbb"		=> "\xe2\x94\xb0",	//BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
		"\x84\xbc"		=> "\xe2\x94\xa5",	//BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
		"\x84\xbd"		=> "\xe2\x94\xb8",	//BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
		"\x84\xbe"		=> "\xe2\x95\x82",	//BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
		"\x87\x40"		=> "\xe2\x91\xa0",	//CIRCLED DIGIT ONE
		"\x87\x41"		=> "\xe2\x91\xa1",	//CIRCLED DIGIT TWO
		"\x87\x42"		=> "\xe2\x91\xa2",	//CIRCLED DIGIT THREE
		"\x87\x43"		=> "\xe2\x91\xa3",	//CIRCLED DIGIT FOUR
		"\x87\x44"		=> "\xe2\x91\xa4",	//CIRCLED DIGIT FIVE
		"\x87\x45"		=> "\xe2\x91\xa5",	//CIRCLED DIGIT SIX
		"\x87\x46"		=> "\xe2\x91\xa6",	//CIRCLED DIGIT SEVEN
		"\x87\x47"		=> "\xe2\x91\xa7",	//CIRCLED DIGIT EIGHT
		"\x87\x48"		=> "\xe2\x91\xa8",	//CIRCLED DIGIT NINE
		"\x87\x49"		=> "\xe2\x91\xa9",	//CIRCLED NUMBER TEN
		"\x87\x4a"		=> "\xe2\x91\xaa",	//CIRCLED NUMBER ELEVEN
		"\x87\x4b"		=> "\xe2\x91\xab",	//CIRCLED NUMBER TWELVE
		"\x87\x4c"		=> "\xe2\x91\xac",	//CIRCLED NUMBER THIRTEEN
		"\x87\x4d"		=> "\xe2\x91\xad",	//CIRCLED NUMBER FOURTEEN
		"\x87\x4e"		=> "\xe2\x91\xae",	//CIRCLED NUMBER FIFTEEN
		"\x87\x4f"		=> "\xe2\x91\xaf",	//CIRCLED NUMBER SIXTEEN
		"\x87\x50"		=> "\xe2\x91\xb0",	//CIRCLED NUMBER SEVENTEEN
		"\x87\x51"		=> "\xe2\x91\xb1",	//CIRCLED NUMBER EIGHTEEN
		"\x87\x52"		=> "\xe2\x91\xb2",	//CIRCLED NUMBER NINETEEN
		"\x87\x53"		=> "\xe2\x91\xb3",	//CIRCLED NUMBER TWENTY
		"\x87\x54"		=> "\xe2\x85\xa0",	//ROMAN NUMERAL ONE
		"\x87\x55"		=> "\xe2\x85\xa1",	//ROMAN NUMERAL TWO
		"\x87\x56"		=> "\xe2\x85\xa2",	//ROMAN NUMERAL THREE
		"\x87\x57"		=> "\xe2\x85\xa3",	//ROMAN NUMERAL FOUR
		"\x87\x58"		=> "\xe2\x85\xa4",	//ROMAN NUMERAL FIVE
		"\x87\x59"		=> "\xe2\x85\xa5",	//ROMAN NUMERAL SIX
		"\x87\x5a"		=> "\xe2\x85\xa6",	//ROMAN NUMERAL SEVEN
		"\x87\x5b"		=> "\xe2\x85\xa7",	//ROMAN NUMERAL EIGHT
		"\x87\x5c"		=> "\xe2\x85\xa8",	//ROMAN NUMERAL NINE
		"\x87\x5d"		=> "\xe2\x85\xa9",	//ROMAN NUMERAL TEN
		"\x87\x5f"		=> "\xe3\x8d\x89",	//SQUARE MIRI
		"\x87\x60"		=> "\xe3\x8c\x94",	//SQUARE KIRO
		"\x87\x61"		=> "\xe3\x8c\xa2",	//SQUARE SENTI
		"\x87\x62"		=> "\xe3\x8d\x8d",	//SQUARE MEETORU
		"\x87\x63"		=> "\xe3\x8c\x98",	//SQUARE GURAMU
		"\x87\x64"		=> "\xe3\x8c\xa7",	//SQUARE TON
		"\x87\x65"		=> "\xe3\x8c\x83",	//SQUARE AARU
		"\x87\x66"		=> "\xe3\x8c\xb6",	//SQUARE HEKUTAARU
		"\x87\x67"		=> "\xe3\x8d\x91",	//SQUARE RITTORU
		"\x87\x68"		=> "\xe3\x8d\x97",	//SQUARE WATTO
		"\x87\x69"		=> "\xe3\x8c\x8d",	//SQUARE KARORII
		"\x87\x6a"		=> "\xe3\x8c\xa6",	//SQUARE DORU
		"\x87\x6b"		=> "\xe3\x8c\xa3",	//SQUARE SENTO
		"\x87\x6c"		=> "\xe3\x8c\xab",	//SQUARE PAASENTO
		"\x87\x6d"		=> "\xe3\x8d\x8a",	//SQUARE MIRIBAARU
		"\x87\x6e"		=> "\xe3\x8c\xbb",	//SQUARE PEEZI
		"\x87\x6f"		=> "\xe3\x8e\x9c",	//SQUARE MM
		"\x87\x70"		=> "\xe3\x8e\x9d",	//SQUARE CM
		"\x87\x71"		=> "\xe3\x8e\x9e",	//SQUARE KM
		"\x87\x72"		=> "\xe3\x8e\x8e",	//SQUARE MG
		"\x87\x73"		=> "\xe3\x8e\x8f",	//SQUARE KG
		"\x87\x74"		=> "\xe3\x8f\x84",	//SQUARE CC
		"\x87\x75"		=> "\xe3\x8e\xa1",	//SQUARE M SQUARED
		"\x87\x7e"		=> "\xe3\x8d\xbb",	//SQUARE ERA NAME HEISEI
		"\x87\x80"		=> "\xe3\x80\x9d",	//REVERSED DOUBLE PRIME QUOTATION MARK
		"\x87\x81"		=> "\xe3\x80\x9f",	//LOW DOUBLE PRIME QUOTATION MARK
		"\x87\x82"		=> "\xe2\x84\x96",	//NUMERO SIGN
		"\x87\x83"		=> "\xe3\x8f\x8d",	//SQUARE KK
		"\x87\x84"		=> "\xe2\x84\xa1",	//TELEPHONE SIGN
		"\x87\x85"		=> "\xe3\x8a\xa4",	//CIRCLED IDEOGRAPH HIGH
		"\x87\x86"		=> "\xe3\x8a\xa5",	//CIRCLED IDEOGRAPH CENTRE
		"\x87\x87"		=> "\xe3\x8a\xa6",	//CIRCLED IDEOGRAPH LOW
		"\x87\x88"		=> "\xe3\x8a\xa7",	//CIRCLED IDEOGRAPH LEFT
		"\x87\x89"		=> "\xe3\x8a\xa8",	//CIRCLED IDEOGRAPH RIGHT
		"\x87\x8a"		=> "\xe3\x88\xb1",	//PARENTHESIZED IDEOGRAPH STOCK
		"\x87\x8b"		=> "\xe3\x88\xb2",	//PARENTHESIZED IDEOGRAPH HAVE
		"\x87\x8c"		=> "\xe3\x88\xb9",	//PARENTHESIZED IDEOGRAPH REPRESENT
		"\x87\x8d"		=> "\xe3\x8d\xbe",	//SQUARE ERA NAME MEIZI
		"\x87\x8e"		=> "\xe3\x8d\xbd",	//SQUARE ERA NAME TAISYOU
		"\x87\x8f"		=> "\xe3\x8d\xbc",	//SQUARE ERA NAME SYOUWA
		"\x87\x90"		=> "\xe2\x89\x92",	//APPROXIMATELY EQUAL TO OR THE IMAGE OF
		"\x87\x91"		=> "\xe2\x89\xa1",	//IDENTICAL TO
		"\x87\x92"		=> "\xe2\x88\xab",	//INTEGRAL
		"\x87\x93"		=> "\xe2\x88\xae",	//CONTOUR INTEGRAL
		"\x87\x94"		=> "\xe2\x88\x91",	//N-ARY SUMMATION
		"\x87\x95"		=> "\xe2\x88\x9a",	//SQUARE ROOT
		"\x87\x96"		=> "\xe2\x8a\xa5",	//UP TACK
		"\x87\x97"		=> "\xe2\x88\xa0",	//ANGLE
		"\x87\x98"		=> "\xe2\x88\x9f",	//RIGHT ANGLE
		"\x87\x99"		=> "\xe2\x8a\xbf",	//RIGHT TRIANGLE
		"\x87\x9a"		=> "\xe2\x88\xb5",	//BECAUSE
		"\x87\x9b"		=> "\xe2\x88\xa9",	//INTERSECTION
		"\x87\x9c"		=> "\xe2\x88\xaa",	//UNION
		"\x88\x9f"		=> "\xe4\xba\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa0"		=> "\xe5\x94\x96",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa1"		=> "\xe5\xa8\x83",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa2"		=> "\xe9\x98\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa3"		=> "\xe5\x93\x80",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa4"		=> "\xe6\x84\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa5"		=> "\xe6\x8c\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa6"		=> "\xe5\xa7\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa7"		=> "\xe9\x80\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa8"		=> "\xe8\x91\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x88\xa9"		=> "\xe8\x8c\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x88\xaa"		=> "\xe7\xa9\x90",	//CJK UNIFIED IDEOGRAPH
		"\x88\xab"		=> "\xe6\x82\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x88\xac"		=> "\xe6\x8f\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x88\xad"		=> "\xe6\xb8\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x88\xae"		=> "\xe6\x97\xad",	//CJK UNIFIED IDEOGRAPH
		"\x88\xaf"		=> "\xe8\x91\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb0"		=> "\xe8\x8a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb1"		=> "\xe9\xaf\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb2"		=> "\xe6\xa2\x93",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb3"		=> "\xe5\x9c\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb4"		=> "\xe6\x96\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb5"		=> "\xe6\x89\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb6"		=> "\xe5\xae\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb7"		=> "\xe5\xa7\x90",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb8"		=> "\xe8\x99\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x88\xb9"		=> "\xe9\xa3\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x88\xba"		=> "\xe7\xb5\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x88\xbb"		=> "\xe7\xb6\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x88\xbc"		=> "\xe9\xae\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x88\xbd"		=> "\xe6\x88\x96",	//CJK UNIFIED IDEOGRAPH
		"\x88\xbe"		=> "\xe7\xb2\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x88\xbf"		=> "\xe8\xa2\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc0"		=> "\xe5\xae\x89",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc1"		=> "\xe5\xba\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc2"		=> "\xe6\x8c\x89",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc3"		=> "\xe6\x9a\x97",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc4"		=> "\xe6\xa1\x88",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc5"		=> "\xe9\x97\x87",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc6"		=> "\xe9\x9e\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc7"		=> "\xe6\x9d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc8"		=> "\xe4\xbb\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x88\xc9"		=> "\xe4\xbc\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x88\xca"		=> "\xe4\xbd\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x88\xcb"		=> "\xe4\xbe\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x88\xcc"		=> "\xe5\x81\x89",	//CJK UNIFIED IDEOGRAPH
		"\x88\xcd"		=> "\xe5\x9b\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x88\xce"		=> "\xe5\xa4\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x88\xcf"		=> "\xe5\xa7\x94",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd0"		=> "\xe5\xa8\x81",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd1"		=> "\xe5\xb0\x89",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd2"		=> "\xe6\x83\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd3"		=> "\xe6\x84\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd4"		=> "\xe6\x85\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd5"		=> "\xe6\x98\x93",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd6"		=> "\xe6\xa4\x85",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd7"		=> "\xe7\x82\xba",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd8"		=> "\xe7\x95\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x88\xd9"		=> "\xe7\x95\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x88\xda"		=> "\xe7\xa7\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x88\xdb"		=> "\xe7\xb6\xad",	//CJK UNIFIED IDEOGRAPH
		"\x88\xdc"		=> "\xe7\xb7\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x88\xdd"		=> "\xe8\x83\x83",	//CJK UNIFIED IDEOGRAPH
		"\x88\xde"		=> "\xe8\x90\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x88\xdf"		=> "\xe8\xa1\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe0"		=> "\xe8\xac\x82",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe1"		=> "\xe9\x81\x95",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe2"		=> "\xe9\x81\xba",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe3"		=> "\xe5\x8c\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe4"		=> "\xe4\xba\x95",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe5"		=> "\xe4\xba\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe6"		=> "\xe5\x9f\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe7"		=> "\xe8\x82\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe8"		=> "\xe9\x83\x81",	//CJK UNIFIED IDEOGRAPH
		"\x88\xe9"		=> "\xe7\xa3\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x88\xea"		=> "\xe4\xb8\x80",	//CJK UNIFIED IDEOGRAPH
		"\x88\xeb"		=> "\xe5\xa3\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x88\xec"		=> "\xe6\xba\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x88\xed"		=> "\xe9\x80\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x88\xee"		=> "\xe7\xa8\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x88\xef"		=> "\xe8\x8c\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf0"		=> "\xe8\x8a\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf1"		=> "\xe9\xb0\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf2"		=> "\xe5\x85\x81",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf3"		=> "\xe5\x8d\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf4"		=> "\xe5\x92\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf5"		=> "\xe5\x93\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf6"		=> "\xe5\x9b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf7"		=> "\xe5\xa7\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf8"		=> "\xe5\xbc\x95",	//CJK UNIFIED IDEOGRAPH
		"\x88\xf9"		=> "\xe9\xa3\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x88\xfa"		=> "\xe6\xb7\xab",	//CJK UNIFIED IDEOGRAPH
		"\x88\xfb"		=> "\xe8\x83\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x88\xfc"		=> "\xe8\x94\xad",	//CJK UNIFIED IDEOGRAPH
		"\x89\x40"		=> "\xe9\x99\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x89\x41"		=> "\xe9\x99\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x89\x42"		=> "\xe9\x9a\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x89\x43"		=> "\xe9\x9f\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x89\x44"		=> "\xe5\x90\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x89\x45"		=> "\xe5\x8f\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x89\x46"		=> "\xe5\xae\x87",	//CJK UNIFIED IDEOGRAPH
		"\x89\x47"		=> "\xe7\x83\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x89\x48"		=> "\xe7\xbe\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x89\x49"		=> "\xe8\xbf\x82",	//CJK UNIFIED IDEOGRAPH
		"\x89\x4a"		=> "\xe9\x9b\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x89\x4b"		=> "\xe5\x8d\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x89\x4c"		=> "\xe9\xb5\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x89\x4d"		=> "\xe7\xaa\xba",	//CJK UNIFIED IDEOGRAPH
		"\x89\x4e"		=> "\xe4\xb8\x91",	//CJK UNIFIED IDEOGRAPH
		"\x89\x4f"		=> "\xe7\xa2\x93",	//CJK UNIFIED IDEOGRAPH
		"\x89\x50"		=> "\xe8\x87\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x89\x51"		=> "\xe6\xb8\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x89\x52"		=> "\xe5\x98\x98",	//CJK UNIFIED IDEOGRAPH
		"\x89\x53"		=> "\xe5\x94\x84",	//CJK UNIFIED IDEOGRAPH
		"\x89\x54"		=> "\xe6\xac\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x89\x55"		=> "\xe8\x94\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x89\x56"		=> "\xe9\xb0\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x89\x57"		=> "\xe5\xa7\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x89\x58"		=> "\xe5\x8e\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x89\x59"		=> "\xe6\xb5\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x89\x5a"		=> "\xe7\x93\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x89\x5b"		=> "\xe9\x96\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x89\x5c"		=> "\xe5\x99\x82",	//CJK UNIFIED IDEOGRAPH
		"\x89\x5d"		=> "\xe4\xba\x91",	//CJK UNIFIED IDEOGRAPH
		"\x89\x5e"		=> "\xe9\x81\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x89\x5f"		=> "\xe9\x9b\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x89\x60"		=> "\xe8\x8d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x89\x61"		=> "\xe9\xa4\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x89\x62"		=> "\xe5\x8f\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x89\x63"		=> "\xe5\x96\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x89\x64"		=> "\xe5\xac\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x89\x65"		=> "\xe5\xbd\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x89\x66"		=> "\xe6\x98\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x89\x67"		=> "\xe6\x9b\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x89\x68"		=> "\xe6\xa0\x84",	//CJK UNIFIED IDEOGRAPH
		"\x89\x69"		=> "\xe6\xb0\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x89\x6a"		=> "\xe6\xb3\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x89\x6b"		=> "\xe6\xb4\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x89\x6c"		=> "\xe7\x91\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x89\x6d"		=> "\xe7\x9b\x88",	//CJK UNIFIED IDEOGRAPH
		"\x89\x6e"		=> "\xe7\xa9\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x89\x6f"		=> "\xe9\xa0\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x89\x70"		=> "\xe8\x8b\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x89\x71"		=> "\xe8\xa1\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x89\x72"		=> "\xe8\xa9\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x89\x73"		=> "\xe9\x8b\xad",	//CJK UNIFIED IDEOGRAPH
		"\x89\x74"		=> "\xe6\xb6\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x89\x75"		=> "\xe7\x96\xab",	//CJK UNIFIED IDEOGRAPH
		"\x89\x76"		=> "\xe7\x9b\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x89\x77"		=> "\xe9\xa7\x85",	//CJK UNIFIED IDEOGRAPH
		"\x89\x78"		=> "\xe6\x82\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x89\x79"		=> "\xe8\xac\x81",	//CJK UNIFIED IDEOGRAPH
		"\x89\x7a"		=> "\xe8\xb6\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x89\x7b"		=> "\xe9\x96\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x89\x7c"		=> "\xe6\xa6\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x89\x7d"		=> "\xe5\x8e\xad",	//CJK UNIFIED IDEOGRAPH
		"\x89\x7e"		=> "\xe5\x86\x86",	//CJK UNIFIED IDEOGRAPH
		"\x89\x80"		=> "\xe5\x9c\x92",	//CJK UNIFIED IDEOGRAPH
		"\x89\x81"		=> "\xe5\xa0\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x89\x82"		=> "\xe5\xa5\x84",	//CJK UNIFIED IDEOGRAPH
		"\x89\x83"		=> "\xe5\xae\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x89\x84"		=> "\xe5\xbb\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x89\x85"		=> "\xe6\x80\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x89\x86"		=> "\xe6\x8e\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x89\x87"		=> "\xe6\x8f\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x89\x88"		=> "\xe6\xb2\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x89\x89"		=> "\xe6\xbc\x94",	//CJK UNIFIED IDEOGRAPH
		"\x89\x8a"		=> "\xe7\x82\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x89\x8b"		=> "\xe7\x84\x94",	//CJK UNIFIED IDEOGRAPH
		"\x89\x8c"		=> "\xe7\x85\x99",	//CJK UNIFIED IDEOGRAPH
		"\x89\x8d"		=> "\xe7\x87\x95",	//CJK UNIFIED IDEOGRAPH
		"\x89\x8e"		=> "\xe7\x8c\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x89\x8f"		=> "\xe7\xb8\x81",	//CJK UNIFIED IDEOGRAPH
		"\x89\x90"		=> "\xe8\x89\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x89\x91"		=> "\xe8\x8b\x91",	//CJK UNIFIED IDEOGRAPH
		"\x89\x92"		=> "\xe8\x96\x97",	//CJK UNIFIED IDEOGRAPH
		"\x89\x93"		=> "\xe9\x81\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x89\x94"		=> "\xe9\x89\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x89\x95"		=> "\xe9\xb4\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x89\x96"		=> "\xe5\xa1\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x89\x97"		=> "\xe6\x96\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x89\x98"		=> "\xe6\xb1\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x89\x99"		=> "\xe7\x94\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x89\x9a"		=> "\xe5\x87\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x89\x9b"		=> "\xe5\xa4\xae",	//CJK UNIFIED IDEOGRAPH
		"\x89\x9c"		=> "\xe5\xa5\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x89\x9d"		=> "\xe5\xbe\x80",	//CJK UNIFIED IDEOGRAPH
		"\x89\x9e"		=> "\xe5\xbf\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x89\x9f"		=> "\xe6\x8a\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa0"		=> "\xe6\x97\xba",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa1"		=> "\xe6\xa8\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa2"		=> "\xe6\xac\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa3"		=> "\xe6\xae\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa4"		=> "\xe7\x8e\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa5"		=> "\xe7\xbf\x81",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa6"		=> "\xe8\xa5\x96",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa7"		=> "\xe9\xb4\xac",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa8"		=> "\xe9\xb4\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x89\xa9"		=> "\xe9\xbb\x84",	//CJK UNIFIED IDEOGRAPH
		"\x89\xaa"		=> "\xe5\xb2\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x89\xab"		=> "\xe6\xb2\x96",	//CJK UNIFIED IDEOGRAPH
		"\x89\xac"		=> "\xe8\x8d\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x89\xad"		=> "\xe5\x84\x84",	//CJK UNIFIED IDEOGRAPH
		"\x89\xae"		=> "\xe5\xb1\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x89\xaf"		=> "\xe6\x86\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb0"		=> "\xe8\x87\x86",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb1"		=> "\xe6\xa1\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb2"		=> "\xe7\x89\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb3"		=> "\xe4\xb9\x99",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb4"		=> "\xe4\xbf\xba",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb5"		=> "\xe5\x8d\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb6"		=> "\xe6\x81\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb7"		=> "\xe6\xb8\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb8"		=> "\xe7\xa9\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x89\xb9"		=> "\xe9\x9f\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x89\xba"		=> "\xe4\xb8\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x89\xbb"		=> "\xe5\x8c\x96",	//CJK UNIFIED IDEOGRAPH
		"\x89\xbc"		=> "\xe4\xbb\xae",	//CJK UNIFIED IDEOGRAPH
		"\x89\xbd"		=> "\xe4\xbd\x95",	//CJK UNIFIED IDEOGRAPH
		"\x89\xbe"		=> "\xe4\xbc\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x89\xbf"		=> "\xe4\xbe\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc0"		=> "\xe4\xbd\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc1"		=> "\xe5\x8a\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc2"		=> "\xe5\x8f\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc3"		=> "\xe5\x98\x89",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc4"		=> "\xe5\xa4\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc5"		=> "\xe5\xab\x81",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc6"		=> "\xe5\xae\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc7"		=> "\xe5\xaf\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc8"		=> "\xe7\xa7\x91",	//CJK UNIFIED IDEOGRAPH
		"\x89\xc9"		=> "\xe6\x9a\x87",	//CJK UNIFIED IDEOGRAPH
		"\x89\xca"		=> "\xe6\x9e\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x89\xcb"		=> "\xe6\x9e\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x89\xcc"		=> "\xe6\xad\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x89\xcd"		=> "\xe6\xb2\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x89\xce"		=> "\xe7\x81\xab",	//CJK UNIFIED IDEOGRAPH
		"\x89\xcf"		=> "\xe7\x8f\x82",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd0"		=> "\xe7\xa6\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd1"		=> "\xe7\xa6\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd2"		=> "\xe7\xa8\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd3"		=> "\xe7\xae\x87",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd4"		=> "\xe8\x8a\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd5"		=> "\xe8\x8b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd6"		=> "\xe8\x8c\x84",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd7"		=> "\xe8\x8d\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd8"		=> "\xe8\x8f\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x89\xd9"		=> "\xe8\x8f\x93",	//CJK UNIFIED IDEOGRAPH
		"\x89\xda"		=> "\xe8\x9d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x89\xdb"		=> "\xe8\xaa\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x89\xdc"		=> "\xe5\x98\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x89\xdd"		=> "\xe8\xb2\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x89\xde"		=> "\xe8\xbf\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x89\xdf"		=> "\xe9\x81\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe0"		=> "\xe9\x9c\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe1"		=> "\xe8\x9a\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe2"		=> "\xe4\xbf\x84",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe3"		=> "\xe5\xb3\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe4"		=> "\xe6\x88\x91",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe5"		=> "\xe7\x89\x99",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe6"		=> "\xe7\x94\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe7"		=> "\xe8\x87\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe8"		=> "\xe8\x8a\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x89\xe9"		=> "\xe8\x9b\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x89\xea"		=> "\xe8\xb3\x80",	//CJK UNIFIED IDEOGRAPH
		"\x89\xeb"		=> "\xe9\x9b\x85",	//CJK UNIFIED IDEOGRAPH
		"\x89\xec"		=> "\xe9\xa4\x93",	//CJK UNIFIED IDEOGRAPH
		"\x89\xed"		=> "\xe9\xa7\x95",	//CJK UNIFIED IDEOGRAPH
		"\x89\xee"		=> "\xe4\xbb\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x89\xef"		=> "\xe4\xbc\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf0"		=> "\xe8\xa7\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf1"		=> "\xe5\x9b\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf2"		=> "\xe5\xa1\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf3"		=> "\xe5\xa3\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf4"		=> "\xe5\xbb\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf5"		=> "\xe5\xbf\xab",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf6"		=> "\xe6\x80\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf7"		=> "\xe6\x82\x94",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf8"		=> "\xe6\x81\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x89\xf9"		=> "\xe6\x87\x90",	//CJK UNIFIED IDEOGRAPH
		"\x89\xfa"		=> "\xe6\x88\x92",	//CJK UNIFIED IDEOGRAPH
		"\x89\xfb"		=> "\xe6\x8b\x90",	//CJK UNIFIED IDEOGRAPH
		"\x89\xfc"		=> "\xe6\x94\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x40"		=> "\xe9\xad\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x41"		=> "\xe6\x99\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x42"		=> "\xe6\xa2\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x43"		=> "\xe6\xb5\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x44"		=> "\xe7\x81\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x45"		=> "\xe7\x95\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x46"		=> "\xe7\x9a\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x47"		=> "\xe7\xb5\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x48"		=> "\xe8\x8a\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x49"		=> "\xe8\x9f\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x4a"		=> "\xe9\x96\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x4b"		=> "\xe9\x9a\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x4c"		=> "\xe8\xb2\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x4d"		=> "\xe5\x87\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x4e"		=> "\xe5\x8a\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x4f"		=> "\xe5\xa4\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x50"		=> "\xe5\x92\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x51"		=> "\xe5\xae\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x52"		=> "\xe5\xb4\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x53"		=> "\xe6\x85\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x54"		=> "\xe6\xa6\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x55"		=> "\xe6\xb6\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x56"		=> "\xe7\xa2\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x57"		=> "\xe8\x93\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x58"		=> "\xe8\xa1\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x59"		=> "\xe8\xa9\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x5a"		=> "\xe9\x8e\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x5b"		=> "\xe9\xaa\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x5c"		=> "\xe6\xb5\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x5d"		=> "\xe9\xa6\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x5e"		=> "\xe8\x9b\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x5f"		=> "\xe5\x9e\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x60"		=> "\xe6\x9f\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x61"		=> "\xe8\x9b\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x62"		=> "\xe9\x88\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x63"		=> "\xe5\x8a\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x64"		=> "\xe5\x9a\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x65"		=> "\xe5\x90\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x66"		=> "\xe5\xbb\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x67"		=> "\xe6\x8b\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x68"		=> "\xe6\x92\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x69"		=> "\xe6\xa0\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x6a"		=> "\xe6\xa0\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x6b"		=> "\xe6\xae\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x6c"		=> "\xe7\x8d\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x6d"		=> "\xe7\xa2\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x6e"		=> "\xe7\xa9\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x6f"		=> "\xe8\xa6\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x70"		=> "\xe8\xa7\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x71"		=> "\xe8\xb5\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x72"		=> "\xe8\xbc\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x73"		=> "\xe9\x83\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x74"		=> "\xe9\x96\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x75"		=> "\xe9\x9a\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x76"		=> "\xe9\x9d\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x77"		=> "\xe5\xad\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x78"		=> "\xe5\xb2\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x79"		=> "\xe6\xa5\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x7a"		=> "\xe9\xa1\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x7b"		=> "\xe9\xa1\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x7c"		=> "\xe6\x8e\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x7d"		=> "\xe7\xac\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x7e"		=> "\xe6\xa8\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x80"		=> "\xe6\xa9\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x81"		=> "\xe6\xa2\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x82"		=> "\xe9\xb0\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x83"		=> "\xe6\xbd\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x84"		=> "\xe5\x89\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x85"		=> "\xe5\x96\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x86"		=> "\xe6\x81\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x87"		=> "\xe6\x8b\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x88"		=> "\xe6\xb4\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x89"		=> "\xe6\xb8\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x8a"		=> "\xe6\xbb\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x8b"		=> "\xe8\x91\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x8c"		=> "\xe8\xa4\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x8d"		=> "\xe8\xbd\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x8e"		=> "\xe4\xb8\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x8f"		=> "\xe9\xb0\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x90"		=> "\xe5\x8f\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x91"		=> "\xe6\xa4\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x92"		=> "\xe6\xa8\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x93"		=> "\xe9\x9e\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x94"		=> "\xe6\xa0\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x95"		=> "\xe5\x85\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x96"		=> "\xe7\xab\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x97"		=> "\xe8\x92\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x98"		=> "\xe9\x87\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x99"		=> "\xe9\x8e\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x9a"		=> "\xe5\x99\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x9b"		=> "\xe9\xb4\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x9c"		=> "\xe6\xa0\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x9d"		=> "\xe8\x8c\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x9e"		=> "\xe8\x90\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8a\x9f"		=> "\xe7\xb2\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa0"		=> "\xe5\x88\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa1"		=> "\xe8\x8b\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa2"		=> "\xe7\x93\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa3"		=> "\xe4\xb9\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa4"		=> "\xe4\xbe\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa5"		=> "\xe5\x86\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa6"		=> "\xe5\xaf\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa7"		=> "\xe5\x88\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa8"		=> "\xe5\x8b\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xa9"		=> "\xe5\x8b\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xaa"		=> "\xe5\xb7\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xab"		=> "\xe5\x96\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xac"		=> "\xe5\xa0\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xad"		=> "\xe5\xa7\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xae"		=> "\xe5\xae\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xaf"		=> "\xe5\xae\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb0"		=> "\xe5\xaf\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb1"		=> "\xe5\xb9\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb2"		=> "\xe5\xb9\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb3"		=> "\xe6\x82\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb4"		=> "\xe6\x84\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb5"		=> "\xe6\x85\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb6"		=> "\xe6\x86\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb7"		=> "\xe6\x8f\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb8"		=> "\xe6\x95\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xb9"		=> "\xe6\x9f\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xba"		=> "\xe6\xa1\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xbb"		=> "\xe6\xa3\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xbc"		=> "\xe6\xac\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xbd"		=> "\xe6\xad\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xbe"		=> "\xe6\xb1\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xbf"		=> "\xe6\xbc\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc0"		=> "\xe6\xbe\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc1"		=> "\xe6\xbd\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc2"		=> "\xe7\x92\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc3"		=> "\xe7\x94\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc4"		=> "\xe7\x9b\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc5"		=> "\xe7\x9c\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc6"		=> "\xe7\xab\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc7"		=> "\xe7\xae\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc8"		=> "\xe7\xb0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xc9"		=> "\xe7\xb7\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xca"		=> "\xe7\xbc\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xcb"		=> "\xe7\xbf\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xcc"		=> "\xe8\x82\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xcd"		=> "\xe8\x89\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xce"		=> "\xe8\x8e\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xcf"		=> "\xe8\xa6\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd0"		=> "\xe8\xab\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd1"		=> "\xe8\xb2\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd2"		=> "\xe9\x82\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd3"		=> "\xe9\x91\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd4"		=> "\xe9\x96\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd5"		=> "\xe9\x96\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd6"		=> "\xe9\x96\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd7"		=> "\xe9\x99\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd8"		=> "\xe9\x9f\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xd9"		=> "\xe9\xa4\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xda"		=> "\xe8\x88\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xdb"		=> "\xe4\xb8\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xdc"		=> "\xe5\x90\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xdd"		=> "\xe5\xb2\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xde"		=> "\xe5\xb7\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xdf"		=> "\xe7\x8e\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe0"		=> "\xe7\x99\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe1"		=> "\xe7\x9c\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe2"		=> "\xe5\xb2\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe3"		=> "\xe7\xbf\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe4"		=> "\xe8\xb4\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe5"		=> "\xe9\x9b\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe6"		=> "\xe9\xa0\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe7"		=> "\xe9\xa1\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe8"		=> "\xe9\xa1\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xe9"		=> "\xe4\xbc\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xea"		=> "\xe4\xbc\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xeb"		=> "\xe5\x8d\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xec"		=> "\xe5\x96\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xed"		=> "\xe5\x99\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xee"		=> "\xe5\x9f\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xef"		=> "\xe5\xa5\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf0"		=> "\xe5\xac\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf1"		=> "\xe5\xaf\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf2"		=> "\xe5\xb2\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf3"		=> "\xe5\xb8\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf4"		=> "\xe5\xb9\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf5"		=> "\xe5\xbf\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf6"		=> "\xe6\x8f\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf7"		=> "\xe6\x9c\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf8"		=> "\xe6\x97\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xf9"		=> "\xe6\x97\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xfa"		=> "\xe6\x9c\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xfb"		=> "\xe6\xa3\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8a\xfc"		=> "\xe6\xa3\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x40"		=> "\xe6\xa9\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x41"		=> "\xe5\xb8\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x42"		=> "\xe6\xaf\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x43"		=> "\xe6\xb0\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x44"		=> "\xe6\xb1\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x45"		=> "\xe7\x95\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x46"		=> "\xe7\xa5\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x47"		=> "\xe5\xad\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x48"		=> "\xe7\xa8\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x49"		=> "\xe7\xb4\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x4a"		=> "\xe5\xbe\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x4b"		=> "\xe8\xa6\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x4c"		=> "\xe8\xa8\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x4d"		=> "\xe8\xb2\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x4e"		=> "\xe8\xb5\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x4f"		=> "\xe8\xbb\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x50"		=> "\xe8\xbc\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x51"		=> "\xe9\xa3\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x52"		=> "\xe9\xa8\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x53"		=> "\xe9\xac\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x54"		=> "\xe4\xba\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x55"		=> "\xe5\x81\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x56"		=> "\xe5\x84\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x57"		=> "\xe5\xa6\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x58"		=> "\xe5\xae\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x59"		=> "\xe6\x88\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x5a"		=> "\xe6\x8a\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x5b"		=> "\xe6\x93\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x5c"		=> "\xe6\xac\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x5d"		=> "\xe7\x8a\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x5e"		=> "\xe7\x96\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x5f"		=> "\xe7\xa5\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x60"		=> "\xe7\xbe\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x61"		=> "\xe8\x9f\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x62"		=> "\xe8\xaa\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x63"		=> "\xe8\xad\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x64"		=> "\xe6\x8e\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x65"		=> "\xe8\x8f\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x66"		=> "\xe9\x9e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x67"		=> "\xe5\x90\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x68"		=> "\xe5\x90\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x69"		=> "\xe5\x96\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x6a"		=> "\xe6\xa1\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x6b"		=> "\xe6\xa9\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x6c"		=> "\xe8\xa9\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x6d"		=> "\xe7\xa0\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x6e"		=> "\xe6\x9d\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x6f"		=> "\xe9\xbb\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x70"		=> "\xe5\x8d\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x71"		=> "\xe5\xae\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x72"		=> "\xe8\x84\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x73"		=> "\xe8\x99\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x74"		=> "\xe9\x80\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x75"		=> "\xe4\xb8\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x76"		=> "\xe4\xb9\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x77"		=> "\xe4\xbb\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x78"		=> "\xe4\xbc\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x79"		=> "\xe5\x8f\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x7a"		=> "\xe5\x90\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x7b"		=> "\xe5\xae\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x7c"		=> "\xe5\xbc\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x7d"		=> "\xe6\x80\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x7e"		=> "\xe6\x95\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x80"		=> "\xe6\x9c\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x81"		=> "\xe6\xb1\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x82"		=> "\xe6\xb1\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x83"		=> "\xe6\xb3\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x84"		=> "\xe7\x81\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x85"		=> "\xe7\x90\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x86"		=> "\xe7\xa9\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x87"		=> "\xe7\xaa\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x88"		=> "\xe7\xac\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x89"		=> "\xe7\xb4\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x8a"		=> "\xe7\xb3\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x8b"		=> "\xe7\xb5\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x8c"		=> "\xe6\x97\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x8d"		=> "\xe7\x89\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x8e"		=> "\xe5\x8e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x8f"		=> "\xe5\xb1\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x90"		=> "\xe5\xb7\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x91"		=> "\xe6\x8b\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x92"		=> "\xe6\x8b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x93"		=> "\xe6\x8c\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x94"		=> "\xe6\xb8\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x95"		=> "\xe8\x99\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x96"		=> "\xe8\xa8\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x97"		=> "\xe8\xb7\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x98"		=> "\xe9\x8b\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x99"		=> "\xe6\xbc\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x9a"		=> "\xe7\xa6\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x9b"		=> "\xe9\xad\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x9c"		=> "\xe4\xba\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x9d"		=> "\xe4\xba\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x9e"		=> "\xe4\xba\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8b\x9f"		=> "\xe4\xbe\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa0"		=> "\xe4\xbe\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa1"		=> "\xe5\x83\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa2"		=> "\xe5\x85\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa3"		=> "\xe7\xab\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa4"		=> "\xe5\x85\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa5"		=> "\xe5\x87\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa6"		=> "\xe5\x8d\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa7"		=> "\xe5\x8c\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa8"		=> "\xe5\x8d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xa9"		=> "\xe5\x8f\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xaa"		=> "\xe5\x96\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xab"		=> "\xe5\xa2\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xac"		=> "\xe5\xb3\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xad"		=> "\xe5\xbc\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xae"		=> "\xe5\xbd\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xaf"		=> "\xe6\x80\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb0"		=> "\xe6\x81\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb1"		=> "\xe6\x81\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb2"		=> "\xe6\x8c\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb3"		=> "\xe6\x95\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb4"		=> "\xe6\xa9\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb5"		=> "\xe6\xb3\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb6"		=> "\xe7\x8b\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb7"		=> "\xe7\x8b\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb8"		=> "\xe7\x9f\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xb9"		=> "\xe8\x83\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xba"		=> "\xe8\x84\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xbb"		=> "\xe8\x88\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xbc"		=> "\xe8\x95\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xbd"		=> "\xe9\x83\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xbe"		=> "\xe9\x8f\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xbf"		=> "\xe9\x9f\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc0"		=> "\xe9\xa5\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc1"		=> "\xe9\xa9\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc2"		=> "\xe4\xbb\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc3"		=> "\xe5\x87\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc4"		=> "\xe5\xb0\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc5"		=> "\xe6\x9a\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc6"		=> "\xe6\xa5\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc7"		=> "\xe5\xb1\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc8"		=> "\xe6\x9b\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xc9"		=> "\xe6\xa5\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xca"		=> "\xe7\x8e\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xcb"		=> "\xe6\xa1\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xcc"		=> "\xe7\xb2\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xcd"		=> "\xe5\x83\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xce"		=> "\xe5\x8b\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xcf"		=> "\xe5\x9d\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd0"		=> "\xe5\xb7\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd1"		=> "\xe9\x8c\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd2"		=> "\xe6\x96\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd3"		=> "\xe6\xac\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd4"		=> "\xe6\xac\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd5"		=> "\xe7\x90\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd6"		=> "\xe7\xa6\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd7"		=> "\xe7\xa6\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd8"		=> "\xe7\xad\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xd9"		=> "\xe7\xb7\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xda"		=> "\xe8\x8a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xdb"		=> "\xe8\x8f\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xdc"		=> "\xe8\xa1\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xdd"		=> "\xe8\xa5\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xde"		=> "\xe8\xac\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xdf"		=> "\xe8\xbf\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe0"		=> "\xe9\x87\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe1"		=> "\xe5\x90\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe2"		=> "\xe9\x8a\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe3"		=> "\xe4\xb9\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe4"		=> "\xe5\x80\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe5"		=> "\xe5\x8f\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe6"		=> "\xe5\x8c\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe7"		=> "\xe7\x8b\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe8"		=> "\xe7\x8e\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xe9"		=> "\xe7\x9f\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xea"		=> "\xe8\x8b\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xeb"		=> "\xe8\xba\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xec"		=> "\xe9\xa7\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xed"		=> "\xe9\xa7\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xee"		=> "\xe9\xa7\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xef"		=> "\xe5\x85\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf0"		=> "\xe6\x84\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf1"		=> "\xe8\x99\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf2"		=> "\xe5\x96\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf3"		=> "\xe7\xa9\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf4"		=> "\xe5\x81\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf5"		=> "\xe5\xaf\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf6"		=> "\xe9\x81\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf7"		=> "\xe9\x9a\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf8"		=> "\xe4\xb8\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xf9"		=> "\xe6\xab\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xfa"		=> "\xe9\x87\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xfb"		=> "\xe5\xb1\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8b\xfc"		=> "\xe5\xb1\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x40"		=> "\xe6\x8e\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x41"		=> "\xe7\xaa\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x42"		=> "\xe6\xb2\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x43"		=> "\xe9\x9d\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x44"		=> "\xe8\xbd\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x45"		=> "\xe7\xaa\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x46"		=> "\xe7\x86\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x47"		=> "\xe9\x9a\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x48"		=> "\xe7\xb2\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x49"		=> "\xe6\xa0\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x4a"		=> "\xe7\xb9\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x4b"		=> "\xe6\xa1\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x4c"		=> "\xe9\x8d\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x4d"		=> "\xe5\x8b\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x4e"		=> "\xe5\x90\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x4f"		=> "\xe8\x96\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x50"		=> "\xe8\xa8\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x51"		=> "\xe7\xbe\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x52"		=> "\xe8\xbb\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x53"		=> "\xe9\x83\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x54"		=> "\xe5\x8d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x55"		=> "\xe8\xa2\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x56"		=> "\xe7\xa5\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x57"		=> "\xe4\xbf\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x58"		=> "\xe5\x82\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x59"		=> "\xe5\x88\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x5a"		=> "\xe5\x85\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x5b"		=> "\xe5\x95\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x5c"		=> "\xe5\x9c\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x5d"		=> "\xe7\x8f\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x5e"		=> "\xe5\x9e\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x5f"		=> "\xe5\xa5\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x60"		=> "\xe5\xbd\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x61"		=> "\xe5\xbe\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x62"		=> "\xe6\x81\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x63"		=> "\xe6\x85\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x64"		=> "\xe6\x85\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x65"		=> "\xe6\x86\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x66"		=> "\xe6\x8e\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x67"		=> "\xe6\x90\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x68"		=> "\xe6\x95\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x69"		=> "\xe6\x99\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x6a"		=> "\xe6\xa1\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x6b"		=> "\xe6\xb8\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x6c"		=> "\xe7\x95\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x6d"		=> "\xe7\xa8\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x6e"		=> "\xe7\xb3\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x6f"		=> "\xe7\xb5\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x70"		=> "\xe7\xb6\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x71"		=> "\xe7\xb9\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x72"		=> "\xe7\xbd\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x73"		=> "\xe8\x8c\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x74"		=> "\xe8\x8d\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x75"		=> "\xe8\x9b\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x76"		=> "\xe8\xa8\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x77"		=> "\xe8\xa9\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x78"		=> "\xe8\xad\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x79"		=> "\xe8\xbb\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x7a"		=> "\xe9\xa0\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x7b"		=> "\xe9\xb6\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x7c"		=> "\xe8\x8a\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x7d"		=> "\xe8\xbf\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x7e"		=> "\xe9\xaf\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x80"		=> "\xe5\x8a\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x81"		=> "\xe6\x88\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x82"		=> "\xe6\x92\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x83"		=> "\xe6\xbf\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x84"		=> "\xe9\x9a\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x85"		=> "\xe6\xa1\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x86"		=> "\xe5\x82\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x87"		=> "\xe6\xac\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x88"		=> "\xe6\xb1\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x89"		=> "\xe6\xbd\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x8a"		=> "\xe7\xa9\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x8b"		=> "\xe7\xb5\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x8c"		=> "\xe8\xa1\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x8d"		=> "\xe8\xa8\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x8e"		=> "\xe6\x9c\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x8f"		=> "\xe4\xbb\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x90"		=> "\xe5\x80\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x91"		=> "\xe5\x80\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x92"		=> "\xe5\x81\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x93"		=> "\xe5\x85\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x94"		=> "\xe5\x88\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x95"		=> "\xe5\x89\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x96"		=> "\xe5\x96\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x97"		=> "\xe5\x9c\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x98"		=> "\xe5\xa0\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x99"		=> "\xe5\xab\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x9a"		=> "\xe5\xbb\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x9b"		=> "\xe6\x86\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x9c"		=> "\xe6\x87\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x9d"		=> "\xe6\x8b\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x9e"		=> "\xe6\x8d\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8c\x9f"		=> "\xe6\xa4\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa0"		=> "\xe6\xa8\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa1"		=> "\xe7\x89\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa2"		=> "\xe7\x8a\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa3"		=> "\xe7\x8c\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa4"		=> "\xe7\xa0\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa5"		=> "\xe7\xa1\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa6"		=> "\xe7\xb5\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa7"		=> "\xe7\x9c\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa8"		=> "\xe8\x82\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xa9"		=> "\xe8\xa6\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xaa"		=> "\xe8\xac\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xab"		=> "\xe8\xb3\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xac"		=> "\xe8\xbb\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xad"		=> "\xe9\x81\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xae"		=> "\xe9\x8d\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xaf"		=> "\xe9\x99\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb0"		=> "\xe9\xa1\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb1"		=> "\xe9\xa8\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb2"		=> "\xe9\xb9\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb3"		=> "\xe5\x85\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb4"		=> "\xe5\x8e\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb5"		=> "\xe5\x8e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb6"		=> "\xe5\xb9\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb7"		=> "\xe5\xbc\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb8"		=> "\xe6\xb8\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xb9"		=> "\xe6\xba\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xba"		=> "\xe7\x8e\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xbb"		=> "\xe7\x8f\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xbc"		=> "\xe7\xb5\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xbd"		=> "\xe8\x88\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xbe"		=> "\xe8\xa8\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xbf"		=> "\xe8\xab\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc0"		=> "\xe9\x99\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc1"		=> "\xe4\xb9\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc2"		=> "\xe5\x80\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc3"		=> "\xe5\x8f\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc4"		=> "\xe5\x91\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc5"		=> "\xe5\x9b\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc6"		=> "\xe5\xa7\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc7"		=> "\xe5\xad\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc8"		=> "\xe5\xb7\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xc9"		=> "\xe5\xba\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xca"		=> "\xe5\xbc\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xcb"		=> "\xe6\x88\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xcc"		=> "\xe6\x95\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xcd"		=> "\xe6\x9e\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xce"		=> "\xe6\xb9\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xcf"		=> "\xe7\x8b\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd0"		=> "\xe7\xb3\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd1"		=> "\xe8\xa2\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd2"		=> "\xe8\x82\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd3"		=> "\xe8\x83\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd4"		=> "\xe8\x8f\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd5"		=> "\xe8\x99\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd6"		=> "\xe8\xaa\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd7"		=> "\xe8\xb7\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd8"		=> "\xe9\x88\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xd9"		=> "\xe9\x9b\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xda"		=> "\xe9\xa1\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xdb"		=> "\xe9\xbc\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xdc"		=> "\xe4\xba\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xdd"		=> "\xe4\xba\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xde"		=> "\xe4\xbc\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xdf"		=> "\xe5\x8d\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe0"		=> "\xe5\x91\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe1"		=> "\xe5\x90\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe2"		=> "\xe5\xa8\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe3"		=> "\xe5\xbe\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe4"		=> "\xe5\xbe\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe5"		=> "\xe6\x82\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe6"		=> "\xe6\xa2\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe7"		=> "\xe6\xaa\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe8"		=> "\xe7\x91\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xe9"		=> "\xe7\xa2\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xea"		=> "\xe8\xaa\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xeb"		=> "\xe8\xaa\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xec"		=> "\xe8\xad\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xed"		=> "\xe9\x86\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xee"		=> "\xe4\xb9\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xef"		=> "\xe9\xaf\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf0"		=> "\xe4\xba\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf1"		=> "\xe4\xbd\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf2"		=> "\xe4\xbe\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf3"		=> "\xe5\x80\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf4"		=> "\xe5\x80\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf5"		=> "\xe5\x85\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf6"		=> "\xe5\x85\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf7"		=> "\xe5\x8a\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf8"		=> "\xe5\x8a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xf9"		=> "\xe5\x8b\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xfa"		=> "\xe5\x8e\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xfb"		=> "\xe5\x8f\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8c\xfc"		=> "\xe5\x90\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x40"		=> "\xe5\x90\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x41"		=> "\xe5\x96\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x42"		=> "\xe5\x9d\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x43"		=> "\xe5\x9e\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x44"		=> "\xe5\xa5\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x45"		=> "\xe5\xad\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x46"		=> "\xe5\xad\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x47"		=> "\xe5\xae\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x48"		=> "\xe5\xb7\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x49"		=> "\xe5\xb7\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x4a"		=> "\xe5\xb7\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x4b"		=> "\xe5\xb9\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x4c"		=> "\xe5\xba\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x4d"		=> "\xe5\xba\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x4e"		=> "\xe5\xba\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x4f"		=> "\xe5\xbc\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x50"		=> "\xe6\x81\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x51"		=> "\xe6\x85\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x52"		=> "\xe6\x8a\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x53"		=> "\xe6\x8b\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x54"		=> "\xe6\x8e\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x55"		=> "\xe6\x94\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x56"		=> "\xe6\x98\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x57"		=> "\xe6\x99\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x58"		=> "\xe6\x9b\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x59"		=> "\xe6\x9d\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x5a"		=> "\xe6\xa0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x5b"		=> "\xe6\xa2\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x5c"		=> "\xe6\xa7\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x5d"		=> "\xe6\xb1\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x5e"		=> "\xe6\xb4\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x5f"		=> "\xe6\xb5\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x60"		=> "\xe6\xb8\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x61"		=> "\xe6\xba\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x62"		=> "\xe7\x94\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x63"		=> "\xe7\x9a\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x64"		=> "\xe7\xa1\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x65"		=> "\xe7\xa8\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x66"		=> "\xe7\xb3\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x67"		=> "\xe7\xb4\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x68"		=> "\xe7\xb4\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x69"		=> "\xe7\xb5\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x6a"		=> "\xe7\xb6\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x6b"		=> "\xe8\x80\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x6c"		=> "\xe8\x80\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x6d"		=> "\xe8\x82\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x6e"		=> "\xe8\x82\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x6f"		=> "\xe8\x85\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x70"		=> "\xe8\x86\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x71"		=> "\xe8\x88\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x72"		=> "\xe8\x8d\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x73"		=> "\xe8\xa1\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x74"		=> "\xe8\xa1\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x75"		=> "\xe8\xac\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x76"		=> "\xe8\xb2\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x77"		=> "\xe8\xb3\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x78"		=> "\xe9\x83\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x79"		=> "\xe9\x85\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x7a"		=> "\xe9\x89\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x7b"		=> "\xe7\xa0\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x7c"		=> "\xe9\x8b\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x7d"		=> "\xe9\x96\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x7e"		=> "\xe9\x99\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x80"		=> "\xe9\xa0\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x81"		=> "\xe9\xa6\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x82"		=> "\xe9\xab\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x83"		=> "\xe9\xb4\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x84"		=> "\xe5\x89\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x85"		=> "\xe5\x8a\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x86"		=> "\xe5\x8f\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x87"		=> "\xe5\x90\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x88"		=> "\xe5\xa3\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x89"		=> "\xe6\x8b\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x8a"		=> "\xe6\xbf\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x8b"		=> "\xe8\xb1\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x8c"		=> "\xe8\xbd\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x8d"		=> "\xe9\xba\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x8e"		=> "\xe5\x85\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x8f"		=> "\xe5\x88\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x90"		=> "\xe5\x91\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x91"		=> "\xe5\x9b\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x92"		=> "\xe7\xa9\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x93"		=> "\xe9\x85\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x94"		=> "\xe9\xb5\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x95"		=> "\xe9\xbb\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x96"		=> "\xe7\x8d\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x97"		=> "\xe6\xbc\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x98"		=> "\xe8\x85\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x99"		=> "\xe7\x94\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x9a"		=> "\xe5\xbf\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x9b"		=> "\xe6\x83\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x9c"		=> "\xe9\xaa\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x9d"		=> "\xe7\x8b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x9e"		=> "\xe8\xbe\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8d\x9f"		=> "\xe6\xad\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa0"		=> "\xe9\xa0\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa1"		=> "\xe4\xbb\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa2"		=> "\xe5\x9b\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa3"		=> "\xe5\x9d\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa4"		=> "\xe5\xa2\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa5"		=> "\xe5\xa9\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa6"		=> "\xe6\x81\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa7"		=> "\xe6\x87\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa8"		=> "\xe6\x98\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xa9"		=> "\xe6\x98\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xaa"		=> "\xe6\xa0\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xab"		=> "\xe6\xa2\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xac"		=> "\xe6\xb7\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xad"		=> "\xe7\x97\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xae"		=> "\xe7\xb4\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xaf"		=> "\xe8\x89\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb0"		=> "\xe9\xad\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb1"		=> "\xe4\xba\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb2"		=> "\xe4\xbd\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb3"		=> "\xe5\x8f\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb4"		=> "\xe5\x94\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb5"		=> "\xe5\xb5\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb6"		=> "\xe5\xb7\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb7"		=> "\xe5\xb7\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb8"		=> "\xe6\x9f\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xb9"		=> "\xe6\xb2\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xba"		=> "\xe7\x91\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xbb"		=> "\xe7\xa0\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xbc"		=> "\xe8\xa9\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xbd"		=> "\xe9\x8e\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xbe"		=> "\xe8\xa3\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xbf"		=> "\xe5\x9d\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc0"		=> "\xe5\xba\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc1"		=> "\xe6\x8c\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc2"		=> "\xe5\x82\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc3"		=> "\xe5\x82\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc4"		=> "\xe5\x86\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc5"		=> "\xe6\x9c\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc6"		=> "\xe5\x93\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc7"		=> "\xe5\xa1\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc8"		=> "\xe5\xa6\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xc9"		=> "\xe5\xae\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xca"		=> "\xe5\xbd\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xcb"		=> "\xe6\x89\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xcc"		=> "\xe6\x8e\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xcd"		=> "\xe6\xa0\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xce"		=> "\xe6\xad\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xcf"		=> "\xe6\xb8\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd0"		=> "\xe7\x81\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd1"		=> "\xe9\x87\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd2"		=> "\xe7\x8a\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd3"		=> "\xe7\xa0\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd4"		=> "\xe7\xa0\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd5"		=> "\xe7\xa5\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd6"		=> "\xe6\x96\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd7"		=> "\xe7\xb4\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd8"		=> "\xe8\x8f\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xd9"		=> "\xe8\xa3\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xda"		=> "\xe8\xbc\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xdb"		=> "\xe9\x9a\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xdc"		=> "\xe5\x89\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xdd"		=> "\xe5\x9c\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xde"		=> "\xe6\x9d\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xdf"		=> "\xe7\xbd\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe0"		=> "\xe8\xb2\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe1"		=> "\xe5\x86\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe2"		=> "\xe5\x9d\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe3"		=> "\xe9\x98\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe4"		=> "\xe5\xa0\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe5"		=> "\xe6\xa6\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe6"		=> "\xe8\x82\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe7"		=> "\xe5\x92\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe8"		=> "\xe5\xb4\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xe9"		=> "\xe5\x9f\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xea"		=> "\xe7\xa2\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xeb"		=> "\xe9\xb7\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xec"		=> "\xe4\xbd\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xed"		=> "\xe5\x89\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xee"		=> "\xe5\x92\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xef"		=> "\xe6\x90\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf0"		=> "\xe6\x98\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf1"		=> "\xe6\x9c\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf2"		=> "\xe6\x9f\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf3"		=> "\xe7\xaa\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf4"		=> "\xe7\xad\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf5"		=> "\xe7\xb4\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf6"		=> "\xe9\x8c\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf7"		=> "\xe6\xa1\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf8"		=> "\xe9\xae\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xf9"		=> "\xe7\xac\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xfa"		=> "\xe5\x8c\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xfb"		=> "\xe5\x86\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8d\xfc"		=> "\xe5\x88\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x40"		=> "\xe5\xaf\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x41"		=> "\xe6\x8b\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x42"		=> "\xe6\x92\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x43"		=> "\xe6\x93\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x44"		=> "\xe6\x9c\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x45"		=> "\xe6\xae\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x46"		=> "\xe8\x96\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x47"		=> "\xe9\x9b\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x48"		=> "\xe7\x9a\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x49"		=> "\xe9\xaf\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x4a"		=> "\xe6\x8d\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x4b"		=> "\xe9\x8c\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x4c"		=> "\xe9\xae\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x4d"		=> "\xe7\x9a\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x4e"		=> "\xe6\x99\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x4f"		=> "\xe4\xb8\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x50"		=> "\xe5\x82\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x51"		=> "\xe5\x8f\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x52"		=> "\xe5\xb1\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x53"		=> "\xe6\x83\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x54"		=> "\xe6\x92\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x55"		=> "\xe6\x95\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x56"		=> "\xe6\xa1\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x57"		=> "\xe7\x87\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x58"		=> "\xe7\x8f\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x59"		=> "\xe7\x94\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x5a"		=> "\xe7\xae\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x5b"		=> "\xe7\xba\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x5c"		=> "\xe8\x9a\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x5d"		=> "\xe8\xae\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x5e"		=> "\xe8\xb3\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x5f"		=> "\xe9\x85\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x60"		=> "\xe9\xa4\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x61"		=> "\xe6\x96\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x62"		=> "\xe6\x9a\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x63"		=> "\xe6\xae\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x64"		=> "\xe4\xbb\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x65"		=> "\xe4\xbb\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x66"		=> "\xe4\xbc\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x67"		=> "\xe4\xbd\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x68"		=> "\xe5\x88\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x69"		=> "\xe5\x8f\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x6a"		=> "\xe5\x8f\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x6b"		=> "\xe5\x97\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x6c"		=> "\xe5\x9b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x6d"		=> "\xe5\xa3\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x6e"		=> "\xe5\xa7\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x6f"		=> "\xe5\xa7\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x70"		=> "\xe5\xa7\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x71"		=> "\xe5\xad\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x72"		=> "\xe5\xb1\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x73"		=> "\xe5\xb8\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x74"		=> "\xe5\xb8\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x75"		=> "\xe5\xbf\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x76"		=> "\xe6\x80\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x77"		=> "\xe6\x8c\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x78"		=> "\xe6\x94\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x79"		=> "\xe5\xad\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x7a"		=> "\xe6\x96\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x7b"		=> "\xe6\x96\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x7c"		=> "\xe6\x97\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x7d"		=> "\xe6\x9e\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x7e"		=> "\xe6\xad\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x80"		=> "\xe6\xad\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x81"		=> "\xe6\xb0\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x82"		=> "\xe7\x8d\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x83"		=> "\xe7\xa5\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x84"		=> "\xe7\xa7\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x85"		=> "\xe7\xb3\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x86"		=> "\xe7\xb4\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x87"		=> "\xe7\xb4\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x88"		=> "\xe8\x82\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x89"		=> "\xe8\x84\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x8a"		=> "\xe8\x87\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x8b"		=> "\xe8\xa6\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x8c"		=> "\xe8\xa9\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x8d"		=> "\xe8\xa9\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x8e"		=> "\xe8\xa9\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x8f"		=> "\xe8\xaa\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x90"		=> "\xe8\xab\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x91"		=> "\xe8\xb3\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x92"		=> "\xe8\xb3\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x93"		=> "\xe9\x9b\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x94"		=> "\xe9\xa3\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x95"		=> "\xe6\xad\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x96"		=> "\xe4\xba\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x97"		=> "\xe4\xbc\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x98"		=> "\xe4\xbe\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x99"		=> "\xe5\x85\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x9a"		=> "\xe5\xad\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x9b"		=> "\xe5\xaf\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x9c"		=> "\xe6\x85\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x9d"		=> "\xe6\x8c\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x9e"		=> "\xe6\x99\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8e\x9f"		=> "\xe6\xac\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa0"		=> "\xe6\xbb\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa1"		=> "\xe6\xb2\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa2"		=> "\xe7\x88\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa3"		=> "\xe7\x92\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa4"		=> "\xe7\x97\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa5"		=> "\xe7\xa3\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa6"		=> "\xe7\xa4\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa7"		=> "\xe8\x80\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa8"		=> "\xe8\x80\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xa9"		=> "\xe8\x87\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xaa"		=> "\xe8\x92\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xab"		=> "\xe8\xbe\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xac"		=> "\xe6\xb1\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xad"		=> "\xe9\xb9\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xae"		=> "\xe5\xbc\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xaf"		=> "\xe8\xad\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb0"		=> "\xe9\xb4\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb1"		=> "\xe7\xab\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb2"		=> "\xe8\xbb\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb3"		=> "\xe5\xae\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb4"		=> "\xe9\x9b\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb5"		=> "\xe4\xb8\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb6"		=> "\xe5\x8f\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb7"		=> "\xe5\x9f\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb8"		=> "\xe5\xa4\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xb9"		=> "\xe5\xab\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xba"		=> "\xe5\xae\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xbb"		=> "\xe6\x82\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xbc"		=> "\xe6\xb9\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xbd"		=> "\xe6\xbc\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xbe"		=> "\xe7\x96\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xbf"		=> "\xe8\xb3\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc0"		=> "\xe5\xae\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc1"		=> "\xe8\x94\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc2"		=> "\xe7\xaf\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc3"		=> "\xe5\x81\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc4"		=> "\xe6\x9f\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc5"		=> "\xe8\x8a\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc6"		=> "\xe5\xb1\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc7"		=> "\xe8\x95\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc8"		=> "\xe7\xb8\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xc9"		=> "\xe8\x88\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xca"		=> "\xe5\x86\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xcb"		=> "\xe5\xb0\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xcc"		=> "\xe6\x8d\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xcd"		=> "\xe8\xb5\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xce"		=> "\xe6\x96\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xcf"		=> "\xe7\x85\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd0"		=> "\xe7\xa4\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd1"		=> "\xe7\xb4\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd2"		=> "\xe8\x80\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd3"		=> "\xe8\xac\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd4"		=> "\xe8\xbb\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd5"		=> "\xe9\x81\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd6"		=> "\xe8\x9b\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd7"		=> "\xe9\x82\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd8"		=> "\xe5\x80\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xd9"		=> "\xe5\x8b\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xda"		=> "\xe5\xb0\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xdb"		=> "\xe6\x9d\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xdc"		=> "\xe7\x81\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xdd"		=> "\xe7\x88\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xde"		=> "\xe9\x85\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xdf"		=> "\xe9\x87\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe0"		=> "\xe9\x8c\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe1"		=> "\xe8\x8b\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe2"		=> "\xe5\xaf\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe3"		=> "\xe5\xbc\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe4"		=> "\xe6\x83\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe5"		=> "\xe4\xb8\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe6"		=> "\xe5\x8f\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe7"		=> "\xe5\xae\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe8"		=> "\xe6\x89\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xe9"		=> "\xe6\x9c\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xea"		=> "\xe6\xae\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xeb"		=> "\xe7\x8b\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xec"		=> "\xe7\x8f\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xed"		=> "\xe7\xa8\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xee"		=> "\xe8\x85\xab",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xef"		=> "\xe8\xb6\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf0"		=> "\xe9\x85\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf1"		=> "\xe9\xa6\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf2"		=> "\xe5\x84\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf3"		=> "\xe5\x8f\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf4"		=> "\xe5\x91\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf5"		=> "\xe5\xaf\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf6"		=> "\xe6\x8e\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf7"		=> "\xe6\xa8\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf8"		=> "\xe7\xb6\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xf9"		=> "\xe9\x9c\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xfa"		=> "\xe5\x9b\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xfb"		=> "\xe5\x8f\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8e\xfc"		=> "\xe5\x91\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x40"		=> "\xe5\xae\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x41"		=> "\xe5\xb0\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x42"		=> "\xe5\xb7\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x43"		=> "\xe4\xbf\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x44"		=> "\xe6\x84\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x45"		=> "\xe6\x8b\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x46"		=> "\xe6\xb4\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x47"		=> "\xe7\xa7\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x48"		=> "\xe7\xa7\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x49"		=> "\xe7\xb5\x82",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x4a"		=> "\xe7\xb9\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x4b"		=> "\xe7\xbf\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x4c"		=> "\xe8\x87\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x4d"		=> "\xe8\x88\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x4e"		=> "\xe8\x92\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x4f"		=> "\xe8\xa1\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x50"		=> "\xe8\xa5\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x51"		=> "\xe8\xae\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x52"		=> "\xe8\xb9\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x53"		=> "\xe8\xbc\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x54"		=> "\xe9\x80\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x55"		=> "\xe9\x85\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x56"		=> "\xe9\x85\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x57"		=> "\xe9\x9b\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x58"		=> "\xe9\x86\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x59"		=> "\xe4\xbb\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x5a"		=> "\xe4\xbd\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x5b"		=> "\xe5\x85\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x5c"		=> "\xe5\x8d\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x5d"		=> "\xe5\xbe\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x5e"		=> "\xe6\x88\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x5f"		=> "\xe6\x9f\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x60"		=> "\xe6\xb1\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x61"		=> "\xe6\xb8\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x62"		=> "\xe7\x8d\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x63"		=> "\xe7\xb8\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x64"		=> "\xe9\x87\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x65"		=> "\xe9\x8a\x83",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x66"		=> "\xe5\x8f\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x67"		=> "\xe5\xa4\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x68"		=> "\xe5\xae\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x69"		=> "\xe6\xb7\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x6a"		=> "\xe7\xa5\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x6b"		=> "\xe7\xb8\xae",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x6c"		=> "\xe7\xb2\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x6d"		=> "\xe5\xa1\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x6e"		=> "\xe7\x86\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x6f"		=> "\xe5\x87\xba",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x70"		=> "\xe8\xa1\x93",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x71"		=> "\xe8\xbf\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x72"		=> "\xe4\xbf\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x73"		=> "\xe5\xb3\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x74"		=> "\xe6\x98\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x75"		=> "\xe7\x9e\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x76"		=> "\xe7\xab\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x77"		=> "\xe8\x88\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x78"		=> "\xe9\xa7\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x79"		=> "\xe5\x87\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x7a"		=> "\xe5\xbe\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x7b"		=> "\xe6\x97\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x7c"		=> "\xe6\xa5\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x7d"		=> "\xe6\xae\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x7e"		=> "\xe6\xb7\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x80"		=> "\xe6\xba\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x81"		=> "\xe6\xbd\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x82"		=> "\xe7\x9b\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x83"		=> "\xe7\xb4\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x84"		=> "\xe5\xb7\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x85"		=> "\xe9\x81\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x86"		=> "\xe9\x86\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x87"		=> "\xe9\xa0\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x88"		=> "\xe5\x87\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x89"		=> "\xe5\x88\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x8a"		=> "\xe6\x89\x80",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x8b"		=> "\xe6\x9a\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x8c"		=> "\xe6\x9b\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x8d"		=> "\xe6\xb8\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x8e"		=> "\xe5\xba\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x8f"		=> "\xe7\xb7\x92",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x90"		=> "\xe7\xbd\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x91"		=> "\xe6\x9b\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x92"		=> "\xe8\x96\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x93"		=> "\xe8\x97\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x94"		=> "\xe8\xab\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x95"		=> "\xe5\x8a\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x96"		=> "\xe5\x8f\x99",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x97"		=> "\xe5\xa5\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x98"		=> "\xe5\xba\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x99"		=> "\xe5\xbe\x90",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x9a"		=> "\xe6\x81\x95",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x9b"		=> "\xe9\x8b\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x9c"		=> "\xe9\x99\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x9d"		=> "\xe5\x82\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x9e"		=> "\xe5\x84\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8f\x9f"		=> "\xe5\x8b\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa0"		=> "\xe5\x8c\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa1"		=> "\xe5\x8d\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa2"		=> "\xe5\x8f\xac",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa3"		=> "\xe5\x93\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa4"		=> "\xe5\x95\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa5"		=> "\xe5\x94\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa6"		=> "\xe5\x98\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa7"		=> "\xe5\xa5\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa8"		=> "\xe5\xa6\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xa9"		=> "\xe5\xa8\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xaa"		=> "\xe5\xae\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xab"		=> "\xe5\xb0\x86",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xac"		=> "\xe5\xb0\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xad"		=> "\xe5\xb0\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xae"		=> "\xe5\xb0\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xaf"		=> "\xe5\xba\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb0"		=> "\xe5\xba\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb1"		=> "\xe5\xbb\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb2"		=> "\xe5\xbd\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb3"		=> "\xe6\x89\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb4"		=> "\xe6\x8a\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb5"		=> "\xe6\x8b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb6"		=> "\xe6\x8e\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb7"		=> "\xe6\x8d\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb8"		=> "\xe6\x98\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xb9"		=> "\xe6\x98\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xba"		=> "\xe6\x98\xad",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xbb"		=> "\xe6\x99\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xbc"		=> "\xe6\x9d\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xbd"		=> "\xe6\xa2\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xbe"		=> "\xe6\xa8\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xbf"		=> "\xe6\xa8\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc0"		=> "\xe6\xb2\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc1"		=> "\xe6\xb6\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc2"		=> "\xe6\xb8\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc3"		=> "\xe6\xb9\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc4"		=> "\xe7\x84\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc5"		=> "\xe7\x84\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc6"		=> "\xe7\x85\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc7"		=> "\xe7\x97\x87",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc8"		=> "\xe7\x9c\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xc9"		=> "\xe7\xa1\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xca"		=> "\xe7\xa4\x81",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xcb"		=> "\xe7\xa5\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xcc"		=> "\xe7\xa7\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xcd"		=> "\xe7\xab\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xce"		=> "\xe7\xac\x91",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xcf"		=> "\xe7\xb2\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd0"		=> "\xe7\xb4\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd1"		=> "\xe8\x82\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd2"		=> "\xe8\x8f\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd3"		=> "\xe8\x92\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd4"		=> "\xe8\x95\x89",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd5"		=> "\xe8\xa1\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd6"		=> "\xe8\xa3\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd7"		=> "\xe8\xa8\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd8"		=> "\xe8\xa8\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xd9"		=> "\xe8\xa9\x94",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xda"		=> "\xe8\xa9\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xdb"		=> "\xe8\xb1\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xdc"		=> "\xe8\xb3\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xdd"		=> "\xe9\x86\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xde"		=> "\xe9\x89\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xdf"		=> "\xe9\x8d\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe0"		=> "\xe9\x90\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe1"		=> "\xe9\x9a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe2"		=> "\xe9\x9e\x98",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe3"		=> "\xe4\xb8\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe4"		=> "\xe4\xb8\x88",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe5"		=> "\xe4\xb8\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe6"		=> "\xe4\xb9\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe7"		=> "\xe5\x86\x97",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe8"		=> "\xe5\x89\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xe9"		=> "\xe5\x9f\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xea"		=> "\xe5\xa0\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xeb"		=> "\xe5\xa3\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xec"		=> "\xe5\xac\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xed"		=> "\xe5\xb8\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xee"		=> "\xe6\x83\x85",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xef"		=> "\xe6\x93\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf0"		=> "\xe6\x9d\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf1"		=> "\xe6\x9d\x96",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf2"		=> "\xe6\xb5\x84",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf3"		=> "\xe7\x8a\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf4"		=> "\xe7\x95\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf5"		=> "\xe7\xa9\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf6"		=> "\xe8\x92\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf7"		=> "\xe8\xad\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf8"		=> "\xe9\x86\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xf9"		=> "\xe9\x8c\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xfa"		=> "\xe5\x98\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xfb"		=> "\xe5\x9f\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x8f\xfc"		=> "\xe9\xa3\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x90\x40"		=> "\xe6\x8b\xad",	//CJK UNIFIED IDEOGRAPH
		"\x90\x41"		=> "\xe6\xa4\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x90\x42"		=> "\xe6\xae\x96",	//CJK UNIFIED IDEOGRAPH
		"\x90\x43"		=> "\xe7\x87\xad",	//CJK UNIFIED IDEOGRAPH
		"\x90\x44"		=> "\xe7\xb9\x94",	//CJK UNIFIED IDEOGRAPH
		"\x90\x45"		=> "\xe8\x81\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x90\x46"		=> "\xe8\x89\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x90\x47"		=> "\xe8\xa7\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x90\x48"		=> "\xe9\xa3\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x90\x49"		=> "\xe8\x9d\x95",	//CJK UNIFIED IDEOGRAPH
		"\x90\x4a"		=> "\xe8\xbe\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x90\x4b"		=> "\xe5\xb0\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x90\x4c"		=> "\xe4\xbc\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x90\x4d"		=> "\xe4\xbf\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x90\x4e"		=> "\xe4\xbe\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x90\x4f"		=> "\xe5\x94\x87",	//CJK UNIFIED IDEOGRAPH
		"\x90\x50"		=> "\xe5\xa8\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x90\x51"		=> "\xe5\xaf\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x90\x52"		=> "\xe5\xaf\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x90\x53"		=> "\xe5\xbf\x83",	//CJK UNIFIED IDEOGRAPH
		"\x90\x54"		=> "\xe6\x85\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x90\x55"		=> "\xe6\x8c\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x90\x56"		=> "\xe6\x96\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x90\x57"		=> "\xe6\x99\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x90\x58"		=> "\xe6\xa3\xae",	//CJK UNIFIED IDEOGRAPH
		"\x90\x59"		=> "\xe6\xa6\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x90\x5a"		=> "\xe6\xb5\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x90\x5b"		=> "\xe6\xb7\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x90\x5c"		=> "\xe7\x94\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x90\x5d"		=> "\xe7\x96\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x90\x5e"		=> "\xe7\x9c\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x90\x5f"		=> "\xe7\xa5\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x90\x60"		=> "\xe7\xa7\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x90\x61"		=> "\xe7\xb4\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x90\x62"		=> "\xe8\x87\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x90\x63"		=> "\xe8\x8a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x90\x64"		=> "\xe8\x96\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x90\x65"		=> "\xe8\xa6\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x90\x66"		=> "\xe8\xa8\xba",	//CJK UNIFIED IDEOGRAPH
		"\x90\x67"		=> "\xe8\xba\xab",	//CJK UNIFIED IDEOGRAPH
		"\x90\x68"		=> "\xe8\xbe\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x90\x69"		=> "\xe9\x80\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x90\x6a"		=> "\xe9\x87\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x90\x6b"		=> "\xe9\x9c\x87",	//CJK UNIFIED IDEOGRAPH
		"\x90\x6c"		=> "\xe4\xba\xba",	//CJK UNIFIED IDEOGRAPH
		"\x90\x6d"		=> "\xe4\xbb\x81",	//CJK UNIFIED IDEOGRAPH
		"\x90\x6e"		=> "\xe5\x88\x83",	//CJK UNIFIED IDEOGRAPH
		"\x90\x6f"		=> "\xe5\xa1\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x90\x70"		=> "\xe5\xa3\xac",	//CJK UNIFIED IDEOGRAPH
		"\x90\x71"		=> "\xe5\xb0\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x90\x72"		=> "\xe7\x94\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x90\x73"		=> "\xe5\xb0\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x90\x74"		=> "\xe8\x85\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x90\x75"		=> "\xe8\xa8\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x90\x76"		=> "\xe8\xbf\x85",	//CJK UNIFIED IDEOGRAPH
		"\x90\x77"		=> "\xe9\x99\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x90\x78"		=> "\xe9\x9d\xad",	//CJK UNIFIED IDEOGRAPH
		"\x90\x79"		=> "\xe7\xac\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x90\x7a"		=> "\xe8\xab\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x90\x7b"		=> "\xe9\xa0\x88",	//CJK UNIFIED IDEOGRAPH
		"\x90\x7c"		=> "\xe9\x85\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x90\x7d"		=> "\xe5\x9b\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x90\x7e"		=> "\xe5\x8e\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x90\x80"		=> "\xe9\x80\x97",	//CJK UNIFIED IDEOGRAPH
		"\x90\x81"		=> "\xe5\x90\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x90\x82"		=> "\xe5\x9e\x82",	//CJK UNIFIED IDEOGRAPH
		"\x90\x83"		=> "\xe5\xb8\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x90\x84"		=> "\xe6\x8e\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x90\x85"		=> "\xe6\xb0\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x90\x86"		=> "\xe7\x82\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x90\x87"		=> "\xe7\x9d\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x90\x88"		=> "\xe7\xb2\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x90\x89"		=> "\xe7\xbf\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x90\x8a"		=> "\xe8\xa1\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x90\x8b"		=> "\xe9\x81\x82",	//CJK UNIFIED IDEOGRAPH
		"\x90\x8c"		=> "\xe9\x85\x94",	//CJK UNIFIED IDEOGRAPH
		"\x90\x8d"		=> "\xe9\x8c\x90",	//CJK UNIFIED IDEOGRAPH
		"\x90\x8e"		=> "\xe9\x8c\x98",	//CJK UNIFIED IDEOGRAPH
		"\x90\x8f"		=> "\xe9\x9a\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x90\x90"		=> "\xe7\x91\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x90\x91"		=> "\xe9\xab\x84",	//CJK UNIFIED IDEOGRAPH
		"\x90\x92"		=> "\xe5\xb4\x87",	//CJK UNIFIED IDEOGRAPH
		"\x90\x93"		=> "\xe5\xb5\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x90\x94"		=> "\xe6\x95\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x90\x95"		=> "\xe6\x9e\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x90\x96"		=> "\xe8\xb6\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x90\x97"		=> "\xe9\x9b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x90\x98"		=> "\xe6\x8d\xae",	//CJK UNIFIED IDEOGRAPH
		"\x90\x99"		=> "\xe6\x9d\x89",	//CJK UNIFIED IDEOGRAPH
		"\x90\x9a"		=> "\xe6\xa4\x99",	//CJK UNIFIED IDEOGRAPH
		"\x90\x9b"		=> "\xe8\x8f\x85",	//CJK UNIFIED IDEOGRAPH
		"\x90\x9c"		=> "\xe9\xa0\x97",	//CJK UNIFIED IDEOGRAPH
		"\x90\x9d"		=> "\xe9\x9b\x80",	//CJK UNIFIED IDEOGRAPH
		"\x90\x9e"		=> "\xe8\xa3\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x90\x9f"		=> "\xe6\xbe\x84",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa0"		=> "\xe6\x91\xba",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa1"		=> "\xe5\xaf\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa2"		=> "\xe4\xb8\x96",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa3"		=> "\xe7\x80\xac",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa4"		=> "\xe7\x95\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa5"		=> "\xe6\x98\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa6"		=> "\xe5\x87\x84",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa7"		=> "\xe5\x88\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa8"		=> "\xe5\x8b\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x90\xa9"		=> "\xe5\xa7\x93",	//CJK UNIFIED IDEOGRAPH
		"\x90\xaa"		=> "\xe5\xbe\x81",	//CJK UNIFIED IDEOGRAPH
		"\x90\xab"		=> "\xe6\x80\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x90\xac"		=> "\xe6\x88\x90",	//CJK UNIFIED IDEOGRAPH
		"\x90\xad"		=> "\xe6\x94\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x90\xae"		=> "\xe6\x95\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x90\xaf"		=> "\xe6\x98\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb0"		=> "\xe6\x99\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb1"		=> "\xe6\xa3\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb2"		=> "\xe6\xa0\x96",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb3"		=> "\xe6\xad\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb4"		=> "\xe6\xb8\x85",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb5"		=> "\xe7\x89\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb6"		=> "\xe7\x94\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb7"		=> "\xe7\x9b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb8"		=> "\xe7\xb2\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x90\xb9"		=> "\xe8\x81\x96",	//CJK UNIFIED IDEOGRAPH
		"\x90\xba"		=> "\xe5\xa3\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x90\xbb"		=> "\xe8\xa3\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x90\xbc"		=> "\xe8\xa5\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x90\xbd"		=> "\xe8\xaa\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x90\xbe"		=> "\xe8\xaa\x93",	//CJK UNIFIED IDEOGRAPH
		"\x90\xbf"		=> "\xe8\xab\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc0"		=> "\xe9\x80\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc1"		=> "\xe9\x86\x92",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc2"		=> "\xe9\x9d\x92",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc3"		=> "\xe9\x9d\x99",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc4"		=> "\xe6\x96\x89",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc5"		=> "\xe7\xa8\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc6"		=> "\xe8\x84\x86",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc7"		=> "\xe9\x9a\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc8"		=> "\xe5\xb8\xad",	//CJK UNIFIED IDEOGRAPH
		"\x90\xc9"		=> "\xe6\x83\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x90\xca"		=> "\xe6\x88\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x90\xcb"		=> "\xe6\x96\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x90\xcc"		=> "\xe6\x98\x94",	//CJK UNIFIED IDEOGRAPH
		"\x90\xcd"		=> "\xe6\x9e\x90",	//CJK UNIFIED IDEOGRAPH
		"\x90\xce"		=> "\xe7\x9f\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x90\xcf"		=> "\xe7\xa9\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd0"		=> "\xe7\xb1\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd1"		=> "\xe7\xb8\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd2"		=> "\xe8\x84\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd3"		=> "\xe8\xb2\xac",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd4"		=> "\xe8\xb5\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd5"		=> "\xe8\xb7\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd6"		=> "\xe8\xb9\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd7"		=> "\xe7\xa2\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd8"		=> "\xe5\x88\x87",	//CJK UNIFIED IDEOGRAPH
		"\x90\xd9"		=> "\xe6\x8b\x99",	//CJK UNIFIED IDEOGRAPH
		"\x90\xda"		=> "\xe6\x8e\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x90\xdb"		=> "\xe6\x91\x82",	//CJK UNIFIED IDEOGRAPH
		"\x90\xdc"		=> "\xe6\x8a\x98",	//CJK UNIFIED IDEOGRAPH
		"\x90\xdd"		=> "\xe8\xa8\xad",	//CJK UNIFIED IDEOGRAPH
		"\x90\xde"		=> "\xe7\xaa\x83",	//CJK UNIFIED IDEOGRAPH
		"\x90\xdf"		=> "\xe7\xaf\x80",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe0"		=> "\xe8\xaa\xac",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe1"		=> "\xe9\x9b\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe2"		=> "\xe7\xb5\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe3"		=> "\xe8\x88\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe4"		=> "\xe8\x9d\x89",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe5"		=> "\xe4\xbb\x99",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe6"		=> "\xe5\x85\x88",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe7"		=> "\xe5\x8d\x83",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe8"		=> "\xe5\x8d\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x90\xe9"		=> "\xe5\xae\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x90\xea"		=> "\xe5\xb0\x82",	//CJK UNIFIED IDEOGRAPH
		"\x90\xeb"		=> "\xe5\xb0\x96",	//CJK UNIFIED IDEOGRAPH
		"\x90\xec"		=> "\xe5\xb7\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x90\xed"		=> "\xe6\x88\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x90\xee"		=> "\xe6\x89\x87",	//CJK UNIFIED IDEOGRAPH
		"\x90\xef"		=> "\xe6\x92\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf0"		=> "\xe6\xa0\x93",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf1"		=> "\xe6\xa0\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf2"		=> "\xe6\xb3\x89",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf3"		=> "\xe6\xb5\x85",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf4"		=> "\xe6\xb4\x97",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf5"		=> "\xe6\x9f\x93",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf6"		=> "\xe6\xbd\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf7"		=> "\xe7\x85\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf8"		=> "\xe7\x85\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x90\xf9"		=> "\xe6\x97\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x90\xfa"		=> "\xe7\xa9\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x90\xfb"		=> "\xe7\xae\xad",	//CJK UNIFIED IDEOGRAPH
		"\x90\xfc"		=> "\xe7\xb7\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x91\x40"		=> "\xe7\xb9\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x91\x41"		=> "\xe7\xbe\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x91\x42"		=> "\xe8\x85\xba",	//CJK UNIFIED IDEOGRAPH
		"\x91\x43"		=> "\xe8\x88\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x91\x44"		=> "\xe8\x88\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x91\x45"		=> "\xe8\x96\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x91\x46"		=> "\xe8\xa9\xae",	//CJK UNIFIED IDEOGRAPH
		"\x91\x47"		=> "\xe8\xb3\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x91\x48"		=> "\xe8\xb7\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x91\x49"		=> "\xe9\x81\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x91\x4a"		=> "\xe9\x81\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x91\x4b"		=> "\xe9\x8a\xad",	//CJK UNIFIED IDEOGRAPH
		"\x91\x4c"		=> "\xe9\x8a\x91",	//CJK UNIFIED IDEOGRAPH
		"\x91\x4d"		=> "\xe9\x96\x83",	//CJK UNIFIED IDEOGRAPH
		"\x91\x4e"		=> "\xe9\xae\xae",	//CJK UNIFIED IDEOGRAPH
		"\x91\x4f"		=> "\xe5\x89\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x91\x50"		=> "\xe5\x96\x84",	//CJK UNIFIED IDEOGRAPH
		"\x91\x51"		=> "\xe6\xbc\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x91\x52"		=> "\xe7\x84\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x91\x53"		=> "\xe5\x85\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x91\x54"		=> "\xe7\xa6\x85",	//CJK UNIFIED IDEOGRAPH
		"\x91\x55"		=> "\xe7\xb9\x95",	//CJK UNIFIED IDEOGRAPH
		"\x91\x56"		=> "\xe8\x86\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x91\x57"		=> "\xe7\xb3\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x91\x58"		=> "\xe5\x99\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x91\x59"		=> "\xe5\xa1\x91",	//CJK UNIFIED IDEOGRAPH
		"\x91\x5a"		=> "\xe5\xb2\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x91\x5b"		=> "\xe6\x8e\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x91\x5c"		=> "\xe6\x9b\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x91\x5d"		=> "\xe6\x9b\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x91\x5e"		=> "\xe6\xa5\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x91\x5f"		=> "\xe7\x8b\x99",	//CJK UNIFIED IDEOGRAPH
		"\x91\x60"		=> "\xe7\x96\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x91\x61"		=> "\xe7\x96\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x91\x62"		=> "\xe7\xa4\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x91\x63"		=> "\xe7\xa5\x96",	//CJK UNIFIED IDEOGRAPH
		"\x91\x64"		=> "\xe7\xa7\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x91\x65"		=> "\xe7\xb2\x97",	//CJK UNIFIED IDEOGRAPH
		"\x91\x66"		=> "\xe7\xb4\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x91\x67"		=> "\xe7\xb5\x84",	//CJK UNIFIED IDEOGRAPH
		"\x91\x68"		=> "\xe8\x98\x87",	//CJK UNIFIED IDEOGRAPH
		"\x91\x69"		=> "\xe8\xa8\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x91\x6a"		=> "\xe9\x98\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x91\x6b"		=> "\xe9\x81\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x91\x6c"		=> "\xe9\xbc\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x91\x6d"		=> "\xe5\x83\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x91\x6e"		=> "\xe5\x89\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x91\x6f"		=> "\xe5\x8f\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x91\x70"		=> "\xe5\x8f\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x91\x71"		=> "\xe5\x80\x89",	//CJK UNIFIED IDEOGRAPH
		"\x91\x72"		=> "\xe5\x96\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x91\x73"		=> "\xe5\xa3\xae",	//CJK UNIFIED IDEOGRAPH
		"\x91\x74"		=> "\xe5\xa5\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x91\x75"		=> "\xe7\x88\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x91\x76"		=> "\xe5\xae\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x91\x77"		=> "\xe5\xb1\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x91\x78"		=> "\xe5\x8c\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x91\x79"		=> "\xe6\x83\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x91\x7a"		=> "\xe6\x83\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x91\x7b"		=> "\xe6\x8d\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x91\x7c"		=> "\xe6\x8e\x83",	//CJK UNIFIED IDEOGRAPH
		"\x91\x7d"		=> "\xe6\x8c\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x91\x7e"		=> "\xe6\x8e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x91\x80"		=> "\xe6\x93\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x91\x81"		=> "\xe6\x97\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x91\x82"		=> "\xe6\x9b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x91\x83"		=> "\xe5\xb7\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x91\x84"		=> "\xe6\xa7\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x91\x85"		=> "\xe6\xa7\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x91\x86"		=> "\xe6\xbc\x95",	//CJK UNIFIED IDEOGRAPH
		"\x91\x87"		=> "\xe7\x87\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x91\x88"		=> "\xe4\xba\x89",	//CJK UNIFIED IDEOGRAPH
		"\x91\x89"		=> "\xe7\x97\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x91\x8a"		=> "\xe7\x9b\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x91\x8b"		=> "\xe7\xaa\x93",	//CJK UNIFIED IDEOGRAPH
		"\x91\x8c"		=> "\xe7\xb3\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x91\x8d"		=> "\xe7\xb7\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x91\x8e"		=> "\xe7\xb6\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x91\x8f"		=> "\xe8\x81\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x91\x90"		=> "\xe8\x8d\x89",	//CJK UNIFIED IDEOGRAPH
		"\x91\x91"		=> "\xe8\x8d\x98",	//CJK UNIFIED IDEOGRAPH
		"\x91\x92"		=> "\xe8\x91\xac",	//CJK UNIFIED IDEOGRAPH
		"\x91\x93"		=> "\xe8\x92\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x91\x94"		=> "\xe8\x97\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x91\x95"		=> "\xe8\xa3\x85",	//CJK UNIFIED IDEOGRAPH
		"\x91\x96"		=> "\xe8\xb5\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x91\x97"		=> "\xe9\x80\x81",	//CJK UNIFIED IDEOGRAPH
		"\x91\x98"		=> "\xe9\x81\xad",	//CJK UNIFIED IDEOGRAPH
		"\x91\x99"		=> "\xe9\x8e\x97",	//CJK UNIFIED IDEOGRAPH
		"\x91\x9a"		=> "\xe9\x9c\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x91\x9b"		=> "\xe9\xa8\x92",	//CJK UNIFIED IDEOGRAPH
		"\x91\x9c"		=> "\xe5\x83\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x91\x9d"		=> "\xe5\xa2\x97",	//CJK UNIFIED IDEOGRAPH
		"\x91\x9e"		=> "\xe6\x86\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x91\x9f"		=> "\xe8\x87\x93",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa0"		=> "\xe8\x94\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa1"		=> "\xe8\xb4\x88",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa2"		=> "\xe9\x80\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa3"		=> "\xe4\xbf\x83",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa4"		=> "\xe5\x81\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa5"		=> "\xe5\x89\x87",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa6"		=> "\xe5\x8d\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa7"		=> "\xe6\x81\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa8"		=> "\xe6\x8d\x89",	//CJK UNIFIED IDEOGRAPH
		"\x91\xa9"		=> "\xe6\x9d\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x91\xaa"		=> "\xe6\xb8\xac",	//CJK UNIFIED IDEOGRAPH
		"\x91\xab"		=> "\xe8\xb6\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x91\xac"		=> "\xe9\x80\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x91\xad"		=> "\xe4\xbf\x97",	//CJK UNIFIED IDEOGRAPH
		"\x91\xae"		=> "\xe5\xb1\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x91\xaf"		=> "\xe8\xb3\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb0"		=> "\xe6\x97\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb1"		=> "\xe7\xb6\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb2"		=> "\xe5\x8d\x92",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb3"		=> "\xe8\xa2\x96",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb4"		=> "\xe5\x85\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb5"		=> "\xe6\x8f\x83",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb6"		=> "\xe5\xad\x98",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb7"		=> "\xe5\xad\xab",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb8"		=> "\xe5\xb0\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x91\xb9"		=> "\xe6\x90\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x91\xba"		=> "\xe6\x9d\x91",	//CJK UNIFIED IDEOGRAPH
		"\x91\xbb"		=> "\xe9\x81\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x91\xbc"		=> "\xe4\xbb\x96",	//CJK UNIFIED IDEOGRAPH
		"\x91\xbd"		=> "\xe5\xa4\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x91\xbe"		=> "\xe5\xa4\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x91\xbf"		=> "\xe6\xb1\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc0"		=> "\xe8\xa9\x91",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc1"		=> "\xe5\x94\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc2"		=> "\xe5\xa0\x95",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc3"		=> "\xe5\xa6\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc4"		=> "\xe6\x83\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc5"		=> "\xe6\x89\x93",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc6"		=> "\xe6\x9f\x81",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc7"		=> "\xe8\x88\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc8"		=> "\xe6\xa5\x95",	//CJK UNIFIED IDEOGRAPH
		"\x91\xc9"		=> "\xe9\x99\x80",	//CJK UNIFIED IDEOGRAPH
		"\x91\xca"		=> "\xe9\xa7\x84",	//CJK UNIFIED IDEOGRAPH
		"\x91\xcb"		=> "\xe9\xa8\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x91\xcc"		=> "\xe4\xbd\x93",	//CJK UNIFIED IDEOGRAPH
		"\x91\xcd"		=> "\xe5\xa0\x86",	//CJK UNIFIED IDEOGRAPH
		"\x91\xce"		=> "\xe5\xaf\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x91\xcf"		=> "\xe8\x80\x90",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd0"		=> "\xe5\xb2\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd1"		=> "\xe5\xb8\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd2"		=> "\xe5\xbe\x85",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd3"		=> "\xe6\x80\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd4"		=> "\xe6\x85\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd5"		=> "\xe6\x88\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd6"		=> "\xe6\x9b\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd7"		=> "\xe6\xb3\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd8"		=> "\xe6\xbb\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x91\xd9"		=> "\xe8\x83\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x91\xda"		=> "\xe8\x85\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x91\xdb"		=> "\xe8\x8b\x94",	//CJK UNIFIED IDEOGRAPH
		"\x91\xdc"		=> "\xe8\xa2\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x91\xdd"		=> "\xe8\xb2\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x91\xde"		=> "\xe9\x80\x80",	//CJK UNIFIED IDEOGRAPH
		"\x91\xdf"		=> "\xe9\x80\xae",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe0"		=> "\xe9\x9a\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe1"		=> "\xe9\xbb\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe2"		=> "\xe9\xaf\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe3"		=> "\xe4\xbb\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe4"		=> "\xe5\x8f\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe5"		=> "\xe5\xa4\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe6"		=> "\xe7\xac\xac",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe7"		=> "\xe9\x86\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe8"		=> "\xe9\xa1\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x91\xe9"		=> "\xe9\xb7\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x91\xea"		=> "\xe6\xbb\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x91\xeb"		=> "\xe7\x80\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x91\xec"		=> "\xe5\x8d\x93",	//CJK UNIFIED IDEOGRAPH
		"\x91\xed"		=> "\xe5\x95\x84",	//CJK UNIFIED IDEOGRAPH
		"\x91\xee"		=> "\xe5\xae\x85",	//CJK UNIFIED IDEOGRAPH
		"\x91\xef"		=> "\xe6\x89\x98",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf0"		=> "\xe6\x8a\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf1"		=> "\xe6\x8b\x93",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf2"		=> "\xe6\xb2\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf3"		=> "\xe6\xbf\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf4"		=> "\xe7\x90\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf5"		=> "\xe8\xa8\x97",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf6"		=> "\xe9\x90\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf7"		=> "\xe6\xbf\x81",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf8"		=> "\xe8\xab\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x91\xf9"		=> "\xe8\x8c\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x91\xfa"		=> "\xe5\x87\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x91\xfb"		=> "\xe8\x9b\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x91\xfc"		=> "\xe5\x8f\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x92\x40"		=> "\xe5\x8f\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x92\x41"		=> "\xe4\xbd\x86",	//CJK UNIFIED IDEOGRAPH
		"\x92\x42"		=> "\xe9\x81\x94",	//CJK UNIFIED IDEOGRAPH
		"\x92\x43"		=> "\xe8\xbe\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x92\x44"		=> "\xe5\xa5\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x92\x45"		=> "\xe8\x84\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x92\x46"		=> "\xe5\xb7\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x92\x47"		=> "\xe7\xab\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x92\x48"		=> "\xe8\xbe\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x92\x49"		=> "\xe6\xa3\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x92\x4a"		=> "\xe8\xb0\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x92\x4b"		=> "\xe7\x8b\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x92\x4c"		=> "\xe9\xb1\x88",	//CJK UNIFIED IDEOGRAPH
		"\x92\x4d"		=> "\xe6\xa8\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x92\x4e"		=> "\xe8\xaa\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x92\x4f"		=> "\xe4\xb8\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x92\x50"		=> "\xe5\x8d\x98",	//CJK UNIFIED IDEOGRAPH
		"\x92\x51"		=> "\xe5\x98\x86",	//CJK UNIFIED IDEOGRAPH
		"\x92\x52"		=> "\xe5\x9d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x92\x53"		=> "\xe6\x8b\x85",	//CJK UNIFIED IDEOGRAPH
		"\x92\x54"		=> "\xe6\x8e\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x92\x55"		=> "\xe6\x97\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x92\x56"		=> "\xe6\xad\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x92\x57"		=> "\xe6\xb7\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x92\x58"		=> "\xe6\xb9\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x92\x59"		=> "\xe7\x82\xad",	//CJK UNIFIED IDEOGRAPH
		"\x92\x5a"		=> "\xe7\x9f\xad",	//CJK UNIFIED IDEOGRAPH
		"\x92\x5b"		=> "\xe7\xab\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x92\x5c"		=> "\xe7\xae\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x92\x5d"		=> "\xe7\xb6\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x92\x5e"		=> "\xe8\x80\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x92\x5f"		=> "\xe8\x83\x86",	//CJK UNIFIED IDEOGRAPH
		"\x92\x60"		=> "\xe8\x9b\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x92\x61"		=> "\xe8\xaa\x95",	//CJK UNIFIED IDEOGRAPH
		"\x92\x62"		=> "\xe9\x8d\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x92\x63"		=> "\xe5\x9b\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x92\x64"		=> "\xe5\xa3\x87",	//CJK UNIFIED IDEOGRAPH
		"\x92\x65"		=> "\xe5\xbc\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x92\x66"		=> "\xe6\x96\xad",	//CJK UNIFIED IDEOGRAPH
		"\x92\x67"		=> "\xe6\x9a\x96",	//CJK UNIFIED IDEOGRAPH
		"\x92\x68"		=> "\xe6\xaa\x80",	//CJK UNIFIED IDEOGRAPH
		"\x92\x69"		=> "\xe6\xae\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x92\x6a"		=> "\xe7\x94\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x92\x6b"		=> "\xe8\xab\x87",	//CJK UNIFIED IDEOGRAPH
		"\x92\x6c"		=> "\xe5\x80\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x92\x6d"		=> "\xe7\x9f\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x92\x6e"		=> "\xe5\x9c\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x92\x6f"		=> "\xe5\xbc\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x92\x70"		=> "\xe6\x81\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x92\x71"		=> "\xe6\x99\xba",	//CJK UNIFIED IDEOGRAPH
		"\x92\x72"		=> "\xe6\xb1\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x92\x73"		=> "\xe7\x97\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x92\x74"		=> "\xe7\xa8\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x92\x75"		=> "\xe7\xbd\xae",	//CJK UNIFIED IDEOGRAPH
		"\x92\x76"		=> "\xe8\x87\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x92\x77"		=> "\xe8\x9c\x98",	//CJK UNIFIED IDEOGRAPH
		"\x92\x78"		=> "\xe9\x81\x85",	//CJK UNIFIED IDEOGRAPH
		"\x92\x79"		=> "\xe9\xa6\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x92\x7a"		=> "\xe7\xaf\x89",	//CJK UNIFIED IDEOGRAPH
		"\x92\x7b"		=> "\xe7\x95\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x92\x7c"		=> "\xe7\xab\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x92\x7d"		=> "\xe7\xad\x91",	//CJK UNIFIED IDEOGRAPH
		"\x92\x7e"		=> "\xe8\x93\x84",	//CJK UNIFIED IDEOGRAPH
		"\x92\x80"		=> "\xe9\x80\x90",	//CJK UNIFIED IDEOGRAPH
		"\x92\x81"		=> "\xe7\xa7\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x92\x82"		=> "\xe7\xaa\x92",	//CJK UNIFIED IDEOGRAPH
		"\x92\x83"		=> "\xe8\x8c\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x92\x84"		=> "\xe5\xab\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x92\x85"		=> "\xe7\x9d\x80",	//CJK UNIFIED IDEOGRAPH
		"\x92\x86"		=> "\xe4\xb8\xad",	//CJK UNIFIED IDEOGRAPH
		"\x92\x87"		=> "\xe4\xbb\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x92\x88"		=> "\xe5\xae\x99",	//CJK UNIFIED IDEOGRAPH
		"\x92\x89"		=> "\xe5\xbf\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x92\x8a"		=> "\xe6\x8a\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x92\x8b"		=> "\xe6\x98\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x92\x8c"		=> "\xe6\x9f\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x92\x8d"		=> "\xe6\xb3\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x92\x8e"		=> "\xe8\x99\xab",	//CJK UNIFIED IDEOGRAPH
		"\x92\x8f"		=> "\xe8\xa1\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x92\x90"		=> "\xe8\xa8\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x92\x91"		=> "\xe9\x85\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x92\x92"		=> "\xe9\x8b\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x92\x93"		=> "\xe9\xa7\x90",	//CJK UNIFIED IDEOGRAPH
		"\x92\x94"		=> "\xe6\xa8\x97",	//CJK UNIFIED IDEOGRAPH
		"\x92\x95"		=> "\xe7\x80\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x92\x96"		=> "\xe7\x8c\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x92\x97"		=> "\xe8\x8b\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x92\x98"		=> "\xe8\x91\x97",	//CJK UNIFIED IDEOGRAPH
		"\x92\x99"		=> "\xe8\xb2\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x92\x9a"		=> "\xe4\xb8\x81",	//CJK UNIFIED IDEOGRAPH
		"\x92\x9b"		=> "\xe5\x85\x86",	//CJK UNIFIED IDEOGRAPH
		"\x92\x9c"		=> "\xe5\x87\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x92\x9d"		=> "\xe5\x96\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x92\x9e"		=> "\xe5\xaf\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x92\x9f"		=> "\xe5\xb8\x96",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa0"		=> "\xe5\xb8\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa1"		=> "\xe5\xba\x81",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa2"		=> "\xe5\xbc\x94",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa3"		=> "\xe5\xbc\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa4"		=> "\xe5\xbd\xab",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa5"		=> "\xe5\xbe\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa6"		=> "\xe6\x87\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa7"		=> "\xe6\x8c\x91",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa8"		=> "\xe6\x9a\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x92\xa9"		=> "\xe6\x9c\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x92\xaa"		=> "\xe6\xbd\xae",	//CJK UNIFIED IDEOGRAPH
		"\x92\xab"		=> "\xe7\x89\x92",	//CJK UNIFIED IDEOGRAPH
		"\x92\xac"		=> "\xe7\x94\xba",	//CJK UNIFIED IDEOGRAPH
		"\x92\xad"		=> "\xe7\x9c\xba",	//CJK UNIFIED IDEOGRAPH
		"\x92\xae"		=> "\xe8\x81\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x92\xaf"		=> "\xe8\x84\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb0"		=> "\xe8\x85\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb1"		=> "\xe8\x9d\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb2"		=> "\xe8\xaa\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb3"		=> "\xe8\xab\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb4"		=> "\xe8\xb6\x85",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb5"		=> "\xe8\xb7\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb6"		=> "\xe9\x8a\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb7"		=> "\xe9\x95\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb8"		=> "\xe9\xa0\x82",	//CJK UNIFIED IDEOGRAPH
		"\x92\xb9"		=> "\xe9\xb3\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x92\xba"		=> "\xe5\x8b\x85",	//CJK UNIFIED IDEOGRAPH
		"\x92\xbb"		=> "\xe6\x8d\x97",	//CJK UNIFIED IDEOGRAPH
		"\x92\xbc"		=> "\xe7\x9b\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x92\xbd"		=> "\xe6\x9c\x95",	//CJK UNIFIED IDEOGRAPH
		"\x92\xbe"		=> "\xe6\xb2\x88",	//CJK UNIFIED IDEOGRAPH
		"\x92\xbf"		=> "\xe7\x8f\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc0"		=> "\xe8\xb3\x83",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc1"		=> "\xe9\x8e\xae",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc2"		=> "\xe9\x99\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc3"		=> "\xe6\xb4\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc4"		=> "\xe5\xa2\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc5"		=> "\xe6\xa4\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc6"		=> "\xe6\xa7\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc7"		=> "\xe8\xbf\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc8"		=> "\xe9\x8e\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x92\xc9"		=> "\xe7\x97\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x92\xca"		=> "\xe9\x80\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x92\xcb"		=> "\xe5\xa1\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x92\xcc"		=> "\xe6\xa0\x82",	//CJK UNIFIED IDEOGRAPH
		"\x92\xcd"		=> "\xe6\x8e\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x92\xce"		=> "\xe6\xa7\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x92\xcf"		=> "\xe4\xbd\x83",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd0"		=> "\xe6\xbc\xac",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd1"		=> "\xe6\x9f\x98",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd2"		=> "\xe8\xbe\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd3"		=> "\xe8\x94\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd4"		=> "\xe7\xb6\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd5"		=> "\xe9\x8d\x94",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd6"		=> "\xe6\xa4\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd7"		=> "\xe6\xbd\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd8"		=> "\xe5\x9d\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x92\xd9"		=> "\xe5\xa3\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x92\xda"		=> "\xe5\xac\xac",	//CJK UNIFIED IDEOGRAPH
		"\x92\xdb"		=> "\xe7\xb4\xac",	//CJK UNIFIED IDEOGRAPH
		"\x92\xdc"		=> "\xe7\x88\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x92\xdd"		=> "\xe5\x90\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x92\xde"		=> "\xe9\x87\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x92\xdf"		=> "\xe9\xb6\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe0"		=> "\xe4\xba\xad",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe1"		=> "\xe4\xbd\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe2"		=> "\xe5\x81\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe3"		=> "\xe5\x81\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe4"		=> "\xe5\x89\x83",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe5"		=> "\xe8\xb2\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe6"		=> "\xe5\x91\x88",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe7"		=> "\xe5\xa0\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe8"		=> "\xe5\xae\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x92\xe9"		=> "\xe5\xb8\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x92\xea"		=> "\xe5\xba\x95",	//CJK UNIFIED IDEOGRAPH
		"\x92\xeb"		=> "\xe5\xba\xad",	//CJK UNIFIED IDEOGRAPH
		"\x92\xec"		=> "\xe5\xbb\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x92\xed"		=> "\xe5\xbc\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x92\xee"		=> "\xe6\x82\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x92\xef"		=> "\xe6\x8a\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf0"		=> "\xe6\x8c\xba",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf1"		=> "\xe6\x8f\x90",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf2"		=> "\xe6\xa2\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf3"		=> "\xe6\xb1\x80",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf4"		=> "\xe7\xa2\x87",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf5"		=> "\xe7\xa6\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf6"		=> "\xe7\xa8\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf7"		=> "\xe7\xb7\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf8"		=> "\xe8\x89\x87",	//CJK UNIFIED IDEOGRAPH
		"\x92\xf9"		=> "\xe8\xa8\x82",	//CJK UNIFIED IDEOGRAPH
		"\x92\xfa"		=> "\xe8\xab\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x92\xfb"		=> "\xe8\xb9\x84",	//CJK UNIFIED IDEOGRAPH
		"\x92\xfc"		=> "\xe9\x80\x93",	//CJK UNIFIED IDEOGRAPH
		"\x93\x40"		=> "\xe9\x82\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x93\x41"		=> "\xe9\x84\xad",	//CJK UNIFIED IDEOGRAPH
		"\x93\x42"		=> "\xe9\x87\x98",	//CJK UNIFIED IDEOGRAPH
		"\x93\x43"		=> "\xe9\xbc\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x93\x44"		=> "\xe6\xb3\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x93\x45"		=> "\xe6\x91\x98",	//CJK UNIFIED IDEOGRAPH
		"\x93\x46"		=> "\xe6\x93\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x93\x47"		=> "\xe6\x95\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x93\x48"		=> "\xe6\xbb\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x93\x49"		=> "\xe7\x9a\x84",	//CJK UNIFIED IDEOGRAPH
		"\x93\x4a"		=> "\xe7\xac\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x93\x4b"		=> "\xe9\x81\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x93\x4c"		=> "\xe9\x8f\x91",	//CJK UNIFIED IDEOGRAPH
		"\x93\x4d"		=> "\xe6\xba\xba",	//CJK UNIFIED IDEOGRAPH
		"\x93\x4e"		=> "\xe5\x93\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x93\x4f"		=> "\xe5\xbe\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x93\x50"		=> "\xe6\x92\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x93\x51"		=> "\xe8\xbd\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x93\x52"		=> "\xe8\xbf\xad",	//CJK UNIFIED IDEOGRAPH
		"\x93\x53"		=> "\xe9\x89\x84",	//CJK UNIFIED IDEOGRAPH
		"\x93\x54"		=> "\xe5\x85\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x93\x55"		=> "\xe5\xa1\xab",	//CJK UNIFIED IDEOGRAPH
		"\x93\x56"		=> "\xe5\xa4\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x93\x57"		=> "\xe5\xb1\x95",	//CJK UNIFIED IDEOGRAPH
		"\x93\x58"		=> "\xe5\xba\x97",	//CJK UNIFIED IDEOGRAPH
		"\x93\x59"		=> "\xe6\xb7\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x93\x5a"		=> "\xe7\xba\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x93\x5b"		=> "\xe7\x94\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x93\x5c"		=> "\xe8\xb2\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x93\x5d"		=> "\xe8\xbb\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x93\x5e"		=> "\xe9\xa1\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x93\x5f"		=> "\xe7\x82\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x93\x60"		=> "\xe4\xbc\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x93\x61"		=> "\xe6\xae\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x93\x62"		=> "\xe6\xbe\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x93\x63"		=> "\xe7\x94\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x93\x64"		=> "\xe9\x9b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x93\x65"		=> "\xe5\x85\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x93\x66"		=> "\xe5\x90\x90",	//CJK UNIFIED IDEOGRAPH
		"\x93\x67"		=> "\xe5\xa0\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x93\x68"		=> "\xe5\xa1\x97",	//CJK UNIFIED IDEOGRAPH
		"\x93\x69"		=> "\xe5\xa6\xac",	//CJK UNIFIED IDEOGRAPH
		"\x93\x6a"		=> "\xe5\xb1\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x93\x6b"		=> "\xe5\xbe\x92",	//CJK UNIFIED IDEOGRAPH
		"\x93\x6c"		=> "\xe6\x96\x97",	//CJK UNIFIED IDEOGRAPH
		"\x93\x6d"		=> "\xe6\x9d\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x93\x6e"		=> "\xe6\xb8\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x93\x6f"		=> "\xe7\x99\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x93\x70"		=> "\xe8\x8f\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x93\x71"		=> "\xe8\xb3\xad",	//CJK UNIFIED IDEOGRAPH
		"\x93\x72"		=> "\xe9\x80\x94",	//CJK UNIFIED IDEOGRAPH
		"\x93\x73"		=> "\xe9\x83\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x93\x74"		=> "\xe9\x8d\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x93\x75"		=> "\xe7\xa0\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x93\x76"		=> "\xe7\xa0\xba",	//CJK UNIFIED IDEOGRAPH
		"\x93\x77"		=> "\xe5\x8a\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x93\x78"		=> "\xe5\xba\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x93\x79"		=> "\xe5\x9c\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x93\x7a"		=> "\xe5\xa5\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x93\x7b"		=> "\xe6\x80\x92",	//CJK UNIFIED IDEOGRAPH
		"\x93\x7c"		=> "\xe5\x80\x92",	//CJK UNIFIED IDEOGRAPH
		"\x93\x7d"		=> "\xe5\x85\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x93\x7e"		=> "\xe5\x86\xac",	//CJK UNIFIED IDEOGRAPH
		"\x93\x80"		=> "\xe5\x87\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x93\x81"		=> "\xe5\x88\x80",	//CJK UNIFIED IDEOGRAPH
		"\x93\x82"		=> "\xe5\x94\x90",	//CJK UNIFIED IDEOGRAPH
		"\x93\x83"		=> "\xe5\xa1\x94",	//CJK UNIFIED IDEOGRAPH
		"\x93\x84"		=> "\xe5\xa1\x98",	//CJK UNIFIED IDEOGRAPH
		"\x93\x85"		=> "\xe5\xa5\x97",	//CJK UNIFIED IDEOGRAPH
		"\x93\x86"		=> "\xe5\xae\x95",	//CJK UNIFIED IDEOGRAPH
		"\x93\x87"		=> "\xe5\xb3\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x93\x88"		=> "\xe5\xb6\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x93\x89"		=> "\xe6\x82\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x93\x8a"		=> "\xe6\x8a\x95",	//CJK UNIFIED IDEOGRAPH
		"\x93\x8b"		=> "\xe6\x90\xad",	//CJK UNIFIED IDEOGRAPH
		"\x93\x8c"		=> "\xe6\x9d\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x93\x8d"		=> "\xe6\xa1\x83",	//CJK UNIFIED IDEOGRAPH
		"\x93\x8e"		=> "\xe6\xa2\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x93\x8f"		=> "\xe6\xa3\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x93\x90"		=> "\xe7\x9b\x97",	//CJK UNIFIED IDEOGRAPH
		"\x93\x91"		=> "\xe6\xb7\x98",	//CJK UNIFIED IDEOGRAPH
		"\x93\x92"		=> "\xe6\xb9\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x93\x93"		=> "\xe6\xb6\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x93\x94"		=> "\xe7\x81\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x93\x95"		=> "\xe7\x87\x88",	//CJK UNIFIED IDEOGRAPH
		"\x93\x96"		=> "\xe5\xbd\x93",	//CJK UNIFIED IDEOGRAPH
		"\x93\x97"		=> "\xe7\x97\x98",	//CJK UNIFIED IDEOGRAPH
		"\x93\x98"		=> "\xe7\xa5\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x93\x99"		=> "\xe7\xad\x89",	//CJK UNIFIED IDEOGRAPH
		"\x93\x9a"		=> "\xe7\xad\x94",	//CJK UNIFIED IDEOGRAPH
		"\x93\x9b"		=> "\xe7\xad\x92",	//CJK UNIFIED IDEOGRAPH
		"\x93\x9c"		=> "\xe7\xb3\x96",	//CJK UNIFIED IDEOGRAPH
		"\x93\x9d"		=> "\xe7\xb5\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x93\x9e"		=> "\xe5\x88\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x93\x9f"		=> "\xe8\x91\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa0"		=> "\xe8\x95\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa1"		=> "\xe8\x97\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa2"		=> "\xe8\xa8\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa3"		=> "\xe8\xac\x84",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa4"		=> "\xe8\xb1\x86",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa5"		=> "\xe8\xb8\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa6"		=> "\xe9\x80\x83",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa7"		=> "\xe9\x80\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa8"		=> "\xe9\x90\x99",	//CJK UNIFIED IDEOGRAPH
		"\x93\xa9"		=> "\xe9\x99\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x93\xaa"		=> "\xe9\xa0\xad",	//CJK UNIFIED IDEOGRAPH
		"\x93\xab"		=> "\xe9\xa8\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x93\xac"		=> "\xe9\x97\x98",	//CJK UNIFIED IDEOGRAPH
		"\x93\xad"		=> "\xe5\x83\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x93\xae"		=> "\xe5\x8b\x95",	//CJK UNIFIED IDEOGRAPH
		"\x93\xaf"		=> "\xe5\x90\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb0"		=> "\xe5\xa0\x82",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb1"		=> "\xe5\xb0\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb2"		=> "\xe6\x86\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb3"		=> "\xe6\x92\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb4"		=> "\xe6\xb4\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb5"		=> "\xe7\x9e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb6"		=> "\xe7\xab\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb7"		=> "\xe8\x83\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb8"		=> "\xe8\x90\x84",	//CJK UNIFIED IDEOGRAPH
		"\x93\xb9"		=> "\xe9\x81\x93",	//CJK UNIFIED IDEOGRAPH
		"\x93\xba"		=> "\xe9\x8a\x85",	//CJK UNIFIED IDEOGRAPH
		"\x93\xbb"		=> "\xe5\xb3\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x93\xbc"		=> "\xe9\xb4\x87",	//CJK UNIFIED IDEOGRAPH
		"\x93\xbd"		=> "\xe5\x8c\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x93\xbe"		=> "\xe5\xbe\x97",	//CJK UNIFIED IDEOGRAPH
		"\x93\xbf"		=> "\xe5\xbe\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc0"		=> "\xe6\xb6\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc1"		=> "\xe7\x89\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc2"		=> "\xe7\x9d\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc3"		=> "\xe7\xa6\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc4"		=> "\xe7\xaf\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc5"		=> "\xe6\xaf\x92",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc6"		=> "\xe7\x8b\xac",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc7"		=> "\xe8\xaa\xad",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc8"		=> "\xe6\xa0\x83",	//CJK UNIFIED IDEOGRAPH
		"\x93\xc9"		=> "\xe6\xa9\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x93\xca"		=> "\xe5\x87\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x93\xcb"		=> "\xe7\xaa\x81",	//CJK UNIFIED IDEOGRAPH
		"\x93\xcc"		=> "\xe6\xa4\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x93\xcd"		=> "\xe5\xb1\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x93\xce"		=> "\xe9\xb3\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x93\xcf"		=> "\xe8\x8b\xab",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd0"		=> "\xe5\xaf\x85",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd1"		=> "\xe9\x85\x89",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd2"		=> "\xe7\x80\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd3"		=> "\xe5\x99\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd4"		=> "\xe5\xb1\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd5"		=> "\xe6\x83\x87",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd6"		=> "\xe6\x95\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd7"		=> "\xe6\xb2\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd8"		=> "\xe8\xb1\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x93\xd9"		=> "\xe9\x81\x81",	//CJK UNIFIED IDEOGRAPH
		"\x93\xda"		=> "\xe9\xa0\x93",	//CJK UNIFIED IDEOGRAPH
		"\x93\xdb"		=> "\xe5\x91\x91",	//CJK UNIFIED IDEOGRAPH
		"\x93\xdc"		=> "\xe6\x9b\x87",	//CJK UNIFIED IDEOGRAPH
		"\x93\xdd"		=> "\xe9\x88\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x93\xde"		=> "\xe5\xa5\x88",	//CJK UNIFIED IDEOGRAPH
		"\x93\xdf"		=> "\xe9\x82\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe0"		=> "\xe5\x86\x85",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe1"		=> "\xe4\xb9\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe2"		=> "\xe5\x87\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe3"		=> "\xe8\x96\x99",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe4"		=> "\xe8\xac\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe5"		=> "\xe7\x81\x98",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe6"		=> "\xe6\x8d\xba",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe7"		=> "\xe9\x8d\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe8"		=> "\xe6\xa5\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x93\xe9"		=> "\xe9\xa6\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x93\xea"		=> "\xe7\xb8\x84",	//CJK UNIFIED IDEOGRAPH
		"\x93\xeb"		=> "\xe7\x95\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x93\xec"		=> "\xe5\x8d\x97",	//CJK UNIFIED IDEOGRAPH
		"\x93\xed"		=> "\xe6\xa5\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x93\xee"		=> "\xe8\xbb\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x93\xef"		=> "\xe9\x9b\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf0"		=> "\xe6\xb1\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf1"		=> "\xe4\xba\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf2"		=> "\xe5\xb0\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf3"		=> "\xe5\xbc\x90",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf4"		=> "\xe8\xbf\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf5"		=> "\xe5\x8c\x82",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf6"		=> "\xe8\xb3\x91",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf7"		=> "\xe8\x82\x89",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf8"		=> "\xe8\x99\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x93\xf9"		=> "\xe5\xbb\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x93\xfa"		=> "\xe6\x97\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x93\xfb"		=> "\xe4\xb9\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x93\xfc"		=> "\xe5\x85\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x94\x40"		=> "\xe5\xa6\x82",	//CJK UNIFIED IDEOGRAPH
		"\x94\x41"		=> "\xe5\xb0\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x94\x42"		=> "\xe9\x9f\xae",	//CJK UNIFIED IDEOGRAPH
		"\x94\x43"		=> "\xe4\xbb\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x94\x44"		=> "\xe5\xa6\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x94\x45"		=> "\xe5\xbf\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x94\x46"		=> "\xe8\xaa\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x94\x47"		=> "\xe6\xbf\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x94\x48"		=> "\xe7\xa6\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x94\x49"		=> "\xe7\xa5\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x94\x4a"		=> "\xe5\xaf\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x94\x4b"		=> "\xe8\x91\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x94\x4c"		=> "\xe7\x8c\xab",	//CJK UNIFIED IDEOGRAPH
		"\x94\x4d"		=> "\xe7\x86\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x94\x4e"		=> "\xe5\xb9\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x94\x4f"		=> "\xe5\xbf\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x94\x50"		=> "\xe6\x8d\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x94\x51"		=> "\xe6\x92\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x94\x52"		=> "\xe7\x87\x83",	//CJK UNIFIED IDEOGRAPH
		"\x94\x53"		=> "\xe7\xb2\x98",	//CJK UNIFIED IDEOGRAPH
		"\x94\x54"		=> "\xe4\xb9\x83",	//CJK UNIFIED IDEOGRAPH
		"\x94\x55"		=> "\xe5\xbb\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x94\x56"		=> "\xe4\xb9\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x94\x57"		=> "\xe5\x9f\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x94\x58"		=> "\xe5\x9a\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x94\x59"		=> "\xe6\x82\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x94\x5a"		=> "\xe6\xbf\x83",	//CJK UNIFIED IDEOGRAPH
		"\x94\x5b"		=> "\xe7\xb4\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x94\x5c"		=> "\xe8\x83\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x94\x5d"		=> "\xe8\x84\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x94\x5e"		=> "\xe8\x86\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x94\x5f"		=> "\xe8\xbe\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x94\x60"		=> "\xe8\xa6\x97",	//CJK UNIFIED IDEOGRAPH
		"\x94\x61"		=> "\xe8\x9a\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x94\x62"		=> "\xe5\xb7\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x94\x63"		=> "\xe6\x8a\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x94\x64"		=> "\xe6\x92\xad",	//CJK UNIFIED IDEOGRAPH
		"\x94\x65"		=> "\xe8\xa6\x87",	//CJK UNIFIED IDEOGRAPH
		"\x94\x66"		=> "\xe6\x9d\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x94\x67"		=> "\xe6\xb3\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x94\x68"		=> "\xe6\xb4\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x94\x69"		=> "\xe7\x90\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x94\x6a"		=> "\xe7\xa0\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x94\x6b"		=> "\xe5\xa9\x86",	//CJK UNIFIED IDEOGRAPH
		"\x94\x6c"		=> "\xe7\xbd\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x94\x6d"		=> "\xe8\x8a\xad",	//CJK UNIFIED IDEOGRAPH
		"\x94\x6e"		=> "\xe9\xa6\xac",	//CJK UNIFIED IDEOGRAPH
		"\x94\x6f"		=> "\xe4\xbf\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x94\x70"		=> "\xe5\xbb\x83",	//CJK UNIFIED IDEOGRAPH
		"\x94\x71"		=> "\xe6\x8b\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x94\x72"		=> "\xe6\x8e\x92",	//CJK UNIFIED IDEOGRAPH
		"\x94\x73"		=> "\xe6\x95\x97",	//CJK UNIFIED IDEOGRAPH
		"\x94\x74"		=> "\xe6\x9d\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x94\x75"		=> "\xe7\x9b\x83",	//CJK UNIFIED IDEOGRAPH
		"\x94\x76"		=> "\xe7\x89\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x94\x77"		=> "\xe8\x83\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x94\x78"		=> "\xe8\x82\xba",	//CJK UNIFIED IDEOGRAPH
		"\x94\x79"		=> "\xe8\xbc\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x94\x7a"		=> "\xe9\x85\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x94\x7b"		=> "\xe5\x80\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x94\x7c"		=> "\xe5\x9f\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x94\x7d"		=> "\xe5\xaa\x92",	//CJK UNIFIED IDEOGRAPH
		"\x94\x7e"		=> "\xe6\xa2\x85",	//CJK UNIFIED IDEOGRAPH
		"\x94\x80"		=> "\xe6\xa5\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x94\x81"		=> "\xe7\x85\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x94\x82"		=> "\xe7\x8b\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x94\x83"		=> "\xe8\xb2\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x94\x84"		=> "\xe5\xa3\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x94\x85"		=> "\xe8\xb3\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x94\x86"		=> "\xe9\x99\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x94\x87"		=> "\xe9\x80\x99",	//CJK UNIFIED IDEOGRAPH
		"\x94\x88"		=> "\xe8\x9d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x94\x89"		=> "\xe7\xa7\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x94\x8a"		=> "\xe7\x9f\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x94\x8b"		=> "\xe8\x90\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x94\x8c"		=> "\xe4\xbc\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x94\x8d"		=> "\xe5\x89\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x94\x8e"		=> "\xe5\x8d\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x94\x8f"		=> "\xe6\x8b\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x94\x90"		=> "\xe6\x9f\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x94\x91"		=> "\xe6\xb3\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x94\x92"		=> "\xe7\x99\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x94\x93"		=> "\xe7\xae\x94",	//CJK UNIFIED IDEOGRAPH
		"\x94\x94"		=> "\xe7\xb2\x95",	//CJK UNIFIED IDEOGRAPH
		"\x94\x95"		=> "\xe8\x88\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x94\x96"		=> "\xe8\x96\x84",	//CJK UNIFIED IDEOGRAPH
		"\x94\x97"		=> "\xe8\xbf\xab",	//CJK UNIFIED IDEOGRAPH
		"\x94\x98"		=> "\xe6\x9b\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x94\x99"		=> "\xe6\xbc\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x94\x9a"		=> "\xe7\x88\x86",	//CJK UNIFIED IDEOGRAPH
		"\x94\x9b"		=> "\xe7\xb8\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x94\x9c"		=> "\xe8\x8e\xab",	//CJK UNIFIED IDEOGRAPH
		"\x94\x9d"		=> "\xe9\xa7\x81",	//CJK UNIFIED IDEOGRAPH
		"\x94\x9e"		=> "\xe9\xba\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x94\x9f"		=> "\xe5\x87\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa0"		=> "\xe7\xae\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa1"		=> "\xe7\xa1\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa2"		=> "\xe7\xae\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa3"		=> "\xe8\x82\x87",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa4"		=> "\xe7\xad\x88",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa5"		=> "\xe6\xab\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa6"		=> "\xe5\xb9\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa7"		=> "\xe8\x82\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa8"		=> "\xe7\x95\x91",	//CJK UNIFIED IDEOGRAPH
		"\x94\xa9"		=> "\xe7\x95\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x94\xaa"		=> "\xe5\x85\xab",	//CJK UNIFIED IDEOGRAPH
		"\x94\xab"		=> "\xe9\x89\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x94\xac"		=> "\xe6\xba\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x94\xad"		=> "\xe7\x99\xba",	//CJK UNIFIED IDEOGRAPH
		"\x94\xae"		=> "\xe9\x86\x97",	//CJK UNIFIED IDEOGRAPH
		"\x94\xaf"		=> "\xe9\xab\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb0"		=> "\xe4\xbc\x90",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb1"		=> "\xe7\xbd\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb2"		=> "\xe6\x8a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb3"		=> "\xe7\xad\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb4"		=> "\xe9\x96\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb5"		=> "\xe9\xb3\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb6"		=> "\xe5\x99\xba",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb7"		=> "\xe5\xa1\x99",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb8"		=> "\xe8\x9b\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x94\xb9"		=> "\xe9\x9a\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x94\xba"		=> "\xe4\xbc\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x94\xbb"		=> "\xe5\x88\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x94\xbc"		=> "\xe5\x8d\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x94\xbd"		=> "\xe5\x8f\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x94\xbe"		=> "\xe5\x8f\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x94\xbf"		=> "\xe5\xb8\x86",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc0"		=> "\xe6\x90\xac",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc1"		=> "\xe6\x96\x91",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc2"		=> "\xe6\x9d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc3"		=> "\xe6\xb0\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc4"		=> "\xe6\xb1\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc5"		=> "\xe7\x89\x88",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc6"		=> "\xe7\x8a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc7"		=> "\xe7\x8f\xad",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc8"		=> "\xe7\x95\x94",	//CJK UNIFIED IDEOGRAPH
		"\x94\xc9"		=> "\xe7\xb9\x81",	//CJK UNIFIED IDEOGRAPH
		"\x94\xca"		=> "\xe8\x88\xac",	//CJK UNIFIED IDEOGRAPH
		"\x94\xcb"		=> "\xe8\x97\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x94\xcc"		=> "\xe8\xb2\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x94\xcd"		=> "\xe7\xaf\x84",	//CJK UNIFIED IDEOGRAPH
		"\x94\xce"		=> "\xe9\x87\x86",	//CJK UNIFIED IDEOGRAPH
		"\x94\xcf"		=> "\xe7\x85\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd0"		=> "\xe9\xa0\x92",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd1"		=> "\xe9\xa3\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd2"		=> "\xe6\x8c\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd3"		=> "\xe6\x99\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd4"		=> "\xe7\x95\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd5"		=> "\xe7\x9b\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd6"		=> "\xe7\xa3\x90",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd7"		=> "\xe8\x95\x83",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd8"		=> "\xe8\x9b\xae",	//CJK UNIFIED IDEOGRAPH
		"\x94\xd9"		=> "\xe5\x8c\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x94\xda"		=> "\xe5\x8d\x91",	//CJK UNIFIED IDEOGRAPH
		"\x94\xdb"		=> "\xe5\x90\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x94\xdc"		=> "\xe5\xa6\x83",	//CJK UNIFIED IDEOGRAPH
		"\x94\xdd"		=> "\xe5\xba\x87",	//CJK UNIFIED IDEOGRAPH
		"\x94\xde"		=> "\xe5\xbd\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x94\xdf"		=> "\xe6\x82\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe0"		=> "\xe6\x89\x89",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe1"		=> "\xe6\x89\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe2"		=> "\xe6\x8a\xab",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe3"		=> "\xe6\x96\x90",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe4"		=> "\xe6\xaf\x94",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe5"		=> "\xe6\xb3\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe6"		=> "\xe7\x96\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe7"		=> "\xe7\x9a\xae",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe8"		=> "\xe7\xa2\x91",	//CJK UNIFIED IDEOGRAPH
		"\x94\xe9"		=> "\xe7\xa7\x98",	//CJK UNIFIED IDEOGRAPH
		"\x94\xea"		=> "\xe7\xb7\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x94\xeb"		=> "\xe7\xbd\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x94\xec"		=> "\xe8\x82\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x94\xed"		=> "\xe8\xa2\xab",	//CJK UNIFIED IDEOGRAPH
		"\x94\xee"		=> "\xe8\xaa\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x94\xef"		=> "\xe8\xb2\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf0"		=> "\xe9\x81\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf1"		=> "\xe9\x9d\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf2"		=> "\xe9\xa3\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf3"		=> "\xe6\xa8\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf4"		=> "\xe7\xb0\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf5"		=> "\xe5\x82\x99",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf6"		=> "\xe5\xb0\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf7"		=> "\xe5\xbe\xae",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf8"		=> "\xe6\x9e\x87",	//CJK UNIFIED IDEOGRAPH
		"\x94\xf9"		=> "\xe6\xaf\x98",	//CJK UNIFIED IDEOGRAPH
		"\x94\xfa"		=> "\xe7\x90\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x94\xfb"		=> "\xe7\x9c\x89",	//CJK UNIFIED IDEOGRAPH
		"\x94\xfc"		=> "\xe7\xbe\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x95\x40"		=> "\xe9\xbc\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x95\x41"		=> "\xe6\x9f\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x95\x42"		=> "\xe7\xa8\x97",	//CJK UNIFIED IDEOGRAPH
		"\x95\x43"		=> "\xe5\x8c\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x95\x44"		=> "\xe7\x96\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x95\x45"		=> "\xe9\xab\xad",	//CJK UNIFIED IDEOGRAPH
		"\x95\x46"		=> "\xe5\xbd\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x95\x47"		=> "\xe8\x86\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x95\x48"		=> "\xe8\x8f\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x95\x49"		=> "\xe8\x82\x98",	//CJK UNIFIED IDEOGRAPH
		"\x95\x4a"		=> "\xe5\xbc\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x95\x4b"		=> "\xe5\xbf\x85",	//CJK UNIFIED IDEOGRAPH
		"\x95\x4c"		=> "\xe7\x95\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x95\x4d"		=> "\xe7\xad\x86",	//CJK UNIFIED IDEOGRAPH
		"\x95\x4e"		=> "\xe9\x80\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x95\x4f"		=> "\xe6\xa1\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x95\x50"		=> "\xe5\xa7\xab",	//CJK UNIFIED IDEOGRAPH
		"\x95\x51"		=> "\xe5\xaa\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x95\x52"		=> "\xe7\xb4\x90",	//CJK UNIFIED IDEOGRAPH
		"\x95\x53"		=> "\xe7\x99\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x95\x54"		=> "\xe8\xac\xac",	//CJK UNIFIED IDEOGRAPH
		"\x95\x55"		=> "\xe4\xbf\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x95\x56"		=> "\xe5\xbd\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x95\x57"		=> "\xe6\xa8\x99",	//CJK UNIFIED IDEOGRAPH
		"\x95\x58"		=> "\xe6\xb0\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x95\x59"		=> "\xe6\xbc\x82",	//CJK UNIFIED IDEOGRAPH
		"\x95\x5a"		=> "\xe7\x93\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x95\x5b"		=> "\xe7\xa5\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x95\x5c"		=> "\xe8\xa1\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x95\x5d"		=> "\xe8\xa9\x95",	//CJK UNIFIED IDEOGRAPH
		"\x95\x5e"		=> "\xe8\xb1\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x95\x5f"		=> "\xe5\xbb\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x95\x60"		=> "\xe6\x8f\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x95\x61"		=> "\xe7\x97\x85",	//CJK UNIFIED IDEOGRAPH
		"\x95\x62"		=> "\xe7\xa7\x92",	//CJK UNIFIED IDEOGRAPH
		"\x95\x63"		=> "\xe8\x8b\x97",	//CJK UNIFIED IDEOGRAPH
		"\x95\x64"		=> "\xe9\x8c\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x95\x65"		=> "\xe9\x8b\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x95\x66"		=> "\xe8\x92\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x95\x67"		=> "\xe8\x9b\xad",	//CJK UNIFIED IDEOGRAPH
		"\x95\x68"		=> "\xe9\xb0\xad",	//CJK UNIFIED IDEOGRAPH
		"\x95\x69"		=> "\xe5\x93\x81",	//CJK UNIFIED IDEOGRAPH
		"\x95\x6a"		=> "\xe5\xbd\xac",	//CJK UNIFIED IDEOGRAPH
		"\x95\x6b"		=> "\xe6\x96\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x95\x6c"		=> "\xe6\xb5\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x95\x6d"		=> "\xe7\x80\x95",	//CJK UNIFIED IDEOGRAPH
		"\x95\x6e"		=> "\xe8\xb2\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x95\x6f"		=> "\xe8\xb3\x93",	//CJK UNIFIED IDEOGRAPH
		"\x95\x70"		=> "\xe9\xa0\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x95\x71"		=> "\xe6\x95\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x95\x72"		=> "\xe7\x93\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x95\x73"		=> "\xe4\xb8\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x95\x74"		=> "\xe4\xbb\x98",	//CJK UNIFIED IDEOGRAPH
		"\x95\x75"		=> "\xe5\x9f\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x95\x76"		=> "\xe5\xa4\xab",	//CJK UNIFIED IDEOGRAPH
		"\x95\x77"		=> "\xe5\xa9\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x95\x78"		=> "\xe5\xaf\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x95\x79"		=> "\xe5\x86\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x95\x7a"		=> "\xe5\xb8\x83",	//CJK UNIFIED IDEOGRAPH
		"\x95\x7b"		=> "\xe5\xba\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x95\x7c"		=> "\xe6\x80\x96",	//CJK UNIFIED IDEOGRAPH
		"\x95\x7d"		=> "\xe6\x89\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x95\x7e"		=> "\xe6\x95\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x95\x80"		=> "\xe6\x96\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x95\x81"		=> "\xe6\x99\xae",	//CJK UNIFIED IDEOGRAPH
		"\x95\x82"		=> "\xe6\xb5\xae",	//CJK UNIFIED IDEOGRAPH
		"\x95\x83"		=> "\xe7\x88\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x95\x84"		=> "\xe7\xac\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x95\x85"		=> "\xe8\x85\x90",	//CJK UNIFIED IDEOGRAPH
		"\x95\x86"		=> "\xe8\x86\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x95\x87"		=> "\xe8\x8a\x99",	//CJK UNIFIED IDEOGRAPH
		"\x95\x88"		=> "\xe8\xad\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x95\x89"		=> "\xe8\xb2\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x95\x8a"		=> "\xe8\xb3\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x95\x8b"		=> "\xe8\xb5\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x95\x8c"		=> "\xe9\x98\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x95\x8d"		=> "\xe9\x99\x84",	//CJK UNIFIED IDEOGRAPH
		"\x95\x8e"		=> "\xe4\xbe\xae",	//CJK UNIFIED IDEOGRAPH
		"\x95\x8f"		=> "\xe6\x92\xab",	//CJK UNIFIED IDEOGRAPH
		"\x95\x90"		=> "\xe6\xad\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x95\x91"		=> "\xe8\x88\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x95\x92"		=> "\xe8\x91\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x95\x93"		=> "\xe8\x95\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x95\x94"		=> "\xe9\x83\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x95\x95"		=> "\xe5\xb0\x81",	//CJK UNIFIED IDEOGRAPH
		"\x95\x96"		=> "\xe6\xa5\x93",	//CJK UNIFIED IDEOGRAPH
		"\x95\x97"		=> "\xe9\xa2\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x95\x98"		=> "\xe8\x91\xba",	//CJK UNIFIED IDEOGRAPH
		"\x95\x99"		=> "\xe8\x95\x97",	//CJK UNIFIED IDEOGRAPH
		"\x95\x9a"		=> "\xe4\xbc\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x95\x9b"		=> "\xe5\x89\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x95\x9c"		=> "\xe5\xbe\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x95\x9d"		=> "\xe5\xb9\x85",	//CJK UNIFIED IDEOGRAPH
		"\x95\x9e"		=> "\xe6\x9c\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x95\x9f"		=> "\xe7\xa6\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa0"		=> "\xe8\x85\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa1"		=> "\xe8\xa4\x87",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa2"		=> "\xe8\xa6\x86",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa3"		=> "\xe6\xb7\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa4"		=> "\xe5\xbc\x97",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa5"		=> "\xe6\x89\x95",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa6"		=> "\xe6\xb2\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa7"		=> "\xe4\xbb\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa8"		=> "\xe7\x89\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x95\xa9"		=> "\xe9\xae\x92",	//CJK UNIFIED IDEOGRAPH
		"\x95\xaa"		=> "\xe5\x88\x86",	//CJK UNIFIED IDEOGRAPH
		"\x95\xab"		=> "\xe5\x90\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x95\xac"		=> "\xe5\x99\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x95\xad"		=> "\xe5\xa2\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x95\xae"		=> "\xe6\x86\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x95\xaf"		=> "\xe6\x89\xae",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb0"		=> "\xe7\x84\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb1"		=> "\xe5\xa5\xae",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb2"		=> "\xe7\xb2\x89",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb3"		=> "\xe7\xb3\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb4"		=> "\xe7\xb4\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb5"		=> "\xe9\x9b\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb6"		=> "\xe6\x96\x87",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb7"		=> "\xe8\x81\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb8"		=> "\xe4\xb8\x99",	//CJK UNIFIED IDEOGRAPH
		"\x95\xb9"		=> "\xe4\xbd\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x95\xba"		=> "\xe5\x85\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x95\xbb"		=> "\xe5\xa1\x80",	//CJK UNIFIED IDEOGRAPH
		"\x95\xbc"		=> "\xe5\xb9\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x95\xbd"		=> "\xe5\xb9\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x95\xbe"		=> "\xe5\xbc\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x95\xbf"		=> "\xe6\x9f\x84",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc0"		=> "\xe4\xb8\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc1"		=> "\xe8\x94\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc2"		=> "\xe9\x96\x89",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc3"		=> "\xe9\x99\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc4"		=> "\xe7\xb1\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc5"		=> "\xe9\xa0\x81",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc6"		=> "\xe5\x83\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc7"		=> "\xe5\xa3\x81",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc8"		=> "\xe7\x99\x96",	//CJK UNIFIED IDEOGRAPH
		"\x95\xc9"		=> "\xe7\xa2\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x95\xca"		=> "\xe5\x88\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x95\xcb"		=> "\xe7\x9e\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x95\xcc"		=> "\xe8\x94\x91",	//CJK UNIFIED IDEOGRAPH
		"\x95\xcd"		=> "\xe7\xae\x86",	//CJK UNIFIED IDEOGRAPH
		"\x95\xce"		=> "\xe5\x81\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x95\xcf"		=> "\xe5\xa4\x89",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd0"		=> "\xe7\x89\x87",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd1"		=> "\xe7\xaf\x87",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd2"		=> "\xe7\xb7\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd3"		=> "\xe8\xbe\xba",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd4"		=> "\xe8\xbf\x94",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd5"		=> "\xe9\x81\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd6"		=> "\xe4\xbe\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd7"		=> "\xe5\x8b\x89",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd8"		=> "\xe5\xa8\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x95\xd9"		=> "\xe5\xbc\x81",	//CJK UNIFIED IDEOGRAPH
		"\x95\xda"		=> "\xe9\x9e\xad",	//CJK UNIFIED IDEOGRAPH
		"\x95\xdb"		=> "\xe4\xbf\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x95\xdc"		=> "\xe8\x88\x97",	//CJK UNIFIED IDEOGRAPH
		"\x95\xdd"		=> "\xe9\x8b\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x95\xde"		=> "\xe5\x9c\x83",	//CJK UNIFIED IDEOGRAPH
		"\x95\xdf"		=> "\xe6\x8d\x95",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe0"		=> "\xe6\xad\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe1"		=> "\xe7\x94\xab",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe2"		=> "\xe8\xa3\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe3"		=> "\xe8\xbc\x94",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe4"		=> "\xe7\xa9\x82",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe5"		=> "\xe5\x8b\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe6"		=> "\xe5\xa2\x93",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe7"		=> "\xe6\x85\x95",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe8"		=> "\xe6\x88\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x95\xe9"		=> "\xe6\x9a\xae",	//CJK UNIFIED IDEOGRAPH
		"\x95\xea"		=> "\xe6\xaf\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x95\xeb"		=> "\xe7\xb0\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x95\xec"		=> "\xe8\x8f\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x95\xed"		=> "\xe5\x80\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x95\xee"		=> "\xe4\xbf\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x95\xef"		=> "\xe5\x8c\x85",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf0"		=> "\xe5\x91\x86",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf1"		=> "\xe5\xa0\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf2"		=> "\xe5\xa5\x89",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf3"		=> "\xe5\xae\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf4"		=> "\xe5\xb3\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf5"		=> "\xe5\xb3\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf6"		=> "\xe5\xb4\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf7"		=> "\xe5\xba\x96",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf8"		=> "\xe6\x8a\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x95\xf9"		=> "\xe6\x8d\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x95\xfa"		=> "\xe6\x94\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x95\xfb"		=> "\xe6\x96\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x95\xfc"		=> "\xe6\x9c\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x96\x40"		=> "\xe6\xb3\x95",	//CJK UNIFIED IDEOGRAPH
		"\x96\x41"		=> "\xe6\xb3\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x96\x42"		=> "\xe7\x83\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x96\x43"		=> "\xe7\xa0\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x96\x44"		=> "\xe7\xb8\xab",	//CJK UNIFIED IDEOGRAPH
		"\x96\x45"		=> "\xe8\x83\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x96\x46"		=> "\xe8\x8a\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x96\x47"		=> "\xe8\x90\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x96\x48"		=> "\xe8\x93\xac",	//CJK UNIFIED IDEOGRAPH
		"\x96\x49"		=> "\xe8\x9c\x82",	//CJK UNIFIED IDEOGRAPH
		"\x96\x4a"		=> "\xe8\xa4\x92",	//CJK UNIFIED IDEOGRAPH
		"\x96\x4b"		=> "\xe8\xa8\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x96\x4c"		=> "\xe8\xb1\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x96\x4d"		=> "\xe9\x82\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x96\x4e"		=> "\xe9\x8b\x92",	//CJK UNIFIED IDEOGRAPH
		"\x96\x4f"		=> "\xe9\xa3\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x96\x50"		=> "\xe9\xb3\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x96\x51"		=> "\xe9\xb5\xac",	//CJK UNIFIED IDEOGRAPH
		"\x96\x52"		=> "\xe4\xb9\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x96\x53"		=> "\xe4\xba\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x96\x54"		=> "\xe5\x82\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x96\x55"		=> "\xe5\x89\x96",	//CJK UNIFIED IDEOGRAPH
		"\x96\x56"		=> "\xe5\x9d\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x96\x57"		=> "\xe5\xa6\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x96\x58"		=> "\xe5\xb8\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x96\x59"		=> "\xe5\xbf\x98",	//CJK UNIFIED IDEOGRAPH
		"\x96\x5a"		=> "\xe5\xbf\x99",	//CJK UNIFIED IDEOGRAPH
		"\x96\x5b"		=> "\xe6\x88\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x96\x5c"		=> "\xe6\x9a\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x96\x5d"		=> "\xe6\x9c\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x96\x5e"		=> "\xe6\x9f\x90",	//CJK UNIFIED IDEOGRAPH
		"\x96\x5f"		=> "\xe6\xa3\x92",	//CJK UNIFIED IDEOGRAPH
		"\x96\x60"		=> "\xe5\x86\x92",	//CJK UNIFIED IDEOGRAPH
		"\x96\x61"		=> "\xe7\xb4\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x96\x62"		=> "\xe8\x82\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x96\x63"		=> "\xe8\x86\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x96\x64"		=> "\xe8\xac\x80",	//CJK UNIFIED IDEOGRAPH
		"\x96\x65"		=> "\xe8\xb2\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x96\x66"		=> "\xe8\xb2\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x96\x67"		=> "\xe9\x89\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x96\x68"		=> "\xe9\x98\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x96\x69"		=> "\xe5\x90\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x96\x6a"		=> "\xe9\xa0\xac",	//CJK UNIFIED IDEOGRAPH
		"\x96\x6b"		=> "\xe5\x8c\x97",	//CJK UNIFIED IDEOGRAPH
		"\x96\x6c"		=> "\xe5\x83\x95",	//CJK UNIFIED IDEOGRAPH
		"\x96\x6d"		=> "\xe5\x8d\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x96\x6e"		=> "\xe5\xa2\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x96\x6f"		=> "\xe6\x92\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x96\x70"		=> "\xe6\x9c\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x96\x71"		=> "\xe7\x89\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x96\x72"		=> "\xe7\x9d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x96\x73"		=> "\xe7\xa9\x86",	//CJK UNIFIED IDEOGRAPH
		"\x96\x74"		=> "\xe9\x87\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x96\x75"		=> "\xe5\x8b\x83",	//CJK UNIFIED IDEOGRAPH
		"\x96\x76"		=> "\xe6\xb2\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x96\x77"		=> "\xe6\xae\x86",	//CJK UNIFIED IDEOGRAPH
		"\x96\x78"		=> "\xe5\xa0\x80",	//CJK UNIFIED IDEOGRAPH
		"\x96\x79"		=> "\xe5\xb9\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x96\x7a"		=> "\xe5\xa5\x94",	//CJK UNIFIED IDEOGRAPH
		"\x96\x7b"		=> "\xe6\x9c\xac",	//CJK UNIFIED IDEOGRAPH
		"\x96\x7c"		=> "\xe7\xbf\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x96\x7d"		=> "\xe5\x87\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x96\x7e"		=> "\xe7\x9b\x86",	//CJK UNIFIED IDEOGRAPH
		"\x96\x80"		=> "\xe6\x91\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x96\x81"		=> "\xe7\xa3\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x96\x82"		=> "\xe9\xad\x94",	//CJK UNIFIED IDEOGRAPH
		"\x96\x83"		=> "\xe9\xba\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x96\x84"		=> "\xe5\x9f\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x96\x85"		=> "\xe5\xa6\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x96\x86"		=> "\xe6\x98\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x96\x87"		=> "\xe6\x9e\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x96\x88"		=> "\xe6\xaf\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x96\x89"		=> "\xe5\x93\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x96\x8a"		=> "\xe6\xa7\x99",	//CJK UNIFIED IDEOGRAPH
		"\x96\x8b"		=> "\xe5\xb9\x95",	//CJK UNIFIED IDEOGRAPH
		"\x96\x8c"		=> "\xe8\x86\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x96\x8d"		=> "\xe6\x9e\x95",	//CJK UNIFIED IDEOGRAPH
		"\x96\x8e"		=> "\xe9\xae\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x96\x8f"		=> "\xe6\x9f\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x96\x90"		=> "\xe9\xb1\x92",	//CJK UNIFIED IDEOGRAPH
		"\x96\x91"		=> "\xe6\xa1\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x96\x92"		=> "\xe4\xba\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x96\x93"		=> "\xe4\xbf\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x96\x94"		=> "\xe5\x8f\x88",	//CJK UNIFIED IDEOGRAPH
		"\x96\x95"		=> "\xe6\x8a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x96\x96"		=> "\xe6\x9c\xab",	//CJK UNIFIED IDEOGRAPH
		"\x96\x97"		=> "\xe6\xb2\xab",	//CJK UNIFIED IDEOGRAPH
		"\x96\x98"		=> "\xe8\xbf\x84",	//CJK UNIFIED IDEOGRAPH
		"\x96\x99"		=> "\xe4\xbe\xad",	//CJK UNIFIED IDEOGRAPH
		"\x96\x9a"		=> "\xe7\xb9\xad",	//CJK UNIFIED IDEOGRAPH
		"\x96\x9b"		=> "\xe9\xba\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x96\x9c"		=> "\xe4\xb8\x87",	//CJK UNIFIED IDEOGRAPH
		"\x96\x9d"		=> "\xe6\x85\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x96\x9e"		=> "\xe6\xba\x80",	//CJK UNIFIED IDEOGRAPH
		"\x96\x9f"		=> "\xe6\xbc\xab",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa0"		=> "\xe8\x94\x93",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa1"		=> "\xe5\x91\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa2"		=> "\xe6\x9c\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa3"		=> "\xe9\xad\x85",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa4"		=> "\xe5\xb7\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa5"		=> "\xe7\xae\x95",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa6"		=> "\xe5\xb2\xac",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa7"		=> "\xe5\xaf\x86",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa8"		=> "\xe8\x9c\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x96\xa9"		=> "\xe6\xb9\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x96\xaa"		=> "\xe8\x93\x91",	//CJK UNIFIED IDEOGRAPH
		"\x96\xab"		=> "\xe7\xa8\x94",	//CJK UNIFIED IDEOGRAPH
		"\x96\xac"		=> "\xe8\x84\x88",	//CJK UNIFIED IDEOGRAPH
		"\x96\xad"		=> "\xe5\xa6\x99",	//CJK UNIFIED IDEOGRAPH
		"\x96\xae"		=> "\xe7\xb2\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x96\xaf"		=> "\xe6\xb0\x91",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb0"		=> "\xe7\x9c\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb1"		=> "\xe5\x8b\x99",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb2"		=> "\xe5\xa4\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb3"		=> "\xe7\x84\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb4"		=> "\xe7\x89\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb5"		=> "\xe7\x9f\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb6"		=> "\xe9\x9c\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb7"		=> "\xe9\xb5\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb8"		=> "\xe6\xa4\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x96\xb9"		=> "\xe5\xa9\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x96\xba"		=> "\xe5\xa8\x98",	//CJK UNIFIED IDEOGRAPH
		"\x96\xbb"		=> "\xe5\x86\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x96\xbc"		=> "\xe5\x90\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x96\xbd"		=> "\xe5\x91\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x96\xbe"		=> "\xe6\x98\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x96\xbf"		=> "\xe7\x9b\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc0"		=> "\xe8\xbf\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc1"		=> "\xe9\x8a\x98",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc2"		=> "\xe9\xb3\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc3"		=> "\xe5\xa7\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc4"		=> "\xe7\x89\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc5"		=> "\xe6\xbb\x85",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc6"		=> "\xe5\x85\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc7"		=> "\xe6\xa3\x89",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc8"		=> "\xe7\xb6\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x96\xc9"		=> "\xe7\xb7\xac",	//CJK UNIFIED IDEOGRAPH
		"\x96\xca"		=> "\xe9\x9d\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x96\xcb"		=> "\xe9\xba\xba",	//CJK UNIFIED IDEOGRAPH
		"\x96\xcc"		=> "\xe6\x91\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x96\xcd"		=> "\xe6\xa8\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x96\xce"		=> "\xe8\x8c\x82",	//CJK UNIFIED IDEOGRAPH
		"\x96\xcf"		=> "\xe5\xa6\x84",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd0"		=> "\xe5\xad\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd1"		=> "\xe6\xaf\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd2"		=> "\xe7\x8c\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd3"		=> "\xe7\x9b\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd4"		=> "\xe7\xb6\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd5"		=> "\xe8\x80\x97",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd6"		=> "\xe8\x92\x99",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd7"		=> "\xe5\x84\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd8"		=> "\xe6\x9c\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x96\xd9"		=> "\xe9\xbb\x99",	//CJK UNIFIED IDEOGRAPH
		"\x96\xda"		=> "\xe7\x9b\xae",	//CJK UNIFIED IDEOGRAPH
		"\x96\xdb"		=> "\xe6\x9d\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x96\xdc"		=> "\xe5\x8b\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x96\xdd"		=> "\xe9\xa4\x85",	//CJK UNIFIED IDEOGRAPH
		"\x96\xde"		=> "\xe5\xb0\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x96\xdf"		=> "\xe6\x88\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe0"		=> "\xe7\xb1\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe1"		=> "\xe8\xb2\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe2"		=> "\xe5\x95\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe3"		=> "\xe6\x82\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe4"		=> "\xe7\xb4\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe5"		=> "\xe9\x96\x80",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe6"		=> "\xe5\x8c\x81",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe7"		=> "\xe4\xb9\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe8"		=> "\xe5\x86\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x96\xe9"		=> "\xe5\xa4\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x96\xea"		=> "\xe7\x88\xba",	//CJK UNIFIED IDEOGRAPH
		"\x96\xeb"		=> "\xe8\x80\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x96\xec"		=> "\xe9\x87\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x96\xed"		=> "\xe5\xbc\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x96\xee"		=> "\xe7\x9f\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x96\xef"		=> "\xe5\x8e\x84",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf0"		=> "\xe5\xbd\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf1"		=> "\xe7\xb4\x84",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf2"		=> "\xe8\x96\xac",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf3"		=> "\xe8\xa8\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf4"		=> "\xe8\xba\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf5"		=> "\xe9\x9d\x96",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf6"		=> "\xe6\x9f\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf7"		=> "\xe8\x96\xae",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf8"		=> "\xe9\x91\x93",	//CJK UNIFIED IDEOGRAPH
		"\x96\xf9"		=> "\xe6\x84\x89",	//CJK UNIFIED IDEOGRAPH
		"\x96\xfa"		=> "\xe6\x84\x88",	//CJK UNIFIED IDEOGRAPH
		"\x96\xfb"		=> "\xe6\xb2\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x96\xfc"		=> "\xe7\x99\x92",	//CJK UNIFIED IDEOGRAPH
		"\x97\x40"		=> "\xe8\xab\xad",	//CJK UNIFIED IDEOGRAPH
		"\x97\x41"		=> "\xe8\xbc\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x97\x42"		=> "\xe5\x94\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x97\x43"		=> "\xe4\xbd\x91",	//CJK UNIFIED IDEOGRAPH
		"\x97\x44"		=> "\xe5\x84\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x97\x45"		=> "\xe5\x8b\x87",	//CJK UNIFIED IDEOGRAPH
		"\x97\x46"		=> "\xe5\x8f\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x97\x47"		=> "\xe5\xae\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x97\x48"		=> "\xe5\xb9\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x97\x49"		=> "\xe6\x82\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x97\x4a"		=> "\xe6\x86\x82",	//CJK UNIFIED IDEOGRAPH
		"\x97\x4b"		=> "\xe6\x8f\x96",	//CJK UNIFIED IDEOGRAPH
		"\x97\x4c"		=> "\xe6\x9c\x89",	//CJK UNIFIED IDEOGRAPH
		"\x97\x4d"		=> "\xe6\x9f\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x97\x4e"		=> "\xe6\xb9\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x97\x4f"		=> "\xe6\xb6\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x97\x50"		=> "\xe7\x8c\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x97\x51"		=> "\xe7\x8c\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x97\x52"		=> "\xe7\x94\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x97\x53"		=> "\xe7\xa5\x90",	//CJK UNIFIED IDEOGRAPH
		"\x97\x54"		=> "\xe8\xa3\x95",	//CJK UNIFIED IDEOGRAPH
		"\x97\x55"		=> "\xe8\xaa\x98",	//CJK UNIFIED IDEOGRAPH
		"\x97\x56"		=> "\xe9\x81\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x97\x57"		=> "\xe9\x82\x91",	//CJK UNIFIED IDEOGRAPH
		"\x97\x58"		=> "\xe9\x83\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x97\x59"		=> "\xe9\x9b\x84",	//CJK UNIFIED IDEOGRAPH
		"\x97\x5a"		=> "\xe8\x9e\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x97\x5b"		=> "\xe5\xa4\x95",	//CJK UNIFIED IDEOGRAPH
		"\x97\x5c"		=> "\xe4\xba\x88",	//CJK UNIFIED IDEOGRAPH
		"\x97\x5d"		=> "\xe4\xbd\x99",	//CJK UNIFIED IDEOGRAPH
		"\x97\x5e"		=> "\xe4\xb8\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x97\x5f"		=> "\xe8\xaa\x89",	//CJK UNIFIED IDEOGRAPH
		"\x97\x60"		=> "\xe8\xbc\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x97\x61"		=> "\xe9\xa0\x90",	//CJK UNIFIED IDEOGRAPH
		"\x97\x62"		=> "\xe5\x82\xad",	//CJK UNIFIED IDEOGRAPH
		"\x97\x63"		=> "\xe5\xb9\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x97\x64"		=> "\xe5\xa6\x96",	//CJK UNIFIED IDEOGRAPH
		"\x97\x65"		=> "\xe5\xae\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x97\x66"		=> "\xe5\xba\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x97\x67"		=> "\xe6\x8f\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x97\x68"		=> "\xe6\x8f\xba",	//CJK UNIFIED IDEOGRAPH
		"\x97\x69"		=> "\xe6\x93\x81",	//CJK UNIFIED IDEOGRAPH
		"\x97\x6a"		=> "\xe6\x9b\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x97\x6b"		=> "\xe6\xa5\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x97\x6c"		=> "\xe6\xa7\x98",	//CJK UNIFIED IDEOGRAPH
		"\x97\x6d"		=> "\xe6\xb4\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x97\x6e"		=> "\xe6\xba\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x97\x6f"		=> "\xe7\x86\x94",	//CJK UNIFIED IDEOGRAPH
		"\x97\x70"		=> "\xe7\x94\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x97\x71"		=> "\xe7\xaa\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x97\x72"		=> "\xe7\xbe\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x97\x73"		=> "\xe8\x80\x80",	//CJK UNIFIED IDEOGRAPH
		"\x97\x74"		=> "\xe8\x91\x89",	//CJK UNIFIED IDEOGRAPH
		"\x97\x75"		=> "\xe8\x93\x89",	//CJK UNIFIED IDEOGRAPH
		"\x97\x76"		=> "\xe8\xa6\x81",	//CJK UNIFIED IDEOGRAPH
		"\x97\x77"		=> "\xe8\xac\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x97\x78"		=> "\xe8\xb8\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x97\x79"		=> "\xe9\x81\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x97\x7a"		=> "\xe9\x99\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x97\x7b"		=> "\xe9\xa4\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x97\x7c"		=> "\xe6\x85\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x97\x7d"		=> "\xe6\x8a\x91",	//CJK UNIFIED IDEOGRAPH
		"\x97\x7e"		=> "\xe6\xac\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x97\x80"		=> "\xe6\xb2\x83",	//CJK UNIFIED IDEOGRAPH
		"\x97\x81"		=> "\xe6\xb5\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x97\x82"		=> "\xe7\xbf\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x97\x83"		=> "\xe7\xbf\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x97\x84"		=> "\xe6\xb7\x80",	//CJK UNIFIED IDEOGRAPH
		"\x97\x85"		=> "\xe7\xbe\x85",	//CJK UNIFIED IDEOGRAPH
		"\x97\x86"		=> "\xe8\x9e\xba",	//CJK UNIFIED IDEOGRAPH
		"\x97\x87"		=> "\xe8\xa3\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x97\x88"		=> "\xe6\x9d\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x97\x89"		=> "\xe8\x8e\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x97\x8a"		=> "\xe9\xa0\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x97\x8b"		=> "\xe9\x9b\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x97\x8c"		=> "\xe6\xb4\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x97\x8d"		=> "\xe7\xb5\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x97\x8e"		=> "\xe8\x90\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x97\x8f"		=> "\xe9\x85\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x97\x90"		=> "\xe4\xb9\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x97\x91"		=> "\xe5\x8d\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x97\x92"		=> "\xe5\xb5\x90",	//CJK UNIFIED IDEOGRAPH
		"\x97\x93"		=> "\xe6\xac\x84",	//CJK UNIFIED IDEOGRAPH
		"\x97\x94"		=> "\xe6\xbf\xab",	//CJK UNIFIED IDEOGRAPH
		"\x97\x95"		=> "\xe8\x97\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x97\x96"		=> "\xe8\x98\xad",	//CJK UNIFIED IDEOGRAPH
		"\x97\x97"		=> "\xe8\xa6\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x97\x98"		=> "\xe5\x88\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x97\x99"		=> "\xe5\x90\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x97\x9a"		=> "\xe5\xb1\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x97\x9b"		=> "\xe6\x9d\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x97\x9c"		=> "\xe6\xa2\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x97\x9d"		=> "\xe7\x90\x86",	//CJK UNIFIED IDEOGRAPH
		"\x97\x9e"		=> "\xe7\x92\x83",	//CJK UNIFIED IDEOGRAPH
		"\x97\x9f"		=> "\xe7\x97\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa0"		=> "\xe8\xa3\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa1"		=> "\xe8\xa3\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa2"		=> "\xe9\x87\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa3"		=> "\xe9\x9b\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa4"		=> "\xe9\x99\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa5"		=> "\xe5\xbe\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa6"		=> "\xe7\x8e\x87",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa7"		=> "\xe7\xab\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa8"		=> "\xe8\x91\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x97\xa9"		=> "\xe6\x8e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x97\xaa"		=> "\xe7\x95\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x97\xab"		=> "\xe5\x8a\x89",	//CJK UNIFIED IDEOGRAPH
		"\x97\xac"		=> "\xe6\xb5\x81",	//CJK UNIFIED IDEOGRAPH
		"\x97\xad"		=> "\xe6\xba\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x97\xae"		=> "\xe7\x90\x89",	//CJK UNIFIED IDEOGRAPH
		"\x97\xaf"		=> "\xe7\x95\x99",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb0"		=> "\xe7\xa1\xab",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb1"		=> "\xe7\xb2\x92",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb2"		=> "\xe9\x9a\x86",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb3"		=> "\xe7\xab\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb4"		=> "\xe9\xbe\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb5"		=> "\xe4\xbe\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb6"		=> "\xe6\x85\xae",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb7"		=> "\xe6\x97\x85",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb8"		=> "\xe8\x99\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x97\xb9"		=> "\xe4\xba\x86",	//CJK UNIFIED IDEOGRAPH
		"\x97\xba"		=> "\xe4\xba\xae",	//CJK UNIFIED IDEOGRAPH
		"\x97\xbb"		=> "\xe5\x83\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x97\xbc"		=> "\xe4\xb8\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x97\xbd"		=> "\xe5\x87\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x97\xbe"		=> "\xe5\xaf\xae",	//CJK UNIFIED IDEOGRAPH
		"\x97\xbf"		=> "\xe6\x96\x99",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc0"		=> "\xe6\xa2\x81",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc1"		=> "\xe6\xb6\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc2"		=> "\xe7\x8c\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc3"		=> "\xe7\x99\x82",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc4"		=> "\xe7\x9e\xad",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc5"		=> "\xe7\xa8\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc6"		=> "\xe7\xb3\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc7"		=> "\xe8\x89\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc8"		=> "\xe8\xab\x92",	//CJK UNIFIED IDEOGRAPH
		"\x97\xc9"		=> "\xe9\x81\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x97\xca"		=> "\xe9\x87\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x97\xcb"		=> "\xe9\x99\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x97\xcc"		=> "\xe9\xa0\x98",	//CJK UNIFIED IDEOGRAPH
		"\x97\xcd"		=> "\xe5\x8a\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x97\xce"		=> "\xe7\xb7\x91",	//CJK UNIFIED IDEOGRAPH
		"\x97\xcf"		=> "\xe5\x80\xab",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd0"		=> "\xe5\x8e\x98",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd1"		=> "\xe6\x9e\x97",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd2"		=> "\xe6\xb7\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd3"		=> "\xe7\x87\x90",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd4"		=> "\xe7\x90\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd5"		=> "\xe8\x87\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd6"		=> "\xe8\xbc\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd7"		=> "\xe9\x9a\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd8"		=> "\xe9\xb1\x97",	//CJK UNIFIED IDEOGRAPH
		"\x97\xd9"		=> "\xe9\xba\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x97\xda"		=> "\xe7\x91\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x97\xdb"		=> "\xe5\xa1\x81",	//CJK UNIFIED IDEOGRAPH
		"\x97\xdc"		=> "\xe6\xb6\x99",	//CJK UNIFIED IDEOGRAPH
		"\x97\xdd"		=> "\xe7\xb4\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x97\xde"		=> "\xe9\xa1\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x97\xdf"		=> "\xe4\xbb\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe0"		=> "\xe4\xbc\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe1"		=> "\xe4\xbe\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe2"		=> "\xe5\x86\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe3"		=> "\xe5\x8a\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe4"		=> "\xe5\xb6\xba",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe5"		=> "\xe6\x80\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe6"		=> "\xe7\x8e\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe7"		=> "\xe7\xa4\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe8"		=> "\xe8\x8b\x93",	//CJK UNIFIED IDEOGRAPH
		"\x97\xe9"		=> "\xe9\x88\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x97\xea"		=> "\xe9\x9a\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x97\xeb"		=> "\xe9\x9b\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x97\xec"		=> "\xe9\x9c\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x97\xed"		=> "\xe9\xba\x97",	//CJK UNIFIED IDEOGRAPH
		"\x97\xee"		=> "\xe9\xbd\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x97\xef"		=> "\xe6\x9a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf0"		=> "\xe6\xad\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf1"		=> "\xe5\x88\x97",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf2"		=> "\xe5\x8a\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf3"		=> "\xe7\x83\x88",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf4"		=> "\xe8\xa3\x82",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf5"		=> "\xe5\xbb\x89",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf6"		=> "\xe6\x81\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf7"		=> "\xe6\x86\x90",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf8"		=> "\xe6\xbc\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x97\xf9"		=> "\xe7\x85\x89",	//CJK UNIFIED IDEOGRAPH
		"\x97\xfa"		=> "\xe7\xb0\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x97\xfb"		=> "\xe7\xb7\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x97\xfc"		=> "\xe8\x81\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x98\x40"		=> "\xe8\x93\xae",	//CJK UNIFIED IDEOGRAPH
		"\x98\x41"		=> "\xe9\x80\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x98\x42"		=> "\xe9\x8c\xac",	//CJK UNIFIED IDEOGRAPH
		"\x98\x43"		=> "\xe5\x91\x82",	//CJK UNIFIED IDEOGRAPH
		"\x98\x44"		=> "\xe9\xad\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x98\x45"		=> "\xe6\xab\x93",	//CJK UNIFIED IDEOGRAPH
		"\x98\x46"		=> "\xe7\x82\x89",	//CJK UNIFIED IDEOGRAPH
		"\x98\x47"		=> "\xe8\xb3\x82",	//CJK UNIFIED IDEOGRAPH
		"\x98\x48"		=> "\xe8\xb7\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x98\x49"		=> "\xe9\x9c\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x98\x4a"		=> "\xe5\x8a\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x98\x4b"		=> "\xe5\xa9\x81",	//CJK UNIFIED IDEOGRAPH
		"\x98\x4c"		=> "\xe5\xbb\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x98\x4d"		=> "\xe5\xbc\x84",	//CJK UNIFIED IDEOGRAPH
		"\x98\x4e"		=> "\xe6\x9c\x97",	//CJK UNIFIED IDEOGRAPH
		"\x98\x4f"		=> "\xe6\xa5\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x98\x50"		=> "\xe6\xa6\x94",	//CJK UNIFIED IDEOGRAPH
		"\x98\x51"		=> "\xe6\xb5\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x98\x52"		=> "\xe6\xbc\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x98\x53"		=> "\xe7\x89\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x98\x54"		=> "\xe7\x8b\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x98\x55"		=> "\xe7\xaf\xad",	//CJK UNIFIED IDEOGRAPH
		"\x98\x56"		=> "\xe8\x80\x81",	//CJK UNIFIED IDEOGRAPH
		"\x98\x57"		=> "\xe8\x81\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x98\x58"		=> "\xe8\x9d\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x98\x59"		=> "\xe9\x83\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x98\x5a"		=> "\xe5\x85\xad",	//CJK UNIFIED IDEOGRAPH
		"\x98\x5b"		=> "\xe9\xba\x93",	//CJK UNIFIED IDEOGRAPH
		"\x98\x5c"		=> "\xe7\xa6\x84",	//CJK UNIFIED IDEOGRAPH
		"\x98\x5d"		=> "\xe8\x82\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x98\x5e"		=> "\xe9\x8c\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x98\x5f"		=> "\xe8\xab\x96",	//CJK UNIFIED IDEOGRAPH
		"\x98\x60"		=> "\xe5\x80\xad",	//CJK UNIFIED IDEOGRAPH
		"\x98\x61"		=> "\xe5\x92\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x98\x62"		=> "\xe8\xa9\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x98\x63"		=> "\xe6\xad\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x98\x64"		=> "\xe8\xb3\x84",	//CJK UNIFIED IDEOGRAPH
		"\x98\x65"		=> "\xe8\x84\x87",	//CJK UNIFIED IDEOGRAPH
		"\x98\x66"		=> "\xe6\x83\x91",	//CJK UNIFIED IDEOGRAPH
		"\x98\x67"		=> "\xe6\x9e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x98\x68"		=> "\xe9\xb7\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x98\x69"		=> "\xe4\xba\x99",	//CJK UNIFIED IDEOGRAPH
		"\x98\x6a"		=> "\xe4\xba\x98",	//CJK UNIFIED IDEOGRAPH
		"\x98\x6b"		=> "\xe9\xb0\x90",	//CJK UNIFIED IDEOGRAPH
		"\x98\x6c"		=> "\xe8\xa9\xab",	//CJK UNIFIED IDEOGRAPH
		"\x98\x6d"		=> "\xe8\x97\x81",	//CJK UNIFIED IDEOGRAPH
		"\x98\x6e"		=> "\xe8\x95\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x98\x6f"		=> "\xe6\xa4\x80",	//CJK UNIFIED IDEOGRAPH
		"\x98\x70"		=> "\xe6\xb9\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x98\x71"		=> "\xe7\xa2\x97",	//CJK UNIFIED IDEOGRAPH
		"\x98\x72"		=> "\xe8\x85\x95",	//CJK UNIFIED IDEOGRAPH
		"\x98\x9f"		=> "\xe5\xbc\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa0"		=> "\xe4\xb8\x90",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa1"		=> "\xe4\xb8\x95",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa2"		=> "\xe4\xb8\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa3"		=> "\xe4\xb8\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa4"		=> "\xe4\xb8\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa5"		=> "\xe4\xb8\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa6"		=> "\xe4\xb8\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa7"		=> "\xe4\xb9\x82",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa8"		=> "\xe4\xb9\x96",	//CJK UNIFIED IDEOGRAPH
		"\x98\xa9"		=> "\xe4\xb9\x98",	//CJK UNIFIED IDEOGRAPH
		"\x98\xaa"		=> "\xe4\xba\x82",	//CJK UNIFIED IDEOGRAPH
		"\x98\xab"		=> "\xe4\xba\x85",	//CJK UNIFIED IDEOGRAPH
		"\x98\xac"		=> "\xe8\xb1\xab",	//CJK UNIFIED IDEOGRAPH
		"\x98\xad"		=> "\xe4\xba\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x98\xae"		=> "\xe8\x88\x92",	//CJK UNIFIED IDEOGRAPH
		"\x98\xaf"		=> "\xe5\xbc\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb0"		=> "\xe4\xba\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb1"		=> "\xe4\xba\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb2"		=> "\xe4\xba\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb3"		=> "\xe4\xba\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb4"		=> "\xe4\xba\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb5"		=> "\xe4\xba\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb6"		=> "\xe4\xba\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb7"		=> "\xe4\xba\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb8"		=> "\xe4\xbb\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x98\xb9"		=> "\xe4\xbb\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x98\xba"		=> "\xe4\xbb\x84",	//CJK UNIFIED IDEOGRAPH
		"\x98\xbb"		=> "\xe4\xbb\x86",	//CJK UNIFIED IDEOGRAPH
		"\x98\xbc"		=> "\xe4\xbb\x82",	//CJK UNIFIED IDEOGRAPH
		"\x98\xbd"		=> "\xe4\xbb\x97",	//CJK UNIFIED IDEOGRAPH
		"\x98\xbe"		=> "\xe4\xbb\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x98\xbf"		=> "\xe4\xbb\xad",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc0"		=> "\xe4\xbb\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc1"		=> "\xe4\xbb\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc2"		=> "\xe4\xbc\x89",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc3"		=> "\xe4\xbd\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc4"		=> "\xe4\xbc\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc5"		=> "\xe4\xbd\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc6"		=> "\xe4\xbd\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc7"		=> "\xe4\xbd\x97",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc8"		=> "\xe4\xbd\x87",	//CJK UNIFIED IDEOGRAPH
		"\x98\xc9"		=> "\xe4\xbd\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x98\xca"		=> "\xe4\xbe\x88",	//CJK UNIFIED IDEOGRAPH
		"\x98\xcb"		=> "\xe4\xbe\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x98\xcc"		=> "\xe4\xbe\x98",	//CJK UNIFIED IDEOGRAPH
		"\x98\xcd"		=> "\xe4\xbd\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x98\xce"		=> "\xe4\xbd\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x98\xcf"		=> "\xe4\xbd\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd0"		=> "\xe4\xbe\x91",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd1"		=> "\xe4\xbd\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd2"		=> "\xe4\xbe\x86",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd3"		=> "\xe4\xbe\x96",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd4"		=> "\xe5\x84\x98",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd5"		=> "\xe4\xbf\x94",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd6"		=> "\xe4\xbf\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd7"		=> "\xe4\xbf\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd8"		=> "\xe4\xbf\x98",	//CJK UNIFIED IDEOGRAPH
		"\x98\xd9"		=> "\xe4\xbf\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x98\xda"		=> "\xe4\xbf\x91",	//CJK UNIFIED IDEOGRAPH
		"\x98\xdb"		=> "\xe4\xbf\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x98\xdc"		=> "\xe4\xbf\x90",	//CJK UNIFIED IDEOGRAPH
		"\x98\xdd"		=> "\xe4\xbf\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x98\xde"		=> "\xe4\xbf\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x98\xdf"		=> "\xe5\x80\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe0"		=> "\xe5\x80\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe1"		=> "\xe5\x80\x94",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe2"		=> "\xe5\x80\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe3"		=> "\xe5\x80\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe4"		=> "\xe5\x80\x85",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe5"		=> "\xe4\xbc\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe6"		=> "\xe4\xbf\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe7"		=> "\xe5\x80\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe8"		=> "\xe5\x80\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x98\xe9"		=> "\xe5\x80\xac",	//CJK UNIFIED IDEOGRAPH
		"\x98\xea"		=> "\xe4\xbf\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x98\xeb"		=> "\xe4\xbf\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x98\xec"		=> "\xe5\x80\x91",	//CJK UNIFIED IDEOGRAPH
		"\x98\xed"		=> "\xe5\x80\x86",	//CJK UNIFIED IDEOGRAPH
		"\x98\xee"		=> "\xe5\x81\x83",	//CJK UNIFIED IDEOGRAPH
		"\x98\xef"		=> "\xe5\x81\x87",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf0"		=> "\xe6\x9c\x83",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf1"		=> "\xe5\x81\x95",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf2"		=> "\xe5\x81\x90",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf3"		=> "\xe5\x81\x88",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf4"		=> "\xe5\x81\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf5"		=> "\xe5\x81\x96",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf6"		=> "\xe5\x81\xac",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf7"		=> "\xe5\x81\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf8"		=> "\xe5\x82\x80",	//CJK UNIFIED IDEOGRAPH
		"\x98\xf9"		=> "\xe5\x82\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x98\xfa"		=> "\xe5\x82\x85",	//CJK UNIFIED IDEOGRAPH
		"\x98\xfb"		=> "\xe5\x82\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x98\xfc"		=> "\xe5\x82\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x99\x40"		=> "\xe5\x83\x89",	//CJK UNIFIED IDEOGRAPH
		"\x99\x41"		=> "\xe5\x83\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x99\x42"		=> "\xe5\x82\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x99\x43"		=> "\xe5\x83\x82",	//CJK UNIFIED IDEOGRAPH
		"\x99\x44"		=> "\xe5\x83\x96",	//CJK UNIFIED IDEOGRAPH
		"\x99\x45"		=> "\xe5\x83\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x99\x46"		=> "\xe5\x83\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x99\x47"		=> "\xe5\x83\xad",	//CJK UNIFIED IDEOGRAPH
		"\x99\x48"		=> "\xe5\x83\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x99\x49"		=> "\xe5\x83\xae",	//CJK UNIFIED IDEOGRAPH
		"\x99\x4a"		=> "\xe5\x83\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x99\x4b"		=> "\xe5\x83\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x99\x4c"		=> "\xe5\x84\x89",	//CJK UNIFIED IDEOGRAPH
		"\x99\x4d"		=> "\xe5\x84\x81",	//CJK UNIFIED IDEOGRAPH
		"\x99\x4e"		=> "\xe5\x84\x82",	//CJK UNIFIED IDEOGRAPH
		"\x99\x4f"		=> "\xe5\x84\x96",	//CJK UNIFIED IDEOGRAPH
		"\x99\x50"		=> "\xe5\x84\x95",	//CJK UNIFIED IDEOGRAPH
		"\x99\x51"		=> "\xe5\x84\x94",	//CJK UNIFIED IDEOGRAPH
		"\x99\x52"		=> "\xe5\x84\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x99\x53"		=> "\xe5\x84\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x99\x54"		=> "\xe5\x84\xba",	//CJK UNIFIED IDEOGRAPH
		"\x99\x55"		=> "\xe5\x84\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x99\x56"		=> "\xe5\x84\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x99\x57"		=> "\xe5\x84\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x99\x58"		=> "\xe5\x84\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x99\x59"		=> "\xe5\x85\x80",	//CJK UNIFIED IDEOGRAPH
		"\x99\x5a"		=> "\xe5\x85\x92",	//CJK UNIFIED IDEOGRAPH
		"\x99\x5b"		=> "\xe5\x85\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x99\x5c"		=> "\xe5\x85\x94",	//CJK UNIFIED IDEOGRAPH
		"\x99\x5d"		=> "\xe5\x85\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x99\x5e"		=> "\xe7\xab\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x99\x5f"		=> "\xe5\x85\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x99\x60"		=> "\xe5\x85\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x99\x61"		=> "\xe5\x85\xae",	//CJK UNIFIED IDEOGRAPH
		"\x99\x62"		=> "\xe5\x86\x80",	//CJK UNIFIED IDEOGRAPH
		"\x99\x63"		=> "\xe5\x86\x82",	//CJK UNIFIED IDEOGRAPH
		"\x99\x64"		=> "\xe5\x9b\x98",	//CJK UNIFIED IDEOGRAPH
		"\x99\x65"		=> "\xe5\x86\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x99\x66"		=> "\xe5\x86\x89",	//CJK UNIFIED IDEOGRAPH
		"\x99\x67"		=> "\xe5\x86\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x99\x68"		=> "\xe5\x86\x91",	//CJK UNIFIED IDEOGRAPH
		"\x99\x69"		=> "\xe5\x86\x93",	//CJK UNIFIED IDEOGRAPH
		"\x99\x6a"		=> "\xe5\x86\x95",	//CJK UNIFIED IDEOGRAPH
		"\x99\x6b"		=> "\xe5\x86\x96",	//CJK UNIFIED IDEOGRAPH
		"\x99\x6c"		=> "\xe5\x86\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x99\x6d"		=> "\xe5\x86\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x99\x6e"		=> "\xe5\x86\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x99\x6f"		=> "\xe5\x86\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x99\x70"		=> "\xe5\x86\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x99\x71"		=> "\xe5\x86\xab",	//CJK UNIFIED IDEOGRAPH
		"\x99\x72"		=> "\xe5\x86\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x99\x73"		=> "\xe5\x86\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x99\x74"		=> "\xe5\x86\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x99\x75"		=> "\xe5\x86\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x99\x76"		=> "\xe5\x86\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x99\x77"		=> "\xe5\x86\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x99\x78"		=> "\xe5\x87\x85",	//CJK UNIFIED IDEOGRAPH
		"\x99\x79"		=> "\xe5\x87\x89",	//CJK UNIFIED IDEOGRAPH
		"\x99\x7a"		=> "\xe5\x87\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x99\x7b"		=> "\xe5\x87\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x99\x7c"		=> "\xe8\x99\x95",	//CJK UNIFIED IDEOGRAPH
		"\x99\x7d"		=> "\xe5\x87\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x99\x7e"		=> "\xe5\x87\xad",	//CJK UNIFIED IDEOGRAPH
		"\x99\x80"		=> "\xe5\x87\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x99\x81"		=> "\xe5\x87\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x99\x82"		=> "\xe5\x87\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x99\x83"		=> "\xe5\x88\x84",	//CJK UNIFIED IDEOGRAPH
		"\x99\x84"		=> "\xe5\x88\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x99\x85"		=> "\xe5\x88\x94",	//CJK UNIFIED IDEOGRAPH
		"\x99\x86"		=> "\xe5\x88\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x99\x87"		=> "\xe5\x88\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x99\x88"		=> "\xe5\x88\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x99\x89"		=> "\xe5\x88\xae",	//CJK UNIFIED IDEOGRAPH
		"\x99\x8a"		=> "\xe5\x88\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x99\x8b"		=> "\xe5\x88\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x99\x8c"		=> "\xe5\x89\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x99\x8d"		=> "\xe5\x89\x84",	//CJK UNIFIED IDEOGRAPH
		"\x99\x8e"		=> "\xe5\x89\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x99\x8f"		=> "\xe5\x89\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x99\x90"		=> "\xe5\x89\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x99\x91"		=> "\xe5\x89\x94",	//CJK UNIFIED IDEOGRAPH
		"\x99\x92"		=> "\xe5\x89\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x99\x93"		=> "\xe5\x89\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x99\x94"		=> "\xe5\x89\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x99\x95"		=> "\xe5\x89\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x99\x96"		=> "\xe5\x89\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x99\x97"		=> "\xe5\x89\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x99\x98"		=> "\xe5\x8a\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x99\x99"		=> "\xe5\x8a\x94",	//CJK UNIFIED IDEOGRAPH
		"\x99\x9a"		=> "\xe5\x8a\x92",	//CJK UNIFIED IDEOGRAPH
		"\x99\x9b"		=> "\xe5\x89\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x99\x9c"		=> "\xe5\x8a\x88",	//CJK UNIFIED IDEOGRAPH
		"\x99\x9d"		=> "\xe5\x8a\x91",	//CJK UNIFIED IDEOGRAPH
		"\x99\x9e"		=> "\xe8\xbe\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x99\x9f"		=> "\xe8\xbe\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa0"		=> "\xe5\x8a\xac",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa1"		=> "\xe5\x8a\xad",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa2"		=> "\xe5\x8a\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa3"		=> "\xe5\x8a\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa4"		=> "\xe5\x8b\x81",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa5"		=> "\xe5\x8b\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa6"		=> "\xe5\x8b\x97",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa7"		=> "\xe5\x8b\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa8"		=> "\xe5\x8b\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x99\xa9"		=> "\xe5\x8b\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x99\xaa"		=> "\xe9\xa3\xad",	//CJK UNIFIED IDEOGRAPH
		"\x99\xab"		=> "\xe5\x8b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x99\xac"		=> "\xe5\x8b\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x99\xad"		=> "\xe5\x8b\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x99\xae"		=> "\xe5\x8b\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x99\xaf"		=> "\xe5\x8b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb0"		=> "\xe5\x8c\x86",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb1"		=> "\xe5\x8c\x88",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb2"		=> "\xe7\x94\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb3"		=> "\xe5\x8c\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb4"		=> "\xe5\x8c\x90",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb5"		=> "\xe5\x8c\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb6"		=> "\xe5\x8c\x95",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb7"		=> "\xe5\x8c\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb8"		=> "\xe5\x8c\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x99\xb9"		=> "\xe5\x8c\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x99\xba"		=> "\xe5\x8c\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x99\xbb"		=> "\xe5\x8c\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x99\xbc"		=> "\xe5\x8c\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x99\xbd"		=> "\xe5\x8d\x80",	//CJK UNIFIED IDEOGRAPH
		"\x99\xbe"		=> "\xe5\x8d\x86",	//CJK UNIFIED IDEOGRAPH
		"\x99\xbf"		=> "\xe5\x8d\x85",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc0"		=> "\xe4\xb8\x97",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc1"		=> "\xe5\x8d\x89",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc2"		=> "\xe5\x8d\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc3"		=> "\xe5\x87\x96",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc4"		=> "\xe5\x8d\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc5"		=> "\xe5\x8d\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc6"		=> "\xe5\x8d\xae",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc7"		=> "\xe5\xa4\x98",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc8"		=> "\xe5\x8d\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x99\xc9"		=> "\xe5\x8d\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x99\xca"		=> "\xe5\x8e\x82",	//CJK UNIFIED IDEOGRAPH
		"\x99\xcb"		=> "\xe5\x8e\x96",	//CJK UNIFIED IDEOGRAPH
		"\x99\xcc"		=> "\xe5\x8e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x99\xcd"		=> "\xe5\x8e\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x99\xce"		=> "\xe5\x8e\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x99\xcf"		=> "\xe5\x8e\xae",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd0"		=> "\xe5\x8e\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd1"		=> "\xe5\x8e\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd2"		=> "\xe5\x8f\x83",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd3"		=> "\xe7\xb0\x92",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd4"		=> "\xe9\x9b\x99",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd5"		=> "\xe5\x8f\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd6"		=> "\xe6\x9b\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd7"		=> "\xe7\x87\xae",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd8"		=> "\xe5\x8f\xae",	//CJK UNIFIED IDEOGRAPH
		"\x99\xd9"		=> "\xe5\x8f\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x99\xda"		=> "\xe5\x8f\xad",	//CJK UNIFIED IDEOGRAPH
		"\x99\xdb"		=> "\xe5\x8f\xba",	//CJK UNIFIED IDEOGRAPH
		"\x99\xdc"		=> "\xe5\x90\x81",	//CJK UNIFIED IDEOGRAPH
		"\x99\xdd"		=> "\xe5\x90\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x99\xde"		=> "\xe5\x91\x80",	//CJK UNIFIED IDEOGRAPH
		"\x99\xdf"		=> "\xe5\x90\xac",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe0"		=> "\xe5\x90\xad",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe1"		=> "\xe5\x90\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe2"		=> "\xe5\x90\xae",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe3"		=> "\xe5\x90\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe4"		=> "\xe5\x90\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe5"		=> "\xe5\x90\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe6"		=> "\xe5\x91\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe7"		=> "\xe5\x92\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe8"		=> "\xe5\x91\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x99\xe9"		=> "\xe5\x92\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x99\xea"		=> "\xe5\x91\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x99\xeb"		=> "\xe5\x91\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x99\xec"		=> "\xe5\x91\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x99\xed"		=> "\xe5\x91\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x99\xee"		=> "\xe5\x92\x92",	//CJK UNIFIED IDEOGRAPH
		"\x99\xef"		=> "\xe5\x91\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf0"		=> "\xe5\x92\x80",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf1"		=> "\xe5\x91\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf2"		=> "\xe5\x92\x84",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf3"		=> "\xe5\x92\x90",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf4"		=> "\xe5\x92\x86",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf5"		=> "\xe5\x93\x87",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf6"		=> "\xe5\x92\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf7"		=> "\xe5\x92\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf8"		=> "\xe5\x92\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x99\xf9"		=> "\xe5\x92\xac",	//CJK UNIFIED IDEOGRAPH
		"\x99\xfa"		=> "\xe5\x93\x84",	//CJK UNIFIED IDEOGRAPH
		"\x99\xfb"		=> "\xe5\x93\x88",	//CJK UNIFIED IDEOGRAPH
		"\x99\xfc"		=> "\xe5\x92\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x40"		=> "\xe5\x92\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x41"		=> "\xe5\x93\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x42"		=> "\xe5\x92\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x43"		=> "\xe5\x92\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x44"		=> "\xe5\x92\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x45"		=> "\xe5\x93\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x46"		=> "\xe5\x93\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x47"		=> "\xe5\x93\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x48"		=> "\xe5\x94\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x49"		=> "\xe5\x94\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x4a"		=> "\xe5\x93\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x4b"		=> "\xe5\x93\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x4c"		=> "\xe5\x93\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x4d"		=> "\xe5\x93\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x4e"		=> "\xe5\x93\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x4f"		=> "\xe5\x94\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x50"		=> "\xe5\x95\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x51"		=> "\xe5\x95\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x52"		=> "\xe5\x95\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x53"		=> "\xe5\x94\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x54"		=> "\xe5\x95\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x55"		=> "\xe5\x95\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x56"		=> "\xe5\x95\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x57"		=> "\xe5\x95\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x58"		=> "\xe5\x94\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x59"		=> "\xe5\x94\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x5a"		=> "\xe5\x95\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x5b"		=> "\xe5\x96\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x5c"		=> "\xe5\x96\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x5d"		=> "\xe5\x92\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x5e"		=> "\xe5\x96\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x5f"		=> "\xe5\x96\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x60"		=> "\xe5\x95\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x61"		=> "\xe5\x95\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x62"		=> "\xe5\x96\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x63"		=> "\xe5\x96\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x64"		=> "\xe5\x96\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x65"		=> "\xe5\x95\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x66"		=> "\xe5\x96\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x67"		=> "\xe5\x96\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x68"		=> "\xe5\x96\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x69"		=> "\xe5\x96\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x6a"		=> "\xe5\x97\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x6b"		=> "\xe5\x97\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x6c"		=> "\xe5\x97\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x6d"		=> "\xe5\x97\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x6e"		=> "\xe5\x97\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x6f"		=> "\xe5\x97\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x70"		=> "\xe5\x97\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x71"		=> "\xe5\x98\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x72"		=> "\xe5\x97\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x73"		=> "\xe5\x98\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x74"		=> "\xe5\x97\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x75"		=> "\xe5\x97\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x76"		=> "\xe5\x98\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x77"		=> "\xe5\x97\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x78"		=> "\xe5\x99\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x79"		=> "\xe5\x99\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x7a"		=> "\xe7\x87\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x7b"		=> "\xe5\x98\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x7c"		=> "\xe5\x98\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x7d"		=> "\xe5\x98\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x7e"		=> "\xe5\x98\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x80"		=> "\xe5\x99\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x81"		=> "\xe5\x99\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x82"		=> "\xe5\x98\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x83"		=> "\xe5\x99\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x84"		=> "\xe5\x99\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x85"		=> "\xe5\x9a\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x86"		=> "\xe5\x9a\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x87"		=> "\xe5\x9a\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x88"		=> "\xe5\x9a\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x89"		=> "\xe5\x9a\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x8a"		=> "\xe5\x9a\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x8b"		=> "\xe5\x9a\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x8c"		=> "\xe5\x9a\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x8d"		=> "\xe5\x9a\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x8e"		=> "\xe5\x9a\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x8f"		=> "\xe5\x9b\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x90"		=> "\xe5\x9a\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x91"		=> "\xe5\x9b\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x92"		=> "\xe5\x9b\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x93"		=> "\xe5\x9b\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x94"		=> "\xe5\x9b\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x95"		=> "\xe5\x9b\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x96"		=> "\xe5\x9b\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x97"		=> "\xe5\x9b\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x98"		=> "\xe5\x9b\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x99"		=> "\xe5\x9b\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x9a"		=> "\xe5\x9b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x9b"		=> "\xe5\x9c\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x9c"		=> "\xe5\x9b\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x9d"		=> "\xe5\x9c\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x9e"		=> "\xe5\x9c\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9a\x9f"		=> "\xe5\x9c\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa0"		=> "\xe5\x9c\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa1"		=> "\xe5\x9c\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa2"		=> "\xe5\x9c\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa3"		=> "\xe5\x9c\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa4"		=> "\xe5\x9c\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa5"		=> "\xe5\x97\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa6"		=> "\xe5\x9c\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa7"		=> "\xe5\x9c\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa8"		=> "\xe5\x9c\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xa9"		=> "\xe5\x9c\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xaa"		=> "\xe5\x9d\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xab"		=> "\xe5\x9c\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xac"		=> "\xe5\x9d\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xad"		=> "\xe5\x9d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xae"		=> "\xe5\x9d\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xaf"		=> "\xe5\x9f\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb0"		=> "\xe5\x9e\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb1"		=> "\xe5\x9d\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb2"		=> "\xe5\x9d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb3"		=> "\xe5\x9e\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb4"		=> "\xe5\x9e\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb5"		=> "\xe5\x9e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb6"		=> "\xe5\x9e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb7"		=> "\xe5\x9e\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb8"		=> "\xe5\x9e\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xb9"		=> "\xe5\x9e\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xba"		=> "\xe5\x9f\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xbb"		=> "\xe5\x9f\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xbc"		=> "\xe5\x9f\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xbd"		=> "\xe5\x9f\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xbe"		=> "\xe5\x9f\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xbf"		=> "\xe5\xa0\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc0"		=> "\xe5\x9f\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc1"		=> "\xe5\x9f\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc2"		=> "\xe5\xa0\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc3"		=> "\xe5\xa0\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc4"		=> "\xe5\xa0\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc5"		=> "\xe5\xa1\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc6"		=> "\xe5\xa0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc7"		=> "\xe5\xa1\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc8"		=> "\xe5\xa1\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xc9"		=> "\xe5\xa1\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xca"		=> "\xe6\xaf\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xcb"		=> "\xe5\xa1\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xcc"		=> "\xe5\xa0\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xcd"		=> "\xe5\xa1\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xce"		=> "\xe5\xa2\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xcf"		=> "\xe5\xa2\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd0"		=> "\xe5\xa2\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd1"		=> "\xe5\xa2\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd2"		=> "\xe5\xa2\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd3"		=> "\xe5\xa3\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd4"		=> "\xe5\xa2\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd5"		=> "\xe5\xa2\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd6"		=> "\xe5\xa2\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd7"		=> "\xe5\xa3\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd8"		=> "\xe5\xa3\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xd9"		=> "\xe5\xa3\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xda"		=> "\xe5\xa3\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xdb"		=> "\xe5\xa3\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xdc"		=> "\xe5\xa3\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xdd"		=> "\xe5\xa3\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xde"		=> "\xe5\xa3\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xdf"		=> "\xe5\xa3\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe0"		=> "\xe5\xa3\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe1"		=> "\xe5\xa3\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe2"		=> "\xe5\xa3\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe3"		=> "\xe5\xa3\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe4"		=> "\xe5\xa3\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe5"		=> "\xe5\xa3\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe6"		=> "\xe5\xa3\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe7"		=> "\xe5\xa4\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe8"		=> "\xe5\xa4\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xe9"		=> "\xe5\xa4\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xea"		=> "\xe5\xa4\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xeb"		=> "\xe6\xa2\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xec"		=> "\xe5\xa4\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xed"		=> "\xe5\xa4\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xee"		=> "\xe5\xa4\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xef"		=> "\xe5\xa4\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf0"		=> "\xe5\xa4\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf1"		=> "\xe5\xa4\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf2"		=> "\xe7\xab\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf3"		=> "\xe5\xa5\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf4"		=> "\xe5\xa5\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf5"		=> "\xe5\xa5\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf6"		=> "\xe5\xa5\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf7"		=> "\xe5\xa5\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf8"		=> "\xe5\xa5\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xf9"		=> "\xe5\xa5\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xfa"		=> "\xe5\xa5\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xfb"		=> "\xe5\xa5\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9a\xfc"		=> "\xe5\xa5\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x40"		=> "\xe5\xa5\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x41"		=> "\xe5\xa6\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x42"		=> "\xe5\xa6\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x43"		=> "\xe4\xbd\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x44"		=> "\xe4\xbe\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x45"		=> "\xe5\xa6\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x46"		=> "\xe5\xa6\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x47"		=> "\xe5\xa7\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x48"		=> "\xe5\xa7\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x49"		=> "\xe5\xa7\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x4a"		=> "\xe5\xa6\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x4b"		=> "\xe5\xa7\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x4c"		=> "\xe5\xa7\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x4d"		=> "\xe5\xa8\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x4e"		=> "\xe5\xa8\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x4f"		=> "\xe5\xa8\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x50"		=> "\xe5\xa8\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x51"		=> "\xe5\xa8\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x52"		=> "\xe5\xa8\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x53"		=> "\xe5\xa9\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x54"		=> "\xe5\xa9\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x55"		=> "\xe5\xa9\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x56"		=> "\xe5\xa8\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x57"		=> "\xe5\xa8\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x58"		=> "\xe5\xa9\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x59"		=> "\xe5\xa9\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x5a"		=> "\xe5\xaa\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x5b"		=> "\xe5\xaa\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x5c"		=> "\xe5\xaa\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x5d"		=> "\xe5\xab\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x5e"		=> "\xe5\xab\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x5f"		=> "\xe5\xaa\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x60"		=> "\xe5\xab\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x61"		=> "\xe5\xab\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x62"		=> "\xe5\xab\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x63"		=> "\xe5\xab\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x64"		=> "\xe5\xab\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x65"		=> "\xe5\xab\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x66"		=> "\xe5\xab\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x67"		=> "\xe5\xac\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x68"		=> "\xe5\xac\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x69"		=> "\xe5\xac\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x6a"		=> "\xe5\xac\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x6b"		=> "\xe5\xab\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x6c"		=> "\xe5\xac\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x6d"		=> "\xe5\xac\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x6e"		=> "\xe5\xac\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x6f"		=> "\xe5\xad\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x70"		=> "\xe5\xad\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x71"		=> "\xe5\xad\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x72"		=> "\xe5\xad\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x73"		=> "\xe5\xad\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x74"		=> "\xe5\xad\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x75"		=> "\xe5\xad\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x76"		=> "\xe5\xad\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x77"		=> "\xe5\xad\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x78"		=> "\xe5\xad\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x79"		=> "\xe5\xad\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x7a"		=> "\xe5\xad\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x7b"		=> "\xe5\xad\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x7c"		=> "\xe6\x96\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x7d"		=> "\xe5\xad\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x7e"		=> "\xe5\xae\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x80"		=> "\xe5\xae\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x81"		=> "\xe5\xae\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x82"		=> "\xe5\xae\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x83"		=> "\xe5\xaf\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x84"		=> "\xe5\xaf\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x85"		=> "\xe5\xaf\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x86"		=> "\xe5\xaf\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x87"		=> "\xe5\xaf\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x88"		=> "\xe5\xaf\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x89"		=> "\xe5\xaf\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x8a"		=> "\xe5\xaf\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x8b"		=> "\xe5\xaf\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x8c"		=> "\xe5\xaf\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x8d"		=> "\xe5\xaf\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x8e"		=> "\xe5\xaf\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x8f"		=> "\xe5\xaf\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x90"		=> "\xe5\xaf\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x91"		=> "\xe5\xb0\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x92"		=> "\xe5\xb0\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x93"		=> "\xe5\xb0\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x94"		=> "\xe5\xb0\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x95"		=> "\xe5\xb0\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x96"		=> "\xe5\xb0\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x97"		=> "\xe5\xb0\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x98"		=> "\xe5\xb0\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x99"		=> "\xe5\xb0\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x9a"		=> "\xe5\xb0\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x9b"		=> "\xe5\xb1\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x9c"		=> "\xe5\xb1\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x9d"		=> "\xe5\xb1\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x9e"		=> "\xe5\xb1\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9b\x9f"		=> "\xe5\xb1\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa0"		=> "\xe5\xb1\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa1"		=> "\xe5\xad\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa2"		=> "\xe5\xb1\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa3"		=> "\xe5\xb1\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa4"		=> "\xe4\xb9\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa5"		=> "\xe5\xb1\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa6"		=> "\xe5\xb1\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa7"		=> "\xe5\xb2\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa8"		=> "\xe5\xb2\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xa9"		=> "\xe5\xb2\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xaa"		=> "\xe5\xa6\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xab"		=> "\xe5\xb2\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xac"		=> "\xe5\xb2\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xad"		=> "\xe5\xb2\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xae"		=> "\xe5\xb2\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xaf"		=> "\xe5\xb2\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb0"		=> "\xe5\xb3\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb1"		=> "\xe5\xb2\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb2"		=> "\xe5\xb3\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb3"		=> "\xe5\xb3\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb4"		=> "\xe5\xb3\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb5"		=> "\xe5\xb3\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb6"		=> "\xe5\xb3\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb7"		=> "\xe5\xb3\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb8"		=> "\xe5\xb6\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xb9"		=> "\xe5\xb3\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xba"		=> "\xe5\xb4\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xbb"		=> "\xe5\xb4\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xbc"		=> "\xe5\xb4\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xbd"		=> "\xe5\xb5\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xbe"		=> "\xe5\xb4\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xbf"		=> "\xe5\xb4\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc0"		=> "\xe5\xb4\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc1"		=> "\xe5\xb4\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc2"		=> "\xe5\xb4\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc3"		=> "\xe5\xb4\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc4"		=> "\xe5\xb4\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc5"		=> "\xe5\xb4\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc6"		=> "\xe5\xb5\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc7"		=> "\xe5\xb5\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc8"		=> "\xe5\xb5\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xc9"		=> "\xe5\xb5\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xca"		=> "\xe5\xb5\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xcb"		=> "\xe5\xb5\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xcc"		=> "\xe5\xb5\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xcd"		=> "\xe5\xb6\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xce"		=> "\xe5\xb6\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xcf"		=> "\xe5\xb6\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd0"		=> "\xe5\xb6\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd1"		=> "\xe5\xb6\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd2"		=> "\xe5\xb6\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd3"		=> "\xe5\xb6\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd4"		=> "\xe5\xb6\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd5"		=> "\xe5\xb6\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd6"		=> "\xe5\xb6\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd7"		=> "\xe5\xb6\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd8"		=> "\xe5\xb7\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xd9"		=> "\xe5\xb7\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xda"		=> "\xe5\xb7\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xdb"		=> "\xe5\xb7\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xdc"		=> "\xe5\xb7\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xdd"		=> "\xe5\xb7\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xde"		=> "\xe5\xb7\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xdf"		=> "\xe5\xb7\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe0"		=> "\xe5\xb7\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe1"		=> "\xe5\xb8\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe2"		=> "\xe5\xb8\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe3"		=> "\xe5\xb8\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe4"		=> "\xe5\xb8\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe5"		=> "\xe5\xb8\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe6"		=> "\xe5\xb8\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe7"		=> "\xe5\xb8\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe8"		=> "\xe5\xb9\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xe9"		=> "\xe5\xb9\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xea"		=> "\xe5\xb9\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xeb"		=> "\xe5\xb9\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xec"		=> "\xe5\xb9\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xed"		=> "\xe5\xb9\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xee"		=> "\xe5\xb9\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xef"		=> "\xe5\xb9\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf0"		=> "\xe5\xb9\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf1"		=> "\xe5\xb9\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf2"		=> "\xe5\xb9\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf3"		=> "\xe5\xb9\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf4"		=> "\xe5\xb9\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf5"		=> "\xe9\xba\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf6"		=> "\xe5\xb9\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf7"		=> "\xe5\xba\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf8"		=> "\xe5\xbb\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xf9"		=> "\xe5\xbb\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xfa"		=> "\xe5\xbb\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xfb"		=> "\xe5\xbb\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9b\xfc"		=> "\xe5\xbb\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x40"		=> "\xe5\xbb\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x41"		=> "\xe5\xbb\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x42"		=> "\xe5\xbb\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x43"		=> "\xe5\xbb\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x44"		=> "\xe5\xbb\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x45"		=> "\xe5\xbb\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x46"		=> "\xe5\xbb\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x47"		=> "\xe5\xbb\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x48"		=> "\xe5\xbb\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x49"		=> "\xe5\xbb\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x4a"		=> "\xe5\xbb\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x4b"		=> "\xe5\xbb\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x4c"		=> "\xe5\xbb\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x4d"		=> "\xe5\xbb\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x4e"		=> "\xe5\xbb\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x4f"		=> "\xe5\xbb\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x50"		=> "\xe5\xbc\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x51"		=> "\xe5\xbc\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x52"		=> "\xe5\xbd\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x53"		=> "\xe5\xbd\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x54"		=> "\xe5\xbc\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x55"		=> "\xe5\xbc\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x56"		=> "\xe5\xbc\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x57"		=> "\xe5\xbc\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x58"		=> "\xe5\xbc\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x59"		=> "\xe5\xbc\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x5a"		=> "\xe5\xbd\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x5b"		=> "\xe5\xbd\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x5c"		=> "\xe5\xbd\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x5d"		=> "\xe5\xbd\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x5e"		=> "\xe5\xbc\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x5f"		=> "\xe5\xbd\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x60"		=> "\xe5\xbd\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x61"		=> "\xe5\xbd\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x62"		=> "\xe5\xbd\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x63"		=> "\xe5\xbd\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x64"		=> "\xe5\xbd\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x65"		=> "\xe5\xbd\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x66"		=> "\xe5\xbd\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x67"		=> "\xe5\xbe\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x68"		=> "\xe5\xbe\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x69"		=> "\xe5\xbd\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x6a"		=> "\xe5\xbe\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x6b"		=> "\xe5\xbe\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x6c"		=> "\xe5\xbe\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x6d"		=> "\xe5\xbe\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x6e"		=> "\xe5\xbe\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x6f"		=> "\xe5\xbe\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x70"		=> "\xe5\xbe\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x71"		=> "\xe5\xbe\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x72"		=> "\xe5\xbe\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x73"		=> "\xe5\xbe\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x74"		=> "\xe5\xbe\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x75"		=> "\xe5\xbf\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x76"		=> "\xe5\xbf\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x77"		=> "\xe5\xbf\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x78"		=> "\xe5\xbf\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x79"		=> "\xe5\xbf\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x7a"		=> "\xe5\xbf\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x7b"		=> "\xe6\x82\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x7c"		=> "\xe5\xbf\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x7d"		=> "\xe6\x80\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x7e"		=> "\xe6\x81\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x80"		=> "\xe6\x80\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x81"		=> "\xe6\x80\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x82"		=> "\xe6\x80\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x83"		=> "\xe6\x80\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x84"		=> "\xe6\x80\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x85"		=> "\xe6\x80\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x86"		=> "\xe6\x80\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x87"		=> "\xe6\x80\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x88"		=> "\xe6\x80\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x89"		=> "\xe6\x80\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x8a"		=> "\xe6\x80\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x8b"		=> "\xe6\x81\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x8c"		=> "\xe6\x81\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x8d"		=> "\xe6\x81\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x8e"		=> "\xe6\x81\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x8f"		=> "\xe6\x81\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x90"		=> "\xe6\x81\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x91"		=> "\xe6\x81\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x92"		=> "\xe6\x81\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x93"		=> "\xe6\x81\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x94"		=> "\xe6\x81\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x95"		=> "\xe6\x81\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x96"		=> "\xe6\x81\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x97"		=> "\xe6\x81\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x98"		=> "\xe6\x81\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x99"		=> "\xe6\x81\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x9a"		=> "\xe6\x82\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x9b"		=> "\xe6\x82\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x9c"		=> "\xe6\x83\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x9d"		=> "\xe6\x82\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x9e"		=> "\xe6\x82\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\x9f"		=> "\xe6\x82\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa0"		=> "\xe6\x82\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa1"		=> "\xe6\x82\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa2"		=> "\xe6\x82\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa3"		=> "\xe6\x82\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa4"		=> "\xe6\x82\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa5"		=> "\xe6\x82\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa6"		=> "\xe6\x83\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa7"		=> "\xe6\x82\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa8"		=> "\xe6\x83\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xa9"		=> "\xe6\x83\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xaa"		=> "\xe6\x82\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xab"		=> "\xe5\xbf\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xac"		=> "\xe6\x82\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xad"		=> "\xe6\x83\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xae"		=> "\xe6\x82\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xaf"		=> "\xe6\x83\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb0"		=> "\xe6\x85\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb1"		=> "\xe6\x84\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb2"		=> "\xe6\x84\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb3"		=> "\xe6\x83\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb4"		=> "\xe6\x83\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb5"		=> "\xe6\x84\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb6"		=> "\xe6\x83\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb7"		=> "\xe6\x83\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb8"		=> "\xe6\x84\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xb9"		=> "\xe6\x84\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xba"		=> "\xe6\x83\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xbb"		=> "\xe6\x83\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xbc"		=> "\xe6\x84\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xbd"		=> "\xe6\x84\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xbe"		=> "\xe6\x85\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xbf"		=> "\xe6\x84\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc0"		=> "\xe6\x84\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc1"		=> "\xe6\x84\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc2"		=> "\xe6\x85\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc3"		=> "\xe6\x84\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc4"		=> "\xe6\x84\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc5"		=> "\xe6\x84\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc6"		=> "\xe6\x84\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc7"		=> "\xe6\x84\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc8"		=> "\xe6\x85\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xc9"		=> "\xe6\x85\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xca"		=> "\xe6\x85\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xcb"		=> "\xe6\x85\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xcc"		=> "\xe6\x85\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xcd"		=> "\xe6\x85\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xce"		=> "\xe6\x85\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xcf"		=> "\xe6\x85\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd0"		=> "\xe6\x85\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd1"		=> "\xe6\x85\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd2"		=> "\xe6\x85\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd3"		=> "\xe6\x85\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd4"		=> "\xe6\x85\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd5"		=> "\xe6\x85\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd6"		=> "\xe6\x85\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd7"		=> "\xe6\x85\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd8"		=> "\xe6\x86\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xd9"		=> "\xe6\x86\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xda"		=> "\xe6\x86\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xdb"		=> "\xe6\x86\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xdc"		=> "\xe6\x86\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xdd"		=> "\xe6\x86\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xde"		=> "\xe6\x86\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xdf"		=> "\xe6\x86\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe0"		=> "\xe6\x86\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe1"		=> "\xe6\x86\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe2"		=> "\xe6\x87\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe3"		=> "\xe6\x87\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe4"		=> "\xe6\x87\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe5"		=> "\xe6\x87\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe6"		=> "\xe6\x87\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe7"		=> "\xe6\x87\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe8"		=> "\xe6\x87\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xe9"		=> "\xe6\x86\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xea"		=> "\xe6\x87\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xeb"		=> "\xe7\xbd\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xec"		=> "\xe6\x87\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xed"		=> "\xe6\x87\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xee"		=> "\xe6\x87\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xef"		=> "\xe6\x87\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf0"		=> "\xe6\x87\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf1"		=> "\xe6\x87\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf2"		=> "\xe6\x87\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf3"		=> "\xe6\x87\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf4"		=> "\xe6\x87\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf5"		=> "\xe6\x87\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf6"		=> "\xe6\x88\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf7"		=> "\xe6\x88\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf8"		=> "\xe6\x88\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xf9"		=> "\xe6\x88\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xfa"		=> "\xe6\x88\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xfb"		=> "\xe6\x88\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9c\xfc"		=> "\xe6\x88\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x40"		=> "\xe6\x88\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x41"		=> "\xe6\x88\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x42"		=> "\xe6\x88\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x43"		=> "\xe6\x88\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x44"		=> "\xe6\x88\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x45"		=> "\xe6\x88\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x46"		=> "\xe6\x88\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x47"		=> "\xe6\x89\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x48"		=> "\xe6\x89\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x49"		=> "\xe6\x89\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x4a"		=> "\xe6\x89\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x4b"		=> "\xe6\x89\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x4c"		=> "\xe6\x89\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x4d"		=> "\xe6\x89\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x4e"		=> "\xe6\x89\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x4f"		=> "\xe6\x8a\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x50"		=> "\xe6\x8a\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x51"		=> "\xe6\x89\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x52"		=> "\xe6\x8a\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x53"		=> "\xe6\x8a\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x54"		=> "\xe6\x8a\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x55"		=> "\xe6\x8b\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x56"		=> "\xe6\x8a\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x57"		=> "\xe6\x8a\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x58"		=> "\xe6\x8b\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x59"		=> "\xe6\x8b\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x5a"		=> "\xe6\x8a\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x5b"		=> "\xe6\x8b\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x5c"		=> "\xe6\x8b\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x5d"		=> "\xe6\x8b\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x5e"		=> "\xe6\x93\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x5f"		=> "\xe6\x8b\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x60"		=> "\xe6\x8b\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x61"		=> "\xe6\x8b\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x62"		=> "\xe6\x8b\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x63"		=> "\xe6\x8b\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x64"		=> "\xe6\x8b\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x65"		=> "\xe6\x8a\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x66"		=> "\xe6\x8b\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x67"		=> "\xe6\x8c\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x68"		=> "\xe6\x8b\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x69"		=> "\xe6\x8b\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x6a"		=> "\xe6\x8c\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x6b"		=> "\xe6\x8c\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x6c"		=> "\xe6\x8c\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x6d"		=> "\xe6\x8b\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x6e"		=> "\xe6\x8b\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x6f"		=> "\xe6\x8d\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x70"		=> "\xe6\x8c\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x71"		=> "\xe6\x8d\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x72"		=> "\xe6\x90\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x73"		=> "\xe6\x8d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x74"		=> "\xe6\x8e\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x75"		=> "\xe6\x8e\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x76"		=> "\xe6\x8e\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x77"		=> "\xe6\x8e\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x78"		=> "\xe6\x8d\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x79"		=> "\xe6\x8e\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x7a"		=> "\xe6\x8e\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x7b"		=> "\xe6\x8e\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x7c"		=> "\xe6\x8e\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x7d"		=> "\xe6\x8e\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x7e"		=> "\xe6\x8d\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x80"		=> "\xe6\x8d\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x81"		=> "\xe6\x8e\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x82"		=> "\xe6\x8f\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x83"		=> "\xe6\x8f\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x84"		=> "\xe6\x8f\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x85"		=> "\xe6\x8f\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x86"		=> "\xe6\x8f\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x87"		=> "\xe6\x8f\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x88"		=> "\xe6\x8f\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x89"		=> "\xe6\x8f\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x8a"		=> "\xe6\x90\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x8b"		=> "\xe6\x90\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x8c"		=> "\xe6\x90\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x8d"		=> "\xe6\x90\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x8e"		=> "\xe6\x90\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x8f"		=> "\xe6\x90\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x90"		=> "\xe6\x94\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x91"		=> "\xe6\x90\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x92"		=> "\xe6\x90\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x93"		=> "\xe6\x90\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x94"		=> "\xe6\x91\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x95"		=> "\xe6\x91\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x96"		=> "\xe6\x91\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x97"		=> "\xe6\x91\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x98"		=> "\xe6\x94\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x99"		=> "\xe6\x92\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x9a"		=> "\xe6\x92\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x9b"		=> "\xe6\x92\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x9c"		=> "\xe6\x92\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x9d"		=> "\xe6\x92\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x9e"		=> "\xe6\x92\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9d\x9f"		=> "\xe6\x93\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa0"		=> "\xe6\x93\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa1"		=> "\xe6\x93\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa2"		=> "\xe6\x93\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa3"		=> "\xe6\x92\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa4"		=> "\xe6\x93\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa5"		=> "\xe6\x93\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa6"		=> "\xe6\x93\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa7"		=> "\xe6\x93\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa8"		=> "\xe8\x88\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xa9"		=> "\xe6\x93\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xaa"		=> "\xe6\x93\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xab"		=> "\xe6\x8a\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xac"		=> "\xe6\x93\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xad"		=> "\xe6\x93\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xae"		=> "\xe6\x94\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xaf"		=> "\xe6\x93\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb0"		=> "\xe6\x93\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb1"		=> "\xe6\x93\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb2"		=> "\xe6\x93\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb3"		=> "\xe6\x94\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb4"		=> "\xe6\x93\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb5"		=> "\xe6\x94\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb6"		=> "\xe6\x94\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb7"		=> "\xe6\x94\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb8"		=> "\xe6\x94\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xb9"		=> "\xe6\x94\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xba"		=> "\xe6\x94\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xbb"		=> "\xe6\x94\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xbc"		=> "\xe6\x94\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xbd"		=> "\xe6\x94\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xbe"		=> "\xe6\x94\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xbf"		=> "\xe6\x94\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc0"		=> "\xe7\x95\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc1"		=> "\xe6\x95\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc2"		=> "\xe6\x95\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc3"		=> "\xe6\x95\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc4"		=> "\xe6\x95\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc5"		=> "\xe6\x95\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc6"		=> "\xe6\x95\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc7"		=> "\xe6\x95\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc8"		=> "\xe6\x95\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xc9"		=> "\xe6\x95\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xca"		=> "\xe6\x96\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xcb"		=> "\xe6\x96\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xcc"		=> "\xe8\xae\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xcd"		=> "\xe6\x96\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xce"		=> "\xe6\x96\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xcf"		=> "\xe6\x96\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd0"		=> "\xe6\x96\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd1"		=> "\xe6\x97\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd2"		=> "\xe6\x97\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd3"		=> "\xe6\x97\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd4"		=> "\xe6\x97\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd5"		=> "\xe6\x97\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd6"		=> "\xe6\x97\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd7"		=> "\xe6\x97\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd8"		=> "\xe6\x97\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xd9"		=> "\xe6\x97\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xda"		=> "\xe6\x97\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xdb"		=> "\xe6\x97\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xdc"		=> "\xe6\x9d\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xdd"		=> "\xe6\x98\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xde"		=> "\xe6\x98\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xdf"		=> "\xe6\x97\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe0"		=> "\xe6\x9d\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe1"		=> "\xe6\x98\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe2"		=> "\xe6\x98\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe3"		=> "\xe6\x98\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe4"		=> "\xe6\x98\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe5"		=> "\xe6\x99\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe6"		=> "\xe6\x99\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe7"		=> "\xe6\x99\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe8"		=> "\xe6\x99\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xe9"		=> "\xe6\x99\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xea"		=> "\xe6\x99\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xeb"		=> "\xe6\x99\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xec"		=> "\xe6\x99\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xed"		=> "\xe6\x99\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xee"		=> "\xe6\x99\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xef"		=> "\xe6\x99\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf0"		=> "\xe6\x99\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf1"		=> "\xe6\x9a\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf2"		=> "\xe6\x9a\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf3"		=> "\xe6\x9a\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf4"		=> "\xe6\x9a\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf5"		=> "\xe6\x9a\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf6"		=> "\xe6\x9a\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf7"		=> "\xe6\x9a\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf8"		=> "\xe6\x9b\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xf9"		=> "\xe6\x9a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xfa"		=> "\xe6\x9b\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xfb"		=> "\xe6\x9a\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9d\xfc"		=> "\xe6\x9a\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x40"		=> "\xe6\x9b\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x41"		=> "\xe6\x9a\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x42"		=> "\xe6\x9b\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x43"		=> "\xe6\x9b\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x44"		=> "\xe6\x9b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x45"		=> "\xe6\x98\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x46"		=> "\xe6\x9b\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x47"		=> "\xe6\x9b\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x48"		=> "\xe6\x9b\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x49"		=> "\xe6\x9b\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x4a"		=> "\xe6\x9b\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x4b"		=> "\xe6\x9c\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x4c"		=> "\xe6\x9c\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x4d"		=> "\xe6\x9c\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x4e"		=> "\xe6\x9c\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x4f"		=> "\xe6\x9c\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x50"		=> "\xe9\x9c\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x51"		=> "\xe6\x9c\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x52"		=> "\xe6\x9c\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x53"		=> "\xe6\x9c\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x54"		=> "\xe6\x9d\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x55"		=> "\xe6\x9c\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x56"		=> "\xe6\x9c\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x57"		=> "\xe6\x9d\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x58"		=> "\xe6\x9d\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x59"		=> "\xe6\x9d\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x5a"		=> "\xe6\x9d\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x5b"		=> "\xe6\x9d\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x5c"		=> "\xe6\x9d\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x5d"		=> "\xe6\x9e\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x5e"		=> "\xe6\x9d\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x5f"		=> "\xe6\x9e\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x60"		=> "\xe6\x9d\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x61"		=> "\xe6\x9d\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x62"		=> "\xe6\x9e\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x63"		=> "\xe6\x9e\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x64"		=> "\xe6\x9e\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x65"		=> "\xe6\x9e\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x66"		=> "\xe6\x9e\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x67"		=> "\xe6\x9e\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x68"		=> "\xe6\x9f\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x69"		=> "\xe6\x9e\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x6a"		=> "\xe6\x9f\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x6b"		=> "\xe6\x9e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x6c"		=> "\xe6\x9f\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x6d"		=> "\xe6\x9e\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x6e"		=> "\xe6\x9f\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x6f"		=> "\xe6\x9f\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x70"		=> "\xe6\x9f\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x71"		=> "\xe6\x9f\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x72"		=> "\xe6\x9f\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x73"		=> "\xe6\x9e\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x74"		=> "\xe6\x9f\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x75"		=> "\xe6\x9f\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x76"		=> "\xe6\x9f\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x77"		=> "\xe6\xaa\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x78"		=> "\xe6\xa0\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x79"		=> "\xe6\xa1\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x7a"		=> "\xe6\xa0\xa9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x7b"		=> "\xe6\xa1\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x7c"		=> "\xe6\xa1\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x7d"		=> "\xe6\xa0\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x7e"		=> "\xe6\xa1\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x80"		=> "\xe6\xa2\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x81"		=> "\xe6\xa0\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x82"		=> "\xe6\xa1\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x83"		=> "\xe6\xa1\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x84"		=> "\xe6\xa1\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x85"		=> "\xe6\xa1\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x86"		=> "\xe6\xa2\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x87"		=> "\xe6\xa2\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x88"		=> "\xe6\xa2\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x89"		=> "\xe6\xa2\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x8a"		=> "\xe6\xa2\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x8b"		=> "\xe6\xa2\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x8c"		=> "\xe6\xa2\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x8d"		=> "\xe6\xaa\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x8e"		=> "\xe6\xa2\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x8f"		=> "\xe6\xa1\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x90"		=> "\xe6\xa2\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x91"		=> "\xe6\xa2\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x92"		=> "\xe6\xa2\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x93"		=> "\xe6\xa4\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x94"		=> "\xe6\xa2\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x95"		=> "\xe6\xa1\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x96"		=> "\xe6\xa4\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x97"		=> "\xe6\xa3\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x98"		=> "\xe6\xa4\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x99"		=> "\xe6\xa3\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x9a"		=> "\xe6\xa4\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x9b"		=> "\xe6\xa4\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x9c"		=> "\xe6\xa3\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x9d"		=> "\xe6\xa4\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x9e"		=> "\xe6\xa3\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9e\x9f"		=> "\xe6\xa3\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa0"		=> "\xe6\xa3\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa1"		=> "\xe6\xa3\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa2"		=> "\xe6\xa4\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa3"		=> "\xe6\xa4\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa4"		=> "\xe6\xa4\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa5"		=> "\xe6\xa3\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa6"		=> "\xe6\xa3\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa7"		=> "\xe6\xa4\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa8"		=> "\xe6\xa3\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xa9"		=> "\xe6\xa3\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xaa"		=> "\xe6\xa3\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xab"		=> "\xe6\xa4\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xac"		=> "\xe6\xa4\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xad"		=> "\xe6\xa4\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xae"		=> "\xe6\xa4\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xaf"		=> "\xe6\xa4\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb0"		=> "\xe6\xa3\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb1"		=> "\xe6\xa5\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb2"		=> "\xe6\xa5\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb3"		=> "\xe6\xa5\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb4"		=> "\xe6\xa5\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb5"		=> "\xe6\xa5\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb6"		=> "\xe6\xa5\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb7"		=> "\xe6\xa5\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb8"		=> "\xe6\xa5\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xb9"		=> "\xe6\xa4\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xba"		=> "\xe6\xa5\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xbb"		=> "\xe6\xa4\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xbc"		=> "\xe6\xa5\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xbd"		=> "\xe6\xa4\xb0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xbe"		=> "\xe6\xa5\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xbf"		=> "\xe6\xa5\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc0"		=> "\xe6\xa5\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc1"		=> "\xe6\xa6\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc2"		=> "\xe6\xa5\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc3"		=> "\xe6\xa6\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc4"		=> "\xe6\xa6\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc5"		=> "\xe6\xa7\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc6"		=> "\xe6\xa6\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc7"		=> "\xe6\xa7\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc8"		=> "\xe6\xa7\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xc9"		=> "\xe6\xa6\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xca"		=> "\xe6\xa7\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xcb"		=> "\xe5\xaf\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xcc"		=> "\xe6\xa7\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xcd"		=> "\xe6\xa7\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xce"		=> "\xe6\xa6\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xcf"		=> "\xe6\xa7\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd0"		=> "\xe6\xa6\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd1"		=> "\xe6\xa8\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd2"		=> "\xe6\xa6\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd3"		=> "\xe6\xa6\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd4"		=> "\xe6\xa6\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd5"		=> "\xe6\xa6\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd6"		=> "\xe6\xa6\xb4",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd7"		=> "\xe6\xa7\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd8"		=> "\xe6\xa7\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xd9"		=> "\xe6\xa8\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xda"		=> "\xe6\xa8\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xdb"		=> "\xe6\xa7\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xdc"		=> "\xe6\xac\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xdd"		=> "\xe6\xa7\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xde"		=> "\xe6\xa7\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xdf"		=> "\xe6\xa7\xa7",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe0"		=> "\xe6\xa8\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe1"		=> "\xe6\xa6\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe2"		=> "\xe6\xa8\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe3"		=> "\xe6\xa7\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe4"		=> "\xe6\xa8\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe5"		=> "\xe6\xa7\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe6"		=> "\xe6\xa8\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe7"		=> "\xe6\xa8\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe8"		=> "\xe6\xab\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xe9"		=> "\xe6\xa8\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xea"		=> "\xe6\xa8\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xeb"		=> "\xe6\xa9\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xec"		=> "\xe6\xa8\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xed"		=> "\xe6\xa9\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xee"		=> "\xe6\xa8\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xef"		=> "\xe6\xa9\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf0"		=> "\xe6\xa9\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf1"		=> "\xe6\xa9\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf2"		=> "\xe6\xa9\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf3"		=> "\xe6\xa9\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf4"		=> "\xe6\xa9\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf5"		=> "\xe6\xa8\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf6"		=> "\xe6\xa8\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf7"		=> "\xe6\xaa\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf8"		=> "\xe6\xaa\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xf9"		=> "\xe6\xaa\xa0",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xfa"		=> "\xe6\xaa\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xfb"		=> "\xe6\xaa\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9e\xfc"		=> "\xe6\xaa\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x40"		=> "\xe6\xaa\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x41"		=> "\xe8\x98\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x42"		=> "\xe6\xaa\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x43"		=> "\xe6\xab\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x44"		=> "\xe6\xab\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x45"		=> "\xe6\xaa\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x46"		=> "\xe6\xaa\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x47"		=> "\xe6\xaa\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x48"		=> "\xe6\xab\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x49"		=> "\xe6\xab\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x4a"		=> "\xe6\xab\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x4b"		=> "\xe6\xaa\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x4c"		=> "\xe6\xab\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x4d"		=> "\xe6\xab\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x4e"		=> "\xe6\xab\xbb",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x4f"		=> "\xe6\xac\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x50"		=> "\xe8\x98\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x51"		=> "\xe6\xab\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x52"		=> "\xe6\xac\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x53"		=> "\xe6\xac\x96",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x54"		=> "\xe9\xac\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x55"		=> "\xe6\xac\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x56"		=> "\xe6\xac\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x57"		=> "\xe6\xac\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x58"		=> "\xe7\x9b\x9c",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x59"		=> "\xe6\xac\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x5a"		=> "\xe9\xa3\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x5b"		=> "\xe6\xad\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x5c"		=> "\xe6\xad\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x5d"		=> "\xe6\xad\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x5e"		=> "\xe6\xad\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x5f"		=> "\xe6\xad\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x60"		=> "\xe6\xad\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x61"		=> "\xe6\xad\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x62"		=> "\xe6\xad\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x63"		=> "\xe6\xad\xa1",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x64"		=> "\xe6\xad\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x65"		=> "\xe6\xad\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x66"		=> "\xe6\xad\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x67"		=> "\xe6\xae\x80",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x68"		=> "\xe6\xae\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x69"		=> "\xe6\xae\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x6a"		=> "\xe6\xae\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x6b"		=> "\xe6\xae\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x6c"		=> "\xe6\xae\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x6d"		=> "\xe6\xae\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x6e"		=> "\xe6\xae\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x6f"		=> "\xe6\xae\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x70"		=> "\xe6\xae\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x71"		=> "\xe6\xae\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x72"		=> "\xe6\xae\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x73"		=> "\xe6\xae\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x74"		=> "\xe6\xae\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x75"		=> "\xe6\xae\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x76"		=> "\xe6\xae\xbc",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x77"		=> "\xe6\xaf\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x78"		=> "\xe6\xaf\x8b",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x79"		=> "\xe6\xaf\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x7a"		=> "\xe6\xaf\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x7b"		=> "\xe6\xaf\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x7c"		=> "\xe6\xaf\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x7d"		=> "\xe6\xaf\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x7e"		=> "\xe6\xaf\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x80"		=> "\xe9\xba\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x81"		=> "\xe6\xb0\x88",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x82"		=> "\xe6\xb0\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x83"		=> "\xe6\xb0\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x84"		=> "\xe6\xb0\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x85"		=> "\xe6\xb0\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x86"		=> "\xe6\xb0\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x87"		=> "\xe6\xb1\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x88"		=> "\xe6\xb1\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x89"		=> "\xe6\xb1\xa2",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x8a"		=> "\xe6\xb1\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x8b"		=> "\xe6\xb2\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x8c"		=> "\xe6\xb2\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x8d"		=> "\xe6\xb2\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x8e"		=> "\xe6\xb2\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x8f"		=> "\xe6\xb2\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x90"		=> "\xe6\xb1\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x91"		=> "\xe6\xb1\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x92"		=> "\xe6\xb1\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x93"		=> "\xe6\xb2\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x94"		=> "\xe6\xb2\x90",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x95"		=> "\xe6\xb3\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x96"		=> "\xe6\xb3\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x97"		=> "\xe6\xb3\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x98"		=> "\xe6\xb2\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x99"		=> "\xe6\xb3\x97",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x9a"		=> "\xe6\xb3\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x9b"		=> "\xe6\xb3\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x9c"		=> "\xe6\xb2\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x9d"		=> "\xe6\xb2\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x9e"		=> "\xe6\xb2\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9f\x9f"		=> "\xe6\xb2\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa0"		=> "\xe6\xb3\x9b",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa1"		=> "\xe6\xb3\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa2"		=> "\xe6\xb3\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa3"		=> "\xe6\xb3\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa4"		=> "\xe6\xb4\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa5"		=> "\xe8\xa1\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa6"		=> "\xe6\xb4\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa7"		=> "\xe6\xb4\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa8"		=> "\xe6\xb4\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xa9"		=> "\xe6\xb4\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xaa"		=> "\xe6\xb4\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xab"		=> "\xe6\xb4\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xac"		=> "\xe6\xb4\xb3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xad"		=> "\xe6\xb4\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xae"		=> "\xe6\xb4\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xaf"		=> "\xe6\xb5\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb0"		=> "\xe6\xb6\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb1"		=> "\xe6\xb5\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb2"		=> "\xe6\xb5\x9a",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb3"		=> "\xe6\xb5\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb4"		=> "\xe6\xb5\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb5"		=> "\xe6\xb6\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb6"		=> "\xe6\xb6\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb7"		=> "\xe6\xbf\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb8"		=> "\xe6\xb6\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xb9"		=> "\xe6\xb7\xb9",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xba"		=> "\xe6\xb8\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xbb"		=> "\xe6\xb8\x8a",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xbc"		=> "\xe6\xb6\xb5",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xbd"		=> "\xe6\xb7\x87",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xbe"		=> "\xe6\xb7\xa6",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xbf"		=> "\xe6\xb6\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc0"		=> "\xe6\xb7\x86",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc1"		=> "\xe6\xb7\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc2"		=> "\xe6\xb7\x9e",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc3"		=> "\xe6\xb7\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc4"		=> "\xe6\xb7\xa8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc5"		=> "\xe6\xb7\x92",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc6"		=> "\xe6\xb7\x85",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc7"		=> "\xe6\xb7\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc8"		=> "\xe6\xb7\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xc9"		=> "\xe6\xb7\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xca"		=> "\xe6\xb7\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xcb"		=> "\xe6\xb7\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xcc"		=> "\xe6\xb7\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xcd"		=> "\xe6\xb8\xad",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xce"		=> "\xe6\xb9\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xcf"		=> "\xe6\xb8\xae",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd0"		=> "\xe6\xb8\x99",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd1"		=> "\xe6\xb9\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd2"		=> "\xe6\xb9\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd3"		=> "\xe6\xb8\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd4"		=> "\xe6\xb8\xa3",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd5"		=> "\xe6\xb9\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd6"		=> "\xe6\xb8\xab",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd7"		=> "\xe6\xb9\xb6",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd8"		=> "\xe6\xb9\x8d",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xd9"		=> "\xe6\xb8\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xda"		=> "\xe6\xb9\x83",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xdb"		=> "\xe6\xb8\xba",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xdc"		=> "\xe6\xb9\x8e",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xdd"		=> "\xe6\xb8\xa4",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xde"		=> "\xe6\xbb\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xdf"		=> "\xe6\xb8\x9d",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe0"		=> "\xe6\xb8\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe1"		=> "\xe6\xba\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe2"		=> "\xe6\xba\xaa",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe3"		=> "\xe6\xba\x98",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe4"		=> "\xe6\xbb\x89",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe5"		=> "\xe6\xba\xb7",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe6"		=> "\xe6\xbb\x93",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe7"		=> "\xe6\xba\xbd",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe8"		=> "\xe6\xba\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xe9"		=> "\xe6\xbb\x84",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xea"		=> "\xe6\xba\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xeb"		=> "\xe6\xbb\x94",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xec"		=> "\xe6\xbb\x95",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xed"		=> "\xe6\xba\x8f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xee"		=> "\xe6\xba\xa5",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xef"		=> "\xe6\xbb\x82",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf0"		=> "\xe6\xba\x9f",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf1"		=> "\xe6\xbd\x81",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf2"		=> "\xe6\xbc\x91",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf3"		=> "\xe7\x81\x8c",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf4"		=> "\xe6\xbb\xac",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf5"		=> "\xe6\xbb\xb8",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf6"		=> "\xe6\xbb\xbe",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf7"		=> "\xe6\xbc\xbf",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf8"		=> "\xe6\xbb\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xf9"		=> "\xe6\xbc\xb1",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xfa"		=> "\xe6\xbb\xaf",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xfb"		=> "\xe6\xbc\xb2",	//CJK UNIFIED IDEOGRAPH
		"\x9f\xfc"		=> "\xe6\xbb\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x40"		=> "\xe6\xbc\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x41"		=> "\xe6\xbc\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x42"		=> "\xe6\xbb\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x43"		=> "\xe6\xbe\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x44"		=> "\xe6\xbd\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x45"		=> "\xe6\xbd\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x46"		=> "\xe6\xbe\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x47"		=> "\xe6\xbe\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x48"		=> "\xe6\xbd\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x49"		=> "\xe6\xbd\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x4a"		=> "\xe6\xbf\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x4b"		=> "\xe6\xbd\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x4c"		=> "\xe6\xbe\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x4d"		=> "\xe6\xbd\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x4e"		=> "\xe6\xbd\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x4f"		=> "\xe6\xbe\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x50"		=> "\xe6\xbe\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x51"		=> "\xe6\xbf\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x52"		=> "\xe6\xbd\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x53"		=> "\xe6\xbe\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x54"		=> "\xe6\xbe\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x55"		=> "\xe6\xbe\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x56"		=> "\xe6\xbe\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x57"		=> "\xe6\xbe\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x58"		=> "\xe6\xbf\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x59"		=> "\xe6\xbe\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x5a"		=> "\xe6\xbf\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x5b"		=> "\xe6\xbf\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x5c"		=> "\xe6\xbf\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x5d"		=> "\xe6\xbf\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x5e"		=> "\xe6\xbf\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x5f"		=> "\xe6\xbf\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x60"		=> "\xe6\xbf\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x61"		=> "\xe6\xbf\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x62"		=> "\xe7\x80\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x63"		=> "\xe7\x80\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x64"		=> "\xe6\xbf\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x65"		=> "\xe7\x80\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x66"		=> "\xe7\x80\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x67"		=> "\xe7\x80\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x68"		=> "\xe6\xbf\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x69"		=> "\xe7\x80\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x6a"		=> "\xe7\x80\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x6b"		=> "\xe6\xbd\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x6c"		=> "\xe7\x80\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x6d"		=> "\xe7\x80\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x6e"		=> "\xe7\x80\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x6f"		=> "\xe7\x80\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x70"		=> "\xe7\x80\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x71"		=> "\xe7\x80\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x72"		=> "\xe7\x81\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x73"		=> "\xe7\x81\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x74"		=> "\xe7\x82\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x75"		=> "\xe7\x82\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x76"		=> "\xe7\x82\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x77"		=> "\xe7\x83\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x78"		=> "\xe7\x82\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x79"		=> "\xe7\x82\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x7a"		=> "\xe7\x82\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x7b"		=> "\xe7\x82\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x7c"		=> "\xe7\x83\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x7d"		=> "\xe7\x83\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x7e"		=> "\xe7\x83\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x80"		=> "\xe7\x83\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x81"		=> "\xe7\x84\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x82"		=> "\xe7\x83\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x83"		=> "\xe7\x84\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x84"		=> "\xe7\x84\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x85"		=> "\xe7\x85\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x86"		=> "\xe7\x85\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x87"		=> "\xe7\x86\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x88"		=> "\xe7\x85\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x89"		=> "\xe7\x85\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x8a"		=> "\xe7\x85\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x8b"		=> "\xe7\x85\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x8c"		=> "\xe7\x85\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x8d"		=> "\xe7\x86\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x8e"		=> "\xe7\x87\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x8f"		=> "\xe7\x86\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x90"		=> "\xe7\x86\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x91"		=> "\xe7\x86\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x92"		=> "\xe7\x86\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x93"		=> "\xe7\x87\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x94"		=> "\xe7\x86\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x95"		=> "\xe7\x86\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x96"		=> "\xe7\x87\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x97"		=> "\xe7\x87\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x98"		=> "\xe7\x87\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x99"		=> "\xe7\x87\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x9a"		=> "\xe7\x87\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x9b"		=> "\xe7\x87\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x9c"		=> "\xe7\x87\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x9d"		=> "\xe7\x87\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x9e"		=> "\xe7\x87\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe0\x9f"		=> "\xe7\x87\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa0"		=> "\xe7\x87\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa1"		=> "\xe7\x88\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa2"		=> "\xe7\x88\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa3"		=> "\xe7\x88\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa4"		=> "\xe7\x88\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa5"		=> "\xe7\x88\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa6"		=> "\xe7\x88\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa7"		=> "\xe7\x88\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa8"		=> "\xe7\x88\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xa9"		=> "\xe7\x88\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xaa"		=> "\xe7\x88\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xab"		=> "\xe7\x88\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xac"		=> "\xe7\x89\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xad"		=> "\xe7\x89\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xae"		=> "\xe7\x89\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xaf"		=> "\xe7\x89\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb0"		=> "\xe7\x89\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb1"		=> "\xe7\x89\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb2"		=> "\xe7\x8a\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb3"		=> "\xe7\x8a\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb4"		=> "\xe7\x8a\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb5"		=> "\xe7\x8a\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb6"		=> "\xe7\x8a\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb7"		=> "\xe7\x8a\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb8"		=> "\xe7\x8a\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xb9"		=> "\xe7\x8a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xba"		=> "\xe7\x8a\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xbb"		=> "\xe7\x8b\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xbc"		=> "\xe7\x8b\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xbd"		=> "\xe7\x8b\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xbe"		=> "\xe7\x8b\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xbf"		=> "\xe7\x8b\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc0"		=> "\xe7\x8b\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc1"		=> "\xe7\x8b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc2"		=> "\xe7\x8b\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc3"		=> "\xe7\x8b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc4"		=> "\xe7\x8b\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc5"		=> "\xe5\x80\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc6"		=> "\xe7\x8c\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc7"		=> "\xe7\x8c\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc8"		=> "\xe7\x8c\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xc9"		=> "\xe7\x8c\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xca"		=> "\xe7\x8c\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xcb"		=> "\xe7\x8c\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xcc"		=> "\xe7\x8c\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xcd"		=> "\xe7\x8c\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xce"		=> "\xe7\x8c\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xcf"		=> "\xe7\x8c\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd0"		=> "\xe7\x8d\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd1"		=> "\xe7\x8d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd2"		=> "\xe9\xbb\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd3"		=> "\xe7\x8d\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd4"		=> "\xe7\x8d\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd5"		=> "\xe7\x8d\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd6"		=> "\xe7\x8d\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd7"		=> "\xe7\x8d\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd8"		=> "\xe7\x8d\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xd9"		=> "\xe7\x8d\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xda"		=> "\xe7\x8d\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xdb"		=> "\xe7\x8f\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xdc"		=> "\xe7\x8e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xdd"		=> "\xe7\x8f\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xde"		=> "\xe7\x8e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xdf"		=> "\xe7\x8f\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe0"		=> "\xe7\x8f\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe1"		=> "\xe7\x8f\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe2"		=> "\xe7\x8f\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe3"		=> "\xe7\x92\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe4"		=> "\xe7\x90\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe5"		=> "\xe7\x91\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe6"		=> "\xe7\x90\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe7"		=> "\xe7\x8f\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe8"		=> "\xe7\x90\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xe9"		=> "\xe7\x90\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xea"		=> "\xe7\x91\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xeb"		=> "\xe7\x90\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xec"		=> "\xe7\x91\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xed"		=> "\xe7\x91\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xee"		=> "\xe7\x91\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xef"		=> "\xe7\x91\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf0"		=> "\xe7\x91\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf1"		=> "\xe7\x91\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf2"		=> "\xe7\x91\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf3"		=> "\xe7\x91\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf4"		=> "\xe7\x91\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf5"		=> "\xe7\x91\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf6"		=> "\xe7\x92\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf7"		=> "\xe7\x92\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf8"		=> "\xe7\x92\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xf9"		=> "\xe7\x93\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xfa"		=> "\xe7\x93\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xfb"		=> "\xe7\x93\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe0\xfc"		=> "\xe7\x8f\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x40"		=> "\xe7\x93\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x41"		=> "\xe7\x93\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x42"		=> "\xe7\x93\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x43"		=> "\xe7\x93\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x44"		=> "\xe7\x93\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x45"		=> "\xe7\x93\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x46"		=> "\xe7\x93\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x47"		=> "\xe7\x93\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x48"		=> "\xe7\x93\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x49"		=> "\xe7\x93\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x4a"		=> "\xe7\x94\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x4b"		=> "\xe7\x94\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x4c"		=> "\xe7\x94\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x4d"		=> "\xe7\x94\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x4e"		=> "\xe7\x94\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x4f"		=> "\xe7\x94\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x50"		=> "\xe7\x94\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x51"		=> "\xe7\x94\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x52"		=> "\xe7\x94\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x53"		=> "\xe7\x94\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x54"		=> "\xe7\x94\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x55"		=> "\xe7\x94\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x56"		=> "\xe7\x95\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x57"		=> "\xe7\x95\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x58"		=> "\xe7\x95\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x59"		=> "\xe7\x95\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x5a"		=> "\xe7\x95\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x5b"		=> "\xe7\x95\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x5c"		=> "\xe7\x95\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x5d"		=> "\xe7\x95\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x5e"		=> "\xe7\x95\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x5f"		=> "\xe7\x95\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x60"		=> "\xe7\x95\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x61"		=> "\xe7\x95\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x62"		=> "\xe7\x95\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x63"		=> "\xe7\x95\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x64"		=> "\xe7\x96\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x65"		=> "\xe7\x96\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x66"		=> "\xe7\x95\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x67"		=> "\xe7\x96\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x68"		=> "\xe7\x96\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x69"		=> "\xe7\x96\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x6a"		=> "\xe7\x96\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x6b"		=> "\xe7\x96\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x6c"		=> "\xe7\x96\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x6d"		=> "\xe7\x96\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x6e"		=> "\xe7\x96\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x6f"		=> "\xe7\x97\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x70"		=> "\xe7\x96\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x71"		=> "\xe7\x97\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x72"		=> "\xe7\x96\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x73"		=> "\xe7\x96\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x74"		=> "\xe7\x96\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x75"		=> "\xe7\x96\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x76"		=> "\xe7\x96\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x77"		=> "\xe7\x97\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x78"		=> "\xe7\x97\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x79"		=> "\xe7\x97\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x7a"		=> "\xe7\x97\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x7b"		=> "\xe7\x97\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x7c"		=> "\xe7\x97\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x7d"		=> "\xe7\x97\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x7e"		=> "\xe7\x97\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x80"		=> "\xe7\x97\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x81"		=> "\xe7\x98\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x82"		=> "\xe7\x97\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x83"		=> "\xe7\x97\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x84"		=> "\xe7\x97\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x85"		=> "\xe7\x97\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x86"		=> "\xe7\x98\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x87"		=> "\xe7\x98\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x88"		=> "\xe7\x98\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x89"		=> "\xe7\x98\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x8a"		=> "\xe7\x98\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x8b"		=> "\xe7\x98\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x8c"		=> "\xe7\x98\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x8d"		=> "\xe7\x98\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x8e"		=> "\xe7\x98\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x8f"		=> "\xe7\x98\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x90"		=> "\xe7\x98\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x91"		=> "\xe7\x98\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x92"		=> "\xe7\x99\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x93"		=> "\xe7\x99\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x94"		=> "\xe7\x99\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x95"		=> "\xe7\x99\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x96"		=> "\xe7\x99\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x97"		=> "\xe7\x99\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x98"		=> "\xe7\x99\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x99"		=> "\xe7\x99\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x9a"		=> "\xe7\x99\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x9b"		=> "\xe7\x99\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x9c"		=> "\xe7\x99\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x9d"		=> "\xe7\x99\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x9e"		=> "\xe7\x99\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\x9f"		=> "\xe7\x99\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa0"		=> "\xe7\x99\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa1"		=> "\xe7\x99\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa2"		=> "\xe7\x99\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa3"		=> "\xe7\x9a\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa4"		=> "\xe7\x9a\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa5"		=> "\xe7\x9a\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa6"		=> "\xe7\x9a\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa7"		=> "\xe7\x9a\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa8"		=> "\xe7\x9a\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xa9"		=> "\xe7\x9a\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xaa"		=> "\xe7\x9a\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xab"		=> "\xe7\x9a\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xac"		=> "\xe7\x9a\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xad"		=> "\xe7\x9a\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xae"		=> "\xe7\x9a\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xaf"		=> "\xe7\x9a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb0"		=> "\xe7\x9a\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb1"		=> "\xe7\x9b\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb2"		=> "\xe7\x9b\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb3"		=> "\xe7\x9b\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb4"		=> "\xe7\x9b\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb5"		=> "\xe7\x9b\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb6"		=> "\xe7\x9b\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb7"		=> "\xe7\x9b\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb8"		=> "\xe7\x9b\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xb9"		=> "\xe7\x9b\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xba"		=> "\xe8\x98\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xbb"		=> "\xe7\x9b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xbc"		=> "\xe7\x9c\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xbd"		=> "\xe7\x9c\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xbe"		=> "\xe7\x9c\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xbf"		=> "\xe7\x9c\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc0"		=> "\xe7\x9c\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc1"		=> "\xe7\x9c\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc2"		=> "\xe7\x9c\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc3"		=> "\xe7\x9c\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc4"		=> "\xe7\x9c\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc5"		=> "\xe7\x9c\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc6"		=> "\xe7\x9c\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc7"		=> "\xe7\x9d\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc8"		=> "\xe7\x9d\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xc9"		=> "\xe7\x9d\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xca"		=> "\xe7\x9d\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xcb"		=> "\xe7\x9d\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xcc"		=> "\xe7\x9d\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xcd"		=> "\xe7\x9d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xce"		=> "\xe7\x9d\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xcf"		=> "\xe7\x9d\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd0"		=> "\xe7\x9e\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd1"		=> "\xe7\x9e\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd2"		=> "\xe7\x9e\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd3"		=> "\xe7\x9e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd4"		=> "\xe7\x9e\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd5"		=> "\xe7\x9e\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd6"		=> "\xe7\x9e\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd7"		=> "\xe7\x9e\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd8"		=> "\xe7\x9e\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xd9"		=> "\xe7\x9e\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xda"		=> "\xe7\x9e\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xdb"		=> "\xe7\x9e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xdc"		=> "\xe7\x9f\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xdd"		=> "\xe7\x9f\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xde"		=> "\xe7\x9f\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xdf"		=> "\xe7\x9f\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe0"		=> "\xe7\x9f\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe1"		=> "\xe7\x9f\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe2"		=> "\xe7\x9f\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe3"		=> "\xe7\x9f\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe4"		=> "\xe7\xa0\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe5"		=> "\xe7\xa0\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe6"		=> "\xe7\xa4\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe7"		=> "\xe7\xa0\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe8"		=> "\xe7\xa4\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xe9"		=> "\xe7\xa1\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xea"		=> "\xe7\xa2\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xeb"		=> "\xe7\xa1\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xec"		=> "\xe7\xa2\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xed"		=> "\xe7\xa1\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xee"		=> "\xe7\xa2\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xef"		=> "\xe7\xa2\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf0"		=> "\xe7\xa2\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf1"		=> "\xe7\xa2\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf2"		=> "\xe7\xa2\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf3"		=> "\xe7\xa2\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf4"		=> "\xe7\xa3\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf5"		=> "\xe7\xa3\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf6"		=> "\xe7\xa3\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf7"		=> "\xe7\xa3\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf8"		=> "\xe7\xa2\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xf9"		=> "\xe7\xa2\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xfa"		=> "\xe7\xa3\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xfb"		=> "\xe7\xa3\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe1\xfc"		=> "\xe7\xa3\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x40"		=> "\xe7\xa3\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x41"		=> "\xe7\xa3\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x42"		=> "\xe7\xa3\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x43"		=> "\xe7\xa3\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x44"		=> "\xe7\xa4\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x45"		=> "\xe7\xa4\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x46"		=> "\xe7\xa4\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x47"		=> "\xe7\xa4\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x48"		=> "\xe7\xa4\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x49"		=> "\xe7\xa4\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x4a"		=> "\xe7\xa5\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x4b"		=> "\xe7\xa5\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x4c"		=> "\xe7\xa5\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x4d"		=> "\xe7\xa5\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x4e"		=> "\xe7\xa5\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x4f"		=> "\xe7\xa5\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x50"		=> "\xe7\xa5\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x51"		=> "\xe7\xa5\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x52"		=> "\xe7\xa5\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x53"		=> "\xe7\xa6\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x54"		=> "\xe7\xa6\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x55"		=> "\xe7\xa6\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x56"		=> "\xe9\xbd\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x57"		=> "\xe7\xa6\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x58"		=> "\xe7\xa6\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x59"		=> "\xe7\xa6\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x5a"		=> "\xe7\xa6\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x5b"		=> "\xe7\xa6\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x5c"		=> "\xe7\xa7\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x5d"		=> "\xe7\xa7\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x5e"		=> "\xe7\xa7\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x5f"		=> "\xe7\xa7\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x60"		=> "\xe7\xa7\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x61"		=> "\xe7\xa7\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x62"		=> "\xe7\xa8\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x63"		=> "\xe7\xa8\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x64"		=> "\xe7\xa8\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x65"		=> "\xe7\xa8\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x66"		=> "\xe7\xa8\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x67"		=> "\xe7\xa8\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x68"		=> "\xe7\xa6\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x69"		=> "\xe7\xa8\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x6a"		=> "\xe7\xa8\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x6b"		=> "\xe7\xa8\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x6c"		=> "\xe7\xa8\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x6d"		=> "\xe7\xa9\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x6e"		=> "\xe7\xa9\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x6f"		=> "\xe7\xa9\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x70"		=> "\xe7\xa9\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x71"		=> "\xe7\xa9\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x72"		=> "\xe7\xa9\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x73"		=> "\xe9\xbe\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x74"		=> "\xe7\xa9\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x75"		=> "\xe7\xa9\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x76"		=> "\xe7\xa9\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x77"		=> "\xe7\xaa\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x78"		=> "\xe7\xaa\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x79"		=> "\xe7\xaa\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x7a"		=> "\xe7\xaa\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x7b"		=> "\xe7\xaa\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x7c"		=> "\xe7\xaa\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x7d"		=> "\xe7\xab\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x7e"		=> "\xe7\xaa\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x80"		=> "\xe7\xaa\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x81"		=> "\xe7\xab\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x82"		=> "\xe7\xab\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x83"		=> "\xe7\xaa\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x84"		=> "\xe9\x82\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x85"		=> "\xe7\xab\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x86"		=> "\xe7\xab\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x87"		=> "\xe7\xab\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x88"		=> "\xe7\xab\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x89"		=> "\xe7\xab\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x8a"		=> "\xe7\xab\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x8b"		=> "\xe7\xab\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x8c"		=> "\xe7\xab\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x8d"		=> "\xe7\xab\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x8e"		=> "\xe7\xab\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x8f"		=> "\xe7\xab\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x90"		=> "\xe7\xab\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x91"		=> "\xe7\xab\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x92"		=> "\xe7\xab\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x93"		=> "\xe7\xac\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x94"		=> "\xe7\xac\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x95"		=> "\xe7\xac\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x96"		=> "\xe7\xac\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x97"		=> "\xe7\xac\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x98"		=> "\xe7\xac\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x99"		=> "\xe7\xac\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x9a"		=> "\xe7\xac\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x9b"		=> "\xe7\xac\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x9c"		=> "\xe7\xac\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x9d"		=> "\xe7\xac\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x9e"		=> "\xe7\xad\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe2\x9f"		=> "\xe7\xad\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa0"		=> "\xe7\xac\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa1"		=> "\xe7\xad\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa2"		=> "\xe7\xac\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa3"		=> "\xe7\xad\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa4"		=> "\xe7\xad\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa5"		=> "\xe7\xad\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa6"		=> "\xe7\xad\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa7"		=> "\xe7\xad\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa8"		=> "\xe7\xad\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xa9"		=> "\xe7\xad\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xaa"		=> "\xe7\xad\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xab"		=> "\xe7\xad\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xac"		=> "\xe7\xad\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xad"		=> "\xe7\xae\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xae"		=> "\xe7\xae\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xaf"		=> "\xe7\xae\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb0"		=> "\xe7\xae\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb1"		=> "\xe7\xae\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb2"		=> "\xe7\xae\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb3"		=> "\xe7\xae\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb4"		=> "\xe7\xae\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb5"		=> "\xe7\xae\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb6"		=> "\xe7\xad\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb7"		=> "\xe7\xae\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb8"		=> "\xe7\xaf\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xb9"		=> "\xe7\xaf\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xba"		=> "\xe7\xaf\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xbb"		=> "\xe7\xaf\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xbc"		=> "\xe7\xae\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xbd"		=> "\xe7\xaf\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xbe"		=> "\xe7\xaf\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xbf"		=> "\xe7\xaf\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc0"		=> "\xe7\xb0\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc1"		=> "\xe7\xb0\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc2"		=> "\xe7\xaf\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc3"		=> "\xe7\xaf\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc4"		=> "\xe7\xb1\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc5"		=> "\xe7\xb0\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc6"		=> "\xe7\xb0\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc7"		=> "\xe7\xb0\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc8"		=> "\xe7\xaf\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xc9"		=> "\xe7\xaf\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xca"		=> "\xe7\xb0\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xcb"		=> "\xe7\xb0\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xcc"		=> "\xe7\xaf\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xcd"		=> "\xe7\xb0\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xce"		=> "\xe7\xb0\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xcf"		=> "\xe7\xb0\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd0"		=> "\xe7\xb0\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd1"		=> "\xe7\xb0\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd2"		=> "\xe7\xb0\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd3"		=> "\xe7\xb0\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd4"		=> "\xe7\xb1\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd5"		=> "\xe7\xb1\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd6"		=> "\xe7\xb1\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd7"		=> "\xe7\xb1\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd8"		=> "\xe7\xb1\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xd9"		=> "\xe7\xb1\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xda"		=> "\xe7\xb1\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xdb"		=> "\xe7\xb1\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xdc"		=> "\xe7\xb1\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xdd"		=> "\xe7\xb1\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xde"		=> "\xe7\xb1\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xdf"		=> "\xe7\xb1\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe0"		=> "\xe7\xb1\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe1"		=> "\xe7\xb2\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe2"		=> "\xe7\xb2\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe3"		=> "\xe7\xb2\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe4"		=> "\xe7\xb2\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe5"		=> "\xe7\xb2\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe6"		=> "\xe7\xb2\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe7"		=> "\xe7\xb2\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe8"		=> "\xe7\xb2\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xe9"		=> "\xe7\xb2\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xea"		=> "\xe7\xb2\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xeb"		=> "\xe7\xb2\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xec"		=> "\xe7\xb2\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xed"		=> "\xe7\xb2\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xee"		=> "\xe7\xb2\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xef"		=> "\xe7\xb3\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf0"		=> "\xe7\xb3\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf1"		=> "\xe7\xb3\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf2"		=> "\xe7\xb3\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf3"		=> "\xe7\xb3\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf4"		=> "\xe7\xb3\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf5"		=> "\xe7\xb3\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf6"		=> "\xe9\xac\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf7"		=> "\xe7\xb3\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf8"		=> "\xe7\xb3\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xf9"		=> "\xe7\xb3\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xfa"		=> "\xe7\xb3\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xfb"		=> "\xe7\xb3\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe2\xfc"		=> "\xe7\xb4\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x40"		=> "\xe7\xb4\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x41"		=> "\xe7\xb4\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x42"		=> "\xe7\xb4\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x43"		=> "\xe7\xb4\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x44"		=> "\xe7\xb5\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x45"		=> "\xe7\xb5\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x46"		=> "\xe7\xb4\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x47"		=> "\xe7\xb4\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x48"		=> "\xe7\xb4\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x49"		=> "\xe7\xb4\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x4a"		=> "\xe7\xb5\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x4b"		=> "\xe7\xb5\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x4c"		=> "\xe7\xb5\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x4d"		=> "\xe7\xb5\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x4e"		=> "\xe7\xb5\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x4f"		=> "\xe7\xb5\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x50"		=> "\xe7\xb5\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x51"		=> "\xe7\xb5\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x52"		=> "\xe7\xb5\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x53"		=> "\xe7\xb6\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x54"		=> "\xe7\xb6\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x55"		=> "\xe7\xb5\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x56"		=> "\xe7\xb6\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x57"		=> "\xe7\xb5\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x58"		=> "\xe7\xb6\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x59"		=> "\xe7\xb6\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x5a"		=> "\xe7\xb6\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x5b"		=> "\xe7\xb6\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x5c"		=> "\xe7\xb6\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x5d"		=> "\xe7\xb7\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x5e"		=> "\xe7\xb6\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x5f"		=> "\xe7\xb6\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x60"		=> "\xe7\xb8\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x61"		=> "\xe7\xb6\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x62"		=> "\xe7\xb6\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x63"		=> "\xe7\xb7\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x64"		=> "\xe7\xb6\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x65"		=> "\xe7\xb6\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x66"		=> "\xe7\xb6\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x67"		=> "\xe7\xb7\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x68"		=> "\xe7\xb7\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x69"		=> "\xe7\xb7\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x6a"		=> "\xe7\xb7\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x6b"		=> "\xe7\xb7\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x6c"		=> "\xe7\xb7\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x6d"		=> "\xe7\xb7\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x6e"		=> "\xe7\xb8\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x6f"		=> "\xe7\xb8\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x70"		=> "\xe7\xb8\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x71"		=> "\xe7\xb8\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x72"		=> "\xe7\xb8\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x73"		=> "\xe7\xb8\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x74"		=> "\xe7\xb8\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x75"		=> "\xe7\xb8\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x76"		=> "\xe7\xb8\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x77"		=> "\xe7\xb8\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x78"		=> "\xe7\xb9\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x79"		=> "\xe7\xb9\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x7a"		=> "\xe7\xb8\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x7b"		=> "\xe7\xb8\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x7c"		=> "\xe7\xb8\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x7d"		=> "\xe7\xb9\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x7e"		=> "\xe7\xb8\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x80"		=> "\xe7\xb8\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x81"		=> "\xe7\xb8\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x82"		=> "\xe7\xb9\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x83"		=> "\xe7\xb9\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x84"		=> "\xe7\xb9\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x85"		=> "\xe7\xb9\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x86"		=> "\xe7\xb9\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x87"		=> "\xe7\xb9\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x88"		=> "\xe7\xb9\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x89"		=> "\xe7\xb9\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x8a"		=> "\xe7\xb9\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x8b"		=> "\xe7\xb9\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x8c"		=> "\xe7\xb9\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x8d"		=> "\xe7\xba\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x8e"		=> "\xe7\xb7\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x8f"		=> "\xe7\xb9\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x90"		=> "\xe8\xbe\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x91"		=> "\xe7\xb9\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x92"		=> "\xe7\xba\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x93"		=> "\xe7\xba\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x94"		=> "\xe7\xba\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x95"		=> "\xe7\xba\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x96"		=> "\xe7\xba\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x97"		=> "\xe7\xba\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x98"		=> "\xe7\xba\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x99"		=> "\xe7\xba\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x9a"		=> "\xe7\xba\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x9b"		=> "\xe7\xba\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x9c"		=> "\xe7\xba\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x9d"		=> "\xe7\xbc\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x9e"		=> "\xe7\xbc\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe3\x9f"		=> "\xe7\xbd\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa0"		=> "\xe7\xbd\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa1"		=> "\xe7\xbd\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa2"		=> "\xe7\xbd\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa3"		=> "\xe7\xbd\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa4"		=> "\xe7\xbd\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa5"		=> "\xe7\xbd\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa6"		=> "\xe7\xbd\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa7"		=> "\xe7\xbd\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa8"		=> "\xe7\xbd\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xa9"		=> "\xe7\xbd\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xaa"		=> "\xe7\xbd\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xab"		=> "\xe7\xbd\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xac"		=> "\xe7\xbd\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xad"		=> "\xe7\xbd\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xae"		=> "\xe7\xbe\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xaf"		=> "\xe7\xbe\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb0"		=> "\xe7\xbe\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb1"		=> "\xe7\xbe\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb2"		=> "\xe7\xbe\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb3"		=> "\xe7\xbe\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb4"		=> "\xe7\xbe\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb5"		=> "\xe7\xbe\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb6"		=> "\xe7\xbe\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb7"		=> "\xe7\xbe\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb8"		=> "\xe7\xbe\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xb9"		=> "\xe7\xbe\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xba"		=> "\xe7\xbe\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xbb"		=> "\xe7\xbe\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xbc"		=> "\xe7\xbe\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xbd"		=> "\xe7\xbe\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xbe"		=> "\xe7\xbe\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xbf"		=> "\xe8\xad\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc0"		=> "\xe7\xbf\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc1"		=> "\xe7\xbf\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc2"		=> "\xe7\xbf\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc3"		=> "\xe7\xbf\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc4"		=> "\xe7\xbf\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc5"		=> "\xe7\xbf\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc6"		=> "\xe7\xbf\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc7"		=> "\xe7\xbf\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc8"		=> "\xe7\xbf\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xc9"		=> "\xe7\xbf\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xca"		=> "\xe9\xa3\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xcb"		=> "\xe8\x80\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xcc"		=> "\xe8\x80\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xcd"		=> "\xe8\x80\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xce"		=> "\xe8\x80\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xcf"		=> "\xe8\x80\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd0"		=> "\xe8\x80\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd1"		=> "\xe8\x80\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd2"		=> "\xe8\x80\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd3"		=> "\xe8\x80\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd4"		=> "\xe8\x80\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd5"		=> "\xe8\x80\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd6"		=> "\xe8\x81\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd7"		=> "\xe8\x81\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd8"		=> "\xe8\x81\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xd9"		=> "\xe8\x81\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xda"		=> "\xe8\x81\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xdb"		=> "\xe8\x81\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xdc"		=> "\xe8\x81\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xdd"		=> "\xe8\x81\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xde"		=> "\xe8\x81\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xdf"		=> "\xe8\x81\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe0"		=> "\xe8\x81\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe1"		=> "\xe8\x81\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe2"		=> "\xe8\x81\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe3"		=> "\xe8\x81\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe4"		=> "\xe8\x81\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe5"		=> "\xe8\x82\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe6"		=> "\xe8\x82\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe7"		=> "\xe8\x82\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe8"		=> "\xe8\x82\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xe9"		=> "\xe8\x82\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xea"		=> "\xe8\x82\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xeb"		=> "\xe8\x82\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xec"		=> "\xe5\x86\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xed"		=> "\xe8\x82\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xee"		=> "\xe8\x83\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xef"		=> "\xe8\x83\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf0"		=> "\xe8\x83\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf1"		=> "\xe8\x83\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf2"		=> "\xe8\x83\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf3"		=> "\xe8\x83\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf4"		=> "\xe8\x83\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf5"		=> "\xe8\x84\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf6"		=> "\xe8\x83\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf7"		=> "\xe8\x83\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf8"		=> "\xe8\x84\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xf9"		=> "\xe8\x84\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xfa"		=> "\xe8\x84\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xfb"		=> "\xe8\x84\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe3\xfc"		=> "\xe8\x85\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x40"		=> "\xe9\x9a\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x41"		=> "\xe8\x85\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x42"		=> "\xe8\x84\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x43"		=> "\xe8\x85\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x44"		=> "\xe8\x85\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x45"		=> "\xe8\x83\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x46"		=> "\xe8\x85\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x47"		=> "\xe8\x85\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x48"		=> "\xe8\x85\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x49"		=> "\xe8\x85\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x4a"		=> "\xe8\x85\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x4b"		=> "\xe8\x86\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x4c"		=> "\xe8\x86\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x4d"		=> "\xe8\x86\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x4e"		=> "\xe8\x86\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x4f"		=> "\xe8\x86\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x50"		=> "\xe8\x86\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x51"		=> "\xe8\x86\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x52"		=> "\xe8\x86\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x53"		=> "\xe8\x86\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x54"		=> "\xe8\x85\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x55"		=> "\xe8\x86\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x56"		=> "\xe8\x86\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x57"		=> "\xe8\x86\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x58"		=> "\xe8\x86\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x59"		=> "\xe8\x86\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x5a"		=> "\xe8\x86\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x5b"		=> "\xe8\x86\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x5c"		=> "\xe8\x87\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x5d"		=> "\xe8\x87\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x5e"		=> "\xe8\x86\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x5f"		=> "\xe8\x87\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x60"		=> "\xe8\x87\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x61"		=> "\xe8\x87\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x62"		=> "\xe8\x87\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x63"		=> "\xe8\x87\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x64"		=> "\xe8\x87\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x65"		=> "\xe8\x87\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x66"		=> "\xe8\x87\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x67"		=> "\xe8\x87\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x68"		=> "\xe8\x87\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x69"		=> "\xe8\x87\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x6a"		=> "\xe8\x87\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x6b"		=> "\xe8\x87\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x6c"		=> "\xe8\x88\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x6d"		=> "\xe8\x88\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x6e"		=> "\xe8\x88\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x6f"		=> "\xe8\x88\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x70"		=> "\xe8\x88\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x71"		=> "\xe8\x88\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x72"		=> "\xe8\x88\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x73"		=> "\xe8\x88\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x74"		=> "\xe8\x88\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x75"		=> "\xe8\x88\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x76"		=> "\xe8\x88\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x77"		=> "\xe8\x88\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x78"		=> "\xe8\x89\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x79"		=> "\xe8\x89\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x7a"		=> "\xe8\x89\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x7b"		=> "\xe8\x89\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x7c"		=> "\xe8\x89\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x7d"		=> "\xe8\x89\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x7e"		=> "\xe8\x89\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x80"		=> "\xe8\x89\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x81"		=> "\xe8\x89\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x82"		=> "\xe8\x89\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x83"		=> "\xe8\x89\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x84"		=> "\xe8\x88\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x85"		=> "\xe8\x89\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x86"		=> "\xe8\x89\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x87"		=> "\xe8\x89\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x88"		=> "\xe8\x89\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x89"		=> "\xe8\x8a\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x8a"		=> "\xe8\x8a\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x8b"		=> "\xe8\x8a\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x8c"		=> "\xe8\x8a\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x8d"		=> "\xe8\x8a\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x8e"		=> "\xe8\x8a\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x8f"		=> "\xe8\x8b\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x90"		=> "\xe8\x8b\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x91"		=> "\xe8\x8b\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x92"		=> "\xe8\x8b\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x93"		=> "\xe8\x8b\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x94"		=> "\xe8\x8b\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x95"		=> "\xe8\x8b\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x96"		=> "\xe8\x8e\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x97"		=> "\xe8\x8c\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x98"		=> "\xe8\x8b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x99"		=> "\xe8\x8b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x9a"		=> "\xe8\x8b\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x9b"		=> "\xe8\x8c\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x9c"		=> "\xe8\x8b\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x9d"		=> "\xe8\x8c\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x9e"		=> "\xe8\x8b\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe4\x9f"		=> "\xe8\x8c\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa0"		=> "\xe8\x8c\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa1"		=> "\xe8\x8c\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa2"		=> "\xe8\x8c\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa3"		=> "\xe8\x8c\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa4"		=> "\xe8\x8d\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa5"		=> "\xe8\x8c\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa6"		=> "\xe8\x8d\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa7"		=> "\xe8\x8d\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa8"		=> "\xe8\x8c\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xa9"		=> "\xe8\x8c\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xaa"		=> "\xe8\x8c\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xab"		=> "\xe8\x8c\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xac"		=> "\xe8\x8e\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xad"		=> "\xe8\x8e\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xae"		=> "\xe8\x8e\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xaf"		=> "\xe8\x8e\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb0"		=> "\xe8\x8e\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb1"		=> "\xe8\x8e\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb2"		=> "\xe8\x8c\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb3"		=> "\xe8\x8e\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb4"		=> "\xe8\x8e\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb5"		=> "\xe8\x8e\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb6"		=> "\xe8\x8d\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb7"		=> "\xe8\x8e\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb8"		=> "\xe8\x8d\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xb9"		=> "\xe8\x8d\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xba"		=> "\xe8\x8e\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbb"		=> "\xe8\x8e\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbc"		=> "\xe8\x8e\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbd"		=> "\xe8\x8f\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbe"		=> "\xe8\x90\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xbf"		=> "\xe8\x8f\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc0"		=> "\xe8\x8f\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc1"		=> "\xe8\x8f\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc2"		=> "\xe8\x90\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc3"		=> "\xe8\x8f\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc4"		=> "\xe8\x90\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc5"		=> "\xe8\x8f\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc6"		=> "\xe8\x8f\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc7"		=> "\xe8\x90\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc8"		=> "\xe8\x8f\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xc9"		=> "\xe8\x8f\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xca"		=> "\xe8\x90\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xcb"		=> "\xe8\x90\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xcc"		=> "\xe8\x90\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xcd"		=> "\xe8\x8e\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xce"		=> "\xe8\x90\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xcf"		=> "\xe8\x94\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd0"		=> "\xe8\x8f\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd1"		=> "\xe8\x91\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd2"		=> "\xe8\x90\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd3"		=> "\xe8\x90\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd4"		=> "\xe8\x95\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd5"		=> "\xe8\x92\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd6"		=> "\xe8\x91\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd7"		=> "\xe8\x91\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd8"		=> "\xe8\x92\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xd9"		=> "\xe8\x91\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xda"		=> "\xe8\x92\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xdb"		=> "\xe8\x91\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xdc"		=> "\xe8\x91\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xdd"		=> "\xe8\x90\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xde"		=> "\xe8\x91\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xdf"		=> "\xe8\x91\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe0"		=> "\xe8\x90\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe1"		=> "\xe8\x93\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe2"		=> "\xe8\x91\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe3"		=> "\xe8\x92\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe4"		=> "\xe8\x92\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe5"		=> "\xe8\x92\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe6"		=> "\xe8\x93\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe7"		=> "\xe8\x93\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe8"		=> "\xe8\x92\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xe9"		=> "\xe8\x93\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xea"		=> "\xe8\x93\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xeb"		=> "\xe8\x93\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xec"		=> "\xe8\x93\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xed"		=> "\xe8\x93\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xee"		=> "\xe8\x92\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xef"		=> "\xe8\x94\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf0"		=> "\xe8\x93\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf1"		=> "\xe8\x93\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf2"		=> "\xe8\x94\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf3"		=> "\xe8\x94\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf4"		=> "\xe8\x94\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf5"		=> "\xe8\x94\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf6"		=> "\xe8\x94\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf7"		=> "\xe8\x94\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf8"		=> "\xe8\x93\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xf9"		=> "\xe8\x95\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xfa"		=> "\xe8\x95\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xfb"		=> "\xe8\x95\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe4\xfc"		=> "\xe8\x95\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x40"		=> "\xe8\x95\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x41"		=> "\xe8\x98\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x42"		=> "\xe8\x95\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x43"		=> "\xe8\x95\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x44"		=> "\xe8\x96\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x45"		=> "\xe8\x96\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x46"		=> "\xe8\x96\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x47"		=> "\xe8\x96\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x48"		=> "\xe8\x96\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x49"		=> "\xe8\x96\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x4a"		=> "\xe8\x95\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x4b"		=> "\xe8\x96\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x4c"		=> "\xe8\x96\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x4d"		=> "\xe8\x97\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x4e"		=> "\xe8\x96\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x4f"		=> "\xe8\x96\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x50"		=> "\xe8\x95\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x51"		=> "\xe8\x95\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x52"		=> "\xe8\x96\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x53"		=> "\xe8\x97\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x54"		=> "\xe8\x96\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x55"		=> "\xe8\x97\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x56"		=> "\xe8\x96\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x57"		=> "\xe8\x97\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x58"		=> "\xe8\x97\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x59"		=> "\xe8\x97\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x5a"		=> "\xe8\x97\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x5b"		=> "\xe8\x97\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x5c"		=> "\xe8\x97\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x5d"		=> "\xe8\x98\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x5e"		=> "\xe8\x98\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x5f"		=> "\xe8\x98\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x60"		=> "\xe8\x97\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x61"		=> "\xe8\x97\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x62"		=> "\xe8\x98\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x63"		=> "\xe8\x98\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x64"		=> "\xe8\x98\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x65"		=> "\xe8\x98\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x66"		=> "\xe8\x98\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x67"		=> "\xe8\x99\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x68"		=> "\xe4\xb9\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x69"		=> "\xe8\x99\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x6a"		=> "\xe8\x99\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x6b"		=> "\xe8\x99\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x6c"		=> "\xe8\x99\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x6d"		=> "\xe8\x9a\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x6e"		=> "\xe8\x9a\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x6f"		=> "\xe8\x9a\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x70"		=> "\xe8\x9a\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x71"		=> "\xe8\x9a\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x72"		=> "\xe8\x9a\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x73"		=> "\xe8\x9a\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x74"		=> "\xe8\x9a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x75"		=> "\xe8\x9b\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x76"		=> "\xe8\x9b\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x77"		=> "\xe8\x9a\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x78"		=> "\xe8\x9b\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x79"		=> "\xe8\xa0\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x7a"		=> "\xe8\x9a\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x7b"		=> "\xe8\x9b\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x7c"		=> "\xe8\x9b\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x7d"		=> "\xe8\x9b\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x7e"		=> "\xe8\x9b\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x80"		=> "\xe8\x9b\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x81"		=> "\xe8\x9b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x82"		=> "\xe8\x9b\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x83"		=> "\xe8\x9c\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x84"		=> "\xe8\x9c\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x85"		=> "\xe8\x9c\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x86"		=> "\xe8\x9c\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x87"		=> "\xe8\x9c\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x88"		=> "\xe8\x9b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x89"		=> "\xe8\x9c\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8a"		=> "\xe8\x9c\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8b"		=> "\xe8\x9c\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8c"		=> "\xe8\x9b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8d"		=> "\xe8\x9c\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8e"		=> "\xe8\x9c\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x8f"		=> "\xe8\x9c\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x90"		=> "\xe8\x9c\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x91"		=> "\xe8\x9c\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x92"		=> "\xe8\x9c\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x93"		=> "\xe8\x9c\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x94"		=> "\xe8\x9c\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x95"		=> "\xe8\x9d\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x96"		=> "\xe8\x9d\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x97"		=> "\xe8\x9d\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x98"		=> "\xe8\x9d\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x99"		=> "\xe8\x9d\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9a"		=> "\xe8\x9d\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9b"		=> "\xe8\x9d\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9c"		=> "\xe8\x9d\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9d"		=> "\xe8\x9d\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9e"		=> "\xe8\x9d\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\x9f"		=> "\xe8\x9d\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa0"		=> "\xe8\x9d\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa1"		=> "\xe8\x9d\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa2"		=> "\xe8\xa0\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa3"		=> "\xe8\x9e\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa4"		=> "\xe8\x9e\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa5"		=> "\xe8\x9e\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa6"		=> "\xe8\x9e\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa7"		=> "\xe8\x9f\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa8"		=> "\xe8\x9e\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xa9"		=> "\xe8\x9f\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaa"		=> "\xe8\x9f\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xab"		=> "\xe9\x9b\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xac"		=> "\xe8\x9e\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xad"		=> "\xe8\x9f\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xae"		=> "\xe8\x9e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xaf"		=> "\xe8\x9f\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb0"		=> "\xe8\x9f\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb1"		=> "\xe8\x9e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb2"		=> "\xe8\x9f\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb3"		=> "\xe8\x9f\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb4"		=> "\xe8\x9f\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb5"		=> "\xe8\xa0\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb6"		=> "\xe8\xa0\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb7"		=> "\xe8\x9f\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb8"		=> "\xe8\x9f\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xb9"		=> "\xe8\x9f\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xba"		=> "\xe8\xa0\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbb"		=> "\xe8\x9f\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbc"		=> "\xe8\xa0\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbd"		=> "\xe8\xa0\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbe"		=> "\xe8\xa0\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xbf"		=> "\xe8\xa0\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc0"		=> "\xe8\xa0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc1"		=> "\xe8\xa0\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc2"		=> "\xe8\xa0\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc3"		=> "\xe8\xa0\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc4"		=> "\xe8\xa0\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc5"		=> "\xe8\xa0\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc6"		=> "\xe8\xa1\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc7"		=> "\xe8\xa1\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc8"		=> "\xe8\xa1\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xc9"		=> "\xe8\xa1\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xca"		=> "\xe8\xa1\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xcb"		=> "\xe8\xa1\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xcc"		=> "\xe8\xa1\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xcd"		=> "\xe8\xa2\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xce"		=> "\xe8\xa1\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xcf"		=> "\xe8\xa2\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd0"		=> "\xe8\xa1\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd1"		=> "\xe8\xa1\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd2"		=> "\xe8\xa2\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd3"		=> "\xe8\xa1\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd4"		=> "\xe8\xa2\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd5"		=> "\xe8\xa2\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd6"		=> "\xe8\xa2\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd7"		=> "\xe8\xa2\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd8"		=> "\xe8\xa2\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xd9"		=> "\xe8\xa2\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xda"		=> "\xe8\xa2\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xdb"		=> "\xe8\xa2\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xdc"		=> "\xe8\xa2\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xdd"		=> "\xe8\xa2\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xde"		=> "\xe8\xa2\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xdf"		=> "\xe8\xa3\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe0"		=> "\xe8\xa3\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe1"		=> "\xe8\xa3\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe2"		=> "\xe8\xa3\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe3"		=> "\xe8\xa3\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe4"		=> "\xe8\xa3\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe5"		=> "\xe8\xa3\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe6"		=> "\xe8\xa4\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe7"		=> "\xe8\xa3\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe8"		=> "\xe8\xa3\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xe9"		=> "\xe8\xa3\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xea"		=> "\xe8\xa3\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xeb"		=> "\xe8\xa4\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xec"		=> "\xe8\xa4\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xed"		=> "\xe8\xa4\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xee"		=> "\xe8\xa4\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xef"		=> "\xe8\xa5\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf0"		=> "\xe8\xa4\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf1"		=> "\xe8\xa4\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf2"		=> "\xe8\xa4\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf3"		=> "\xe8\xa4\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf4"		=> "\xe8\xa5\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf5"		=> "\xe8\xa5\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf6"		=> "\xe8\xa4\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf7"		=> "\xe8\xa4\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf8"		=> "\xe8\xa4\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xf9"		=> "\xe8\xa5\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xfa"		=> "\xe8\xa4\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xfb"		=> "\xe8\xa5\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe5\xfc"		=> "\xe8\xa5\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x40"		=> "\xe8\xa5\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x41"		=> "\xe8\xa5\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x42"		=> "\xe8\xa5\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x43"		=> "\xe8\xa5\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x44"		=> "\xe8\xa5\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x45"		=> "\xe8\xa5\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x46"		=> "\xe8\xa5\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x47"		=> "\xe8\xa5\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x48"		=> "\xe8\xa6\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x49"		=> "\xe8\xa6\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x4a"		=> "\xe8\xa6\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x4b"		=> "\xe8\xa6\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x4c"		=> "\xe8\xa6\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x4d"		=> "\xe8\xa6\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x4e"		=> "\xe8\xa6\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x4f"		=> "\xe8\xa6\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x50"		=> "\xe8\xa6\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x51"		=> "\xe8\xa6\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x52"		=> "\xe8\xa6\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x53"		=> "\xe8\xa6\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x54"		=> "\xe8\xa6\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x55"		=> "\xe8\xa6\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x56"		=> "\xe8\xa7\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x57"		=> "\xe8\xa7\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x58"		=> "\xe8\xa7\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x59"		=> "\xe8\xa7\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x5a"		=> "\xe8\xa7\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x5b"		=> "\xe8\xa7\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x5c"		=> "\xe8\xa7\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x5d"		=> "\xe8\xa8\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x5e"		=> "\xe8\xa8\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x5f"		=> "\xe8\xa8\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x60"		=> "\xe8\xa8\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x61"		=> "\xe8\xa8\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x62"		=> "\xe8\xa8\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x63"		=> "\xe8\xa8\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x64"		=> "\xe8\xa8\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x65"		=> "\xe8\xa9\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x66"		=> "\xe8\xa9\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x67"		=> "\xe8\xa9\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x68"		=> "\xe8\xa9\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x69"		=> "\xe8\xa9\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x6a"		=> "\xe8\xa9\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x6b"		=> "\xe8\xa9\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x6c"		=> "\xe8\xa9\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x6d"		=> "\xe8\xa9\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x6e"		=> "\xe8\xaa\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x6f"		=> "\xe8\xaa\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x70"		=> "\xe8\xaa\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x71"		=> "\xe8\xaa\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x72"		=> "\xe8\xaa\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x73"		=> "\xe8\xaa\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x74"		=> "\xe8\xaa\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x75"		=> "\xe8\xaa\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x76"		=> "\xe8\xaa\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x77"		=> "\xe8\xaa\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x78"		=> "\xe8\xab\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x79"		=> "\xe8\xab\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x7a"		=> "\xe8\xab\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x7b"		=> "\xe8\xab\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x7c"		=> "\xe8\xab\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x7d"		=> "\xe8\xab\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x7e"		=> "\xe8\xab\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x80"		=> "\xe8\xab\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x81"		=> "\xe8\xab\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x82"		=> "\xe8\xac\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x83"		=> "\xe8\xab\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x84"		=> "\xe8\xab\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x85"		=> "\xe8\xab\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x86"		=> "\xe8\xab\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x87"		=> "\xe8\xab\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x88"		=> "\xe8\xac\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x89"		=> "\xe8\xac\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8a"		=> "\xe8\xac\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8b"		=> "\xe8\xab\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8c"		=> "\xe8\xac\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8d"		=> "\xe8\xac\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8e"		=> "\xe8\xac\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x8f"		=> "\xe8\xac\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x90"		=> "\xe8\xac\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x91"		=> "\xe9\x9e\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x92"		=> "\xe8\xac\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x93"		=> "\xe8\xac\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x94"		=> "\xe8\xac\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x95"		=> "\xe8\xac\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x96"		=> "\xe8\xad\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x97"		=> "\xe8\xad\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x98"		=> "\xe8\xad\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x99"		=> "\xe8\xad\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9a"		=> "\xe8\xad\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9b"		=> "\xe8\xad\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9c"		=> "\xe8\xad\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9d"		=> "\xe8\xad\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9e"		=> "\xe8\xad\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\x9f"		=> "\xe8\xad\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa0"		=> "\xe8\xad\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa1"		=> "\xe8\xad\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa2"		=> "\xe8\xad\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa3"		=> "\xe8\xad\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa4"		=> "\xe8\xae\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa5"		=> "\xe8\xae\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa6"		=> "\xe8\xae\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa7"		=> "\xe8\xae\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa8"		=> "\xe8\xae\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xa9"		=> "\xe8\xae\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaa"		=> "\xe8\xae\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xab"		=> "\xe8\xae\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xac"		=> "\xe8\xb0\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xad"		=> "\xe8\xb1\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xae"		=> "\xe8\xb0\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xaf"		=> "\xe8\xb1\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb0"		=> "\xe8\xb1\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb1"		=> "\xe8\xb1\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb2"		=> "\xe8\xb1\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb3"		=> "\xe8\xb1\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb4"		=> "\xe8\xb1\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb5"		=> "\xe8\xb1\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb6"		=> "\xe8\xb1\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb7"		=> "\xe8\xb1\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb8"		=> "\xe8\xb2\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xb9"		=> "\xe8\xb2\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xba"		=> "\xe8\xb2\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbb"		=> "\xe8\xb2\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbc"		=> "\xe8\xb2\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbd"		=> "\xe8\xb2\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbe"		=> "\xe8\xb2\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xbf"		=> "\xe8\xb1\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc0"		=> "\xe8\xb2\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc1"		=> "\xe6\x88\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc2"		=> "\xe8\xb2\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc3"		=> "\xe8\xb2\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc4"		=> "\xe8\xb2\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc5"		=> "\xe8\xb2\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc6"		=> "\xe8\xb2\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc7"		=> "\xe8\xb2\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc8"		=> "\xe8\xb2\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xc9"		=> "\xe8\xb3\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xca"		=> "\xe8\xb3\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xcb"		=> "\xe8\xb3\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xcc"		=> "\xe8\xb3\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xcd"		=> "\xe8\xb3\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xce"		=> "\xe8\xb3\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xcf"		=> "\xe8\xb3\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd0"		=> "\xe8\xb3\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd1"		=> "\xe8\xb4\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd2"		=> "\xe8\xb4\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd3"		=> "\xe8\xb4\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd4"		=> "\xe8\xb4\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd5"		=> "\xe8\xb4\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd6"		=> "\xe8\xb4\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd7"		=> "\xe8\xb4\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd8"		=> "\xe9\xbd\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xd9"		=> "\xe8\xb4\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xda"		=> "\xe8\xb3\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xdb"		=> "\xe8\xb4\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xdc"		=> "\xe8\xb4\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xdd"		=> "\xe8\xb5\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xde"		=> "\xe8\xb5\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xdf"		=> "\xe8\xb5\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe0"		=> "\xe8\xb5\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe1"		=> "\xe8\xb6\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe2"		=> "\xe8\xb6\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe3"		=> "\xe8\xb7\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe4"		=> "\xe8\xb6\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe5"		=> "\xe8\xb6\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe6"		=> "\xe8\xb7\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe7"		=> "\xe8\xb7\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe8"		=> "\xe8\xb7\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xe9"		=> "\xe8\xb7\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xea"		=> "\xe8\xb7\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xeb"		=> "\xe8\xb7\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xec"		=> "\xe8\xb7\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xed"		=> "\xe8\xb7\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xee"		=> "\xe8\xb7\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xef"		=> "\xe8\xb7\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf0"		=> "\xe8\xb7\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf1"		=> "\xe8\xb8\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf2"		=> "\xe8\xb8\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf3"		=> "\xe8\xb7\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf4"		=> "\xe8\xb8\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf5"		=> "\xe8\xb8\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf6"		=> "\xe8\xb8\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf7"		=> "\xe8\xb8\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf8"		=> "\xe8\xb9\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xf9"		=> "\xe8\xb8\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xfa"		=> "\xe8\xb8\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xfb"		=> "\xe8\xb8\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe6\xfc"		=> "\xe8\xb9\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x40"		=> "\xe8\xb9\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x41"		=> "\xe8\xb9\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x42"		=> "\xe8\xb9\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x43"		=> "\xe8\xb9\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x44"		=> "\xe8\xb9\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x45"		=> "\xe8\xb9\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x46"		=> "\xe8\xb9\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x47"		=> "\xe8\xb9\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x48"		=> "\xe8\xb8\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x49"		=> "\xe8\xb9\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x4a"		=> "\xe8\xb9\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x4b"		=> "\xe8\xb9\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x4c"		=> "\xe8\xb9\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x4d"		=> "\xe8\xb9\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x4e"		=> "\xe8\xba\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x4f"		=> "\xe8\xba\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x50"		=> "\xe8\xba\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x51"		=> "\xe8\xba\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x52"		=> "\xe8\xba\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x53"		=> "\xe8\xba\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x54"		=> "\xe8\xba\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x55"		=> "\xe8\xba\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x56"		=> "\xe8\xba\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x57"		=> "\xe8\xba\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x58"		=> "\xe8\xba\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x59"		=> "\xe8\xba\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x5a"		=> "\xe8\xba\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x5b"		=> "\xe8\xba\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x5c"		=> "\xe8\xbb\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x5d"		=> "\xe8\xba\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x5e"		=> "\xe8\xba\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x5f"		=> "\xe8\xbb\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x60"		=> "\xe8\xbb\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x61"		=> "\xe8\xbb\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x62"		=> "\xe8\xbb\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x63"		=> "\xe8\xbb\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x64"		=> "\xe8\xbb\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x65"		=> "\xe8\xbb\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x66"		=> "\xe8\xbb\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x67"		=> "\xe8\xbb\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x68"		=> "\xe8\xbc\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x69"		=> "\xe8\xbc\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x6a"		=> "\xe8\xbc\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x6b"		=> "\xe8\xbc\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x6c"		=> "\xe8\xbc\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x6d"		=> "\xe8\xbc\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x6e"		=> "\xe8\xbc\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x6f"		=> "\xe8\xbc\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x70"		=> "\xe8\xbc\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x71"		=> "\xe8\xbc\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x72"		=> "\xe8\xbc\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x73"		=> "\xe8\xbc\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x74"		=> "\xe8\xbc\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x75"		=> "\xe8\xbc\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x76"		=> "\xe8\xbd\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x77"		=> "\xe8\xbd\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x78"		=> "\xe8\xbc\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x79"		=> "\xe8\xbd\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x7a"		=> "\xe8\xbd\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x7b"		=> "\xe8\xbd\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x7c"		=> "\xe8\xbd\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x7d"		=> "\xe8\xbd\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x7e"		=> "\xe8\xbd\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x80"		=> "\xe8\xbd\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x81"		=> "\xe8\xbd\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x82"		=> "\xe8\xbd\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x83"		=> "\xe8\xbe\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x84"		=> "\xe8\xbe\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x85"		=> "\xe8\xbe\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x86"		=> "\xe8\xbe\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x87"		=> "\xe8\xbe\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x88"		=> "\xe8\xbe\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x89"		=> "\xe8\xbf\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8a"		=> "\xe8\xbf\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8b"		=> "\xe8\xbf\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8c"		=> "\xe8\xbf\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8d"		=> "\xe8\xbf\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8e"		=> "\xe9\x82\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x8f"		=> "\xe8\xbf\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x90"		=> "\xe9\x80\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x91"		=> "\xe8\xbf\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x92"		=> "\xe8\xbf\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x93"		=> "\xe9\x80\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x94"		=> "\xe9\x80\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x95"		=> "\xe9\x80\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x96"		=> "\xe9\x80\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x97"		=> "\xe9\x80\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x98"		=> "\xe9\x80\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x99"		=> "\xe9\x80\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9a"		=> "\xe9\x80\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9b"		=> "\xe9\x80\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9c"		=> "\xe9\x80\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9d"		=> "\xe9\x80\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9e"		=> "\xe8\xbf\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\x9f"		=> "\xe9\x81\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa0"		=> "\xe9\x81\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa1"		=> "\xe9\x81\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa2"		=> "\xe9\x81\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa3"		=> "\xe9\x80\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa4"		=> "\xe9\x81\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa5"		=> "\xe9\x80\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa6"		=> "\xe9\x81\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa7"		=> "\xe9\x81\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa8"		=> "\xe9\x81\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xa9"		=> "\xe9\x81\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaa"		=> "\xe9\x81\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xab"		=> "\xe9\x81\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xac"		=> "\xe9\x9a\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xad"		=> "\xe9\x81\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xae"		=> "\xe9\x82\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xaf"		=> "\xe9\x81\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb0"		=> "\xe9\x82\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb1"		=> "\xe9\x82\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb2"		=> "\xe9\x82\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb3"		=> "\xe9\x82\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb4"		=> "\xe9\x82\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb5"		=> "\xe9\x82\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb6"		=> "\xe9\x82\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb7"		=> "\xe9\x82\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb8"		=> "\xe9\x82\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xb9"		=> "\xe9\x83\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xba"		=> "\xe9\x83\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbb"		=> "\xe6\x89\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbc"		=> "\xe9\x83\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbd"		=> "\xe9\x84\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbe"		=> "\xe9\x84\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xbf"		=> "\xe9\x84\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc0"		=> "\xe9\x84\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc1"		=> "\xe9\x84\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc2"		=> "\xe9\x85\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc3"		=> "\xe9\x85\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc4"		=> "\xe9\x85\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc5"		=> "\xe9\x85\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc6"		=> "\xe9\x85\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc7"		=> "\xe9\x85\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc8"		=> "\xe9\x85\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xc9"		=> "\xe9\x85\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xca"		=> "\xe9\x86\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xcb"		=> "\xe9\x86\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xcc"		=> "\xe9\x86\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xcd"		=> "\xe9\x86\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xce"		=> "\xe9\x86\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xcf"		=> "\xe9\x86\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd0"		=> "\xe9\x86\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd1"		=> "\xe9\x86\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd2"		=> "\xe9\x86\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd3"		=> "\xe9\x86\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd4"		=> "\xe9\x87\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd5"		=> "\xe9\x87\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd6"		=> "\xe9\x87\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd7"		=> "\xe9\x87\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd8"		=> "\xe9\x87\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xd9"		=> "\xe9\x87\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xda"		=> "\xe9\x87\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xdb"		=> "\xe9\x87\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xdc"		=> "\xe9\x87\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xdd"		=> "\xe9\x87\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xde"		=> "\xe9\x87\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xdf"		=> "\xe9\x87\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe0"		=> "\xe9\x88\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe1"		=> "\xe9\x87\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe2"		=> "\xe9\x88\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe3"		=> "\xe9\x88\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe4"		=> "\xe9\x88\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe5"		=> "\xe9\x88\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe6"		=> "\xe9\x89\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe7"		=> "\xe9\x89\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe8"		=> "\xe9\x89\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xe9"		=> "\xe9\x89\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xea"		=> "\xe9\x89\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xeb"		=> "\xe9\x89\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xec"		=> "\xe9\x8a\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xed"		=> "\xe9\x88\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xee"		=> "\xe9\x89\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xef"		=> "\xe9\x89\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf0"		=> "\xe9\x8a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf1"		=> "\xe9\x8a\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf2"		=> "\xe9\x8a\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf3"		=> "\xe9\x8a\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf4"		=> "\xe9\x89\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf5"		=> "\xe9\x8b\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf6"		=> "\xe9\x8a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf7"		=> "\xe9\x8a\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf8"		=> "\xe9\x8b\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xf9"		=> "\xe9\x8c\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xfa"		=> "\xe9\x8b\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xfb"		=> "\xe9\x8d\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe7\xfc"		=> "\xe9\x8c\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x40"		=> "\xe9\x8c\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x41"		=> "\xe9\x8c\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x42"		=> "\xe9\x8c\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x43"		=> "\xe9\x8c\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x44"		=> "\xe9\x8c\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x45"		=> "\xe9\x8c\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x46"		=> "\xe9\x8c\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x47"		=> "\xe9\x8d\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x48"		=> "\xe9\x8d\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x49"		=> "\xe9\x8d\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x4a"		=> "\xe9\x8d\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x4b"		=> "\xe9\x8d\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x4c"		=> "\xe9\x8e\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x4d"		=> "\xe9\x8e\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x4e"		=> "\xe9\x8e\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x4f"		=> "\xe9\x8e\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x50"		=> "\xe9\x8e\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x51"		=> "\xe9\x8f\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x52"		=> "\xe9\x8f\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x53"		=> "\xe9\x8f\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x54"		=> "\xe9\x8f\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x55"		=> "\xe9\x8f\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x56"		=> "\xe9\x8f\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x57"		=> "\xe9\x8f\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x58"		=> "\xe9\x8f\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x59"		=> "\xe9\x8f\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x5a"		=> "\xe9\x8f\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x5b"		=> "\xe9\x90\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x5c"		=> "\xe9\x90\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x5d"		=> "\xe9\x90\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x5e"		=> "\xe9\x90\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x5f"		=> "\xe9\x90\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x60"		=> "\xe9\x90\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x61"		=> "\xe9\x90\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x62"		=> "\xe9\x90\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x63"		=> "\xe9\x90\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x64"		=> "\xe9\x90\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x65"		=> "\xe9\x90\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x66"		=> "\xe9\x91\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x67"		=> "\xe9\x91\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x68"		=> "\xe9\x91\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x69"		=> "\xe9\x91\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x6a"		=> "\xe9\x91\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x6b"		=> "\xe9\x91\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x6c"		=> "\xe9\x91\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x6d"		=> "\xe9\x91\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x6e"		=> "\xe9\x88\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x6f"		=> "\xe9\x91\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x70"		=> "\xe9\x91\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x71"		=> "\xe9\x91\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x72"		=> "\xe9\x91\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x73"		=> "\xe9\x91\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x74"		=> "\xe9\x91\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x75"		=> "\xe9\x91\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x76"		=> "\xe9\x92\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x77"		=> "\xe9\x91\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x78"		=> "\xe9\x96\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x79"		=> "\xe9\x96\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x7a"		=> "\xe9\x96\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x7b"		=> "\xe9\x96\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x7c"		=> "\xe9\x96\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x7d"		=> "\xe9\x96\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x7e"		=> "\xe9\x96\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x80"		=> "\xe9\x96\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x81"		=> "\xe9\x96\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x82"		=> "\xe9\x96\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x83"		=> "\xe9\x96\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x84"		=> "\xe9\x96\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x85"		=> "\xe9\x96\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x86"		=> "\xe9\x96\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x87"		=> "\xe9\x96\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x88"		=> "\xe9\x97\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x89"		=> "\xe6\xbf\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8a"		=> "\xe9\x97\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8b"		=> "\xe9\x97\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8c"		=> "\xe9\x97\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8d"		=> "\xe9\x97\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8e"		=> "\xe9\x97\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x8f"		=> "\xe9\x97\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x90"		=> "\xe9\x97\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x91"		=> "\xe9\x97\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x92"		=> "\xe9\x97\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x93"		=> "\xe9\x97\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x94"		=> "\xe9\x98\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x95"		=> "\xe9\x98\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x96"		=> "\xe9\x98\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x97"		=> "\xe9\x98\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x98"		=> "\xe9\x99\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x99"		=> "\xe9\x99\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9a"		=> "\xe9\x99\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9b"		=> "\xe9\x99\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9c"		=> "\xe9\x99\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9d"		=> "\xe9\x99\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9e"		=> "\xe9\x99\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\x9f"		=> "\xe9\x99\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa0"		=> "\xe9\x99\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa1"		=> "\xe9\x99\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa2"		=> "\xe9\x99\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa3"		=> "\xe9\x99\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa4"		=> "\xe9\x9a\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa5"		=> "\xe9\x9a\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa6"		=> "\xe9\x9a\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa7"		=> "\xe9\x9a\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa8"		=> "\xe9\x9a\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xa9"		=> "\xe9\x9a\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaa"		=> "\xe9\x9a\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xab"		=> "\xe9\x9a\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xac"		=> "\xe9\x9a\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xad"		=> "\xe9\x9a\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xae"		=> "\xe9\x9a\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xaf"		=> "\xe9\x9a\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb0"		=> "\xe9\x9a\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb1"		=> "\xe9\x9b\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb2"		=> "\xe9\x9b\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb3"		=> "\xe9\x9b\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb4"		=> "\xe9\x9b\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb5"		=> "\xe8\xa5\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb6"		=> "\xe9\x9b\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb7"		=> "\xe9\x9c\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb8"		=> "\xe9\x9b\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xb9"		=> "\xe9\x9b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xba"		=> "\xe9\x9c\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbb"		=> "\xe9\x9c\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbc"		=> "\xe9\x9c\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbd"		=> "\xe9\x9c\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbe"		=> "\xe9\x9c\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xbf"		=> "\xe9\x9c\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc0"		=> "\xe9\x9c\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc1"		=> "\xe9\x9c\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc2"		=> "\xe9\x9c\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc3"		=> "\xe9\x9c\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc4"		=> "\xe9\x9c\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc5"		=> "\xe9\x9c\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc6"		=> "\xe9\x9c\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc7"		=> "\xe9\x9c\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc8"		=> "\xe9\x9c\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xc9"		=> "\xe9\x9d\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xca"		=> "\xe9\x9d\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xcb"		=> "\xe9\x9d\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xcc"		=> "\xe9\x9d\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xcd"		=> "\xe9\x9d\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xce"		=> "\xe9\x9d\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xcf"		=> "\xe9\x9d\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd0"		=> "\xe9\x9d\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd1"		=> "\xe9\x9d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd2"		=> "\xe9\x9d\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd3"		=> "\xe5\x8b\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd4"		=> "\xe9\x9d\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd5"		=> "\xe9\x9d\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd6"		=> "\xe9\x9d\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd7"		=> "\xe9\x9e\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd8"		=> "\xe9\x9d\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xd9"		=> "\xe9\x9e\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xda"		=> "\xe9\x9d\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xdb"		=> "\xe9\x9e\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xdc"		=> "\xe9\x9e\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xdd"		=> "\xe9\x9e\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xde"		=> "\xe9\x9e\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xdf"		=> "\xe9\x9e\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe0"		=> "\xe9\x9e\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe1"		=> "\xe9\x9e\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe2"		=> "\xe9\x9e\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe3"		=> "\xe9\x9e\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe4"		=> "\xe9\x9e\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe5"		=> "\xe9\x9f\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe6"		=> "\xe9\x9f\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe7"		=> "\xe9\x9f\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe8"		=> "\xe9\x9f\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xe9"		=> "\xe9\x9f\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xea"		=> "\xe9\x9f\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xeb"		=> "\xe9\xbd\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xec"		=> "\xe9\x9f\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xed"		=> "\xe7\xab\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xee"		=> "\xe9\x9f\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xef"		=> "\xe9\x9f\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf0"		=> "\xe9\xa0\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf1"		=> "\xe9\xa0\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf2"		=> "\xe9\xa0\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf3"		=> "\xe9\xa0\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf4"		=> "\xe9\xa0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf5"		=> "\xe9\xa0\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf6"		=> "\xe9\xa0\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf7"		=> "\xe9\xa1\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf8"		=> "\xe9\xa1\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xf9"		=> "\xe9\xa1\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xfa"		=> "\xe9\xa1\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xfb"		=> "\xe9\xa1\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe8\xfc"		=> "\xe9\xa1\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x40"		=> "\xe9\xa1\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x41"		=> "\xe9\xa1\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x42"		=> "\xe9\xa1\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x43"		=> "\xe9\xa2\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x44"		=> "\xe9\xa2\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x45"		=> "\xe9\xa2\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x46"		=> "\xe9\xa2\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x47"		=> "\xe9\xa3\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x48"		=> "\xe9\xa3\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x49"		=> "\xe9\xa3\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x4a"		=> "\xe9\xa3\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x4b"		=> "\xe9\xa3\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x4c"		=> "\xe9\xa4\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x4d"		=> "\xe9\xa4\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x4e"		=> "\xe9\xa4\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x4f"		=> "\xe9\xa4\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x50"		=> "\xe9\xa4\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x51"		=> "\xe9\xa4\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x52"		=> "\xe9\xa4\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x53"		=> "\xe9\xa4\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x54"		=> "\xe9\xa4\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x55"		=> "\xe9\xa4\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x56"		=> "\xe9\xa4\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x57"		=> "\xe9\xa4\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x58"		=> "\xe9\xa4\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x59"		=> "\xe9\xa4\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x5a"		=> "\xe9\xa5\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x5b"		=> "\xe9\xa5\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x5c"		=> "\xe9\xa5\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x5d"		=> "\xe9\xa5\x90",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x5e"		=> "\xe9\xa5\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x5f"		=> "\xe9\xa5\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x60"		=> "\xe9\xa5\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x61"		=> "\xe9\xa5\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x62"		=> "\xe9\xa5\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x63"		=> "\xe9\xa6\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x64"		=> "\xe9\xa6\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x65"		=> "\xe9\xa6\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x66"		=> "\xe9\xa6\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x67"		=> "\xe9\xa6\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x68"		=> "\xe9\xa6\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x69"		=> "\xe9\xa7\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x6a"		=> "\xe9\xa7\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x6b"		=> "\xe9\xa7\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x6c"		=> "\xe9\xa7\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x6d"		=> "\xe9\xa7\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x6e"		=> "\xe9\xa7\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x6f"		=> "\xe9\xa7\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x70"		=> "\xe9\xa7\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x71"		=> "\xe9\xa7\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x72"		=> "\xe9\xa7\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x73"		=> "\xe9\xa7\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x74"		=> "\xe9\xa8\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x75"		=> "\xe9\xa8\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x76"		=> "\xe9\xa8\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x77"		=> "\xe9\xa7\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x78"		=> "\xe9\xa8\x99",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x79"		=> "\xe9\xa8\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x7a"		=> "\xe9\xa8\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x7b"		=> "\xe9\xa9\x85",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x7c"		=> "\xe9\xa9\x82",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x7d"		=> "\xe9\xa9\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x7e"		=> "\xe9\xa9\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x80"		=> "\xe9\xa8\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x81"		=> "\xe9\xa9\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x82"		=> "\xe9\xa9\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x83"		=> "\xe9\xa9\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x84"		=> "\xe9\xa9\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x85"		=> "\xe9\xa9\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x86"		=> "\xe9\xa9\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x87"		=> "\xe9\xa9\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x88"		=> "\xe9\xa9\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x89"		=> "\xe9\xa9\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8a"		=> "\xe9\xa9\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8b"		=> "\xe9\xa9\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8c"		=> "\xe9\xaa\xad",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8d"		=> "\xe9\xaa\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8e"		=> "\xe9\xaa\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x8f"		=> "\xe9\xab\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x90"		=> "\xe9\xab\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x91"		=> "\xe9\xab\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x92"		=> "\xe9\xab\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x93"		=> "\xe9\xab\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x94"		=> "\xe9\xab\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x95"		=> "\xe9\xab\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x96"		=> "\xe9\xab\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x97"		=> "\xe9\xab\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x98"		=> "\xe9\xab\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x99"		=> "\xe9\xab\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9a"		=> "\xe9\xab\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9b"		=> "\xe9\xab\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9c"		=> "\xe9\xab\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9d"		=> "\xe9\xab\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9e"		=> "\xe9\xab\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\x9f"		=> "\xe9\xab\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa0"		=> "\xe9\xac\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa1"		=> "\xe9\xac\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa2"		=> "\xe9\xac\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa3"		=> "\xe9\xac\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa4"		=> "\xe9\xac\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa5"		=> "\xe9\xac\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa6"		=> "\xe9\xac\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa7"		=> "\xe9\xac\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa8"		=> "\xe9\xac\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xa9"		=> "\xe9\xac\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaa"		=> "\xe9\xac\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xab"		=> "\xe9\xac\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xac"		=> "\xe9\xac\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xad"		=> "\xe9\xac\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xae"		=> "\xe9\xad\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xaf"		=> "\xe9\xad\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb0"		=> "\xe9\xad\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb1"		=> "\xe9\xad\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb2"		=> "\xe9\xad\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb3"		=> "\xe9\xad\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb4"		=> "\xe9\xad\x98",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb5"		=> "\xe9\xad\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb6"		=> "\xe9\xae\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb7"		=> "\xe9\xae\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb8"		=> "\xe9\xae\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xb9"		=> "\xe9\xae\x96",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xba"		=> "\xe9\xae\x97",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbb"		=> "\xe9\xae\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbc"		=> "\xe9\xae\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbd"		=> "\xe9\xae\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbe"		=> "\xe9\xae\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xbf"		=> "\xe9\xaf\x80",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc0"		=> "\xe9\xaf\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc1"		=> "\xe9\xae\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc2"		=> "\xe9\xaf\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc3"		=> "\xe9\xaf\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc4"		=> "\xe9\xaf\x91",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc5"		=> "\xe9\xaf\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc6"		=> "\xe9\xaf\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc7"		=> "\xe9\xaf\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc8"		=> "\xe9\xaf\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xc9"		=> "\xe9\xaf\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xca"		=> "\xe9\xaf\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xcb"		=> "\xe9\xb0\xba",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xcc"		=> "\xe9\xaf\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xcd"		=> "\xe9\xaf\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xce"		=> "\xe9\xaf\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xcf"		=> "\xe9\xb0\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd0"		=> "\xe9\xb0\x94",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd1"		=> "\xe9\xb0\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd2"		=> "\xe9\xb0\x93",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd3"		=> "\xe9\xb0\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd4"		=> "\xe9\xb0\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd5"		=> "\xe9\xb0\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd6"		=> "\xe9\xb0\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd7"		=> "\xe9\xb0\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd8"		=> "\xe9\xb0\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xd9"		=> "\xe9\xb0\xae",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xda"		=> "\xe9\xb0\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xdb"		=> "\xe9\xb0\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xdc"		=> "\xe9\xb0\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xdd"		=> "\xe9\xb0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xde"		=> "\xe9\xb0\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xdf"		=> "\xe9\xb1\x87",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe0"		=> "\xe9\xb0\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe1"		=> "\xe9\xb1\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe2"		=> "\xe9\xb0\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe3"		=> "\xe9\xb1\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe4"		=> "\xe9\xb1\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe5"		=> "\xe9\xb1\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe6"		=> "\xe9\xb1\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe7"		=> "\xe9\xb1\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe8"		=> "\xe9\xb3\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xe9"		=> "\xe9\xb3\xac",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xea"		=> "\xe9\xb3\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xeb"		=> "\xe9\xb4\x89",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xec"		=> "\xe9\xb4\x88",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xed"		=> "\xe9\xb3\xab",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xee"		=> "\xe9\xb4\x83",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xef"		=> "\xe9\xb4\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf0"		=> "\xe9\xb4\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf1"		=> "\xe9\xb4\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf2"		=> "\xe9\xb6\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf3"		=> "\xe9\xb4\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf4"		=> "\xe9\xb4\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf5"		=> "\xe9\xb5\x84",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf6"		=> "\xe9\xb4\x95",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf7"		=> "\xe9\xb4\x92",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf8"		=> "\xe9\xb5\x81",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xf9"		=> "\xe9\xb4\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xfa"		=> "\xe9\xb4\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xfb"		=> "\xe9\xb5\x86",	//CJK UNIFIED IDEOGRAPH
		"\xe9\xfc"		=> "\xe9\xb5\x88",	//CJK UNIFIED IDEOGRAPH
		"\xea\x40"		=> "\xe9\xb5\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xea\x41"		=> "\xe9\xb5\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xea\x42"		=> "\xe9\xb5\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xea\x43"		=> "\xe9\xb5\x91",	//CJK UNIFIED IDEOGRAPH
		"\xea\x44"		=> "\xe9\xb5\x90",	//CJK UNIFIED IDEOGRAPH
		"\xea\x45"		=> "\xe9\xb5\x99",	//CJK UNIFIED IDEOGRAPH
		"\xea\x46"		=> "\xe9\xb5\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xea\x47"		=> "\xe9\xb6\x89",	//CJK UNIFIED IDEOGRAPH
		"\xea\x48"		=> "\xe9\xb6\x87",	//CJK UNIFIED IDEOGRAPH
		"\xea\x49"		=> "\xe9\xb6\xab",	//CJK UNIFIED IDEOGRAPH
		"\xea\x4a"		=> "\xe9\xb5\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xea\x4b"		=> "\xe9\xb5\xba",	//CJK UNIFIED IDEOGRAPH
		"\xea\x4c"		=> "\xe9\xb6\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xea\x4d"		=> "\xe9\xb6\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xea\x4e"		=> "\xe9\xb6\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xea\x4f"		=> "\xe9\xb6\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xea\x50"		=> "\xe9\xb7\x84",	//CJK UNIFIED IDEOGRAPH
		"\xea\x51"		=> "\xe9\xb7\x81",	//CJK UNIFIED IDEOGRAPH
		"\xea\x52"		=> "\xe9\xb6\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xea\x53"		=> "\xe9\xb6\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xea\x54"		=> "\xe9\xb6\xba",	//CJK UNIFIED IDEOGRAPH
		"\xea\x55"		=> "\xe9\xb7\x86",	//CJK UNIFIED IDEOGRAPH
		"\xea\x56"		=> "\xe9\xb7\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xea\x57"		=> "\xe9\xb7\x82",	//CJK UNIFIED IDEOGRAPH
		"\xea\x58"		=> "\xe9\xb7\x99",	//CJK UNIFIED IDEOGRAPH
		"\xea\x59"		=> "\xe9\xb7\x93",	//CJK UNIFIED IDEOGRAPH
		"\xea\x5a"		=> "\xe9\xb7\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xea\x5b"		=> "\xe9\xb7\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xea\x5c"		=> "\xe9\xb7\xad",	//CJK UNIFIED IDEOGRAPH
		"\xea\x5d"		=> "\xe9\xb7\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xea\x5e"		=> "\xe9\xb7\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xea\x5f"		=> "\xe9\xb8\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xea\x60"		=> "\xe9\xb8\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xea\x61"		=> "\xe9\xb8\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xea\x62"		=> "\xe9\xb9\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xea\x63"		=> "\xe9\xb9\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xea\x64"		=> "\xe9\xb9\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xea\x65"		=> "\xe9\xba\x81",	//CJK UNIFIED IDEOGRAPH
		"\xea\x66"		=> "\xe9\xba\x88",	//CJK UNIFIED IDEOGRAPH
		"\xea\x67"		=> "\xe9\xba\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xea\x68"		=> "\xe9\xba\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xea\x69"		=> "\xe9\xba\x92",	//CJK UNIFIED IDEOGRAPH
		"\xea\x6a"		=> "\xe9\xba\x95",	//CJK UNIFIED IDEOGRAPH
		"\xea\x6b"		=> "\xe9\xba\x91",	//CJK UNIFIED IDEOGRAPH
		"\xea\x6c"		=> "\xe9\xba\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xea\x6d"		=> "\xe9\xba\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xea\x6e"		=> "\xe9\xba\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xea\x6f"		=> "\xe9\xba\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xea\x70"		=> "\xe9\xba\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xea\x71"		=> "\xe9\xba\xad",	//CJK UNIFIED IDEOGRAPH
		"\xea\x72"		=> "\xe9\x9d\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xea\x73"		=> "\xe9\xbb\x8c",	//CJK UNIFIED IDEOGRAPH
		"\xea\x74"		=> "\xe9\xbb\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xea\x75"		=> "\xe9\xbb\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xea\x76"		=> "\xe9\xbb\x90",	//CJK UNIFIED IDEOGRAPH
		"\xea\x77"		=> "\xe9\xbb\x94",	//CJK UNIFIED IDEOGRAPH
		"\xea\x78"		=> "\xe9\xbb\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xea\x79"		=> "\xe9\xbb\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xea\x7a"		=> "\xe9\xbb\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xea\x7b"		=> "\xe9\xbb\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xea\x7c"		=> "\xe9\xbb\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xea\x7d"		=> "\xe9\xbb\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xea\x7e"		=> "\xe9\xbb\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xea\x80"		=> "\xe9\xbb\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xea\x81"		=> "\xe9\xbb\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xea\x82"		=> "\xe9\xbb\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xea\x83"		=> "\xe9\xbb\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xea\x84"		=> "\xe9\xbb\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xea\x85"		=> "\xe9\xbb\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xea\x86"		=> "\xe9\xbb\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xea\x87"		=> "\xe9\xbc\x87",	//CJK UNIFIED IDEOGRAPH
		"\xea\x88"		=> "\xe9\xbc\x88",	//CJK UNIFIED IDEOGRAPH
		"\xea\x89"		=> "\xe7\x9a\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xea\x8a"		=> "\xe9\xbc\x95",	//CJK UNIFIED IDEOGRAPH
		"\xea\x8b"		=> "\xe9\xbc\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xea\x8c"		=> "\xe9\xbc\xac",	//CJK UNIFIED IDEOGRAPH
		"\xea\x8d"		=> "\xe9\xbc\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xea\x8e"		=> "\xe9\xbd\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xea\x8f"		=> "\xe9\xbd\x92",	//CJK UNIFIED IDEOGRAPH
		"\xea\x90"		=> "\xe9\xbd\x94",	//CJK UNIFIED IDEOGRAPH
		"\xea\x91"		=> "\xe9\xbd\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xea\x92"		=> "\xe9\xbd\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xea\x93"		=> "\xe9\xbd\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xea\x94"		=> "\xe9\xbd\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xea\x95"		=> "\xe9\xbd\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xea\x96"		=> "\xe9\xbd\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xea\x97"		=> "\xe9\xbd\xac",	//CJK UNIFIED IDEOGRAPH
		"\xea\x98"		=> "\xe9\xbd\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xea\x99"		=> "\xe9\xbd\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xea\x9a"		=> "\xe9\xbd\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xea\x9b"		=> "\xe9\xbd\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xea\x9c"		=> "\xe9\xbe\x95",	//CJK UNIFIED IDEOGRAPH
		"\xea\x9d"		=> "\xe9\xbe\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xea\x9e"		=> "\xe9\xbe\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xea\x9f"		=> "\xe5\xa0\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xea\xa0"		=> "\xe6\xa7\x87",	//CJK UNIFIED IDEOGRAPH
		"\xea\xa1"		=> "\xe9\x81\x99",	//CJK UNIFIED IDEOGRAPH
		"\xea\xa2"		=> "\xe7\x91\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xea\xa3"		=> "\xe5\x87\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xea\xa4"		=> "\xe7\x86\x99",	//CJK UNIFIED IDEOGRAPH
		"\xed\x40"		=> "\xe7\xba\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xed\x41"		=> "\xe8\xa4\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xed\x42"		=> "\xe9\x8d\x88",	//CJK UNIFIED IDEOGRAPH
		"\xed\x43"		=> "\xe9\x8a\x88",	//CJK UNIFIED IDEOGRAPH
		"\xed\x44"		=> "\xe8\x93\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xed\x45"		=> "\xe4\xbf\x89",	//CJK UNIFIED IDEOGRAPH
		"\xed\x46"		=> "\xe7\x82\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xed\x47"		=> "\xe6\x98\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xed\x48"		=> "\xe6\xa3\x88",	//CJK UNIFIED IDEOGRAPH
		"\xed\x49"		=> "\xe9\x8b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xed\x4a"		=> "\xe6\x9b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xed\x4b"		=> "\xe5\xbd\x85",	//CJK UNIFIED IDEOGRAPH
		"\xed\x4c"		=> "\xe4\xb8\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xed\x4d"		=> "\xe4\xbb\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xed\x4e"		=> "\xe4\xbb\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xed\x4f"		=> "\xe4\xbc\x80",	//CJK UNIFIED IDEOGRAPH
		"\xed\x50"		=> "\xe4\xbc\x83",	//CJK UNIFIED IDEOGRAPH
		"\xed\x51"		=> "\xe4\xbc\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xed\x52"		=> "\xe4\xbd\x96",	//CJK UNIFIED IDEOGRAPH
		"\xed\x53"		=> "\xe4\xbe\x92",	//CJK UNIFIED IDEOGRAPH
		"\xed\x54"		=> "\xe4\xbe\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xed\x55"		=> "\xe4\xbe\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xed\x56"		=> "\xe4\xbe\x94",	//CJK UNIFIED IDEOGRAPH
		"\xed\x57"		=> "\xe4\xbf\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xed\x58"		=> "\xe5\x81\x80",	//CJK UNIFIED IDEOGRAPH
		"\xed\x59"		=> "\xe5\x80\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xed\x5a"		=> "\xe4\xbf\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xed\x5b"		=> "\xe5\x80\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xed\x5c"		=> "\xe5\x81\x86",	//CJK UNIFIED IDEOGRAPH
		"\xed\x5d"		=> "\xe5\x81\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xed\x5e"		=> "\xe5\x81\x82",	//CJK UNIFIED IDEOGRAPH
		"\xed\x5f"		=> "\xe5\x82\x94",	//CJK UNIFIED IDEOGRAPH
		"\xed\x60"		=> "\xe5\x83\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xed\x61"		=> "\xe5\x83\x98",	//CJK UNIFIED IDEOGRAPH
		"\xed\x62"		=> "\xe5\x85\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xed\x63"		=> "\xe5\x85\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xed\x64"		=> "\xe5\x86\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xed\x65"		=> "\xe5\x86\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xed\x66"		=> "\xe5\x87\xac",	//CJK UNIFIED IDEOGRAPH
		"\xed\x67"		=> "\xe5\x88\x95",	//CJK UNIFIED IDEOGRAPH
		"\xed\x68"		=> "\xe5\x8a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xed\x69"		=> "\xe5\x8a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xed\x6a"		=> "\xe5\x8b\x80",	//CJK UNIFIED IDEOGRAPH
		"\xed\x6b"		=> "\xe5\x8b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xed\x6c"		=> "\xe5\x8c\x80",	//CJK UNIFIED IDEOGRAPH
		"\xed\x6d"		=> "\xe5\x8c\x87",	//CJK UNIFIED IDEOGRAPH
		"\xed\x6e"		=> "\xe5\x8c\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xed\x6f"		=> "\xe5\x8d\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xed\x70"		=> "\xe5\x8e\x93",	//CJK UNIFIED IDEOGRAPH
		"\xed\x71"		=> "\xe5\x8e\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xed\x72"		=> "\xe5\x8f\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xed\x73"		=> "\xef\xa8\x8e",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\x74"		=> "\xe5\x92\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xed\x75"		=> "\xe5\x92\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xed\x76"		=> "\xe5\x92\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xed\x77"		=> "\xe5\x93\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xed\x78"		=> "\xe5\x96\x86",	//CJK UNIFIED IDEOGRAPH
		"\xed\x79"		=> "\xe5\x9d\x99",	//CJK UNIFIED IDEOGRAPH
		"\xed\x7a"		=> "\xe5\x9d\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xed\x7b"		=> "\xe5\x9e\xac",	//CJK UNIFIED IDEOGRAPH
		"\xed\x7c"		=> "\xe5\x9f\x88",	//CJK UNIFIED IDEOGRAPH
		"\xed\x7d"		=> "\xe5\x9f\x87",	//CJK UNIFIED IDEOGRAPH
		"\xed\x7e"		=> "\xef\xa8\x8f",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\x80"		=> "\xef\xa8\x90",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\x81"		=> "\xe5\xa2\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xed\x82"		=> "\xe5\xa2\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xed\x83"		=> "\xe5\xa4\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xed\x84"		=> "\xe5\xa5\x93",	//CJK UNIFIED IDEOGRAPH
		"\xed\x85"		=> "\xe5\xa5\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xed\x86"		=> "\xe5\xa5\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xed\x87"		=> "\xe5\xa5\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xed\x88"		=> "\xe5\xa6\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xed\x89"		=> "\xe5\xa6\xba",	//CJK UNIFIED IDEOGRAPH
		"\xed\x8a"		=> "\xe5\xad\x96",	//CJK UNIFIED IDEOGRAPH
		"\xed\x8b"		=> "\xe5\xaf\x80",	//CJK UNIFIED IDEOGRAPH
		"\xed\x8c"		=> "\xe7\x94\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xed\x8d"		=> "\xe5\xaf\x98",	//CJK UNIFIED IDEOGRAPH
		"\xed\x8e"		=> "\xe5\xaf\xac",	//CJK UNIFIED IDEOGRAPH
		"\xed\x8f"		=> "\xe5\xb0\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xed\x90"		=> "\xe5\xb2\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xed\x91"		=> "\xe5\xb2\xba",	//CJK UNIFIED IDEOGRAPH
		"\xed\x92"		=> "\xe5\xb3\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xed\x93"		=> "\xe5\xb4\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xed\x94"		=> "\xe5\xb5\x93",	//CJK UNIFIED IDEOGRAPH
		"\xed\x95"		=> "\xef\xa8\x91",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\x96"		=> "\xe5\xb5\x82",	//CJK UNIFIED IDEOGRAPH
		"\xed\x97"		=> "\xe5\xb5\xad",	//CJK UNIFIED IDEOGRAPH
		"\xed\x98"		=> "\xe5\xb6\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xed\x99"		=> "\xe5\xb6\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xed\x9a"		=> "\xe5\xb7\x90",	//CJK UNIFIED IDEOGRAPH
		"\xed\x9b"		=> "\xe5\xbc\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xed\x9c"		=> "\xe5\xbc\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xed\x9d"		=> "\xe5\xbd\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xed\x9e"		=> "\xe5\xbe\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xed\x9f"		=> "\xe5\xbf\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa0"		=> "\xe6\x81\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa1"		=> "\xe6\x82\x85",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa2"		=> "\xe6\x82\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa3"		=> "\xe6\x83\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa4"		=> "\xe6\x83\x95",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa5"		=> "\xe6\x84\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa6"		=> "\xe6\x83\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa7"		=> "\xe6\x84\x91",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa8"		=> "\xe6\x84\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xed\xa9"		=> "\xe6\x84\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xed\xaa"		=> "\xe6\x86\x98",	//CJK UNIFIED IDEOGRAPH
		"\xed\xab"		=> "\xe6\x88\x93",	//CJK UNIFIED IDEOGRAPH
		"\xed\xac"		=> "\xe6\x8a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xed\xad"		=> "\xe6\x8f\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xed\xae"		=> "\xe6\x91\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xed\xaf"		=> "\xe6\x92\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb0"		=> "\xe6\x93\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb1"		=> "\xe6\x95\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb2"		=> "\xe6\x98\x80",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb3"		=> "\xe6\x98\x95",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb4"		=> "\xe6\x98\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb5"		=> "\xe6\x98\x89",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb6"		=> "\xe6\x98\xae",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb7"		=> "\xe6\x98\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb8"		=> "\xe6\x98\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xed\xb9"		=> "\xe6\x99\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xed\xba"		=> "\xe6\x99\x97",	//CJK UNIFIED IDEOGRAPH
		"\xed\xbb"		=> "\xe6\x99\x99",	//CJK UNIFIED IDEOGRAPH
		"\xed\xbc"		=> "\xef\xa8\x92",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\xbd"		=> "\xe6\x99\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xed\xbe"		=> "\xe6\x9a\x99",	//CJK UNIFIED IDEOGRAPH
		"\xed\xbf"		=> "\xe6\x9a\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc0"		=> "\xe6\x9a\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc1"		=> "\xe6\x9a\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc2"		=> "\xe6\x9b\xba",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc3"		=> "\xe6\x9c\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc4"		=> "\xef\xa4\xa9",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\xc5"		=> "\xe6\x9d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc6"		=> "\xe6\x9e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc7"		=> "\xe6\xa1\x92",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc8"		=> "\xe6\x9f\x80",	//CJK UNIFIED IDEOGRAPH
		"\xed\xc9"		=> "\xe6\xa0\x81",	//CJK UNIFIED IDEOGRAPH
		"\xed\xca"		=> "\xe6\xa1\x84",	//CJK UNIFIED IDEOGRAPH
		"\xed\xcb"		=> "\xe6\xa3\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xed\xcc"		=> "\xef\xa8\x93",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\xcd"		=> "\xe6\xa5\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xed\xce"		=> "\xef\xa8\x94",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\xcf"		=> "\xe6\xa6\x98",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd0"		=> "\xe6\xa7\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd1"		=> "\xe6\xa8\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd2"		=> "\xe6\xa9\xab",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd3"		=> "\xe6\xa9\x86",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd4"		=> "\xe6\xa9\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd5"		=> "\xe6\xa9\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd6"		=> "\xe6\xab\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd7"		=> "\xe6\xab\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd8"		=> "\xe6\xaf\x96",	//CJK UNIFIED IDEOGRAPH
		"\xed\xd9"		=> "\xe6\xb0\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xed\xda"		=> "\xe6\xb1\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xed\xdb"		=> "\xe6\xb2\x86",	//CJK UNIFIED IDEOGRAPH
		"\xed\xdc"		=> "\xe6\xb1\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xed\xdd"		=> "\xe6\xb3\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xed\xde"		=> "\xe6\xb4\x84",	//CJK UNIFIED IDEOGRAPH
		"\xed\xdf"		=> "\xe6\xb6\x87",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe0"		=> "\xe6\xb5\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe1"		=> "\xe6\xb6\x96",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe2"		=> "\xe6\xb6\xac",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe3"		=> "\xe6\xb7\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe4"		=> "\xe6\xb7\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe5"		=> "\xe6\xb7\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe6"		=> "\xe6\xb7\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe7"		=> "\xe6\xb8\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe8"		=> "\xe6\xb9\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xed\xe9"		=> "\xe6\xb8\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xed\xea"		=> "\xe6\xb8\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xed\xeb"		=> "\xe6\xba\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xed\xec"		=> "\xe6\xbe\x88",	//CJK UNIFIED IDEOGRAPH
		"\xed\xed"		=> "\xe6\xbe\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xed\xee"		=> "\xe6\xbf\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xed\xef"		=> "\xe7\x80\x85",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf0"		=> "\xe7\x80\x87",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf1"		=> "\xe7\x80\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf2"		=> "\xe7\x82\x85",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf3"		=> "\xe7\x82\xab",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf4"		=> "\xe7\x84\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf5"		=> "\xe7\x84\x84",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf6"		=> "\xe7\x85\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf7"		=> "\xe7\x85\x86",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf8"		=> "\xe7\x85\x87",	//CJK UNIFIED IDEOGRAPH
		"\xed\xf9"		=> "\xef\xa8\x95",	//CJK COMPATIBILITY IDEOGRAPH
		"\xed\xfa"		=> "\xe7\x87\x81",	//CJK UNIFIED IDEOGRAPH
		"\xed\xfb"		=> "\xe7\x87\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xed\xfc"		=> "\xe7\x8a\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xee\x40"		=> "\xe7\x8a\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xee\x41"		=> "\xe7\x8c\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xee\x42"		=> "\xef\xa8\x96",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x43"		=> "\xe7\x8d\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xee\x44"		=> "\xe7\x8e\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xee\x45"		=> "\xe7\x8f\x89",	//CJK UNIFIED IDEOGRAPH
		"\xee\x46"		=> "\xe7\x8f\x96",	//CJK UNIFIED IDEOGRAPH
		"\xee\x47"		=> "\xe7\x8f\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xee\x48"		=> "\xe7\x8f\x92",	//CJK UNIFIED IDEOGRAPH
		"\xee\x49"		=> "\xe7\x90\x87",	//CJK UNIFIED IDEOGRAPH
		"\xee\x4a"		=> "\xe7\x8f\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xee\x4b"		=> "\xe7\x90\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xee\x4c"		=> "\xe7\x90\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xee\x4d"		=> "\xe7\x90\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xee\x4e"		=> "\xe7\x90\xae",	//CJK UNIFIED IDEOGRAPH
		"\xee\x4f"		=> "\xe7\x91\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xee\x50"		=> "\xe7\x92\x89",	//CJK UNIFIED IDEOGRAPH
		"\xee\x51"		=> "\xe7\x92\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xee\x52"		=> "\xe7\x94\x81",	//CJK UNIFIED IDEOGRAPH
		"\xee\x53"		=> "\xe7\x95\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xee\x54"		=> "\xe7\x9a\x82",	//CJK UNIFIED IDEOGRAPH
		"\xee\x55"		=> "\xe7\x9a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xee\x56"		=> "\xe7\x9a\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xee\x57"		=> "\xe7\x9a\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xee\x58"		=> "\xe7\x9a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xee\x59"		=> "\xef\xa8\x97",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x5a"		=> "\xe7\x9d\x86",	//CJK UNIFIED IDEOGRAPH
		"\xee\x5b"		=> "\xe5\x8a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xee\x5c"		=> "\xe7\xa0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xee\x5d"		=> "\xe7\xa1\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xee\x5e"		=> "\xe7\xa1\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xee\x5f"		=> "\xe7\xa1\xba",	//CJK UNIFIED IDEOGRAPH
		"\xee\x60"		=> "\xe7\xa4\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xee\x61"		=> "\xef\xa8\x98",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x62"		=> "\xef\xa8\x99",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x63"		=> "\xef\xa8\x9a",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x64"		=> "\xe7\xa6\x94",	//CJK UNIFIED IDEOGRAPH
		"\xee\x65"		=> "\xef\xa8\x9b",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x66"		=> "\xe7\xa6\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xee\x67"		=> "\xe7\xab\x91",	//CJK UNIFIED IDEOGRAPH
		"\xee\x68"		=> "\xe7\xab\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xee\x69"		=> "\xef\xa8\x9c",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x6a"		=> "\xe7\xab\xab",	//CJK UNIFIED IDEOGRAPH
		"\xee\x6b"		=> "\xe7\xae\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xee\x6c"		=> "\xef\xa8\x9d",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x6d"		=> "\xe7\xb5\x88",	//CJK UNIFIED IDEOGRAPH
		"\xee\x6e"		=> "\xe7\xb5\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xee\x6f"		=> "\xe7\xb6\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xee\x70"		=> "\xe7\xb6\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xee\x71"		=> "\xe7\xb7\x96",	//CJK UNIFIED IDEOGRAPH
		"\xee\x72"		=> "\xe7\xb9\x92",	//CJK UNIFIED IDEOGRAPH
		"\xee\x73"		=> "\xe7\xbd\x87",	//CJK UNIFIED IDEOGRAPH
		"\xee\x74"		=> "\xe7\xbe\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xee\x75"		=> "\xef\xa8\x9e",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x76"		=> "\xe8\x8c\x81",	//CJK UNIFIED IDEOGRAPH
		"\xee\x77"		=> "\xe8\x8d\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xee\x78"		=> "\xe8\x8d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xee\x79"		=> "\xe8\x8f\x87",	//CJK UNIFIED IDEOGRAPH
		"\xee\x7a"		=> "\xe8\x8f\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xee\x7b"		=> "\xe8\x91\x88",	//CJK UNIFIED IDEOGRAPH
		"\xee\x7c"		=> "\xe8\x92\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xee\x7d"		=> "\xe8\x95\x93",	//CJK UNIFIED IDEOGRAPH
		"\xee\x7e"		=> "\xe8\x95\x99",	//CJK UNIFIED IDEOGRAPH
		"\xee\x80"		=> "\xe8\x95\xab",	//CJK UNIFIED IDEOGRAPH
		"\xee\x81"		=> "\xef\xa8\x9f",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x82"		=> "\xe8\x96\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xee\x83"		=> "\xef\xa8\xa0",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x84"		=> "\xef\xa8\xa1",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x85"		=> "\xe8\xa0\x87",	//CJK UNIFIED IDEOGRAPH
		"\xee\x86"		=> "\xe8\xa3\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xee\x87"		=> "\xe8\xa8\x92",	//CJK UNIFIED IDEOGRAPH
		"\xee\x88"		=> "\xe8\xa8\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xee\x89"		=> "\xe8\xa9\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xee\x8a"		=> "\xe8\xaa\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xee\x8b"		=> "\xe8\xaa\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xee\x8c"		=> "\xe8\xab\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xee\x8d"		=> "\xef\xa8\xa2",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x8e"		=> "\xe8\xab\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xee\x8f"		=> "\xe8\xad\x93",	//CJK UNIFIED IDEOGRAPH
		"\xee\x90"		=> "\xe8\xad\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xee\x91"		=> "\xe8\xb3\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xee\x92"		=> "\xe8\xb3\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xee\x93"		=> "\xe8\xb4\x92",	//CJK UNIFIED IDEOGRAPH
		"\xee\x94"		=> "\xe8\xb5\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xee\x95"		=> "\xef\xa8\xa3",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x96"		=> "\xe8\xbb\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xee\x97"		=> "\xef\xa8\xa4",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x98"		=> "\xef\xa8\xa5",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x99"		=> "\xe9\x81\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xee\x9a"		=> "\xe9\x83\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xee\x9b"		=> "\xef\xa8\xa6",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\x9c"		=> "\xe9\x84\x95",	//CJK UNIFIED IDEOGRAPH
		"\xee\x9d"		=> "\xe9\x84\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xee\x9e"		=> "\xe9\x87\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xee\x9f"		=> "\xe9\x87\x97",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa0"		=> "\xe9\x87\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa1"		=> "\xe9\x87\xad",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa2"		=> "\xe9\x87\xae",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa3"		=> "\xe9\x87\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa4"		=> "\xe9\x87\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa5"		=> "\xe9\x88\x86",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa6"		=> "\xe9\x88\x90",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa7"		=> "\xe9\x88\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa8"		=> "\xe9\x88\xba",	//CJK UNIFIED IDEOGRAPH
		"\xee\xa9"		=> "\xe9\x89\x80",	//CJK UNIFIED IDEOGRAPH
		"\xee\xaa"		=> "\xe9\x88\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xee\xab"		=> "\xe9\x89\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xee\xac"		=> "\xe9\x89\x99",	//CJK UNIFIED IDEOGRAPH
		"\xee\xad"		=> "\xe9\x89\x91",	//CJK UNIFIED IDEOGRAPH
		"\xee\xae"		=> "\xe9\x88\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xee\xaf"		=> "\xe9\x89\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb0"		=> "\xe9\x8a\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb1"		=> "\xe9\x89\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb2"		=> "\xe9\x89\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb3"		=> "\xe9\x8b\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb4"		=> "\xe9\x8b\x97",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb5"		=> "\xe9\x8b\x99",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb6"		=> "\xe9\x8b\x90",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb7"		=> "\xef\xa8\xa7",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\xb8"		=> "\xe9\x8b\x95",	//CJK UNIFIED IDEOGRAPH
		"\xee\xb9"		=> "\xe9\x8b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xee\xba"		=> "\xe9\x8b\x93",	//CJK UNIFIED IDEOGRAPH
		"\xee\xbb"		=> "\xe9\x8c\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xee\xbc"		=> "\xe9\x8c\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xee\xbd"		=> "\xe9\x8b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xee\xbe"		=> "\xef\xa8\xa8",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\xbf"		=> "\xe9\x8c\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc0"		=> "\xe9\x8b\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc1"		=> "\xe9\x8c\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc2"		=> "\xe9\x8c\x82",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc3"		=> "\xe9\x8d\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc4"		=> "\xe9\x8d\x97",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc5"		=> "\xe9\x8e\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc6"		=> "\xe9\x8f\x86",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc7"		=> "\xe9\x8f\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc8"		=> "\xe9\x8f\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xee\xc9"		=> "\xe9\x90\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xee\xca"		=> "\xe9\x91\x85",	//CJK UNIFIED IDEOGRAPH
		"\xee\xcb"		=> "\xe9\x91\x88",	//CJK UNIFIED IDEOGRAPH
		"\xee\xcc"		=> "\xe9\x96\x92",	//CJK UNIFIED IDEOGRAPH
		"\xee\xcd"		=> "\xef\xa7\x9c",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\xce"		=> "\xef\xa8\xa9",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\xcf"		=> "\xe9\x9a\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd0"		=> "\xe9\x9a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd1"		=> "\xe9\x9c\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd2"		=> "\xe9\x9c\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd3"		=> "\xe9\x9d\x83",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd4"		=> "\xe9\x9d\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd5"		=> "\xe9\x9d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd6"		=> "\xe9\x9d\x91",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd7"		=> "\xe9\x9d\x95",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd8"		=> "\xe9\xa1\x97",	//CJK UNIFIED IDEOGRAPH
		"\xee\xd9"		=> "\xe9\xa1\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xee\xda"		=> "\xef\xa8\xaa",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\xdb"		=> "\xef\xa8\xab",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\xdc"		=> "\xe9\xa4\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xee\xdd"		=> "\xef\xa8\xac",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\xde"		=> "\xe9\xa6\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xee\xdf"		=> "\xe9\xa9\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe0"		=> "\xe9\xab\x99",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe1"		=> "\xe9\xab\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe2"		=> "\xe9\xad\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe3"		=> "\xe9\xad\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe4"		=> "\xe9\xae\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe5"		=> "\xe9\xae\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe6"		=> "\xe9\xae\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe7"		=> "\xe9\xb0\x80",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe8"		=> "\xe9\xb5\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xee\xe9"		=> "\xe9\xb5\xab",	//CJK UNIFIED IDEOGRAPH
		"\xee\xea"		=> "\xef\xa8\xad",	//CJK COMPATIBILITY IDEOGRAPH
		"\xee\xeb"		=> "\xe9\xb8\x99",	//CJK UNIFIED IDEOGRAPH
		"\xee\xec"		=> "\xe9\xbb\x91",	//CJK UNIFIED IDEOGRAPH
		"\xee\xef"		=> "\xe2\x85\xb0",	//SMALL ROMAN NUMERAL ONE
		"\xee\xf0"		=> "\xe2\x85\xb1",	//SMALL ROMAN NUMERAL TWO
		"\xee\xf1"		=> "\xe2\x85\xb2",	//SMALL ROMAN NUMERAL THREE
		"\xee\xf2"		=> "\xe2\x85\xb3",	//SMALL ROMAN NUMERAL FOUR
		"\xee\xf3"		=> "\xe2\x85\xb4",	//SMALL ROMAN NUMERAL FIVE
		"\xee\xf4"		=> "\xe2\x85\xb5",	//SMALL ROMAN NUMERAL SIX
		"\xee\xf5"		=> "\xe2\x85\xb6",	//SMALL ROMAN NUMERAL SEVEN
		"\xee\xf6"		=> "\xe2\x85\xb7",	//SMALL ROMAN NUMERAL EIGHT
		"\xee\xf7"		=> "\xe2\x85\xb8",	//SMALL ROMAN NUMERAL NINE
		"\xee\xf8"		=> "\xe2\x85\xb9",	//SMALL ROMAN NUMERAL TEN
		"\xee\xf9"		=> "\xef\xbf\xa2",	//FULLWIDTH NOT SIGN
		"\xee\xfa"		=> "\xef\xbf\xa4",	//FULLWIDTH BROKEN BAR
		"\xee\xfb"		=> "\xef\xbc\x87",	//FULLWIDTH APOSTROPHE
		"\xee\xfc"		=> "\xef\xbc\x82",	//FULLWIDTH QUOTATION MARK
		"\xfa\x40"		=> "\xe2\x85\xb0",	//SMALL ROMAN NUMERAL ONE
		"\xfa\x41"		=> "\xe2\x85\xb1",	//SMALL ROMAN NUMERAL TWO
		"\xfa\x42"		=> "\xe2\x85\xb2",	//SMALL ROMAN NUMERAL THREE
		"\xfa\x43"		=> "\xe2\x85\xb3",	//SMALL ROMAN NUMERAL FOUR
		"\xfa\x44"		=> "\xe2\x85\xb4",	//SMALL ROMAN NUMERAL FIVE
		"\xfa\x45"		=> "\xe2\x85\xb5",	//SMALL ROMAN NUMERAL SIX
		"\xfa\x46"		=> "\xe2\x85\xb6",	//SMALL ROMAN NUMERAL SEVEN
		"\xfa\x47"		=> "\xe2\x85\xb7",	//SMALL ROMAN NUMERAL EIGHT
		"\xfa\x48"		=> "\xe2\x85\xb8",	//SMALL ROMAN NUMERAL NINE
		"\xfa\x49"		=> "\xe2\x85\xb9",	//SMALL ROMAN NUMERAL TEN
		"\xfa\x4a"		=> "\xe2\x85\xa0",	//ROMAN NUMERAL ONE
		"\xfa\x4b"		=> "\xe2\x85\xa1",	//ROMAN NUMERAL TWO
		"\xfa\x4c"		=> "\xe2\x85\xa2",	//ROMAN NUMERAL THREE
		"\xfa\x4d"		=> "\xe2\x85\xa3",	//ROMAN NUMERAL FOUR
		"\xfa\x4e"		=> "\xe2\x85\xa4",	//ROMAN NUMERAL FIVE
		"\xfa\x4f"		=> "\xe2\x85\xa5",	//ROMAN NUMERAL SIX
		"\xfa\x50"		=> "\xe2\x85\xa6",	//ROMAN NUMERAL SEVEN
		"\xfa\x51"		=> "\xe2\x85\xa7",	//ROMAN NUMERAL EIGHT
		"\xfa\x52"		=> "\xe2\x85\xa8",	//ROMAN NUMERAL NINE
		"\xfa\x53"		=> "\xe2\x85\xa9",	//ROMAN NUMERAL TEN
		"\xfa\x54"		=> "\xef\xbf\xa2",	//FULLWIDTH NOT SIGN
		"\xfa\x55"		=> "\xef\xbf\xa4",	//FULLWIDTH BROKEN BAR
		"\xfa\x56"		=> "\xef\xbc\x87",	//FULLWIDTH APOSTROPHE
		"\xfa\x57"		=> "\xef\xbc\x82",	//FULLWIDTH QUOTATION MARK
		"\xfa\x58"		=> "\xe3\x88\xb1",	//PARENTHESIZED IDEOGRAPH STOCK
		"\xfa\x59"		=> "\xe2\x84\x96",	//NUMERO SIGN
		"\xfa\x5a"		=> "\xe2\x84\xa1",	//TELEPHONE SIGN
		"\xfa\x5b"		=> "\xe2\x88\xb5",	//BECAUSE
		"\xfa\x5c"		=> "\xe7\xba\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x5d"		=> "\xe8\xa4\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x5e"		=> "\xe9\x8d\x88",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x5f"		=> "\xe9\x8a\x88",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x60"		=> "\xe8\x93\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x61"		=> "\xe4\xbf\x89",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x62"		=> "\xe7\x82\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x63"		=> "\xe6\x98\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x64"		=> "\xe6\xa3\x88",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x65"		=> "\xe9\x8b\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x66"		=> "\xe6\x9b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x67"		=> "\xe5\xbd\x85",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x68"		=> "\xe4\xb8\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x69"		=> "\xe4\xbb\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x6a"		=> "\xe4\xbb\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x6b"		=> "\xe4\xbc\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x6c"		=> "\xe4\xbc\x83",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x6d"		=> "\xe4\xbc\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x6e"		=> "\xe4\xbd\x96",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x6f"		=> "\xe4\xbe\x92",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x70"		=> "\xe4\xbe\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x71"		=> "\xe4\xbe\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x72"		=> "\xe4\xbe\x94",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x73"		=> "\xe4\xbf\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x74"		=> "\xe5\x81\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x75"		=> "\xe5\x80\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x76"		=> "\xe4\xbf\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x77"		=> "\xe5\x80\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x78"		=> "\xe5\x81\x86",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x79"		=> "\xe5\x81\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x7a"		=> "\xe5\x81\x82",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x7b"		=> "\xe5\x82\x94",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x7c"		=> "\xe5\x83\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x7d"		=> "\xe5\x83\x98",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x7e"		=> "\xe5\x85\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x80"		=> "\xe5\x85\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x81"		=> "\xe5\x86\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x82"		=> "\xe5\x86\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x83"		=> "\xe5\x87\xac",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x84"		=> "\xe5\x88\x95",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x85"		=> "\xe5\x8a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x86"		=> "\xe5\x8a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x87"		=> "\xe5\x8b\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x88"		=> "\xe5\x8b\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x89"		=> "\xe5\x8c\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x8a"		=> "\xe5\x8c\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x8b"		=> "\xe5\x8c\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x8c"		=> "\xe5\x8d\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x8d"		=> "\xe5\x8e\x93",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x8e"		=> "\xe5\x8e\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x8f"		=> "\xe5\x8f\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x90"		=> "\xef\xa8\x8e",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfa\x91"		=> "\xe5\x92\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x92"		=> "\xe5\x92\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x93"		=> "\xe5\x92\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x94"		=> "\xe5\x93\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x95"		=> "\xe5\x96\x86",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x96"		=> "\xe5\x9d\x99",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x97"		=> "\xe5\x9d\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x98"		=> "\xe5\x9e\xac",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x99"		=> "\xe5\x9f\x88",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x9a"		=> "\xe5\x9f\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x9b"		=> "\xef\xa8\x8f",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfa\x9c"		=> "\xef\xa8\x90",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfa\x9d"		=> "\xe5\xa2\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x9e"		=> "\xe5\xa2\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xfa\x9f"		=> "\xe5\xa4\x8b",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa0"		=> "\xe5\xa5\x93",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa1"		=> "\xe5\xa5\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa2"		=> "\xe5\xa5\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa3"		=> "\xe5\xa5\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa4"		=> "\xe5\xa6\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa5"		=> "\xe5\xa6\xba",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa6"		=> "\xe5\xad\x96",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa7"		=> "\xe5\xaf\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa8"		=> "\xe7\x94\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xa9"		=> "\xe5\xaf\x98",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xaa"		=> "\xe5\xaf\xac",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xab"		=> "\xe5\xb0\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xac"		=> "\xe5\xb2\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xad"		=> "\xe5\xb2\xba",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xae"		=> "\xe5\xb3\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xaf"		=> "\xe5\xb4\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb0"		=> "\xe5\xb5\x93",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb1"		=> "\xef\xa8\x91",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfa\xb2"		=> "\xe5\xb5\x82",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb3"		=> "\xe5\xb5\xad",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb4"		=> "\xe5\xb6\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb5"		=> "\xe5\xb6\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb6"		=> "\xe5\xb7\x90",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb7"		=> "\xe5\xbc\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb8"		=> "\xe5\xbc\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xb9"		=> "\xe5\xbd\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xba"		=> "\xe5\xbe\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xbb"		=> "\xe5\xbf\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xbc"		=> "\xe6\x81\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xbd"		=> "\xe6\x82\x85",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xbe"		=> "\xe6\x82\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xbf"		=> "\xe6\x83\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc0"		=> "\xe6\x83\x95",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc1"		=> "\xe6\x84\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc2"		=> "\xe6\x83\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc3"		=> "\xe6\x84\x91",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc4"		=> "\xe6\x84\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc5"		=> "\xe6\x84\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc6"		=> "\xe6\x86\x98",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc7"		=> "\xe6\x88\x93",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc8"		=> "\xe6\x8a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xc9"		=> "\xe6\x8f\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xca"		=> "\xe6\x91\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xcb"		=> "\xe6\x92\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xcc"		=> "\xe6\x93\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xcd"		=> "\xe6\x95\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xce"		=> "\xe6\x98\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xcf"		=> "\xe6\x98\x95",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd0"		=> "\xe6\x98\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd1"		=> "\xe6\x98\x89",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd2"		=> "\xe6\x98\xae",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd3"		=> "\xe6\x98\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd4"		=> "\xe6\x98\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd5"		=> "\xe6\x99\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd6"		=> "\xe6\x99\x97",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd7"		=> "\xe6\x99\x99",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xd8"		=> "\xef\xa8\x92",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfa\xd9"		=> "\xe6\x99\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xda"		=> "\xe6\x9a\x99",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xdb"		=> "\xe6\x9a\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xdc"		=> "\xe6\x9a\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xdd"		=> "\xe6\x9a\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xde"		=> "\xe6\x9b\xba",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xdf"		=> "\xe6\x9c\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xe0"		=> "\xef\xa4\xa9",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfa\xe1"		=> "\xe6\x9d\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xe2"		=> "\xe6\x9e\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xe3"		=> "\xe6\xa1\x92",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xe4"		=> "\xe6\x9f\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xe5"		=> "\xe6\xa0\x81",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xe6"		=> "\xe6\xa1\x84",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xe7"		=> "\xe6\xa3\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xe8"		=> "\xef\xa8\x93",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfa\xe9"		=> "\xe6\xa5\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xea"		=> "\xef\xa8\x94",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfa\xeb"		=> "\xe6\xa6\x98",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xec"		=> "\xe6\xa7\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xed"		=> "\xe6\xa8\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xee"		=> "\xe6\xa9\xab",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xef"		=> "\xe6\xa9\x86",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf0"		=> "\xe6\xa9\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf1"		=> "\xe6\xa9\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf2"		=> "\xe6\xab\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf3"		=> "\xe6\xab\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf4"		=> "\xe6\xaf\x96",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf5"		=> "\xe6\xb0\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf6"		=> "\xe6\xb1\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf7"		=> "\xe6\xb2\x86",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf8"		=> "\xe6\xb1\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xf9"		=> "\xe6\xb3\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xfa"		=> "\xe6\xb4\x84",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xfb"		=> "\xe6\xb6\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfa\xfc"		=> "\xe6\xb5\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x40"		=> "\xe6\xb6\x96",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x41"		=> "\xe6\xb6\xac",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x42"		=> "\xe6\xb7\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x43"		=> "\xe6\xb7\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x44"		=> "\xe6\xb7\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x45"		=> "\xe6\xb7\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x46"		=> "\xe6\xb8\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x47"		=> "\xe6\xb9\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x48"		=> "\xe6\xb8\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x49"		=> "\xe6\xb8\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x4a"		=> "\xe6\xba\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x4b"		=> "\xe6\xbe\x88",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x4c"		=> "\xe6\xbe\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x4d"		=> "\xe6\xbf\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x4e"		=> "\xe7\x80\x85",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x4f"		=> "\xe7\x80\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x50"		=> "\xe7\x80\xa8",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x51"		=> "\xe7\x82\x85",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x52"		=> "\xe7\x82\xab",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x53"		=> "\xe7\x84\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x54"		=> "\xe7\x84\x84",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x55"		=> "\xe7\x85\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x56"		=> "\xe7\x85\x86",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x57"		=> "\xe7\x85\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x58"		=> "\xef\xa8\x95",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x59"		=> "\xe7\x87\x81",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x5a"		=> "\xe7\x87\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x5b"		=> "\xe7\x8a\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x5c"		=> "\xe7\x8a\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x5d"		=> "\xe7\x8c\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x5e"		=> "\xef\xa8\x96",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x5f"		=> "\xe7\x8d\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x60"		=> "\xe7\x8e\xbd",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x61"		=> "\xe7\x8f\x89",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x62"		=> "\xe7\x8f\x96",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x63"		=> "\xe7\x8f\xa3",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x64"		=> "\xe7\x8f\x92",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x65"		=> "\xe7\x90\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x66"		=> "\xe7\x8f\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x67"		=> "\xe7\x90\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x68"		=> "\xe7\x90\xaa",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x69"		=> "\xe7\x90\xa9",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x6a"		=> "\xe7\x90\xae",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x6b"		=> "\xe7\x91\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x6c"		=> "\xe7\x92\x89",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x6d"		=> "\xe7\x92\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x6e"		=> "\xe7\x94\x81",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x6f"		=> "\xe7\x95\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x70"		=> "\xe7\x9a\x82",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x71"		=> "\xe7\x9a\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x72"		=> "\xe7\x9a\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x73"		=> "\xe7\x9a\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x74"		=> "\xe7\x9a\xa6",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x75"		=> "\xef\xa8\x97",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x76"		=> "\xe7\x9d\x86",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x77"		=> "\xe5\x8a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x78"		=> "\xe7\xa0\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x79"		=> "\xe7\xa1\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x7a"		=> "\xe7\xa1\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x7b"		=> "\xe7\xa1\xba",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x7c"		=> "\xe7\xa4\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x7d"		=> "\xef\xa8\x98",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x7e"		=> "\xef\xa8\x99",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x80"		=> "\xef\xa8\x9a",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x81"		=> "\xe7\xa6\x94",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x82"		=> "\xef\xa8\x9b",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x83"		=> "\xe7\xa6\x9b",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x84"		=> "\xe7\xab\x91",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x85"		=> "\xe7\xab\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x86"		=> "\xef\xa8\x9c",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x87"		=> "\xe7\xab\xab",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x88"		=> "\xe7\xae\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x89"		=> "\xef\xa8\x9d",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x8a"		=> "\xe7\xb5\x88",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x8b"		=> "\xe7\xb5\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x8c"		=> "\xe7\xb6\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x8d"		=> "\xe7\xb6\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x8e"		=> "\xe7\xb7\x96",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x8f"		=> "\xe7\xb9\x92",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x90"		=> "\xe7\xbd\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x91"		=> "\xe7\xbe\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x92"		=> "\xef\xa8\x9e",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x93"		=> "\xe8\x8c\x81",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x94"		=> "\xe8\x8d\xa2",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x95"		=> "\xe8\x8d\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x96"		=> "\xe8\x8f\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x97"		=> "\xe8\x8f\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x98"		=> "\xe8\x91\x88",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x99"		=> "\xe8\x92\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x9a"		=> "\xe8\x95\x93",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x9b"		=> "\xe8\x95\x99",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x9c"		=> "\xe8\x95\xab",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x9d"		=> "\xef\xa8\x9f",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\x9e"		=> "\xe8\x96\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xfb\x9f"		=> "\xef\xa8\xa0",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xa0"		=> "\xef\xa8\xa1",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xa1"		=> "\xe8\xa0\x87",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xa2"		=> "\xe8\xa3\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xa3"		=> "\xe8\xa8\x92",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xa4"		=> "\xe8\xa8\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xa5"		=> "\xe8\xa9\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xa6"		=> "\xe8\xaa\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xa7"		=> "\xe8\xaa\xbe",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xa8"		=> "\xe8\xab\x9f",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xa9"		=> "\xef\xa8\xa2",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xaa"		=> "\xe8\xab\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xab"		=> "\xe8\xad\x93",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xac"		=> "\xe8\xad\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xad"		=> "\xe8\xb3\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xae"		=> "\xe8\xb3\xb4",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xaf"		=> "\xe8\xb4\x92",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xb0"		=> "\xe8\xb5\xb6",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xb1"		=> "\xef\xa8\xa3",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xb2"		=> "\xe8\xbb\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xb3"		=> "\xef\xa8\xa4",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xb4"		=> "\xef\xa8\xa5",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xb5"		=> "\xe9\x81\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xb6"		=> "\xe9\x83\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xb7"		=> "\xef\xa8\xa6",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xb8"		=> "\xe9\x84\x95",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xb9"		=> "\xe9\x84\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xba"		=> "\xe9\x87\x9a",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xbb"		=> "\xe9\x87\x97",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xbc"		=> "\xe9\x87\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xbd"		=> "\xe9\x87\xad",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xbe"		=> "\xe9\x87\xae",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xbf"		=> "\xe9\x87\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc0"		=> "\xe9\x87\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc1"		=> "\xe9\x88\x86",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc2"		=> "\xe9\x88\x90",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc3"		=> "\xe9\x88\x8a",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc4"		=> "\xe9\x88\xba",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc5"		=> "\xe9\x89\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc6"		=> "\xe9\x88\xbc",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc7"		=> "\xe9\x89\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc8"		=> "\xe9\x89\x99",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xc9"		=> "\xe9\x89\x91",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xca"		=> "\xe9\x88\xb9",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xcb"		=> "\xe9\x89\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xcc"		=> "\xe9\x8a\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xcd"		=> "\xe9\x89\xb7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xce"		=> "\xe9\x89\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xcf"		=> "\xe9\x8b\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd0"		=> "\xe9\x8b\x97",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd1"		=> "\xe9\x8b\x99",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd2"		=> "\xe9\x8b\x90",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd3"		=> "\xef\xa8\xa7",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xd4"		=> "\xe9\x8b\x95",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd5"		=> "\xe9\x8b\xa0",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd6"		=> "\xe9\x8b\x93",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd7"		=> "\xe9\x8c\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd8"		=> "\xe9\x8c\xa1",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xd9"		=> "\xe9\x8b\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xda"		=> "\xef\xa8\xa8",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xdb"		=> "\xe9\x8c\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xdc"		=> "\xe9\x8b\xbf",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xdd"		=> "\xe9\x8c\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xde"		=> "\xe9\x8c\x82",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xdf"		=> "\xe9\x8d\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe0"		=> "\xe9\x8d\x97",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe1"		=> "\xe9\x8e\xa4",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe2"		=> "\xe9\x8f\x86",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe3"		=> "\xe9\x8f\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe4"		=> "\xe9\x8f\xb8",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe5"		=> "\xe9\x90\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe6"		=> "\xe9\x91\x85",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe7"		=> "\xe9\x91\x88",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe8"		=> "\xe9\x96\x92",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xe9"		=> "\xef\xa7\x9c",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xea"		=> "\xef\xa8\xa9",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xeb"		=> "\xe9\x9a\x9d",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xec"		=> "\xe9\x9a\xaf",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xed"		=> "\xe9\x9c\xb3",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xee"		=> "\xe9\x9c\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xef"		=> "\xe9\x9d\x83",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xf0"		=> "\xe9\x9d\x8d",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xf1"		=> "\xe9\x9d\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xf2"		=> "\xe9\x9d\x91",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xf3"		=> "\xe9\x9d\x95",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xf4"		=> "\xe9\xa1\x97",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xf5"		=> "\xe9\xa1\xa5",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xf6"		=> "\xef\xa8\xaa",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xf7"		=> "\xef\xa8\xab",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xf8"		=> "\xe9\xa4\xa7",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xf9"		=> "\xef\xa8\xac",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfb\xfa"		=> "\xe9\xa6\x9e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xfb"		=> "\xe9\xa9\x8e",	//CJK UNIFIED IDEOGRAPH
		"\xfb\xfc"		=> "\xe9\xab\x99",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x40"		=> "\xe9\xab\x9c",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x41"		=> "\xe9\xad\xb5",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x42"		=> "\xe9\xad\xb2",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x43"		=> "\xe9\xae\x8f",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x44"		=> "\xe9\xae\xb1",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x45"		=> "\xe9\xae\xbb",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x46"		=> "\xe9\xb0\x80",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x47"		=> "\xe9\xb5\xb0",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x48"		=> "\xe9\xb5\xab",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x49"		=> "\xef\xa8\xad",	//CJK COMPATIBILITY IDEOGRAPH
		"\xfc\x4a"		=> "\xe9\xb8\x99",	//CJK UNIFIED IDEOGRAPH
		"\xfc\x4b"		=> "\xe9\xbb\x91",	//CJK UNIFIED IDEOGRAPH
	);
}