﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using LFS_External;
using LFS_External.InSim;

namespace LFS_ExternalExample
{
    public partial class MainForm : Form
    {
        private InSimInterface insim;

        // InSim connection settings.
        private InSimSettings settings = new InSimSettings(
            "127.0.0.1",
            29999,
            0,
            Flags.InSimFlags.ISF_MCI | Flags.InSimFlags.ISF_MSO_COLS,
            '!',
            1000,
            String.Empty,
            "^3LFS External",
            5);

        // Connection and player dictionaries.
        private Dictionary<byte, Connection> connections = new Dictionary<byte, Connection>(); 
        private Dictionary<byte, Player> players = new Dictionary<byte, Player>();

        public MainForm()
        {
            InitializeComponent();

            // Connect to LFS.
            InSimConnect();
        }

        // Use this method to connect to InSim so you are able to catch any exception that might occure
        private void InSimConnect()
        {
            try
            {
                // Create InSim object.
                insim = new InSimInterface(settings);
                insim.ConnectionLost += insim_ConnectionLost;
                insim.Reconnected += insim_Reconnected;

                // Hookup InSim packet events...
                InitializeInSimEvents();

                // Connect to LFS.
                insim.Connect();

                // Request connections/players to be sent.
                insim.Request_NCN_AllConnections(255);
                insim.Request_NPL_AllPlayers(255);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        // You should only enable the events you need to gain maximum performance. All events are enable by default.
        private void InitializeInSimEvents()
        {
            // Add InSim packet events here!
            insim.NCN_Received += insim_NCN_Received;
            insim.CNL_Received += insim_CNL_Received;
            insim.CPR_Received += insim_CPR_Received;
            insim.NPL_Received += insim_NPL_Received;
            insim.PLP_Received += insim_PLP_Received;
            insim.TOC_Received += insim_TOC_Received;
            insim.MSO_Received += insim_MSO_Received;
        }

        // Always call .Close() on application exit
        protected override void OnClosed(EventArgs e)
        {
            if (insim != null)
            {
                insim.Close();
            }

            base.OnClosed(e);
        }

        // Invokes action onto main program thread when needed.
        private void DoInvoke(Action action)
        {
            // Check if method being called from different thread.
            if (InvokeRequired)
            {
                // Invoke action onto main thread.
                Invoke(action);
            }
            else
            {
                // Call action normally.
                action();
            }
        }

        // Occurs when connection was lost due to an unknown reason
        private void insim_ConnectionLost()
        {
            // Invoke onto main program thread.
            DoInvoke(() =>
            {
                MessageBox.Show("Connection to InSim lost due to an unknown reason.");
            });
        }

        // Occurs when connection was recovert automatically
        private void insim_Reconnected()
        {
            // Invoke onto main program thread.
            DoInvoke(() =>
            {
                MessageBox.Show("Automatically reconnected to InSim.");
            });
        }

        // Player chat and system messages.
        private void insim_MSO_Received(Packets.IS_MSO mso)
        {
            // Invoke onto main program thread.
            DoInvoke(() =>
            {
                // Add newline to end of string.
                string message = String.Format("{0}{1}", mso.Msg, Environment.NewLine);

                // Add to textbox.
                msoTextBox.AppendText(message);
            });
        }

        // New client joined the server.
        private void insim_NCN_Received(Packets.IS_NCN ncn)
        {
            // Add new connection to connections list.
            connections.Add(ncn.UCID, new Connection
            {
                UniqueID = ncn.UCID,
                Username = ncn.UName,
                PlayerName = ncn.PName,
                IsAdmin = ncn.Admin,
                Flags = ncn.Flags,
            });
        }

        // Client left the server.
        private void insim_CNL_Received(Packets.IS_CNL cnl)
        {
            // Remove connection from list.
            connections.Remove(cnl.UCID);
        }

        // Client changed name or plate.
        private void insim_CPR_Received(Packets.IS_CPR cpr)
        {
            // Update connection.
            connections[cpr.UCID].PlayerName = cpr.PName;

            // Update player.
            Player player = players.Values.SingleOrDefault(p => p.UniqueID == cpr.UCID);
            if (player != null)
            {
                player.PlayerName = cpr.PName;
                player.Plate = cpr.Plate;
            }
        }

        // Player joins the race. If PLID already exists, then player leaves pit.
        private void insim_NPL_Received(Packets.IS_NPL npl)
        {
            if (players.ContainsKey(npl.PLID))
            {
                // Player left pits (garage screen).
            }
            else
            {
                // Add player to players list.
                players.Add(npl.PLID, new Player
                {
                    AddedMass = npl.H_Mass,
                    CarName = npl.CName,
                    Flags = npl.Flags,
                    Passengers = npl.Pass,
                    Plate = npl.Plate,
                    PlayerID = npl.PLID,
                    UniqueID = npl.UCID,
                    PlayerName = npl.PName,
                    PlayerType = (PType)npl.PType,
                    SkinName = npl.SName,
                    Tyre_FL = npl.Tyre_FL,
                    Tyre_FR = npl.Tyre_FR,
                    Tyre_RL = npl.Tyre_RL,
                    Tyre_RR = npl.Tyre_RR,
                });
            }
        }

        // Player leaves the race (spectate)
        private void insim_PLP_Received(Packets.IS_PLP plp)
        {
            // Remove player from players list.
            players.Remove(plp.PLID);
        }

        // Car was taken over by another player.
        private void insim_TOC_Received(Packets.IS_TOC toc)
        {
            players[toc.PLID].UniqueID = toc.NewUCID;
            players[toc.PLID].PlayerID = toc.PLID;
        }
    }
}
