﻿using System;
using System.Windows.Forms;
using InSimDotNet;
using InSimDotNet.Packets;

namespace InSimUI {
    public partial class MainForm : Form {
        private InSim _insim;

        public MainForm() {
            InitializeComponent();

            ActiveControl = _connectButton;
        }

        private void DoInvoke(Action action) {
            // Check if invoke is required, if not call normally.
            if (InvokeRequired) {
                Invoke(action);
            }
            else {
                action();
            }
        }

        protected override void OnClosed(EventArgs e) {
            if (_insim != null) {
                _insim.Dispose();
            }

            base.OnClosed(e);
        }

        private void OnConnectButtonClick(object sender, EventArgs e) {
            try {
                Cursor = Cursors.WaitCursor;

                InitializeInSim();

                // Connect to LFS!
                _insim.Initialize(new InSimSettings {
                    Host = _hostTextBox.Text,
                    Port = (int)_portNumericUpDown.Value,
                    Admin = _adminTextBox.Text,
                });
            }
            catch (Exception ex) {
                MessageBox.Show(
                    this,
                    String.Format("Error: {0}", ex.Message),
                    "InSim Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
            }
            finally {
                Cursor = Cursors.Default;
            }
        }

        private void InitializeInSim() {
            // If InSim object already created, dispose of it...
            if (_insim != null) {
                _insim.Unbind<IS_MSO>(MessageOut);
                _insim.Initialized -= OnInSimInitialized;
                _insim.Disconnected -= OnInSimDisconnected;
                _insim.InSimError -= OnInSimInSimError;
                _insim.Dispose();
            }

            // Create new InSim object.
            // Bind packet handler and events.
            _insim = new InSim();
            _insim.Bind<IS_MSO>(MessageOut);
            _insim.Initialized += new EventHandler<InitializeEventArgs>(OnInSimInitialized);
            _insim.Disconnected += new EventHandler<DisconnectedEventArgs>(OnInSimDisconnected);
            _insim.InSimError += new EventHandler<InSimErrorEventArgs>(OnInSimInSimError);
        }

        private void OnInSimInitialized(object sender, InitializeEventArgs e) {
            // When initialized update buttons.
            DoInvoke(UpdateButtons);
        }

        private void OnInSimDisconnected(object sender, DisconnectedEventArgs e) {
            // When disconnected updated buttons.
            DoInvoke(UpdateButtons);
        }

        private void OnInSimInSimError(object sender, InSimErrorEventArgs e) {
            // Show error message and update buttons.
            DoInvoke(() => {
                MessageBox.Show(
                    this,
                    String.Format("Error: {0}", e.Exception.Message),
                    "InSim Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                UpdateButtons();
            });
        }

        private void MessageOut(InSim insim, IS_MSO mso) {
            // Append MSO message onto textbox.
            DoInvoke(() => {
                _messagesTextBox.AppendText(String.Format("{0}{1}", mso.Msg, Environment.NewLine));
            });
        }

        private void OnDisconnectButtonClick(object sender, EventArgs e) {
            if (_insim != null) {
                _insim.Disconnect();
            }
        }

        private void UpdateButtons() {
            _connectButton.Enabled = !_insim.IsConnected;
            _disconnectButton.Enabled = _insim.IsConnected;
        }
    }
}
