#include "outgaugeClient.h"
#include "wheelInterface.h"
#include <stdio.h>

LPDIRECTINPUT8 di8;
LPDIRECTINPUTDEVICE8 diDevice;
DIJOYSTATE2 diDevState;

LPWSTR diErrCap = L"DirectInput error";

UINT acqAttempts = 0;

BOOL deviceFound = false;		//This can probably be done better
BOOL poll = true;

HANDLE pollMutex;

HRESULT pollErr;			//Error reported when trying to poll the device
HRESULT acqErr;				//Error reported when trying to acquire the device in PollDevice() function
HRESULT reAcqErr;			//Error reported when trying to reacquire the device before sending RPM data
HRESULT sendRPMErr;			//Error reported when trying to send RPM data to the device

CONST ULONG G27ID = 0xC29B046D;		//G27 PID		
CONST ULONG DFPID = 0xC298046D;		//Debug only

CONST DWORD ESCAPE_COMMAND_LEDS = 0;
CONST DWORD LEDS_VERSION_NUMBER = 0x00000001;
struct LedsRpmData
{
	FLOAT currentRPM;
    FLOAT rpmFirstLedTurnsOn;
    FLOAT rpmRedLine;
};
struct WheelData
{
    DWORD size;
    DWORD versionNbr;
    LedsRpmData rpmData;
};

//DEBUG only, send these RPM values to the wheel
FLOAT minRPM = 4000;
FLOAT currRPM = 5000;
FLOAT redline = 7500;
FLOAT step = 100;

BOOL CALLBACK DevEnumCallback(const DIDEVICEINSTANCE* instance, VOID* context)
{
	HRESULT hr;
	if(FAILED(hr = di8->CreateDevice(instance->guidInstance, &diDevice, NULL)))
	{
		//Error when enumerating a device
		return DIENUM_CONTINUE;
	}

	//Check the device PID and VID
	if((instance->guidProduct.Data1 == G27ID) || (instance->guidProduct.Data1 == DFPID))
	{
		//G27 found, stop enumerating
		deviceFound = true;
		return DIENUM_STOP;
	}
	
	//Invalid PID/VID, do not store the device and continue enumarating
	return DIENUM_CONTINUE;
}

HRESULT InitializeDI(HWND hWnd)
{
	HRESULT hr;
	//Initialize DirectInput
	if(FAILED(hr = DirectInput8Create(GetModuleHandle(NULL), DIRECTINPUT_VERSION, IID_IDirectInput8, (VOID**)&di8, NULL)))
	{
		MessageBox(NULL, L"Cannot initialize DirectInput. Error code" + hr, diErrCap, MB_ICONERROR | MB_OK);
		return hr;
	}

	//Iterate through input devices and find G27
	if(FAILED(hr = di8->EnumDevices(DI8DEVCLASS_GAMECTRL, DevEnumCallback, NULL, DIEDFL_ATTACHEDONLY)))
	{
		//Error when enumareting devices, tell user and report back
		MessageBox(NULL, L"Error when enumerating game devices.", diErrCap, MB_ICONERROR | MB_OK);
		return hr;
	}

	//If G27 wasn't found, tell user and report error
	if(!deviceFound)
	{
		MessageBox(NULL, L"G27 wheel not found. Is the device connected and set to native mode?", L"G27 not found", MB_ICONERROR | MB_OK);
		return E_FAIL;
	}

	//Set expected data format
	if (FAILED(hr = diDevice->SetDataFormat(&c_dfDIJoystick2)))
	{
		WCHAR text[255];
		wsprintf(text, L"%p error setting data format", hr);
		MessageBox(NULL, text, diErrCap, MB_ICONERROR | MB_OK);
		return hr;
	}

	/*Set the cooperative level to let DirectInput know how this device should
	interact with the system and with other DirectInput applications.*/
	if (FAILED(hr = diDevice->SetCooperativeLevel(hWnd, DISCL_EXCLUSIVE | DISCL_BACKGROUND)))
	{
		WCHAR text[255];
		wsprintf(text, L"%p error setting cooperative level", hr);
		MessageBox(NULL, text, diErrCap, MB_ICONERROR | MB_OK);
		return hr;
	}

	//Initialize mutex
	pollMutex = CreateMutex(NULL, FALSE, NULL);

	return S_OK;
}

VOID PollDevice(void)
{
	pollErr = diDevice->Poll();
	if(FAILED(pollErr))
	{
		//Try to acquire device until we acquire it.
		while(true) {
			acqErr = diDevice->Acquire();
			acqAttempts++;
			if(acqErr == S_OK || acqAttempts > 5)
				break;
		}

		//Unrecoverable error, report failure
        if ((pollErr == DIERR_INVALIDPARAM) || (pollErr == DIERR_NOTINITIALIZED)) {
            return;
        }

		//Another app has priority, wait until it frees the device.
		//WARNING: This might be a problem!
        if (pollErr == DIERR_OTHERAPPHASPRIO) {
            return;
        }
	}
		
	//Get the device's state
	if (FAILED(pollErr = diDevice->GetDeviceState(sizeof(DIJOYSTATE2), &diDevState)))
	{
		return;
	}
}

DWORD WINAPI ContPolling(LPVOID lpArg)
{
	HWND hWnd = (HWND)lpArg;
	DWORD waitResult;

	//Prepare LEDs command
	WheelData wheelData;
	ZeroMemory(&wheelData, sizeof(wheelData));

	wheelData.size = sizeof(WheelData);
	wheelData.versionNbr = LEDS_VERSION_NUMBER;
	wheelData.rpmData.currentRPM = 5000;
	wheelData.rpmData.rpmFirstLedTurnsOn = minRPM;
	wheelData.rpmData.rpmRedLine = redline;
	
	while(true)
	{
		wheelData.rpmData.currentRPM = GetRPM();
		if(wheelData.rpmData.currentRPM < 100)
		{
			wheelData.rpmData.currentRPM = 5000;
		}

		DIEFFESCAPE data;
		ZeroMemory(&data, sizeof(data));

		data.dwSize = sizeof(DIEFFESCAPE);
		data.dwCommand = ESCAPE_COMMAND_LEDS;
		data.lpvInBuffer = &wheelData;
		data.cbInBuffer = sizeof(wheelData);

		//Clear errors
		pollErr = acqErr = reAcqErr = sendRPMErr = 0;

		PollDevice();
		
		//Reaquire device before sending RPM data
		//reAcqErr = diDevice->Acquire();
		while(true)
		{
			reAcqErr = diDevice->Acquire();
			acqAttempts++;
			if(acqAttempts > 10 || reAcqErr == S_OK)
				break;
		}
		HRESULT sendRPMErr = diDevice->Escape(&data);

		WCHAR* pollErrDesc = TranslateError(pollErr, L"Poll error");
		WCHAR* acqErrDesc = TranslateError(acqErr, L"Acq error");
		//WCHAR* reAcqErrDesc = TranslateError(reAcqErr, L"Reacq error");
		WCHAR* sendErrDesc = TranslateError(sendRPMErr, L"Send RPM error");

		CHAR ogRpm[16];
		sprintf_s(ogRpm, sizeof(ogRpm), "%g", wheelData.rpmData.currentRPM);

		//Prepare info
		WCHAR* text = (WCHAR*)malloc((wcslen(pollErrDesc) + wcslen(acqErrDesc) + wcslen(sendErrDesc) + strlen(ogRpm) + 6 + 6 + 2 + 18)*sizeof(WCHAR));
		//WCHAR text[1024];
		ZeroMemory(text, sizeof(text));
		wsprintf(text, L"%s %s %s X-axis: %d Acqs: %d RPM: %S",
				 pollErrDesc, acqErrDesc, sendErrDesc, diDevState.lX, acqAttempts, ogRpm);

		//Print info to the Editbox
		SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)text);

		free(pollErrDesc);
		free(acqErrDesc);
		free(sendErrDesc);
		free(text);

		//Change RPM
		/*currRPM += step;
		if(currRPM >= redline || currRPM <= minRPM)
		{
			step*=-1;
		}*/
		
		acqAttempts = 0;
		Sleep(50);

		waitResult = WaitForSingleObject(pollMutex, INFINITE);
		if(waitResult == WAIT_OBJECT_0)
		{
			if(!poll)
			{
				break;
			}
		}
		ReleaseMutex(pollMutex);
	}

	return S_OK;
}

VOID TerminatePolling(void)
{
	DWORD waitResult = WaitForSingleObject(pollMutex, INFINITE);

	switch(waitResult)
	{
		case WAIT_OBJECT_0:
			poll = false;
			ReleaseMutex(pollMutex);
			break;
	}
}

VOID UnacquireDevice(void)
{
	diDevice->Unacquire();
}

WCHAR* TranslateError(HRESULT hr, WCHAR* errFunc)
{
	WCHAR* errDesc;		//Human-readable description of the error
	WCHAR* errType;		//A type of known error the application has encountered
	
	//Known errors
	switch(hr)
	{
		case S_OK:
			errType = L"OK";
			break;
		case DIERR_INPUTLOST:
			errType = L"Input lost";
			break;
		case DIERR_NOTACQUIRED:
			errType = L"Not acquired";
			break;
		case DIERR_ACQUIRED:
			errType = L"Aquired";
			break;
		case E_ACCESSDENIED:
			errType = L"Access denied";
			break;
		default:
			errType = (WCHAR*)malloc(sizeof(WCHAR)*9);
			wsprintf(errType, L"%p", hr);
			break;
	}

	errDesc = (WCHAR*)malloc((wcslen(errFunc) + wcslen(errType) + 3)*sizeof(WCHAR));
	wsprintf(errDesc, L"%s: %s", errFunc, errType);

	free(errType);
	
	return errDesc;
}