package ogt;

public class ParsedData {
	
	public byte gear = 1; 
	public float speed = 0.0f;
	public int rpm = 0;
	public float fuel = 1.0f;
	
	public boolean shiftL = false;
	public boolean pitspeedL = false;
	public boolean oilL = false;
	public boolean batteryL = false;
	
	private int showLights = 0;
	/* ShowLights bits - info taken from insim.h
		DL_SHIFT,					// bit 0	- shift light
		DL_FULLBEAM,			// bit 1	- full beam
		DL_HANDBRAKE,		// bit 2	- handbrake
		DL_PITSPEED,			// bit 3	- pit speed limiter
		DL_TC,						// bit 4	- TC active or switched off
		DL_SIGNAL_L,			// bit 5	- left turn signal
		DL_SIGNAL_R,			// bit 6	- right turn signal
		DL_SIGNAL_ANY,		// bit 7	- shared turn signal
		DL_OILWARN,				// bit 8	- oil pressure warning
		DL_BATTERY,				// bit 9	- battery warning
		DL_ABS,					// bit 10 - ABS active or switched off
		DL_SPARE,				// bit 11
		DL_NUM;
	*/
	
	public int redline = 0;
	
	private char[] carCode = new char[3];
	private char[] prevCarCode = new char[3];
	
	private boolean refreshCarCode = true;
	
	public void parsePacket(byte[] inData)
	{
			//Get the car code to set the redline correctly
			carCode[0] = (char)inData[4];
			carCode[1] = (char)inData[5];
			carCode[2] = (char)inData[6];
			if(refreshCarCode)		//Set new redline if needed.
			{
				System.arraycopy(carCode, 0, prevCarCode, 0, 3);
				setRedline();
				refreshCarCode = false;
			}
			
			refreshCarCode = checkCarCode();
		
			gear = inData[10];
			speed = segmentToFloat(inData, 12) * 3.6f;	//Store speed in km/h
			rpm = (int)segmentToFloat(inData, 16);
			fuel = segmentToFloat(inData, 28);
			showLights = segmentToInt(inData, 44);
			parseDashLights();
	}
	
	/** Check if the car code has changed, returns
	 * true if it has.
	 */
	private boolean checkCarCode()
	{
		if(carCode[0] == prevCarCode[0])
		{
			if(carCode[1] == prevCarCode[1])
			{
				if(carCode[2] == prevCarCode[2])
				{
					return false;
				}
			}
		}
		
		return true;
	}
	
	private float segmentToFloat(byte[] inData, int beg)
	{
		int bits = 0;
		for(int shiftBy = 0; shiftBy < 4; shiftBy++)
        {
            bits |= ((int)inData[beg++] & 0xFF) << (shiftBy * 8);
        }
		
        return Float.intBitsToFloat(bits);
	}
	
	private int segmentToInt(byte[] inData, int beg)
	{
		int retVal = 0;
		for(int shiftBy = 0; shiftBy < 4; shiftBy++)
        {
            retVal |= ((int)inData[beg++] & 0xFF) << (shiftBy * 8);
        }
		
		return retVal;
	}
	
	private void parseDashLights()
	{
		//Is shift light on?
		int mask = 0x01;
		shiftL = ((showLights & mask) == mask) ? true : false;
		
		//Is pit limiter light on?
		mask = 0x06;
		pitspeedL = ((showLights & mask) == mask) ? true : false;
		
		//Is oil press light on?
		mask = 0x100;
		oilL = ((showLights & mask) == mask) ? true : false;
		
		//Is charge light on?
		mask = 0x200;
		batteryL = ((showLights & mask) == mask) ? true : false;
	}
	
	private void setRedline()
	{
		if(carCode[0] == 'B')
		{
			redline = 20000;						//BF1
		}
		else if(carCode[0] == 'F')	//The F-cars
		{
			if(carCode[1] == 'B')
			{
				redline = 9000;					//FBM
			}
			else if(carCode[1] == 'O')
			{
				redline = 7500;					//FOX
			}
			else if(carCode[1] == 'X')		//The FX-cars
			{
				if(carCode[2] == 'O')
				{
					redline = 7500;				//FXO
				}
				else
				{
					redline = 7500;				//FXR
				}
			}
			else if(carCode[1] == 'Z')	//The FZ-cars
			{
				if(carCode[2] == 'R')
				{
					redline = 8500;				//FZR
				}
				else
				{
					redline = 8000;				//FZ5
				}
			}
			else if(carCode[1] == '0')
			{
				redline = 9500;					//FO8
			}
		}
		else if(carCode[0] == 'L')	//The L-cars
		{
			redline = 9000;						//Both LX4 and LX6
		}
		else if(carCode[0] == 'M')			
		{
			redline = 13000;						//MRT5
		}
		else if(carCode[0] == 'R')	//The R-cars
		{
			if(carCode[1] == 'A')
			{
				redline = 7000;					//RAC
			}
			else
			{
				redline = 7500;					//RB4
			}
		}
		else if(carCode[0] == 'U')	//The U-cars
		{
			if(carCode[2] == 'R')
			{
				redline = 9000;					//UFR
			}
			else 
			{
				redline = 7000;					//UF1
			}
		}
		else if(carCode[0] == 'X')	//The X-cars
		{
			if(carCode[1] == 'F')			//The XF-cars
			{
				redline = 8000;					//XFG and XFR
			}
			else if(carCode[1] == 'R')	//The XR-cars
			{
				if(carCode[2] == 'G')
				{	
					redline = 7000;				//XRG
				}
				else
				{
					redline = 7500;				//XRT and XRR
				}
			}
		}	
	}
}
