﻿using System;
using System.Windows.Forms;
using Spark;
using Spark.Packets;
using SparkGui.Properties;

namespace SparkGui
{
    public partial class MainForm : Form
    {
        private InSim insim;

        public MainForm()
        {
            InitializeComponent();
            InitializeInSim();

            LoadSettings();
            UpdateButtons();

            ActiveControl = connectButton;
        }

        private void DoInvoke(Action action)
        {
            // Invoke operation onto main GUI thread if nessesary.
            if (InvokeRequired)
            {
                Invoke(action);
            }
            else
            {
                action();
            }
        }

        private void InitializeInSim()
        {
            insim = new InSim();

            // Bind events.
            insim.Connected += new EventHandler<ConnectedEventArgs>(insim_Connected);
            insim.Disconnected += new EventHandler<DisconnectedEventArgs>(insim_Disconnected);
            insim.InSimError += new EventHandler<InSimErrorEventArgs>(insim_InSimError);

            // Bind packets.
            insim.Bind<IS_MSO>(MessageOut);
        }

        private void LoadSettings()
        {
            // Set max and min port values.
            insimPortNumericUpDown.Minimum = 0;
            insimPortNumericUpDown.Maximum = UInt16.MaxValue;

            hostAddressTextBox.Text = Settings.Default.HostAddress;
            adminPassTextBox.Text = Settings.Default.AdminPass;
            insimPortNumericUpDown.Value = Settings.Default.InSimPort;
        }

        private void SaveSettings()
        {
            Settings.Default.HostAddress = hostAddressTextBox.Text;
            Settings.Default.InSimPort = (ushort)insimPortNumericUpDown.Value;
            Settings.Default.AdminPass = adminPassTextBox.Text;
            Settings.Default.Save();
        }

        protected override void OnClosed(EventArgs e)
        {
            SaveSettings();

            base.OnClosed(e);
        }

        private void UpdateButtons()
        {
            // Set UI enabled/disabled.
            connectButton.Enabled = !insim.IsConnected;
            disconnectButton.Enabled = insim.IsConnected;
            hostAddressTextBox.Enabled = !insim.IsConnected;
            insimPortNumericUpDown.Enabled = !insim.IsConnected;
            adminPassTextBox.Enabled = !insim.IsConnected;
        }

        private void insim_Connected(object sender, ConnectedEventArgs e)
        {
            DoInvoke(UpdateButtons);
        }

        private void insim_Disconnected(object sender, DisconnectedEventArgs e)
        {
            DoInvoke(UpdateButtons);
        }

        private void insim_InSimError(object sender, InSimErrorEventArgs e)
        {
            DoInvoke(() =>
                {
                    UpdateButtons();
                    MessageBox.Show(
                        e.Exception.Message,
                        "InSim Error",
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Error);
                });
        }

        private void MessageOut(IS_MSO mso)
        {
            // Add message to UI.
            DoInvoke(() =>
                {
                    messagesTextBox.Text += String.Format(
                        "{0}{1}",
                        mso.Msg,
                        Environment.NewLine);
                });
        }

        private void connectButton_Click(object sender, EventArgs e)
        {
            const string AppName = "^3SparkGUI";

            try
            {
                // Connect to InSim.
                insim.Connect(
                    Settings.Default.HostAddress,
                    Settings.Default.InSimPort);

                // Initialize InSim.
                insim.Send(new IS_ISI
                {
                    Admin = Settings.Default.AdminPass,
                    IName = AppName,
                });
            }
            catch (Exception ex)
            {
                MessageBox.Show(
                    ex.Message,
                    "InSim Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
            }
        }

        private void disconnectButton_Click(object sender, EventArgs e)
        {
            // Close InSim.
            insim.Disconnect();
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }

                // Dispose InSim when closing.
                if (insim != null)
                {
                    insim.Dispose();
                }
            }

            base.Dispose(disposing);
        }
    }
}
