#include "StdAfx.h"
#include "WheelRangeSetter.h"

WheelRangeSetter::WheelRangeSetter(ConfigLoader* inCfgLdr)
{	
	cfgLdr = inCfgLdr;

	//Initialize variables
	dwSize = 1024;
	dwType = REG_DWORD;
	logitechKey = NULL;
	validRegEntriesCount = 0;
	lfsWindow = NULL;

	//Wait until LFS is launched
	while(lfsWindow == NULL)
	{
		lfsWindow = FindWindow(L"LFS", NULL);
		Sleep(500);		//Wait for 500 msecs after every attempt to get LFS handle. We don't want the CPU pegged at 100% by continouous FindWindow calls.
	}
}

WheelRangeSetter::~WheelRangeSetter(void)
{
}

/** Reads the registry and finds all
* subkeys in the HKCR\Software\Logitech\Gaming Software\GlobalDeviceSettings
* that containg DWORD OperatingRange value.
* Returns false if no such subkey is found
*/
bool WheelRangeSetter::ProbeRegistry(void)
{
	//Declare local variables
	HKEY tempSubKey = NULL;
	DWORD totalSubKeys = 0;
	const unsigned char subKeyNameMaxLen = 255;

	//Tell user what we're doing
	std::wcout << "Beginning registry probe..." << std::endl;

	//Open logitech registry key
	opRes = RegOpenKeyEx(HKEY_CURRENT_USER, L"Software\\Logitech\\Gaming Software\\GlobalDeviceSettings\\", NULL, KEY_ALL_ACCESS, &logitechKey);
	if(opRes != ERROR_SUCCESS)
	{
		std::wcout << L"(WheelRangeSetter)Critical: Cannot open HKCR\\Software\\Logitech\\Gaming Software\\GlobalDeviceSettings" << std::endl;
		std::wcout << L"Error code returned: " << opRes << std::endl;
		return false;
	}

	//Find how many subkeys are there in this key
	opRes = RegQueryInfoKey(logitechKey, NULL, NULL, NULL, &totalSubKeys, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
	if(opRes != ERROR_SUCCESS)
	{
		std::wcout << L"(WheelRangeSetter)Critical: Cannot query Logitech key for number of subkeys." << std::endl;
		std::wcout << L"Error code returned: " << opRes << std::endl;
		return false;
	}

	std::wcout << "Total subkeys found: " << totalSubKeys << std::endl;

	//Now we know with how many subkeys we have to deal with
	//so let's make sure we have where to store their names
	validRegEntries = new WCHAR*[totalSubKeys];
	for(DWORD idx = 0; idx < totalSubKeys; idx++)
	{
		validRegEntries[idx] = new WCHAR[subKeyNameMaxLen];
	}


	//Now find all registry subkeys that containn DWORD OperatingRange value
	for(DWORD idx = 0; idx < totalSubKeys; idx++)
	{
		//(Re)initialize variables
		HKEY tempRegKey = NULL;
		DWORD tempORValue = NULL;
		WCHAR tempRegKeyName[subKeyNameMaxLen];
		dwSize = 1024;

		std::wcout << "Probing subkey with index " << idx << std::endl;

		//Get the name of a subkey
		opRes = RegEnumKeyEx(logitechKey, idx, tempRegKeyName, &dwSize, NULL, NULL, NULL, NULL);
		if(opRes != ERROR_SUCCESS)
		{	
			std::wcout << L"(WheelRangeSetter)Non-critical: Cannot get name of subkey index " << idx << std::endl;
			std::wcout << L"Error code returned: " << opRes << std::endl;
			continue;
		}

		//Open that key
		opRes = RegOpenKeyEx(logitechKey, tempRegKeyName, NULL, KEY_ALL_ACCESS, &tempRegKey);
		if(opRes != ERROR_SUCCESS)
		{	
			std::wcout << L"(WheelRangeSetter)Non-critical: Cannot open subkey name " << tempRegKeyName << std::endl;
			std::wcout << L"Error code returned: " << opRes << std::endl;
			continue;
		}

		dwSize = 1024; //DO NOT change max size of DWORD!

		//Find if there is a DWORD OperatingRange value present in the subkey
		opRes = RegQueryValueEx(tempRegKey, L"OperatingRange", NULL, &dwType, (LPBYTE)&tempORValue, &dwSize);
		if(opRes == ERROR_FILE_NOT_FOUND)
		{	
			std::wcout << L"(WheelRangeSetter)Note: Cannot find DWORD OperatingRange in subkey name " << tempRegKeyName << std::endl;
			continue;
		}
		else if(opRes != ERROR_SUCCESS)
		{
			std::wcout << L"(WheelRangeSetter)Non-critical: Error while probing subkey name " << tempRegKeyName << std::endl;
			std::wcout << L"Error code returned: " << opRes << std::endl;
			continue;
		}

		//DWORD OperatingRange found
		//Add the name of this subkey to the lost
		if(tempORValue != NULL)
		{
			wcscpy(validRegEntries[validRegEntriesCount++], tempRegKeyName);
		}
		
		//Close the subkey
		RegCloseKey(tempRegKey);
	}

	if(validRegEntriesCount == 0)
	{
		std::wcout << "(WheelRangeSetter)Critical: No registry entries for supported wheels have been found!" << std::endl;
		return false;
	}

	//List names of all subkeys containing the DWORD OperatingRange value
	std::wcout << "Registry probing complete. Number of valid entries: " << validRegEntriesCount << std::endl;
	std::wcout << "List of valid subkeys:" << std::endl;
	for(DWORD idx = 0; idx < validRegEntriesCount; idx++)
	{
		std::wcout << idx+1 << ") " << validRegEntries[idx] << std::endl;
	}

	return true;

}


/** Sets new wheel range value in the registry.
 * Returns false if unsuccessfull
 */
bool WheelRangeSetter::SetRange(DWORD newRange)
{
  bool hasSucceeded = false;	//Stays false if we cannot change wheel range in any valid subkey.
  
  //Reset opRes as it might have a non-NULL value
  opRes = NULL;
  HKEY tempSubKey = NULL;
  
  //Set new wheel range for each valid device
  for(DWORD idx = 0; idx < validRegEntriesCount; idx++)
  {
	  //Open a valid subkey
	  opRes = RegOpenKeyEx(logitechKey, validRegEntries[idx], NULL, KEY_ALL_ACCESS, &tempSubKey);
      if(opRes != ERROR_SUCCESS)
      {
		  std::wcout << "Non-critical: Cannot open subkey name " << validRegEntries[idx] << std::endl;
          continue;
      }
    
      //Get current value of DWORD OperatingRange.
      //This is more of a safety measure than necessity
      DWORD currentRange = NULL;
      opRes = RegQueryValueEx(tempSubKey, L"OperatingRange", NULL, &dwType, (LPBYTE)&currentRange, &dwSize);
      if(opRes != ERROR_SUCCESS)
      {
		  std::wcout << "(WheelRangeSetter)Non-critical: Cannot retrieve OperatingRange value from key name " << validRegEntries[idx] <<". Skipping this key." << std::endl;
          RegCloseKey(tempSubKey);
          continue;
      }
    
      //If newRange value differs from the currentRange, set range to newRange
      if(newRange != currentRange)
      {
         opRes = RegSetValueEx(tempSubKey, L"OperatingRange", NULL, dwType, (LPBYTE)&newRange, dwSize);
         if(opRes != ERROR_SUCCESS)
         {
			 std::wcout << "(WheelRangeSetter)Non-critical: Error setting new OperatingRange in key name " << validRegEntries[idx] << std::endl;
         }
         else
         {
	         hasSucceeded = true;
         }
    }
    
    RegCloseKey(tempSubKey);
  }

  //If AutoAltTab is allowed, minimize and restore LFS to make LGS use the new range
  if(cfgLdr->GetAutoAltTab())
  {
	  ShowWindow(lfsWindow, SW_FORCEMINIMIZE);
	  Sleep(cfgLdr->GetRestoreDelay());
	  ShowWindow(lfsWindow, SW_RESTORE);
  }
  
  return hasSucceeded;
  
}
    
    

