#include "StdAfx.h"
#include "ConFileReader.h"

ConFileReader::ConFileReader(void)
{
}

/** Prepares the ConFileReader
* for getting updated wheel range data
* from LFS. Returns false if the
* preparation is unsuccessful.
*/
bool ConFileReader::Initialize()
{
	if(!ProbeMiscDirectory())
	{
		return false;
	}

	return ReadInitialWheelRange();
}

/** Finds all .con files in
* "data/misc" directory and stores theris names.
* We need those to get wheel range directly from LFS.
*/
bool ConFileReader::ProbeMiscDirectory()
{
	HANDLE fileHndl = NULL;		//Handle to a file returned by FindFirstFile()

	WIN32_FIND_DATA	fileInfo;	//Struct holding the file info
	
	//Get the first .con file in the directory
	fileHndl = FindFirstFile(L"./data/misc/*.con", &fileInfo);

	//No .con file found, getting data from LFS unavailable
	if(fileHndl == INVALID_HANDLE_VALUE)
	{
		std::wcout << "(ConFileReader)Non-critical: No .con file found in data/misc directory." << std::endl;
		return false;
	}

	//We have the first file, let's store it's name
	conFiles.push_back(fileInfo.cFileName);

	//Find possible other .con files
	while(FindNextFile(fileHndl, &fileInfo))
	{
		conFiles.push_back(fileInfo.cFileName);
	}

	return true;
}

/** Reads wheel range from all .con files
* and stores current values. We need those
* to keep track of any changes to wheel range
* made by LFS.
*/
bool ConFileReader::ReadInitialWheelRange()
{
	DWORD wheelRange = NULL;
	bool hasSucceeded = false;
	
	for(unsigned int idx = 0; idx < conFiles.size(); idx++)
	{
		wheelRange = ReadConFile(&conFiles.at(idx));
		if(wheelRange == NULL)
		{
			wheelRangeValues.push_back((DWORD)0);
			continue;
		}
		wheelRangeValues.push_back(wheelRange);
		hasSucceeded = true;
	}

	return hasSucceeded;
}

/** Reads all .con files and compares
* current wheel range value in each file
* to it's initial value. If these values are not
* equal, returns current value.
* If there is no change in any of the .con files,
* returns NULL.
*/
DWORD ConFileReader::GetUpdatedWheelRange()
{
	DWORD readRange = NULL;
	for(unsigned int idx = 0; idx < conFiles.size(); idx++)
	{
		if(wheelRangeValues.at(idx) == 0)
		{
			continue;
		}
		readRange = ReadConFile(&conFiles.at(idx));
		if(readRange != wheelRangeValues.at(idx))
		{
			return readRange;
		}
	}

	return NULL;
}


/** Reads a .con file and extracts needed bytes 9, 10 and 11.
* Returns NULL if data cannot be read or are invalid,
* read wheel range is returned otherwise.
*/
DWORD ConFileReader::ReadConFile(std::wstring* fileName)
{
	std::wstring fullFileName = L"./data/misc/";
	fullFileName.append(*fileName);

	std::ifstream fReader(fullFileName.c_str(), std::ios::in);
	if(!fReader)	//Can we open the file for reading?
	{
		std::wcout << "(ConFileReader)Non-critical: Cannot read file " << fileName << "." << std::endl;
		return NULL;
	}
	
	fReader.seekg(0, std::ios::end);
	size_t fileLen = fReader.tellg();
	//Store data from the file here
	char* data = new char[fileLen];

	//Read the file
	fReader.seekg(NULL, std::ios::beg);
	fReader.read(data, fileLen);
	fReader.close();

	unsigned char* wheelRangeBytes = new unsigned char[3];
	wheelRangeBytes[0] = (unsigned char)data[9];
	wheelRangeBytes[1] = (unsigned char)data[10];
	wheelRangeBytes[2] = (unsigned char)data[11];

	DWORD wheelRange = CalcWheelRange(wheelRangeBytes);
	if((wheelRange < 90) || (wheelRange > 900))		//Does wheel range read from the file make sense?
	{
		std::wcout << "(ConFileReader)File " << *fileName << " is invalid, skipping." << std::endl;
		return NULL;
	}
	
	return wheelRange;
}

/** Get the wheel range out of the data
* we read from .con file. Returns 0 if
* the data we got does't make sense.
* Note: You might not want to figure out the way
* LFS stores the wheel range just by analyzing this function:)
*/
DWORD ConFileReader::CalcWheelRange(unsigned char data[3])
{
	int addVal = 0;
	//3rd byte is a flag byte, so let's see what value it has
	if(data[2] == 66)	//3rd byte set to 66, wheel range is from 90 to 127 deg
	{
		return data[1]/2;
	}
	else if(data[2] == 67)	//3rd byte set to 67, wheel range is from 128 to 511
	{
		if(data[0] == 128)
		{
			addVal = 1;
		}
		if(data[1] < 129)
		{
			return data[1] + 128 + addVal;
		}
		else
		{
			return 2 * (data[1] - 128) + 256 + addVal;
		}
	}
	else if(data[2] == 68)	//3rd byte set to 38, wheel range is from 512 to 900
	{
		if(data[0] == 64)
		{
			addVal = 1;
		}
		else if(data[0] == 128)
		{
			addVal = 2;
		}
		else if(data[0] == 192)
		{
			addVal = 3;
		}
		return 4 * (data[1]) + 512 + addVal;
	}
  
	return 0;
}


ConFileReader::~ConFileReader(void)
{
}
