#!/usr/bin/env python

import struct
import math

INSIM_VERSION = 4

# Enum for packet types
ISP_NONE = 0
ISP_ISI = 1
ISP_VER = 2
ISP_TINY = 3
ISP_SMALL = 4
ISP_STA = 5
ISP_SCH = 6
ISP_SFP = 7
ISP_SCC = 8
ISP_CPP = 9
ISP_ISM = 10
ISP_MSO = 11
ISP_III = 12
ISP_MST = 13
ISP_MTC = 14
ISP_MOD = 15
ISP_VTN = 16
ISP_RST = 17
ISP_NCN = 18
ISP_CNL = 19
ISP_CPR = 20
ISP_NPL = 21
ISP_PLP = 22
ISP_PLL = 23
ISP_LAP = 24
ISP_SPX = 25
ISP_PIT = 26
ISP_PSF = 27
ISP_PLA = 28
ISP_CCH = 29
ISP_PEN = 30
ISP_TOC = 31
ISP_FLG = 32
ISP_PFL = 33
ISP_FIN = 34
ISP_RES = 35
ISP_REO = 36
ISP_NLP = 37
ISP_MCI = 38
ISP_MSX = 39
ISP_MSL = 40
ISP_CRS = 41
ISP_BFN = 42
ISP_AXI = 43
ISP_AXO = 44
ISP_BTN = 45
ISP_BTC = 46
ISP_BTT = 47
ISP_RIP = 48
ISP_SSH = 49

# Enum for IS_TINY sub-type
TINY_NONE = 0
TINY_VER = 1
TINY_CLOSE = 2
TINY_PING = 3
TINY_REPLY = 4
TINY_VTC = 5
TINY_SCP = 6
TINY_SST = 7
TINY_GTH = 8
TINY_MPE = 9
TINY_ISM = 10
TINY_REN = 11
TINY_CLR = 12
TINY_NCN = 13
TINY_NPL = 14
TINY_RES = 15
TINY_NLP = 16
TINY_MCI = 17
TINY_REO = 18
TINY_RST = 19
TINY_AXI = 20
TINY_AXC = 21
TINY_RIP = 22

# Enum for IS_SMALL sub-type
SMALL_NONE = 0
SMALL_SSP = 1
SMALL_SSG = 2
SMALL_VTA = 3
SMALL_TMS = 4
SMALL_STP = 5
SMALL_RTP = 6
SMALL_NLI = 7

# Bit flags for ISI Flags
ISF_RES_0 = 1
ISF_RES_1 = 2
ISF_LOCAL = 4
ISF_MSO_COLS = 8
ISF_NLP = 16
ISF_MCI = 32

# Enum for IS_MSO UserType
MSO_SYSTEM = 0
MSO_USER = 1
MSO_PREFIX = 2
MSO_O = 3

# Enum for IS_MSL Sound
SND_SILENT = 0
SND_MESSAGE = 1
SND_SYSMESSAGE = 2
SND_INVALIDKEY = 3
SND_ERROR = 4

# Enum for IS_VTN Action
VOTE_NONE = 0
VOTE_END = 1
VOTE_RESTART = 2
VOTE_QUALIFY = 3

# Enum for IS_PLA Fact
PITLANE_EXIT = 0
PITLANE_ENTER = 1
PITLANE_NO_PURPOSE = 2
PITLANE_DT = 3
PITLANE_SG = 4

# Enum for IS_STA InGameCam
VIEW_FOLLOW = 0
VIEW_HELI = 1
VIEW_CAM = 2
VIEW_DRIVER = 3
VIEW_CUSTOM = 4
VIEW_ANOTHER = 255

# Enum for IS_CNL Reason
LEAVR_DISCO = 0
LEAVR_TIMEOUT = 1
LEAVR_LOSTCONN = 2
LEAVR_KICKED = 3
LEAVR_BANNED = 4
LEAVR_SECURITY = 5

# Enum for IS_PEN Penalty
PENALTY_NONE = 0
PENALTY_DT = 1
PENALTY_DT_VALID = 2
PENALTY_SG = 3
PENALTY_SG_VALID = 4
PENALTY_30 = 5
PENALTY_45 = 6

# Enum for IS_PEN Reason
PENR_UNKNOWN = 1
PENR_ADMIN = 2
PENR_WRONG_WAY = 3
PENR_FALSE_START = 4
PENR_SPEEDING = 5
PENR_STOP_SHORT = 6
PENR_STOP_LATE = 7

# Enum for IS_PIT Tyres
TYRE_R1 = 0
TYRE_R2 = 1
TYRE_R3 = 2
TYRE_R4 = 3
TYRE_ROAD_SUPER = 4
TYRE_ROAD_NORMAL = 5
TYRE_HYBRID = 6
TYRE_KNOBBLY = 7
TYRE_NOT_CHANGED = 255

# Bit flags for IS_STA Flags
ISS_GAME = 1
ISS_REPLAY = 2
ISS_PAUSED = 4
ISS_SHIFTU = 8
ISS_SHIFTU_HIGH = 16
ISS_SHIFTU_FOLLOW = 32
ISS_SHIFTU_NO_OPT = 64
ISS_SHOW_2D = 128
ISS_FRONT_END = 256
ISS_MULTI = 512
ISS_MPSPEEDUP =    1024
ISS_WINDOWED = 2048
ISS_SOUND_MUTE = 4096
ISS_VIEW_OVERRIDE = 8192
ISS_VISIBLE = 16384

# Bit flags for IS_PIT Work
PSE_NOTHING = 1
PSE_STOP = 2
PSE_FR_DAM = 4
PSE_FR_WHL = 8
PSE_LE_FR_DAM = 16
PSE_LE_FR_WHL = 32
PSE_RI_FR_DAM = 64
PSE_RI_FR_WHL = 128
PSE_RE_DAM = 256
PSE_RE_WHL = 512
PSE_LE_RE_DAM = 1024
PSE_LE_RE_WHL = 2048
PSE_RI_RE_DAM = 4096
PSE_RI_RE_WHL = 8192
PSE_BODY_MINOR = 16384
PSE_BODY_MAJOR = 32768
PSE_SETUP = 65536
PSE_REFUEL = 131072
PSE_NUM = 262144

# Bit flags for IS_NPL Flags
PIF_SWAPSIDE = 1
PIF_RESERVED_2 = 2
PIF_RESERVED_4 = 4
PIF_AUTOGEARS = 8
PIF_SHIFTER = 16
PIF_RESERVED_32 = 32
PIF_HELP_B = 64
PIF_AXIS_CLUTCH = 128
PIF_INPITS = 256
PIF_AUTOCLUTCH = 512
PIF_MOUSE = 1024
PIF_KB_NO_HELP = 2048
PIF_KB_STABILISED = 4096
PIF_CUSTOM_VIEW = 8192

# Bit flags for IS_RES Confirm
CONF_MENTIONED = 1
CONF_CONFIRMED = 2
CONF_PENALTY_DT = 4
CONF_PENALTY_SG = 8
CONF_PENALTY_30 = 16
CONF_PENALTY_45 = 32
CONF_DID_NOT_PIT = 64
CONF_DISQ = (CONF_PENALTY_DT | CONF_PENALTY_SG | CONF_DID_NOT_PIT)
CONF_TIME = (CONF_PENALTY_30 | CONF_PENALTY_45)

# Bit flags for IS_RST Flags
HOSTF_CAN_VOTE = 1
HOSTF_CAN_SELECT = 2
HOSTF_MID_RACE = 32
HOSTF_MUST_PIT = 64
HOSTF_CAN_RESET = 128
HOSTF_FCV = 256
HOSTF_CRUISE = 512

# Bit flags for CompCar Info
CCI_BLUE = 1
CCI_YELLOW = 2
CCI_LAG    = 32
CCI_FIRST = 64
CCI_LAST = 128

# Enum for IS_BFN SubT
BFN_DEL_BTN = 0
BFN_CLEAR = 1
BFN_USER_CLEAR = 2
BFN_REQUEST = 3
INST_ALWAYS_ON = 128

# Bit flags for IS_BTN BStyle
ISB_C1 = 1
ISB_C2 = 2
ISB_C4 = 4
ISB_CLICK = 8
ISB_LIGHT = 16
ISB_DARK = 32
ISB_LEFT = 64
ISB_RIGHT = 128

# Bit flags for BTN CFlags
ISB_LMB = 1
ISB_RMB = 2
ISB_CTRL = 4
ISB_SHIFT = 8

# Enum for IS_RIP Error
RIP_OK = 0
RIP_ALREADY = 1
RIP_DEDICATED = 2
RIP_WRONG_MODE = 3
RIP_NOT_REPLAY = 4
RIP_CORRUPTED = 5
RIP_NOT_FOUND = 6
RIP_UNLOADABLE = 7
RIP_DEST_OOB = 8
RIP_UNKNOWN = 9
RIP_USER = 10
RIP_OOS = 11

# Enum for IS_RIP Options
RIPOPT_LOOP = 1
RIPOPT_SKINS = 2

# Enum for IS_SSH Error
SSH_OK = 0
SSH_DEDICATED = 1
SSH_CORRUPTED = 2
SSH_NO_SAVE = 3

# Bit flags for IS_NPL SetF
SETF_SYMM_WHEELS = 1
SETF_TC_ENABLE = 2
SETF_ABS_ENABLE = 4


def _eat_null_chars(str):
    return str.rstrip('\000')


class IS_ISI(object):
    """InSim Init - packet to initialise the InSim system.

    """    
    pack_struct = struct.Struct('4B2HBcH15sx15sx')
    def __init__(self, ReqI=0, Zero=0, UDPPort=0, Flags=0, Sp0=0, Prefix='\x00', Interval=0, Admin='', IName='pyinsim'):
        """Create a new IS_ISI packet.

        Args:
            ReqI     : If non-zero LFS will send an ``IS_VER`` packet
            UDPPort  : Port for UDP replies from LFS (0 to 65535)
            Flags    : ``ISF_*`` bit flags for options
            Prefix   : Special host message prefix character
            Interval : Time in ms between ``IS_NLP`` or ``IS_MCI`` packets (0 = none)
            Admin    : Admin password (if set in LFS)
            IName    : A short name for your program

        """        
        self.Size = 44
        self.Type = ISP_ISI
        self.ReqI = ReqI
        self.Zero = Zero
        self.UDPPort = UDPPort
        self.Flags = Flags
        self.Sp0 = Sp0
        self.Prefix = Prefix
        self.Interval = Interval
        self.Admin = Admin
        self.IName = IName
    def pack(self):
        return self.pack_struct.pack(self.Size, self.Type, self.ReqI, self.Zero, self.UDPPort, self.Flags, self.Sp0, self.Prefix, self.Interval, self.Admin, self.IName)

class IS_VER(object):
    """VERsion. 

    """    
    pack_struct = struct.Struct('4B7sx5sxH')
    def __init__(self, ReqI=0, Zero=0, Version='', Product='', InSimVer=0):
        """Initialise a new IS_VER packet.

        """        
        self.Size = 20
        self.Type = ISP_VER
        self.ReqI = ReqI
        self.Zero = Zero
        self.Version = Version
        self.Product = Product
        self.InSimVer = InSimVer
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.Version, self.Product, self.InSimVer = self.pack_struct.unpack(data)
        self.Product = _eat_null_chars(self.Product)
        self.Version = _eat_null_chars(self.Version)
        return self

class IS_TINY(object):
    """General purpose packet.

    """    
    struct_pack = struct.Struct('4B')
    def __init__(self, ReqI=0, SubT=TINY_NONE):
        """Initialise a new IS_TINY packet.

        Args:
            ReqI : zero (0) unless in response to a request.
            SubT : subtype from ``TINY_*`` enumeration (e.g. ``TINY_REN``)

        """        
        self.Size = 4
        self.Type = ISP_TINY
        self.ReqI = ReqI
        self.SubT = SubT
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.SubT)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.SubT = self.struct_pack.unpack(data)
        return self

class IS_SMALL(object):
    """General purpose packet.

    """    
    struct_pack = struct.Struct('4BI')
    def __init__(self, ReqI=0, SubT=SMALL_NONE, UVal=0):
        """Initialise a new IS_SMALL packet.

        Args:
            ReqI : zero (0) unless in response to a request.
            SubT : subtype from ``SMALL_*`` enumeration (e.g. ``SMALL_SSP``)
            UVal : value (e.g. for ``SMALL_SSP`` this would be the OutSim packet rate)

        """        
        self.Size = 8
        self.Type = ISP_SMALL
        self.ReqI = ReqI
        self.SubT = SubT
        self.UVal = UVal
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.SubT, self.UVal)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.SubT, self.UVal = self.struct_pack.unpack(data)
        return self
    
class IS_STA(object):
    """STAte packet, sent whenever the data in the packet changes. To request
    this packet send a ``IS_TINY`` with a ``ReqI`` of non-zero and a ``SubT`` of ``TINY_STA``.

    """    
    struct_pack = struct.Struct('4BfH10B5sx2B')
    def __init__(self):
        """Initialise a new IS_STA packet.

        """        
        self.Size = 28
        self.Type = ISP_STA
        self.ReqI = 0
        self.Zero = 0
        self.ReplaySpeed = 0.0
        self.Flags = 0
        self.InGameCam = 0
        self.ViewPLID = 0
        self.NumP = 0
        self.NumConns = 0
        self.NumFinished = 0
        self.RaceInProg = 0
        self.QualMins = 0
        self.RaceLaps = 0
        self.Spare2 = 0
        self.Spare3 = 0
        self.Track = ''
        self.Weather = 0
        self.Wind = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.ReplaySpeed, self.Flags, self.InGameCam, self.ViewPLID, self.NumP, self.NumConns, self.NumFinished, self.RaceInProg, self.QualMins, self.RaceLaps, self.Spare2, self.Spare3, self.Track, self.Weather, self.Wind = self.struct_pack.unpack(data)
        self.Track = _eat_null_chars(self.Track)
        return self

class IS_SCH(object):
    """Single CHaracter

    """
    struct_pack = struct.Struct('4Bc3B')
    def __init__(self, ReqI=0, CharB='\x00', Flags=0):
        """Initialise a new IS_SCH packet.

        Args:
            ReqI  : 0
            CharB : key to press
            Flags : bit 0 : SHIFT / bit 1 : CTRL

        """
        self.Size = 8
        self.Type = ISP_SCH
        self.ReqI = ReqI
        self.Zero = 0
        self.CharB = CharB
        self.Flags = Flags
        self.Spare2 = 0
        self.Spare3 = 0
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Zero, self.CharB, self.Flags, self.Spare2, self.Spare3)

class IS_SFP(object):
    """State Flags Pack. Send this packet to set the game state. Other states
    must be set by using key-presses or slash commands.

    """
    struct_pack = struct.Struct('4BH2B')
    def __init__(self, ReqI=0, Flag=0, OffOn=0):
        """Initialise a new IS_SFP packet.

        Args:
            ReqI  : ReqI as received in the request packet
            Flag  : ``ISS_*`` state flags
            OffOn : 0 = off / 1 = on

        """
        self.Size = 8
        self.Type = ISP_SFP
        self.ReqI = ReqI
        self.Zero = 0
        self.Flag = Flag
        self.OffOn = OffOn
        self.Sp3 = 0
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Zero, self.Flag, self.OffOn, self.Sp3)

class IS_SCC(object):
    """Set Car Camera - Simplified camera packet (not SHIFT+U mode)

    """
    struct_pack = struct.Struct('8B')
    def __init__(self, ReqI=0, ViewPLID=0, InGameCam=0):
        """Initialise a new IS_SCC packet.

        Args:
            ReqI      : 0
            ViewPLID  : UniqueID of player to view
            InGameCam : InGameCam (as reported in StatePack)

        """
        self.Size = 8
        self.Type = ISP_SCC
        self.ReqI = ReqI
        self.Zero = 0
        self.ViewPLID = ViewPLID
        self.InGameCam = InGameCam
        self.Sp2 = 0
        self.Sp3 = 0
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Zero, self.ViewPLID, self.InGameCam, self.Sp2, self.Sp3)

class IS_CPP(object):
    """Cam Pos Pack - Full camera packet (in car or SHIFT+U mode)

    """
    struct_pack = struct.Struct('4B3i3H2Bf2H')
    def __init__(self, ReqI=0, Pos=[0,0,0], H=0, P=0, R=0, ViewPLID=0, InGameCam=0, FOV=0.0, Time=0, Flags=0):
        """Initialise a new IS_CPP packet.

        Args:
            ReqI      : instruction : 0 / or reply : ReqI as received in the ``TINY_SCP``
            Pos       : Position vector
            H         : heading - 0 points along Y axis
            P         : pitch - 0 means looking at horizon
            R         : roll - 0 means no roll
            ViewPLID  : Unique ID of viewed player (0 = none)
            InGameCam : InGameCam (as reported in StatePack)
            FOV       : FOV in degrees
            Time      : Time to get there (0 means instant + reset)
            Flags     : state flags from ``ISS_*``

        """
        self.Size = 32
        self.Type = ISP_CPP
        self.ReqI = ReqI
        self.Zero = 0
        self.Pos = Pos
        self.H = H
        self.P = P
        self.R = R
        self.ViewPLID = ViewPLID
        self.InGameCam = InGameCam
        self.FOV = FOV
        self.Time = Time
        self.Flags = Flags
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Zero, self.Pos[0], self.Pos[1], self.Pos[2], self.H, self.P, self.R, self.ViewPLID, self.InGameCam, self.FOV, self.Time, self.Flags)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.Pos[0], self.Pos[1], self.Pos[2], self.H, self.P, self.R, self.ViewPLID, self.InGameCam, self.FOV, self.Time, self.Flags = self.struct_pack.unpack(data)
        return self

class IS_ISM(object):
    """InSim Multi

    LFS will send this packet when a host is started or joined.

    """
    struct_pack = struct.Struct('8B31sx')
    def __init__(self):
        """Initialise a new IS_ISM packet.

        """
        self.Size = 40
        self.Type = ISP_ISM
        self.ReqI = 0
        self.Zero = 0
        self.Host = 0
        self.Sp1 = 0
        self.Sp2 = 0
        self.Sp3 = 0
        self.HName = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.Host, self.Sp1, self.Sp2, self.Sp3, self.HName = self.struct_pack.unpack(data)
        self.HName = _eat_null_chars(self.HName)
        return self

class IS_MSO(object):
    """MSg Out - system messages and user messages

    """
    struct_pack = struct.Struct('8B127sx')
    def __init__(self):
        """Initialise a new IS_MSO packet.

        """
        self.Size = 136
        self.Type = ISP_MSO
        self.ReqI = 0
        self.Zero = 0
        self.UCID =0
        self.PLID = 0
        self.UserType = 0
        self.TextStart = 0
        self.Msg = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.UCID, self.PLID, self.UserType, self.TextStart, self.Msg = self.struct_pack.unpack(data)
        self.Msg = _eat_null_chars(self.Msg)
        return self

class IS_III(object):
    """InsIm Info - /i message from user to host's InSim

    """
    struct_pack = struct.Struct('8B63sx')
    def __init__(self):
        """Initialise a new IS_III packet.

        """
        self.Size = 72
        self.Type = ISP_III
        self.ReqI = 0
        self.Zero = 0
        self.UCID = 0
        self.PLID = 0
        self.Sp2 = 0
        self.Sp3 = 0
        self.Msg = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.UCID, self.PLID, self.Sp2, self.Sp3, self.Msg = self.struct_pack.unpack(data)
        self.Msg = _eat_null_chars(self.Msg)
        return self

class IS_MST(object):
    """MSg Type - send to LFS to type message or command

    """
    struct_pack = struct.Struct('4B63sx')
    def __init__(self, ReqI=0, Msg=''):
        """Initialise a new IS_MST packet.

        Args:
            ReqI : 0
            Msg  : message (64 characters)

        """
        self.Size = 68
        self.Type = ISP_MST
        self.ReqI = ReqI
        self.Zero = 0
        self.Msg = Msg
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Zero, self.Msg)

class IS_MTC(object):
    """Msg To Connection - hosts only - send to a connection or a player

    """
    struct_pack = struct.Struct('8B63sx')
    def __init__(self, ReqI=0, UCID=0, PLID=0, Msg=''):
        """Initialise a new IS_MTC packet.

        Args:
            ReqI : 0
            UCID : connection's unique id (0 = host)
            PLID : player's unique id (if zero, use :attr:`UCID`)
            Msg  : Message (64 characters)

        """
        self.Size = 72
        self.Type = ISP_MTC
        self.ReqI = ReqI
        self.Zero = 0
        self.UCID = UCID
        self.PLID = PLID
        self.Sp2 = 0
        self.Sp3 = 0
        self.Msg = Msg
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Zero, self.UCID, self.PLID, self.Sp2, self.Sp3, self.Msg)

class IS_MOD(object):
    """MODe : send to LFS to change screen mode

    """
    struct_pack = struct.Struct('4B4i')
    def __init__(self, ReqI=0, Bits16=0, RR=0, Width=0, Height=0):
        """Initialise a new IS_MOD packet.

        Args:
            ReqI   : 0
            Bits16 : set to choose 16-bit
            RR     : refresh rate - zero for default
            Width  : 0 means go to window
            Height : 0 means go to window

        """
        self.Size = 20
        self.Type = ISP_MOD
        self.ReqI = ReqI
        self.Zero = 0
        self.Bits16 = Bits16
        self.RR = RR
        self.Width = Width
        self.Height = Height
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Zero, self.Bits16, self.RR, self.Width, self.Height)

class IS_VTN(object):
    """VoTe Notify

    """
    struct_pack = struct.Struct('8B')
    def __init__(self, ReqI=0, UCID=0, Action=0):
        """Initialise a new IS_VTN packet.

        Args:
            ReqI   : 0
            UCID   : connection's unique id
            Action : Vote action from ``VOTE_*``

        """
        self.Size = 8
        self.Type = ISP_VTN
        self.ReqI = ReqI
        self.Zero = 0
        self.UCID = UCID
        self.Action = Action
        self.Spare2 = 0
        self.Spare3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.UCID, self.Action, self.Spare2, self.Spare3 = self.struct_pack.unpack(data)
        return self

class IS_RST(object):
    """Race STart

    """
    struct_pack = struct.Struct('8B5sx2B6H')
    def __init__(self):
        """Initialise a new IS_RST packet.

        """
        self.Size = 28
        self.Type = ISP_RST
        self.ReqI = 0
        self.Zero = 0
        self.RaceLaps = 0
        self.QualMins = 0
        self.NumP = 0
        self.Spare = 0
        self.Track = ''
        self.Weather = 0
        self.Wind = 0
        self.Flags = 0
        self.NumNodes = 0
        self.Finish = 0
        self.Split1 = 0
        self.Split2 = 0
        self.Split3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.RaceLaps, self.QualMins, self.NumP, self.Spare, self.Track, self.Weather, self.Wind, self.Flags, self.NumNodes, self.Finish, self.Split1, self.Split2, self.Split3 = self.struct_pack.unpack(data)
        self.Track = _eat_null_chars(self.Track)
        return self

class IS_NCN(object):
    """New ConN

    """
    struct_pack = struct.Struct('4B23sx23sx4B')
    def __init__(self):
        """Initialise a new IS_NCN packet.

        """
        self.Size = 56
        self.Type = ISP_NCN
        self.ReqI = 0
        self.UCID = 0
        self.UName = ''
        self.PName = ''
        self.Admin = 0
        self.Total = 0
        self.Flags = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.UName, self.PName, self.Admin, self.Total, self.Flags, self.Sp3 = self.struct_pack.unpack(data)
        self.UName = _eat_null_chars(self.UName)
        self.PName = _eat_null_chars(self.PName)
        return self

class IS_CNL(object):
    """ConN Leave

    """
    struct_pack = struct.Struct('8B')
    def __init__(self):
        """Initialise a new IS_CNL packet.

        """
        self.Size = 8
        self.Type = ISP_CNL
        self.ReqI = 0
        self.UCID = 0
        self.Reason = 0
        self.Total = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.Reason, self.Total, self.Sp2, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_CPR(object):
    """Conn Player Rename

    """
    struct_pack = struct.Struct('4B23sx7sx')
    def __init__(self):
        """Initialise a new IS_CPR packet.

        """
        self.Size = 36
        self.Type = ISP_CPR
        self.ReqI = 0
        self.UCID = 0
        self.PName = ''
        self.Plate = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.PName, self.Plate = self.struct_pack.unpack(data)
        self.PName = _eat_null_chars(self.PName)
        #self.Plate = _eat_null_chars(self.Plate) # No trailing zero on Plate.
        return self

class IS_NPL(object):
    """New PLayer joining race (if PLID already exists, then leaving pits)

    """
    struct_pack = struct.Struct('6BH23sx8s3sx15sx8Bi4B')
    def __init__(self):
        """Initialise a new IS_NPL packet.
        
        """
        self.Size = 76
        self.Type = ISP_NPL
        self.ReqI = 0
        self.PLID = 0
        self.UCID = 0
        self.PType = 0
        self.Flags = 0
        self.PName = ''
        self.Plate = ''
        self.CName = ''
        self.SName = ''
        self.Tyres = [0, 0, 0, 0]
        self.H_Mass = 0
        self.H_TRes = 0
        self.Model = 0
        self.Pass = 0
        self.Spare = 0
        self.SetF = 0
        self.NumP = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.UCID, self.PType, self.Flags, self.PName, self.Plate, self.CName, self.SName, self.Tyres[0], self.Tyres[1], self.Tyres[2], self.Tyres[3], self.H_Mass, self.H_TRes, self.Model, self.Pass, self.Spare, self.SetF, self.NumP, self.Sp2, self.Sp3 = self.struct_pack.unpack(data)
        self.PName = _eat_null_chars(self.PName)
        #self.Plate = _eat_null_chars(self.Plate) # No trailing zero
        self.CName = _eat_null_chars(self.CName)
        self.SName = _eat_null_chars(self.SName)
        return self

class IS_PLP(object):
    """PLayer Pits (go to settings - stays in player list)

    """
    struct_pack = struct.Struct('4B')
    def __init__(self):
        """Initialise a new IS_PLP packet.

        """
        self.Size = 4
        self.Type = ISP_PLP
        self.ReqI = 0
        self.PLID = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID = self.struct_pack.unpack(data)
        return self

class IS_PLL(object):
    """PLayer Leave race (spectate - removed from player list)

    """
    struct_pack = struct.Struct('4B')
    def __init__(self):
        """Initialise a new IS_PLL packet.

        """
        self.Size = 4
        self.Type = ISP_PLL
        self.ReqI = 0
        self.PLID = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID = self.struct_pack.unpack(data)
        return self

class IS_LAP(object):
    """LAP time

    """
    struct_pack = struct.Struct('4B2I2H4B')
    def __init__(self):
        """Initialise a new IS_LAP packet.

        """
        self.Size = 20
        self.Type = ISP_LAP
        self.ReqI = 0
        self.PLID = 0
        self.LTime = 0
        self.ETime = 0
        self.LapsDone = 0
        self.Flags = 0
        self.Sp0 = 0
        self.Penalty = 0
        self.NumStops = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.LTime, self.ETime, self.LapsDone, self.Flags, self.Sp0, self.Penalty, self.NumStops, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_SPX(object):
    """SPlit X time

    """
    struct_pack = struct.Struct('4B2I4B')
    def __init__(self):
        """Initialise a new IS_SPX packet.

        """
        self.Size = 16
        self.Type = ISP_SPX
        self.ReqI = 0
        self.PLID = 0
        self.STime = 0
        self.ETime = 0
        self.Split = 0
        self.Penalty = 0
        self.NumStops = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.STime, self.ETime, self.Split, self.Penalty, self.NumStops, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_PIT(object):
    """PIT stop (stop at pit garage)

    """
    struct_pack = struct.Struct('4B2H8B2I')
    def __init__(self):
        """Initialise a new IS_PIT packet.

        """
        self.Size = 24
        self.Type = ISP_PIT
        self.ReqI = 0
        self.PLID = 0
        self.LapsDone = 0
        self.Flags = 0
        self.Sp0 = 0
        self.Penalty = 0
        self.NumStops = 0
        self.Sp3 = 0
        self.Tyres = [0, 0, 0, 0]
        self.Work = 0
        self.Spare = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.LapsDone, self.Flags, self.Sp0, self.Penalty, self.NumStops, self.Sp3, self.Tyres[0], self.Tyres[1], self.Tyres[2], self.Tyres[3], self.Work, self.Spare = self.struct_pack.unpack(data)
        return self

class IS_PSF(object):
    """Pit Stop Finished

    """
    struct_pack = struct.Struct('4B2I')
    def __init__(self):
        """Initialise a new IS_PSF packet.

        """
        self.Size = 12
        self.Type = ISP_PSF
        self.ReqI = 0
        self.PLID = 0
        self.STime = 0
        self.Spare = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.STime, self.Spare = self.struct_pack.unpack(data)
        return self

class IS_PLA(object):
    """Pit LAne

    """
    struct_pack = struct.Struct('8B')
    def __init__(self):
        """Initialise a new IS_PLA packet.

        """
        self.Size = 8
        self.Type = ISP_PLA
        self.ReqI = 0
        self.PLID = 0
        self.Fact = 0
        self.Sp1 = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.Fact, self.Sp1, self.Sp2, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_CCH(object):
    """Camera CHange

    """
    struct_pack = struct.Struct('8B')
    def __init__(self):
        """Initialise a new IS_CCH packet.

        """
        self.Size = 8
        self.Type = ISP_CCH
        self.ReqI = 0
        self.PLID = 0
        self.Camera = 0
        self.Sp1 = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.Camera, self.Sp1, self.Sp2, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_PEN(object):
    """PENalty (given or cleared)

    """
    struct_pack = struct.Struct('8B')
    def __init__(self):
        """Initialise a new IS_PEN packet.

        """
        self.Size = 8
        self.Type = ISP_PEN
        self.ReqI = 0
        self.PLID = 0
        self.OldPen = 0
        self.NewPen = 0
        self.Reason = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.OldPen, self.NewPen, self.Reason, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_TOC(object):
    """Take Over Car

    """
    struct_pack = struct.Struct('8B')
    def __init__(self, ReqI=0, PLID=0, OldUCID=0, NewUCID=0, Sp2=0, Sp3=0):
        """Initialise a new IS_TOC packet.

        """
        self.Size = 8
        self.Type = ISP_TOC
        self.ReqI = 0
        self.PLID = 0
        self.OldUCID = 0
        self.NewUCID = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.OldUCID, self.NewUCID, self.Sp2, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_FLG(object):
    """FLaG (yellow or blue flag changed)

    """
    struct_pack = struct.Struct('8B')
    def __init__(self):
        """Initialise a new IS_FLG packet.

        """
        self.Size = 8
        self.Type = ISP_FLG
        self.ReqI = 0
        self.PLID = 0
        self.OffOn = 0
        self.Flag = 0
        self.CarBehind = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.OffOn, self.Flag, self.CarBehind, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_PFL(object):
    """Player FLags (help flags changed)

    """
    struct_pack = struct.Struct('4B2H')
    def __init__(self, ReqI=0, PLID=0, Flags=0, Spare=0):
        """Initialise a new IS_PFL packet.

        """
        self.Size = 8
        self.Type = ISP_PFL
        self.ReqI = 0
        self.PLID = 0
        self.Flags = 0
        self.Spare = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.Flags, self.Spare = self.struct_pack.unpack(data)
        return self

class IS_FIN(object):
    """FINished race notification (not a final result - use :class:`IS_RES`)

    """
    struct_pack = struct.Struct('4B2I4B2H')
    def __init__(self):
        """Initialise a new IS_FIN packet.

        """
        self.Size = 20
        self.Type = ISP_FIN
        self.ReqI = 0
        self.PLID = 0
        self.TTime = 0
        self.BTime = 0
        self.SpA = 0
        self.NumStops = 0
        self.Confirm = 0
        self.SpB = 0
        self.LapsDone = 0
        self.Flags = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.TTime, self.BTime, self.SpA, self.NumStops, self.Confirm, self.SpB, self.LapsDone, self.Flags = self.struct_pack.unpack(data)
        return self

class IS_RES(object):
    """RESult (qualify or confirmed finish)

    """
    struct_pack = struct.Struct('4B23sx23sx7sx3sx2I4B2H2BH')
    def __init__(self):
        """Initialise a new IS_RES packet.

        """
        self.Size = 84
        self.Type = ISP_RES
        self.ReqI = 0
        self.PLID = 0
        self.UName = ''
        self.PName = ''
        self.Plate = ''
        self.CName = ''
        self.TTime = 0
        self.BTime = 0
        self.SpA = 0
        self.NumStops = 0
        self.Confirm = 0
        self.SpB = 0
        self.LapsDone = 0
        self.Flags = 0
        self.ResultNum = 0
        self.NumRes = 0
        self.PSeconds = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.UName, self.PName, self.Plate, self.CName, self.TTime, self.BTime, self.SpA, self.NumStops, self.Confirm, self.SpB, self.LapsDone, self.Flags, self.ResultNum, self.NumRes, self.PSeconds = self.struct_pack.unpack(data)
        self.UName = _eat_null_chars(self.UName)
        self.PName = _eat_null_chars(self.PName)
        #self.Plate = _eat_null_chars(self.Plate) # No trailing zero
        self.CName = _eat_null_chars(self.CName)
        return self

class IS_REO(object):
    """REOrder (when race restarts after qualifying)

    """
    struct_pack = struct.Struct('4B')
    def __init__(self, ReqI=0, NumP=0, PLID=[]):
        """Initialise a new IS_reo packet.

        Args:
            ReqI : 0 unless this is a reply to an ``TINY_REO`` request
            NumP : number of players in race
            PLID : all PLIDs in new order

        """
        self.Size = 36
        self.Type = ISP_REO
        self.ReqI = ReqI
        self.NumP = len(PLID)
        self.PLID = PLID
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.NumP) + struct.pack('%dB' % self.NumP, *self.PLID).ljust(32, '\x00')
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.NumP = self.struct_pack.unpack(data[:4])
        self.PLID = data[4:4+self.NumP]
        return self

class IS_NLP(object):
    """Node and Lap Packet - variable size

    """
    struct_pack = struct.Struct('4B')
    def __init__(self):
        """Initialise a new IS_NLP packet.

        """
        self.Size = 4
        self.Type = ISP_NLP
        self.ReqI = 0
        self.NumP = 0
        self.Info = []
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.NumP = self.struct_pack.unpack(data[:4])
        data = data[4:]
        append = self.Info.append
        nodelap = NodeLap
        for i in xrange(0, self.NumP * 6, 6):
            append(nodelap(data, i))
        return self

class NodeLap(object):
    """Car info in 6 bytes - there is an array of these in the :class:`IS_NLP`

    """
    struct_pack = struct.Struct('2H2B')
    def __init__(self, data, index):
        """Initialise a new NodeLap sub-packet.

        """
        self.Node, self.Lap, self.PLID, self.Position = self.struct_pack.unpack(data[index:index+6])

class IS_MCI(object):
    """Multi Car Info - if more than 8 in race then more than one of these is sent

    """
    struct_pack = struct.Struct('4B')
    def __init__(self):
        """Initialise a new IS_MCI packet.

        """
        self.Size = 4
        self.Type = ISP_MCI
        self.ReqI = 0
        self.NumC = 0
        self.Info = []
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.NumC = self.struct_pack.unpack(data[:4])
        data = data[4:]
        append = self.Info.append
        compcar = CompCar
        for i in xrange(0, self.NumC * 28, 28):
            append(compcar(data, i))
        return self

class CompCar(object):
    """Car info in 28 bytes - there is an array of these in the :class:`IS_MCI`

    """
    struct_pack = struct.Struct('2H4B3i3Hh')
    def __init__(self, data, index):
        """Initialise a new CompCar sub-packet.

        """
        self.Node, self.Lap, self.PLID, self.Position, self.Info, self.Sp3, self.X, self.Y, self.Z, self.Speed, self.Direction, self.Heading, self.AngVel = self.struct_pack.unpack(data[index:index+28])
        
class IS_MSX(object):
    """MSg eXtended - like ``IS_MST`` but longer (not for commands)

    """
    struct_pack = struct.Struct('4B95sx')
    def __init__(self, ReqI=0, Msg=''):
        """Initialise a new IS_MSX packet.

        Args:
            ReqI : 0
            Msg  : last byte must be zero
        
        """
        self.Size = 100
        self.Type = ISP_MSX
        self.ReqI = ReqI
        self.Zero = 0
        self.Msg = Msg
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Zero, self.Msg)

class IS_MSL(object):
    """MSg Local - message to appear on local computer only

    """
    struct_pack = struct.Struct('4B127sx')
    def __init__(self, ReqI=0, Sound=0, Msg=''):
        """Initialise a new IS_MSL packet.

        Args:
            ReqI  : 0
            Sound : Sound from ``SND_*`` enumeration.
            Msg   : Message

        """
        self.Size = 132
        self.Type = ISP_MSL
        self.ReqI = ReqI
        self.Sound = Sound
        self.Msg = Msg
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Sound, self.Msg)

class IS_CRS(object):
    """Car ReSet

    """
    struct_pack = struct.Struct('4B')
    def __init__(self):
        """Initialise a new IS_CRS packet.

        """
        self.Size = 4
        self.Type = ISP_CRS
        self.ReqI = 0
        self.PLID = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID = self.struct_pack.unpack(data)
        return self

class IS_BFN(object):
    """Button FunctioN - delete buttons / receive button requests

    """
    struct_pack = struct.Struct('8B')
    def __init__(self, ReqI=0, SubT=0, UCID=0, ClickID=0, Inst=0):
        """Initialise a new IS_BFN packet.

        Args:
            ReqI    : 0
            SubT    : subtype, from ``BFN_*`` enumeration
            UCID    : connection to send to or from (0 = local / 255 = all)
            ClickID : ID of button to delete (if ``SubT`` is ``BFN_DEL_BTN``)
            Inst    : used internally by InSim

        """
        self.Size = 8
        self.Type = ISP_BFN
        self.ReqI = ReqI
        self.SubT = SubT
        self.UCID = UCID
        self.ClickID = ClickID
        self.Inst = Inst
        self.Sp3 = 0
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.SubT, self.UCID, self.ClickID, self.Inst, self.Sp3)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.SubT, self.UCID, self.ClickID, self.Inst, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_AXI(object):
    """AutoX Info

    """
    struct_pack = struct.Struct('6BH31sx')
    def __init__(self):
        """Initialise a new IS_AXI packet.

        Args:
            ReqI    : 0 unless this is a reply to an ``TINY_AXI`` request
            AXStart : autocross start position
            NumCP   : number of checkpoints
            NumO    : number of objects
            LName   : the name of the layout last loaded (if loaded locally)

        """
        self.Size = 40
        self.Type = ISP_AXI
        self.ReqI = 0
        self.Zero = 0
        self.AXStart = 0
        self.NumCP = 0
        self.NumO = 0
        self.LName = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.AXStart, self.NumCP, self.NumO, self.LName = self.struct_pack.unpack(data)
        self.LName = _eat_null_chars(self.LName)
        return self

class IS_AXO(object):
    """AutoX Object

    """
    struct_pack = struct.Struct('4B')
    def __init__(self):
        """Initialise a new IS_AXO packet.

        Args:
            ReqI : 0
            PLID : player's unique id

        """
        self.Size = 4
        self.Type = ISP_AXO
        self.ReqI = 0
        self.PLID = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID = self.struct_pack.unpack(data)
        return self

class IS_BTN(object):
    """BuTtoN - button header - followed by 0 to 240 characters

    """
    struct_pack = struct.Struct('12B')
    def __init__(self, ReqI=0, UCID=0, ClickID=0, Inst=0, BStyle=0, TypeIn=0, L=0, T=0, W=0, H=0, Text=''):
        """Initialise a new IS_BTN packet.

        Args:
            ReqI    : non-zero (returned in ``IS_BTC`` and ``IS_BTT`` packets)
            UCID    : connection to display the button (0 = local / 255 = all)
            ClickID : button ID (0 to 239)
            Inst    : some extra flags from ``INST_*``
            BStyl   : button style flags from ``ISB_*``
            TypeIn  : max chars to type in
            L       : left: 0 - 200
            T       : top: 0 - 200
            W       : width: 0 - 200
            H       : height: 0 - 200
            Text    : 0 to 240 characters of text

        """
        self.Size = 12
        self.Type = ISP_BTN
        self.ReqI = ReqI
        self.UCID = UCID
        self.ClickID = ClickID
        self.Inst = Inst
        self.BStyle = BStyle
        self.TypeIn = TypeIn
        self.L = L
        self.T = T
        self.W = W
        self.H = H
        self.Text = Text
    def pack(self):
        TEXT_SIZE = int(math.ceil(len(self.Text) / 4.0)) * 4
        return self.struct_pack.pack(self.Size + TEXT_SIZE, self.Type, self.ReqI, self.UCID, self.ClickID, self.Inst, self.BStyle, self.TypeIn, self.L, self.T, self.W, self.H) + struct.pack('%ds' % TEXT_SIZE, self.Text)

class IS_BTC(object):
    """BuTton Click - sent back when user clicks a button

    """
    struct_pack = struct.Struct('8B')
    def __init__(self):
        """Initialise a new IS_BTC packet.

        """
        self.Size = 8
        self.Type = ISP_BTC
        self.ReqI = 0
        self.UCID = 0
        self.ClickID = 0
        self.Inst = 0
        self.CFlags = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.ClickID, self.Inst, self.CFlags, self.Sp3 = self.struct_pack.unpack(data)
        return self

class IS_BTT(object):
    """BuTton Type - sent back when user types into a text entry button

    """
    struct_pack = struct.Struct('8B95sx')
    def __init__(self):
        """Initialise a new IS_BTT packet.

        """
        self.Size = 104
        self.Type = ISP_BTT
        self.ReqI = 0
        self.UCID = 0
        self.ClickID = 0
        self.Inst = 0
        self.TypeIn = 0
        self.Sp3 = 0
        self.Text = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.ClickID, self.Inst, self.TypeIn, self.Sp3, self.Text = self.struct_pack.unpack(data)
        self.Text = _eat_null_chars(self.Text)
        return self

class IS_RIP(object):
    """Replay Information Packet

    """
    struct_pack = struct.Struct('8B2H63sx')
    def __init__(self, ReqI=0, Error=0, MPR=0, Paused=0, Options=0, CTime=0, TTime=0, RName=''):
        """Initialise a new IS_RIP packet.

        Args:
            ReqI    : request : non-zero / reply : same value returned
            Error   : 0 or 1 = OK / options from ``RIP_*``
            MPR     : 0 = SPR / 1 = MPR
            Paused  : request : pause on arrival / reply : paused state
            Options : various options from ``RIPOPT_*``
            CTime   : (hundredths) request : destination / reply : position
            TTime   : (hundredths) request : zero / reply : replay length
            RName   : zero or replay name: last byte must be zero

        """
        self.Size = 76
        self.Type = ISP_RIP
        self.ReqI = ReqI
        self.Error = Error
        self.MPR = MPR
        self.Paused = Paused
        self.Options = Options
        self.Sp3 = 0
        self.CTime = CTime
        self.TTime = TTime
        self.RName = RName
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Error, self.MPR, self.Paused, self.Options, self.Sp3, self.CTime, self.TTime, self.RName)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Error, self.MPR, self.Paused, self.Options, self.Sp3, self.CTime, self.TTime, self.RName = self.struct_pack.unpack(data)
        self.RName = _eat_null_chars(self.RName)
        return self

class IS_SSH(object):
    """ScreenSHot

    """
    struct_pack = struct.Struct('8B31sx')
    def __init__(self, ReqI=0, Error=0, BMP=''):
        """Initialise a new IS_SSH packet.

        Args:
            ReqI  : request : non-zero / reply : same value returned
            Error : 0 = OK / other values from ``SSH_*``
            BMP   : name of screenshot file: last byte must be zero

        """
        self.Size = 40
        self.Type = ISP_SSH
        self.ReqI = ReqI
        self.Error = Error
        self.Sp0 = 0
        self.Sp1 = 0
        self.Sp2 = 0
        self.Sp3 = 0
        self.BMP = BMP
    def pack(self):
        return self.struct_pack.pack(self.Size, self.Type, self.ReqI, self.Error, self.Sp0, self.Sp1, self.Sp2, self.Sp3, self.BMP)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Error, self.Sp0, self.Sp1, self.Sp2, self.Sp3, self.BMP = self.struct_pack.unpack(data)
        self.BMP = _eat_null_chars(self.BMP)
        return self 
    