#!/usr/bin/env python

import re
import strmanip
from insim import *


COLOUR_REGEX = re.compile('\^[0-9]')


class Plugin(object):
    """Class for implementing a pyinsim plugin."""
    
    # These are non-instanced so all plugins share them.
    host_map = {}
    plugin_map = {}
    message_map = {}
    
    def __init__(self):
        """Create a new plugin."""
        self.callback_map = {}
        self.data = {}
        self.name = self.__class__.__name__
    
    def initialized(self):
        """Called when the plugin is initialized."""
        pass
    
    def uninitialized(self):
        """Called when the plugin is uninitialized."""
        pass
    
    def connected(self, host):
        """Called when a host is connected."""
        pass
    
    def disconnected(self, host):
        """Called when a host disconnects."""
        pass
    
    def handle_packet(self, host, packet):
        """Called when a packet is received from a host."""
        type = packet.Type
        if type in self.callback_map:
            [c(host, packet) for c in self.callback_map[type]]
    
    def bind(self, type, callback):
        """Bind a packet callback handler."""
        if type in self.callback_map:
            self.callback_map[type].append(callback)
        else:
            self.callback_map[type] = [callback]
    
    def unbind(self, type, callback):
        """Unbind a packet callback handler."""
        if type in self.callback_map and callback in self.callback_map[type]:
            self.callback_map[type].remove(callback)
            if not self.callback_map[type]:
                del self.callback_map[type]
    
    def subscribe(self, msg, callback):
        """Subscribe to messages from other plugins."""
        msg = msg.lower()
        if msg in self.message_map:
            self.message_map[msg].append(callback)
        else:
            self.message_map[msg] = [callback]
    
    def unsubcribe(self, msg, callback):
        """Unsubscribe from messages from other plugins."""
        msg = msg.lower()
        if msg in self.message_map and callback in self.message_map[msg]:
            self.message_map[msg].remove(callback)
            if not self.message_map[msg]:
                del self.message_map[msg]
    
    def notify(self, msg, *args):
        """Notify other plugins of a message."""
        msg = msg.lower()
        if msg in self.message_map:
            [m(*args) for m in self.message_map[msg]]
            
    def find_plugin(self, name):
        """Find another plugin by name"""
        if name in self.plugin_map:
            return self.plugin_map[name]
        return None
        
    
def get_plugin_subclasses():
    return Plugin.__subclasses__()    
           
def stripcols(str):
    """Strip color codes from a string."""
    return COLOUR_REGEX.sub('', str)

def tounicode(str, cols=True, default='L'):
    """Convert a LFS encoded string to unicode."""
    return strmanip.toUnicode(str, default, cols)

def fromunicode(ustr, default='L'):
    """Convert a uncode string to a LFS encoded string."""
    return strmanip.fromUnicode(ustr, default)

def time(ms):
    """Convert milliseconds into hours, minutes, seconds and thousanths."""
    h = ms / 3600000
    m = ms / 60000 % 60
    s = ms / 1000 % 60
    t = ms % 1000
    return [h, m, s, t]

def timestr(ms, hours=False):
    """Convert milliseconds into a formatted time string (e.g. h:mm:ss.ttt)."""
    h, m, s, t = time(ms)
    if h or hours:
        return '%d.%.2d:%.2d.%.3d' % (h, m, s, t)
    return '%d:%.2d.%.3d' % (m, s, t)

def mps(speed):
    """Convert speed to meters per second."""
    return speed / 327,68

def mph(speed):
    """Convert speed to miles per hour."""
    return speed / 146.486067

def kph(speed):
    """Convert speed to kilometers per hour."""
    return speed / 91.02

def length(length):
    """Convert LFS length into meters."""
    return length / 65536.0

def miles(length):
    """Convert length to miles."""
    return length(length) / 1609.344

def km(length):
    """Convert length to kilometers."""
    return length(length) / 1000.0

def deg(radians):
    """Convert radians to degrees."""
    return radians * 57.295773

def rad(degrees):
    """Convert degrees to radians."""
    return degrees * 0.01745329

def rpm(radians):
    """Convert radians to RPM."""
    return radians * 9.549295;

def dist(a=(0,0,0), b=(0,0,0)):
    """Determine the distance between two points."""
    return math.sqrt((b[0] - a[0]) * (b[0] - a[0]) + (b[1] - a[1]) * (b[1] - a[1]) + (b[2] - a[2]) * (b[2] - a[2]))

def intersects(a=(0, 0, 0, 0), b=(0, 0, 0, 0)):
    """Determine if two rectangles are intersecting."""
    x1 = a[0] + a[2]
    y1 = a[1] + a[3]
    x3 = b[0] + b[2]
    y3 = b[1] + b[3]
    return not (x1 < b[0] or x3 < a[0] or y1 < b[1] or y3 < a[1])

    