#!/usr/bin/env python

import socket
import array
import asyncore
import insim


_BUFFER_SIZE = 2048
_RESERVED_REQI = 255
_PACKET_MAP = {insim.ISP_ISI: insim.IS_ISI, insim.ISP_VER: insim.IS_VER, insim.ISP_TINY: insim.IS_TINY, 
              insim.ISP_SMALL: insim.IS_SMALL, insim.ISP_STA: insim.IS_STA, insim.ISP_SCH: insim.IS_SCH, 
              insim.ISP_SFP: insim.IS_SFP, insim.ISP_SCC: insim.IS_SCC, insim.ISP_CPP: insim.IS_CPP, 
              insim.ISP_ISM: insim.IS_ISM, insim.ISP_MSO: insim.IS_MSO, insim.ISP_III: insim.IS_III, 
              insim.ISP_MST: insim.IS_MST, insim.ISP_MTC: insim.IS_MTC, insim.ISP_MOD: insim.IS_MOD,
              insim.ISP_VTN: insim.IS_VTN, insim.ISP_RST: insim.IS_RST, insim.ISP_NCN: insim.IS_NCN, 
              insim.ISP_CNL: insim.IS_CNL, insim.ISP_CPR: insim.IS_CPR, insim.ISP_NPL: insim.IS_NPL, 
              insim.ISP_PLP: insim.IS_PLP, insim.ISP_PLL: insim.IS_PLL, insim.ISP_LAP: insim.IS_LAP, 
              insim.ISP_SPX: insim.IS_SPX, insim.ISP_PIT: insim.IS_PIT, insim.ISP_PSF: insim.IS_PSF, 
              insim.ISP_PLA: insim.IS_PLA, insim.ISP_CCH: insim.IS_CCH, insim.ISP_PEN: insim.IS_PEN,
              insim.ISP_TOC: insim.IS_TOC, insim.ISP_FLG: insim.IS_FLG, insim.ISP_PFL: insim.IS_PFL, 
              insim.ISP_FIN: insim.IS_FIN, insim.ISP_RES: insim.IS_RES, insim.ISP_REO: insim.IS_REO, 
              insim.ISP_NLP: insim.IS_NLP, insim.ISP_MCI: insim.IS_MCI, insim.ISP_MSX: insim.IS_MSX, 
              insim.ISP_MSL: insim.IS_MSL, insim.ISP_CRS: insim.IS_CRS, insim.ISP_BFN: insim.IS_BFN, 
              insim.ISP_AXI: insim.IS_AXI, insim.ISP_AXO: insim.IS_AXO, insim.ISP_BTN: insim.IS_BTN,
              insim.ISP_BTC: insim.IS_BTC, insim.ISP_BTT: insim.IS_BTT, insim.ISP_RIP: insim.IS_RIP, 
              insim.ISP_SSH: insim.IS_SSH,}


def run(**kwargs):
    asyncore.loop(**kwargs)


class _TcpClient(asyncore.dispatcher):
    def __init__(self, dispatch_to):
        asyncore.dispatcher.__init__(self)
        self.create_socket(socket.AF_INET, socket.SOCK_STREAM)
        self.dispatch_to = dispatch_to
        self.send_buff = ''
        self.recv_buff = array.array('B')
        self.temp_buff = array.array('B', [0]*_BUFFER_SIZE)
        
    def __len__(self):
        return self.recv_buff.buffer_info()[1]
        
    def handle_connect(self):
        self.dispatch_to.handle_connect()
        
    def handle_close(self):
        self.dispatch_to.handle_failed()
        self.close()
        
    def handle_error(self):
        self.dispatch_to.handle_error()
        self.close()
        
    def send(self, data):
        self.send_buff += data
        
    def writeable(self):
        return len(self.send_buff)
    
    def handle_write(self):
        sent = self.socket.send(self.send_buff)
        self.send_buff = self.send_buff[sent:]
        
    def handle_read(self):
        # TODO: Optimise TCP code, too much buffer copying.
        recv = self.socket.recv_into(self.temp_buff)
        if recv:
            self.recv_buff.extend(self.temp_buff[:recv])
            while len(self) and len(self) >= self.recv_buff[0]:
                size = self.recv_buff[0]
                self.dispatch_to.handle_tcp(self.recv_buff[:size])
                self.recv_buff = self.recv_buff[size:]
        else:
            self.close()
            self.dispatch_to.handle_lost()
        
        
class _UdpClient(asyncore.dispatcher):
    def __init__(self, dispatch_to):
        asyncore.dispatcher.__init__(self)
        self.create_socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.dispatch_to = dispatch_to
        self.temp_buff = array.array('B', [0]*_BUFFER_SIZE)
        
    def handle_read(self):
        recv = self.socket.recv_into(self.temp_buff)
        if recv:
            self.dispatch_to.handle_udp(self.temp_buff[:recv])
        else:
            self.close()
            
    def handle_error(self):
        self.dispatch_to.handle_error()
        self.close()            
            
            
class Host(object):
    """Class to represent a LFS host."""
    def __init__(self, name, dispatch_to):
        """Create a new host object."""
        self.name = name
        self.dispatch_to = dispatch_to
        self.tcp = _TcpClient(dispatch_to=self)     
        self.udp = None
        self.hostaddr = ()
        self.state = None
        self.conns = {}
        self.players = {}
        
    def init(self, host, port, is_relay=False, **kwargs):
        self.hostaddr = (host, port)        
        self.tcp.connect((host, port))
        self.send(insim.ISP_ISI, **kwargs)
        if 'UDPPort' in kwargs and kwargs['UDPPort']:
            self.udp = _UdpClient(dispatch_to=self)               
            self.udp.bind((host, kwargs['UDPPort']))
        self.send(insim.ISP_TINY, ReqI=_RESERVED_REQI, SubT=insim.TINY_SST)
        self.send(insim.ISP_TINY, ReqI=_RESERVED_REQI, SubT=insim.TINY_ISM)            
    
    def send(self, type, **kwargs):
        """Send a packet to the host."""
        self.tcp.send(_PACKET_MAP[type](**kwargs).pack())
        
    def sendm(self, msg, ucid=0, plid=0):
        """Send a message to the host, or to a specific connection or player."""
        if ucid or plid:
            self.send(insim.ISP_MTC, Msg=msg, UCID=ucid, PLID=plid)
        elif len(msg) < 64:
            self.send(insim.ISP_MST, Msg=msg)
        else:
            self.send(insim.ISP_MSX, Msg=msg)
        
    def handle_connect(self):
        self.dispatch_to.handle_connect(self)
    
    def handle_failed(self):
        # Close UDP if TCP failed.
        # TODO: Move UDP init into handle_connect?
        if self.udp:
            self.udp.close()     
        self.dispatch_to.handle_failed(self)
    
    def handle_error(self):
        import sys
        self.dispatch_to.handle_error(self, sys.exc_info())
        
    def handle_lost(self):
        # Close UDP when TCP lost.
        if self.udp:
            self.udp.close()          
        self.dispatch_to.handle_lost(self)      
        
    def handle_tcp(self, data):
        type = data[1]
        p = _PACKET_MAP[type]().unpack(data)
        
        if type == insim.ISP_TINY and p.SubT == insim.TINY_NONE:
            self.send(insim.ISP_TINY, SubT=insim.TINY_NONE) # Keep alive.
        elif type == insim.ISP_STA:
            self.state = p
        elif type == insim.ISP_ISM:
            self.send(insim.ISP_TINY, ReqI=_RESERVED_REQI, SubT=insim.TINY_NCN)
            self.send(insim.ISP_TINY, ReqI=_RESERVED_REQI, SubT=insim.TINY_NPL)
        elif type == insim.ISP_NCN:
            self.conns[p.UCID] = p
        elif type == insim.ISP_NPL:
            self.players[p.PLID] = p
        
        # Don't dispatch for packets we requested.
        if p.ReqI != _RESERVED_REQI:
            self.dispatch_to.handle_packet(self, p)
        
        # Do these after dispatch so conns and players can still be accessed
        # in plugins that handle those packets.
        if type == insim.ISP_CNL:
            del self.conns[p.UCID]
        elif type == insim.ISP_PLL:
            del self.players[p.PLID]
            
    def handle_udp(self, data):
        # TODO: Add OutSim and OutGauge.
        type = data[1]
        if type == insim.ISP_MCI:
            mci = insim.IS_MCI().unpack(data)
            self.dispatch_to.handle_packet(self, mci)
        elif type == insim.ISP_NLP:
            nlp = insim.IS_NLP().unpack(data)
            self.dispatch_to.handle_packet(self, nlp)
   
