<?php
/**
 * This class will handle anything to do with Live For Speed data types or
 * parsing lfs data from LFS to your screen be it via HTML or the console.
 *
 * @author Dygear (Mark Tomlin) <Dygear@gmail.com>
 * @version 0.4.0
 * @license http://opensource.org/licenses/mit-license.php The MIT License
 * @copyright Copyright (c) 2010, PHPInSim Mod Development Team
 * @package PRISM
 * @package LFSWorldSDK
 * @subpackage LFS
*/

class LFS
{

	/**
	*	Strings
	*/

	/**
	 * @const number	Makes no change to the string.
	 */
	const STRING_DEFAULT			= 0;

	/**
	 * @const number	Removes color codes from lfs string.
	 */
	const STRING_STRIP_COLOR		= 1;

	/**
	 * @const number	Removes language codes from the lfs string.
	 */
	const STRING_STRIP_LANG			= 2;

	/**
	 * @const number	Removes all formatting from the lfs string.
	 *					SELF::STRING_STRIP_COLOR & SELF::STRING_STRIP_LANG
	 */
	const STRING_STRIP				= 3;

	/**
	 * @const number	Converts color codes to HTML & UTF-8.
	 */
	const STRING_HTML    			= 4;

	/**
	 * @const number	Converts color codes to ECMA-48 escape color codes & UTF-8.
	 */
	const STRING_CONSOLE			= 8;

	/**
	*	Tracks
	*/

	/**
	 * @const number	Makes no change to format that the function gets.
	 */
	const TRACK_DEFAULT				= 0;

	/**
	 * @const number	Displays Short Code (BL1R, AS7)
	 */
	const TRACK_TINY				= 1;

	/**
	 * @const number	Displays the 'CHMOD' version of the track name. (001, 460)
	 */
	const TRACK_SHORT				= 2;

	/**
	 * @const number	Displays Long Name (Blackwood Grand Prix Reversed, Aston National)
	 */
	const TRACK_LONG				= 4;

	/**
	*	Time
	*	/!\ To Do /!\
	*	TIME_*
	*	TIME_*_0LEADING
	*	TIME_*_0IFNEXTSET
	*	TIME_*_ALL
	*	TIME_ALL
	*	TIME_LFS
	*/

	/**
	*	Car
	*/

	/**
	 * @const number	Makes no change to the LFS format for car name.
	 */
	const CAR_DEFAULT				= 0;

	/**
	 * @const number	Displays a Bitwise string of cars that are enabled.
	 */
	const CAR_BITWISE				= 1;

	/**
	 * @const number	Displays short code (FOX, BF1)
	 */
	const CAR_TINY					= 2;

	/**
	 * @const number	Displays long name (Formula V8)
	 */
	const CAR_LONG					= 4;

	/**
	 * @const number	Places output into an array.
	 */
	const CAR_ARRAY					= 8;

	/**
	 * @const number	The key will be the int that is used in place of the car name.
	 */
	const CAR_ARRAY_KEY_INT			= 16;

	/**
	 * @const number	Key (FOX or F08, ect).
	 */
	const CAR_ARRAY_KEY_TINY		= 32;

	/**
	 * @const number	Key Long (Formula XR or Formula V8, ect)
	 */
	const CAR_ARRAY_KEY_LONG		= 64;

	/**
	 * @static array	Code Page from LFS to Universal Standard.
	 */
	static $CodePages = array('^L'=>'CP1252','^G'=>'ISO-8859-7','^C'=>'CP1251','^E'=>'ISO-8859-2','^T'=>'ISO-8859-9','^B'=>'ISO-8859-13','^J'=>'SJIS-win','^S'=>'CP936','^K'=>'CP949','^H'=>'CP950');

	/**
	 * @static array	LFS Charater Escape Codes.
	 */
	static $CharMap = array('^a'=>'*','^c'=>':','^d'=>'\\','^h'=>'#','^l'=>'<','^q'=>'?','^r'=>'>','^s'=>'/','^t'=>'"','^v'=>'|');

	/**
	 * @static array	LFS Color Codes to ECMA-48 color standard.
	 */
	static $ECMA48 = array('^0'=>"\033[30m",'^1'=>"\033[31m",'^2'=>"\033[32m",'^3'=>"\033[33m",'^4'=>"\033[34m",'^5'=>"\033[35m",'^6'=>"\033[36m",'^7'=>"\033[37m",'^8'=>"\033[39m",'^9'=>"\033[0m");

	/**
	 * @static array	LFS Color Codes to HTML color standard.
	 */
	static $HTML = array('^0'=>'000','^1'=>'F00','^2'=>'0F0','^3'=>'FF0','^4'=>'00F','^5'=>'F0F','^6'=>'0FF','^7'=>'FFF');

	/**
	 * Converts LFS markup into HTML (via spans) and UTF-8.
	 *
	 * @uses	strlen()
	 * @uses	is_numeric()
	 * @uses	substr()
	 * @uses	self::$HTML
	 * @param	string	String to convert
	 * @param	number	Bitflag from self::STRING_* consts.
	 * @return	string	String with Markup.
	*/
	final static function HTML($str, $bit)
	{
		$isTagOpen = FALSE;

		for ($i = 0, $j = 1, $l = strlen($str); $i < $l; ++$i, ++$j)
		{
			if ($str{$i} == '^' && is_numeric($str{$j}))
			{
				if ($isTagOpen == TRUE)
				{
					$isTagOpen = FALSE;
					$str = substr($str, 0, $i) . '</span>' . substr($str, $i);
					$i += 7; $j += 7; $l = strlen($str);
				}
				if ($str{$j} < 8)
				{
					$isTagOpen = TRUE;
					$str = substr($str, 0, $i) . '<span style="color: #' . self::$HTML[$str[$j]] . ';">' . substr($str, $i);
					$i += 27; $j += 27;
					$l = strlen($str);
				}
				$str[$i] = NULL; # Remove ^.
				$str[$j] = NULL; # Remove Int.
			}
		}

		// Close any tag left open.
		if ($isTagOpen)
			$str .= '</span>';

		return $str;
	}

	/**
	 * Converts LFS markup into a ECMA-48 compliant output.
	 *
	 * @uses	strlen()
	 * @uses	is_numeric()
	 * @uses	substr()
	 * @uses	self::$ECMA48
	 * @param	string	String to convert
	 * @param	number	Bitflag from self::STRING_* consts.
	 * @return	string	String with Markup.
	*/
	final static function Console($str)
	{
		$isTagOpen = FALSE;
		for ($i = 0, $j = 1, $l = strlen($str); $i < $l; ++$i, ++$j)
		{
			if ($str{$i} == '^' && is_numeric($str{$j}))
			{
				$deltaLen = strlen(self::$ECMA48[$str[$j]]);
				// Set tag status
				if ($str{$j} < 8)
					$isTagOpen = TRUE;
				else
					$isTagOpen = FALSE;
				// Place ECMA-48 Charaters into String.
				$str = substr($str, 0, $i) . self::$ECMA48[$str[$j]] . substr($str, $i);
				// Move Pointer Past The Change.
				$i += $deltaLen; $j += $deltaLen;
				// Get New String Length.
				$l = strlen($str);
				// Remove The Formatting.
				$str[$i] = NULL; # Remove ^
				$str[$j] = NULL; # Remove Int.
			}
		}
		// Close any tag left open.
		if ($isTagOpen)
			$str .= "\033[0m";

		return $str;
	}
}

?>