﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Timers;
using Spark;
using Spark.Helpers;
using Spark.Packets;

namespace SparkCruise
{
    public class CruiseApp
    {
        private Dictionary<byte, User> _users = new Dictionary<byte, User>();
        private Dictionary<byte, IS_NPL> _players = new Dictionary<byte, IS_NPL>();
        private UserRepository _repository = new UserRepository();
        private Dictionary<string, Action<User, string[]>> _commands;
        private InSim _insim;
        private Timer _heartbeat;

        public CruiseApp()
        {
            Console.WriteLine("SparkCruise");
            Console.WriteLine("-----------");
            Console.WriteLine();

            try
            {
                InitializeCommands();
                InitializeInSim();
                InitializeHeartbeat();

                // We're done!
                Message("Welcome to ^3SparkCruise^7!");
                Console.WriteLine("The cruise server is running!");
                Console.WriteLine();

                // Stop program from exiting while InSim is connected.
                _insim.Run();
            }
            catch (InSimException ex)
            {
                Console.WriteLine("Error: {0}", ex.Message);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Error: {0}", ex);
            }
        }

        private void InitializeInSim()
        {
            _insim = new InSim();

            // Bind events.
            _insim.Bind<IS_ISM>(JoinedMultiplayer);
            _insim.Bind<IS_RST>(RaceStarted);
            _insim.Bind<IS_NCN>(ConnectionJoined);
            _insim.Bind<IS_CNL>(ConnectionLeft);
            _insim.Bind<IS_NPL>(PlayerJoined);
            _insim.Bind<IS_PLL>(PlayerLeft);
            _insim.Bind<IS_MCI>(CarUpdate);
            _insim.Bind<IS_MSO>(MessageReceived);
            _insim.InSimError += new EventHandler<InSimErrorEventArgs>(_insim_InSimError);

            // Connect to LFS and initialize InSim.
            _insim.Connect("127.0.0.1", 29999);
            _insim.Send(new IS_ISI
            {
                IName = "^3SparkCruise",
                Admin = string.Empty, // Admin password here!
                Flags = InSimFlags.ISF_MCI,
                Interval = 1000, // Milliseconds.
                Prefix = '!'
            });

            // Request host info.
            _insim.Send(new IS_TINY { ReqI = 255, SubT = TinyType.TINY_ISM });
        }

        private void InitializeHeartbeat()
        {
            _heartbeat = new Timer();
            _heartbeat.Interval = 1000; // Milliseconds.
            _heartbeat.AutoReset = true;
            _heartbeat.Elapsed += new ElapsedEventHandler(_heartbeat_Elapsed);
            _heartbeat.Start();
        }

        private void _insim_InSimError(object sender, InSimErrorEventArgs e)
        {
            // Any errors which occur in the event handlers will end up here.
            Console.WriteLine("InSim Error: {0}", e.Exception);
            Process.GetCurrentProcess().Kill(); // Bye bye!
        }

        private void RaceStarted(IS_RST rst)
        {
            if (!rst.Flags.HasFlag(RaceFlags.HOSTF_CRUISE))
            {
                Message("^1NOT IN CRUISE MODE!");
            }
        }

        private void JoinedMultiplayer(IS_ISM ism)
        {
            // On joining host, request all connections and players.
            _insim.Send(new IS_TINY { ReqI = 255, SubT = TinyType.TINY_NCN });
            _insim.Send(new IS_TINY { ReqI = 255, SubT = TinyType.TINY_NPL });
            // Request race start packet.
            _insim.Send(new IS_TINY { ReqI = 255, SubT = TinyType.TINY_RST });
        }

        private void ConnectionJoined(IS_NCN ncn)
        {
            if (ncn.UCID == 0) return; // Ignore host.

            // Load user.
            User user = LoadUser(ncn);
            UpdateUserSession(ncn, user);

            // Add to users map.
            _users[user.ConnectionId] = user;

            // Welcome message.
            Message(user, "Welcome to ^3SparkCruise^7!");
        }

        private User LoadUser(IS_NCN ncn)
        {
            User user;
            if (!_repository.TryGetUser(ncn.UName, out user))
            {
                // Create new default user.
                user = User.CreateUser(
                    Guid.NewGuid(),
                    ncn.UName,
                    1000, // Starting cash.
                    0.0,
                    DateTime.Now,
                    100,
                    0,
                    ncn.PName);
                user.Cars.Add(Car.CreateCar(
                    Guid.NewGuid(),
                    user.UserId,
                    "UF1")); // Start cars.

                // Add to repository and save.
                _repository.Add(user);
                _repository.Save();
            }
            return user;
        }

        private static void UpdateUserSession(IS_NCN ncn, User user)
        {
            user.ConnectionId = ncn.UCID;
            user.Playername = ncn.PName;
            user.IsAdmin = (ncn.Admin == 1);
            user.CurrentCar = null;
        }

        private void ConnectionLeft(IS_CNL cnl)
        {
            if (cnl.UCID == 0) return; // Ignore host.

            SaveUser(cnl.UCID);

            // Remove user from map.
            _users.Remove(cnl.UCID);
        }

        private void SaveUser(byte ucid)
        {
            _repository.Save();

            // Remove user from object context.
            var user = _users[ucid];
            _repository.Detach(user);
        }

        private void PlayerJoined(IS_NPL npl)
        {
            _players[npl.PLID] = npl;

            CheckUserOwnsCar(npl);
        }

        private void CheckUserOwnsCar(IS_NPL npl)
        {
            var user = _users[npl.UCID];
            if (user.HasCar(npl.CName))
            {
                user.CurrentCar = npl.CName;
            }
            else
            {
                Message("/spec {0}", user.Username); // Send spectate command.
                Message(user, "You do not own the {0}", npl.CName);
            }
        }

        private void PlayerLeft(IS_PLL pll)
        {
            GetUser(pll.PLID).CurrentCar = null;
            _players.Remove(pll.PLID);
        }

        private void CarUpdate(IS_MCI mci)
        {
            foreach (var car in mci.CompCars)
            {
                User user;
                if (TryGetUser(car.PLID, out user))
                {
                    UpdateDistance(car, user);

                    // Save speed for use in heartbeat.
                    user.Speed = car.Speed;
                }
            }
        }

        private void UpdateDistance(CompCar car, User user)
        {
            if (user.LastX != 0 && user.LastY != 0 && user.LastZ != 0)
            {
                var distance = MathHelper.Distance(
                    user.LastX,
                    user.LastY,
                    user.LastZ,
                    car.X,
                    car.Y,
                    car.Z);
                user.Distance += MathHelper.LengthToMetres(distance);
            }
            user.LastX = car.X;
            user.LastY = car.Y;
            user.LastZ = car.Z;
        }

        private void _heartbeat_Elapsed(object sender, ElapsedEventArgs e)
        {
            if (!_insim.IsConnected) return;

            foreach (var user in _users.Values)
            {
                if (!user.IsHost)
                {
                    UpdateUser(user);
                    UpdateOnScreenDisplay(user);
                }
            }
        }

        private int GetRandom(int min, int max)
        {
            return new Random().Next(min, max);
        }

        private void UpdateUser(User user)
        {
            // Update bonus.
            if (user.Bonus >= 100)
            {
                int bonus = GetRandom(400, 500);
                user.Cash += bonus;
                user.Bonus = 0;
                Message(user, "You have received a ${0} bonus!", bonus);
            }

            // Update health.
            if (user.Health <= 0)
            {
                int fine = GetRandom(400, 500);
                user.Cash -= fine;
                user.Health = 100;
                Message(user, "You have received a ${0} doctors bill!", fine);
            }

            // Update cash.
            if (MathHelper.SpeedToKph(user.Speed) > 5)
            {
                user.Cash++;

                user.Bonus += 1;
                user.Health -= 0.6; // Made this slightly odd so it wouldn't be so predicatble.
            }
        }

        private void UpdateOnScreenDisplay(User user)
        {
            string[] buttons =
            {
                "^3SparkCruise",
                string.Format("^7Cash: ^3${0}", user.Cash),
                string.Format("^7Bonus: ^3{0:f1}%", user.Bonus),
                string.Format("^7Health: ^3{0:f1}", user.Health),
                string.Format("^7Distance: ^3{0:f2} Km", (user.Distance / 1000)),
                string.Format("^7Car: ^3{0}", (user.CurrentCar == null) ? "None" : user.CurrentCar),
            };

            for (byte i = 0, id = 1, top = 140; i < buttons.Length; i++, id++, top += 4)
            {
                _insim.Send(new IS_BTN
                {
                    ReqI = 255,
                    UCID = user.ConnectionId,
                    ClickID = id,
                    BStyle = ButtonStyles.ISB_DARK | ButtonStyles.ISB_LEFT,
                    T = top,
                    L = 1,
                    W = 20,
                    H = 4,
                    Text = buttons[i],
                });
            }
        }

        #region Helpers
        private void Message(string message, params object[] args)
        {
            _insim.Send(message, args);
        }

        private void Message(User user, string message, params object[] args)
        {
            _insim.Send(user.ConnectionId, 0, "^3|^7 " + message, args);
        }

        private User GetUser(byte plid)
        {
            IS_NPL npl;
            if (_players.TryGetValue(plid, out npl))
            {
                return _users[npl.UCID];
            }
            return null;
        }

        private bool TryGetUser(byte plid, out User user)
        {
            user = GetUser(plid);
            return user != null;
        }

        private bool TryGetUser(string username, out User user)
        {
            user = (from u in _users.Values
                    where u.Username.Equals(username, StringComparison.InvariantCultureIgnoreCase)
                    select u).SingleOrDefault();
            return user != null;
        }

        private bool TryParseCommand(IS_MSO mso, out string[] args)
        {
            if (mso.UserType == UserType.MSO_PREFIX)
            {
                var message = mso.Msg.Substring(mso.TextStart);
                args = message.Split();
                return args.Length > 0;
            }
            args = null;
            return false;
        }
        #endregion

        #region Commands
        private void InitializeCommands()
        {
            _commands = new Dictionary<string, Action<User, string[]>>
            {
                { "!help", CommandHelp },
                { "!admins", CommandAdmins },
                { "!prices", CommandPrices },
                { "!garage", CommandGarage },
                { "!buy", CommandBuy },
                { "!sell", CommandSell },
                { "!send", CommandSend },
                { "!give", CommandGive },
                { "!pitlane", CommandPitlane },
                { "!showoff", CommandShowoff },
                { "!top", CommandTop },
            };
        }

        private void MessageReceived(IS_MSO mso)
        {
            string[] args;
            if (TryParseCommand(mso, out args))
            {
                var command = args[0].ToLower();

                Action<User, string[]> action;
                if (_commands.TryGetValue(command, out action))
                {
                    var user = _users[mso.UCID];
                    action(user, args);
                }
            }
        }

        private bool ValidateArgs(User user, string[] args, string message, int length)
        {
            if (args.Length == length)
            {
                return true;
            }
            Message(user, message);
            return false;
        }

        private void CommandHelp(User user, string[] args)
        {
            Message(user, "Help:");
            Message(user, "!help - Show this message");
            Message(user, "!admins - See which admins are currently online");
            Message(user, "!prices - See a list of car prices");
            Message(user, "!garage - See what cars you currently own");
            Message(user, "!buy <car> - Buy the specified car");
            Message(user, "!sell <car> - Sell the specified car");
            Message(user, "!send <username> <cash> - Send cash to a user");
            Message(user, "!give <username> <car> - Give a car to a user");
            Message(user, "!pitlane - Return to the pitlane ($500 fine)");
            Message(user, "!showoff - Show off your stats to your friends!");
            Message(user, "!top - Show the top users on ^3SparkCruise");
        }

        private void CommandAdmins(User user, string[] args)
        {
            var admins = from u in _users.Values
                         where u.IsAdmin
                         orderby u.JoinDate
                         select u;

            if (admins.Any())
            {
                Message(user, "Admins:");
                foreach (var admin in admins)
                {
                    Message(user, "{0} ^7({1})", admin.Playername, admin.Username);
                }
            }
            else
            {
                Message(user, "There are no admins online");
            }
        }

        private void CommandPrices(User user, string[] args)
        {
            Message(user, "Prices:");
            foreach (var price in Dealer.GetAllPrices())
            {
                Message(user, "{0}: ${1}", price.Car, price.Price);
            }
        }

        private void CommandGarage(User user, string[] args)
        {
            if (user.Cars.Any())
            {
                Message(user, "Garage:");
                foreach (var car in user.Cars)
                {
                    int price = Dealer.GetCarPrice(car.Name);
                    Message(user, "{0}: ${1}", car.Name, price);
                }
            }
            else
            {
                Message(user, "You do not own any cars");
            }
        }

        private void CommandBuy(User user, string[] args)
        {
            if (!ValidateArgs(user, args, "Usage: !buy <car>", 2)) return;

            var carname = args[1].ToUpper();

            int price;
            if (!Dealer.TryGetPrice(carname, out price))
            {
                Message(user, "The car {0} does not exist", carname);
            }
            else if (user.HasCar(carname))
            {
                Message(user, "You already own the {0}", carname);
            }
            else if (user.Cash < price)
            {
                Message(user, "You need ${0} more to afford the {1} (${2})", price - user.Cash, carname, price);
            }
            else
            {
                user.Cash -= price;
                user.Cars.Add(Car.CreateCar(Guid.NewGuid(), user.UserId, carname));

                Message(user, "You have bought the {0} for ${1}", carname, price);
                Message("{0}^7 has bought the {1}", user.Playername, carname);
            }
        }

        private void CommandSell(User user, string[] args)
        {
            if (!ValidateArgs(user, args, "Usage: !sell <car>", 2)) return;

            var carname = args[1].ToUpper();

            int price;
            Car car;
            if (!Dealer.TryGetPrice(carname, out price))
            {
                Message(user, "The {0} does not exist", carname);
            }
            else if (!user.TryGetCar(carname, out car))
            {
                Message(user, "You do not own the {0}", carname);
            }
            else
            {
                user.Cash += price;
                _repository.Remove(car);
                Message(user, "You have sold the {0} for ${1}", carname, price);

                // Spectate driver if they've just sold the car they're driving.
                if (user.CurrentCar == carname)
                {
                    Message("/spec {0}", user.Username);
                    Message(user, "You do not own the {0} anymore", carname);
                }
            }
        }

        private void CommandSend(User user, string[] args)
        {
            if (!ValidateArgs(user, args, "Usage: !send <username> <cash>", 3)) return;

            User other;
            int amount;
            if (!TryGetUser(args[1], out other))
            {
                Message(user, "The user {0} does not exist", args[1]);
            }
            else if (!int.TryParse(args[2], out amount))
            {
                Message(user, "{0} is not a valid amount of cash", args[2]);
            }
            else if (user.Cash < amount)
            {
                Message(user, "You don't have ${0} to give away", amount);
            }
            else
            {
                user.Cash -= amount;
                Message(user, "You have given {0} ^7the amount of ${1}", other.Playername, amount);

                other.Cash += amount;
                Message(other, "You have been given ${0} by {1}", amount, user.Playername);
            }
        }

        private void CommandGive(User user, string[] args)
        {
            if (!ValidateArgs(user, args, "Usage: !give <username> <car>", 3)) return;

            var carname = args[2].ToUpper();

            Car car;
            User other;
            if (!TryGetUser(args[1], out other))
            {
                Message(user, "The user {0} does not exist", args[1]);
            }
            else if (!Dealer.CarExists(carname))
            {
                Message(user, "The car {0} does not exist", carname);
            }
            else if (!user.TryGetCar(carname, out car))
            {
                Message(user, "You do not own the {0}", carname);
            }
            else if (other.HasCar(carname))
            {
                Message(user, "{0} ^7already owns the {1}", other.Playername, carname);
            }
            else
            {
                other.Cars.Add(Car.CreateCar(Guid.NewGuid(), other.UserId, carname));
                Message(other, "{0} ^7has given you the {1}", user.Playername, carname);

                _repository.Remove(car);
                Message(user, "You have given {0} ^7the {1}", other.Playername, carname);

                if (user.CurrentCar == carname)
                {
                    Message("/spec {0}", user.Username);
                    Message(user, "You don't own the {0} anymore", carname);
                }
            }
        }

        private void CommandPitlane(User user, string[] args)
        {
            user.Cash -= 500;

            Message("/pitlane {0}", user.Username);
            Message(user, "You have been fined ${0} for returning to the pitlane", 500);
        }

        private void CommandShowoff(User user, string[] args)
        {
            Message("Stats for {0}^7:", user.Playername);
            Message("Cash: ^3${0}", user.Cash);
            Message("Cars: ^3{0}", user.Cars.Count);
            Message("Distance: ^3{0:f2} Km", user.Distance / 1000);

            var joined = ((DateTime.Now - user.JoinDate).TotalHours / 24) + 1;
            Message("Joined: ^3{0:f0} days ago", joined);
        }

        private void CommandTop(User user, string[] args)
        {
            var topUsers = (from u in _repository.GetUsers()
                            orderby u.Cash descending
                            select u).Take(10);

            if (topUsers.Any())
            {
                Message(user, "Top Users:");

                int position = 0;
                foreach (var topUser in topUsers)
                {
                    Message(
                        user,
                        "{0}. {1} ^7- ${2}",
                        ++position,
                        topUser.Playername,
                        topUser.Cash);
                }
            }
            else
            {
                Message(user, "There are no top users :(");
            }
        }
        #endregion
    }
}
