using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using InSim;
using InSim.Enums;
using InSim.Structs;
using Tools;
using d3d8_proxy;
using System.Threading;
using System.Reflection;
using LFS;
using System.IO;
using OutGauge;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace DigitalGauges
{
    public partial class MainForm : Form
    {
        #region variables & constructor
        
        private const string appName            = "Digital Speedo";

        private InSimWriter isWriter            = null;
        private InSimReader isReader            = null;
        private GfxOutput gfxOutput             = null;
        private Pitlane pitlane                 = null;
        private OutGaugeReader ogReader         = null;

        private string lfsPath                  = string.Empty;
        private string adminPass                = string.Empty;
        private int isSendPort                  = 29999;
        private int isRecvPort                  = 30011;
        private int ogRecvPort                  = 30003;
        private bool lfsConnected               = false;
        private bool ogNeedRequest              = true;        
        public bool specHide                    = false;
        private string currCar                  = string.Empty;
        private bool startMinimized             = false;
        private bool visibleState               = false;
        private byte playerPLID                 = 0;
        private byte viewPLID                   = 0;
        private int mciIndex                    = -1;
        private string colorLfs                 = "ffffff";
        private string colorCustom              = "ffffff";
        private bool customColor                = false;
        private bool ogDirect                   = true;
        private bool ogAutoCfg                  = true;
        private bool autoClose                  = false;
        private bool closePrograms              = false;
        private const int WM_SYSCOMMAND         = 0x0112;
        private const int SC_MINIMIZE           = 0xF020;     
        private const int SC_MAXIMIZE           = 0xF030;

        private struct WndState
        {
            public const string Default    = "Default";  
            public const string Minimized  = "Minimized";
            public const string Normal     = "Normal";
            public const string Maximized  = "Maximized";
            public const string Fullscreen = "Fullscreen";
        }

        private SortedList customSettings       = new SortedList();
        private GaugeSettings currCarSettings   = new GaugeSettings();
        private GaugeSettings currWndSettings   = new GaugeSettings();
        private GaugeState    gaugeState        = new GaugeState();
        private SortedList ogClients            = new SortedList();

        private delegate void ConnectedEvent(bool connected);
                
        [DllImport("user32.dll")]
        public static extern int PostMessage(
                  IntPtr hWnd,      // handle to destination window
                  uint Msg,       // message
                  long wParam,  // first message parameter
                  long lParam   // second message parameter
                  );

        public MainForm()
        {
            InitializeComponent();
        }

        #endregion

        #region insim events

        private void isReader_KeepAliveEvent(IS_TINY packet)
        {
            isWriter.Send(packet);
        }

        private void isReader_InitialRequestEvent()
        {
            this.Invoke(new ConnectedEvent(this.SetConnected), true);
            this.Invoke(new MethodInvoker(delegate()
                {
                    timerConnect.Enabled = false;
                }));

            RequestPlayers();  //in case we connected during race (we need playerID)            
            RequestState();
        }

        private void isReader_OutGaugeEvent(InSim.Structs.OutGaugePack packet)
        {
            OutGaugeEvent(packet);            
        }

        private void ogReader_OutGaugeEvent(OutGaugePack packet)
        {
            OutGaugeEvent(packet);
        }

        private void isReader_MultiCarInfoEvent(IS_MCI packet)
        {
            if (visibleState == false)
                return;

            if (currCarSettings.speedVisibleType != (int)VisibleTypeSpeed.PitLane)
                return;

            if (mciIndex == -1 || mciIndex >= packet.NumC || packet.Info[mciIndex].PLID != viewPLID)
            {
                for(int i=0; i<packet.NumC; i++)
                {
                    if (packet.Info[i].PLID == viewPLID)
                    {
                        mciIndex = i;
                        break;
                    }
                }
            }

            if (mciIndex == -1 || mciIndex >= packet.NumC)
                return;            

            //check if its not incorrect packet, same index but different player
            if (packet.Info[mciIndex].PLID != viewPLID)
                return; 

            CompCar cc  = packet.Info[mciIndex];            
            bool pits   = pitlane.Contains(cc.X, cc.Y);            

            if (pits == gaugeState.inPits)
                return;

            gaugeState.inPits = pits;            
            
            if (gfxOutput.LibraryLoaded)
            {
                if (pits != gaugeState.speedVisible)
                {
                    if (pits)
                        gfxOutput.SetSpeedPosSize(currCarSettings.speedPosSize.x, currCarSettings.speedPosSize.y, currCarSettings.speedPosSize.size);
                    else
                    {
                        gaugeState.speed = 0;
                        gfxOutput.SetSpeed(0);
                    }

                    gfxOutput.SetSpeedVisible(pits);                                        
                    gaugeState.speedVisible = pits;
                }   
            }            
        }

        private void isReader_StateEvent(InSim.Structs.IS_STA packet)
        {
            pitlane.Load(packet.Track);
            
            viewPLID = packet.ViewPLID;
            Log.Trace("plid {0}", viewPLID);

            visibleState = true;
            if ((packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_GAME) == 0 && (packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_REPLAY) == 0)
                visibleState = false;

            if ((packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_VISIBLE) == 0)
                visibleState = false;
                
            if ((packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_SHIFTU) != 0)
                visibleState = false;

            if ((packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_PAUSED) != 0)
                visibleState = false;

            if (packet.RaceInProg == 0)
                visibleState = false;

            if (currCarSettings.speedVisibleType == (int)VisibleTypeSpeed.No && currCarSettings.gearVisibleType == (int)VisibleType.No && currCarSettings.ledVisibleType == (int)VisibleType.No)
                visibleState = false;

            if (specHide && packet.ViewPLID != playerPLID)
                visibleState = false;

            //speed
            bool speed = false;
            if (visibleState && currCarSettings.speedVisibleType != (int)VisibleTypeSpeed.No)
            {
                speed = true;
                if (currCarSettings.speedVisibleType == (int)VisibleTypeSpeed.PitLane)
                {
                    if (gaugeState.inPits == false)
                        speed = false;
                }
            }

            //gear
            bool gear = false;
            if (visibleState && currCarSettings.gearVisibleType != (int)VisibleType.No)
                gear = true;

            //led
            bool led = false;
            if (visibleState && currCarSettings.ledVisibleType != (int)VisibleType.No)
                led = true;

            //rpm
            bool rpm = false;
            if (visibleState && currCarSettings.rpmVisibleType != (int)VisibleType.No)
                rpm = true;

            if (gfxOutput.LibraryLoaded)
            {
                if (speed != gaugeState.speedVisible)
                {
                    if (speed)
                        gfxOutput.SetSpeedPosSize(currCarSettings.speedPosSize.x, currCarSettings.speedPosSize.y, currCarSettings.speedPosSize.size);
                    else
                    {
                        gaugeState.speed = 0;
                        gfxOutput.SetSpeed(0);
                    }

                    gfxOutput.SetSpeedVisible(speed);                                        
                    gaugeState.speedVisible = speed;
                }   

                if (gear != gaugeState.gearVisible)
                {
                    if (gear)
                        gfxOutput.SetGearPosSize(currCarSettings.gearPosSize.x, currCarSettings.gearPosSize.y, currCarSettings.gearPosSize.size);                    
                    else
                    {
                        gaugeState.gear = 1; //neutral
                        gfxOutput.SetGear(1);
                    }

                    gfxOutput.SetGearVisible(gear);
                    gaugeState.gearVisible = gear;
                }   

                if (led != gaugeState.ledVisible)
                {
                    if (led)
                        gfxOutput.SetLedPosSize(currCarSettings.ledPosSize.x, currCarSettings.ledPosSize.y, currCarSettings.ledPosSize.size);                    
                    else
                    {
                        gaugeState.led = (int)GfxOutput.Led.Off;
                        gfxOutput.SetLed((int)GfxOutput.Led.Off);
                    }

                    gfxOutput.SetLedVisible(led);
                    gaugeState.ledVisible = led;
                }

                if (rpm != gaugeState.rpmVisible)
                {
                    if (rpm)
                        gfxOutput.SetRpmPosSize(currCarSettings.rpmPosSize.x, currCarSettings.rpmPosSize.y, currCarSettings.rpmPosSize.size);                    
                    else
                    {
                        gaugeState.rpm = 0;
                        gfxOutput.SetRpm(0);
                    }

                    gfxOutput.SetRpmVisible(rpm);
                    gaugeState.rpmVisible = rpm;
                }
            }
                        
            if ((packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_FRONT_END) != 0 || ((packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_GAME) == 0 && (packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_REPLAY) == 0))
                ogNeedRequest = true;
            
            if (ogDirect && ((packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_GAME) != 0 || (packet.Flags & (ushort)ISS_STATE_FLAGS.ISS_REPLAY) != 0) && ogNeedRequest)
            {                
                RequestOutGauge(1);                  
                
                ogNeedRequest = false;

                Log.Trace("request OutGauge");
            }
        }

        private void isReader_PlayerJoinEvent(IS_NPL packet)
        {
            if ((packet.PType & (byte)PTYPE.AI) == 0 && (packet.PType & (byte)PTYPE.REMOTE) == 0)
                playerPLID = packet.PLID;
        }

        #endregion

        #region window events

        #region main form

        private void MainForm_Load(object sender, EventArgs e)
        {            
            //first activation of tabPageStartup raises lvPrograms_ItemChecked, which enables save button            
            //to prevent that, we activate this tab before settings load
            tcPages.SelectedTab = tabPageStartup;
            tcPages.SelectedTab = tabPageConnection;
                                    
            cbWindowState.Items.Add(WndState.Minimized);
            cbWindowState.Items.Add(WndState.Normal);
            cbWindowState.Items.Add(WndState.Maximized);
            
            LoadDefaultSettings();
            LoadSettings();            

            if (startMinimized)
                this.WindowState = FormWindowState.Minimized;
        }

        protected override void OnLoad(EventArgs args)
        {
            base.OnLoad(args);

            Application.Idle += new EventHandler(MainForm_Loaded);
        }

        private void MainForm_Loaded(object sender, EventArgs args)
        {
            Application.Idle -= new EventHandler(MainForm_Loaded);                       

            try
            {
                ExtractResource("pitlane.cfg", false);           
                lbCars.SelectedIndex = 0;
            }
            catch(Exception ex)
            {
                Log.Error(ex.ToString());
            }

            StartPrograms();
            Start();    
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            Stop();
            StopPrograms();
        }

        private void timerConnectLfs_Tick(object sender, EventArgs e)
        {
            InitLfsConnection();
        }

        private void timerIsConnected_Tick(object sender, EventArgs e)
        {
            if (!lfsConnected)
                return;

            if (Math.Abs((DateTime.Now - isReader.LastPacket).TotalSeconds) > 45)
            {
                SetConnected(false);
                timerConnect.Enabled = true;
            }
        }

        private void btnSave_Click(object sender, EventArgs e)
        {
            bool restart = false;

            if (isSendPort != nbIsSendPort.Value)
            { 
                isSendPort    = (int)nbIsSendPort.Value;
                restart     = true;
            }

            if (isRecvPort != nbIsRecvPort.Value)
            { 
                isRecvPort    = (int)nbIsRecvPort.Value;
                restart     = true;
            }

            if (ogRecvPort != nbOgRecvPort.Value)
            { 
                ogRecvPort  = (int)nbOgRecvPort.Value;
                restart     = true;
            }

            if (adminPass.CompareTo(tbAdminPass.Text) != 0)
            { 
                adminPass   = tbAdminPass.Text;
                restart     = true;
            }
            
            if (lfsPath.CompareTo(tbLfsPath.Text) != 0)
            {
                lfsPath     = tbLfsPath.Text;
                restart     = true;
            }          
  
            if (ogDirect != rbOgDirect.Checked)
            {
                if (ogDirect)
                    RequestOutGauge(0);

                ogDirect    = rbOgDirect.Checked;
                restart     = true;
            }

            if (ogAutoCfg != cbOgAutoCfg.Checked)
            {
                ogAutoCfg   = cbOgAutoCfg.Checked;                
                restart     = true;
            }

            if (lbOgPorts.Items.Count != ogClients.Count)
                restart = true;
            else
            {
                foreach(int port in lbOgPorts.Items)
                {
                    if (ogClients.ContainsKey(port) == false)
                    {
                        restart = true;
                        break;
                    }
                }
            }

            if (restart)
            {
                Stop();
                Start();
            }
            
            SaveSettings();
            EnableBtnSave(false);
        }

        protected override void WndProc(ref Message m)
        {
            if (m.Msg == 0x5FA && autoClose == true)
            {   
                if (gfxOutput.LibraryLoaded)
                {
                    StringBuilder sb = new StringBuilder(512);
                    gfxOutput.GetFilePath(sb);

                    if (lfsPath.ToLower().CompareTo(sb.ToString().ToLower()) == 0)
                        this.Close();
                }
            }

            base.WndProc(ref m);
        }

        #endregion

        #region tab connection

        private void btnLfsPath_Click(object sender, EventArgs e)
        {
            ofDlg.InitialDirectory  = (lfsPath.Length > 0) ? Path.GetDirectoryName(lfsPath) : "c:\\";
            ofDlg.Filter            = "LFS.exe|LFS.exe|Executables|*.exe";
            if (ofDlg.ShowDialog(this) == DialogResult.OK)
            {
                tbLfsPath.Text = ofDlg.FileName;
                EnableBtnSave(true);
            }
        }

        private void btnOgPortRemove_Click(object sender, EventArgs e)
        {
            lbOgPorts.Items.RemoveAt(lbOgPorts.SelectedIndex);

            EnableBtnSave(true);
        }

        private void tbAdminPass_TextChanged(object sender, EventArgs e)
        {
            EnableBtnSave(true);
        }

        private void nbIsSendPort_ValueChanged(object sender, EventArgs e)
        {
            EnableBtnSave(true);
        }

        private void nbIsRecvPort_ValueChanged(object sender, EventArgs e)
        {
            EnableBtnSave(true);
        }

        private void nbOgRecvPort_ValueChanged(object sender, EventArgs e)
        {
            EnableBtnSave(true);
        }

        private void rbOgIndirect_CheckedChanged(object sender, EventArgs e)
        {
            nbOgRecvPort.Enabled    = !rbOgDirect.Checked;

            EnableBtnSave(true);
        }

        private void cbOgAutoCfg_CheckedChanged(object sender, EventArgs e)
        {
            gbOutGauge.Enabled = !cbOgAutoCfg.Checked;
            EnableBtnSave(true);
        }

        private void btnOgPortAdd_Click(object sender, EventArgs e)
        {
            int port    = (int)nbOgNewPort.Value;
            if (lbOgPorts.Items.Contains(port) == true)
                goto failed;
                
            if (port == isSendPort)
                goto failed;

            if (port == isRecvPort)
                goto failed;

            if (port == ogRecvPort && ogDirect == false)
                goto failed;

            lbOgPorts.Items.Add(port);
            EnableBtnSave(true);
            
        failed:
                return;
        }

        private void lbOgPorts_SelectedIndexChanged(object sender, EventArgs e)
        {
            btnOgPortRemove.Enabled = (lbOgPorts.SelectedIndex >= 0);
        }

        #endregion

        #region tab gauges

        #region speed

        private void cbSpeedVisible_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbSpeedVisible.SelectedIndex == (int)VisibleTypeSpeed.No)
                pnlSpeedPosSize.Enabled = false;
            else
                pnlSpeedPosSize.Enabled = true;

            if (currWndSettings.speedVisibleType == cbSpeedVisible.SelectedIndex)
                return;

            currWndSettings.speedVisibleType = cbSpeedVisible.SelectedIndex;                   
            if (currWndSettings == currCarSettings)
                RequestState();

            EnableBtnSave(true);
        }

        private void nbSpeedX_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.speedPosSize.x == nbSpeedX.Value)
                return;

            currWndSettings.speedPosSize.x = (int)nbSpeedX.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetSpeedPosSize(currCarSettings.speedPosSize.x, currCarSettings.speedPosSize.y, currCarSettings.speedPosSize.size);

            EnableBtnSave(true);
        }

        private void nbSpeedY_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.speedPosSize.y == nbSpeedY.Value)
                return;

            currWndSettings.speedPosSize.y = (int)nbSpeedY.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetSpeedPosSize(currCarSettings.speedPosSize.x, currCarSettings.speedPosSize.y, currCarSettings.speedPosSize.size);

            EnableBtnSave(true);
        }

        private void nbSpeedSize_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.speedPosSize.size == nbSpeedSize.Value)
                return;

            currWndSettings.speedPosSize.size = (int)nbSpeedSize.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetSpeedPosSize(currCarSettings.speedPosSize.x, currCarSettings.speedPosSize.y, currCarSettings.speedPosSize.size);

            EnableBtnSave(true);
        }

        #endregion

        #region gear

        private void cbGearVisible_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbGearVisible.SelectedIndex != (int)VisibleType.Yes)
                pnlGearPosSize.Enabled = false;
            else
                pnlGearPosSize.Enabled = true;

            if (currWndSettings.gearVisibleType == cbGearVisible.SelectedIndex)
                return;

            currWndSettings.gearVisibleType = cbGearVisible.SelectedIndex; 
            
            if (currWndSettings == currCarSettings)
                RequestState();

            EnableBtnSave(true);
        }

        private void nbGearX_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.gearPosSize.x == nbGearX.Value)
                return;

            currWndSettings.gearPosSize.x = (int)nbGearX.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetGearPosSize(currCarSettings.gearPosSize.x, currCarSettings.gearPosSize.y, currCarSettings.gearPosSize.size);

            EnableBtnSave(true);
        }

        private void nbGearY_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.gearPosSize.y == nbGearY.Value)
                return;

            currWndSettings.gearPosSize.y = (int)nbGearY.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetGearPosSize(currCarSettings.gearPosSize.x, currCarSettings.gearPosSize.y, currCarSettings.gearPosSize.size);

            EnableBtnSave(true);
        }

        private void nbGearSize_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.gearPosSize.size == nbGearSize.Value)
                return;

            currWndSettings.gearPosSize.size = (int)nbGearSize.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetGearPosSize(currCarSettings.gearPosSize.x, currCarSettings.gearPosSize.y, currCarSettings.gearPosSize.size);

            EnableBtnSave(true);
        }

        #endregion

        #region led

        private void cbLedVisible_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbLedVisible.SelectedIndex != (int)VisibleType.Yes)
                pnlLedPosSize.Enabled = false;
            else
                pnlLedPosSize.Enabled = true;

            if (currWndSettings.ledVisibleType == cbLedVisible.SelectedIndex)
                return;

            currWndSettings.ledVisibleType = cbLedVisible.SelectedIndex;   
            if (currWndSettings == currCarSettings)    
                RequestState();

            EnableBtnSave(true);
        }

        private void nbLedX_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.ledPosSize.x == nbLedX.Value)
                return;

            currWndSettings.ledPosSize.x = (int)nbLedX.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetLedPosSize(currCarSettings.ledPosSize.x, currCarSettings.ledPosSize.y, currCarSettings.ledPosSize.size);

            EnableBtnSave(true);
        }

        private void nbLedY_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.ledPosSize.y == nbLedY.Value)
                return;

            currWndSettings.ledPosSize.y = (int)nbLedY.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetLedPosSize(currCarSettings.ledPosSize.x, currCarSettings.ledPosSize.y, currCarSettings.ledPosSize.size);

            EnableBtnSave(true);
        }

        private void nbLedSize_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.ledPosSize.size == nbLedSize.Value)
                return;

            currWndSettings.ledPosSize.size = (int)nbLedSize.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetLedPosSize(currCarSettings.ledPosSize.x, currCarSettings.ledPosSize.y, currCarSettings.ledPosSize.size);

            EnableBtnSave(true);
        }

        #endregion        

        #region rpm

        private void cbRpmVisible_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbRpmVisible.SelectedIndex != (int)VisibleType.Yes)
                pnlRpmPosSize.Enabled = false;
            else
                pnlRpmPosSize.Enabled = true;

            if (currWndSettings.rpmVisibleType == cbRpmVisible.SelectedIndex)
                return;

            currWndSettings.rpmVisibleType = cbRpmVisible.SelectedIndex;                   
            if (currWndSettings == currCarSettings)
                RequestState();

            EnableBtnSave(true);
        }

        private void nbRpmX_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.rpmPosSize.x == nbRpmX.Value)
                return;

            currWndSettings.rpmPosSize.x = (int)nbRpmX.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetRpmPosSize(currCarSettings.rpmPosSize.x, currCarSettings.rpmPosSize.y, currCarSettings.rpmPosSize.size);

            EnableBtnSave(true);
        }

        private void nbRpmY_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.rpmPosSize.y == nbRpmY.Value)
                return;

            currWndSettings.rpmPosSize.y = (int)nbRpmY.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetRpmPosSize(currCarSettings.rpmPosSize.x, currCarSettings.rpmPosSize.y, currCarSettings.rpmPosSize.size);

            EnableBtnSave(true);
        }        

        private void nbRpmSize_ValueChanged(object sender, EventArgs e)
        {
            if (currWndSettings.rpmPosSize.size == nbRpmSize.Value)
                return;

            currWndSettings.rpmPosSize.size = (int)nbRpmSize.Value;
            if (currWndSettings == currCarSettings && gfxOutput.LibraryLoaded)
                gfxOutput.SetRpmPosSize(currCarSettings.rpmPosSize.x, currCarSettings.rpmPosSize.y, currCarSettings.rpmPosSize.size);

            EnableBtnSave(true);
        }

        #endregion

        private void rbColorCustom_CheckedChanged(object sender, EventArgs e)
        {
            customColor         = rbColorCustom.Checked;
            btnColor.Enabled    = customColor;

            if (!customColor)
            {
                pnlColor.BackColor  = Color.LightGray;
                colorLfs            = GetLfsClockColor(Path.GetDirectoryName(lfsPath));
            }

            string col = (customColor) ? colorCustom : colorLfs;
            SetDigitsColor(col, customColor);

            EnableBtnSave(true);
        }

        private void btnColor_Click(object sender, EventArgs e)
        {
            clrDlg.FullOpen = true;
            clrDlg.Color    = ParseColor(colorCustom);

            if (clrDlg.ShowDialog() == DialogResult.OK)
            {
                Color col   = clrDlg.Color;
                colorCustom =  String.Format("{0:x2}{1:x2}{2:x2}", col.R, col.G, col.B);

                SetDigitsColor(colorCustom, true);
                EnableBtnSave(true);
            }
        }     

        private void cbSpecHide_CheckedChanged(object sender, EventArgs e)
        {
            specHide = cbSpecHide.Checked;
                
            RequestState();
            EnableBtnSave(true);
        }

        private void lbCars_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                currWndSettings     = customSettings[lbCars.SelectedItem.ToString()] as GaugeSettings;
            
                cbSpeedVisible.SelectedIndex    = ValidateValue(0, cbSpeedVisible.Items.Count - 1, currWndSettings.speedVisibleType); 
                nbSpeedX.Value                  = ValidateValue(nbSpeedX.Minimum, nbSpeedX.Maximum, currWndSettings.speedPosSize.x);   
                nbSpeedY.Value                  = ValidateValue(nbSpeedY.Minimum, nbSpeedY.Maximum, currWndSettings.speedPosSize.y); 
                nbSpeedSize.Value               = ValidateValue(nbSpeedSize.Minimum, nbSpeedSize.Maximum, currWndSettings.speedPosSize.size);

                cbGearVisible.SelectedIndex     = ValidateValue(0, cbGearVisible.Items.Count - 1, currWndSettings.gearVisibleType);
                nbGearX.Value                   = ValidateValue(nbGearX.Minimum, nbGearX.Maximum, currWndSettings.gearPosSize.x);   
                nbGearY.Value                   = ValidateValue(nbGearY.Minimum, nbGearY.Maximum, currWndSettings.gearPosSize.y);   
                nbGearSize.Value                = ValidateValue(nbGearSize.Minimum, nbGearSize.Maximum, currWndSettings.gearPosSize.size);   

                cbLedVisible.SelectedIndex      = ValidateValue(0, cbLedVisible.Items.Count - 1, currWndSettings.ledVisibleType); 
                nbLedX.Value                    = ValidateValue(nbLedX.Minimum, nbLedX.Maximum, currWndSettings.ledPosSize.x);   
                nbLedY.Value                    = ValidateValue(nbLedY.Minimum, nbLedY.Maximum, currWndSettings.ledPosSize.y);   
                nbLedSize.Value                 = ValidateValue(nbLedSize.Minimum, nbLedSize.Maximum, currWndSettings.ledPosSize.size);   

                cbRpmVisible.SelectedIndex      = ValidateValue(0, cbRpmVisible.Items.Count - 1, currWndSettings.rpmVisibleType); 
                nbRpmX.Value                    = ValidateValue(nbRpmX.Minimum, nbRpmX.Maximum, currWndSettings.rpmPosSize.x);   
                nbRpmY.Value                    = ValidateValue(nbRpmY.Minimum, nbRpmY.Maximum, currWndSettings.rpmPosSize.y);   
                nbRpmSize.Value                 = ValidateValue(nbRpmSize.Minimum, nbRpmSize.Maximum, currWndSettings.rpmPosSize.size);   
            }
            catch(Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        #endregion

        #region tab startup

        private void cbStartMinimized_CheckedChanged(object sender, EventArgs e)
        {                      
            startMinimized = cbStartMinimized.Checked;                            
            EnableBtnSave(true);
        }      

        private void btnProgramPath_Click(object sender, EventArgs e)
        {
            ofDlg.InitialDirectory = (lfsPath.Length > 0) ? Path.GetDirectoryName(lfsPath) : "c:\\";
            ofDlg.Filter            = "Executables|*.bat;*.cmd;*.com;*.exe";
            if (ofDlg.ShowDialog(this) == DialogResult.OK)
            {
                tbProgramPath.Text          = ofDlg.FileName;
                tbProgramParameters.Enabled = true;
                cbWindowState.Enabled       = true;
                btnProgramAdd.Enabled       = true;                
                tbProgramParameters.Text    = string.Empty;
                cbWindowState.SelectedIndex = 1;

                ProgramPath_AddLfsOptions();                
            }
        }

        private void btnProgramAdd_Click(object sender, EventArgs e)
        {               
            ListViewItem lvi = null;            
            if (lvPrograms.Items.ContainsKey(tbProgramPath.Text))
            {
                ListViewItem[] lvia = lvPrograms.Items.Find(tbProgramPath.Text, false);
                if (lvia.Length == 0)
                    return;

                lvi = lvia[0];
                lvi.SubItems[1].Text = tbProgramParameters.Text;   
                lvi.SubItems[2].Text = cbWindowState.Text;
            }
            else
            {
                lvi = new ListViewItem();                

                lvi.Name            = tbProgramPath.Text;
                lvi.Text            = Path.GetFileName(tbProgramPath.Text);
                lvi.Checked         = true;
                lvi.SubItems.Add(tbProgramParameters.Text);   
                lvi.SubItems.Add(cbWindowState.Text);
            
                lvPrograms.Items.Add(lvi);
            }
                
            if (lvi == null)
                return;

            tbProgramPath.Text          = string.Empty;
            tbProgramParameters.Text    = string.Empty;
            tbProgramParameters.Enabled = false;
            cbWindowState.Enabled       = false;
            cbWindowState.SelectedIndex = 1;
            btnProgramAdd.Enabled       = false;

            EnableBtnSave(true);
            lvPrograms.SelectedItems.Clear();
        }
        
        private void lvPrograms_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            EnableBtnSave(true);
        }

        private void lvPrograms_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (lvPrograms.SelectedItems.Count == 0)
            {
                btnProgramRemove.Enabled    = false;
                btnProgramMoveUp.Enabled    = false;
                btnProgramMoveDown.Enabled  = false;                

                return;
            }            

            try
            {
                ListViewItem lvi            = lvPrograms.SelectedItems[0];
                btnProgramRemove.Enabled    = true;
                btnProgramMoveUp.Enabled    = (lvi.Index > 0);
                btnProgramMoveDown.Enabled  = (lvi.Index < lvPrograms.Items.Count - 1);

                tbProgramPath.Text          = lvi.Name;
                tbProgramParameters.Text    = lvi.SubItems[1].Text;
                
                ProgramPath_AddLfsOptions(); 
                cbWindowState.Text          = lvi.SubItems[2].Text;

                tbProgramParameters.Enabled = true;
                cbWindowState.Enabled       = true;
                btnProgramAdd.Enabled       = true;               
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        private void btnProgramRemove_Click(object sender, EventArgs e)
        {
            if (lvPrograms.SelectedItems.Count == 0)
                return;

            lvPrograms.Items.Remove(lvPrograms.SelectedItems[0]);

            tbProgramPath.Text          = string.Empty;
            tbProgramParameters.Text    = string.Empty;
            tbProgramParameters.Enabled = false;
            cbWindowState.Enabled       = false;
            cbWindowState.SelectedIndex = 1;
            btnProgramAdd.Enabled       = false;

            EnableBtnSave(true);
            lvPrograms.SelectedItems.Clear();
        }

        private void btnProgramMoveUp_Click(object sender, EventArgs e)
        {
            if (lvPrograms.SelectedItems.Count == 0)
                return;

            try
            {
                ListViewItem lvi    = lvPrograms.SelectedItems[0];
                int idx             = lvi.Index;
                lvPrograms.Items.Remove(lvi);
                lvPrograms.Items.Insert(idx - 1, lvi);

                 EnableBtnSave(true);
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }            
        }

        private void btnProgramMoveDown_Click(object sender, EventArgs e)
        {
            if (lvPrograms.SelectedItems.Count == 0)
                return;

            try
            {
                ListViewItem lvi    = lvPrograms.SelectedItems[0];
                int idx             = lvi.Index;
                lvPrograms.Items.Remove(lvi);
                lvPrograms.Items.Insert(idx + 1, lvi);

                 EnableBtnSave(true);
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        private void cbClosePrograms_CheckedChanged(object sender, EventArgs e)
        {
            closePrograms = cbClosePrograms.Checked;
            EnableBtnSave(true);
        }

        private void cbAutoClose_CheckedChanged(object sender, EventArgs e)
        {
            autoClose = cbAutoClose.Checked;
            EnableBtnSave(true);
        }

        #endregion

        #endregion

        #region methods

        #region configuration

        private void LoadDefaultSettings()
        {
            try
            {
                customSettings.Clear();
                lbCars.Items.Clear();

                tbLfsPath.Text          = lfsPath       = string.Empty;
                nbIsSendPort.Value      = isSendPort    = 29999;
                nbIsRecvPort.Value      = isRecvPort    = 30011;
                nbOgRecvPort.Value      = ogRecvPort    = 30003;
                tbAdminPass.Text        = adminPass     = string.Empty;
                cbSpecHide.Checked      = specHide      = false;
                rbColorLfs.Checked      = !(customColor = false);
                rbOgDirect.Checked      = ogDirect      = true;
                cbOgAutoCfg.Checked     = ogAutoCfg     = true;
                cbAutoClose.Checked     = autoClose     = false;
                cbClosePrograms.Checked = closePrograms = false;
                cbWindowState.SelectedIndex             = 1;

                string[] cars ={"BF1", "FBM", "FO8", "FOX", "FXO", "FXR", "FZ5", "FZR", "LX4", "LX6", "MRT", "RAC", "RB4", "UF1", "UFR", "XFG", "XFR", "XRG", "XRR", "XRT"};
                foreach(string car in cars)
                {
                    GaugeSettings gs        = new GaugeSettings();

                    gs.speedVisibleType     = (int)VisibleTypeSpeed.Yes;
                    gs.speedPosSize.x       = 47;
                    gs.speedPosSize.y       = 770;
                    gs.speedPosSize.size    = 35;

                    gs.gearVisibleType      = (int)VisibleType.Yes;
                    gs.gearPosSize.x        = 138;
                    gs.gearPosSize.y        = 767;
                    gs.gearPosSize.size     = 50;

                    gs.ledVisibleType       = (int)VisibleType.Yes;
                    gs.ledPosSize.x         = 45;
                    gs.ledPosSize.y         = 815;
                    gs.ledPosSize.size      = 65;

                    gs.rpmVisibleType       = (int)VisibleType.No;
                    gs.rpmPosSize.x         = 80;
                    gs.rpmPosSize.y         = 815;
                    gs.rpmPosSize.size      = 30;
                    
                    customSettings.Add(car, gs);                    
                    lbCars.Items.Add(car);                
                }                               
            }
            catch(Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        /*
        private void LoadSettings_06()
        {
            XmlConfig cfg = new XmlConfig(Application.ExecutablePath);
            cfg.Load();

            try
            {
                tbLfsPath.Text      = lfsPath     = cfg.ReadString("lfs_path", string.Empty);
                nbIsSendPort.Value  = isSendPort  = cfg.ReadInt("send_port", 29999);                
                tbAdminPass.Text    = adminPass   = cfg.ReadString("admin_pass", string.Empty);
                cbSpecHide.Checked  = specHide    = Convert.ToBoolean(Int16.Parse(cfg.ReadString("spec_hide", "0")));

                string id               = string.Empty;
                SortedList sl           = cfg.ReadMultiple("gauges", ref id);                
                if (sl == null)
                {
                    LoadDefaultSettings();
                    return;
                }

                string[] cars ={"BF1", "FBM", "FO8", "FOX", "FXO", "FXR", "FZ5", "FZR", "LX4", "LX6", "MRT", "RAC", "RB4", "UF1", "UFR", "XFG", "XFR", "XRG", "XRR", "XRT"};
                foreach(string car in cars)
                {                
                    GaugeSettings gs        = new GaugeSettings();
                    
                    gs.speedVisibleType     = Int32.Parse(sl["speed_visible"].ToString());
                    gs.speedPosSize.x       = Int32.Parse(sl["speed_x"].ToString());
                    gs.speedPosSize.y       = Int32.Parse(sl["speed_y"].ToString());  
                    gs.speedPosSize.size    = Int32.Parse(sl["speed_size"].ToString());
                    
                    gs.gearVisibleType      = Int32.Parse(sl["gear_visible"].ToString());
                    gs.gearPosSize.x        = Int32.Parse(sl["gear_x"].ToString());
                    gs.gearPosSize.y        = Int32.Parse(sl["gear_y"].ToString());  
                    gs.gearPosSize.size     = Int32.Parse(sl["gear_size"].ToString());
                    
                    gs.ledVisibleType       = Int32.Parse(sl["led_visible"].ToString());
                    gs.ledPosSize.x         = Int32.Parse(sl["led_x"].ToString());
                    gs.ledPosSize.y         = Int32.Parse(sl["led_y"].ToString());  
                    gs.ledPosSize.size      = Int32.Parse(sl["led_size"].ToString());

                    gs.rpmVisibleType       = Int32.Parse(sl["rpm_visible"].ToString());
                    gs.rpmPosSize.x         = Int32.Parse(sl["rpm_x"].ToString());
                    gs.rpmPosSize.y         = Int32.Parse(sl["rpm_y"].ToString());  
                    gs.rpmPosSize.size      = Int32.Parse(sl["rpm_size"].ToString());

                    if (customSettings.ContainsKey(car))
                        customSettings[car] = gs;
                    else
                    {
                        customSettings.Add(car, gs);  
                        lbCars.Items.Add(car);
                    }
                }                                                
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());

                LoadDefaultSettings();
            }
            finally 
            {
                EnableBtnSave(false);
            }
        }
        */

        private void LoadSettings()
        {            
            XmlConfig cfg = new XmlConfig(Application.ExecutablePath);
            cfg.Load();

            try
            {
                string version      = cfg.ReadString("app_version", string.Empty);
                if (version == string.Empty)
                {
                    //LoadSettings_06();
                    return;
                }

                tbLfsPath.Text              = lfsPath           = cfg.ReadString("lfs_path", string.Empty);
                nbIsSendPort.Value          = isSendPort        = cfg.ReadInt("send_port", 29999);
                nbIsRecvPort.Value          = isRecvPort        = cfg.ReadInt("recv_port", 30011);
                nbOgRecvPort.Value          = ogRecvPort        = cfg.ReadInt("og_recv_port", 30003);
                tbAdminPass.Text            = adminPass         = cfg.ReadString("admin_pass", string.Empty);
                cbSpecHide.Checked          = specHide          = Convert.ToBoolean(Int16.Parse(cfg.ReadString("spec_hide", "0")));
                cbStartMinimized.Checked    = startMinimized    = Convert.ToBoolean(Int16.Parse(cfg.ReadString("start_min", "0")));
                rbColorCustom.Checked       = customColor       = Convert.ToBoolean(Int16.Parse(cfg.ReadString("custom_color", "0")));
                rbColorLfs.Checked          = !customColor;
                colorCustom                 = cfg.ReadString("color", "ffffff");
                rbOgDirect.Checked          = ogDirect          = Convert.ToBoolean(Int16.Parse(cfg.ReadString("og_direct", "1")));
                rbOgIndirect.Checked        = !ogDirect;
                cbOgAutoCfg.Checked         = ogAutoCfg         = Convert.ToBoolean(Int16.Parse(cfg.ReadString("og_autocfg", "1")));
                cbAutoClose.Checked         = autoClose         = Convert.ToBoolean(Int16.Parse(cfg.ReadString("auto_close", "0")));
                cbClosePrograms.Checked     = closePrograms     = Convert.ToBoolean(Int16.Parse(cfg.ReadString("close_programs", "0")));

                string ports                = cfg.ReadString("og_ports", string.Empty);
                string[] sa = ports.Split(';');
                foreach (string s in sa)
                {
                    if (s.Length == 0)
                        continue;

                    lbOgPorts.Items.Add(Int32.Parse(s));                    
                }

                SortedList sl       = null;
                string id           = string.Empty;
                while((sl = cfg.ReadMultiple("programs", ref id)) != null)
                {
                    ListViewItem lvi    = new ListViewItem();
                    lvi.Name            = sl["path"].ToString();
                    lvi.Text            = id;
                    lvi.Checked         = Convert.ToBoolean(Int16.Parse(sl["enabled"].ToString()));
                    lvi.SubItems.Add(sl["params"].ToString()); 
                    if (sl.ContainsKey("wnd_state"))
                        lvi.SubItems.Add(sl["wnd_state"].ToString());                     
                    else
                        lvi.SubItems.Add(WndState.Normal);

                    lvPrograms.Items.Add(lvi);
                }

                while((sl = cfg.ReadMultiple("gauges", ref id)) != null)
                {
                    GaugeSettings gs        = new GaugeSettings();
                    
                    gs.speedVisibleType     = Int32.Parse(sl["speed_visible"].ToString());
                    gs.speedPosSize.x       = Int32.Parse(sl["speed_x"].ToString());
                    gs.speedPosSize.y       = Int32.Parse(sl["speed_y"].ToString());  
                    gs.speedPosSize.size    = Int32.Parse(sl["speed_size"].ToString());
                    
                    gs.gearVisibleType      = Int32.Parse(sl["gear_visible"].ToString());
                    gs.gearPosSize.x        = Int32.Parse(sl["gear_x"].ToString());
                    gs.gearPosSize.y        = Int32.Parse(sl["gear_y"].ToString());  
                    gs.gearPosSize.size     = Int32.Parse(sl["gear_size"].ToString());
                    
                    gs.ledVisibleType       = Int32.Parse(sl["led_visible"].ToString());
                    gs.ledPosSize.x         = Int32.Parse(sl["led_x"].ToString());
                    gs.ledPosSize.y         = Int32.Parse(sl["led_y"].ToString());  
                    gs.ledPosSize.size      = Int32.Parse(sl["led_size"].ToString());

                    gs.rpmVisibleType       = Int32.Parse(sl["rpm_visible"].ToString());
                    gs.rpmPosSize.x         = Int32.Parse(sl["rpm_x"].ToString());
                    gs.rpmPosSize.y         = Int32.Parse(sl["rpm_y"].ToString());  
                    gs.rpmPosSize.size      = Int32.Parse(sl["rpm_size"].ToString());
                
                    if (customSettings.ContainsKey(id))
                        customSettings[id] = gs;
                    else
                    {
                        customSettings.Add(id, gs);  
                        lbCars.Items.Add(id);
                    }
                }                
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());

                LoadDefaultSettings();
            }
            finally 
            {
                EnableBtnSave(false);
            }            
        }

        private void SaveSettings()
        {
            try
            {
                XmlConfig cfg = new XmlConfig(Application.ExecutablePath);

                cfg.Write("app_version", Assembly.GetExecutingAssembly().GetName().Version.ToString());
                cfg.Write("lfs_path", lfsPath);
                cfg.Write("send_port", isSendPort.ToString());
                cfg.Write("recv_port", isRecvPort.ToString());
                cfg.Write("og_recv_port", ogRecvPort.ToString());
                cfg.Write("admin_pass", adminPass);
                cfg.Write("spec_hide", Convert.ToInt16(specHide).ToString());
                cfg.Write("start_min", Convert.ToInt16(startMinimized).ToString());
                cfg.Write("custom_color", Convert.ToInt16(customColor).ToString());
                cfg.Write("color", colorCustom);
                cfg.Write("og_direct", Convert.ToInt16(ogDirect).ToString());
                cfg.Write("og_autocfg", Convert.ToInt16(ogAutoCfg).ToString());
                cfg.Write("auto_close", Convert.ToInt16(autoClose).ToString());
                cfg.Write("close_programs", Convert.ToInt16(closePrograms).ToString());
               
                string ports = string.Empty;
                foreach(int port in lbOgPorts.Items)
                    ports += String.Format("{0};", port);
                cfg.Write("og_ports", ports);

                foreach(ListViewItem lvi in lvPrograms.Items)
                {
                    SortedList sl = new SortedList();
                    sl.Add("path", lvi.Name);
                    sl.Add("params", lvi.SubItems[1].Text);
                    sl.Add("wnd_state", lvi.SubItems[2].Text);
                    sl.Add("enabled", Convert.ToInt16(lvi.Checked).ToString());

                    cfg.WriteMultiple("programs", lvi.Text, sl);
                }

                foreach(DictionaryEntry de in customSettings)
                    cfg.WriteMultiple("gauges", de.Key.ToString(), CreateGaugeCfg(de.Value as GaugeSettings));

                
                cfg.Save();
            }
            catch(Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        private SortedList CreateGaugeCfg(GaugeSettings gs)
        {
            SortedList sl = new SortedList();

            try
            {
                sl.Add("speed_visible", gs.speedVisibleType.ToString());
                sl.Add("speed_x", gs.speedPosSize.x.ToString());
                sl.Add("speed_y", gs.speedPosSize.y.ToString());
                sl.Add("speed_size", gs.speedPosSize.size.ToString());

                sl.Add("gear_visible", gs.gearVisibleType.ToString());
                sl.Add("gear_x", gs.gearPosSize.x.ToString());
                sl.Add("gear_y", gs.gearPosSize.y.ToString());
                sl.Add("gear_size", gs.gearPosSize.size.ToString());

                sl.Add("led_visible", gs.ledVisibleType.ToString());
                sl.Add("led_x", gs.ledPosSize.x.ToString());
                sl.Add("led_y", gs.ledPosSize.y.ToString());
                sl.Add("led_size", gs.ledPosSize.size.ToString());

                sl.Add("rpm_visible", gs.rpmVisibleType.ToString());
                sl.Add("rpm_x", gs.rpmPosSize.x.ToString());
                sl.Add("rpm_y", gs.rpmPosSize.y.ToString());
                sl.Add("rpm_size", gs.rpmPosSize.size.ToString());
            }
            catch(Exception ex)
            {
                Log.Error(ex.ToString());
            }

            return sl;
        }

        #endregion        

        private void InitLfsConnection()
        {
            ushort flags = 0;
            flags |= (ushort)ISF.ISF_MCI;

            IS_ISI packet = new IS_ISI((ushort)isRecvPort, adminPass, "Digital Speedo", flags);
            isWriter.Send(packet);
        }

        private void SetConnected(bool val)
        {
            lfsConnected                = val;
            timerIsConnected.Enabled    = val;
            
            if (!lfsConnected)
            {
                ogNeedRequest = true;

                if (gfxOutput.LibraryLoaded)
                    gfxOutput.ResetOG();
            }

            tsConnectState.Text = (val) ? "connected" : "not connected";
            if (val &&  gfxOutput.LibraryLoaded == false)
                tsConnectState.Text += ", d3d8.dll not valid";            
        }

        private bool ValidatePorts()
        {
            string msg  = string.Empty;
            if (isSendPort == isRecvPort)
            {
                msg = String.Format("Ports conflict: {0} - IS send port, IS recv port", isRecvPort);
                goto failed;
            }
  
            if (ogAutoCfg == false && ogDirect == false && isSendPort == ogRecvPort)
            {
                msg = String.Format("Ports conflict: {0} - IS send port, OG recv port", ogRecvPort);
                goto failed;
            }

            if (ogAutoCfg == false && ogDirect == false && isRecvPort == ogRecvPort)
            {
                msg = String.Format("Ports conflict: {0} - IS recv port, OG recv port", ogRecvPort);
                goto failed;                
            }

            if (lbOgPorts.Items.Contains(isSendPort))
            {
                msg = String.Format("Ports conflict: {0} - IS send port, OG gateway send port", isSendPort);
                goto failed;                
            }

            if (lbOgPorts.Items.Contains(isRecvPort))
            {
                msg = String.Format("Ports conflict: {0} - IS recv port, OG gateway send port", isRecvPort);
                goto failed;                
            }

            if (lbOgPorts.Items.Contains(ogRecvPort) && ogDirect == false)
            {
                msg = String.Format("Ports conflict: {0} - OG recv port, OG gateway send port", isRecvPort);
                goto failed;                
            }
            
            return true;

        failed:
            Log.Error(msg);
            MessageBox.Show(msg, appName, MessageBoxButtons.OK, MessageBoxIcon.Error);

            return false;
        }

        private void RequestState()
        {
            if (isWriter == null || isWriter.Connected == false)
                return;

            IS_TINY req = new IS_TINY(1, TINY.TINY_SST);
            isWriter.Send(req);
        }
        
        private void RequestOutGauge(uint delay)
        {
            if (isWriter == null || isWriter.Connected == false)
                return;

            IS_SMALL req = new IS_SMALL(0, SMALL.SMALL_SSG, delay);
			isWriter.Send(req);
        }

        private void RequestPlayers()
        {
            if (isWriter == null || isWriter.Connected == false)
                return;

            IS_TINY req = new IS_TINY(1, TINY.TINY_NPL);
            isWriter.Send(req);
        }

        private void OutGaugeEvent(InSim.Structs.OutGaugePack packet)
        {
            if (currCar != packet.Car)
                SetSettings(packet.Car);

            if (gfxOutput.LibraryLoaded)
            {
                if (gaugeState.speedVisible && packet.Speed != gaugeState.speed)
                {
                    gaugeState.speed = (int)packet.Speed;
                    gfxOutput.SetSpeed((int)packet.Speed);
                }

                if (gaugeState.gearVisible && packet.Gear != gaugeState.gear)
                {
                    gaugeState.gear = (int)packet.Gear;
                    gfxOutput.SetGear((int)packet.Gear);
                }
                
                if (gaugeState.ledVisible)
                {
                    int led = 0;
                    if ((packet.Flags & (ushort)OG_FLAGS.OG_REDLINE) != 0)
                        led = (int)GfxOutput.Led.Redline;
                    else if ((packet.Flags & (ushort)OG_FLAGS.OG_SHIFTLIGHT) != 0 || (packet.ShowLights & (uint)DL_FLAGS.DL_SHIFT) != 0)
                        led = (int)GfxOutput.Led.ShiftLight;            
                    else
                        led = (int)GfxOutput.Led.Off;

                    if (led != gaugeState.led)
                    {
                        gaugeState.led = led;
                        gfxOutput.SetLed(led);
                    }
                }

                if (gaugeState.rpmVisible && packet.RPM != gaugeState.rpm)
                {
                    gaugeState.rpm = (int)packet.RPM;
                    gfxOutput.SetRpm((int)packet.RPM);
                }

                foreach(InSimWriter iw in ogClients.Values)
                    iw.Send(packet);
            }
        }

        private Color ParseColor(string color)
        {
            if (color.Length != 6)
                return Color.White;

            int r = int.Parse(color.Substring(0, 2), System.Globalization.NumberStyles.HexNumber);
            int g = int.Parse(color.Substring(2, 2), System.Globalization.NumberStyles.HexNumber);
            int b = int.Parse(color.Substring(4, 2), System.Globalization.NumberStyles.HexNumber);

            return Color.FromArgb(r, g, b);
        }

        private void SetDigitsColor(string color, bool updatePanel)
        {       
            if (color.Length != 6)
                return;

            int r = int.Parse(color.Substring(0, 2), System.Globalization.NumberStyles.HexNumber);
            int g = int.Parse(color.Substring(2, 2), System.Globalization.NumberStyles.HexNumber);
            int b = int.Parse(color.Substring(4, 2), System.Globalization.NumberStyles.HexNumber);

            if (updatePanel)
                pnlColor.BackColor = Color.FromArgb(r, g, b);

            if (gfxOutput != null && gfxOutput.LibraryLoaded)
                gfxOutput.SetDigitsColor(r, g, b);
        }

        private void ExtractResource(string resName, bool overWrite)
		{
			try
			{			
			    string fp       = Path.GetDirectoryName(Application.ExecutablePath);
                fp              += "\\" + resName;

                if (overWrite == false && File.Exists(fp))
                    return;
                               
				Assembly Asm    = Assembly.GetExecutingAssembly();			
				//string name   = Asm.GetName().Name + "." + resName;
                string name     = GetType().Namespace + "." + resName;                
				Stream res      = Asm.GetManifestResourceStream(name);
                if (res == null)
                    throw new Exception("Unknown resource: " + name);

                Stream file     = File.Create(fp);
                BinaryReader br = new BinaryReader(res);                                
                BinaryWriter bw = new BinaryWriter(file);

                bw.Write(br.ReadBytes((int)res.Length));
                bw.Flush();

                bw.Close();
                br.Close();
			}
			catch(Exception ex)
			{
                Log.Error(ex.ToString());
			}
        }

        private string GetLfsClockColor(string path)
        {         
            string result = "ffffff";

            try
            {                
                string fp       = path + "\\cfg.txt";
                StreamReader sr = new StreamReader(fp, System.Text.Encoding.Default);

                while(true)
				{
					string s = sr.ReadLine();
					if(s == null)
						break;

					if (s.ToLower().StartsWith("clock text"))
					{
						s           = s.ToLower().Replace("clock text ", string.Empty);
						string[] sa = s.Split(' ');
						
                        if (sa.Length == 3)
                            result = String.Format("{0:x2}{1:x2}{2:x2}", Int16.Parse(sa[0]), Int16.Parse(sa[1]), Int16.Parse(sa[2]));
					}
				}
				
				sr.Close();
            }
            catch(Exception ex)
            {
                Log.Error(ex.ToString());
            }

            return result;
        }

        private void GetLfsOutGaugeConfig(string path)
        {         
            try
            {                
                string fp       = path + "\\cfg.txt";
                StreamReader sr = new StreamReader(fp, System.Text.Encoding.Default);

                while(true)
				{
					string s = sr.ReadLine();
					if(s == null)
						break;

					if (s.ToLower().StartsWith("outgauge mode"))
					{
						s           = s.ToLower().Replace("outgauge mode ", string.Empty);						
                        if (s == "0")
                            rbOgDirect.Checked  = ogDirect  = true;
                        else
                        {
                            rbOgDirect.Checked      = ogDirect  = false;
                            rbOgIndirect.Checked    = true;
                        }
					}

                    if (s.ToLower().StartsWith("outgauge port"))
					{
						s                   = s.ToLower().Replace("outgauge port ", string.Empty);						
                        nbOgRecvPort.Value  = ogRecvPort    = Int32.Parse(s);
					}
				}
				
				sr.Close();
            }
            catch(Exception ex)
            {
                Log.Error(ex.ToString());
            }            
        }

        private int ValidateValue(int min, int max, int val)
        {
            val = (val > min) ? val : min;
            val = (val < max) ? val : max;

            return val;
        }

        private int ValidateValue(decimal min, decimal max, int val)
        {
            val = (val > (int)min) ? val : (int)min;
            val = (val < (int)max) ? val : (int)max;

            return val;
        }

        private void EnableBtnSave(bool enable)
        {
            btnSave.Enabled = enable;
        }
        
        private void SetSettings(string car)
        {            
            if (gfxOutput.LibraryLoaded)
            {                
                gfxOutput.SetSpeedVisible(false);
                gaugeState.speedVisible = false;

                gfxOutput.SetGearVisible(false);
                gaugeState.gearVisible  = false;

                gfxOutput.SetLedVisible(false);
                gaugeState.ledVisible   = false;

                gfxOutput.SetRpmVisible(false);
                gaugeState.rpmVisible   = false;
            }

            if (customSettings.ContainsKey(car))
            {
                currCarSettings      = customSettings[car] as GaugeSettings;    
            
                this.Invoke(new MethodInvoker(delegate()
                {
                    lbCars.SelectedIndex = lbCars.FindStringExact(car);
                }));                
            }
            else
            {
                currCarSettings.speedVisibleType    = (int)VisibleTypeSpeed.No;
                currCarSettings.gearVisibleType     = (int)VisibleType.No;
                currCarSettings.ledVisibleType      = (int)VisibleType.No;
                currCarSettings.rpmVisibleType      = (int)VisibleType.No;
            }

            if (currCar == string.Empty) 
                RequestState();

            currCar     = car;            
        }

        private void Start()
        {            
            try
            {              
                gfxOutput   = new GfxOutput();

                if (ogAutoCfg && lfsPath.Length > 0)
                    GetLfsOutGaugeConfig(Path.GetDirectoryName(lfsPath));

                if (ValidatePorts() == false)
                    return;

                isWriter    = new InSimWriter("127.0.0.1", isSendPort);
                isReader    = new InSimReader(isRecvPort);                   
                                               
                isReader.KeepAliveEvent         += new InSimReader.KeepAliveDelegate(isReader_KeepAliveEvent);
                isReader.InitialRequestEvent    += new InSimReader.InitialRequestDelegate(isReader_InitialRequestEvent);
                isReader.OutGaugeEvent          += new InSimReader.OutGaugeDelegate(isReader_OutGaugeEvent);
                isReader.StateEvent             += new InSimReader.StateDelegate(isReader_StateEvent);
                isReader.PlayerJoinEvent        += new InSimReader.PlayerJoinDelegate(isReader_PlayerJoinEvent);
                isReader.MultiCarInfoEvent      += new InSimReader.MultiCarInfoDelegate(isReader_MultiCarInfoEvent);

                if (ogDirect == false && isRecvPort != ogRecvPort)
                {
                    ogReader                = new OutGaugeReader(ogRecvPort);
                    ogReader.OutGaugeEvent  += new OutGaugeReader.OutGaugeDelegate(ogReader_OutGaugeEvent);
                }

                //OutGauge gateway clients
                foreach(int port in lbOgPorts.Items)
                {
                    InSimWriter iw  = new InSimWriter("127.0.0.1", port);

                    ogClients.Add(port, iw);
                }
                
                InitLfsConnection();
                timerConnect.Enabled = true;

                if (lfsPath.Length > 0)
                {
                    string fp   = Path.GetDirectoryName(lfsPath);
                    colorLfs    = GetLfsClockColor(fp);
                    gfxOutput.Load(fp);

                    if (gfxOutput.LibraryLoaded)
                    {
                        gfxOutput.SetHwnd(this.Handle);
                        cbAutoClose.Enabled = true;

                        string col  = (customColor) ? colorCustom : colorLfs;
                        SetDigitsColor(col, customColor);                                                
                    }
                }

                pitlane = new Pitlane();
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        private void Stop()
        {
            try
            {
                Log.Trace("shutdown");

                if (lfsConnected)
                {
                    //stop outgauge
                    if (ogDirect)
                        RequestOutGauge(0);
                    
                    //close
                    IS_TINY msg = new IS_TINY(0, TINY.TINY_CLOSE);
                    isWriter.Send(msg);
                }

                SetConnected(false);

                currCar                 = string.Empty;
                timerConnect.Enabled    = false;                
                cbAutoClose.Enabled     = false;

                if (isReader != null)
                {
                    isReader.Close();
                    isReader.KeepAliveEvent         -= new InSimReader.KeepAliveDelegate(isReader_KeepAliveEvent);
                    isReader.InitialRequestEvent    -= new InSimReader.InitialRequestDelegate(isReader_InitialRequestEvent);
                    isReader.OutGaugeEvent          -= new InSimReader.OutGaugeDelegate(isReader_OutGaugeEvent);
                    isReader.StateEvent             -= new InSimReader.StateDelegate(isReader_StateEvent);
                    isReader.PlayerJoinEvent        -= new InSimReader.PlayerJoinDelegate(isReader_PlayerJoinEvent);
                    isReader.MultiCarInfoEvent      -= new InSimReader.MultiCarInfoDelegate(isReader_MultiCarInfoEvent);
                    isReader = null;
                }

                if (isWriter != null)
                {                 
                    isWriter.Close();
                    isWriter = null;
                }

                if (ogReader != null)
                {
                    ogReader.Close();                    
                    ogReader.OutGaugeEvent -= new OutGaugeReader.OutGaugeDelegate(ogReader_OutGaugeEvent);
                    ogReader = null;
                }

                foreach(InSimWriter iw in ogClients.Values)
                    iw.Close();

                ogClients.Clear();
                
                gfxOutput.Free();
            }            
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        private void ProgramPath_AddLfsOptions()
        {            
            if (tbProgramPath.Text.ToLower() == tbLfsPath.Text.ToLower())
            {
               if (cbWindowState.Items.Contains(WndState.Default) == false)
                   cbWindowState.Items.Insert(0, WndState.Default);

               if (cbWindowState.Items.Contains(WndState.Fullscreen) == false)
                   cbWindowState.Items.Add(WndState.Fullscreen);                            
            }
            else
            {
                cbWindowState.Items.Remove(WndState.Default);
                cbWindowState.Items.Remove(WndState.Fullscreen);
            }
        }

        private void StartPrograms()
        {            
            try
            {                               
                foreach(ListViewItem lvi in lvPrograms.Items)
                {
                    if (lvi.Checked == false)
                        continue;

                    //check if program isnt running already
                    string fn       = lvi.Text.Replace(Path.GetExtension(lvi.Text), string.Empty);
                    Process[] pa    = Process.GetProcessesByName(fn);                          
                    if(pa.Length > 0)
                    {
                        bool found = false;
                        foreach(Process pf in pa)
                        {
                            if (pf.MainModule.FileName.ToLower() == lvi.Name.ToLower())
                            {
                                found = true;
                                break;
                            }
                        }

                        if (found)
                            continue;
                    }

                    

                    Process p = new Process();
                    
                    p.StartInfo.FileName            = lvi.Name;
                    p.StartInfo.WorkingDirectory    = Path.GetDirectoryName(lvi.Name);
                    p.StartInfo.Arguments           = lvi.SubItems[1].Text; 
                    p.StartInfo.UseShellExecute     = true;
                    
                    string state    = lvi.SubItems[2].Text;                    
                    int msg         = 0;
                    bool lfs        = (p.StartInfo.FileName.ToLower() == tbLfsPath.Text.ToLower());                                        

                    switch(state)
                    {
                        case WndState.Minimized:
                            if (lfs == false)
                                p.StartInfo.WindowStyle = ProcessWindowStyle.Minimized;
                            else
                                ModifyLfsCfgFile(true);
                            msg = SC_MINIMIZE;
                            break;
                        case WndState.Maximized:
                            if (lfs == false)
                                p.StartInfo.WindowStyle = ProcessWindowStyle.Maximized;
                            else
                                ModifyLfsCfgFile(true);
                            msg = SC_MAXIMIZE;
                            break;
                        case WndState.Normal:
                            if (lfs)
                                ModifyLfsCfgFile(true);
                            break;
                        case WndState.Fullscreen:
                            if (lfs)
                                ModifyLfsCfgFile(false);
                            break;
                        default:
                            break;
                    }
                        
                    p.Start();  

                    if (msg != 0)
                    {
                        p.WaitForInputIdle(); 
                        PostMessage(p.MainWindowHandle, WM_SYSCOMMAND, msg, 0);                                    
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        private void ModifyLfsCfgFile(bool enable)
        {
            FileStream fs   = null;
            string fp       = Path.GetDirectoryName(lfsPath) + "\\cfg.txt";
     
            try
            {
                fs          = new FileStream(fp, FileMode.Open, FileAccess.ReadWrite, FileShare.None);

                int length  = (int)fs.Length;   
                byte[] buf  = new byte[length];           

                UTF8Encoding temp = new UTF8Encoding(true);
                int count = fs.Read(buf, 0, length);
                if (count > 0)
                {                    
                    string s    = temp.GetString(buf);
                    string st   = "Start Windowed";
                    int idx     = s.IndexOf(st) + st.Length + 1;
                    byte[] num  = new byte[1];
                    num[0]      = (byte)((enable) ? 0x31 : 0x30);
                    
                    fs.Seek(idx, SeekOrigin.Begin);
                    fs.Write(num, 0, 1);                    
                }
             }
             catch (Exception ex)
             {
                Log.Error(ex.ToString());
             }
             finally
             {         
                if (fs != null)
                    fs.Close();
             }
        }

        private void StopPrograms()
        {
            if (closePrograms == false)
                return;

            try
            {                               
                foreach(ListViewItem lvi in lvPrograms.Items)
                {
                    if (lvi.Checked == false)
                        continue;

                    string fn       = lvi.Text.Replace(Path.GetExtension(lvi.Text), string.Empty);
                    Process[] pa    = Process.GetProcessesByName(fn);                          
                    if(pa.Length > 0)
                    {
                        foreach(Process pf in pa)
                        {
                            if (pf.MainModule.FileName.ToLower() == lvi.Name.ToLower())
                            {
                                pf.CloseMainWindow();
                                break;
                            }
                        }
                    }                  
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
            
        }

        #endregion
    }  
}
