using System;
using System.Runtime.InteropServices;
using System.Drawing;

namespace InSim.Structs
{		
    public struct CompCar // Car info in 28 bytes - there is an array of these in the MCI (below)
    {
	    public ushort	Node;		// current path node
	    public ushort	Lap;		// current lap
	    public byte	    PLID;		// player's unique id
	    public byte	    Position;	// current race position : 0 = unknown, 1 = leader, etc...
	    public byte	    Info;		// flags and other info - see below
	    public byte	    Sp3;
	    public float	X;			// X map (65536 = 1 metre)
	    public float	Y;			// Y map (65536 = 1 metre)
	    public float	Z;			// Z alt (65536 = 1 metre)
	    public float	Speed;		// speed (32768 = 100 m/s)
	    public ushort	Direction;	// direction of car's motion : 0 = world y direction, 32768 = 180 deg
	    public ushort	Heading;	// direction of forward axis : 0 = world y direction, 32768 = 180 deg
	    public short	AngVel;		// signed, rate of change of heading : (16384 = 360 deg/s)

        public CompCar(byte[] data, ref int offset)
        {            
            Node        = BitConverter.ToUInt16(data, offset);
            offset      += 2;

            Lap         = BitConverter.ToUInt16(data, offset);
            offset      += 2;

            PLID        = data[offset];
            offset      += 1;

            Position    = data[offset];
            offset      += 1;

            Info        = data[offset];
            offset      += 1;

            Sp3         = data[offset];
            offset      += 1;

            X			= (float)(BitConverter.ToInt32(data, offset)) / 65536; //convert to meters
            offset      += 4;

            Y			= (float)(BitConverter.ToInt32(data, offset)) / 65536;
            offset      += 4;

            Z			= (float)(BitConverter.ToInt32(data, offset)) / 65536;
            offset      += 4;

            Speed       = (float)(BitConverter.ToUInt16(data, offset) / 327.68); //convert to m/s
            offset      += 2;

            Direction   = BitConverter.ToUInt16(data, offset);
            offset      += 2;

            Heading     = BitConverter.ToUInt16(data, offset);
            offset      += 2;

            AngVel      = BitConverter.ToInt16(data, offset);
            offset      += 2;
        }
    }

    public struct IS_BFN // Button FunctioN - delete buttons / receive button requests
    {
	    public byte	Size;		// 8
	    public byte	Type;		// ISP_BFN
	    public byte	ReqI;		// 0
	    public byte	SubT;		// subtype, from BFN_ enumeration (see below)

	    public byte	UCID;		// connection to send to or from (0 = local / 255 = all)
	    public byte	ClickID;	// ID of button to delete (if SubT is BFN_DEL_BTN)
	    public byte	Inst;		// used internally by InSim
	    public byte	Sp3;

        public IS_BFN(byte clickID, byte ucid, byte subT)
        {
            Size    = 8;
            Type    = (byte)Enums.ISP.ISP_BFN;
            ReqI    = 0;
            SubT    = subT;
            UCID    = ucid;
            ClickID = clickID;
            Inst    = 0;
            Sp3     = 0;
        }
    };

    public struct IS_BTN // ButtoN - button header - followed by 0 to 240 characters
    {
	    public byte	Size;		// 12 + TEXT_SIZE (a multiple of 4)
	    public byte	Type;		// ISP_BTN
	    public byte	ReqI;		// non-zero (returned in IS_BTC and IS_BTT packets)
	    public byte	UCID;		// connection to display the button (0 = local / 255 = all)

	    public byte	ClickID;	// button ID (0 to 239)
	    public byte	Inst;		// some extra flags - see below
	    public byte	BStyle;		// button style flags - see below
	    public byte	TypeIn;		// max chars to type in - see below

	    public byte	L;			// left   : 0 - 200
	    public byte	T;			// top    : 0 - 200
	    public byte	W;			// width  : 0 - 200
	    public byte	H;			// height : 0 - 200

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 240)]
    	public string Text; // 0 to 240 characters of text

        public IS_BTN(byte clickId, byte ucid, string msg, byte x, byte y, byte w, byte h)
        {
            if (msg.Length > 240)
                throw new Exception("IS_BTN: msg to long, 240 chars max");

            if (x > 200)
                throw new Exception("IS_BTN: x to big, 200 max");

            if (y > 200)
                throw new Exception("IS_BTN: y to big, 200 max");

            if (w > 200)
                throw new Exception("IS_BTN: w to big, 200 max");

            if (h > 200)
                throw new Exception("IS_BTN: h to big, 200 max");

            Size    = 252;
            Type    = (byte)Enums.ISP.ISP_BTN;
            ReqI    = 1;
            UCID    = ucid;

            ClickID = clickId;
            Inst    = 0;
            BStyle  = (byte)Enums.BTN_STYLE.ISB_LEFT;
            TypeIn  = 0;

            L       = x;
            T       = y;
            W       = w;
            H       = h;

            Text    = msg;
        }
    }

    public struct IS_CNL // ConN Leave
    {
	    public byte	Size;		// 8
	    public byte	Type;		// ISP_CNL
	    public byte	ReqI;		// 0
	    public byte	UCID;		// unique id of the connection which left

	    public byte	Reason;		// leave reason (see below)
	    public byte	Total;		// number of connections including host
	    public byte	Sp2;
	    public byte	Sp3;

        public IS_CNL(byte[] data)
        {      		
            int offset  = 0;

			Size		= data[offset];
  			offset 	   += 1;

            Type		= data[offset];
  			offset 	   += 1;

            ReqI		= data[offset];
  			offset 	   += 1;

            UCID		= data[offset];
  			offset 	   += 1;

            Reason		= data[offset];
  			offset 	   += 1;

            Total		= data[offset];
  			offset 	   += 1;

            Sp2 		= data[offset];
  			offset 	   += 1;

            Sp3 		= data[offset];
  			offset 	   += 1;
        }
    }

	public struct IS_ISI // InSim Init - packet to initialise the InSim system
  	{
    	public byte Size;		// 44
    	public byte Type;		// ISP_ISI
    	public byte ReqI;		// If non-zero LFS will send an IS_VER packet
    	public byte Zero;		// 0

    	public ushort UDPPort;	// Port for UDP replies from LFS (0 to 65535)
    	public ushort Flags;	// Bit flags for options (see below)

    	public byte Sp0;		// 0
    	public byte Prefix;		// Special host message prefix character
    	public ushort Interval;	// Time in ms between NLP or MCI (0 = none)

    	[MarshalAs(UnmanagedType.ByValTStr, SizeConst = 16)]
        public string Admin;                // Admin password (if set in LFS) 
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 16)]
        public string IName;                // A short name for your program

    	public IS_ISI(ushort updPort, string adminPass, string progName, ushort flags)
    	{
      		Size		= 44;
      		Type 		= (byte)Enums.ISP.ISP_ISI;
      		ReqI 		= 1;
      		Zero 		= 0;
      		UDPPort 	= updPort;
      		Flags 		= flags;
      		Sp0 		= 0;
      		Prefix 		= 0;
      		Interval 	= 1;
      		Admin 		= adminPass;
      		IName 		= progName;
    	}
	}    

    public struct IS_MCI // Multi Car Info - if more than 8 in race then more than one of these is sent
    {
	    public byte Size;		// 4 + NumP * 28
	    public byte Type;		// ISP_MCI
	    public byte ReqI;		// 0 unless this is a reply to an TINY_MCI request
	    public byte NumC;		// number of valid CompCar structs in this packet

	    public CompCar[] Info;	// car info for each player, 1 to 8 of these (NumC)

        public IS_MCI(byte[] data)
        {
            int offset      = 0;

            Size		= data[offset];
  			offset 	   += 1;

            Type		= data[offset];
  			offset 	   += 1;

            ReqI		= data[offset];
  			offset 	   += 1;

            NumC		= data[offset];
  			offset 	   += 1;

            Info        = new CompCar[8];

            for (int i=0; i<NumC; i++)
                Info[i] = new CompCar(data, ref offset);
        }
    }

    public struct IS_MSO // MSg Out - system messages and user messages 
    {
	    public byte	Size;		// 136
	    public byte	Type;		// ISP_MSO
	    public byte	ReqI;		// 0
	    public byte	Zero;

	    public byte	UCID;		// connection's unique id (0 = host)
	    public byte	PLID;		// player's unique id (if zero, use UCID)
	    public byte	UserType;	// set if typed by a user (see User Values below) 
	    public byte	TextStart;	// first character of the actual text (after player name)

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 128)]
	    public string	Msg;

        public IS_MSO(byte[] data)
        {
            System.Text.Encoding enc = System.Text.Encoding.ASCII;
      		
            int offset  = 0;

			Size		= data[offset];
  			offset 	   += 1;

            Type		= data[offset];
  			offset 	   += 1;

            ReqI		= data[offset];
  			offset 	   += 1;

            Zero		= data[offset];
  			offset 	   += 1;

            UCID		= data[offset];
  			offset 	   += 1;

            PLID		= data[offset];
  			offset 	   += 1;

            UserType	= data[offset];
  			offset 	   += 1;

            TextStart	= data[offset];
  			offset 	   += 1;

            Msg 	    = enc.GetString(data, offset, 128);
        }
    }

    public struct IS_MST // MSg Type - send to LFS to type message or command
    {
	    public byte	Size;		// 68
	    public byte	Type;		// ISP_MST
	    public byte	ReqI;		// 0
	    public byte	Zero;
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 64)]
	    public string Msg;  	// last byte must be zero

        public IS_MST(string msg)
		{
			Size 	= 68;
			Type	= (byte)Enums.ISP.ISP_MST;
			ReqI	= 0;
            Zero    = 0;
			Msg     = msg;
		}
    };

    public struct IS_NCN // New ConN
    {
	    public byte	Size;		// 56
	    public byte	Type;		// ISP_NCN
	    public byte	ReqI;		// 0 unless this is a reply to a TINY_NCN request
	    public byte	UCID;		// new connection's unique id (0 = host)

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 24)]
	    public string	UName;	// username
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 24)]
	    public string	PName;	// nickname

	    public byte	Admin;		// 1 if admin
	    public byte	Total;		// number of connections including host
	    public byte	Flags;		// bit 2 : remote
	    public byte	Sp3;

        public IS_NCN(byte[] data)
        {
            System.Text.Encoding enc = System.Text.Encoding.ASCII;      		
            int offset  = 0;

            Size		= data[offset];
  			offset 	   += 1;

            Type		= data[offset];
  			offset 	   += 1;

            ReqI		= data[offset];
  			offset 	   += 1;
 
            UCID        = data[offset];
            offset     += 1;

            UName 	    = enc.GetString(data, offset, 24);
      		offset 	   += 24;

            PName 	    = enc.GetString(data, offset, 24);
      		offset 	   += 24;

            Admin		= data[offset];
  			offset 	   += 1;

            Total		= data[offset];
  			offset 	   += 1;

            Flags		= data[offset];
  			offset 	   += 1;

            Sp3	    	= data[offset];
        }
    }

    public struct IS_NPL // New PLayer joining race (if PLID already exists, then leaving pits)
	{
        public byte	Size;		// 76
	    public byte	Type;		// ISP_NPL
	    public byte	ReqI;		// 0 unless this is a reply to an TINY_NPL request
	    public byte	PLID;		// player's newly assigned unique id

	    public byte	UCID;		// connection's unique id
	    public byte	PType;		// bit 0 : female / bit 1 : AI / bit 2 : remote
	    public ushort Flags;	// player flags

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 24)]
        public string PName;    //nickname
	    [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 8)]
        public string Plate;    // number plate - NO ZERO AT END!

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 4)]
        public string CName;    // car name
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 16)]
        public string SName;    // skin name - MAX_CAR_TEX_NAME

        public byte[] Tyres;    // compounds

	    public byte	H_Mass;		// added mass (kg)
	    public byte	H_TRes;		// intake restriction
	    public byte	Model;		// driver model
	    public byte	Pass;		// passengers byte

	    public int	Spare;

	    public byte	Sp0;
	    public byte	NumP;		// number in race (same when leaving pits, 1 more if new)
	    public byte	Sp2;
	    public byte	Sp3;
		
		public IS_NPL(byte[] data)
    	{
            System.Text.Encoding enc = System.Text.Encoding.ASCII;      		
            int offset  = 0;

			Size		= data[offset];
  			offset 	   += 1;

            Type		= data[offset];
  			offset 	   += 1;

            ReqI		= data[offset];
  			offset 	   += 1;

            PLID		= data[offset];
  			offset 	   += 1;

            UCID		= data[offset];
  			offset 	   += 1;

            PType		= data[offset];
  			offset 	   += 1;

            Flags	    = BitConverter.ToUInt16(data, offset);
            offset 	   += 2;
            
            PName 	    = enc.GetString(data, offset, 24);
      		offset 	   += 24;
	    
            Plate 	    = enc.GetString(data, offset, 8);
      		offset 	   += 8;

            CName 	    = enc.GetString(data, offset, 4);
      		offset 	   += 4;

            SName 	    = enc.GetString(data, offset, 16);
      		offset 	   += 16;

            Tyres       = new byte[4];
            Buffer.BlockCopy(data, offset, Tyres, 0, 4);
      		offset 	   += 4;

            H_Mass		= data[offset];
  			offset 	   += 1;

            H_TRes		= data[offset];
  			offset 	   += 1;

            Model		= data[offset];
  			offset 	   += 1;

            Pass		= data[offset];
  			offset 	   += 1;

            Spare	    = BitConverter.ToInt32(data, offset);
            offset 	   += 4;

            Sp0		    = data[offset];
  			offset 	   += 1;

            NumP		= data[offset];
  			offset 	   += 1;

            Sp2		    = data[offset];
  			offset 	   += 1;

            Sp3		    = data[offset];            
		}
    }

    public struct IS_PFL // Player FLags (help flags changed)
	{
        public byte	    Size;		// 4
	    public byte	    Type;		// ISP_PFL
	    public byte	    ReqI;		// 0
	    public byte	    PLID;		// player's unique id
        public ushort	Flags;		// player flags (see below)
	    public ushort	Spare;
		
		public IS_PFL(byte[] data)
    	{
            int offset  = 0;

			Size		= data[offset];
  			offset 	   += 1;

            Type		= data[offset];
  			offset 	   += 1;

            ReqI		= data[offset];
  			offset 	   += 1;

            PLID		= data[offset];
  			offset 	   += 1;

            Flags	    = BitConverter.ToUInt16(data, offset);
            offset 	   += 2;

            Spare	    = BitConverter.ToUInt16(data, offset);
            offset 	   += 2;
		}
    }

	public struct IS_PLA // Pit Lane
	{
		public byte Size;		// 8
		public byte Type;		// ISP_PLA
		public byte ReqI;		// 0
		public byte	PLID;		// player's unique id
		public byte	Fact;		// pit lane fact (see below)
		public byte	Sp1;
		public byte	Sp2;
		public byte	Sp3;
		
		public IS_PLA(byte[] data)
		{
			int offset	= 0;
			
			Size		= data[offset];
  			offset 	   += 1;
  			
  			Type		= data[offset];
  			offset 	   += 1;
  			
			ReqI		= data[offset];
  			offset 	   += 1;
			
  			PLID		= data[offset];
  			offset 	   += 1;
  			
			Fact		= data[offset];
  			offset 	   += 1;
  			
  			Sp1			= data[offset];
  			offset 	   += 1;
  			
  			Sp2			= data[offset];
  			offset 	   += 1;
  			
  			Sp3			= data[offset];
  			offset 	   += 1;  			
		}
	}

    public struct IS_PLL // PLayer Leave race (spectate - removed from player list)
	{
        public byte	Size;		// 4
	    public byte	Type;		// ISP_PLL
	    public byte	ReqI;		// 0
	    public byte	PLID;		// player's unique id
		
		public IS_PLL(byte[] data)
    	{
            int offset  = 0;

			Size		= data[offset];
  			offset 	   += 1;

            Type		= data[offset];
  			offset 	   += 1;

            ReqI		= data[offset];
  			offset 	   += 1;

            PLID		= data[offset];
  			offset 	   += 1;
		}
    }

    public struct IS_SMALL // General purpose 8 byte packet
	{
		public byte Size;		// always 8
		public byte Type;		// always ISP_SMALL
		public byte ReqI;		// 0 unless it is an info request or a reply to an info request
		public byte SubT;		// subtype, from SMALL_ enumeration (e.g. SMALL_SSP)
		public uint UVal;		// value (e.g. for SMALL_SSP this would be the OutSim packet rate)
		
		public IS_SMALL(byte reqI, Enums.SMALL subT, uint uVal)
		{
			Size	= 8;
			Type	= (byte)Enums.ISP.ISP_SMALL;
			ReqI	= reqI;
			SubT	= (byte)subT;
			UVal	= uVal;
		}
	}

    public struct IS_STA  // State
	{
		public byte		Size;			// 28
		public byte		Type;			// ISP_STA
		public byte		ReqI;			// ReqI if replying to a request packet
		public byte		Zero;

		public float	ReplaySpeed;	// 4-byte float - 1.0 is normal speed
		public ushort	Flags;			// ISS state flags (see below)
		public byte		InGameCam;		// Which type of camera is selected (see below)
		public byte		ViewPLID;		// Unique ID of viewed player (0 = none)

		public byte		NumP;			// Number of players in race
		public byte		NumConns;		// Number of connections including host
		public byte		NumFinished;	// Number finished or qualified
		public byte		RaceInProg;		// 0 - no race / 1 - race / 2 - qualifying

		public byte		QualMins;
		public byte		RaceLaps;		// see "RaceLaps" near the top of this document
		public byte		Spare2;
		public byte		Spare3;

		[MarshalAs(UnmanagedType.ByValTStr, SizeConst = 6)]
		public string	Track;	// short name for track e.g. FE2R
		public byte	Weather;		// 0,1,2...
		public byte	Wind;			// 0=off 1=weak 2=strong

		public IS_STA(byte[] data)
		{
			System.Text.Encoding enc = System.Text.Encoding.ASCII;
  			int offset	= 0;

  			Size		= data[offset];
  			offset 	   += 1;
  			
  			Type		= data[offset];
  			offset 	   += 1;
  			
			ReqI		= data[offset];
  			offset 	   += 1;
  			
  			Zero		= data[offset];
  			offset 	   += 1;
  			      			
  			ReplaySpeed	= BitConverter.ToSingle(data, offset);
  			offset 		+= 4;
  		
			Flags		= BitConverter.ToUInt16(data, offset);
	  		offset 		+= 2;
	
	  		InGameCam	= data[offset];
  			offset 	   += 1;
  			
  			ViewPLID	= data[offset];
  			offset 	   += 1;
  			
  			NumP		= data[offset];
  			offset 	   += 1;

  			NumConns	= data[offset];
  			offset 	   += 1;
  			
  			NumFinished	= data[offset];
  			offset 	   += 1;
  			
  			RaceInProg	= data[offset];
  			offset 	   += 1;
  			
  			QualMins	= data[offset];
  			offset 	   += 1;

  			RaceLaps	= data[offset];
  			offset 	   += 1;
  			
  			Spare2		= data[offset];
  			offset 	   += 1;
  			
  			Spare3		= data[offset];
  			offset 	   += 1;
  			
			Track 		= enc.GetString(data, offset, 6);
  			offset	   += 6;
		
  			Weather		= data[offset];
  			offset 	   += 1;
  			
  			Wind		= data[offset];
  			offset 	   += 1;
		}
	}

    public struct IS_TINY // General purpose 4 byte packet
	{
		public byte Size;		// always 4
		public byte Type;		// always ISP_TINY
		public byte ReqI;		// 0 unless it is an info request or a reply to an info request
		public byte SubT;		// subtype, from TINY_ enumeration (e.g. TINY_RACE_END)
		
		public IS_TINY(byte reqI, Enums.TINY subT)
		{
			Size 	= 4;
			Type	= (byte)Enums.ISP.ISP_TINY;
			ReqI	= reqI;
			SubT	= (byte)subT;
		}
		
		public IS_TINY(byte[] data)
    	{
      		int offset	= 0;
      		
      		Size		= data[offset];
      		offset 	   += 1;
      		
      		Type		= data[offset];
      		offset 	   += 1;

      		ReqI		= data[offset];
      		offset 	   += 1;

      		SubT		= data[offset];      		
		}
	}

    public struct IS_VER // VERsion
	{
		public byte	Size;			// 20
		public byte	Type;			// ISP_VERSION
		public byte	ReqI;			// ReqI as received in the request packet
		public byte	Zero;
		[MarshalAs(UnmanagedType.ByValTStr, SizeConst = 8)]
        public string Version;      // LFS version, e.g. 0.3G
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 6)]
        public string Product;      // Product : DEMO or S1
		public ushort InSimVer;		// InSim Version : increased when InSim packets change
		
		public IS_VER(byte[] data)
    	{
			System.Text.Encoding enc = System.Text.Encoding.ASCII;
      		int offset	= 0;
      		
      		Size		= data[offset];
      		offset 	   += 1;
      		
      		Type		= data[offset];
      		offset 	   += 1;
      		
      		ReqI		= data[offset];
      		offset 	   += 1;
      		
      		Zero		= data[offset];
      		offset 	   += 1;
      		
      		Version 	= enc.GetString(data, offset, 8);
      		offset	   += 8;
      		
      		Product 	= enc.GetString(data, offset, 6);
      		offset	   += 6;
      		
      		InSimVer	= BitConverter.ToUInt16(data, offset);
		}
	}

    public struct OutGaugePack
	{
		public uint 	Time;		// time in milliseconds (to check order)

		[MarshalAs(UnmanagedType.ByValTStr, SizeConst = 4)]
		public string Car;			// Car name
		public ushort Flags;			// OG_FLAGS (see below)
		public byte Gear;			// Reverse:0, Neutral:1, First:2...
		public byte SpareB;
		public float Speed;			// M/S
		public float RPM;			// RPM
		public float Turbo;			// BAR
		public float EngTemp;		// C
		public float Fuel;			// 0 to 1
		public float	OilPressure;		// BAR
		public float	OilTemp;
        public uint DashLights;
        public uint ShowLights;
		public float	Throttle;		// 0 to 1
		public float	Brake;			// 0 to 1
		public float	Clutch;			// 0 to 1
		[MarshalAs(UnmanagedType.ByValTStr, SizeConst = 16)]
		public string	Display1;	// Usually Fuel
		[MarshalAs(UnmanagedType.ByValTStr, SizeConst = 16)]
		public string 	Display2;	// Usually Settings
		
		public int		ID;				// optional - only if GameID is specified
		
		public OutGaugePack(byte[] data)
    	{
			System.Text.Encoding enc = System.Text.Encoding.ASCII;
      		int offset	= 0;
      		
      		Time 		= BitConverter.ToUInt32(data, offset);
      		offset 		+= 4;
      		      		
			Car 		= enc.GetString(data, offset, 4);
      		offset 		+= 4;

      		Flags		= BitConverter.ToUInt16(data, offset);
      		offset 		+= 2;
      		
      		Gear		= data[offset];
      		offset 		+= 1;
      		
      		SpareB		= data[offset];
      		offset 		+= 1;
      		
      		if ((Flags & (ushort)Enums.OG_FLAGS.OG_KM) != 0)
      			Speed = (int)(BitConverter.ToSingle(data, offset) * 3.6); //convert to kmh
      		else
      			Speed = (int)(BitConverter.ToSingle(data, offset) * 2.237); //convert to mph
      		offset 		+= 4;
      		
      		RPM			= (int)BitConverter.ToSingle(data, offset);
      		offset 		+= 4;
      		
      		Turbo		= BitConverter.ToSingle(data, offset);
      		offset 		+= 4;
      		
      		EngTemp		= BitConverter.ToSingle(data, offset);
      		offset 		+= 4;
      		
      		Fuel		= BitConverter.ToSingle(data, offset);
      		offset 		+= 4;
      		
      		OilPressure	= BitConverter.ToSingle(data, offset);
      		offset 		+= 4;

            OilTemp = BitConverter.ToSingle(data, offset);
      		offset 		+= 4;

            DashLights = BitConverter.ToUInt32(data, offset);
      		offset 		+= 4;

            ShowLights = BitConverter.ToUInt32(data, offset);
      		offset 		+= 4;
      		
      		Throttle	= BitConverter.ToSingle(data, offset);
      		offset 		+= 4;
      		
      		Brake		= BitConverter.ToSingle(data, offset);
      		offset 		+= 4;
      		
      		Clutch		= BitConverter.ToSingle(data, offset);
      		offset 		+= 4;
      		
      		Display1 	= enc.GetString(data, offset, 16);
      		offset 		+= 16;
      		
      		Display2 	= enc.GetString(data, offset, 16);
      		offset 		+= 16;
      		
      		if (data.Length == 96)
      			ID = BitConverter.ToInt32(data, offset);
      		else
      			ID = 0;
		}
		
	}    

    public struct OutSimPack
	{
		public uint 	Time;		// time in milliseconds (to check order)
		public Vector	AngVel;		// 3 floats, angular velocity vector
		public float	Heading;	// anticlockwise from above (Z)
        public float	Pitch;		// anticlockwise from right (X)
	    public float	Roll;		// anticlockwise from front (Y)
	    public Vector	Accel;		// 3 floats X, Y, Z
	    public Vector	Vel;		// 3 floats X, Y, Z
	    public Vec		Pos;		// 3 ints   X, Y, Z (1m = 65536)

	    public int		ID;			// optional - only if GameID is specified
		
		public OutSimPack(byte[] data)
    	{
      		int offset	= 0;
      		
      		Time 		= BitConverter.ToUInt32(data, offset);
      		offset 	   += 4;
      		
      		AngVel		= new Vector(data, ref offset);
      		
      		Heading		= BitConverter.ToSingle(data, offset);
      		offset	   += 4;
      		
            Pitch		= BitConverter.ToSingle(data, offset);
      		offset	   += 4;
      		
            Roll		= BitConverter.ToSingle(data, offset);
      		offset	   += 4;

            Accel		= new Vector(data, ref offset);
            Vel 		= new Vector(data, ref offset);
            Pos 		= new Vec(data, ref offset);

            if (data.Length == 68)
      			ID = BitConverter.ToInt32(data, offset);
      		else
      			ID = 0;
		}
	}

    public struct Vec
	{
		public float X;
		public float Y;
		public float Z;
		
		public Vec(byte[] data, ref int offset)
		{
			X			= (float)(BitConverter.ToInt32(data, offset)) / 65536; //convert to meters
  			offset 		+= 4;
  			
  			Y			= (float)(BitConverter.ToInt32(data, offset)) / 65536;
  			offset 		+= 4;
  			
  			Z			= (float)(BitConverter.ToInt32(data, offset)) / 65536;
  			offset 		+= 4;
		}
	}

    public struct Vector
	{
		public float X;
		public float Y;
		public float Z;
		
		public Vector(byte[] data, ref int offset)
		{
			X			= BitConverter.ToSingle(data, offset);
  			offset 		+= 4;
  			
  			Y			= BitConverter.ToSingle(data, offset);
  			offset 		+= 4;
  			
  			Z			= BitConverter.ToSingle(data, offset);
  			offset 		+= 4;
		}
	}





}


