using System;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using Tools;
using System.Text;

namespace InSim
{	
	public class StateObject 
	{
		// Client socket.
		public Socket workSocket = null;
		// Size of receive buffer.
		public const int BufferSize = 32768;
		// Receive buffer.
		public byte[] buffer = new byte[BufferSize];
		// Received data string.
		public StringBuilder sb = new StringBuilder();
	}

	public class InSimReader
    {
        #region variables 

        private byte[] 		data 		= new byte[512];
		private int			dataSize;
		private Socket		sock;
		private int			recvPort;
		private bool		connected	= false;
		private DateTime	lastPacket	= DateTime.Now;

        #region delegates

        public  delegate void			ConnJoinDelegate(Structs.IS_NCN packet);
		public  event ConnJoinDelegate	ConnJoinEvent;

        public  delegate void			ConnLeaveDelegate(Structs.IS_CNL packet);
		public  event ConnLeaveDelegate	ConnLeaveEvent;

        public  delegate void			InitialRequestDelegate();
		public  event InitialRequestDelegate	InitialRequestEvent;

        public  delegate void			KeepAliveDelegate(Structs.IS_TINY packet);
		public  event KeepAliveDelegate	KeepAliveEvent;

        public  delegate void			MsgOutDelegate(Structs.IS_MSO packet);
		public  event MsgOutDelegate	MsgOutEvent;

        public  delegate void			MultiCarInfoDelegate(Structs.IS_MCI packet);
		public  event MultiCarInfoDelegate	MultiCarInfoEvent;

		public  delegate void			OutGaugeDelegate(Structs.OutGaugePack packet);
		public  event OutGaugeDelegate	OutGaugeEvent;
		
        public  delegate void			OutSimDelegate(Structs.OutSimPack packet);
		public  event OutSimDelegate	OutSimEvent;

        public  delegate void			PitlaneDelegate(Structs.IS_PLA packet);
		public  event PitlaneDelegate	PitlaneEvent;
		
        public  delegate void			PlayerFlagsDelegate(Structs.IS_PFL packet);
		public  event PlayerFlagsDelegate	PlayerFlagsEvent;

        public  delegate void			PlayerJoinDelegate(Structs.IS_NPL packet);
		public  event PlayerJoinDelegate	PlayerJoinEvent;

		public  delegate void			PlayerLeaveDelegate(Structs.IS_PLL packet);
		public  event PlayerLeaveDelegate	PlayerLeaveEvent;

		public  delegate void			StateDelegate(Structs.IS_STA packet);
		public  event StateDelegate		StateEvent;        

        #endregion

        #endregion

        public InSimReader(int port)
		{
			recvPort = port;
			
			try
			{	
				sock = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
    			IPEndPoint end = new IPEndPoint(IPAddress.Any, recvPort);
    			sock.Bind(end);
    			
    			StartReceive();
    			connected = true;
    			
    			Log.Trace(String.Format("InsimReader: {0} - OK", recvPort));
			}
			catch (Exception ex)
			{
				Log.Error(ex.ToString());
			}
		}
		
		public void Close()
		{
			try
			{
				connected = false;
				
				sock.Shutdown(SocketShutdown.Both);
				sock.Close();
			}
			catch(Exception ex)
			{
				Log.Error(ex.ToString());
			}
		}
		
		private void StartReceive()
		{
			try
			{
    			sock.BeginReceive(data, 0, data.Length, SocketFlags.None, new AsyncCallback(ReceiveCallback), data);
			}
			catch(Exception ex)
			{
				Log.Error(ex.ToString());
			}
		}
		
		private void ReceiveCallback(IAsyncResult ar)
		{
			try
			{
				if (!connected)
					return;
			
				dataSize = sock.EndReceive(ar);				
          		if (dataSize > 0)
    			{
    				HandlePacket();
        			StartReceive();
    			}
			}
			catch(SocketException ex)
			{
				Log.Error(ex.ToString());
			}
			catch(Exception ex)
			{
				Log.Error(ex.ToString());
			}
		}
		
		private void HandlePacket()
		{
			lastPacket = DateTime.Now;
						
			if (dataSize == 92 || dataSize == 96) //probably OutGauge packet
			{				
				if (OutGaugeEvent != null)
                {
                    Structs.OutGaugePack packet = new Structs.OutGaugePack(data);				
					OutGaugeEvent(packet);
                }
			}
			else if (dataSize == 64 || dataSize == 68) //probably OutSim packet
			{
                if (OutSimEvent != null)
                {
				    Structs.OutSimPack packet = new Structs.OutSimPack(data);				
					OutSimEvent(packet);
                }
			}
			else
			{
				Enums.ISP type = (Enums.ISP)data[1];	
				//Log.Trace("{0} - {1} bytes", type.ToString(), dataSize);
				switch (type)
				{	
                    case Enums.ISP.ISP_CNL: // ConN Leave
					{																
						if (ConnLeaveEvent != null)
                        {
                            Structs.IS_CNL packet = new Structs.IS_CNL(data);						
							ConnLeaveEvent(packet);
                        }
						break;
					}
                    case Enums.ISP.ISP_MCI: // Multi Car Info
					{	
						if (MultiCarInfoEvent != null)
                        {
						    Structs.IS_MCI packet = new Structs.IS_MCI(data);						
							MultiCarInfoEvent(packet);
                        }						
						break;
					}
                    case Enums.ISP.ISP_MSO: // MSg Out - system messages and user messages 
					{	
						if (MsgOutEvent != null)
                        {
						    Structs.IS_MSO packet = new Structs.IS_MSO(data);						
							MsgOutEvent(packet);
                        }						
						break;
					}
                    case Enums.ISP.ISP_NCN: // New ConN
					{																
						if (ConnJoinEvent != null)
                        {
                            Structs.IS_NCN packet = new Structs.IS_NCN(data);						
							ConnJoinEvent(packet);
                        }
						break;
					}
					case Enums.ISP.ISP_NPL: // New PLayer joining race (if PLID already exists, then leaving pits)
					{																
						if (PlayerJoinEvent != null)
                        {
                            Structs.IS_NPL packet = new Structs.IS_NPL(data);						
							PlayerJoinEvent(packet);
                        }
						break;
					}				
                    case Enums.ISP.ISP_PLA: // Pit Lane
					{                        
						if (PitlaneEvent != null)
                        {
                            Structs.IS_PLA  packet = new InSim.Structs.IS_PLA(data);												
							PitlaneEvent(packet);
                        }
						break;
					}
                    case Enums.ISP.ISP_PFL: // Player FLags (help flags changed)
					{	
						if (PlayerFlagsEvent != null)
                        {
						    Structs.IS_PFL packet = new Structs.IS_PFL(data);												
							PlayerFlagsEvent(packet);
                        }
						
						break;
					}
                    case Enums.ISP.ISP_PLL: // PLayer Leave race (spectate - removed from player list)
					{	
						if (PlayerLeaveEvent != null)
                        {
						    Structs.IS_PLL packet = new Structs.IS_PLL(data);						
							PlayerLeaveEvent(packet);
                        }						
						break;
					}                    
                    case Enums.ISP.ISP_SMALL: // General purpose 8 byte packet
					{
						break;
					}
                    case Enums.ISP.ISP_STA: // State
					{	
						if (StateEvent != null)
                        {
						    Structs.IS_STA packet = new Structs.IS_STA(data);						
							StateEvent(packet);
                        }						
						break;
					}
                    case Enums.ISP.ISP_TINY: // General purpose 4 byte packet
					{
						Structs.IS_TINY packet = new Structs.IS_TINY(data);
						
						switch((Enums.TINY)packet.SubT)
						{
							case Enums.TINY.TINY_NONE:
								if (KeepAliveEvent != null)
									KeepAliveEvent(packet);
								break;
							default:
								break;
						}
						break;
					}
                    case Enums.ISP.ISP_VER: // VERsion
					{
                        if (InitialRequestEvent != null)
                        {
						    Structs.IS_VER packet 	= new Structs.IS_VER(data);												
							InitialRequestEvent();
                        }						
						break;
					}
					default:						    										
						break;
				}
			}
		}
				
		public DateTime LastPacket
		{
			get
			{
				return lastPacket;
			}
		}

        public bool Connected
        {
            get
            {
                return connected;
            }
        }
	}
	
	public class InSimWriter
	{
		private bool connected		= false;
		private UdpClient client 	= null;
		
		public InSimWriter(string host, int port)
		{
			try
			{
				client 		= new UdpClient(host, port);
				connected	= true;
				
				Log.Trace(String.Format("InsimWriter: {0} - OK", port));
			}
			catch (SocketException se)
			{
				Log.Error(se.ToString());
			}				
		}
		
		public void Close()
		{
			try
			{
				connected	= false;
				client.Close();
			}
			catch(Exception ex)
			{
				Log.Error(ex.ToString());
			}
		}
		
		public void Send(object packet)
		{
			try
			{
				if (!connected)
					return;
				
				byte[] data = DataConverter.PacketToData(packet);
				client.Send(data, data.Length);
			}
			catch(Exception ex)
			{
				Log.Error(ex.ToString());
			}
		}		

        public bool Connected
        {
            get
            {
                return connected;
            }
        }
	}

}
