﻿/*
  Copyright (C) 2010 Mikael Forsberg <mikael@liveforspeed.se>
  
  This source file is subject to the MIT license.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using Spark;
using Spark.Packets;

namespace Sparktris
{
    class Game
    {
        public delegate void EndingHandler();
        public event EndingHandler ending;

        public Game(string ipAddress, int port, string password)
        {
            this.ipAddress = ipAddress;
            this.port      = port;
            this.password  = password;

            buttonFactory = new ButtonFactory(insim);
        }

        #region Button Callbacks
        public void start_clicked(ClickableButton b)
        {
            b.hide();
            gameStarted = true;
            gameThread.Interrupt();
        }

        public void left_clicked(ClickableButton b)
        {
            lock (f)
            {
                if (f.shapeLeft())
                {
                    f.draw();
                }
            }
        }

        public void down_clicked(ClickableButton b)
        {
            lock (f)
            {
                if (true == f.shapeDown())
                {
                    gameThread.Interrupt();
                    f.draw();
                }
            }
        }

        public void right_clicked(ClickableButton b)
        {
            lock (f)
            {
                if (f.shapeRight())
                {
                    f.draw();
                }
            }
        }

        public void rotate_clicked(ClickableButton b)
        {
            lock (f)
            {
                if (f.shapeRotate())
                {
                    f.draw();
                }
            }
        }
        #endregion

        public void run()
        {
            gameThread = Thread.CurrentThread;

            try
            {
                insim.Connect(ipAddress, port);
            }
            catch (Exception)
            {
                System.Windows.Forms.MessageBox.Show("Connection failed", "Sparktris");

                if (null != ending)
                {
                    ending();
                }

                return;
            }

            insim.Send(new IS_ISI { Admin = password, IName = "Sparktris" });

            createButtons();

            buttons.showGroup("background");
            buttons.showGroup("logo");
            buttons.showGroup("start");

            try
            {
                while (true)
                {
                    while (false == gameStarted)
                    {
                        try
                        {
                            Thread.Sleep(10000);
                        }
                        catch (ThreadInterruptedException) { }
                    }

                    if (f != null)
                    {
                        f.destroy();
                        buttons.hideGroup("gameover");
                    }

                    play();

                    gameStarted = false;
                }
            }
            catch (ThreadAbortException)
            {
                if (f != null)
                {
                    f.destroy();
                }

                buttons.destroyAll();

                try
                {
                    insim.Disconnect();
                }
                catch (Exception) { }
            }
        }

        public void play()
        {
            buttons.hideGroup("logo");
            buttons.showGroup("score");
            buttons.showGroup("controls");

            score = 0;

            scoreButton.setText("0");
            scoreButton.show();

            f = new Field(12, 9, buttonFactory, new ShapeFactory());

            f.setSize(72, 132);
            f.setPosition(64, 38);
            f.nextShape();

            int completedRows;

            while (true)
            {
                try
                {
                    Thread.Sleep(900);

                    lock(f)
                    {
                        if (false == f.shapeDown())
                        {
                            f.commitShape();
                            score += 1;

                            if (0 < (completedRows = f.removeCompletedRows()))
                            {
                                 score += (int) Math.Pow((double)completedRows + 1, (double) 3);
                            }

                            scoreButton.setText(score.ToString());
                            scoreButton.show();

                            f.nextShape();

                            if (!f.shapeHasLegalPosition())
                            {
                                f.draw();
                                break;
                            }
                        }

                        f.draw();
                    }
                }
                catch (ThreadInterruptedException)
                {
                }
            }

            buttons.hideGroup("controls");
            buttons.showGroup("gameover");
            buttons.showGroup("start");
        }

        private void createButtons()
        {
            ClickableButton start   = (ClickableButton) buttons.Add("start", buttonFactory.createClickableButton());
            
            Button leftBackground   = buttons.Add("controls", buttonFactory.createButton());
            Button downBackground   = buttons.Add("controls", buttonFactory.createButton());
            Button rightBackground  = buttons.Add("controls", buttonFactory.createButton());
            Button rotateBackground = buttons.Add("controls", buttonFactory.createButton());

            ClickableButton left    = (ClickableButton) buttons.Add("controls", buttonFactory.createClickableButton());
            ClickableButton down    = (ClickableButton) buttons.Add("controls", buttonFactory.createClickableButton());
            ClickableButton right   = (ClickableButton) buttons.Add("controls", buttonFactory.createClickableButton());
            ClickableButton rotate  = (ClickableButton) buttons.Add("controls", buttonFactory.createClickableButton());

            Button scoreBackground  = buttons.Add("score", buttonFactory.createButton());
            Button score            = buttons.Add("score", buttonFactory.createButton());
            Button gameOver         = buttons.Add("gameover", buttonFactory.createButton());

            start.setText("Play!");
            start.setPosition(84, 100);
            start.setSize(32, 10);
            start.setStyle(ButtonStyles.ISB_C1 | ButtonStyles.ISB_DARK);
            start.clicked += new ClickableButton.ClickHandler(start_clicked);

            leftBackground.setStyle(ButtonStyles.ISB_DARK);
            leftBackground.setPosition(76, 172);
            leftBackground.setSize(16, 10);

            downBackground.setStyle(ButtonStyles.ISB_DARK);
            downBackground.setPosition(92, 182);
            downBackground.setSize(16, 10);

            rightBackground.setStyle(ButtonStyles.ISB_DARK);
            rightBackground.setPosition(108, 172);
            rightBackground.setSize(16, 10);

            rotateBackground.setStyle(ButtonStyles.ISB_DARK);
            rotateBackground.setPosition(92, 172);
            rotateBackground.setSize(16, 10);

            left.setPosition(77, 173);
            left.setSize(14, 8);
            left.setText("<");
            left.setStyle(ButtonStyles.ISB_C1 | ButtonStyles.ISB_LIGHT);
            left.clicked += new ClickableButton.ClickHandler(left_clicked);

            down.setPosition(93, 183);
            down.setSize(14, 8);
            down.setText("V");
            down.setStyle(ButtonStyles.ISB_C1 | ButtonStyles.ISB_LIGHT);
            down.clicked += new ClickableButton.ClickHandler(down_clicked);

            right.setPosition(109, 173);
            right.setSize(14, 8);
            right.setText(">");
            right.setStyle(ButtonStyles.ISB_C1 | ButtonStyles.ISB_LIGHT);
            right.clicked += new ClickableButton.ClickHandler(right_clicked);

            rotate.setPosition(93, 173);
            rotate.setSize(14, 8);
            rotate.setText("@");
            rotate.setStyle(ButtonStyles.ISB_C1 | ButtonStyles.ISB_LIGHT);
            rotate.clicked += new ClickableButton.ClickHandler(rotate_clicked);

            scoreBackground.setPosition(62, 15);
            scoreBackground.setSize(76, 22);
            scoreBackground.setStyle(ButtonStyles.ISB_DARK);

            score.setPosition(64, 17);
            score.setSize(72, 18);
            score.setStyle(ButtonStyles.ISB_LIGHT | ButtonStyles.ISB_C1);

            scoreButton = score;

            gameOver.setText("Game Over");
            gameOver.setStyle(ButtonStyles.ISB_DARK | ButtonStyles.ISB_C1 | ButtonStyles.ISB_C4);
            gameOver.setPosition(84, 80);
            gameOver.setSize(32, 10);

            createBackground(63, 16, 74, 155);
            createLogo(60, 16, 78, 18);
        }

        private void createBackground(int x, int y, int width, int height)
        {
            Button b;

            b = buttons.Add("background", buttonFactory.createButton());

            b.setPosition(x - 1, y - 1);
            b.setSize(width + 2, height + 2);
            b.setStyle(ButtonStyles.ISB_DARK);

            for (int i = 0; i < 4; ++i)
            {
                b = buttons.Add("background", buttonFactory.createButton());

                b.setPosition(x, y);
                b.setSize(width, height);
                b.setStyle(ButtonStyles.ISB_LIGHT);
            }
        }

        private void createLogo(int x, int y, int width, int height)
        {
            Button b;

            b = buttons.Add("logo", buttonFactory.createButton());

            b.setPosition(62, 15);
            b.setSize(76, 22);
            b.setStyle(ButtonStyles.ISB_DARK);

            b = buttons.Add("logo", buttonFactory.createButton());

            b.setPosition(64, 17);
            b.setSize(72, 18);
            b.setStyle(ButtonStyles.ISB_LIGHT | ButtonStyles.ISB_C1);

            char[] chars = "SPARKTRIS".ToCharArray();

            int charWidth = width / chars.Length;
            int nudge = 0;

            for (int i = 0; i < chars.Length; ++i)
            {
                nudge = (i + 1) % 2;

                b = buttons.Add("logo", buttonFactory.createButton());

                b.setPosition(x + (i * charWidth), y + nudge);
                b.setSize(charWidth * 2, height);
                b.setStyle(ButtonStyles.ISB_C2);
                b.setText(chars[i].ToString());

                b = buttons.Add("logo", buttonFactory.createButton());

                b.setPosition(x + (i * charWidth), y + nudge);
                b.setSize(charWidth * 2, height);
                b.setStyle(ButtonStyles.ISB_C1);
                b.setText(chars[i].ToString());

                b.move(-1, -1);
            }
        }

        private InSim insim = new InSim();
        private string ipAddress;
        private int port;
        private string password;

        private ButtonFactory buttonFactory;
        private ButtonRegistry buttons = new ButtonRegistry();

        private Field f;

        private int score = 0;
        private Button scoreButton;

        private Thread gameThread;
        private bool gameStarted = false;
    }
}
