﻿/*
  Copyright (C) 2010 Mikael Forsberg <mikael@liveforspeed.se>
  
  This source file is subject to the MIT license.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Spark;
using Spark.Packets;

namespace Sparktris
{
    class Field
    {
        public Field(int rows, int columns, ButtonFactory buttonFactory, ShapeFactory shapeFactory)
        {
            this.rows         = rows;
            this.columns      = columns;
            this.shapeFactory = shapeFactory;

            for (int i = 0; i < columns; ++i)
            {
                completedRow |= 1 << i;
            }

            for (int i = 0; i < rows; ++i)
            {
                field.Add(0);

                for (int j = 0; j < columns; ++j)
                {
                    buttons.Add(buttonFactory.createButton());
                }
            }
        }

        public void setSize(int width, int height)
        {
            this.width  = width;
            this.height = height;

            foreach (Button b in buttons)
            {
                b.setSize(width / columns, height / rows);
            }
        }

        public void setPosition(int x, int y)
        {
            this.x = x;
            this.y = y;

            int n = 0;

            for (int i = 0; i < rows; ++i)
            {
                for (int j = 0; j < columns; ++j)
                {
                    buttons[n++].setPosition(x + ((width / columns) * j), y + ((height / rows) * i));
                }
            }
        }

        public void nextShape()
        {
            shape = shapeFactory.getShape();
            shape.setPosition((columns / 2) - (shape.width / 2), shape.y);

            if (0 != (columns % 2) && 0 == (shape.width % 2))
            {
                shape.move(1, 0);
            }
        }

        private bool shapeMove(int x, int y)
        {
            shape.move(x, y);

            if (shapeHasLegalPosition())
            {
                return true;
            }
            else
            {
                shape.move(-x, -y);
                return false;
            }
        }

        public bool shapeDown()
        {
            return shapeMove(0, 1);
        }

        public bool shapeLeft()
        {
            return shapeMove(-1, 0);
        }

        public bool shapeRight()
        {
            return shapeMove(1, 0);
        }

        public bool shapeRotate()
        {
            shape.rotate();

            if (shapeHasLegalPosition())
            {
                return true;
            }
            else
            {
                shape.rotate(-1);
                return false;
            }
        }

        public bool shapeHasLegalPosition()
        {
            int shapeBits;

            for (int i = 0; i < shape.height; ++i)
            {
                for (int j = 0; j < shape.width; ++j)
                {
                    // there is a pixel here and it's outside the screen
                    if (0 < (shape.shape[i] & (1 << j)) && (0 > shape.x + j || columns <= shape.x + j || rows <= shape.y + i))
                    {
                        return false;
                    }
                }

                if (shape.y + i >= field.Count)
                    break;

                shapeBits = shape.x >= 0 ? (shape.shape[i] << shape.x) : (shape.shape[i] >> Math.Abs(shape.x));

                // one or more pixels of the shape are intersecting with pixels on the field
                if (0 < (field[shape.y + i] & shapeBits))
                {
                    return false;
                }
            }

            return true;
        }

        public void commitShape()
        {
            for (int i = 0; i < shape.height; ++i)
            {
                if (shape.y + i < field.Count)
                {
                    field[shape.y + i] |= shape.x >= 0 ? (shape.shape[i] << shape.x) : (shape.shape[i] >> Math.Abs(shape.x));
                }
            }
        }

        public int removeCompletedRows()
        {
            int numCompletedRows = 0;

            for (int i = 0; i < rows; ++i)
            {
                if (field[i] == completedRow)
                {
                    ++numCompletedRows;

                    for (int p = 0; p < columns; ++p)
                    {
                        buttons[(i * columns) + p].hide();
                        System.Threading.Thread.Sleep(20);
                    }

                    for (int j = i; j > 0; --j)
                    {
                        field[j] = field[j - 1];
                    }
                }
            }

            return numCompletedRows;
        }

        public void draw()
        {
            int n = 0;

            for (int i = 0; i < rows; ++i)
            {
                for (int j = 0; j < columns; ++j)
                {
                    buttons[n].setStyle(0 == (field[i] & (1 << j)) ? (ButtonStyles) 0 : ButtonStyles.ISB_DARK);

                    if (shape.y <= i && (shape.y + (shape.height - 1)) >= i)
                    {
                        buttons[n].setStyle(0 == (shape.shape[i - shape.y] & (1 << (j - shape.x))) ? buttons[n].getStyle() : ButtonStyles.ISB_DARK);
                    }

                    buttons[n++].show();
                }
            }
        }

        public void destroy()
        {
            foreach (Button b in buttons)
            {
                b.hide(true);
            }
        }

        private int rows, columns, width, height, x, y, completedRow;
        private ShapeFactory shapeFactory;
        private Shape shape;

        private List<int> field      = new List<int>();
        private List<Button> buttons = new List<Button>();
    }
}
