#include "ctrack.h"
#include <wx/tokenzr.h>

//-----------------------------------------------------------------------------
// Static members

// array to map display name(s) to a track code
// syntax: <primary_name> ':' [<alternate_name> ':']* <code>

wxString cTrack::s_DefaultTrackCodes[] = {
 _T("GP Track:BL1"),
 _T("Rallycross:BL2"),
 _T("Car Park:BL3"),
 _T("Classic:SO1"),
 _T("Sprint 1:Sprint Track 1:SO2"),
 _T("Sprint 2:Sprint Track 2:SO3"),
 _T("City Long:SO4"),
 _T("Town Course:SO5"),
 _T("Chicane Route:SO6"),
 _T("Club:FE1"),
 _T("Green:Green Track:FE2"),
 _T("Gold:Gold Track:FE3"),
 _T("Black:Black Track:FE4"),
 _T("Rallycross:FE5"),
 _T("RallyX Green:FE6"),
 _T("Autocross:AU1"),
 _T("Skid Pad:AU2"),
 _T("Drag Strip:AU3"),
 _T("Eight Lane Drag:AU4"),
 _T("Oval:KY1"),
 _T("National:KY2"),
 _T("GP Long:KY3"),
 _T("International:WE1"),
 _T("Cadet:AS1"),
 _T("Club:AS2"),
 _T("National:AS3"),
 _T("Historic:AS4"),
 _T("Grand Prix:AS5"),
 _T("Grand Touring:AS6"),
 _T("North:AS7")
};

wxString cTrack::s_DefaultVenueCodes[] = {
 _T("BL:Blackwood:0"),
 _T("SO:South City:1"),
 _T("FE:Fern Bay:2"),
 _T("AU:Autocross:3"),
 _T("KY:Kyoto Ring:4"),
 _T("WE:Westhill:5"),
 _T("AS:Aston:6")
};

wxArrayString cTrack::s_TrackCodes;
wxArrayString cTrack::s_VenueCodes;

//-----------------------------------------------------------------------------

cTrack::cTrack()
{
}

//-----------------------------------------------------------------------------

cTrack::~cTrack()
{
}

//-----------------------------------------------------------------------------
// Initialise static data

void cTrack::InitTracks(const wxString& configFile)
{
  wxASSERT(s_TrackCodes.IsEmpty());

  if (!::ReadConfigFile(configFile, s_TrackCodes)) {
    // load from file failed; read hard-coded data
    for (int i = 0; i < sizeof(s_DefaultTrackCodes) / sizeof(wxString); i++) {
      s_TrackCodes.Add(s_DefaultTrackCodes[i]);
    }
  }
}

void cTrack::InitVenues(const wxString& configFile)
{
  wxASSERT(s_VenueCodes.IsEmpty());

  if (!::ReadConfigFile(configFile, s_VenueCodes)) {
    // load from file failed; read hard-coded data
    for (int i = 0; i < sizeof(s_DefaultVenueCodes) / sizeof(wxString); i++) {
      s_VenueCodes.Add(s_DefaultVenueCodes[i]);
    }
  }
}

//-----------------------------------------------------------------------------
// Convert the full track name (venue+config) to the track code
// - name = venue name
// - config = configuration name
// Returns an empty string if not found

wxString cTrack::Name2Code(wxString& name, wxString& config)
{
  wxString abbrev = Name2Venue(name); // 2-letter venue abbreviation

  // remove "reverse" indicator from config name
  bool reverse = false;
  wxString cfg = config;
  if (cfg.Right(4).CmpNoCase(_T("_REV")) == 0) {
    reverse = true;
    cfg = cfg.Left(config.Length() - 4);
  }
  // only compare on the first 11 chars, because the max. length is 15, and the last 4 may be overwritten with '_REV'
  cfg = cfg.Left(11);

  // search the list
  for (size_t i = 0; i < s_TrackCodes.GetCount(); i++) {
    wxArrayString names = wxStringTokenize(s_TrackCodes[i], _T(":"));
    wxASSERT(names.GetCount() >= 2);
    wxString code = names[names.GetCount() - 1];

    if (code.Left(2) != abbrev) continue; // different venue abbreviation

    for (size_t j = 0; j < names.GetCount() - 1; j++) {
      if (names[j].Left(11).CmpNoCase(cfg) != 0) continue; // different name

      // found it
      if (reverse) code += _T("R");

      // canonicalize config name
      config = names[j];
      if (reverse) config += _T(" (reverse)");

      return code;
    }
  }

  // not found
  wxFAIL;
  return wxEmptyString;
}

//-----------------------------------------------------------------------------
// Convert the track/venue name to the 2-character venue code
// - name = track name

wxString cTrack::Name2Venue(const wxString& name)
{
  // search the list
  for (size_t i = 0; i < s_VenueCodes.GetCount(); i++) {
    wxString venue = s_VenueCodes[i].AfterFirst(':').BeforeFirst(':');
    if (venue.CmpNoCase(name) != 0) continue; // different name
    return s_VenueCodes[i].BeforeFirst(':');
  }

  // not found
  wxFAIL;
  return name.Left(2).Upper(); // last resort: use first 2 characters
}

//-----------------------------------------------------------------------------
// Get the list of codes
// - codes = list that will be filled with the codes

void cTrack::GetCodes(wxArrayString& codes)
{
  codes.Empty();
  for (size_t i = 0; i < s_TrackCodes.GetCount(); i++) {
    codes.Add(s_TrackCodes[i].AfterLast(':'));
  }
  codes.Sort();
}

//-----------------------------------------------------------------------------
// Get the list of names (<venue code> - <primary_config_name>)
// - names = list that will be filled with the names

void cTrack::GetNames(wxArrayString& names)
{
  names.Empty();
  for (size_t i = 0; i < s_TrackCodes.GetCount(); i++) {
    names.Add(s_TrackCodes[i].AfterLast(':') + _T(" - ") + s_TrackCodes[i].BeforeFirst(':'));
  }
  names.Sort();
}

//-----------------------------------------------------------------------------
// Convert the track code to the 3-digit LFS World track number
// - code = track code
// Returns an empty string if not found

wxString cTrack::Code2Lfsw(const wxString& code)
{
  wxString result;

  // search the list
  for (size_t i = 0; i < s_VenueCodes.GetCount(); i++) {
    wxString venue = s_VenueCodes[i].BeforeFirst(':'); // 2-character venue code
    wxString result = s_VenueCodes[i].AfterLast(':'); // first digit: LFSW venue number
    if (code.Left(2) != venue) continue;

    // found the venue
    wxASSERT(result.Len() == 1);

    // second digit = number of the track's config, minus 1 (!)
    long config = 0;
    code.Mid(2).ToLong(&config);
    result += wxString::Format(_T("%d"), config - 1);

    // last digit: 1 for reverse, 0 for normal direction
    result += (code.Right(1) == _T("R")) ? _T("1") : _T("0");
    return result;
  }

  // not found
  wxFAIL;
  return wxEmptyString;
}
