#include "cproperties.h"
#include "cmgr.h"
#include <wx/dcclient.h>

//-----------------------------------------------------------------------------
// Property enumeration

enum ENUM_ROWS {
  ROW_PLAYERNAME = 0,
  ROW_CARNAME,
  ROW_LAPTIME,
  ROW_SPLITS,
  ROW_SECTORS,
  ROW_LFSVERSION,
  ROW_HLVC,
  ROW_BRAKES,
  ROW_RIDEHEIGHT,
  ROW_STIFFNESS,
  ROW_DAMPING_BUMP,
  ROW_DAMPING_REBOUND,
  ROW_ANTIROLL,
  ROW_GEARS,
  ROW_FINALDRIVE,
  ROW_TYRETYPE,
  ROW_TYRETEMP,
  ROW_SAMPLEFREQ
};

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cProperties, cPane)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cProperties::cProperties(wxWindow* parent)
: cPane(parent)
{
  m_List = new wxListView(this, -1, wxDefaultPosition, wxDefaultSize, wxLC_REPORT | wxNO_BORDER | wxLC_HRULES | wxLC_VRULES);

  m_TopSizer = new wxBoxSizer(wxVERTICAL);
  m_TopSizer->Add(m_List, wxSizerFlags(1).Expand().Proportion(1));
  SetSizer(m_TopSizer);

  // calculate font height
  wxClientDC dc(this);
  dc.SetFont(*wxNORMAL_FONT);
  wxCoord sizeX;
  dc.GetTextExtent(_T("Wg"), &sizeX, &m_FontHeight);

  // create rows for the properties
  m_List->InsertColumn(0, wxEmptyString);
  m_List->InsertItem(ROW_PLAYERNAME, wxEmptyString);
  m_List->InsertItem(ROW_CARNAME, wxEmptyString);
  m_List->InsertItem(ROW_LAPTIME, wxEmptyString);
  m_List->InsertItem(ROW_SPLITS, wxEmptyString);
  m_List->InsertItem(ROW_SECTORS, wxEmptyString);
  m_List->InsertItem(ROW_LFSVERSION, wxEmptyString);
  m_List->InsertItem(ROW_HLVC, wxEmptyString);

  m_List->InsertItem(ROW_BRAKES, wxEmptyString);
  m_List->InsertItem(ROW_RIDEHEIGHT, wxEmptyString);
  m_List->InsertItem(ROW_STIFFNESS, wxEmptyString);
  m_List->InsertItem(ROW_DAMPING_BUMP, wxEmptyString);
  m_List->InsertItem(ROW_DAMPING_REBOUND, wxEmptyString);
  m_List->InsertItem(ROW_ANTIROLL, wxEmptyString);
  m_List->InsertItem(ROW_GEARS, wxEmptyString);
  m_List->InsertItem(ROW_FINALDRIVE, wxEmptyString);
  m_List->InsertItem(ROW_TYRETYPE, wxEmptyString);
  m_List->InsertItem(ROW_TYRETEMP, wxEmptyString);
  m_List->InsertItem(ROW_SAMPLEFREQ, wxEmptyString);

  TranslateTexts();

  RedirectEvents(m_List, this, false, true); // redirect escape-key events to parent
  RedirectEvents(m_List, NULL, true, false); // redirect mouse-wheel events to window-at-pointer
}

//-----------------------------------------------------------------------------

cProperties::~cProperties()
{
}

//-----------------------------------------------------------------------------
// Re-read the properties from all the laps

void cProperties::UpdateAll()
{
  // delete exisiting columns
  for (int c = m_List->GetColumnCount() - 1; c > 0; c--) m_List->DeleteColumn(c);

  // re-create a column for each lap
  for (size_t l = 0; l < MGR->GetLapCount(); l++) {
    cLap* lap = MGR->GetLap(l);

    size_t col = l + 1;
    m_List->InsertColumn(col, lap->GetName());

    wxString buf;

    // general info
    m_List->SetItem(ROW_PLAYERNAME, col, lap->m_PlayerName);
    buf = lap->m_Car.m_Name + _T(" (") + lap->m_CarCode + _T(")");
    m_List->SetItem(ROW_CARNAME, col, buf);
    m_List->SetItem(ROW_LFSVERSION, col, lap->m_LfsVersion);
    m_List->SetItem(ROW_LAPTIME, col, ::FormatTime(lap->m_Split[lap->m_SplitCount - 1]));

    // split times
    buf = ::FormatTime(lap->m_Split[0]);
    for (int s = 1; s < lap->m_SplitCount - 1; s++) {
      buf += _T(", ") + ::FormatTime(lap->m_Split[s]);
    }
    m_List->SetItem(ROW_SPLITS, col, buf);

    // sector times
    buf = ::FormatTime(lap->m_Split[0]);
    for (int s = 1; s < lap->m_SplitCount; s++) {
      buf += _T(", ") + ::FormatTime(lap->m_Split[s] - lap->m_Split[s-1]);
    }
    m_List->SetItem(ROW_SECTORS, col, buf);

    // HLVC
    switch (lap->m_Hlvc) {
      case 1 : buf = _TT(ID_TXT_PR_HLVC1, "Legal"); break;
      case 2 : buf = _TT(ID_TXT_PR_HLVC2, "Illegal"); break;
      default : buf = _TT(ID_TXT_PR_HLVC0, "Unknown"); break;
    }
    m_List->SetItem(ROW_HLVC, col, buf);

    // car setup data
    wxString front = _TT(ID_TXT_PR_FRONT, "front");
    wxString rear = _TT(ID_TXT_PR_REAR, "rear");

    float frontBrake = lap->m_Car.m_Brake[0];
    float totalBrake = frontBrake + lap->m_Car.m_Brake[2];
    buf.Printf(_T("%.0f, %.0f%% %s"),
        totalBrake / 2,
        frontBrake * 100 / totalBrake,
        front.c_str());
    m_List->SetItem(ROW_BRAKES, col, buf);

    buf.Printf(_T("%.3f %s, %.3f %s"),
        lap->m_Car.m_MaxDeflect[2], rear.c_str(),
        lap->m_Car.m_MaxDeflect[0], front.c_str());
    m_List->SetItem(ROW_RIDEHEIGHT, col, buf);

    buf.Printf(_T("%.1f %s, %.1f %s"),
        lap->m_Car.m_Spring[2] / 1000, rear.c_str(),
        lap->m_Car.m_Spring[0] / 1000, front.c_str());
    m_List->SetItem(ROW_STIFFNESS, col, buf);

    buf.Printf(_T("%.1f %s, %.1f %s"),
        lap->m_Car.m_Damp_Bump[2] / 1000, rear.c_str(),
        lap->m_Car.m_Damp_Bump[0] / 1000, front.c_str());
    m_List->SetItem(ROW_DAMPING_BUMP, col, buf);

    buf.Printf(_T("%.1f %s, %.1f %s"),
        lap->m_Car.m_Damp_Rebound[2] / 1000, rear.c_str(),
        lap->m_Car.m_Damp_Rebound[0] / 1000, front.c_str());
    m_List->SetItem(ROW_DAMPING_REBOUND, col, buf);

    buf.Printf(_T("%.1f %s, %.1f %s"),
        lap->m_Car.m_AntiRoll_Rear / 1000, rear.c_str(),
        lap->m_Car.m_AntiRoll_Front / 1000, front.c_str());
    m_List->SetItem(ROW_ANTIROLL, col, buf);

    buf = wxEmptyString;
    for (int g = 0; g < lap->m_Car.m_GearCount; g++) {
      if (g > 0) buf += _T(", ");
      if (g < MAX_GEAR_RATIOS) {
        buf += wxString::Format(_T("%.2f"), lap->m_Car.m_GearRatio[g]);
      }
      else {
        buf += _T("?");
      }
    }
    m_List->SetItem(ROW_GEARS, col, buf);

    buf.Printf(_T("%.2f"), lap->m_Car.m_FinalDrive);
    m_List->SetItem(ROW_FINALDRIVE, col, buf);

    if (lap->m_Car.m_TyreType[2] == lap->m_Car.m_TyreType[0]) {
      // same tyre types on front and rear
      buf.Printf(_T("%s"), cCar::TyreCode2Name(lap->m_Car.m_TyreType[2]).c_str());
    }
    else {
      // different tyre types
      buf.Printf(_T("%s %s, %s %s"),
          cCar::TyreCode2Name(lap->m_Car.m_TyreType[2]).c_str(), rear.c_str(),
          cCar::TyreCode2Name(lap->m_Car.m_TyreType[0]).c_str(), front.c_str());
    }
    m_List->SetItem(ROW_TYRETYPE, col, buf);

    buf.Printf(_T("%.1f | %.1f %s, %.1f | %.1f %s"),
        lap->m_AvgTemp[2], lap->m_AvgTemp[3], rear.c_str(),
        lap->m_AvgTemp[0], lap->m_AvgTemp[1], front.c_str());
    m_List->SetItem(ROW_TYRETEMP, col, buf);

    // file sampling frequency
    buf.Printf(_T("%.0f Hz"), 1.0 / lap->GetTimeInterval());
    m_List->SetItem(ROW_SAMPLEFREQ, col, buf);
  }

  ::SetColumnWidths(m_List);
}

//-----------------------------------------------------------------------------

wxSize cProperties::DoGetBestSize() const
{
  // calculate total width
  int width = 0;
  for (int c = 0; c < m_List->GetColumnCount(); c++) width += m_List->GetColumnWidth(c);
  if (m_List->GetColumnCount() == 1) width *= 2;
  width += m_List->GetColumnCount() * 2;

  // calculate total height
  int height = (m_List->GetItemCount() + 2) * (m_FontHeight + 2);

  return wxSize(width, height);
}

//-----------------------------------------------------------------------------
// Translation

void cProperties::TranslateTexts()
{
  m_List->SetItemText(ROW_PLAYERNAME,      _TT(ID_TXT_PR_PLAYERNAME,      "Player"));
  m_List->SetItemText(ROW_CARNAME,         _TT(ID_TXT_PR_CARNAME,         "Car"));
  m_List->SetItemText(ROW_LAPTIME,         _TT(ID_TXT_PR_LAPTIME,         "Lap time"));
  m_List->SetItemText(ROW_SPLITS,          _TT(ID_TXT_PR_SPLITS,          "Split times"));
  m_List->SetItemText(ROW_SECTORS,         _TT(ID_TXT_PR_SECTORS,         "Sector times"));
  m_List->SetItemText(ROW_LFSVERSION,      _TT(ID_TXT_PR_LFSVERSION,      "LFS version"));
  m_List->SetItemText(ROW_HLVC,            _TT(ID_TXT_PR_HLVC,            "HLVC"));
  m_List->SetItemText(ROW_BRAKES,          _TT(ID_TXT_PR_BRAKES,          "Brakes"));
  m_List->SetItemText(ROW_RIDEHEIGHT,      _TT(ID_TXT_PR_RIDEHEIGHT,      "Ride height"));
  m_List->SetItemText(ROW_STIFFNESS,       _TT(ID_TXT_PR_STIFFNESS,       "Stiffness"));
  m_List->SetItemText(ROW_DAMPING_BUMP,    _TT(ID_TXT_PR_DAMPING_BUMP,    "Bump damping"));
  m_List->SetItemText(ROW_DAMPING_REBOUND, _TT(ID_TXT_PR_DAMPING_REBOUND, "Rebound damping"));
  m_List->SetItemText(ROW_ANTIROLL,        _TT(ID_TXT_PR_ANTIROLL,        "Anti roll"));
  m_List->SetItemText(ROW_GEARS,           _TT(ID_TXT_PR_GEARS,           "Gears"));
  m_List->SetItemText(ROW_FINALDRIVE,      _TT(ID_TXT_PR_FINALDRIVE,      "Final drive"));
  m_List->SetItemText(ROW_TYRETYPE,        _TT(ID_TXT_PR_TYRETYPE,        "Tyre type"));
  m_List->SetItemText(ROW_TYRETEMP,        _TT(ID_TXT_PR_TYRETEMP,        "Tyre temperature (avg)"));
  m_List->SetItemText(ROW_SAMPLEFREQ,      _TT(ID_TXT_PR_SAMPLEFREQ,      "Sampling"));

  ::SetColumnWidths(m_List);
}
