#include "cdrivingline.h"
#include "cmgr.h"
#include "clap.h"
#include "ctrackpath.h"
#include <wx/dcclient.h>
#include <wx/cursor.h>
#include <wx/config.h>

// default scale factor
#define DEFAULT_SCALE 1.0f

// default visible range (= #pixels in visible part of window when at minimum size)
#define DEFAULT_RANGE 100

// multiplication/division factor for zooming in/out (for mouse clicks and for mouse wheel)
#define ZOOM_FACTOR_BUTTON 1.5f
#define ZOOM_FACTOR_WHEEL 1.25f

// diameter of the track cursor
#define CURSOR_DIAMETER 4

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cDrivingLine, cPane)
  EVT_LEFT_DCLICK(cDrivingLine::OnMouseLeftDoubleClick)
  EVT_LEFT_DOWN(cDrivingLine::OnMouseLeftClick)
  EVT_LEFT_UP(cDrivingLine::OnMouseLeftRelease)
  EVT_MIDDLE_DOWN(cDrivingLine::OnMouseMiddleClick)
  EVT_MOTION(cDrivingLine::OnMouseMove)
  EVT_MOUSEWHEEL(cDrivingLine::OnMouseWheel)
  EVT_RIGHT_DCLICK(cDrivingLine::OnMouseRightDoubleClick)
  EVT_RIGHT_DOWN(cDrivingLine::OnMouseRightClick)
  EVT_MENU(wxID_ANY, cDrivingLine::OnMenuClick)
  EVT_PAINT(cDrivingLine::OnPaint)
  EVT_SIZE(cDrivingLine::OnSize)
  EVT_SCROLLWIN(cDrivingLine::OnScrollWin)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cDrivingLine::cDrivingLine(wxWindow* parent)
: cPane(parent)
{
  m_Zoom = 1.0f;

  m_CursorDist = -1.0f;
  m_CursorDrawn = false;
  m_AutoScroll = false;

  m_DragMode = false;
  m_ClickPos = -1;

  m_ShowPath = true;
  m_Path = NULL;
  m_PolygonsValid = false;
  m_Polygons = NULL;
  m_PolygonSize = NULL;
  m_PolygonCount = 0;

  ShowBitmap(true);
  m_Bitmap = NULL;
  m_BitmapValid = false;

  cLang::AppendCheckItem(&m_Context, ID_MENU_SHOW_PATH, _T("Show path"));
  cLang::AppendCheckItem(&m_Context, ID_MENU_SHOW_BITMAP, _T("Show map"));
  m_Context.AppendSeparator();
  cLang::AppendMenuItem(&m_Context, ID_MENU_ZOOM_RESET, _T("Reset zoom"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_ZOOM_IN, _T("Zoom in"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_ZOOM_OUT, _T("Zoom out"));
  m_Context.AppendSeparator();
  cLang::AppendCheckItem(&m_Context, ID_MENU_AUTOSCROLL, _T("Auto-scroll"));
}

//-----------------------------------------------------------------------------

cDrivingLine::~cDrivingLine()
{
  if (HasCapture()) ReleaseMouse();

  m_PolygonsValid = false;
  delete[] m_Polygons;
  delete[] m_PolygonSize;

  m_BitmapDc.SelectObjectAsSource(wxNullBitmap);
  delete m_Bitmap;
  m_Bitmap = NULL;
}

//-----------------------------------------------------------------------------
// Plot the trackmap

void cDrivingLine::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  wxPaintDC dc(this);
  dc.SetLogicalFunction(wxCOPY);

  if (MGR->IsEmpty()) return;

  // plot the track
  DrawBitmap(dc);
  DrawTrack(dc);
  DrawLaps(dc);

  // re-plot the cursor
  if ((m_CursorDrawn) && IsExposed(m_CursorX, m_CursorY)) {
    m_CursorDrawn = false;
    PutTrackCursor(dc, m_CursorDist);
  }
}

//-----------------------------------------------------------------------------
// Draw the track path

void cDrivingLine::DrawTrack(wxDC& dc)
{
  if (!m_ShowPath) return;

  if (!m_PolygonsValid) {
    // need to renew the polygons
    delete[] m_Polygons;
    m_Polygons = NULL;
    delete[] m_PolygonSize;
    m_PolygonSize = NULL;
    m_PolygonCount = 0;
    m_PolygonsValid = true;

    m_Path = MGR->GetTrackPath();
    CHECK_PTR_NULL(m_Path);
    if (m_Path == NULL) return; // no track path loaded

    // allocate
    m_PolygonCount = m_Path->m_NodeCount;
    m_Polygons = new wxPoint[4 * m_PolygonCount];
    m_PolygonSize = new int[m_PolygonCount];

    // calculate coordinates
    for (int n = 0; n < m_PolygonCount; n ++) {
      m_PolygonSize[n] = 4;
      int nn = (n + 1) % m_PolygonCount; // next node
      m_Polygons[4*n].x = Track2CoordX(m_Path->m_RoadLeftX[n]);
      m_Polygons[4*n].y = Track2CoordY(m_Path->m_RoadLeftY[n]);
      m_Polygons[4*n+1].x = Track2CoordX(m_Path->m_RoadRightX[n]);
      m_Polygons[4*n+1].y = Track2CoordY(m_Path->m_RoadRightY[n]);
      m_Polygons[4*n+2].x = Track2CoordX(m_Path->m_RoadRightX[nn]);
      m_Polygons[4*n+2].y = Track2CoordY(m_Path->m_RoadRightY[nn]);
      m_Polygons[4*n+3].x = Track2CoordX(m_Path->m_RoadLeftX[nn]);
      m_Polygons[4*n+3].y = Track2CoordY(m_Path->m_RoadLeftY[nn]);
    }
  }

  if (m_Polygons == NULL) return; // drawing track not possible

  CHECK_PTR(m_Path);
  wxCoord offsetX = Track2CoordX(m_Path->m_RoadLeftX[0]) - m_Polygons[0].x;
  wxCoord offsetY = Track2CoordY(m_Path->m_RoadLeftY[0]) - m_Polygons[0].y;

  // draw the polygons
  wxColour col(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY);
  dc.SetPen(wxPen(col));
  dc.SetBrush(wxBrush(col));
  dc.DrawPolyPolygon(m_PolygonCount, m_PolygonSize, m_Polygons, offsetX, offsetY);

  // draw the finish line
  dc.SetPen(*wxBLACK_PEN);
  dc.DrawLine(
      Track2CoordX(m_Path->m_RoadLeftX[m_Path->m_FinishNode]),
      Track2CoordY(m_Path->m_RoadLeftY[m_Path->m_FinishNode]),
      Track2CoordX(m_Path->m_RoadRightX[m_Path->m_FinishNode]),
      Track2CoordY(m_Path->m_RoadRightY[m_Path->m_FinishNode]));

#ifdef __WXDEBUG__
  // debug only: show origin
  dc.SetPen(*wxRED_PEN);
  dc.DrawCircle(Track2CoordX(0), Track2CoordY(0), 2);
#endif
}

//-----------------------------------------------------------------------------
// Draw the track bitmap
// For more info on the bitmaps see http://www.lfsforum.net/showthread.php?t=37411

void cDrivingLine::DrawBitmap(wxDC& dc)
{
  if (!m_ShowBitmap) return;

  if (!m_BitmapValid) {
    // discard old bitmap and load new one
    m_BitmapDc.SelectObjectAsSource(wxNullBitmap);
    delete m_Bitmap;
    m_Bitmap = NULL;
    SetBackgroundColour(*wxWHITE);

    m_BitmapFile = MGR->GetBitmapFile();
    if (!m_BitmapFile.IsEmpty()) {
      // load file
      wxBeginBusyCursor();
      m_Bitmap = new wxBitmap(m_BitmapFile, wxBITMAP_TYPE_TIF);
      if (m_Bitmap->IsOk()) {
        // load OK
        m_BitmapDc.SelectObjectAsSource(*m_Bitmap);

        // set window background colour from background of bitmap
        wxColour col;
        m_BitmapDc.GetPixel(0, 0, &col);
        SetBackgroundColour(col);

        // set offset (the point with track coordinates (0,0) is exactly in the centre of the bitmap
        m_BitmapOffsetX = - m_Bitmap->GetWidth()/2;
        m_BitmapOffsetY = - m_Bitmap->GetHeight()/2;
      }
      else {
        // load failed
        delete m_Bitmap;
        m_Bitmap = NULL;
        wxLogDebug(_T("Could not load track bitmap file \"%s\""), m_BitmapFile.c_str());
      }
      wxEndBusyCursor();
    }
    m_BitmapValid = true;
  }

  if ((m_Bitmap == NULL) || (!m_BitmapDc.IsOk())) return; // nothing to draw

  // draw it
  double oldScaleX, oldScaleY;
  dc.GetUserScale(&oldScaleX, &oldScaleY); // save current user scale
  dc.SetUserScale(m_Zoom, m_Zoom);

  wxCoord width = (wxCoord)(GetClientSize().GetWidth() / m_Zoom) + 1;
  wxCoord height = (wxCoord)(GetClientSize().GetHeight() / m_Zoom) + 1;
  float xsrc = Coord2TrackX(0) - m_BitmapOffsetX;
  float ysrc = m_Bitmap->GetHeight() - (Coord2TrackY(0) - m_BitmapOffsetY);
  dc.Blit(0, 0, width, height, &m_BitmapDc, (wxCoord)xsrc, (wxCoord)ysrc);

  dc.SetUserScale(oldScaleX, oldScaleY); // restore user scale
}

//-----------------------------------------------------------------------------
// Draw the trajectory of each lap

void cDrivingLine::DrawLaps(wxDC& dc)
{
  for (size_t i = 0; i < MGR->GetLapCount(); i++) {
    cLap* lap = MGR->GetLap(i);
    if (!lap->IsShown()) continue;

    int width = lap->GetCar().GetWidth() * m_Zoom;
    if (width < 1) width = 1;
    dc.SetPen(wxPen(lap->GetColour(), width, wxSOLID));

    // get screen coordinates of first state
    const cCarState* state = lap->GetState(0);
    wxPoint prev(Track2CoordX(state->GetPosX()), Track2CoordY(state->GetPosY()));

    // plot the lap's trajectory
    for (size_t s = 1; s < lap->GetStateCount(); s++) {
      // draw line from previous position to current position
      state = lap->GetState(s);
      wxPoint curr(Track2CoordX(state->GetPosX()), Track2CoordY(state->GetPosY()));
      dc.DrawLine(prev, curr);

      prev = curr;
    }
  }
}

//-----------------------------------------------------------------------------
// Set the "show bitmap" flag

void cDrivingLine::ShowBitmap(bool flag)
{
  m_ShowBitmap = flag;

  if (m_ShowBitmap && !MGR->IsEmpty()) {
    SetBackgroundColour(*wxBLACK);
  }
  else {
    SetBackgroundColour(*wxWHITE);
  }
}

//-----------------------------------------------------------------------------
// Draw the cursor at a new position
// - distance = distance_in_lap (in m) that corresponds to the cursor position
// - type = type of cursor-set command
// If distance < 0 then cursor is removed

void cDrivingLine::DoSetTrackCursorPos(cGraphView* WXUNUSED(view), float distance, int type)
{
  if (m_DragMode) return; // ignore calls while dragging
  if (MGR->IsEmpty()) return; // empty display
  if ((type == CURSORSET_FOLLOW) && (!m_AutoScroll)) return; // do not follow mouse, unless auto-scroll is on

  // move track cursor to its new position
  wxClientDC dc(this);
  RemoveTrackCursor(dc);
  PutTrackCursor(dc, distance);

  // check if we need to scroll the window to keep the cursor visible
  if (m_CursorDist >= 0.0f) {
    // horizontal scrolling
    wxCoord marginX = GetSize().GetWidth() / 4; // minimum distance to window border before scrolling occurs
    int dx = 0;                                 // amount (in pixels) that the centre needs to be moved
    if (m_CursorX < marginX)
        dx = marginX - m_CursorX;
    if (m_CursorX > GetClientSize().GetWidth() - marginX)
        dx = GetClientSize().GetWidth() - marginX - m_CursorX;

    // vertical scrolling
    wxCoord marginY = GetSize().GetHeight() / 4;
    int dy = 0;
    if (m_CursorY < marginY)
        dy = marginY - m_CursorY;
    if (m_CursorY > GetClientSize().GetHeight() - marginY)
        dy = GetClientSize().GetHeight() - marginY - m_CursorY;

    // scroll it
    if ((dx != 0) || (dy != 0)) {
      RemoveTrackCursor(dc);
      ScrollView(dx, dy);
      PutTrackCursor(dc, distance);
    }
  }
}

//-----------------------------------------------------------------------------
// Draw the cursor at a new position
// - dc = device context to draw in
// - distance = distance-in-lap for cursor position

void cDrivingLine::PutTrackCursor(wxDC& dc, float distance)
{
  wxASSERT(!m_CursorDrawn);

  m_CursorDist = distance;
  if (distance < 0) return;

  cLap* lap = MGR->GetLeader();
  if (lap == NULL) return;
  CHECK_PTR(lap);
  float x, y;
  lap->GetPositionAt(distance, x, y);

  m_CursorX = Track2CoordX(x);
  m_CursorY = Track2CoordY(y);
  DrawTrackCursor(dc, m_CursorX, m_CursorY);
  m_CursorDrawn = true;
}

//-----------------------------------------------------------------------------
// Remove the cursor from its current position

void cDrivingLine::RemoveTrackCursor(wxDC& dc)
{
  if (!m_CursorDrawn) return;
  DrawTrackCursor(dc, m_CursorX, m_CursorY);
  m_CursorDrawn = false;
}

//-----------------------------------------------------------------------------
// Actually draw the cursor
// - dc = device context to draw in
// - x, y = coordinates of the cursor position

void cDrivingLine::DrawTrackCursor(wxDC& dc, wxCoord x, wxCoord y)
{
  dc.SetPen(*wxWHITE_PEN);
  dc.SetBrush(*wxWHITE_BRUSH);
  dc.SetLogicalFunction(wxINVERT);
  dc.DrawCircle(x, y, CURSOR_DIAMETER);
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnSize(wxSizeEvent& event)
{
  // force full redraw when window is resized
  Refresh();
  event.Skip();
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseLeftClick(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler
  if (MGR->IsEmpty()) return; // empty display

  if (event.CmdDown()) {
    // zoom in
    SetZoom(m_Zoom * ZOOM_FACTOR_BUTTON);
    m_ClickPos = -1;
  }
  else {
    // prepare for drag-scrolling
    wxClientDC dc(this);
    m_ClickPos = DecodeMouse(event.GetLogicalPosition(dc));
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseMove(wxMouseEvent& event)
{
  if (MGR->IsEmpty()) return;

  // get logical position of mouse
  wxClientDC dc(this);
  wxPoint pos = event.GetLogicalPosition(dc);

  if (event.LeftIsDown()) {
    if (!m_DragMode) {
      // enter dragging mode
      m_DragMode = true;
      CaptureMouse();
      SetCursor(wxCursor(wxCURSOR_HAND));
    }
    else {
      // already dragging
      wxClientDC dc(this);
      RemoveTrackCursor(dc);
      ScrollView(pos.x - m_MovePos.x, pos.y - m_MovePos.y);
      PutTrackCursor(dc, m_CursorDist);
    }

    m_MovePos = pos;
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseLeftRelease(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler

  if (m_DragMode) {
    // end drag
    SetCursor(wxNullCursor);
    if (HasCapture()) ReleaseMouse();
    m_DragMode = false;
  }
  else {
    if (MGR->IsEmpty()) return; // empty display
    if (m_ClickPos < 0) return; // no valid click

    ::SetTrackCursorPos_Send(m_ClickPos, CURSORSET_FORCE);
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseLeftDoubleClick(wxMouseEvent& event)
{
  if (event.CmdDown()) {
    // zoom in
    SetZoom(m_Zoom * ZOOM_FACTOR_BUTTON);
    m_ClickPos = -1;
  }
  else {
    ResetZoom();
  }
}

void cDrivingLine::OnMouseRightDoubleClick(wxMouseEvent& event)
{
  if (event.CmdDown()) {
    // zoom out
    SetZoom(m_Zoom / ZOOM_FACTOR_BUTTON);
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::SetZoom(float zoom)
{
  if (m_Zoom == zoom) return;
  m_Zoom = zoom;
  m_PolygonsValid = false;
  Refresh();
}

void cDrivingLine::ResetZoom()
{
  SetZoom(1.0);
}

//-----------------------------------------------------------------------------
void cDrivingLine::UpdateAll()
{
  if (m_Path != MGR->GetTrackPath()) {
    // just loaded a new track
    m_PolygonsValid = false;
  }
  if (m_BitmapFile != MGR->GetBitmapFile()) {
    // just loaded a track on a new venue
    m_BitmapValid = false;
  }

  if (MGR->GetLapCount() > 0) {
    // reset view centre to starting line
    const cCarState* state = MGR->GetLap(0)->GetState(0);
    m_CentreX = state->GetPosX();
    m_CentreY = state->GetPosY();
  }

  Refresh();

  SetMinSize(wxSize(DEFAULT_RANGE, DEFAULT_RANGE));
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseWheel(wxMouseEvent& event)
{
  if (MGR->IsEmpty()) return;

  float factor = ((event.GetWheelRotation() > 0) ? ZOOM_FACTOR_WHEEL : (1.0f / ZOOM_FACTOR_WHEEL));
  SetZoom(m_Zoom * factor);
}

//-----------------------------------------------------------------------------
// Return the distance-in-lap that corresponds to a mouse position

float cDrivingLine::DecodeMouse(wxPoint pos)
{
  if (MGR->IsEmpty()) return -1;

  // find state that is nearest to this position
  float nearest_dist = -1; // distance-in-lap of state that is nearest to mouse position
  float nearest_pix = IMPOSSIBLY_HIGH_VALUE; // square of distance in pixels to that state

  // scan all laps
  for (size_t i = 0; i < MGR->GetLapCount(); i++) {
    cLap* lap = MGR->GetLap(i);
    // scan all states
    for (size_t s = 0; s < lap->GetStateCount(); s++) {
      const cCarState* state = lap->GetState(s);
      float dX = Track2CoordX(state->GetPosX()) - pos.x;
      float dY = Track2CoordY(state->GetPosY()) - pos.y;
      float d = (dX * dX) + (dY * dY);
      if (d < nearest_pix) {
        nearest_pix = d;
        nearest_dist = state->GetDistance();
      }
    }
  }

  if (sqrt(nearest_pix) > CLICK_DETECT_THRESHOLD) return -1;

  return nearest_dist;
}

//-----------------------------------------------------------------------------
// Scroll the display

void cDrivingLine::ScrollView(int dx, int dy)
{
  if ((dx == 0) && (dy == 0)) return;

  wxASSERT(!m_CursorDrawn);

  m_CentreX -= ((float)dx) / m_Zoom;
  m_CentreY += ((float)dy) / m_Zoom;

  ScrollWindow(dx, dy);
  Update();
}

//-----------------------------------------------------------------------------

wxSize cDrivingLine::DoGetBestSize() const
{
  int range = 2 * DEFAULT_RANGE;
  return wxSize(range, range);
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseRightClick(wxMouseEvent& event)
{
  if (event.CmdDown()) {
    // zoom out
    SetZoom(m_Zoom / ZOOM_FACTOR_BUTTON);
  }
  else {
    if (HasCapture()) return; // don't show context menu while a drag is going on

    // show context menu
    bool hasLaps = (MGR->GetLapCount() > 0);
    bool hasPath = (MGR->GetTrackPath() != NULL);
    bool hasBitmap = (!MGR->GetBitmapFile().IsEmpty());

    m_Context.Check(ID_MENU_SHOW_PATH, m_ShowPath);
    m_Context.Enable(ID_MENU_SHOW_PATH, hasPath || !hasLaps);
    m_Context.Check(ID_MENU_SHOW_BITMAP, m_ShowBitmap);
    m_Context.Enable(ID_MENU_SHOW_BITMAP, hasBitmap || !hasLaps);

    m_Context.Enable(ID_MENU_ZOOM_IN, hasLaps);
    m_Context.Enable(ID_MENU_ZOOM_OUT, hasLaps);
    m_Context.Enable(ID_MENU_ZOOM_RESET, hasLaps && (m_Zoom != 1.0f));
    m_Context.Check(ID_MENU_AUTOSCROLL, m_AutoScroll);
    PopupMenu(&m_Context);
  }
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cDrivingLine::OnMenuClick(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_ZOOM_IN :
      SetZoom(m_Zoom * ZOOM_FACTOR_BUTTON);
      break;

    case ID_MENU_ZOOM_OUT :
      SetZoom(m_Zoom / ZOOM_FACTOR_BUTTON);
      break;

    case ID_MENU_ZOOM_RESET :
      ResetZoom();
      break;

    case ID_MENU_AUTOSCROLL :
      m_AutoScroll = !m_AutoScroll;
      break;

    case ID_MENU_SHOW_PATH :
      m_ShowPath = !m_ShowPath;
      Refresh();
      break;

    case ID_MENU_SHOW_BITMAP :
      ShowBitmap(!m_ShowBitmap);
      Refresh();
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cDrivingLine::LoadConfig(wxRegConfig* config, const wxString& key)
{
  config->Read(key + _T("/autoscroll"), &m_AutoScroll);
  double zoom = m_Zoom;
  if (config->Read(key + _T("/zoom"), &zoom)) m_Zoom = zoom;
  config->Read(key + _T("/show_path"), &m_ShowPath);
  config->Read(key + _T("/show_bitmap"), &m_ShowBitmap);
  ShowBitmap(m_ShowBitmap);
}


void cDrivingLine::SaveConfig(wxRegConfig* config, const wxString& key)
{
  config->Write(key + _T("/autoscroll"), m_AutoScroll);
  config->Write(key + _T("/zoom"), m_Zoom);
  config->Write(key + _T("/show_path"), m_ShowPath);
  config->Write(key + _T("/show_bitmap"), m_ShowBitmap);
}

//-----------------------------------------------------------------------------
// Translation

void cDrivingLine::TranslateTexts()
{
  cLang::TranslateMenu(&m_Context);
}
