#include "ctrackmap.h"
#include "cmgr.h"
#include "clap.h"
#include <wx/dcclient.h>
#include <wx/cursor.h>
#include <wx/config.h>

// thickness of the lap trajectory (in pixels)
#define TRAJECTORY_WIDTH 2

// half the length of the sector markers (in pixels)
#define MARKER_LENGTH2 4

// minimal length (in meters) for a sector
#define MIN_SECTOR_LENGTH 50

// max. number of line segments that the track is plotted in
#define MAX_SEGMENTS 1000

// margin between map and border of window
#define MAP_MARGIN 8

// minimum length of a drag-selected part of the track
#define MIN_SELECTION_LENGTH 25

// distance (in meters) for scrolling with the mouse wheel
#define SCROLL_DIST 50

// distance (in pixels) between sector buttons
#define BUTTON_SPACING 2

// margin when setting an auto-detected sector
#define SECTOR_MARGIN 15.0f

// minimum brake value / maximum throttle value (in %) for auto-detecting a sector
#define BRAKE_PEDAL_THRESHOLD 5.0f
#define THROTTLE_PEDAL_THRESHOLD 90.0f

// minimum time between sector-start events (in s)
#define SECTOR_DURATION_THRESHOLD 2.0f

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cTrackMap, cPane)
  EVT_LEFT_DCLICK(cTrackMap::OnMouseLeftDoubleClick)
  EVT_LEFT_DOWN(cTrackMap::OnMouseLeftClick)
  EVT_LEFT_UP(cTrackMap::OnMouseLeftRelease)
  EVT_MIDDLE_DOWN(cTrackMap::OnMouseMiddleClick)
  EVT_RIGHT_DOWN(cTrackMap::OnMouseRightClick)
  EVT_MENU(wxID_ANY, cTrackMap::OnMenuClick)
  EVT_MOTION(cTrackMap::OnMouseMove)
  EVT_MOUSEWHEEL(cTrackMap::OnMouseWheel)
  EVT_ENTER_WINDOW(cTrackMap::OnMouseEntering)
  EVT_LEAVE_WINDOW(cTrackMap::OnMouseLeaving)
  EVT_PAINT(cTrackMap::OnPaint)
  EVT_SIZE(cTrackMap::OnSize)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------
// Static members

wxPen cTrackMap::s_Pen_Sel(wxColour(MEDIUM_GREY, MEDIUM_GREY, MEDIUM_GREY), TRAJECTORY_WIDTH);
wxPen cTrackMap::s_Pen_UnSel(wxColour(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY), TRAJECTORY_WIDTH);
wxPen cTrackMap::s_ButtonPen(wxColour(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY));
wxPen cTrackMap::s_SfPen(wxColour(200, 0, 0), 3);
wxPen cTrackMap::s_SectorPen(wxColour(0, 0, 0), 2);
wxPen cTrackMap::s_SectorDragPen(wxColour(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY), 2);

//-----------------------------------------------------------------------------

cTrackMap::cTrackMap(wxWindow* parent)
: cPane(parent)
{
  m_DragMode = false;
  m_ClickPos = -1;
  m_ClickMarker = -1;
  m_CursorDrawn = false;

  m_ShowSectorToolbar = true;
  m_LockSectors = false;
  m_SectorsChanged = false;
  ClearSectors();

  m_CursorDist = -1;
  m_ShowDistance = false;

  SetBackgroundColour(*wxWHITE);

  m_ResetSectorsMenu = new wxMenu;
  cLang::AppendMenuItem(m_ResetSectorsMenu, ID_MENU_SECTOR_AUTODETECT, _T("Auto-detect"));
  cLang::AppendMenuItem(m_ResetSectorsMenu, ID_MENU_SECTOR_RESET_CLEAR, _T("Clear all"));
  cLang::AppendMenuItem(m_ResetSectorsMenu, ID_MENU_SECTOR_RESET_TIMELINES, _T("LFS splits"));
  m_ResetSectorsMenu->AppendSeparator();
  cLang::AppendMenuItem(m_ResetSectorsMenu, ID_MENU_SECTOR_UNDO, _T("Undo changes"));

  cLang::AppendMenuItem(&m_Context, ID_MENU_ZOOM_RESET, _T("Reset zoom"));
  m_Context.AppendSeparator();
  cLang::AppendMenuItem(&m_Context, ID_MENU_SECTOR_ADD_DEL, _T("Add sector"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_SECTOR_ADD_SELECTION, _T("Use selection as sector"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_SECTOR_RESET, _T("Reset sectors"), m_ResetSectorsMenu);
  cLang::AppendCheckItem(&m_Context, ID_MENU_SECTOR_LOCK, _T("Lock sectors"));
  m_Context.AppendSeparator();
  cLang::AppendCheckItem(&m_Context, ID_MENU_SHOWSECTORBUTTONS, _T("Show sector buttons"));
  cLang::AppendCheckItem(&m_Context, ID_MENU_SHOWDISTANCE, _T("Show cursor distance"));

  AdjustDimensions();
}

//-----------------------------------------------------------------------------

cTrackMap::~cTrackMap()
{
}

//-----------------------------------------------------------------------------
// Re-calculate the dimensioning

void cTrackMap::AdjustDimensions()
{
  wxCoord sizeX, sizeY;
  wxClientDC dc(this);
  dc.SetFont(*wxNORMAL_FONT);

  int width, height; // room available for the track map itself
  GetClientSize(&width, &height);

  dc.GetTextExtent(LARGEST_VALUE_TEXT, &sizeX, &sizeY);
  m_ValueStartY = height - sizeY;
  if (m_ShowDistance) height -= sizeY + BASE_MARGIN; // room for distance value at the bottom

  if (m_ShowSectorToolbar) {
    dc.GetTextExtent(_T("99"), &sizeX, &sizeY); // max size of button labels
    ::Maximise(sizeX, sizeY);                   // make square buttons
    m_ButtonSize = sizeX + 8;                   // 4 pixels at each side, for padding and button border

    width -= m_ButtonSize + BASE_MARGIN;        // room for sector buttons at the right
    m_ButtonStart.x = width;
    m_ButtonStart.y = BASE_MARGIN;
  }

  width -= 2 * MAP_MARGIN;
  height -= 2 * MAP_MARGIN;

  if (MGR->IsEmpty()) {
    // no laps loaded
    m_MinX = 0.0f;
    m_MaxY = 1.0f;
    m_Scale = 1.0f;
    m_MarginX = MAP_MARGIN;
    m_MarginY = MAP_MARGIN;
    m_SelStart = 0.0f;
    m_SelEnd = 0.0f;
    return;
  }

  cLap* lap = MGR->GetLeader();
  CHECK_PTR(lap);
  wxASSERT(lap->GetStateCount() > 0);
  m_MinX = lap->GetMinX();
  m_MaxY = lap->GetMaxY();
  float rangeX = lap->GetMaxX() - m_MinX;
  float rangeY = m_MaxY - lap->GetMinY();
  float scaleX = width / rangeX;
  float scaleY = height / rangeY;
  m_Scale = (scaleX < scaleY) ? scaleX : scaleY;
  m_MarginX = MAP_MARGIN + (width - (wxCoord)(rangeX * m_Scale)) / 2;
  m_MarginY = MAP_MARGIN + (height - (wxCoord)(rangeY * m_Scale)) / 2;
}

//-----------------------------------------------------------------------------
// Plot the trackmap

void cTrackMap::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  wxPaintDC dc(this);
  dc.SetFont(*wxNORMAL_FONT);

  if (MGR->IsEmpty()) {
    // empty track map
    m_SelStart = IMPOSSIBLY_LOW_VALUE;
    m_SelEnd = IMPOSSIBLY_HIGH_VALUE;
    m_ClickPos = -1;
    m_ClickMarker = -1;
    return;
  }

  DoPaint(dc);

  if (m_ShowSectorToolbar) {
    // draw the sector buttons
    dc.SetPen(s_ButtonPen);
    dc.SetBrush(*wxTRANSPARENT_BRUSH);

    // draw "all track" button
    DrawSectorButton(dc, 0);
    ::DrawTextCentered(dc, _TT(ID_TXT_TM_ALL_SECTORS, "A"),
        m_ButtonStart.x + m_ButtonSize / 2 + 1, Button2CoordY(0) + m_ButtonSize / 2);

    // draw the other buttons
    for (int i = 0; i < m_SectorCount; i++) {
      DrawSectorButton(dc, i + 1);
      ::DrawTextCentered(dc, wxString::Format(_T("%d"), i + 1),
          m_ButtonStart.x + m_ButtonSize / 2 + 1, Button2CoordY(i + 1) + m_ButtonSize / 2);
    }
  }

  // re-plot the cursor
  if ((m_CursorDrawn) && IsExposed(m_CursorX, m_CursorY)) {
    m_CursorDrawn = false;
    PutTrackCursor(dc, m_CursorDist);
  }
}


void cTrackMap::DoPaint(wxDC& dc)
{
  cLap* lap = MGR->GetLeader();
  if (lap == NULL) return;
  CHECK_PTR(lap);
  wxASSERT(lap->GetStateCount() > 0);

  dc.SetLogicalFunction(wxCOPY);

  // draw the sector markers
  for (int i = 0; i < m_SectorCount - 1; i++) {
    if (m_DragMode && (i == m_ClickMarker)) {
      dc.SetPen(s_SectorDragPen);
    }
    else {
      dc.SetPen(s_SectorPen);
    }
    DrawSectorMarker(dc, m_SectorEnd[i]);
  }

  // get screen coordinates of first state
  const cCarState* state = lap->GetState(0);
  wxCoord lastX = Track2CoordX(state->GetPosX());
  wxCoord lastY = Track2CoordY(state->GetPosY());

  // plot the lap's trajectory
  size_t step = 1 + (lap->GetStateCount() / MAX_SEGMENTS);
  for (size_t s = step; s < lap->GetStateCount(); s += step) {
    state = lap->GetState(s);
    wxCoord stateX = Track2CoordX(state->GetPosX());
    wxCoord stateY = Track2CoordY(state->GetPosY());

    if ((state->GetDistance() < m_SelStart) || (state->GetDistance() > m_SelEnd)) {
      dc.SetPen(s_Pen_UnSel);
    }
    else {
      dc.SetPen(s_Pen_Sel);
    }
    dc.DrawLine(lastX, lastY, stateX, stateY);

    lastX = stateX;
    lastY = stateY;
  }

  // draw the start/finish line (after the trajectory, so it won't be overwritten)
  dc.SetPen(s_SfPen);
  DrawSectorMarker(dc, 0.0f);

  // plot the distance value
  if (m_ShowDistance && (m_CursorDist >= 0.0f)) {
    dc.SetFont(*wxNORMAL_FONT);
    wxString value;
    value.Printf(_T("%.0f"), m_CursorDist);
    DrawTextRightAligned(dc, value, GetClientSize().GetWidth() - BASE_MARGIN, m_ValueStartY);
  }
}

//-----------------------------------------------------------------------------
// Convert track coordinates to window coordinates

wxCoord cTrackMap::Track2CoordX(float tc)
{
  return m_MarginX + (wxCoord)((tc - m_MinX) * m_Scale);
}

wxCoord cTrackMap::Track2CoordY(float tc)
{
  return m_MarginY + (wxCoord)((m_MaxY - tc) * m_Scale);
}

//-----------------------------------------------------------------------------
// Place the cursor at a new position
// - distance = distance_in_lap (in m) that corresponds to the cursor position
// - type = type of cursor-set command
// If distance < 0 then cursor is removed

void cTrackMap::DoSetTrackCursorPos(cGraphView* WXUNUSED(view), float distance, int type)
{
  if (MGR->IsEmpty()) return; // empty display

  if ((m_DragMode) && (m_ClickMarker < 0)) return; // don't show cursor while drag-selecting
  if ((m_DragMode) && (type == CURSORSET_PLAYBACK)) return; // ignore playback while dragging

  // draw the cursor at its new position
  wxClientDC dc(this);
  RemoveTrackCursor(dc);
  PutTrackCursor(dc, distance);

  // update value
  if (m_ShowDistance) {
    RefreshRect(wxRect(0, m_ValueStartY, GetClientSize().GetWidth(), GetClientSize().GetHeight()));
    Update(); // repaint immediately
  }
}

//-----------------------------------------------------------------------------
// Draw the cursor at a new position
// - dc = device context to draw in
// - distance = distance-in-lap for cursor position

void cTrackMap::PutTrackCursor(wxDC& dc, float distance)
{
  wxASSERT(!m_CursorDrawn);

  m_CursorDist = distance;
  if (distance < 0) return;

  cLap* lap = MGR->GetLeader();
  if (lap == NULL) return;
  CHECK_PTR(lap);
  float x, y;
  lap->GetPositionAt(distance, x, y);

  m_CursorX = Track2CoordX(x);
  m_CursorY = Track2CoordY(y);
  DrawTrackCursor(dc, m_CursorX, m_CursorY);
  m_CursorDrawn = true;
}

//-----------------------------------------------------------------------------
// Remove the cursor from its current position

void cTrackMap::RemoveTrackCursor(wxDC& dc)
{
  if (!m_CursorDrawn) return;
  DrawTrackCursor(dc, m_CursorX, m_CursorY);
  m_CursorDrawn = false;
}

//-----------------------------------------------------------------------------
// Actually draw the cursor
// - dc = device context to draw in
// - x, y = coordinates of the cursor position

void cTrackMap::DrawTrackCursor(wxDC& dc, wxCoord x, wxCoord y)
{
  dc.SetPen(*wxWHITE_PEN);
  dc.SetBrush(*wxWHITE_BRUSH);
  dc.SetLogicalFunction(wxINVERT);
  dc.DrawCircle(x, y, 3);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnSize(wxSizeEvent& event)
{
  AdjustDimensions();
  Refresh(); // force full redraw when window is resized

  event.Skip();
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseLeftClick(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler
  if (MGR->IsEmpty()) return; // empty track map

  int button = -1;
  m_ClickPos = DecodeMouse(event, &button);
  m_ClickMarker = Dist2Marker(m_ClickPos);
  if (m_LockSectors) m_ClickMarker = -1; // ignore clicks on markers if sectors are locked for editing

  // handle click on sector button
  if (button < 0) return; // clicks on other parts are handled on release of button
  if (button == 0) {
    // first button = select whole track
    m_SelStart = 0.0f;
    m_SelEnd = MGR->GetTrackLength();
  }
  else {
    // second button = first sector, third button = second sector, etc.
    wxASSERT(button <= m_SectorCount);
    m_SelStart = (button == 1) ? 0.0f : m_SectorEnd[button - 2];
    m_SelEnd = m_SectorEnd[button - 1];
  }

  // re-draw the trajectory
  wxClientDC dc(this);
  DoPaint(dc);

  // synchronise other panes/graphs
  ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseLeftDoubleClick(wxMouseEvent& event)
{
  int button = -1;
  float dist = DecodeMouse(event, &button);

  if (button >= 0) return; // click on a button: do nothing

  if (dist < 0.0f) {
    // click outside trajectory: reset zoom
    ResetZoom();
    return;
  }

  // click on trajectory: select sector
  int sector = Dist2Sector(dist);
  if (sector < 0) return;
  float start = (sector == 0) ? 0.0f : m_SectorEnd[sector - 1];
  float end = m_SectorEnd[sector];
  ::SetTrackSelection_Send(start, end);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseMove(wxMouseEvent& event)
{
  int button = -1;
  float dist = DecodeMouse(event, &button);
  if (m_ShowSectorToolbar) SelectButton(button);

  if (dist < 0) {
    // mouse not near track line
    SetCursor(wxNullCursor); // indicate that clicks will be ignored
    if (m_DragMode) ::SetTrackCursorPos_Send(-1.0f, CURSORSET_FOLLOW); // hide track cursor when dragging
    return;
  }

  SetCursor(wxCursor(wxCURSOR_HAND)); // indicate that clicks will be recognised

  if (!event.LeftIsDown()) return; // not drag-selecting
  if (m_ClickPos < 0) return; // last click was not near the track line

  // enter drag mode
  m_DragMode = true;

  wxClientDC dc(this);
  RemoveTrackCursor(dc);

  if (m_ClickMarker < 0) {
    // drag-selecting a part of the track: change selection
    if (fabs(dist - m_ClickPos) >= MIN_SELECTION_LENGTH) {
      if (dist < m_ClickPos) {
        m_SelStart = dist;
        m_SelEnd = m_ClickPos;
      }
      else {
        m_SelStart = m_ClickPos;
        m_SelEnd = dist;
      }
    }
  }

  // re-draw the trajectory
  DoPaint(dc);

  // let track cursor follow mouse pointer
  ::SetTrackCursorPos_Send(dist, CURSORSET_FOLLOW);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseLeftRelease(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler

  if (MGR->IsEmpty()) return; // empty track map
  if (m_ClickPos < 0) return; // last click was not near the track line

  if (m_DragMode) {
    // end drag mode
    m_DragMode = false;

    if (m_ClickMarker < 0) {
      // drag-selecting a part of the track: set the track selection in the graphs
      if (m_SelStart > m_SelEnd) ::SwapValues(m_SelStart, m_SelEnd);
      ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
    }
    else {
      // dragging a marker: move it to new position
      float newPos = DecodeMouse(event);
      float oldPos = m_SectorEnd[m_ClickMarker];
      DeleteSector(m_ClickMarker);
      if (AddSector(newPos)) {
        // move selection if it coincided with marker
        if (m_SelStart == oldPos) m_SelStart = newPos;
        if (m_SelEnd == oldPos) m_SelEnd = newPos;
        if (m_SelStart > m_SelEnd) ::SwapValues(m_SelStart, m_SelEnd);
        ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
      }
      else {
        // move failed - restore old position
        AddSector(oldPos);
      }
      Refresh();
    }
  }
  else {
    // simple click+release
    if (event.ShiftDown()) {
      // modify selected part
      if (m_ClickPos < m_SelStart) {
        m_SelEnd = m_SelStart;
        m_SelStart = m_ClickPos;
      }
      else {
        m_SelEnd = m_ClickPos;
      }
      ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
    }
    else {
      // set track cursor at this point on the track
      ::SetTrackCursorPos_Send(m_ClickPos, CURSORSET_FORCE);
    }
  }

  m_ClickPos = -1; // 'forget' last click
}

//-----------------------------------------------------------------------------
// Return the distance-in-lap that corresponds to a mouse position

float cTrackMap::DecodeMouse(wxMouseEvent& event, int* button)
{
  if (MGR->IsEmpty()) return -1;

  // get logical position of click
  wxClientDC dc(this);
  wxPoint pos = event.GetLogicalPosition(dc);

  // check the sector buttons
  if (button != NULL) {
    *button = -1;
    if ((pos.x >= m_ButtonStart.x) && (pos.x < m_ButtonStart.x + m_ButtonSize)) {
      *button = Coord2ButtonY(pos.y);
    }
    if (*button > 0) return -1.0f;
  }

  // find state that is nearest to this position
  float nearest_dist = -1; // distance-in-lap of state that is nearest to mouse position
  float nearest_pix = IMPOSSIBLY_HIGH_VALUE; // distance in pixels to that state
  cLap* lap = MGR->GetLeader();
  CHECK_PTR(lap);
  for (size_t s = 0; s < lap->GetStateCount(); s++) {
    const cCarState* state = lap->GetState(s);
    float dX = (float)(Track2CoordX(state->GetPosX()) - pos.x);
    float dY = (float)(Track2CoordY(state->GetPosY()) - pos.y);
    float d = sqrt((dX * dX) + (dY * dY));
    if (d < nearest_pix) {
      nearest_pix = d;
      nearest_dist = state->GetDistance();
    }
  }

  if (nearest_pix > CLICK_DETECT_THRESHOLD) return -1.0f;

  return nearest_dist;
}

//-----------------------------------------------------------------------------

void cTrackMap::DoSetTrackSelection(float start, float end)
{
  m_SelStart = start;
  m_SelEnd = end;

  if (MGR->IsEmpty()) return; // empty display

  wxClientDC dc(this);
  RemoveTrackCursor(dc);
  DoPaint(dc); // re-draw the trajectory
  PutTrackCursor(dc, m_CursorDist);
}

//-----------------------------------------------------------------------------

void cTrackMap::ResetZoom()
{
  if (MGR->IsEmpty()) return;

  m_SelStart = 0.0f;
  m_SelEnd = MGR->GetTrackLength();

  // re-draw the trajectory
  wxClientDC dc(this);
  DoPaint(dc);

  // synchronise other panes/graphs
  ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseWheel(wxMouseEvent& event)
{
  if (MGR->IsEmpty()) return;

  float diff;
  if (event.GetWheelRotation() > 0) {
    diff = -SCROLL_DIST;
    if (m_SelStart + diff < 0.0f) diff = -m_SelStart;
  }
  else {
    diff = SCROLL_DIST;
    if (m_SelEnd + diff > MGR->GetTrackLength()) diff = MGR->GetTrackLength() - m_SelEnd;
  }
  ::SetTrackSelection_Send(m_SelStart + diff, m_SelEnd + diff);
  if (m_CursorDist >= 0.0f) ::SetTrackCursorPos_Send(m_CursorDist + diff, CURSORSET_FOLLOW);
}

//-----------------------------------------------------------------------------

void cTrackMap::UpdateAll()
{
  wxString track;
  if (!MGR->IsEmpty()) track = MGR->GetTrackCode();
  if (m_TrackCode != track) {
    // track has changed
    SaveSectorSet(); // save current sectors

    // get new track data
    m_TrackCode = track;
    GetSectorSet();
    m_SectorsChanged = false;
  }

  AdjustDimensions();

  SetMinSize(wxSize(100, 100)); // TODO: set a more sensible size?
  Refresh();
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseRightClick(wxMouseEvent& event)
{
  if (m_DragMode) return; // don't show context menu while a drag is going on

  m_ClickPos = DecodeMouse(event);

  m_ClickMarker = -1;
  if (m_ClickPos < 0) {
    // click not on track
    m_Context.Enable(ID_MENU_SECTOR_ADD_DEL, (m_ClickPos >= 0.0f)); // disable adding/deleting sector
  }
  else {
    // find out if the click is near a sector marker
    m_ClickMarker = Dist2Marker(m_ClickPos);
    if (m_ClickMarker >= 0) m_ClickPos = m_SectorEnd[m_ClickMarker]; // "snap" click position to sector marker

    // disable submenu if click is on LAST sector marker (= finish line)
    m_Context.Enable(ID_MENU_SECTOR_ADD_DEL,
        !m_LockSectors && ((m_SectorCount == 0) || (m_ClickMarker < m_SectorCount - 1)));

    // set track cursor at this point on the track
    ::SetTrackCursorPos_Send(m_ClickPos, CURSORSET_FOLLOW);
  }

  if (m_ClickMarker < 0) {
    // click not on a marker - can insert one here
    m_Context.SetLabel(ID_MENU_SECTOR_ADD_DEL, _TT(ID_MENU_SECTOR_ADD_DEL, "Add sector"));
  }
  else {
    // click on a marker - can delete this one
    m_Context.SetLabel(ID_MENU_SECTOR_ADD_DEL, _TT(ID_TXT_TM_DEL_SECTOR, "Delete sector"));
  }

  m_Context.Enable(ID_MENU_ZOOM_RESET,
      (!MGR->IsEmpty()) &&
      ((m_SelStart > 0.0f) || ((m_SelEnd > 0.0f) && (m_SelEnd < MGR->GetTrackLength()))));

  m_Context.Enable(ID_MENU_SECTOR_ADD_SELECTION,
      (!MGR->IsEmpty()) && !m_LockSectors &&
      ((m_SelStart > MIN_SECTOR_LENGTH) || (m_SelEnd < MGR->GetTrackLength() - MIN_SECTOR_LENGTH)));

  m_Context.Enable(ID_MENU_SECTOR_RESET, (!MGR->IsEmpty()) && !m_LockSectors);
  m_Context.Enable(ID_MENU_SECTOR_UNDO, m_SectorsChanged);
  m_Context.Check(ID_MENU_SECTOR_LOCK, m_LockSectors);
  m_Context.Check(ID_MENU_SHOWSECTORBUTTONS, m_ShowSectorToolbar);
  m_Context.Check(ID_MENU_SHOWDISTANCE, m_ShowDistance);

  PopupMenu(&m_Context);
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cTrackMap::OnMenuClick(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_ZOOM_RESET :
      ResetZoom();
      break;

    case ID_MENU_SHOWDISTANCE :
      m_ShowDistance = !m_ShowDistance;
      AdjustDimensions();
      Refresh();
      break;

    case ID_MENU_SHOWSECTORBUTTONS :
      m_ShowSectorToolbar = !m_ShowSectorToolbar;
      if (!m_ShowSectorToolbar) m_SelectedButton = -1;
      AdjustDimensions();
      Refresh();
      break;

    case ID_MENU_SECTOR_ADD_DEL :
      if (m_ClickMarker < 0) {
        AddSector(m_ClickPos);
      }
      else {
        DeleteSector(m_ClickMarker);
      }
      Refresh();
      m_SectorsChanged = true;
      break;

    case ID_MENU_SECTOR_AUTODETECT :
      SetAutodetectSectors();
      Refresh();
      m_SectorsChanged = true;
      break;

    case ID_MENU_SECTOR_UNDO :
      GetSectorSet();
      Refresh();
      m_SectorsChanged = false;
      break;

    case ID_MENU_SECTOR_RESET_CLEAR :
      ClearSectors();
      Refresh();
      m_SectorsChanged = true;
      break;

    case ID_MENU_SECTOR_RESET_TIMELINES :
      SetTimelineSectors();
      Refresh();
      m_SectorsChanged = true;
      break;

    case ID_MENU_SECTOR_ADD_SELECTION :
      SetSelectionSector();
      Refresh();
      m_SectorsChanged = true;
      break;

    case ID_MENU_SECTOR_LOCK :
      m_LockSectors = !m_LockSectors;
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cTrackMap::LoadConfig(wxRegConfig* config, const wxString& key)
{
  config->Read(key + _T("/showdistance"), &m_ShowDistance);
  config->Read(key + _T("/sectors/showtoolbar"), &m_ShowSectorToolbar);
  config->Read(key + _T("/sectors/lock"), &m_LockSectors);

  int i = 0;
  while (true) {
    wxString subkey;
    subkey.Printf(_T("/sectors/set%d"), i);
    wxString settings;
    if (!config->Read(key + subkey, &settings)) break;

    m_SectorSets.Add(settings);
    i += 1;
  }
}


void cTrackMap::SaveConfig(wxRegConfig* config, const wxString& key)
{
  config->Write(key + _T("/showdistance"), m_ShowDistance);
  config->Write(key + _T("/sectors/showtoolbar"), m_ShowSectorToolbar);
  config->Write(key + _T("/sectors/lock"), m_LockSectors);

  SaveSectorSet();
  for (size_t i = 0; i < m_SectorSets.GetCount(); i++) {
    wxString subkey;
    subkey.Printf(_T("/sectors/set%d"), i);
    config->Write(key + subkey, m_SectorSets[i]);
  }
}

//-----------------------------------------------------------------------------
// Convert button number to window coordinate and vice versa

wxCoord cTrackMap::Button2CoordY(int button) const
{
  return m_ButtonStart.y + button * (m_ButtonSize + BUTTON_SPACING);
}

int cTrackMap::Coord2ButtonY(wxCoord coord) const
{
  if (coord < m_ButtonStart.y) return -1;
  if ((coord - m_ButtonStart.y) % (m_ButtonSize + BUTTON_SPACING) > m_ButtonSize) return -1;

  int button = (coord - m_ButtonStart.y) / (m_ButtonSize + BUTTON_SPACING);
  return (button < GetButtonCount()) ? button : -1;
}

//-----------------------------------------------------------------------------
// Set the 'selected' button
// - button = button number (-1 = none)

void cTrackMap::SelectButton(int button)
{
  wxASSERT(m_ShowSectorToolbar);
  wxASSERT(button < GetButtonCount()); // button number must be valid

  if (button == m_SelectedButton) return; // no change

  wxClientDC dc(this);

  // remove current selection
  if (m_SelectedButton >= 0) DrawSectorButton(dc, m_SelectedButton, false);

  m_SelectedButton = button;

  // draw new selection
  if (m_SelectedButton >= 0) DrawSectorButton(dc, m_SelectedButton, true);
}

//-----------------------------------------------------------------------------
// Draw the 'selected' box around a button, with the current pen
// - dc = current device context
// - button = button number

void cTrackMap::DrawSectorButton(wxDC& dc, int button, bool highlight)
{
  wxASSERT(m_ShowSectorToolbar);
  wxASSERT((button >= 0) && (button < GetButtonCount())); // button number must be valid

  ::DrawButton(dc,
      wxRect(m_ButtonStart.x, Button2CoordY(button), m_ButtonSize, m_ButtonSize),
      highlight);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseEntering(wxMouseEvent& event)
{
  int button = -1;
  DecodeMouse(event, &button);
  if (m_ShowSectorToolbar) SelectButton(button);
}

void cTrackMap::OnMouseLeaving(wxMouseEvent& WXUNUSED(event))
{
  if (m_ShowSectorToolbar) SelectButton(-1);
}

//-----------------------------------------------------------------------------
// Draw the line for a sector marker, with the current pen
// - dc = current device context
// - dist = distance-in-lap for marker

void cTrackMap::DrawSectorMarker(wxDC& dc, float dist)
{
  cLap* lap = MGR->GetLeader();
  CHECK_PTR(lap);

  // get the corresponding state
  const cCarState* state = lap->GetState(lap->FindStateAt(dist));

  // get direction of marker line
  float len = (float)MARKER_LENGTH2 / m_Scale;
  float dx = state->GetRightVector().x * len; // X-distance
  float dy = state->GetRightVector().y * len; // Y-distance

  // draw the line
  dc.DrawLine(
      Track2CoordX(state->GetPosX() + dx), Track2CoordY(state->GetPosY() + dy),
      Track2CoordX(state->GetPosX() - dx), Track2CoordY(state->GetPosY() - dy));
}

//-----------------------------------------------------------------------------
// Delete all sector data

void cTrackMap::ClearSectors()
{
  m_SectorCount = 0;
  m_SelectedButton = -1;
  m_ClickMarker = -1;
}

//-----------------------------------------------------------------------------
// Insert a sector into the array
// - dist = distance-in-lap of end of new sector
// Returns false if value could not be inserted

bool cTrackMap::AddSector(float dist)
{
  wxASSERT(!MGR->IsEmpty());

  if (m_SectorCount >= MAX_SECTORS) return false;                     // too many sectors
  if (dist < MIN_SECTOR_LENGTH) return false;                         // too near to start line
  if (dist > MGR->GetTrackLength() - MIN_SECTOR_LENGTH) return false; // too near to finish line

  if (m_SectorCount == 0) {
    m_SectorCount = 2;
    m_SectorEnd[0] = dist;
    m_SectorEnd[1] = MGR->GetTrackLength();
  }
  else {
    // find the insertion point in the array
    int index = 0;
    while (dist > m_SectorEnd[index] - MIN_SECTOR_LENGTH) {
      if (fabs(dist - m_SectorEnd[index]) < MIN_SECTOR_LENGTH) return false; // too near to an existing value
      index += 1;
    }
    wxASSERT(index < m_SectorCount); // sanity check

    // move existing values and add new value
    for (int i = m_SectorCount; i > index; i--) {
      m_SectorEnd[i] = m_SectorEnd[i - 1];
    }
    m_SectorEnd[index] = dist;
    m_SectorCount += 1;
  }

  // for (int i = 0; i < m_SectorCount; i++) wxLogDebug(_T("sector %d = %.0f"), i, m_SectorEnd[i]);

  return true;
}

//-----------------------------------------------------------------------------
// Delete a sector
// - index = index of the sector in the array

void cTrackMap::DeleteSector(int index)
{
  wxASSERT(index >= 0);
  wxASSERT(index < m_SectorCount - 1); // illegal to delete last sector

  if (m_SectorCount == 2) {
    // can't have just one sector
    ClearSectors();
  }
  else {
    // move existing values
    for (int i = index; i < m_SectorCount - 1; i++) {
      m_SectorEnd[i] = m_SectorEnd[i + 1];
    }
    m_SectorCount -= 1;
  }

  // for (int i = 0; i < m_SectorCount; i++) wxLogDebug(_T("sector %d = %.0f"), i, m_SectorEnd[i]);
}

//-----------------------------------------------------------------------------
// Derive sectors from split times

void cTrackMap::SetTimelineSectors()
{
  cLap* lap = MGR->GetLeader();
  CHECK_PTR(lap);

  ClearSectors();
  if (!lap->IsComplete()) return; // incomplete laps have inconsistent split times
  for (int s = 0; s < lap->GetSplitCount() - 1; s++) {
    AddSector(lap->GetDistanceAt(lap->GetSplit(s)));
  }
}

//-----------------------------------------------------------------------------
// Derive sectors from driver behaviour

void cTrackMap::SetAutodetectSectors()
{
  cLap* lap = MGR->GetLeader();
  CHECK_PTR(lap);

  ClearSectors();

  wxArrayInt events;             // indexes of state with (possible) sector-start events

  // scan states for brake presses
  bool wasPressed = false;       // was the pedal pressed in the previous state?
  for (size_t s = 0; s < lap->GetStateCount(); s++) {
    const cCarState* state = lap->GetState(s);

    if (state->GetLogData(LOGTYPE_BRAKE, 0) > BRAKE_PEDAL_THRESHOLD) {
      if (!wasPressed) events.Add(s);
      wasPressed = true;
    }
    else {
      wasPressed = false;
    }
  }

  // scan states for throttle releases
  wasPressed = true;
  size_t sRelease = 0; // index of state at throttle-release
  for (size_t s = 0; s < lap->GetStateCount(); s++) {
    const cCarState* state = lap->GetState(s);

    if (state->GetLogData(LOGTYPE_THROTTLE, 0) > THROTTLE_PEDAL_THRESHOLD) {
      if ((!wasPressed) &&
          (state->GetLogData(LOGTYPE_GEAR, 0) <= lap->GetState(sRelease)->GetLogData(LOGTYPE_GEAR, 0))) {
        // NB: ignore upshifts
        events.Add(sRelease);
      }
      wasPressed = true;
    }
    else {
      if (wasPressed) sRelease = s;
      wasPressed = false;
    }
  }

  // add sectors
  float last = IMPOSSIBLY_LOW_VALUE; // time-in-lap of previous event
  for (size_t i = 0; i < events.GetCount(); i++) {
    const cCarState* state = lap->GetState(events[i]);
    if ((state->GetTime() - last) > SECTOR_DURATION_THRESHOLD) {
      AddSector(state->GetDistance() - SECTOR_MARGIN);
    }
    last = state->GetTime();
  }
}

//-----------------------------------------------------------------------------
// Read/Save the current sector data from/in m_SectorSets

bool cTrackMap::GetSectorSet()
{
  ClearSectors();
  if (m_TrackCode.IsEmpty()) return false; // no track

  // find the element that has sector data for the current track
  wxString settings;
  for (size_t i = 0; i < m_SectorSets.GetCount(); i++) {
    if (m_SectorSets[i].StartsWith(m_TrackCode + _T("|"))) {
      settings = m_SectorSets[i].AfterFirst(_T('|'));
      break;
    }
  }
  if (settings.IsEmpty()) return false; // no data found

  // decode the sector settings string
  wxArrayString keys;
  wxArrayString values;
  ::DecodeKeyValueList(settings, keys, values);
  for (size_t k = 0; k < keys.GetCount(); k++) {
    if (keys[k] == _T("format")) {
      if (wxAtoi(values[k]) != 1) return false; // unknown format
    }
    else if (keys[k] == _T("dist")) {
      AddSector(wxAtof(values[k]));
    }
  }
  return true;
}


void cTrackMap::SaveSectorSet()
{
  if (m_TrackCode.IsEmpty()) return; // no data

  // construct settings string
  wxString settings = m_TrackCode + _T("|format=1;");
  for (int i = 0; i < m_SectorCount; i++) {
    settings += wxString::Format(_T("dist=%f;"), m_SectorEnd[i]);
  }

  // find the element that has sector data for the current track
  for (size_t i = 0; i < m_SectorSets.GetCount(); i++) {
    if (m_SectorSets[i].StartsWith(m_TrackCode + _T("|"))) {
      // replace contents of element
      m_SectorSets[i] = settings;
      return;
    }
  }
  // not found - add it
  m_SectorSets.Add(settings);
}

//-----------------------------------------------------------------------------
// Use the current selection as sector

void cTrackMap::SetSelectionSector()
{
  AddSector(m_SelStart);
  AddSector(m_SelEnd);

  // remove all intermediate sectors
  for (int i = m_SectorCount - 1; i >= 0; i--) {
    if ((m_SectorEnd[i] > m_SelStart) && (m_SectorEnd[i] < m_SelEnd)) DeleteSector(i);
  }
}

//-----------------------------------------------------------------------------
// Determine the sector that corresponds with a distance-in-lap (-1 = none)

int cTrackMap::Dist2Sector(float dist)
{
  wxASSERT(!MGR->IsEmpty());
  wxASSERT(dist >= 0.0f);
  wxASSERT(dist <= MGR->GetTrackLength());

  if (dist < m_SectorEnd[0]) return 0;
  for (int i = 1; i < m_SectorCount; i++) {
    if ((dist >= m_SectorEnd[i - 1]) && (dist < m_SectorEnd[i])) return i;
  }
  return -1;
}

//-----------------------------------------------------------------------------
// Determine the sector marker that corresponds with a distance-in-lap (-1 = none)

int cTrackMap::Dist2Marker(float dist)
{
  for (int i = 0; i < m_SectorCount; i++) {
    if (fabs(dist - m_SectorEnd[i]) < MIN_SECTOR_LENGTH) return i;
  }
  return -1;
}

//-----------------------------------------------------------------------------
// Translation

void cTrackMap::TranslateTexts()
{
  cLang::TranslateMenu(&m_Context);
}
