#include "clap.h"
#include "ccolourlist.h"
#include "ctrack.h"
#include "curl.h"
#include "cfiledlg.h"
#include "cmgr.h"
#include "ccsvdlg.h"
#include "capp.h"
#include <math.h>
#include <wx/filename.h>
#include <wx/stopwatch.h>
#include <wx/utils.h>

#define GCALC_RANGE 5

//-----------------------------------------------------------------------------
// Define an array of cLap

#include <wx/arrimpl.cpp>
WX_DEFINE_OBJARRAY(cLapArray);

//-----------------------------------------------------------------------------
// - fileName = full filename
// - url = URL to download the file from (may be empty)

cLap::cLap(const wxString& fileName, const wxString& url)
{
  wxASSERT(wxFileExists(fileName) || !url.IsEmpty());
  wxASSERT(url.IsEmpty() || url.StartsWith(_T("http://")));

  m_FileName = fileName;
  m_UrlName = url;

  if (!url.IsEmpty()) {
    // construct filename for downloaded RAF file
    m_LfswName = cFileDlg::LfswUrl2LapName(url);
    wxFileName wxfn(MGR->GetRafDir(), m_LfswName, _T("raf"));
    m_FileName = wxfn.GetFullPath();
  }

  m_IsComplete = false;
  m_Naming = LAPNAME_NONE;
  SetNaming(LAPNAME_FILE);

  m_TrackLength = 0.0f;
  m_Status = LAPSTATUS_CREATED;
  m_Shown = false;

  m_MinX = IMPOSSIBLY_HIGH_VALUE;
  m_MaxX = IMPOSSIBLY_LOW_VALUE;
  m_MinY = IMPOSSIBLY_HIGH_VALUE;
  m_MaxY = IMPOSSIBLY_LOW_VALUE;

  m_Colour = cColourList::Alloc();
}

//-----------------------------------------------------------------------------

cLap::~cLap()
{
  cColourList::Free(m_Colour);
}

//-----------------------------------------------------------------------------
// Get the display name
// - naming = naming scheme to use

wxString cLap::GetName(lapname_t naming) const
{
  CHECK_THIS;
  wxASSERT(naming != LAPNAME_NONE);
  wxASSERT(!m_FileName.IsEmpty());

  if (naming == m_Naming) return m_Name;

  wxString name;
  switch (naming) {
    case LAPNAME_FILE :
      wxFileName::SplitPath(m_FileName, NULL, NULL, &name, NULL);
      break;

    case LAPNAME_TIME :
      wxASSERT(IsLoaded());
      name = ::FormatTime(m_Split[m_SplitCount - 1]);
      break;

    case LAPNAME_TIME_DRIVER :
      wxASSERT(IsLoaded());
      name.Printf(_T("%s  %s"),
          ::FormatTime(m_Split[m_SplitCount - 1]).c_str(),
          m_PlayerName.c_str());
      break;

    case LAPNAME_TIME_CAR_DRIVER :
      wxASSERT(IsLoaded());
      name.Printf(_T("%s  %s - %s"),
          ::FormatTime(m_Split[m_SplitCount - 1]).c_str(),
          m_CarCode.c_str(),
          m_PlayerName.c_str());
      break;

    default:
      wxFAIL;
  }

  return name;
}

//-----------------------------------------------------------------------------
// Change the display naming

void cLap::SetNaming(lapname_t naming)
{
  wxASSERT(naming != LAPNAME_NONE);
  m_Name = GetName(naming);
  m_Naming = naming;
}

//-----------------------------------------------------------------------------
// Change the file name

void cLap::SetFileName(const wxString& newName)
{
  wxASSERT(!IsOpened());

  m_FileName = newName;
  SetNaming(m_Naming); // refresh name
}

//-----------------------------------------------------------------------------
// Start the loading of a RAF file
// Returns false if unable to load

bool cLap::LoadStart()
{
  wxASSERT(!IsOpened());
  wxASSERT(m_Status == LAPSTATUS_CREATED);

  if (!m_UrlName.IsEmpty()) {
    // first download the RAF file
    cUrl urlReader(m_UrlName);
    if (!urlReader.ReadToFile(m_FileName)) {
      LoadFailed(_T("Could not download file"));
    }
  }

  m_Status = LAPSTATUS_LOADING;

  if (!Open(m_FileName)) return LoadFailed(_T("Could not open file"));

  // NB: RAF file format is documented on http://www.liveforspeed.net/?page=RAF

  // ----- initialization -----
  for (int i = 0; i < 4; i++) m_AvgTemp[i] = 0.0f;

  // ----- read the header -----
  wxString str;
  ReadString(str, 6); // read fixed header string 'LFSRAF'
  if (str != "LFSRAF") return LoadFailed(_T("Not a RAF file"));

  Skip(2); // game version, game revision

  wxInt8 format;
  ReadByte(format);
  if (format != 2) return LoadFailed(wxString::Format(_T("Unknown RAF format version (%d)"), format));

  wxInt8 interval;
  ReadByte(interval);
  if (interval > 0) {
    m_TimeInterval = interval / 1000.0f; // convert from ms to seconds
  }
  else {
    // no valid interval (S2 patch Y and earlier)
    m_TimeInterval = 0.01f; // default: 100 states per second
  }

  Skip(2); // empty

  ReadWord(m_HeaderSize);
  ReadWord(m_BlockSize);
  ReadWord(m_WheelBlockSize);
  ReadWord(m_WheelBlockOffset);
  // check block sizes (for the current format version)
  wxASSERT(m_HeaderSize == 1024);
  wxASSERT(m_BlockSize == 192);
  wxASSERT(m_WheelBlockSize == 32);
  wxASSERT(m_WheelBlockOffset == 64);

  ReadInt(m_BlockCount);

  // sanity check on block count: max laptime between 0 and 30 minutes
  if ((m_BlockCount <= 0) || (m_BlockCount > 100*60*30))
      return LoadFailed(wxString::Format(_T("File format error (block count = %d)"), m_BlockCount));

  ReadString(m_TrackCode, 4); // will get empty string for S2 patch W and earlier
  ReadFloat(m_TrackLength);
  ReadString(m_PlayerName, 32);
  UnEscapeLFS(m_PlayerName);

  ReadString(m_Car.m_Name, 32);
  ReadString(m_TrackName, 32);
  ReadString(m_ConfigName, 16);

  // determine car code
  m_CarCode = cCar::Name2Code(m_Car.m_Name);

  // determine track code, and canonicalize track+config name
  wxString code = cTrack::Name2Code(m_TrackName, m_ConfigName);
  if ((code.IsEmpty()) && (m_TrackCode.IsEmpty()))
      return LoadFailed(wxString::Format(_T("Unknown track (%s %s)"), m_TrackName.c_str(), m_ConfigName.c_str()));

  if (m_TrackCode.IsEmpty()) m_TrackCode = code;
  // NB: since S2 patch X the track code can be read from the RAF file, but
  // we still need to decode it, to support older versions and to canonicalize the track+config name
  // However, we'll try to detect any differences
  wxASSERT(code == m_TrackCode);

  // sanity check on track length
  if (m_TrackLength <= 0.0f)
      return LoadFailed(wxString::Format(_T("Can't handle files for track %s"), m_TrackCode.c_str()));

  m_FullTrackName = m_TrackName + _T(" " ) + m_ConfigName;

  ReadString(m_Weather, 16);
  ReadString(m_LfsVersion, 8);
  ReadByte(m_PlayerFlags);

  wxInt8 wheels;
  ReadByte(wheels);
  if (wheels != 4) return LoadFailed(_T("Can only handle cars with 4 wheels"));

  ReadByte(m_Hlvc);
  ReadByte(m_SplitCount);
  for (int i = 0; i < 4; i++) ReadInt(m_Split[i]);

  // ----- read static car data -----
  GoTo(188);
  Skip(4 + 4); // mass, unsprung mass
  ReadFloat(m_Car.m_AntiRoll_Rear);
  ReadFloat(m_Car.m_AntiRoll_Front);
  ReadFloat(m_Car.m_FinalDrive);
  ReadByte(m_Car.m_GearCount);
  Skip(3);
  for (int i = 0; i < MAX_GEAR_RATIOS; i++) ReadFloat(m_Car.m_GearRatio[i]);

  // ----- read static wheel data -----
  for (int w = 0; w < 4; w++) {
    GoTo(512 + w * 128);

    float x, y;
    ReadFloat(x);
    ReadFloat(y);
    Skip(4); // Z position

    m_Wheelmap[w] = 0;
    if (x > 0) m_Wheelmap[w] += 1; // this is a right wheel
    if (y < 0) m_Wheelmap[w] += 2; // this is a rear wheel

    m_Car.m_WheelX[m_Wheelmap[w]] = x;
    m_Car.m_WheelY[m_Wheelmap[w]] = y;

    ReadFloat(m_Car.m_Radius[m_Wheelmap[w]]);
    Skip(4); // width
    ReadFloat(m_Car.m_MaxDeflect[m_Wheelmap[w]]);
    Skip(4 + 1);
    ReadByte(m_Car.m_TyreType[m_Wheelmap[w]]);
    Skip(2);
    ReadFloat(m_Car.m_Spring[m_Wheelmap[w]]);
    ReadFloat(m_Car.m_Damp_Bump[m_Wheelmap[w]]);
    ReadFloat(m_Car.m_Damp_Rebound[m_Wheelmap[w]]);
    ReadFloat(m_Car.m_Brake[m_Wheelmap[w]]);
  }

  // check validity of wheel mapping
  wxASSERT(m_Wheelmap[0] != m_Wheelmap[1]);
  wxASSERT(m_Wheelmap[0] != m_Wheelmap[2]);
  wxASSERT(m_Wheelmap[0] != m_Wheelmap[3]);
  wxASSERT(m_Wheelmap[1] != m_Wheelmap[2]);
  wxASSERT(m_Wheelmap[1] != m_Wheelmap[3]);
  wxASSERT(m_Wheelmap[2] != m_Wheelmap[3]);

  // ----- prepare to read the states -----
  m_PrevTime = IMPOSSIBLY_LOW_VALUE;
  m_Log.Alloc(m_BlockCount);
  m_BlockIndex = 0;

  m_Status = LAPSTATUS_LOADING;
  return true;
}

//-----------------------------------------------------------------------------
// Load the next batch of data
// - ms = allowed time (in ms) to go on reading
// Returns false if error or ready

bool cLap::LoadNext(int ms)
{
  wxASSERT(m_Status == LAPSTATUS_LOADING);
  wxStopWatch sw;

  // load more states
  while (m_BlockIndex < m_BlockCount) {
    if (Eof()) return LoadFailed(_T("Unexpected end-of-file"));

    GoTo(m_HeaderSize + m_BlockIndex * m_BlockSize); // go to start of block

    cCarState* state = new cCarState(&m_Car);
    ReadFloat(state->m_Throttle);
    ReadFloat(state->m_Brake);
    ReadFloat(state->m_Steer);
    ReadFloat(state->m_Clutch);
    ReadFloat(state->m_Handbrake);
    ReadByte(state->m_Gear);
    Skip(3); // empty
    ReadFloat(state->m_Speed);
    Skip(4); // car distance

    wxInt32 pos;
    ReadInt(pos); state->m_Pos.x = (float)pos / 65536.0f;
    ReadInt(pos); state->m_Pos.y = (float)pos / 65536.0f;
    ReadInt(pos); state->m_Pos.z = (float)pos / 65536.0f;

    ReadFloat(state->m_Rpm);
    ReadFloat(state->m_Distance);

    if (m_BlockIndex == 0) {
      // initialise time-in-lap by estimating from distance and current speed
      state->m_TimeInLap = state->m_Distance / state->m_Speed;
    }
    else {
      state->m_TimeInLap = m_PrevTime + m_TimeInterval;
    }

    wxInt16 dir;
    ReadWord(dir); state->m_Right.x = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Right.y = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Right.z = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Fwd.x = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Fwd.y = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Fwd.z = (float)dir / 32767.0f;
    state->m_Right.Normalize(); // vectors should have unit length
    state->m_Fwd.Normalize();

    // read dynamic wheel data
    for (int w = 0; w < 4; w++) {
      GoTo(m_HeaderSize + m_BlockIndex * m_BlockSize + m_WheelBlockOffset + w * m_WheelBlockSize);
      ReadFloat(state->m_Deflect[m_Wheelmap[w]]);
      ReadFloat(state->m_WheelAngle[m_Wheelmap[w]]);
      ReadFloat(state->m_ForceLat[m_Wheelmap[w]]);
      ReadFloat(state->m_ForceLong[m_Wheelmap[w]]);
      ReadFloat(state->m_Load[m_Wheelmap[w]]);
      ReadFloat(state->m_AngVel[m_Wheelmap[w]]);
      ReadFloat(state->m_Camber[m_Wheelmap[w]]);

      wxInt8 temp;
      ReadByte(temp);
      m_AvgTemp[m_Wheelmap[w]] += temp;

      ReadByte(temp); // will be zero for S2 patch Y and earlier
      int slipfraction = (temp < 0) ? (temp + 256) : temp; // convert from signed byte to value 0 - 255
      wxASSERT((slipfraction >= 0) && (slipfraction <=255));
      state->m_TractionBudget[m_Wheelmap[w]] = 1.0f - (float)slipfraction / 255.0f;
    }

    wxASSERT(m_Log.GetCount() == (size_t)m_BlockIndex);
    if (Eof()) return LoadFailed(_T("Unexpected end-of-file"));
    m_Log.Add(state);

    // calculated state data
    ::Minimise(m_MinX, state->GetPosX());
    ::Maximise(m_MaxX, state->GetPosX());
    ::Minimise(m_MinY, state->GetPosY());
    ::Maximise(m_MaxY, state->GetPosY());

    // calculated state data that needs data from other states
    int calc, prev, next; // index of state to calculate for, and the previous and next states
    if (m_BlockIndex <= GCALC_RANGE) {
      // first couple of states: calculation not needed
    }
    else if (m_BlockIndex < m_BlockCount - 1) {
      // intermediate states: normal calculation
      next = m_BlockIndex;
      calc = m_BlockIndex - GCALC_RANGE;
      wxASSERT(calc >= 1);
      prev = calc - GCALC_RANGE;
      ::Maximise(prev, 0);
      m_Log[calc].CalculateDerivedData(m_Log[prev], m_Log[next]);
    }
    else if (m_BlockIndex == m_BlockCount - 1) {
      // last state: calculate for the last couple of states
      for (calc = m_BlockIndex - GCALC_RANGE; calc < m_BlockIndex; calc++) {
        prev = calc - GCALC_RANGE;
        ::Maximise(prev, 0); // defensive
        m_Log[calc].CalculateDerivedData(m_Log[prev], m_Log[m_BlockIndex]);
      }
    }

    // prepare for next block
    m_BlockIndex += 1;
    m_PrevTime = state->m_TimeInLap;

    if (sw.Time() > ms) return true; // time's up!
  }

  // done reading the file data
  Close();

  // determine completeness
  float laptime = GetLaptime();
  if ((laptime >= (m_BlockCount - 1) * m_TimeInterval) &&
      (laptime <= (m_BlockCount + 1) * m_TimeInterval)) {
    m_IsComplete = true;
  }

  // sanity check on m_Distance
  if (m_Log[0].m_Distance >= m_Log[m_Log.GetCount() - 1].m_Distance)
      return LoadFailed(_T("File format error (descending distance-in-lap)"));

  // take average of tyre temp
  for (int i = 0; i < 4; i++) m_AvgTemp[i] = m_AvgTemp[i] / m_Log.GetCount();

  // determine if this file is on LFS World
  if (m_LfswName.IsEmpty()) {
    // check if file name conforms to LFSW naming pattern
    wxFileName wxfn(m_FileName);
    wxString file = wxfn.GetName();
    wxString user, car, track;
    long lapTime;
    cFileDlg::LfswName2LapData(file, user, car, track, &lapTime);
    if ((car == m_CarCode) && (track == m_TrackCode) && (lapTime == m_Split[m_SplitCount - 1])) {
      // conforms to naming pattern, with same car+track+laptime -> probably
      m_LfswName = cFileDlg::LapData2LfswName(user, car, track, lapTime);
    }
  }
  if (m_Hlvc != 1) m_LfswName = wxEmptyString; // not HLVC legal -> certainly not

  // find the corresponnding replay
  FindReplayFile();

  // loading successful
  wxASSERT(m_LoadError.IsEmpty());
  wxASSERT(m_Status == LAPSTATUS_LOADING);
  m_Status = LAPSTATUS_LOAD_OK;
  m_Shown = true;

  return false; // reading complete
}

//-----------------------------------------------------------------------------
// Utility function to signal an error in loading
// - error = error description
// Always return false

bool cLap::LoadFailed(wxString error)
{
  wxASSERT(m_Status == LAPSTATUS_LOADING);
  m_Status = LAPSTATUS_LOAD_FAIL;
  m_LoadError = error;
  Close();
  return false;
}

//-----------------------------------------------------------------------------
// Find the state at (or just before) a certain distance
// - distance = distance-in-lap
// Returns the index of the state

size_t cLap::FindStateAt(float distance) const
{
  wxASSERT(m_Status == LAPSTATUS_LOAD_OK);

  // use binary search
  size_t lo = 0;
  size_t hi = m_Log.GetCount() - 1;

  if (distance <= GetState(lo)->GetDistance()) return lo;
  if (distance >= GetState(hi)->GetDistance()) return hi;

  wxASSERT(GetState(lo)->GetDistance() <= GetState(hi)->GetDistance());

  while (hi - lo >= 2) {
    size_t mid = (hi + lo) / 2;
    if (GetState(mid)->GetDistance() < distance) {
      lo = mid;
    }
    else {
      hi = mid;
    }
  }
  return lo;
}

//-----------------------------------------------------------------------------
// Find the car's position at a certain distance
// - distance = distance-in-lap
// - x, y = receive the X and Y coordinates

void cLap::GetPositionAt(float distance, float& x, float& y) const
{
  wxASSERT(IsLoaded());

  size_t index = FindStateAt(distance);
  const cCarState* state1 = GetState(index);
  const cCarState* state2 = (index >= m_Log.GetCount() - 1) ? state1 : GetState(index + 1);

  float lambda = 0.0f; // interpolation scalar
  float dist1 = state1->GetDistance();
  float dist2 = state2->GetDistance();
  if (dist2 - dist1 > EPSILON) lambda = (distance - dist1) / (dist2 - dist1);

  x = (1.0f - lambda) * state1->GetPosX() + lambda * state2->GetPosX();
  y = (1.0f - lambda) * state1->GetPosY() + lambda * state2->GetPosY();
}

//-----------------------------------------------------------------------------
// Get the car state data at a certain distance
// - distance = distance-in-lap
// - log = logtype
// - wheel = wheel type

float cLap::GetLogDataAt(float distance, int log, int wheel) const
{
  wxASSERT(IsLoaded());
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  size_t index = FindStateAt(distance);

  // distance and value of this state
  float dist1 = GetState(index)->GetDistance();
  float val1 = GetState(index)->GetLogData(log, wheel);

  if (index == m_Log.GetCount() - 1)
    return val1;

  // distance and value of the next state
  float dist2 = GetState(index + 1)->GetDistance();
  float val2 = GetState(index + 1)->GetLogData(log, wheel);

  // calculate value
  float lambda = 0.0f; // interpolation scalar
  if (dist2 - dist1 > EPSILON) lambda = (distance - dist1) / (dist2 - dist1);

  float result;
  if (cCarState::IsInteger(log)) {
    // integer logdata: take nearest state
    result = (lambda < 0.5f) ? val1 : val2;
  }
  else {
    // float logdata: calculate by interpolation
    result = (1.0f - lambda) * val1 + lambda * val2;
  }

  return result;
}

//-----------------------------------------------------------------------------
// Find the distance-in-lap at a certain time
// - time = time from start of lap

float cLap::GetDistanceAt(float time)
{
  wxASSERT(m_Status == LAPSTATUS_LOAD_OK);

  // use binary search
  size_t lo = 0;
  size_t hi = m_Log.GetCount() - 1;

  if (time <= GetState(lo)->GetTime()) return GetState(lo)->GetDistance();
  if (time >= GetState(hi)->GetTime()) return GetState(hi)->GetDistance();

  wxASSERT(GetState(lo)->GetTime() <= GetState(hi)->GetTime());

  while (hi - lo >= 2) {
    size_t mid = (hi + lo) / 2;
    if (GetState(mid)->GetTime() < time) {
      lo = mid;
    }
    else {
      hi = mid;
    }
  }

  float time1 = GetState(lo)->GetTime();
  float time2 = GetState(hi)->GetTime();
  float lambda = (time - time1) / (time2 - time1); // interpolation scalar
  return (1.0f - lambda) * GetState(lo)->GetDistance() + lambda * GetState(hi)->GetDistance();
}

//-----------------------------------------------------------------------------
// Export lap data to a CSV file
// - fileName = full pathname fo file (will overwrite existing file!)
// - start, end = distance-in-lap for start and end of selected part of the lap
// Returns success or failure of operation

bool cLap::Export(const wxString& fileName, float start, float end)
{
  wxFile file;

  wxBusyCursor cursor; // activate hourglass

  // create file
  file.Create(fileName, true);
  if (!file.IsOpened()) return false;

  // read export settings
  wxString separator = APP->GetCsvSettings()->GetDecimalSeparator();
  wxString wrap = APP->GetCsvSettings()->GetFieldWrapCharacter();
  bool header = APP->GetCsvSettings()->GetHeaderLine();

  // write header line
  wxString line;
  if (header) {
    line = _T("\"d\"");
    for (int i = LOGTYPE_FIRST; i < LOGTYPE_LAST; i++) {
      wxString name = cCarState::GetLogtypeName(i);
      if (cCarState::IsPerWheel(i)) {
        // 4 columns, one for each wheel
        for (int w = 0; w < 4; w++) {
          line += wxString::Format(_T(",\"%s[%d]\""), name.c_str(), w);
        }
      }
      else {
        // 1 column
        line += wxString::Format(_T(",\"%s\""), name.c_str());
      }
    }
    if (!wrap.IsEmpty()) line.Replace(_T("\""), wrap);
    file.Write(line + "\n");
  }

  // write states
  for (size_t s = FindStateAt(start); s < m_Log.GetCount(); s++) {
    const cCarState* state = GetState(s);
    if (state->GetDistance() > end) break; // past end of selection

    line = wrap + wxString::Format(_T("%.1f"), state->GetDistance()) + wrap;
    for (int i = LOGTYPE_FIRST; i < LOGTYPE_LAST; i++) {
      if (cCarState::IsPerWheel(i)) {
        // 4 values, one for each wheel
        for (int w = 0; w < 4; w++) {
          line += _T(",") + wrap + wxString::Format(_T("%f"), state->GetLogData(i, w)) + wrap;
        }
      }
      else {
        // 1 value
        line += _T(",") + wrap + wxString::Format(_T("%f"), state->GetLogData(i, 0)) + wrap;
      }
    }
    line.Replace(_T("."), separator);
    file.Write(line + "\n");
  }

  // done
  file.Close();
  return true;
}

//-----------------------------------------------------------------------------
// Look for a replay file that contains this lap
// Returns success of search

bool cLap::FindReplayFile()
{
  wxASSERT(m_ReplayFile.IsEmpty());
  wxString replayDir = MGR->GetReplayDir();
  if (replayDir.IsEmpty()) return false;

  wxFileName replay(replayDir, wxEmptyString, _T("spr"));

  // build the list of possible names
  wxArrayString names;
  names.Add(m_LfswName);                                                // standard LFSW-style name
  names.Add(m_LfswName.BeforeLast(_T('0')));                            // standard LFSW-style name without last zero
  names.Add(m_FileName.AfterLast(wxFILE_SEP_PATH).BeforeLast(_T('.'))); // same file name as the RAF file

  // check the names for existence, top at the first hit
  for (size_t i = 0; i < names.GetCount(); i++) {
    replay.SetName(names[i]);
    wxASSERT(replay.IsOk());
    if (replay.FileExists()) {
      m_ReplayFile = replay.GetFullPath();
      return true;
    }
  }

  return false;
}
