#include "cfiledlg.h"
#include "cfile.h"
#include "clap.h"
#include "ctrack.h"
#include "cmgr.h"
#include "cpubstatdlg.h"
#include "curl.h"
#include <wx/dir.h>
#include <wx/datetime.h>
#include <wx/dirdlg.h>
#include <wx/filefn.h>
#include <wx/msgdlg.h>
#include <wx/textdlg.h>
#include <wx/config.h>
#include <wx/filename.h>
#include <wx/utils.h>
#include <wx/tokenzr.h>

// sorting indicators on columns (ascending/descending)
#define SORT_INDICATOR_ASC _T("> ")
#define SORT_INDICATOR_DESC _T("< ")

// base URL for loading Pubstat data from LFS World
#define LFSW_PUBSTAT _T("http://www.lfsworld.net/pubstat/get_stat2.php?version=1.4")

// base URL for loading RAF files from LFS World
#define LFSW_GET_RAF _T("http://www.lfsworld.net/get_raf2.php?file=")

// base URL for loading SPR files from LFS World
#define LFSW_GET_SPR _T("http://www.lfsworld.net/get_spr2.php?file=")

// timeouts on loading data from LFS World (in ms)
#define LFSW_DEFAULT_FILESIZE 8192
#define LFSW_TIMEOUT 10000
#define LFSW_SLEEP 100

//-----------------------------------------------------------------------------
// Enums

// columns in the local file list
enum ENUM_COLS_0 {
  COL_0_FILENAME = 0,
  COL_0_CARCODE,
  COL_0_TRACKCODE,
  COL_0_LAPTIME,
  COL_0_DATE,
  COL_0_RACER,
  COL_0_STEER,
  COL_0_RHD_LHD,
  COL_0_LFSVERSION,
  COL_0_SAMPLEFREQ
};

// columns in the LFSW file list
enum ENUM_COLS_1 {
  COL_1_CHARTPOS = 0,
  COL_1_USERNAME,
  COL_1_STEER,
  COL_1_RHD_LHD,
  COL_1_LAPTIME,
  COL_1_WR_PERC,
  COL_1_WR_DIST
};

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cFileDlg, wxDialog)
  EVT_BUTTON(wxID_ANY, cFileDlg::OnButton)
  EVT_CHECKBOX(wxID_ANY, cFileDlg::OnCheckBox)
  EVT_CHOICE(wxID_ANY, cFileDlg::OnChangeFilter)

  EVT_NOTEBOOK_PAGE_CHANGED(wxID_ANY, cFileDlg::OnPageChange)

  EVT_LIST_KEY_DOWN(wxID_ANY, cFileDlg::OnKeyDown)
  EVT_LIST_COL_CLICK(wxID_ANY, cFileDlg::OnColClick)
  EVT_LIST_ITEM_ACTIVATED(wxID_ANY, cFileDlg::OnItemActivate)
  EVT_LIST_ITEM_DESELECTED(wxID_ANY, cFileDlg::OnChangeSelection)
  EVT_LIST_ITEM_SELECTED(wxID_ANY, cFileDlg::OnChangeSelection)
  EVT_LIST_ITEM_RIGHT_CLICK(wxID_ANY, cFileDlg::OnItemRightClick)

  EVT_MENU(wxID_ANY, cFileDlg::OnMenuClick)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------
// - parent = parent window

cFileDlg::cFileDlg(wxWindow* parent)
: wxDialog(parent, -1, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER)
{
  m_Initialising = true;
  m_PubstatDialog = NULL;
  SetSize(600, 400); // default size to accommodate all controls

  // context menu on file list
  cLang::AppendMenuItem(&m_Context, ID_MENU_RENAME, _T("Rename"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_DELETE, _T("Delete"));
  m_Context.AppendSeparator();
  cLang::AppendMenuItem(&m_Context, ID_MENU_SELECTALL, _T("Select all\tCtrl+A"));

  // the notebook with the file lists
  m_Notebook = new wxNotebook(this, -1, wxDefaultPosition, wxDefaultSize, wxNO_BORDER | wxNB_NOPAGETHEME);
  for (int i = 0; i < 2; i++) {
    m_Page[i] = new wxPanel(m_Notebook, wxID_ANY);
    m_Page[i]->SetBackgroundColour(this->GetBackgroundColour());
    m_SortKey[i] = 1;
  }

  // list of local files
  m_List[0] = new wxListView(m_Page[0], ID_CTRL_FD_FILE_LIST_LOCAL, wxDefaultPosition, wxDefaultSize, wxLC_REPORT | wxSIMPLE_BORDER);
  m_List[0]->InsertColumn(COL_0_FILENAME, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_CARCODE, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_TRACKCODE, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_LAPTIME, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_DATE, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_RACER, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_STEER, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_RHD_LHD, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_LFSVERSION, wxEmptyString);
  m_List[0]->InsertColumn(COL_0_SAMPLEFREQ, wxEmptyString);

  m_DeleteBtn = new wxButton(m_Page[0], ID_CTRL_FD_DELETE, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);
  m_RenameBtn = new wxButton(m_Page[0], ID_CTRL_FD_RENAME, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);

  // list of LFSW files
  m_List[1] = new wxListView(m_Page[1], ID_CTRL_FD_FILE_LIST_LFSW, wxDefaultPosition, wxDefaultSize, wxLC_REPORT | wxSIMPLE_BORDER);
  m_List[1]->InsertColumn(COL_1_CHARTPOS, wxEmptyString);
  m_List[1]->InsertColumn(COL_1_USERNAME, wxEmptyString);
  m_List[1]->InsertColumn(COL_1_STEER, wxEmptyString);
  m_List[1]->InsertColumn(COL_1_RHD_LHD, wxEmptyString);
  m_List[1]->InsertColumn(COL_1_LAPTIME, wxEmptyString);
  m_List[1]->InsertColumn(COL_1_WR_PERC, wxEmptyString);
  m_List[1]->InsertColumn(COL_1_WR_DIST, wxEmptyString);

  m_GetListBtn = new wxButton(m_Page[1], ID_CTRL_FD_GET_LIST, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);

  wxColour col = m_Notebook->GetThemeBackgroundColour();
  if (col.Ok()) {
    for (int i = 0; i < 2; i++) {
      // set colour that fits with WinXP theme
      m_List[i]->SetBackgroundColour(col);
    }
  }

  // controls for filtering the lists
  wxArrayString carCodes;
  cCar::GetNames(carCodes);
  m_CarChoice = new wxChoice(this, ID_CTRL_FD_CHOICE_CAR, wxDefaultPosition, wxDefaultSize, carCodes);
  wxArrayString trackCodes;
  cTrack::GetNames(trackCodes);
  m_TrackChoice = new wxChoice(this, ID_CTRL_FD_CHOICE_TRACK, wxDefaultPosition, wxDefaultSize, trackCodes);
  m_ReverseCheck = new wxCheckBox(this, ID_CTRL_FD_CHECK_REVERSE, wxEmptyString);

  // other controls
  m_CarTxt = new wxStaticText(this, ID_TXT_FD_CAR, wxEmptyString);
  m_TrackTxt = new wxStaticText(this, ID_TXT_FD_TRACK, wxEmptyString);
  m_AllBtn = new wxButton(this, ID_CTRL_FD_ALL, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);
  m_FolderBtn = new wxButton(this, ID_CTRL_FD_FOLDER, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);
  m_CancelBtn = new wxButton(this, wxID_CANCEL, wxEmptyString);
  m_OpenBtn = new wxButton(this, wxID_OK, wxEmptyString);
  m_OpenBtn->SetDefault();

  // populate sizers on the notebook pages
  m_ButtonSizer[0] = new wxBoxSizer(wxVERTICAL);
  m_ButtonSizer[0]->Add(m_DeleteBtn, wxSizerFlags(1).Expand().Proportion(0));
  m_ButtonSizer[0]->AddSpacer(BASE_MARGIN);
  m_ButtonSizer[0]->Add(m_RenameBtn, wxSizerFlags(1).Expand().Proportion(0));

  m_ButtonSizer[1] = new wxBoxSizer(wxVERTICAL);
  m_ButtonSizer[1]->Add(m_GetListBtn, wxSizerFlags(1).Expand().Proportion(0));

  for (int i = 0; i < 2; i++) {
    m_PageSizer[i] = new wxBoxSizer(wxHORIZONTAL);
    m_PageSizer[i]->Add(m_List[i], wxSizerFlags(1).Expand().Proportion(1).Border(wxALL, BASE_MARGIN));
    m_PageSizer[i]->Add(m_ButtonSizer[i], wxSizerFlags(1).Expand().Proportion(0).Border(wxALL, BASE_MARGIN));
  }

  // populate sizers on the main window
  m_FilterSizer = new wxBoxSizer(wxHORIZONTAL);
  m_FilterSizer->Add(m_CarTxt, wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddSpacer(BASE_MARGIN);
  m_FilterSizer->Add(m_CarChoice, wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddSpacer(4 * BASE_MARGIN);
  m_FilterSizer->Add(m_TrackTxt, wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddSpacer(BASE_MARGIN);
  m_FilterSizer->Add(m_TrackChoice, wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddSpacer(BASE_MARGIN);
  m_FilterSizer->Add(m_ReverseCheck, wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddStretchSpacer();
  m_FilterSizer->Add(m_AllBtn);

  m_MainButtonSizer = new wxBoxSizer(wxHORIZONTAL);
  m_MainButtonSizer->Add(m_FolderBtn);
  m_MainButtonSizer->AddStretchSpacer();
  m_MainButtonSizer->Add(m_CancelBtn);
  m_MainButtonSizer->AddSpacer(BASE_MARGIN);
  m_MainButtonSizer->Add(m_OpenBtn);

  m_TopSizer = new wxBoxSizer(wxVERTICAL);
  m_TopSizer->Add(m_FilterSizer, wxSizerFlags(1).Expand().Proportion(0).Border(wxALL, BASE_MARGIN));
  m_TopSizer->Add(m_Notebook, wxSizerFlags(1).Expand().Proportion(1).Border(wxALL, BASE_MARGIN));
  m_TopSizer->Add(m_MainButtonSizer, wxSizerFlags(1).Expand().Proportion(0).Border(wxALL, BASE_MARGIN));

  TranslateTexts();
  for (int i = 0; i < 2; i++) m_Page[i]->SetSizer(m_PageSizer[i]);
  SetSizer(m_TopSizer);

  m_Notebook->AddPage(m_Page[0], _TT(ID_TXT_FD_LOCAL, _T("Local")));
  m_Notebook->AddPage(m_Page[1], _T("LFS World")); // NB not translated
  m_Notebook->SetSelection(0);

  m_Initialising = false;
}

//-----------------------------------------------------------------------------

cFileDlg::~cFileDlg()
{
}

//-----------------------------------------------------------------------------
// Set the filters according to the user's choices, and refresh the local file list

void cFileDlg::ApplyFilters()
{
  SetFilters(m_CarChoice->GetStringSelection(), m_TrackChoice->GetStringSelection(), m_ReverseCheck->GetValue());
  ReadLocalFiles();
}

//-----------------------------------------------------------------------------
// Set the car and track filters, and show them in the filter controls
// - car = new car filter     (code / code + description / empty / m_AllString)
// - track = new track filter (code / code + description / empty / m_AllString)
// - reverse = reverse track direction?

void cFileDlg::SetFilters(const wxString& car, const wxString& track, bool reverse)
{
  int index;
  wxArrayString choices;

  // determine new car filter
  wxString newCar;
  if (car == m_AllString) {
    newCar = wxEmptyString;
  }
  else {
    newCar = car.Left(3);
  }

  // find new car code in choice list, and select that entry
  index = 0; // default: select first entry
  choices = m_CarChoice->GetStrings();
  for (size_t i = 0; i < choices.GetCount(); i++) {
    if (!choices[i].StartsWith(newCar)) continue;

    // found it
    index = i;
    break;
  }
  m_CarChoice->SetSelection(index);
  m_CarFilter = newCar;

  // determine new car filter
  wxString newTrack;
  if (m_Adding) {
    // not allowed to change track when adding
    newTrack = m_TrackFilter.Left(3);
    reverse = m_TrackFilter.EndsWith(_T("R"));
  }
  else {
    if (track == m_AllString) {
      newTrack = wxEmptyString;
      reverse = false;
    }
    else {
      newTrack = track.Left(3);
      if (track.EndsWith(_T("R"))) reverse = true;
    }
  }

  // find track code in choice list, and select that entry
  index = 0; // default: select first entry
  choices = m_TrackChoice->GetStrings();
  if (track != m_AllString) {
    for (size_t i = 0; i < choices.GetCount(); i++) {
      if (!choices[i].StartsWith(newTrack)) continue;

      // found it
      index = i;
      break;
    }
  }

  m_TrackChoice->SetSelection(index);
  m_ReverseCheck->SetValue(reverse);
  m_TrackFilter = newTrack;
  if (reverse) m_TrackFilter += _T("R");

  wxLogDebug(_T("Filters: car='%s', track='%s', rev=%d"), m_CarFilter.c_str(), m_TrackFilter.c_str(), reverse?1:0);
}

//-----------------------------------------------------------------------------
// Get the names of the selected files

void cFileDlg::GetPaths(wxArrayString& paths) const
{
  int index = m_Notebook->GetSelection();
  long item = m_List[index]->GetFirstSelected();

  while (item >= 0) {
    cFileDlgItemData* data = (cFileDlgItemData*)m_List[index]->GetItemData(item);
    paths.Add(data->filename);
    item = m_List[index]->GetNextSelected(item);
  }
}

//-----------------------------------------------------------------------------
// Read the local RAF files and fill the file list

void cFileDlg::ReadLocalFiles()
{
  if (m_Notebook->GetSelection() != 0) return; // for local files only
  if (m_Initialising) return;

  m_List[0]->Freeze();

  // clear list
  m_List[0]->DeleteAllItems();
  for (size_t i = 0; i < m_ItemData[0].GetCount(); i++) delete m_ItemData[0][i];
  m_ItemData[0].Clear();

  // enumerate files
  wxArrayString fileList;
  wxDir::GetAllFiles(MGR->GetRafDir(), &fileList, _T("*.raf"), wxDIR_FILES);
  cFile file;
  for (size_t f = 0; f < fileList.GetCount(); f++) {
    // read data from file
    if (!file.Open(fileList[f])) continue;
    wxString header;
    file.ReadString(header, 6);
    file.Skip(2); // game version, game revision
    wxInt8 format;
    file.ReadByte(format);
    if ((header != "LFSRAF") || (format != 2)) {
      // not an LFS RAF file, or unknown format version
      file.Close();
      continue;
    }
    wxInt8 interval; // sampling interval in ms
    file.ReadByte(interval);
    if (interval == 0) interval = 10; // default (S2 patch Y and earlier): 100 Hz
    file.GoTo(28);
    float trackLength;
    file.ReadFloat(trackLength);
    wxString racerName;
    file.ReadString(racerName, 32);
    UnEscapeLFS(racerName);
    wxString car;
    file.ReadString(car, 32);
    wxString track;
    file.ReadString(track, 32);
    wxString config;
    file.ReadString(config, 16);
    wxString lfsVersion;
    file.GoTo(160);
    file.ReadString(lfsVersion, 8);
    wxInt32 playerFlags;
    file.ReadInt(playerFlags);
    file.GoTo(171);
    wxInt8 splits;
    file.ReadByte(splits);
    file.GoTo(172 + (splits - 1) * 4);
    wxInt32 lapTime;
    file.ReadInt(lapTime);
    file.Close();

    // get car and track codes
    wxString carCode = cCar::Name2Code(car);
    wxString trackCode = cTrack::Name2Code(track, config);
    wxString trackDesc = trackCode + _T(" - ") + config;

    // apply filters
    if ((!m_CarFilter.IsEmpty()) && (carCode != m_CarFilter)) continue;
    if ((!m_TrackFilter.IsEmpty()) && (trackCode != m_TrackFilter)) continue;

    // convert fields
    wxString lapName;
    wxFileName::SplitPath(fileList[f], NULL, NULL, &lapName, NULL);

    // add item to list
    size_t item = m_List[0]->GetItemCount();
    m_List[0]->InsertItem(item, lapName);
    m_List[0]->SetItem(item, COL_0_FILENAME, lapName);
    m_List[0]->SetItem(item, COL_0_CARCODE, carCode);
    m_List[0]->SetItem(item, COL_0_TRACKCODE, trackCode);
    m_List[0]->SetItem(item, COL_0_LAPTIME, ::FormatTime(lapTime));
    wxDateTime fileDate(wxFileModificationTime(fileList[f]));
    m_List[0]->SetItem(item, COL_0_DATE, fileDate.FormatISODate());
    m_List[0]->SetItem(item, COL_0_RACER, racerName);
    m_List[0]->SetItem(item, COL_0_STEER, Flags2Steer(playerFlags));
    m_List[0]->SetItem(item, COL_0_LFSVERSION, lfsVersion);
    m_List[0]->SetItem(item, COL_0_SAMPLEFREQ, wxString::Format(_T("%.0f Hz"), 1000.0f / interval));
    m_List[0]->SetItem(item, COL_0_RHD_LHD, Flags2RhdLhd(playerFlags));

    // show files that shouldn't be loaded in grey:
    // 1. files that are already loaded (provided the user wants to add files)
    // 2. files that can't be loaded
    if ((m_Adding && (MGR->IsLoaded(fileList[f]))) || (trackLength <= EPSILON)) {
      m_List[0]->SetItemTextColour(item, wxColour(MEDIUM_GREY, MEDIUM_GREY, MEDIUM_GREY));
    }

    // attach item data
    cFileDlgItemData* data = new cFileDlgItemData;
    data->type = 0;
    data->filename = fileList[f];
    data->name = lapName;
    data->car = carCode;
    data->track = trackCode;
    data->laptime = lapTime;
    data->filedate = fileDate.FormatISODate();
    data->lfsversion = lfsVersion;
    data->player = racerName;
    data->flags = playerFlags;
    data->sampling = 1000.0f / interval;
    m_ItemData[0].Add(data);
    m_List[0]->SetItemData(item, (long)data);
  }

  ::SetColumnWidths(m_List[0]);
  DoSort(0);
  if (m_List[0]->GetItemCount() > 0) {
    m_List[0]->Select(0);
    m_List[0]->SetFocus();
  }

  // ready
  EnableButtons();
  m_List[0]->Thaw();
}

//-----------------------------------------------------------------------------
// Read the files (=hotlaps) from LFS World and fill the file list
// Interface is described on http://www.lfsforum.net/showthread.php?t=14480

void cFileDlg::ReadLfswFiles()
{
  if (m_Notebook->GetSelection() != 1) return; // for LFSW files only
  if (m_Initialising) return;
  wxASSERT(m_PubstatDialog != NULL);
  wxASSERT(!m_TrackFilter.IsEmpty());
  wxASSERT(!m_CarFilter.IsEmpty());

  wxString lfswError;
  wxArrayString line;

  // get LFSW ident-key
  wxString identKey = m_PubstatDialog->GetIdentKey(false);

  wxBusyCursor cursor; // activate hourglass

  // construct URL
  wxString track = cTrack::Code2Lfsw(m_TrackFilter);
  wxString url, file;
  url.Printf(_T("%s&idk=%s&action=ch&track=%s&car=%s"), LFSW_PUBSTAT, identKey.c_str(), track.c_str(), m_CarFilter.c_str());

  // retrieve from LFSW
  cUrl urlReader(url);
  if (!urlReader.ReadToString(&file)) {
    lfswError = _T("Could not connect to LFS World");
  }
  else{
    // split result into lines
    line = wxStringTokenize(file, _T("\r\n"));

    long dummy;
    switch (line.GetCount()) {
      case 0 :
        lfswError = _T("No response from LFS World");
        break;
      case 1 :
        // result should begin with a number; if not, it's an error message from LFSW
        if (!line[0].ToLong(&dummy)) {
          lfswError = _T("LFS World error: ") + line[0];
        }
        break;
      default :
        // OK
        break;
    }
  }

  if (!lfswError.IsEmpty()) {
    // report error
    wxMessageBox(lfswError, _T("Error"), wxOK, this);
    return;
  }

  m_List[1]->Freeze();

  // clear list
  m_List[1]->DeleteAllItems();
  for (size_t i = 0; i < m_ItemData[1].GetCount(); i++) delete m_ItemData[1][i];
  m_ItemData[1].Clear();

  // parse each line
  float wrTime = IMPOSSIBLY_HIGH_VALUE; // laptime of the World Record (= first item in list)
  for (size_t i = 0; i < line.GetCount(); i++) {
    wxArrayString field;
    field = wxStringTokenize(line[i], _T(" "));
    if (field.GetCount() < 7) {
      wxLogDebug(_T("LFSW result, line %d: Not enough fields in '%s'"), i, line[i].c_str());
      continue;
    }

    // convert fields
    int chartPos = i + 1;
    long lapTime = 0;
    field[4].ToLong(&lapTime);
    if ((chartPos == 1) && (lapTime != 0)) wrTime = (float)lapTime;

    wxString userName = field[6]; // if name contains spaces then it was split into multiple fields
    for (size_t j = 7; j < field.GetCount(); j++) userName += _T(" ") + field[j];

    long playerFlags;
    field[5].ToLong(&playerFlags);

    long sprId;
    field[0].ToLong(&sprId);

    // create item in list and fill columns
    size_t item = m_List[1]->GetItemCount();
    m_List[1]->InsertItem(item, userName);
    m_List[1]->SetItem(item, COL_1_CHARTPOS, wxString::Format(_T("%d"), chartPos));
    m_List[1]->SetItem(item, COL_1_USERNAME, userName);
    m_List[1]->SetItem(item, COL_1_STEER, Flags2Steer(playerFlags));
    m_List[1]->SetItem(item, COL_1_RHD_LHD, Flags2RhdLhd(playerFlags));
    m_List[1]->SetItem(item, COL_1_LAPTIME, ::FormatTime(lapTime));
    m_List[1]->SetItem(item, COL_1_WR_PERC, wxString::Format(_T("%5.2f%%"), 100 * lapTime / wrTime));
    m_List[1]->SetItem(item, COL_1_WR_DIST, ::FormatTime(lapTime - wrTime, true));
    m_List[1]->SetItem(item, COL_1_RHD_LHD, Flags2RhdLhd(playerFlags));

    // attach item data
    cFileDlgItemData* data = new cFileDlgItemData;
    data->type = 1;
    data->filename = LFSW_GET_RAF + cFileDlg::LapData2LfswName(userName, m_CarFilter, m_TrackFilter, lapTime);
    data->car = m_CarFilter;
    data->track = m_TrackFilter;
    data->chartpos = chartPos;
    data->laptime = lapTime;
    data->player = userName;
    data->flags = playerFlags;
    data->sprnumber = sprId;
    m_ItemData[1].Add(data);
    m_List[1]->SetItemData(item, (long)data);
  }

  ::SetColumnWidths(m_List[1]);
  DoSort(1);
  if (m_List[1]->GetItemCount() > 0) {
    m_List[1]->Select(0);
    m_List[1]->SetFocus();
  }

  // ready
  EnableButtons();
  m_List[1]->Thaw();
}

//-----------------------------------------------------------------------------
// Comparison function, called by wxListCtrl::SortItems

int wxCALLBACK cFileDlg_Compare(long item1, long item2, long key)
{
  wxASSERT(item1 != NULL);
  wxASSERT(item2 != NULL);
  if ((item1 == NULL) || (item2 == NULL)) return 0; // defensive

  cFileDlgItemData* data1 = (cFileDlgItemData*)item1;
  cFileDlgItemData* data2 = (cFileDlgItemData*)item2;
  wxASSERT(data1->type == data2->type);

  int result = 0;

  int column = abs(key) - 1;
  switch (data1->type) {
    case 0 : // local
      switch (column) {
        case COL_0_FILENAME :
          result = data1->name.CmpNoCase(data2->name);
          break;
        case COL_0_CARCODE :
          result = data1->car.CmpNoCase(data2->car);
          break;
        case COL_0_TRACKCODE :
          result = data1->track.CmpNoCase(data2->track);
          break;
        case COL_0_LAPTIME :
          result = data1->laptime - data2->laptime;
          break;
        case COL_0_DATE :
          result = data1->filedate.CmpNoCase(data2->filedate);
          break;
        case COL_0_RACER :
          result = data1->player.CmpNoCase(data2->player);
          break;
        case COL_0_LFSVERSION :
          result = data1->lfsversion.CmpNoCase(data2->lfsversion);
          break;
        case COL_0_SAMPLEFREQ :
          result = data1->sampling - data2->sampling;
          break;
        case COL_0_STEER :
          result = cFileDlg::Flags2Steer(data1->flags).CmpNoCase(cFileDlg::Flags2Steer(data2->flags));
          break;
        case COL_0_RHD_LHD :
          result = cFileDlg::Flags2RhdLhd(data1->flags).CmpNoCase(cFileDlg::Flags2RhdLhd(data2->flags));
          break;
        default :
          wxFAIL;
      }
      break;

    case 1 : // LFSW
      switch (column) {
        case COL_1_CHARTPOS :
          result = data1->chartpos - data2->chartpos;
          break;
        case COL_1_USERNAME :
          result = data1->player.CmpNoCase(data2->player);
          break;
        case COL_1_LAPTIME :
        case COL_1_WR_PERC :
        case COL_1_WR_DIST :
          result = data1->laptime - data2->laptime;
          break;
        case COL_1_STEER :
          result = cFileDlg::Flags2Steer(data1->flags).CmpNoCase(cFileDlg::Flags2Steer(data2->flags));
          break;
        case COL_1_RHD_LHD :
          result = cFileDlg::Flags2RhdLhd(data1->flags).CmpNoCase(cFileDlg::Flags2RhdLhd(data2->flags));
          break;
        default :
          wxFAIL;
      }
      break;

    default :
      wxFAIL;
  }

  if (key < 0) result = -result; // reversed sorting order

  // if no difference then sort on lap name
  if (result == 0) result = data1->name.CmpNoCase(data2->name);

  return result;
}

//-----------------------------------------------------------------------------
// Sort the list of files on the current column
// - index = index of file list

void cFileDlg::DoSort(int index)
{
  wxASSERT((index >= 0) && (index <= 1));
  if (m_List[index]->GetItemCount() == 0) return; // nothing to do

  // un-select any selected item in the list
  long item = m_List[index]->GetFirstSelected();
  while (item >= 0) {
    m_List[index]->Select(item, false);
    item = m_List[index]->GetNextSelected(item);
  }

  // do the sorting
  m_List[index]->SortItems(cFileDlg_Compare, m_SortKey[index]);
}

//-----------------------------------------------------------------------------
// A column in the list was clicked

void cFileDlg::OnColClick(wxListEvent& event)
{
  int index = m_Notebook->GetSelection();
  long col = event.GetColumn();

  if ((col < 0) || (col >= m_List[index]->GetColumnCount())) {
    event.Skip(); // not a valid column
    return;
  }

  int key = col + 1;
  if (abs(m_SortKey[index]) == key) {
    // already sorted on this column - reverse order
    key = -m_SortKey[index];
  }
  else {
    // sort on new column
    if ((index == 0) && (col == COL_0_DATE)) {
      // for "Date" column: start with descending order
      key *= -1;
    }
  }
  SetSortKey(index, key);

  // select and set focus on the first item
  m_List[index]->Select(0);
  m_List[index]->Focus(0);
}

//-----------------------------------------------------------------------------
// Set the sort key
// - index = index of file list
// - key = new sort key (see declaration of m_SortKey for description)

void cFileDlg::SetSortKey(int index, int key)
{
  wxASSERT((index >= 0) && (index <= 1));

  // remove the indicator of the current sorted column
  int oldcol = abs(m_SortKey[index]) - 1;
  wxString text = GetColumnText(index, oldcol);
  if ((text.Left(2) == SORT_INDICATOR_ASC) || (text.Left(2) == SORT_INDICATOR_DESC)) {
    text = text.Mid(2);
    SetColumnText(index, oldcol, text);
  }

  m_SortKey[index] = key;
  DoSort(index);

  // set the indicator of the new sorted column
  int newcol = abs(m_SortKey[index]) - 1;
  text = GetColumnText(index, newcol);
  if (m_SortKey[index] > 0) {
    text = SORT_INDICATOR_ASC + text;
  }
  else {
    text = SORT_INDICATOR_DESC + text;
  }
  SetColumnText(index, newcol, text);
  ::SetColumnWidths(m_List[index]);
}

//-----------------------------------------------------------------------------

void cFileDlg::DoEnter()
{
  int index = m_Notebook->GetSelection();
  if (IsModal() && (m_List[index]->GetFirstSelected() >= 0)) EndModal(wxID_OK);
}

//-----------------------------------------------------------------------------

void cFileDlg::OnButton(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_CTRL_FD_ALL :
      SetFilters(wxEmptyString, wxEmptyString);
      ReadLocalFiles();
      Layout();
      break;

    case ID_CTRL_FD_FOLDER :
      {
        wxString folder = MGR->GetRafDir();
        wxDirDialog dirDlg(this, _TT(ID_TXT_FD_SELECT_FOLDER, "Select RAF folder"), folder);
        if (dirDlg.ShowModal() != wxID_OK) return; // no choice was made

        folder = dirDlg.GetPath();
        if (folder == MGR->GetRafDir()) return; // no change
        MGR->SetRafDir(folder);

        // display the RAF files in the new folder
        m_Notebook->SetSelection(0);
        SetFilters(wxEmptyString, wxEmptyString);
        ReadLocalFiles();
        Layout();
      }
      break;

    case ID_CTRL_FD_DELETE :
      DeleteSelection();
      break;

    case ID_CTRL_FD_RENAME :
      RenameSelection();
      break;

    case ID_CTRL_FD_GET_LIST :
      ReadLfswFiles();
      break;

    case wxID_OK :
      DoEnter();
      break;

    default :
      event.Skip();
      return;
  }
}

//-----------------------------------------------------------------------------
// Prepare things and show the (modal) dialog
// - caption = dialog caption
// - add = add files to already loaded files (or clear all and load new ones)?
// - ident = dialog to fetch the LFSW ID

int cFileDlg::DoShowModal(const wxString& caption, bool add, cPubstatDlg* ident)
{
  SetTitle(caption);
  m_PubstatDialog = ident;

  m_Adding = add && (MGR->GetLapCount() > 0);

  if (m_Adding) {
    m_OpenBtn->SetLabel(_TT(ID_TXT_FD_ADD, "&Add"));

    // set the filter to the car&track of the first loaded lap
    m_CarFilter = MGR->GetLap(0)->GetCarCode();
    m_TrackFilter = MGR->GetTrackCode();
    SetFilters(m_CarFilter, m_TrackFilter);
  }
  else {
    m_OpenBtn->SetLabel(_TT(ID_CTRL_FD_OPEN, "&Open"));
  }

  EnableButtons();
  ReadLocalFiles();

  CentreOnParent();
  return ShowModal();
}

//-----------------------------------------------------------------------------
// Get/Set the text of a column of the file list
// - index = index of file list

wxString cFileDlg::GetColumnText(int index, int col) const
{
  wxASSERT((index >= 0) && (index <= 1));
  wxASSERT(col < m_List[index]->GetColumnCount());

  wxListItem info;
  info.m_mask = wxLIST_MASK_TEXT;
  info.m_itemId = 0;

  if (!m_List[index]->GetColumn(col, info)) return wxEmptyString;
  return info.m_text;
}

void cFileDlg::SetColumnText(int index, int col, const wxString& text)
{
  wxASSERT((index >= 0) && (index <= 1));
  wxASSERT(col < m_List[index]->GetColumnCount());

  wxListItem item;
  item.m_mask = wxLIST_MASK_TEXT;
  m_List[index]->GetColumn(col, item);
  item.SetText(text);
  m_List[index]->SetColumn(col, item);
}

//-----------------------------------------------------------------------------
// Enable or disable the controls after a page change

void cFileDlg::OnPageChange(wxNotebookEvent& WXUNUSED(event))
{
  EnableButtons();
  ApplyFilters();
}

//-----------------------------------------------------------------------------
// Enable or disable the controls

void cFileDlg::EnableButtons()
{
  bool optionAll; // should the "all" entry be in the lists?
  wxArrayString selection;

  switch (m_Notebook->GetSelection()) {
    case 0 :  // local file list
      optionAll = true;
      GetPaths(selection);
      m_DeleteBtn->Enable(selection.GetCount() > 0);
      m_RenameBtn->Enable(selection.GetCount() == 1);
      break;

    case 1 :  // LFSW list
      optionAll = false;
      break;
  }
  m_AllBtn->Enable(optionAll);

  // add or remove the "all" entry
  bool renewFilters = false;
  int index;
  index = m_CarChoice->FindString(m_AllString);
  if (optionAll && (index == wxNOT_FOUND)) m_CarChoice->Insert(m_AllString, 0);
  if (!optionAll && (index != wxNOT_FOUND)) {
    if (m_CarChoice->GetSelection() == index) renewFilters = true;
    m_CarChoice->Delete(index);
  }

  index = m_TrackChoice->FindString(m_AllString);
  if (optionAll && (index == wxNOT_FOUND)) m_TrackChoice->Insert(m_AllString, 0);
  if (!optionAll && (index != wxNOT_FOUND)) {
    if (m_TrackChoice->GetSelection() == index) renewFilters = true;
    m_TrackChoice->Delete(index);
  }

  // re-set the filters when needed
  if (renewFilters) {
    SetFilters(m_CarChoice->GetStringSelection(), m_TrackChoice->GetStringSelection(), m_ReverseCheck->GetValue());
  }

  // when adding files, the track filter is disabled
  m_TrackChoice->Enable(!m_Adding);
  m_ReverseCheck->Enable(!m_Adding && (m_TrackChoice->GetStringSelection() != m_AllString));
}

//-----------------------------------------------------------------------------

void cFileDlg::OnKeyDown(wxListEvent& event)
{
  switch (event.GetKeyCode()) {
    case WXK_DELETE :
      // Delete = delete selected items
      DeleteSelection();
      break;

    case WXK_F2 :
      // F2 = rename selected item
      RenameSelection();
      break;

    case 'A' :
      // Ctrl-A = select all
      if (!wxGetKeyState(WXK_CONTROL)) {
        event.Skip();
        return;
      }
      SelectAll();
      break;

    default :
      event.Skip();
      return;
  }
}

//-----------------------------------------------------------------------------
// Select all files

void cFileDlg::SelectAll()
{
  if(m_Notebook->GetSelection() != 0) return; // for local files only

  for (int i = 0; i < m_List[0]->GetItemCount(); i++) m_List[0]->Select(i);
}

//-----------------------------------------------------------------------------
// Delete all selected files (after confirmation by user)

void cFileDlg::DeleteSelection()
{
  wxASSERT(m_Notebook->GetSelection() == 0); // for local files only

  // get names of selected files
  wxArrayString selection;
  GetPaths(selection);
  if (selection.IsEmpty()) return;

  // ask user to confirm deletion
  wxString prompt;
  if (selection.GetCount() == 1) {
    // mention name of the selected item
    prompt = wxString::Format(_TT(ID_TXT_FD_DELETE_ONE, "Delete '%s'?"),
        m_List[0]->GetItemText(m_List[0]->GetFirstSelected()).c_str());
  }
  else {
    // mention number of selected items
    prompt = wxString::Format(_TT(ID_TXT_FD_DELETE_MANY, "Delete %d files?"), selection.GetCount());
  }
  if (wxMessageBox(prompt, _TT(ID_TXT_FD_WARNING, "Warning"), wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION, this) == wxNO)
      return;

  // delete files
  for (size_t f = 0; f < selection.GetCount(); f++) {
    ::wxRemoveFile(selection[f]);
  }

  // renew list
  ReadLocalFiles();
}

//-----------------------------------------------------------------------------
// Rename the selected file

void cFileDlg::RenameSelection()
{
  wxASSERT(m_Notebook->GetSelection() == 0); // for local files only

  // get names of selected file
  wxArrayString selection;
  GetPaths(selection);
  if (selection.GetCount() != 1) return;

  // get display name of selected file
  wxString dispName = m_List[0]->GetItemText(m_List[0]->GetFirstSelected());

  // prompt for new name
  wxTextEntryDialog dlg(this, _TT(ID_TXT_FD_RENAME_PROMPT, "New name for file"),
      _TT(ID_TXT_FD_RENAME_CAPTION, "Rename file"), dispName);
  if (dlg.ShowModal() == wxID_CANCEL) return;

  wxString newName = MGR->GetRafDir() + wxFILE_SEP_PATH + dlg.GetValue() + _T(".raf");
  if (::wxFileExists(newName) && (selection[0].CmpNoCase(newName) != 0)) {
    wxMessageBox(_TT(ID_TXT_FD_FILE_EXISTS, "File already exists"), _TT(ID_TXT_FD_ERROR, "Error"),
        wxOK | wxICON_ERROR, this);
    return;
  }

  // rename file
  ::wxRenameFile(selection[0], newName);
  MGR->RenameLap(selection[0], newName);
  ::LayoutMainFrame_Send(); // causes main frame to update the file names

  // renew list
  ReadLocalFiles();
}

//-----------------------------------------------------------------------------
// The right mouse button was clicked over an item in the list

void cFileDlg::OnItemRightClick(wxListEvent& WXUNUSED(event))
{
  if(m_Notebook->GetSelection() != 0) return; // for local files only

  wxArrayString selection;
  GetPaths(selection);

  m_Context.Enable(ID_MENU_RENAME, selection.GetCount() == 1);
  m_Context.Enable(ID_MENU_DELETE, !selection.IsEmpty());
  m_Context.Enable(ID_MENU_SELECTALL, !selection.IsEmpty());
  PopupMenu(&m_Context);
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cFileDlg::OnMenuClick(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_RENAME :
      RenameSelection();
      break;

    case ID_MENU_DELETE :
      DeleteSelection();
      break;

    case ID_MENU_SELECTALL :
      SelectAll();
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Convert the player flag to a display string

wxString cFileDlg::Flags2Steer(long flags)
{
  wxString steer = _T("w");           // wheel
  if (flags & 1024) steer = _T("m");  // mouse
  if (flags & 2048) steer = _T("kn"); // keyboard not stabilised
  if (flags & 4096) steer = _T("ks"); // keyboard stabilised
  return steer;
}

wxString cFileDlg::Flags2RhdLhd(long flags)
{
  wxString drive;
  drive = (flags & 1) ? _T("L") : _T("R");
  return drive;
}

//-----------------------------------------------------------------------------
// Convert an LFSW URL to an LFSW-style lapname

wxString cFileDlg::LfswUrl2LapName(const wxString& url)
{
  wxASSERT(url.StartsWith(LFSW_GET_RAF));
  wxString base = LFSW_GET_RAF;
  return url.Mid(base.Len());
}

//-----------------------------------------------------------------------------
// Convert an LFSW-style lapname to an URL for downloading the replay

wxString cFileDlg::LfswName2ReplayUrl(const wxString& lapName)
{
  wxString result = LFSW_GET_SPR + lapName;
  return result;
}

//-----------------------------------------------------------------------------
// Convert lap data into an LFSW-style lapname, and back
// - user = LFSW account name (NOT player name from a RAF file!!)
// - car, track = car/track code
// - lapTime = laptime in ms

wxString cFileDlg::LapData2LfswName(const wxString& user, const wxString& car, const wxString& track, long lapTime)
{
  wxString result;
  wxString resTime; // the part of the result that contains the laptime

  resTime.Printf(_T("%d%05d"), lapTime / 60000, lapTime % 60000);
  result.Printf(_T("%s_%s_%s_%s"), user.c_str(), track.c_str(), car.c_str(), resTime.c_str());
  result.Replace(_T(" "), _T("_")); // replace spaces with underscores

  return result;
}

bool cFileDlg::LfswName2LapData(const wxString& lapName, wxString& user, wxString& car, wxString& track, long* lapTime)
{
  wxString time = lapName.AfterLast(_T('_'));
  if (!time.ToLong(lapTime)) return false;
  // time string is in <minutes><seconds><milliseconds> notation - convert to number of milliseconds
  long minutes = *lapTime / 100000;
  *lapTime = (minutes * 60000) + (*lapTime % 100000);

  wxString rest = lapName.BeforeLast(_T('_'));
  car = rest.AfterLast(_T('_'));

  rest = rest.BeforeLast(_T('_'));
  track = rest.AfterLast(_T('_'));

  user = rest.BeforeLast(_T('_'));
  return (!user.IsEmpty()) && (!car.IsEmpty()) && (!track.IsEmpty()) && (*lapTime > 0);
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cFileDlg::LoadConfig(wxRegConfig* config, const wxString& key)
{
  wxString car, track;
  config->Read(key + _T("/car_filter"), &car);
  config->Read(key + _T("/track_filter"), &track);
  SetFilters(car, track);

  int sortkey;
  sortkey = m_SortKey[0];
  config->Read(key + _T("/sortkey"), &sortkey);
  SetSortKey(0, sortkey);
  sortkey = m_SortKey[1];
  config->Read(key + _T("/sortkey_lfsw"), &sortkey);
  SetSortKey(1, sortkey);

  int sizeX, sizeY;
  GetSize(&sizeX, &sizeY);
  config->Read(key + _T("/size/x"), &sizeX);
  config->Read(key + _T("/size/y"), &sizeY);
  SetSize(sizeX, sizeY);
}


void cFileDlg::SaveConfig(wxRegConfig* config, const wxString& key)
{
  config->Write(key + _T("/car_filter"), m_CarFilter);
  config->Write(key + _T("/track_filter"), m_TrackFilter);
  config->Write(key + _T("/size/x"), GetSize().GetWidth());
  config->Write(key + _T("/size/y"), GetSize().GetHeight());
  config->Write(key + _T("/sortkey"), m_SortKey[0]);
  config->Write(key + _T("/sortkey_lfsw"), m_SortKey[1]);
}

//-----------------------------------------------------------------------------
//

void cFileDlg::TranslateTexts()
{
  cLang::TranslateMenu(&m_Context);

  m_AllString = _TT(ID_TXT_FD_FILTER_ALL, "(all)");

  // notebook pages
  if (m_Notebook->GetPageCount() > 0) m_Notebook->SetPageText(0, _TT(ID_TXT_FD_LOCAL, _T("Local")));

  // column names in the local file list
  SetColumnText(0, COL_0_FILENAME,   _TT(ID_TXT_FD_FILENAME,       "File"));
  SetColumnText(0, COL_0_CARCODE,    _TT(ID_TXT_FD_CARCODE,        "Car"));
  SetColumnText(0, COL_0_TRACKCODE,  _TT(ID_TXT_FD_TRACKCODE,      "Track"));
  SetColumnText(0, COL_0_LAPTIME,    _TT(ID_TXT_FD_LAPTIME,        "Laptime"));
  SetColumnText(0, COL_0_DATE,       _TT(ID_TXT_FD_DATE,           "Date"));
  SetColumnText(0, COL_0_RACER,      _TT(ID_TXT_FD_PLAYER,         "Player name"));
  SetColumnText(0, COL_0_STEER,      _TT(ID_TXT_FD_STEER,          "Steer"));
  SetColumnText(0, COL_0_RHD_LHD,    _TT(ID_TXT_FD_RHD_LHD,        "Side"));
  SetColumnText(0, COL_0_LFSVERSION, _TT(ID_TXT_FD_LFS_VERSION,    "LFS version"));
  SetColumnText(0, COL_0_SAMPLEFREQ, _TT(ID_TXT_FD_SAMPLEFREQ,     "Sampling"));

  // column names in the LFSW file list
  SetColumnText(1, COL_1_CHARTPOS,   _TT(ID_TXT_FD_LFSW_CHARTPOS,  "Pos"));
  SetColumnText(1, COL_1_USERNAME,   _TT(ID_TXT_FD_LFSW_RACERNAME, "Racer name"));
  SetColumnText(1, COL_1_STEER,      _TT(ID_TXT_FD_STEER,          "Steer"));
  SetColumnText(1, COL_1_RHD_LHD,    _TT(ID_TXT_FD_RHD_LHD,        "Side"));
  SetColumnText(1, COL_1_LAPTIME,    _TT(ID_TXT_FD_LAPTIME,        "Laptime"));
  SetColumnText(1, COL_1_WR_PERC,    _TT(ID_TXT_FD_WR_PERC,        "% WR"));
  SetColumnText(1, COL_1_WR_DIST,    _TT(ID_TXT_FD_WR_DIST,        "Distance"));

  for (int index = 0; index < 2; index++) SetSortKey(index, m_SortKey[index]);

  // control labels
  m_CarTxt->SetLabel(_TT(ID_TXT_FD_CAR, "Car"));
  m_TrackTxt->SetLabel(_TT(ID_TXT_FD_TRACK, "Track"));
  m_ReverseCheck->SetLabel(_TT(ID_CTRL_FD_CHECK_REVERSE, "Reverse"));
  m_AllBtn->SetLabel(_TT(ID_CTRL_FD_ALL, "A&ll"));
  m_FolderBtn->SetLabel(_TT(ID_CTRL_FD_FOLDER, "&Folder..."));
  m_DeleteBtn->SetLabel(_TT(ID_CTRL_FD_DELETE, "&Delete"));
  m_RenameBtn->SetLabel(_TT(ID_CTRL_FD_RENAME, "&Rename"));
  m_GetListBtn->SetLabel(_TT(ID_CTRL_FD_GET_LIST, "Get &List"));
  m_CancelBtn->SetLabel(_TT(ID_CTRL_FD_CANCEL, "Cancel"));
  // NB label of m_OpenBtn is set in DoShowModal
}
