#include "cevent.h"
#include "capp.h"
#include "cmainframe.h"

//-----------------------------------------------------------------------------
// Send a custom event
// - type = event type
// - clientInt, clientData, string = attributes of the event

void SendCommandEvent(WXTYPE type, int clientInt, void* clientData, const wxString& string)
{
  wxWindow* frame = APP->GetFrame();
  if (frame == NULL) return; // pointer can be NULL at startup

  wxCommandEvent event(type, frame->GetId());
  event.SetClientData(clientData);
  event.SetEventObject(frame);
  event.SetInt(clientInt);
  event.SetString(string);
  wxPostEvent(frame, event);
}

//-----------------------------------------------------------------------------
// Send an event to close a view (graph or pane)
// - view = view to close

void CloseView_Send(wxWindow* view)
{
  wxASSERT(view != NULL);
  SendCommandEvent((WXTYPE)EVT_LRA_CLOSEVIEW, 0, view);
}

void CloseView_Decode(wxCommandEvent& event, wxWindow*& view)
{
  wxASSERT(event.GetEventType() == EVT_LRA_CLOSEVIEW);
  view = (wxWindow*)(event.GetClientData());
  wxASSERT(view != NULL);
}

DEFINE_EVENT_TYPE(EVT_LRA_CLOSEVIEW)

//-----------------------------------------------------------------------------
// Send an event move a graph in the list of graphs
// - view = graphview to move
// - change = number of positions to change (-1 = move one position up)

void MoveGraph_Send(cGraphView* view, int change)
{
  CHECK_PTR(view);
  SendCommandEvent((WXTYPE)EVT_LRA_MOVEGRAPH, change, view);
}

void MoveGraph_Decode(wxCommandEvent& event, cGraphView*& view, int& change)
{
  wxASSERT(event.GetEventType() == EVT_LRA_MOVEGRAPH);
  view = (cGraphView*)(event.GetClientData());
  CHECK_PTR(view);
  change = event.GetInt();
}

DEFINE_EVENT_TYPE(EVT_LRA_MOVEGRAPH)

//-----------------------------------------------------------------------------
// Send an event to delete a lap
// - lap = lap to delete

void DeleteLap_Send(cLap* lap)
{
  CHECK_PTR(lap);
  SendCommandEvent((WXTYPE)EVT_LRA_DELETELAP, 0, lap);
}

void DeleteLap_Decode(wxCommandEvent& event, cLap*& lap)
{
  wxASSERT(event.GetEventType() == EVT_LRA_DELETELAP);
  lap = (cLap*)(event.GetClientData());
  CHECK_PTR(lap);
}

DEFINE_EVENT_TYPE(EVT_LRA_DELETELAP)

//-----------------------------------------------------------------------------
// Send an event to show or hide a lap in the graphs
// - lap = lap to show or hide (when NULL: show/hide all laps)
// - show - show or hide?

void ShowLap_Send(cLap* lap, bool shown)
{
  CHECK_PTR_NULL(lap);
  SendCommandEvent((WXTYPE)EVT_LRA_SHOWLAP, ((shown) ? 1 : 0), lap);
}

void ShowLap_Decode(wxCommandEvent& event, cLap*& lap, bool& shown)
{
  wxASSERT(event.GetEventType() == EVT_LRA_SHOWLAP);
  lap = (cLap*)(event.GetClientData());
  CHECK_PTR_NULL(lap);
  shown = (event.GetInt() != 0);
}

DEFINE_EVENT_TYPE(EVT_LRA_SHOWLAP)

//-----------------------------------------------------------------------------
// Send an event to set the track cursor(s) at a new position
// - pos = distance_in_lap (in m) that corresponds to the cursor position (< 0 = hide cursor)
// - type = type of cursor-set command

void SetTrackCursorPos_Send(float pos, int type)
{
  int intPos = *((int*)&pos);
  SendCommandEvent((WXTYPE)EVT_LRA_SETTRACKCURSORPOS, intPos, (void*)(type));
}

void SetTrackCursorPos_Decode(wxCommandEvent& event, float& pos, int& type)
{
  wxASSERT(event.GetEventType() == EVT_LRA_SETTRACKCURSORPOS);
  int intPos = event.GetInt();
  pos = *((float*)&intPos);
  type = (int)event.GetClientData();
}

DEFINE_EVENT_TYPE(EVT_LRA_SETTRACKCURSORPOS)

//-----------------------------------------------------------------------------
// Send an event to set the visible ('selected') part of the lap
// - start, end = start and end of the selection

void SetTrackSelection_Send(float start, float end)
{
  wxASSERT(end - start > EPSILON);
  int intStart = *((int*)&start);
  int intEnd = *((int*)&end);
  SendCommandEvent((WXTYPE)EVT_LRA_SETTRACKSELECTION, intStart, (void*)intEnd);
}

void SetTrackSelection_Decode(wxCommandEvent& event, float& start, float& end)
{
  wxASSERT(event.GetEventType() == EVT_LRA_SETTRACKSELECTION);
  int intStart = event.GetInt();
  start = *((float*)&intStart);
  int intEnd = (int)(event.GetClientData());
  end = *((float*)&intEnd);
  wxASSERT(start <= end);
}

DEFINE_EVENT_TYPE(EVT_LRA_SETTRACKSELECTION)

//-----------------------------------------------------------------------------
// Send an event to make all graphs equal height

void EqualHeightGraphs_Send()
{
  SendCommandEvent((WXTYPE)EVT_LRA_EQUALHEIGHTGRAPHS, 0, NULL);
}

DEFINE_EVENT_TYPE(EVT_LRA_EQUALHEIGHTGRAPHS)

//-----------------------------------------------------------------------------
// Send an event to open a RAF file
// - filename = pathname of file

void OpenFile_Send(const wxString& filename)
{
  SendCommandEvent((WXTYPE)EVT_LRA_OPENFILE, 0, NULL, filename);
}

void OpenFile_Decode(wxCommandEvent& event, wxString& filename)
{
  wxASSERT(event.GetEventType() == EVT_LRA_OPENFILE);
  filename = event.GetString();
}

DEFINE_EVENT_TYPE(EVT_LRA_OPENFILE)

//-----------------------------------------------------------------------------
// Send an event to perform an action on the presets
// - action = action to perform (0 = reload presets, 1 = select a preset)
// - name = name of preset

void Preset_Send(int action, const wxString& name)
{
  wxASSERT((action >= 0) && (action <= 1));
  SendCommandEvent((WXTYPE)EVT_LRA_PRESET, action, NULL, name);
}

void Preset_Decode(wxCommandEvent& event, int& action, wxString& name)
{
  wxASSERT(event.GetEventType() == EVT_LRA_PRESET);
  action = event.GetInt();
  name = event.GetString();
}

DEFINE_EVENT_TYPE(EVT_LRA_PRESET)

//-----------------------------------------------------------------------------
// Send an event to synchronise the graph views

void SyncGraphs_Send(cGraphView* view)
{
  CHECK_PTR(view);
  SendCommandEvent((WXTYPE)EVT_LRA_SYNCGRAPHS, 0, view);
}

void SyncGraphs_Decode(wxCommandEvent& event, cGraphView*& view)
{
  wxASSERT(event.GetEventType() == EVT_LRA_SYNCGRAPHS);
  view = (cGraphView*)(event.GetClientData());
  CHECK_PTR(view);
}

DEFINE_EVENT_TYPE(EVT_LRA_SYNCGRAPHS)

//-----------------------------------------------------------------------------
// Send an event to update the layout of the main frame

void LayoutMainFrame_Send()
{
  SendCommandEvent((WXTYPE)EVT_LRA_LAYOUTFRAME, 0, NULL);
}

DEFINE_EVENT_TYPE(EVT_LRA_LAYOUTFRAME)

//-----------------------------------------------------------------------------
// Send an event to view the replay of a lap
// - lap = lap to view

void ReplayLap_Send(cLap* lap)
{
  CHECK_PTR(lap);
  SendCommandEvent((WXTYPE)EVT_LRA_REPLAYLAP, 0, lap);
}

void ReplayLap_Decode(wxCommandEvent& event, cLap*& lap)
{
  wxASSERT(event.GetEventType() == EVT_LRA_REPLAYLAP);
  lap = (cLap*)(event.GetClientData());
  CHECK_PTR(lap);
}

DEFINE_EVENT_TYPE(EVT_LRA_REPLAYLAP)
