#include "ctrackpath.h"
#include <math.h>

#define HEADERSIZE 16
#define NODEBLOCKSIZE 40

//-----------------------------------------------------------------------------

cTrackPath::cTrackPath()
{
  m_RoadLeftX = NULL;
  m_RoadLeftY = NULL;
  m_RoadRightX = NULL;
  m_RoadRightY = NULL;
}

//-----------------------------------------------------------------------------

cTrackPath::~cTrackPath()
{
  delete[] m_RoadLeftX;
  delete[] m_RoadLeftY;
  delete[] m_RoadRightX;
  delete[] m_RoadRightY;
}

//-----------------------------------------------------------------------------
// Open the PTH file and read its data
// - fileName = full filename
// Returns success of operation

bool cTrackPath::Load(const wxString& fileName)
{
  wxASSERT(!IsOpened());

  if (!Open(fileName)) return false;

  // NB: PTH file format is documented in PTH.txt, included with the PTH and SMX files,
  // available from http://www.liveforspeed.net/?page=coderfiles

  // read the header
  wxString str;
  ReadString(str, 6); // read fixed header string 'LFSPTH'
  if (str != "LFSPTH") return false;

  wxInt8 version, revision;
  ReadByte(version);
  ReadByte(revision);
  if ((version != 0) || (revision != 0)) return false;

  ReadInt(m_NodeCount);
  if (m_NodeCount <= 0) return false;

  ReadInt(m_FinishNode);

  m_RoadLeftX = new float[m_NodeCount];
  m_RoadLeftY = new float[m_NodeCount];
  m_RoadRightX = new float[m_NodeCount];
  m_RoadRightY = new float[m_NodeCount];

  // read the nodes
  for (int i = 0; i < m_NodeCount; i++) {
    if (Eof()) return false;

    GoTo(HEADERSIZE + i * NODEBLOCKSIZE); // go to start of node block

    // centre point of node
    wxInt32 pos;
    ReadInt(pos);
    float centreX = (float)pos / 65536.0f;
    ReadInt(pos);
    float centreY = (float)pos / 65536.0f;
    Skip(4);  // centre Z

    // direction vector
    float dirX, dirY;
    ReadFloat(dirX);
    ReadFloat(dirY);
    Skip(4    // dir Z
        + 8); // limit of track (left/right)

    float roadL, roadR; // distance from centre point to road limit (left/right)
    ReadFloat(roadL);
    ReadFloat(roadR);

    m_RoadLeftX[i] = centreX + dirY * roadL;
    m_RoadLeftY[i] = centreY - dirX * roadL;
    m_RoadRightX[i] = centreX + dirY * roadR;
    m_RoadRightY[i] = centreY - dirX * roadR;
  }

  Close();

  return true;
}
