#ifndef _CDRIVINGLINE_H_
#define _CDRIVINGLINE_H_

#include "global.h"
#include "cpane.h"
#include <wx/window.h>
#include <wx/pen.h>
#include <wx/menu.h>
#include <wx/dcmemory.h>

class cTrackPath;

//-----------------------------------------------------------------------------
// The pane with the driving lines across the track

class cDrivingLine : public cPane
{
private:
  float m_Zoom;             // user zoom factor (= number of pixels per meter)
  float m_CentreX;          // X track coordinate corresponding with centre of track display
  float m_CentreY;

  float m_ClickPos;         // distance-in-lap of last click
  wxPoint m_MovePos;        // position of last mouse-motion event
  bool m_DragMode;          // are we drag-scrolling?

  bool m_AutoScroll;        // automatically scroll the display?
  wxMenu m_Context;         // context menu

  void DrawTrack(wxDC& dc);
  void DrawBitmap(wxDC& dc);
  void DrawLaps(wxDC& dc);

  // handling the track cursor
  bool m_CursorDrawn;       // is there currently a track cursor?
  float m_CursorDist;       // distance-in-lap of the cursor
  wxCoord m_CursorX;        // the X coordinate where the track cursor has been drawn
  wxCoord m_CursorY;

  void DrawTrackCursor(wxDC& dc, wxCoord x, wxCoord y);
  void PutTrackCursor(wxDC& dc, float distance);
  void RemoveTrackCursor(wxDC& dc);

  // drawing the track path
  bool m_ShowPath;          // show the track path?
  cTrackPath* m_Path;       // the track path on which the polygon data is based
  bool m_PolygonsValid;     // are the current polygon values correct?
  wxPoint* m_Polygons;      // a poly-polygon, for use in wxDC::DrawPolyPolygon
  int* m_PolygonSize;       // the number of nodes for each polygon (always 4)
  int m_PolygonCount;       // the number of polygons (= number of nodes in the track path)

  // drawing the track bitmap
  bool m_ShowBitmap;        // show the track bitmap?
  wxBitmap* m_Bitmap;       // the bitmap
  bool m_BitmapValid;       // is the bitmap valid for the current track?
  wxString m_BitmapFile;    // the filename from which the bitmap was loaded
  wxMemoryDC m_BitmapDc;    // device context fro drawing from the loaded bitmap
  float m_BitmapOffsetX;    // X track coordinate of the topleft corner of the bitmap
  float m_BitmapOffsetY;

  void ShowBitmap(bool flag);

  // zooming etc.
  void ScrollView(int dx, int dy);
  void SetZoom(float zoom);
  void ResetZoom();
  float DecodeMouse(wxPoint pos);

  void OnMouseLeftClick(wxMouseEvent& event);
  void OnMouseLeftRelease(wxMouseEvent& event);
  void OnMouseMove(wxMouseEvent& event);
  void OnMouseWheel(wxMouseEvent& event);
  void OnMouseLeftDoubleClick(wxMouseEvent& event);
  void OnMouseMiddleClick(wxMouseEvent& WXUNUSED(event)) { ResetZoom(); }
  void OnMouseRightClick(wxMouseEvent& event);
  void OnMouseRightDoubleClick(wxMouseEvent& event);

  void OnMenuClick(wxCommandEvent& event);

  // wxScrolledWindow auto-generates wxScrollWinEvent events when the mouse is dragged outside the window
  // if handling them is left to wxScrolledWindow, these cause repeated redraw (= 'flicker')
  void OnScrollWin(wxScrollWinEvent& WXUNUSED(event)) {}

  void OnPaint(wxPaintEvent& event);
  void OnSize(wxSizeEvent& event);

  DECLARE_EVENT_TABLE()

public:
  cDrivingLine(wxWindow* parent);
  virtual ~cDrivingLine();

  virtual wxSize DoGetBestSize() const;

  virtual void UpdateAll();
  virtual void DoShowLap(cLap* WXUNUSED(lap), bool WXUNUSED(shown)) { Refresh(); }
  virtual void DoSetTrackCursorPos(cGraphView* view, float distance, int type);

  // Convert track coordinates to window coordinates, and vice versa
  inline wxCoord Track2CoordX(float tc)
    { return (GetClientSize().GetWidth() / 2) + (wxCoord)((tc - m_CentreX) * m_Zoom); }

  inline wxCoord Track2CoordY(float tc)
    { return (GetClientSize().GetHeight() / 2) + (wxCoord)((m_CentreY - tc) * m_Zoom); }

  float Coord2TrackX(wxCoord wc)
    { return (wc - (GetClientSize().GetWidth() / 2)) / m_Zoom + m_CentreX; }

  float Coord2TrackY(wxCoord wc)
    { return m_CentreY - (wc - (GetClientSize().GetHeight() / 2)) / m_Zoom; }

  virtual void TranslateTexts();

  virtual void LoadConfig(wxRegConfig* config, const wxString& key);
  virtual void SaveConfig(wxRegConfig* config, const wxString& key);
};

#endif
