#ifndef _GLOBAL_H_
#define _GLOBAL_H_

//-----------------------------------------------------------------------------
// standard includes

#include <wx/string.h>
#include <wx/debug.h>
#include <wx/event.h>
#include <wx/log.h>
#include <wx/pen.h>
#include "cevent.h"
#include "clang.h"

// for Borland compiler: stop using precompiled headers
#ifdef __BORLANDC__
#pragma hdrstop
#endif

//-----------------------------------------------------------------------------
// check wxWidgets version

#include <wx/version.h>
#if wxCHECK_VERSION(2, 8, 0)
  // OK
#else
  #error wxWidgets version is too old
#endif

//-----------------------------------------------------------------------------
// global constants

// identification
#define APPLICATION_NAME "LFS Replay Analyser"
#define APPLICATION_VERSION "1.1"

// the folder for the language files
#define LANGUAGE_FOLDER _T("language")

// the folder for the car/track configuration files
#define CONFIG_FOLDER _T("config")

// the folder for the track bitmap files
#define BITMAP_FOLDER _T("bitmaps")

// the filename of the LRA manual
#define LRA_MANUAL _T("manual.html")

// the URL of the LRA website (forum thread)
#define LRA_WEBSITE _T("http://www.lfsforum.net/showthread.php?t=20056")

// the URL where you can get an LFSW Pubstat key
#define LFSW_PUBSTAT_PAGE _T("http://www.lfsworld.net/?win=settings&whichTab=pubstat")

// the minimum float discernable from zero (prevents division-by-zero, and loss of precision)
#define EPSILON 2.0e-6f

// extreme values (e.g. for initialization of a "minimum" variable)
#define IMPOSSIBLY_LOW_VALUE -1.0e20f
#define IMPOSSIBLY_HIGH_VALUE 1.0e20f

// some compilers (MinGW) do not define M_PI_2 and M_PI in math.h
#ifndef M_PI_2
#define M_PI_2 1.5707963f
#endif
#ifndef M_PI
#define M_PI 3.1415926f
#endif

// units conversion
#define RAD2DEG (180.0f / M_PI)
#define ONE_G 9.80665f
#define RADPSEC2RPM (60.0f / (2.0f * M_PI))

// basic whitespace in window layout
#define BASE_MARGIN 4

// largest value (in text format) that must fit in the displays
#define LARGEST_VALUE_TEXT _T("-99999.9")

// max. distance (in pixels) for detecting a click on a curve
#define CLICK_DETECT_THRESHOLD 12

// colours
#define MEDIUM_GREY 140
#define LIGHT_GREY 200
#define LIGHTER_GREY 220

// types of cursor-set commands
enum CURSORSET_IDS {
  CURSORSET_FOLLOW = 0, // follow the mouse pointer (crosshair)
  CURSORSET_FORCE,      // forced setting of cursor (mouse-click)
  CURSORSET_PLAYBACK    // playback of laps
};

//-----------------------------------------------------------------------------
// shortcuts for global objects / functions

#define MGR (cMgr::GetMgr())
#define APP ((cApp*)wxTheApp)
#define CONFIG (APP->GetConfig())

//-----------------------------------------------------------------------------
// global functions

inline void Minimise(int& val, int other) { if (val > other) val = other; }
inline void Maximise(int& val, int other) { if (val < other) val = other; }
inline void Minimise(float& val, float other) { if (val > other) val = other; }
inline void Maximise(float& val, float other) { if (val < other) val = other; }
void SwapValues(float& a, float& b) { float temp = a; a = b; b = temp; }

float FloorOneSD(float val);
float CeilOneSD(float val);
int FirstSD(float val);

wxString FormatTime(int t, bool shortformat=false);
bool ReadConfigFile(const wxString& fileName, wxArrayString& lines);
void SplitStringList(const wxString& list, char separator, wxArrayString& parts);
void DecodeKeyValueList(const wxString& list, wxArrayString& keys, wxArrayString& values);

class wxListCtrl;
void SetColumnWidths(wxListCtrl* list);

void UnEscapeLFS(wxString& string);
wxString Escape(const wxString& string, const wxString& special);
wxString UnEscape(const wxString& string, const wxString& special);

class wxDC;
void DrawTextCentered(wxDC& dc, const wxString& text, wxCoord x, wxCoord y);
void DrawTextCenteredH(wxDC& dc, const wxString& text, wxCoord x, wxCoord y);
void DrawTextRightAligned(wxDC& dc, const wxString& text, wxCoord x, wxCoord y);
void DrawButton(wxDC& dc, const wxRect& rect, bool highlight);

//-----------------------------------------------------------------------------
// very simple pointer checking and object counting

#ifdef __WXDEBUG__

class cObjChecker
{
private:
  static int s_ObjCount;
  cObjChecker* m_Marker;

public:
  cObjChecker();
  ~cObjChecker();

  static int GetCount() { return s_ObjCount; }

  inline bool IsValid() const { return (m_Marker == this + 1); }
};

// to enable pointer checking on a class, put this in the public section of the class definition
#define ENABLE_OBJCHECKER cObjChecker m_PtrChecker;

// to check a non-NULL pointer
#define CHECK_PTR(ptr) wxASSERT((ptr != NULL) && (ptr->m_PtrChecker.IsValid()))

// to check a pointer that may also be NULL
#define CHECK_PTR_NULL(ptr) wxASSERT((ptr == NULL) || (ptr->m_PtrChecker.IsValid()))

// to check the current object
#define CHECK_THIS CHECK_PTR(this)

#else

// macros are empty if not compiled in debug mode
#define ENABLE_OBJCHECKER
#define CHECK_PTR(ptr)
#define CHECK_PTR_NULL(ptr)
#define CHECK_THIS

#endif

#endif
