#include "cmgr.h"
#include "ctrack.h"
#include "ctrackpath.h"
#include <wx/filename.h>
#include <wx/dir.h>

#define REL_PATH_SMX wxFILE_SEP_PATH + _T("smx")

// the name of the LFS executable
#define LFS_EXE _T("lfs.exe")

// the name of the folder with replays
#define LFS_SPR_FOLDER _T("data\\spr")

// the file extension for track bitmaps
#define BITMAP_FILE_EXTENSION _T(".tif")

//-----------------------------------------------------------------------------

cMgr::cMgr()
{
  m_TrackPath = NULL;
  ClearAll();
  m_LapNaming = LAPNAME_FILE;

  cCarState::DefineTypes();

  m_RafDir = wxGetCwd();
}

//-----------------------------------------------------------------------------

cMgr::~cMgr()
{
  delete m_TrackPath;
  s_Mgr = NULL;
}

//----------------------------------------------------------------------------

cMgr* cMgr::GetMgr()
{
  // singleton pattern
  if (s_Mgr == NULL) s_Mgr = new cMgr;
  return s_Mgr;
}

//----------------------------------------------------------------------------

cMgr* cMgr::s_Mgr = 0;

//-----------------------------------------------------------------------------
// Comparison function, to be used in sorting the list of laps

int cMgr::Compare(cLap** a, cLap** b)
{
  // sort on display name, case-insensitive
  return ((*a)->GetName().CmpNoCase((*b)->GetName()));
}

//-----------------------------------------------------------------------------
// Change the lap naming scheme

void cMgr::SetLapNaming(lapname_t naming)
{
  wxASSERT(naming != LAPNAME_NONE);
  if (m_LapNaming == naming) return; // no change

  m_LapNaming = naming;
  for (size_t i = 0; i < m_Laps.GetCount(); i++) {
    m_Laps[i].SetNaming(m_LapNaming);
  }

  SortLaps(); // re-sort the laps
}

//-----------------------------------------------------------------------------
// Add a lap (= RAF file) that has been loaded
// - lap = lap to add
// - error = receives the error description of lap can't be added

bool cMgr::AddLap(cLap* lap, wxString& error)
{
  wxASSERT(lap->IsLoaded());

  // check if lap is for the same track as the others
  if ((m_Laps.GetCount() > 0) && (lap->GetTrackCode() != GetTrackCode())) {
    error = wxString::Format(_T("File contains data from a different track (%s)"), lap->GetTrackCode().c_str());
    return false;
  }

  // add the lap
  m_Laps.Add(lap);
  lap->SetNaming(m_LapNaming);
  SortLaps(); // keep the laps sorted

  // first lap only: sort out the track data
  if (m_Laps.GetCount() == 1) {
    // try to load track path
    wxASSERT(m_TrackPath == NULL); // no track path loaded yet
    wxString file = lap->GetFileName();
    wxString dir = file.BeforeLast(wxFILE_SEP_PATH).BeforeLast(wxFILE_SEP_PATH) + REL_PATH_SMX;
    wxString suffix = wxFILE_SEP_PATH + lap->GetTrackCode() + _T(".pth");

    // first try: relative to path of RAF file
    wxString pthFile = dir + suffix;

    // second try: m_SmxDir
    if (!wxDirExists(m_SmxDir)) m_SmxDir = wxEmptyString; // directory no longer exists
    if ((!wxFileExists(pthFile)) && (!m_SmxDir.IsEmpty())) {
      pthFile = m_SmxDir + suffix;
    }

    if (wxFileExists(pthFile)) {
      m_TrackPath = new cTrackPath();
      if (m_TrackPath->Load(pthFile)) {
        // load successful
        if (m_SmxDir.IsEmpty())
            m_SmxDir = pthFile.BeforeLast(wxFILE_SEP_PATH); // remember directory
      }
      else {
        // load failed
        delete m_TrackPath;
        m_TrackPath = NULL;
      }
    }
    else {
      wxLogDebug(_T("Could not find track outline file \"%s.pth\""), lap->GetTrackCode().c_str());
    }

    // find track bitmap file
    if (!m_BitmapDir.IsEmpty()) {
      m_BitmapFile = m_BitmapDir + wxFILE_SEP_PATH + cTrack::Name2Venue(lap->GetTrackName()) + BITMAP_FILE_EXTENSION;
      if (!wxFileExists(m_BitmapFile)) m_BitmapFile = wxEmptyString;
    }
  }

  FindLeader();

  return true;
}

//-----------------------------------------------------------------------------

void cMgr::DeleteLap(cLap* lap)
{
  wxASSERT(lap != NULL);
  int index = m_Laps.Index(*lap);
  wxASSERT(index != wxNOT_FOUND);

  m_Laps.RemoveAt(index);
  if (m_Laps.IsEmpty()) ClearAll();
  FindLeader();
}

//-----------------------------------------------------------------------------
// Check if a lap file is already loaded

bool cMgr::IsLoaded(const wxString& fileName)
{
  for (size_t i = 0; i < m_Laps.GetCount(); i++) {
    if (fileName == m_Laps[i].GetFileName()) return true;
  }
  return false;
}

//-----------------------------------------------------------------------------
// Delete all data

void cMgr::ClearAll()
{
  m_Laps.Clear();
  m_Leader = NULL;
  delete m_TrackPath;
  m_TrackPath = NULL;
  m_BitmapFile.Empty();
}

//-----------------------------------------------------------------------------
// Change the filename of a loaded lap

void cMgr::RenameLap(const wxString& oldName, const wxString& newName)
{
  for (size_t i = 0; i < m_Laps.GetCount(); i++) {
    if (m_Laps[i].GetFileName().CmpNoCase(oldName) == 0) {
      m_Laps[i].SetFileName(newName);
    }
  }
}

//-----------------------------------------------------------------------------
// Determine the 'leader' lap

void cMgr::FindLeader()
{
  if (IsEmpty()) {
    m_Leader = NULL;
    return;
  }

  m_Leader = &m_Laps[0];
  for (size_t i = 1; i < m_Laps.GetCount(); i++) {
    if (m_Laps[i].IsComplete() && (m_Laps[i].GetLaptime() < m_Leader->GetLaptime())) m_Leader = &m_Laps[i];
  }
}

//-----------------------------------------------------------------------------
// Return the directory for replay files

wxString cMgr::GetReplayDir() const
{
  if (m_LfsDir.IsEmpty()) return wxEmptyString;
  wxString result = m_LfsDir + wxFILE_SEP_PATH + LFS_SPR_FOLDER;
  return result;
}

//-----------------------------------------------------------------------------
// Update the directory for RAF files, and also the location for lfs.exe

void cMgr::SetRafDir(const wxString& dir)
{
  m_RafDir = dir;

  // check if lfs.exe is there
  wxString newDir = dir.BeforeLast(wxFILE_SEP_PATH).BeforeLast(wxFILE_SEP_PATH);
  if (newDir.CmpNoCase(m_LfsDir) == 0) return; // no change

  if (wxFileExists(newDir + wxFILE_SEP_PATH + LFS_EXE)) m_LfsDir = newDir;
}

//-----------------------------------------------------------------------------
// Update the directory for track bitmaps

void cMgr::SetBitmapDir(const wxString& dir)
{
  m_BitmapDir = wxEmptyString;
  if (!wxDirExists(dir)) return;

  // check if the directory has bitmap files
  wxDir d(dir);
  wxString spec = _T("*");
  spec += BITMAP_FILE_EXTENSION;
  if (d.HasFiles(spec)) {
    m_BitmapDir = dir;
  }
  else {
    wxLogMessage(_T("No track bitmaps found in folder '%s'"), dir.c_str());
  }
}
