#include "clang.h"
#include <wx/menu.h>
#include <wx/file.h>
#include <wx/textfile.h>
#include <wx/dir.h>
#include <wx/filename.h>
#include <wx/log.h>

// highest allowable string ID
#define MAX_ID 9999

// the name of the default language
#define DEFAULT_LANGUAGE _T("English")

// the extension for language files
#define LANGUAGE_FILE_EXT _T(".txt")

//-----------------------------------------------------------------------------
// Static members

wxString cLang::m_LanguageDir;
wxString cLang::m_Language = DEFAULT_LANGUAGE;

wxArrayString cLang::m_DefText;
wxArrayString cLang::m_DefHelp;

wxArrayString cLang::m_SelText;
wxArrayString cLang::m_SelHelp;

//-----------------------------------------------------------------------------

cLang::cLang()
{
}

//-----------------------------------------------------------------------------

cLang::~cLang()
{
}

//-----------------------------------------------------------------------------
// Set the folder for the language files

void cLang::SetFolder(const wxString& folder)
{
  wxASSERT(m_LanguageDir.IsEmpty()); // function is intended to be called only once

  wxASSERT(!folder.IsEmpty());
  m_LanguageDir = folder;

  // load strings from file (if present)
  Load(DEFAULT_LANGUAGE, m_DefText, m_DefHelp);
}

//-----------------------------------------------------------------------------
// Get a list of the available languages

void cLang::GetLanguageList(wxArrayString& list)
{
  wxASSERT(!m_LanguageDir.IsEmpty()); // SetFolder() must have been called

  list.Clear();
  list.Add(DEFAULT_LANGUAGE);
  if (!wxDirExists(m_LanguageDir)) return;

  wxArrayString fileList;
  wxDir::GetAllFiles(m_LanguageDir, &fileList, _T("*") LANGUAGE_FILE_EXT, wxDIR_FILES);
  for (size_t f = 0; f < fileList.GetCount(); f++) {
    if (f >= MAX_LANGUAGES) break; // defensive

    wxString langName;
    wxFileName::SplitPath(fileList[f], NULL, NULL, &langName, NULL);

    if (langName == DEFAULT_LANGUAGE) continue;
    if (!isalpha(langName[0])) continue;
    list.Add(langName);
  }

  list.Sort();
}

//-----------------------------------------------------------------------------
// Set the selected language
// - name = name of language
// Returns success of operation

bool cLang::SetLanguage(const wxString& name)
{
  // remove current strings and return to default language
  m_SelText.Empty();
  m_SelHelp.Empty();
  m_Language = DEFAULT_LANGUAGE;

  if (name.IsEmpty() || (name == DEFAULT_LANGUAGE)) return true;

  // try to load non-default language file
  if (!Load(name, m_SelText, m_SelHelp)) return false;

  m_Language = name;
  return true;
}

//-----------------------------------------------------------------------------
// Load a language file
// - name = name of language
// - txt = array to load the text strings into
// - hlp = array to load the help strings into
// Returns success of operation

bool cLang::Load(const wxString& name, wxArrayString& txt, wxArrayString& hlp)
{
  wxASSERT(!m_LanguageDir.IsEmpty()); // SetFolder() must have been called
  wxString fileName = m_LanguageDir + wxFILE_SEP_PATH + name + LANGUAGE_FILE_EXT;
  wxASSERT(wxFileExists(fileName));
  if (!wxFileExists(fileName)) return false; // defensive

  wxTextFile file;
  file.Open(fileName);

  for (size_t i = 0; i < file.GetLineCount(); i++) {
    wxString line = file.GetLine(i);

    // trim comments and spaces
    int p = line.Find('#');
    if (p != -1) line.Truncate(p);
    line.Trim().Trim(false);
    if (line.IsEmpty()) continue; // nothing left

    // parse line
    int index = wxAtoi(line);
    size_t id = index + LANG_START_ID;
    wxASSERT(id >= LANG_START_ID);
    wxASSERT(id <= MAX_ID);
    if ((id < LANG_START_ID) || (id > MAX_ID)) continue; // defensive

    bool isHelp = (line[4] == 'h') || (line[4] == 'H');

    line = line.Mid(6).Trim().Trim(false);
    if (line.IsEmpty()) continue;

    // add string to the selected language
    for (int i = txt.GetCount(); i <= index; i++) {
      txt.Add(wxEmptyString);
      hlp.Add(wxEmptyString);
    }
    if (isHelp) {
      hlp[index] = line;
    }
    else {
      txt[index] = line;
    }

  }
  wxASSERT(txt.GetCount() == hlp.GetCount());
  wxLogDebug(_T("Loaded language file %s, %d strings"), name.c_str(), txt.GetCount());

  file.Close();
  return true;
}

//-----------------------------------------------------------------------------
// Lookup translated text (normal text or help string)
// - id = ID of the string
// - def = default string

const wxString& cLang::Lookup(int id, const wxString& def, wxArrayString& selArr, wxArrayString& defArr)
{
  if ((id < ID_MENU_FIRST) || (id > ID_MENU_LAST_TRANSLATABLE)) return def; // not a translatable item

  size_t index = id - LANG_START_ID;

  // add default string to the default language
  if (!def.IsEmpty()) {
    for (size_t i = defArr.GetCount(); i <= index; i++) {
      defArr.Add(wxEmptyString);
    }
    wxASSERT(defArr[index].IsEmpty() || (defArr[index] == def)); // sanity check
    if (defArr[index].IsEmpty()) defArr[index] = def;
  }

  // lookup in the user-selected language
  if ((index < selArr.GetCount()) && (!selArr[index].IsEmpty())) {
    return selArr[index];
  }

  // lookup in the default language
  if (index < defArr.GetCount()) {
    return defArr[index];
  }

  return def;
}

//-----------------------------------------------------------------------------
// Get the translated text
// - id = ID of the string
// - def = default string

const wxString& cLang::TranslateText(int id, const wxString& def)
{
  return Lookup(id, def, m_SelText, m_DefText);
}

//-----------------------------------------------------------------------------
// Get the translated help string
// - id = ID of the string
// - def = default string

const wxString& cLang::TranslateHelp(int id, const wxString& def)
{
  return Lookup(id, def, m_SelHelp, m_DefHelp);
}

//-----------------------------------------------------------------------------
// Translate all items in a menu
// - menu = the menu (may be null)

void cLang::TranslateMenu(wxMenu* menu)
{
  if (menu == NULL) return;

  wxMenuItemList& list = menu->GetMenuItems();

  // iterate over the list
  wxMenuItemList::iterator iter;
  for (iter = list.begin(); iter != list.end(); ++iter) {
    wxMenuItem* item = *iter;
    if (item->GetKind() == wxITEM_SEPARATOR) continue;

    // translate the item (NB: no default string available!)
    wxString newLabel = TranslateText(item->GetId(), wxEmptyString);
    if (newLabel.IsEmpty()) continue; // not a translatable item
    item->SetItemLabel(newLabel);
    item->SetHelp(TranslateHelp(item->GetId(), wxEmptyString));

    // recursively translate the submenu
    TranslateMenu(item->GetSubMenu());
  }
}

//-----------------------------------------------------------------------------
// Write the strings in the default language to a file
// Returns success of operation

bool cLang::DumpDefault()
{
  wxASSERT(!m_LanguageDir.IsEmpty()); // SetFolder() must have been called
  wxASSERT(m_DefText.GetCount() >= m_DefHelp.GetCount()); // sanity check

  wxString fileName = m_LanguageDir + wxFILE_SEP_PATH + DEFAULT_LANGUAGE + LANGUAGE_FILE_EXT;
  wxFile file;
  file.Create(fileName, true);
  if (!file.IsOpened()) return false;

  for (size_t i = 0; i < m_DefText.GetCount(); i++) {
    if (m_DefText[i].IsEmpty()) continue;
    file.Write(wxString::Format(_T("%04dt %s\n"), i, m_DefText[i].c_str()));

    if ((i < m_DefHelp.GetCount()) && (!m_DefHelp[i].IsEmpty())) {
      file.Write(wxString::Format(_T("%04dh %s\n"), i, m_DefHelp[i].c_str()));
    }
  }

  file.Close();
  return true;
}

//-----------------------------------------------------------------------------
// Helper functions for creating menu items

void cLang::AppendMenuItem(wxMenu* menu, int id, const wxString& label, const wxString& help)
{
  menu->Append(id, TranslateText(id, label), TranslateHelp(id, help));
}

void cLang::AppendMenuItem(wxMenu* menu, int id, const wxString& label, wxMenu* submenu, const wxString& help)
{
  menu->Append(id, TranslateText(id, label), submenu, TranslateHelp(id, help));
}

void cLang::AppendCheckItem(wxMenu* menu, int id, const wxString& label, const wxString& help)
{
  menu->AppendCheckItem(id, TranslateText(id, label), TranslateHelp(id, help));
}
