#include "cgraph_h.h"
#include "cgraphview.h"
#include "ccurve_h.h"
#include "cruler.h"
#include "cmgr.h"
#include <wx/dcclient.h>

// default number of bins
#define DEFAULT_BIN_COUNT 60

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cGraphH, cGraph)
  EVT_MOTION(cGraphH::OnMouseMove)
  EVT_ENTER_WINDOW(cGraphH::OnMouseEntering)
  EVT_LEAVE_WINDOW(cGraphH::OnMouseLeaving)
  EVT_PAINT(cGraphH::OnPaint)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cGraphH::cGraphH(cGraphView* parent)
: cGraph(parent, GRAPHTYPE_H)
{
  m_Lap = NULL;

  m_Logtype = LOGTYPE_FIRST;
  m_Wheel = WHEELTYPE_FIRST;

  AdjustBbox(); // init bounding box
}

//-----------------------------------------------------------------------------

cGraphH::~cGraphH()
{
}

//-----------------------------------------------------------------------------
// Get the rbb mode to start select-dragging (RBBMODE_NONE = veto)
// - event = event that signals start of drag

rbbmode_t cGraphH::GetRbbMode(wxMouseEvent& event)
{
  return (event.ShiftDown()) ? RBBMODE_BOTH : RBBMODE_XONLY;
}

//-----------------------------------------------------------------------------
// Add a curve to the graph
// - lap = lap for which to add a curve

void cGraphH::AddCurve(cLap* lap)
{
  CHECK_PTR(lap);

  // add the curve
  cCurveH* curve = new cCurveH(lap);
  curve->SetLogtype(m_Logtype, m_Wheel);
  m_Curves.Add(curve);

  if (m_Curves.GetCount() == 1) {
    // automatically show first curve
    SetLap(curve->GetLap());
  }
  else {
    // hide additional curves
    curve->Show(false);
  }

  // update display
  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Delete a curve
// - lap = lap whose curve is to be deleted

void cGraphH::DeleteCurve(cLap* lap)
{
  CHECK_PTR(lap);

  // find the curve for this lap
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    if (m_Curves[c].GetLap() != lap) continue;

    // found it
    if (m_Lap == lap) SetLap(NULL);
    m_Curves.RemoveAt(c);
    if ((m_Lap == NULL) && (m_Curves.GetCount() > 0)) SetLap(m_Curves[0].GetLap());
    AdjustBbox();
    Refresh();
    return;
  }
}

//-----------------------------------------------------------------------------
// Delete all curves

void cGraphH::DeleteAllCurves()
{
  m_Curves.Clear();
  m_Lap = NULL;
  m_SelStart = 0.0f;
  m_SelEnd = 1.0f;

  ZoomReset();
  AdjustBbox();
}

//-----------------------------------------------------------------------------
// Plot the graph

void cGraphH::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  CHECK_PTR_NULL(m_Lap);
  wxASSERT((m_Lap != NULL) || (m_Curves.IsEmpty()));
  wxASSERT((m_Lap == NULL) || (MGR->FindLap(m_Lap) >= 0));

  wxPaintDC dc(this);

  int screenX, screenY;
  dc.GetSize(&screenX, &screenY);
  if ((screenX <= 0) || (screenY <= 0)) return;

  // plot the grid
  float value;
  wxCoord coord;
  // Y axis (1 vertical line)
  dc.SetPen(cGraph::s_AxisPen);
  coord = Value2CoordX(0.0f);
  dc.DrawLine(coord, 0, coord, screenY);
  // Y grid (horizontal lines)
  value = GetGridEndY();
  value -= fmod(value, m_GridStepY);
  value += m_GridStepY;
  coord = 0;
  dc.SetPen(cGraph::s_GridPen);
  while (coord <= screenY) {
    coord = Value2CoordY(value);
    dc.DrawLine(0, coord, screenX, coord);
    value -= m_GridStepY;
  }

  // plot the curves (only the selected one)
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    m_Curves[c].Plot(dc, *this, m_SelStart, m_SelEnd);
  }

  // update the rulers
  if (m_RulerX != NULL) m_RulerX->Refresh();
  if (m_RulerY != NULL) m_RulerY->Refresh();

  // plot the crosshair (if it was present)
  if (m_CrossHairVisible) {
    dc.SetPen(*wxBLACK_PEN);
    dc.CrossHair(m_CrossHairPos.x, m_CrossHairPos.y);
  }
}

//-----------------------------------------------------------------------------
// Set the type of data that is shown
// - log = type of log-data
// - wheel = index of the wheel for which the data must be shown

void cGraphH::SetLogtype(int log, int wheel)
{
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  if ((m_Logtype == log) && (m_Wheel == wheel)) return; // no change
  m_Logtype = log;
  m_Wheel = wheel;

  // set it for each curve
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    ((cCurveH&)m_Curves[c]).SetLogtype(log, wheel);
  }

  // update display
  ZoomReset();
  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Set the lap
// - lap = the new lap, or NULL if there is no lap

void cGraphH::SetLap(cLap* lap)
{
  CHECK_PTR_NULL(m_Lap);
  wxASSERT((lap == NULL) || (MGR->FindLap(lap) >= 0));

  if (m_Lap == lap) return; // no change

  // hide current lap
  int index = FindCurve(m_Lap);
  if (index != wxNOT_FOUND) m_Curves[index].Show(false);

  m_Lap = lap;

  // show the new lap
  index = FindCurve(m_Lap);
  if (index != wxNOT_FOUND) m_Curves[index].Show(true);

  // update display
  Refresh();
}

//-----------------------------------------------------------------------------
// Post-processing (= correction) of the bounding box

void cGraphH::OnAdjustBbox()
{
  wxASSERT(m_MinX < m_MaxX);

  if (MGR->IsEmpty()) {
    // no curves - change defaults (from cGraph::AdjustBbox)
    m_MinY = 0.0f;
    m_MaxY = 100.0f;
    return;
  }

  float bins;
  if (cCarState::IsSymmetric(m_Logtype, m_Wheel)) {
    // logtype has both positive and negative values - make range symmetric around axis
    float range = fabs(m_MaxX);
    ::Maximise(range, fabs(m_MinX));

    bins = DEFAULT_BIN_COUNT / 2;          // #bins for positive half(!) of range
    float step = FloorOneSD(range / bins); // get a 'nice' step value
    bins = ceil(range / step);             // extend #bins
    bins += 0.5f;                          // add a center bin
    range = step * bins;                   // extend range to exactly hold the bins

    bins *= 2.0f;
    m_MinX = -range;
    m_MaxX = range;
  }
  else {
    // logtype only has positive values - let range begin at zero
    wxASSERT(m_MinX >= 0.0f);
    wxASSERT(m_MaxX > 0.0f);
    m_MinX = 0.0f;
    float range = m_MaxX;

    bins = DEFAULT_BIN_COUNT;
    float step = FloorOneSD(range / bins); // get a 'nice' step value
    bins = ceil(range / step);             // extend #bins
    range = step * bins;                   // extend range to exactly hold the bins

    m_MaxX = range;
  }

  // integer logtypes:
  if (cCarState::IsInteger(m_Logtype)) {
    // extend range with 0.5 around extreme values
    m_MinX -= 0.5;
    m_MaxX += 0.5;
    // one bin for each value
    bins = m_MaxX - m_MinX;
  }

  SetBins((int)bins);

  // synchronise with other histograms
  RequestSync();
}

//-----------------------------------------------------------------------------
// Set the number of bins

void cGraphH::SetBins(int bins)
{
  m_BinCount = bins;

  // set the bins for each curve
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    ((cCurveH&)m_Curves[c]).SetBins(m_MinX, m_MaxX, m_BinCount);
  }
}

//-----------------------------------------------------------------------------
// Set the visible ('selected') part of the lap

void cGraphH::DoSetTrackSelection(float start, float end)
{
  m_SelStart = start;
  m_SelEnd = end;

  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

wxString cGraphH::SaveSettings() const
{
  wxString settings;

  settings += wxString::Format(_T("data=%s;"), cCarState::GetLogtypeCode(GetLogtype()).c_str());
  settings += wxString::Format(_T("wheel=%s;"), cCarState::GetWheeltypeCode(GetWheeltype()).c_str());
  settings += wxString::Format(_T("lap=%d;"), MGR->FindLap(GetLap()));

  return settings;
}


void cGraphH::LoadSettings(const wxString& settings)
{
  // get current settings
  int data = GetLogtype();
  int wheel = GetWheeltype();
  int lap = MGR->FindLap(GetLap());

  // parse new settings
  wxArrayString keys;
  wxArrayString values;
  ::DecodeKeyValueList(settings, keys, values);
  for (size_t k = 0; k < keys.GetCount(); k++) {
    int val = wxAtoi(values[k]);
    if (keys[k] == _T("data")) {
      data = cCarState::FindLogtypeCode(values[k]);
    }
    else if (keys[k] == _T("wheel")) {
      wheel = cCarState::FindWheeltypeCode(values[k]);
    }
    else if (keys[k] == _T("lap")) {
      if (val < (int)(MGR->GetLapCount())) lap = val;
    }
  }
  // fallback to current settings if new ones incorrect
  if (data == wxNOT_FOUND) data = GetLogtype();
  if (wheel == wxNOT_FOUND) wheel = GetWheeltype();

  // put new settings
  SetLogtype(data, wheel);
  if (lap >= 0) SetLap(MGR->GetLap(lap));

  UpdateAll();
}

//-----------------------------------------------------------------------------

void cGraphH::OnMouseMove(wxMouseEvent& event)
{
  event.Skip(); // always pass event to parent class (for rbb handling)

  // move the crosshair
  RemoveCrossHair();
  DrawCrossHairAtPointer();
}

//-----------------------------------------------------------------------------

void cGraphH::OnMouseEntering(wxMouseEvent& WXUNUSED(event))
{
  DrawCrossHairAtPointer();
}

void cGraphH::OnMouseLeaving(wxMouseEvent& WXUNUSED(event))
{
  RemoveCrossHair();
}

//-----------------------------------------------------------------------------
// Sync'ing

void cGraphH::DoSync()
{
  ::SyncGraphs_Send(m_Parent);
}


void cGraphH::DoSynchronise(cGraph* graph)
{
  if (graph->GetType() != m_Type) return; // not a histogram
  cGraphH* hist = (cGraphH*)graph;

  if (hist->GetLogtype() != m_Logtype) return; // histogram shows other data

  bool changed = false;

  if (m_MinX > hist->GetMinX()) { m_MinX = hist->GetMinX(); changed = true; }
  if (m_MaxX < hist->GetMaxX()) { m_MaxX = hist->GetMaxX(); changed = true; }

  if (changed) {
    SetBins(hist->GetBins());
    // wxLogDebug(_T("%d adapts bbox from %d: %.2f to %.2f (%d bins)"), GetId(), hist->GetId(), m_MinX, m_MaxX, m_BinCount);

    AdjustDimensions();
    Refresh();
  }
  else {
    if ((m_MinX < hist->GetMinX()) || (m_MaxX > hist->GetMaxX())) {
      // make other histograms synchronise with this one
      RequestSync();
    }
  }
}
