#include "ccurve_y.h"
#include "cgraph.h"

//-----------------------------------------------------------------------------

cCurveY::cCurveY(cLap* lap)
: cCurve(lap)
{
  m_Logtype = LOGTYPE_FIRST;
  m_Wheel = WHEELTYPE_FIRST;
  m_DeltaLap = NULL;

  AdjustBbox();
}

//-----------------------------------------------------------------------------

cCurveY::~cCurveY()
{
}

//-----------------------------------------------------------------------------
// Prepare the curve's data for plotting
// - start, end = start and end of the part of the lap to be plotted
// - maxPoints = maximum number of plot points

void cCurveY::Rewind(float start, float end, size_t maxPoints)
{
  wxASSERT(end > start);

  m_NextState = m_Lap->FindStateAt(start);
  m_LastState = end;
  m_Finished = false;

  int stateCount = (int)(m_Lap->GetStateCount() * (end - start) / (GetMaxX() - GetMinX()));
  m_StateStep = stateCount / maxPoints;
  if (m_StateStep < 1) m_StateStep = 1;
}

//-----------------------------------------------------------------------------
// Get the next point on the curve
// x, y = receive the coordinates of the point

bool cCurveY::GetNextPoint(float& x, float& y)
{
  if (m_Finished) return false; // no more points to plot

  if (m_NextState >= m_Lap->GetStateCount()) {
    // past last data point
    m_NextState = m_Lap->GetStateCount() - 1;
    m_Finished = true; // no more points after this one
  }

  if (m_Lap->GetState(m_NextState)->GetDistance() > m_LastState) m_Finished = true;
  x = m_Lap->GetState(m_NextState)->GetDistance();
  y = m_Lap->GetState(m_NextState)->GetLogData(m_Logtype, m_Wheel);

  if (m_DeltaLap != NULL) {
    if (m_DeltaLap == m_Lap) {
      // in rare cases, a diff with itself does not yield zero, so we set it explicitly
      y = 0.0f;
    }
    else {
      // calculate difference with other lap
      y -= m_DeltaLap->GetLogDataAt(x, m_Logtype, m_Wheel);
    }
  }

  m_NextState += m_StateStep;
  return true;
}

//-----------------------------------------------------------------------------
// Get the curve values corresponding with a certain point in the lap
// - distance = distance-in-lap
// - x, y = X and Y values of the curve at that point

bool cCurveY::GetValueAt(float distance, float& x, float& y)
{
  x = distance;
  y = m_Lap->GetLogDataAt(distance, m_Logtype, m_Wheel);
  return true;
}

//-----------------------------------------------------------------------------
// Set the type of data that is shown
// - log = type of log-data
// - wheel = index of the wheel for which the data must be shown

void cCurveY::SetLogtype(int log, int wheel)
{
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  m_Logtype = log;
  m_Wheel = wheel;
  AdjustBbox();
}

//-----------------------------------------------------------------------------
// Set a reference to the delta-lap
// - lap = new delta-lap (may be NULL)

void cCurveY::SetDeltaLap(cLap* lap)
{
  m_DeltaLap = lap;
  AdjustBbox();
}
