#!/usr/bin/env python
#
# Copyright Alex McBride 2008-2010.
#
# This file is part of pyinsim.
#
# pyinsim is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# pyinsim is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with pyinsim. If not, see <http://www.gnu.org/licenses/>.

"""InSim module for the Python programming language.

"""

import socket
import struct
import threading
import traceback
import re
import select
import math
import array
import time
import _strmanip

__all__ = ['ADMIN_LEN', 'BFN_CLEAR', 'BFN_DEL_BTN', 'BFN_REQUEST', 'BFN_USER_CLEAR', 'CCI_BLUE', 'CCI_FIRST', 'CCI_LAG', 'CCI_LAST', 'CCI_YELLOW', 'CLOSE_LOST', 'CLOSE_REQUEST', 'COL_BLACK', 'COL_BLUE', 'COL_DARK_GREEN', 'COL_LIGHT_BLUE', 'COL_LIGHT_GREEN', 'COL_ORIGINAL', 'COL_PURPLE', 'COL_RED', 'COL_WHITE', 'COL_YELLOW', 'CONF_CONFIRMED', 'CONF_DID_NOT_PIT', 'CONF_DISQ', 'CONF_MENTIONED', 'CONF_PENALTY_30', 'CONF_PENALTY_45', 'CONF_PENALTY_DT', 'CONF_PENALTY_SG', 'CONF_TIME', 'CompCar', 'DL_ABS', 'DL_BATTERY', 'DL_FULLBEAM', 'DL_HANDBRAKE', 'DL_NUM', 'DL_OILWARN', 'DL_PITSPEED', 'DL_SHIFT', 'DL_SIGNAL_ANY', 'DL_SIGNAL_L', 'DL_SIGNAL_R', 'DL_SPARE', 'DL_TC', 'EVT_ALL', 'EVT_CLOSED', 'EVT_CONNECTED', 'EVT_ERROR', 'EVT_TIMEOUT', 'FLG_BLUE', 'FLG_OFF', 'FLG_ON', 'FLG_YELLOW', 'HInfo', 'HOSTF_CAN_RESET', 'HOSTF_CAN_SELECT', 'HOSTF_CAN_VOTE', 'HOSTF_CRUISE', 'HOSTF_FCV', 'HOSTF_MID_RACE', 'HOSTF_MUST_PIT', 'HOS_LICENSED', 'HOS_S1', 'HOS_S2', 'HOS_SPECPASS', 'INSIM_VERSION', 'INST_ALWAYS_ON', 'IRP_ARP', 'IRP_ARQ', 'IRP_ERR', 'IRP_HLR', 'IRP_HOS', 'IRP_SEL', 'IR_ARP', 'IR_ARQ', 'IR_ERR', 'IR_ERR_ADMIN', 'IR_ERR_HOSTNAME', 'IR_ERR_NOSPEC', 'IR_ERR_PACKET1', 'IR_ERR_PACKET2', 'IR_ERR_SPEC', 'IR_HLR', 'IR_HOS', 'IR_SEL', 'ISB_C1', 'ISB_C2', 'ISB_C4', 'ISB_CLICK', 'ISB_CTRL', 'ISB_DARK', 'ISB_LEFT', 'ISB_LIGHT', 'ISB_LMB', 'ISB_RIGHT', 'ISB_RMB', 'ISB_SHIFT', 'ISF_LOCAL', 'ISF_MCI', 'ISF_MSO_COLS', 'ISF_NLP', 'ISF_RES_0', 'ISF_RES_1', 'ISP_AXI', 'ISP_AXO', 'ISP_BFN', 'ISP_BTC', 'ISP_BTN', 'ISP_BTT', 'ISP_CCH', 'ISP_CNL', 'ISP_CPP', 'ISP_CPR', 'ISP_CRS', 'ISP_FIN', 'ISP_FLG', 'ISP_III', 'ISP_ISI', 'ISP_ISM', 'ISP_LAP', 'ISP_MCI', 'ISP_MOD', 'ISP_MSL', 'ISP_MSO', 'ISP_MST', 'ISP_MSX', 'ISP_MTC', 'ISP_NCN', 'ISP_NLP', 'ISP_NONE', 'ISP_NPL', 'ISP_PEN', 'ISP_PFL', 'ISP_PIT', 'ISP_PLA', 'ISP_PLL', 'ISP_PLP', 'ISP_PSF', 'ISP_REO', 'ISP_RES', 'ISP_RIP', 'ISP_RST', 'ISP_SCC', 'ISP_SCH', 'ISP_SFP', 'ISP_SMALL', 'ISP_SPX', 'ISP_SSH', 'ISP_STA', 'ISP_TINY', 'ISP_TOC', 'ISP_VER', 'ISP_VTN', 'ISS_FRONT_END', 'ISS_GAME', 'ISS_MPSPEEDUP', 'ISS_MULTI', 'ISS_PAUSED', 'ISS_REPLAY', 'ISS_SHIFTU', 'ISS_SHIFTU_FOLLOW', 'ISS_SHIFTU_HIGH', 'ISS_SHIFTU_NO_OPT', 'ISS_SHOW_2D', 'ISS_SOUND_MUTE', 'ISS_VIEW_OVERRIDE', 'ISS_VISIBLE', 'ISS_WINDOWED', 'IS_AXI', 'IS_AXO', 'IS_BFN', 'IS_BTC', 'IS_BTN', 'IS_BTT', 'IS_CCH', 'IS_CNL', 'IS_CPP', 'IS_CPR', 'IS_CRS', 'IS_FIN', 'IS_FLG', 'IS_III', 'IS_ISI', 'IS_ISM', 'IS_LAP', 'IS_MCI', 'IS_MOD', 'IS_MSL', 'IS_MSO', 'IS_MST', 'IS_MSX', 'IS_MTC', 'IS_NCN', 'IS_NLP', 'IS_NPL', 'IS_PEN', 'IS_PFL', 'IS_PIT', 'IS_PLA', 'IS_PLL', 'IS_PLP', 'IS_PSF', 'IS_REO', 'IS_RES', 'IS_RIP', 'IS_RST', 'IS_SCC', 'IS_SCH', 'IS_SFP', 'IS_SMALL', 'IS_SPX', 'IS_SSH', 'IS_STA', 'IS_TINY', 'IS_TOC', 'IS_VER', 'IS_VTN', 'IS_X_MAX', 'IS_X_MIN', 'IS_Y_MAX', 'IS_Y_MIN', 'InSim', 'Error', 'InSimRelay', 'VersionError', 'LEAVR_BANNED', 'LEAVR_DISCO', 'LEAVR_KICKED', 'LEAVR_LOSTCONN', 'LEAVR_SECURITY', 'LEAVR_TIMEOUT', 'MAX_PORT', 'MIN_PORT', 'MSO_O', 'MSO_PREFIX', 'MSO_SYSTEM', 'MSO_USER', 'NodeLap', 'OG_BAR', 'OG_KM', 'OG_TURBO', 'OUT_OUTGAUGE', 'OUT_OUTSIM', 'OutGauge', 'OutGaugePack', 'OutSim', 'OutSimPack', 'PENALTY_30', 'PENALTY_45', 'PENALTY_DT', 'PENALTY_DT_VALID', 'PENALTY_NONE', 'PENALTY_SG', 'PENALTY_SG_VALID', 'PENR_ADMIN', 'PENR_FALSE_START', 'PENR_SPEEDING', 'PENR_STOP_LATE', 'PENR_STOP_SHORT', 'PENR_UNKNOWN', 'PENR_WRONG_WAY', 'PIF_AUTOCLUTCH', 'PIF_AUTOGEARS', 'PIF_AXIS_CLUTCH', 'PIF_CUSTOM_VIEW', 'PIF_HELP_B', 'PIF_INPITS', 'PIF_KB_NO_HELP', 'PIF_KB_STABILISED', 'PIF_MOUSE', 'PIF_RESERVED_2', 'PIF_RESERVED_32', 'PIF_RESERVED_4', 'PIF_SHIFTER', 'PIF_SWAPSIDE', 'PITLANE_DT', 'PITLANE_ENTER', 'PITLANE_EXIT', 'PITLANE_NO_PURPOSE', 'PITLANE_SG', 'PSE_BODY_MAJOR', 'PSE_BODY_MINOR', 'PSE_FR_DAM', 'PSE_FR_WHL', 'PSE_LE_FR_DAM', 'PSE_LE_FR_WHL', 'PSE_LE_RE_DAM', 'PSE_LE_RE_WHL', 'PSE_NOTHING', 'PSE_NUM', 'PSE_REFUEL', 'PSE_RE_DAM', 'PSE_RE_WHL', 'PSE_RI_FR_DAM', 'PSE_RI_FR_WHL', 'PSE_RI_RE_DAM', 'PSE_RI_RE_WHL', 'PSE_SETUP', 'PSE_STOP', 'PYINSIM_VERSION', 'RIPOPT_LOOP', 'RIPOPT_SKINS', 'RIP_ALREADY', 'RIP_CORRUPTED', 'RIP_DEDICATED', 'RIP_DEST_OOB', 'RIP_NOT_FOUND', 'RIP_NOT_REPLAY', 'RIP_OK', 'RIP_OOS', 'RIP_UNKNOWN', 'RIP_UNLOADABLE', 'RIP_USER', 'RIP_WRONG_MODE', 'SETF_ABS_ENABLE', 'SETF_SYMM_WHEELS', 'SETF_TC_ENABLE', 'SMALL_NLI', 'SMALL_NONE', 'SMALL_RTP', 'SMALL_SSG', 'SMALL_SSP', 'SMALL_STP', 'SMALL_TMS', 'SMALL_VTA', 'SND_ERROR', 'SND_INVALIDKEY', 'SND_MESSAGE', 'SND_SILENT', 'SND_SYSMESSAGE', 'SSH_CORRUPTED', 'SSH_DEDICATED', 'SSH_NO_SAVE', 'SSH_OK', 'TINY_AXC', 'TINY_AXI', 'TINY_CLOSE', 'TINY_CLR', 'TINY_GTH', 'TINY_ISM', 'TINY_MCI', 'TINY_MPE', 'TINY_NCN', 'TINY_NLP', 'TINY_NONE', 'TINY_NPL', 'TINY_PING', 'TINY_REN', 'TINY_REO', 'TINY_REPLY', 'TINY_RES', 'TINY_RIP', 'TINY_RST', 'TINY_SCP', 'TINY_SST', 'TINY_VER', 'TINY_VTC', 'TYRE_HYBRID', 'TYRE_KNOBBLY', 'TYRE_NOT_CHANGED', 'TYRE_R1', 'TYRE_R2', 'TYRE_R3', 'TYRE_R4', 'TYRE_ROAD_NORMAL', 'TYRE_ROAD_SUPER', 'VIEW_ANOTHER', 'VIEW_CAM', 'VIEW_CUSTOM', 'VIEW_DRIVER', 'VIEW_FOLLOW', 'VIEW_HELI', 'VOTE_END', 'VOTE_NONE', 'VOTE_QUALIFY', 'VOTE_RESTART', 'WEA_BRIGHTCLEAR', 'WEA_CLOUDYSUNSETDUSK', 'WEA_OVERCAST', 'WND_HIGH', 'WND_LOW', 'WND_NONE', 'checkAdmin', 'checkIP', 'checkPort', 'collision', 'degrees', 'distance', 'fromUnicode', 'km', 'kph', 'lookup', 'metres', 'miles', 'mph', 'mps', 'mps2kph', 'mps2mph', 'newEvent', 'parseCmd', 'rad2degrees', 'rad2rpm', 'sleep', 'stripCols', 'time2ms', 'timer', 'timestr', 'toUnicode', 'version']

# General constants
INSIM_VERSION = 4
PYINSIM_VERSION = '1.6.4'
MAX_PORT = 65535
MIN_PORT = 0
ADMIN_LEN = 16
_BUFFER_SIZE = 2048
_IP_REGEX = re.compile(r'\b(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2\
[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25\
[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b')
_COLOUR_REGEX = re.compile('\^[0-9]')
_CURR_EVT = 255

# Enum for pyinsim events
EVT_CONNECTED = 60
EVT_CLOSED = 61
EVT_ERROR = 62
EVT_TIMEOUT = 63
EVT_ALL = 64

# Enum for close reasons
CLOSE_LOST = 1
CLOSE_REQUEST = 2

# Enum for packet types
ISP_NONE = 0
ISP_ISI = 1
ISP_VER = 2
ISP_TINY = 3
ISP_SMALL = 4
ISP_STA = 5
ISP_SCH = 6
ISP_SFP = 7
ISP_SCC = 8
ISP_CPP = 9
ISP_ISM = 10
ISP_MSO = 11
ISP_III = 12
ISP_MST = 13
ISP_MTC = 14
ISP_MOD = 15
ISP_VTN = 16
ISP_RST = 17
ISP_NCN = 18
ISP_CNL = 19
ISP_CPR = 20
ISP_NPL = 21
ISP_PLP = 22
ISP_PLL = 23
ISP_LAP = 24
ISP_SPX = 25
ISP_PIT = 26
ISP_PSF = 27
ISP_PLA = 28
ISP_CCH = 29
ISP_PEN = 30
ISP_TOC = 31
ISP_FLG = 32
ISP_PFL = 33
ISP_FIN = 34
ISP_RES = 35
ISP_REO = 36
ISP_NLP = 37
ISP_MCI = 38
ISP_MSX = 39
ISP_MSL = 40
ISP_CRS = 41
ISP_BFN = 42
ISP_AXI = 43
ISP_AXO = 44
ISP_BTN = 45
ISP_BTC = 46
ISP_BTT = 47
ISP_RIP = 48
ISP_SSH = 49
# Special packets
_ISP_NODELAP = 101
_ISP_COMPCAR = 102
OUT_OUTGAUGE = 103
OUT_OUTSIM = 104
# InSim Relay Packets
IRP_ARQ = 250
IRP_ARP = 251
IRP_HLR = 252
IRP_HOS = 253
IRP_SEL = 254
IRP_ERR = 255
# Special InSim relay packets.
_IRP_HINFO = 105

# Enum for IS_TINY sub-type
TINY_NONE = 0
TINY_VER = 1
TINY_CLOSE = 2
TINY_PING = 3
TINY_REPLY = 4
TINY_VTC = 5
TINY_SCP = 6
TINY_SST = 7
TINY_GTH = 8
TINY_MPE = 9
TINY_ISM = 10
TINY_REN = 11
TINY_CLR = 12
TINY_NCN = 13
TINY_NPL = 14
TINY_RES = 15
TINY_NLP = 16
TINY_MCI = 17
TINY_REO = 18
TINY_RST = 19
TINY_AXI = 20
TINY_AXC = 21
TINY_RIP = 22

# Enum for IS_SMALL sub-type
SMALL_NONE = 0
SMALL_SSP = 1
SMALL_SSG = 2
SMALL_VTA = 3
SMALL_TMS = 4
SMALL_STP = 5
SMALL_RTP = 6
SMALL_NLI = 7

# Bit flags for ISI Flags
ISF_RES_0 = 1
ISF_RES_1 = 2
ISF_LOCAL = 4
ISF_MSO_COLS = 8
ISF_NLP = 16
ISF_MCI = 32

# Enum for IS_MSO UserType
MSO_SYSTEM = 0
MSO_USER = 1
MSO_PREFIX = 2
MSO_O = 3

# Enum for IS_MSL Sound
SND_SILENT = 0
SND_MESSAGE = 1
SND_SYSMESSAGE = 2
SND_INVALIDKEY = 3
SND_ERROR = 4

# Enum for IS_VTN Action
VOTE_NONE = 0
VOTE_END = 1
VOTE_RESTART = 2
VOTE_QUALIFY = 3

# Enum for IS_PLA Fact
PITLANE_EXIT = 0
PITLANE_ENTER = 1
PITLANE_NO_PURPOSE = 2
PITLANE_DT = 3
PITLANE_SG = 4

# Enum for IS_STA InGameCam
VIEW_FOLLOW = 0
VIEW_HELI = 1
VIEW_CAM = 2
VIEW_DRIVER = 3
VIEW_CUSTOM = 4
VIEW_ANOTHER = 255

# Enum for IS_CNL Reason
LEAVR_DISCO = 0
LEAVR_TIMEOUT = 1
LEAVR_LOSTCONN = 2
LEAVR_KICKED = 3
LEAVR_BANNED = 4
LEAVR_SECURITY = 5

# Enum for IS_PEN Penalty
PENALTY_NONE = 0
PENALTY_DT = 1
PENALTY_DT_VALID = 2
PENALTY_SG = 3
PENALTY_SG_VALID = 4
PENALTY_30 = 5
PENALTY_45 = 6

# Enum for IS_PEN Reason
PENR_UNKNOWN = 1
PENR_ADMIN = 2
PENR_WRONG_WAY = 3
PENR_FALSE_START = 4
PENR_SPEEDING = 5
PENR_STOP_SHORT = 6
PENR_STOP_LATE = 7

# Enum for IS_PIT Tyres
TYRE_R1 = 0
TYRE_R2 = 1
TYRE_R3 = 2
TYRE_R4 = 3
TYRE_ROAD_SUPER = 4
TYRE_ROAD_NORMAL = 5
TYRE_HYBRID = 6
TYRE_KNOBBLY = 7
TYRE_NOT_CHANGED = 255

# Bit flags for IS_STA Flags
ISS_GAME = 1
ISS_REPLAY = 2
ISS_PAUSED = 4
ISS_SHIFTU = 8
ISS_SHIFTU_HIGH    = 16
ISS_SHIFTU_FOLLOW = 32
ISS_SHIFTU_NO_OPT = 64
ISS_SHOW_2D = 128
ISS_FRONT_END = 256
ISS_MULTI = 512
ISS_MPSPEEDUP =    1024
ISS_WINDOWED = 2048
ISS_SOUND_MUTE = 4096
ISS_VIEW_OVERRIDE = 8192
ISS_VISIBLE = 16384

# Bit flags for IS_PIT Work
PSE_NOTHING = 1
PSE_STOP = 2
PSE_FR_DAM = 4
PSE_FR_WHL = 8
PSE_LE_FR_DAM = 16
PSE_LE_FR_WHL = 32
PSE_RI_FR_DAM = 64
PSE_RI_FR_WHL = 128
PSE_RE_DAM = 256
PSE_RE_WHL = 512
PSE_LE_RE_DAM = 1024
PSE_LE_RE_WHL = 2048
PSE_RI_RE_DAM = 4096
PSE_RI_RE_WHL = 8192
PSE_BODY_MINOR = 16384
PSE_BODY_MAJOR = 32768
PSE_SETUP = 65536
PSE_REFUEL = 131072
PSE_NUM = 262144

# Bit flags for IS_NPL Flags
PIF_SWAPSIDE = 1
PIF_RESERVED_2 = 2
PIF_RESERVED_4 = 4
PIF_AUTOGEARS = 8
PIF_SHIFTER = 16
PIF_RESERVED_32 = 32
PIF_HELP_B = 64
PIF_AXIS_CLUTCH = 128
PIF_INPITS = 256
PIF_AUTOCLUTCH = 512
PIF_MOUSE = 1024
PIF_KB_NO_HELP = 2048
PIF_KB_STABILISED = 4096
PIF_CUSTOM_VIEW = 8192

# Bit flags for IS_RES Confirm
CONF_MENTIONED = 1
CONF_CONFIRMED = 2
CONF_PENALTY_DT = 4
CONF_PENALTY_SG = 8
CONF_PENALTY_30 = 16
CONF_PENALTY_45 = 32
CONF_DID_NOT_PIT = 64
CONF_DISQ = (CONF_PENALTY_DT | CONF_PENALTY_SG | CONF_DID_NOT_PIT)
CONF_TIME = (CONF_PENALTY_30 | CONF_PENALTY_45)

# Bit flags for IS_RST Flags
HOSTF_CAN_VOTE = 1
HOSTF_CAN_SELECT = 2
HOSTF_MID_RACE = 32
HOSTF_MUST_PIT = 64
HOSTF_CAN_RESET = 128
HOSTF_FCV = 256
HOSTF_CRUISE = 512

# Bit flags for CompCar Info
CCI_BLUE = 1
CCI_YELLOW = 2
CCI_LAG    = 32
CCI_FIRST = 64
CCI_LAST = 128

# Recomended area for IS_BTN T, L, W, H
IS_X_MIN = 0
IS_X_MAX = 110
IS_Y_MIN = 30
IS_Y_MAX = 170

# Enum for IS_BFN SubT
BFN_DEL_BTN = 0
BFN_CLEAR = 1
BFN_USER_CLEAR = 2
BFN_REQUEST = 3
INST_ALWAYS_ON = 128

# Bit flags for IS_BTN BStyle
ISB_C1 = 1
ISB_C2 = 2
ISB_C4 = 4
ISB_CLICK = 8
ISB_LIGHT = 16
ISB_DARK = 32
ISB_LEFT = 64
ISB_RIGHT = 128

# Bit flags for BTN CFlags
ISB_LMB = 1
ISB_RMB = 2
ISB_CTRL = 4
ISB_SHIFT = 8

# Enum for Weather
WEA_BRIGHTCLEAR = 0
WEA_OVERCAST = 1
WEA_CLOUDYSUNSETDUSK = 2

# Enum for Wind
WND_NONE = 0
WND_LOW = 1
WND_HIGH = 2

# Enum for IS_FLG Flag
FLG_BLUE = 1
FLG_YELLOW = 2

# Enum for IS_FLG OffOn
FLG_OFF = 0
FLG_ON = 1

# Enum for IS_RIP Error
RIP_OK = 0
RIP_ALREADY = 1
RIP_DEDICATED = 2
RIP_WRONG_MODE = 3
RIP_NOT_REPLAY = 4
RIP_CORRUPTED = 5
RIP_NOT_FOUND = 6
RIP_UNLOADABLE = 7
RIP_DEST_OOB = 8
RIP_UNKNOWN = 9
RIP_USER = 10
RIP_OOS = 11

# Enum for IS_RIP Options
RIPOPT_LOOP    = 1
RIPOPT_SKINS = 2

# Enum for IS_SSH Error
SSH_OK = 0
SSH_DEDICATED = 1
SSH_CORRUPTED = 2
SSH_NO_SAVE = 3

# Bit flags for IS_NPL SetF
SETF_SYMM_WHEELS = 1
SETF_TC_ENABLE = 2
SETF_ABS_ENABLE    = 4

# Colour constants.
COL_BLACK = '^0'
COL_RED = '^1'
COL_LIGHT_GREEN = '^2'
COL_YELLOW = '^3'
COL_BLUE = '^4'
COL_PURPLE = '^5'
COL_LIGHT_BLUE = '^6'
COL_WHITE = '^7'
COL_DARK_GREEN = '^8'
COL_ORIGINAL = '^9'

# Bits for OutGaugePack Flags
OG_TURBO = 8192
OG_KM = 16384
OG_BAR = 32768

# Bits for OutGaugePack DashLights and ShowLights
DL_SHIFT = 1
DL_FULLBEAM = 2
DL_HANDBRAKE = 4
DL_PITSPEED = 8
DL_TC = 16
DL_SIGNAL_L = 32
DL_SIGNAL_R = 64
DL_SIGNAL_ANY = 128
DL_OILWARN = 256
DL_BATTERY = 512
DL_ABS = 1024
DL_SPARE = 2048
DL_NUM= 4096

# InSim Relay host flags
HOS_SPECPASS = 1
HOS_LICENSED = 2
HOS_S1 = 4
HOS_S2 = 8
 
# InSim Relay error codes
IR_ERR_PACKET1 = 1
IR_ERR_PACKET2 = 2
IR_ERR_HOSTNAME = 3
IR_ERR_ADMIN = 4
IR_ERR_SPEC = 5
IR_ERR_NOSPEC = 6


def checkIP(ipStr):
    """Check if a string contains a valid IP address.
    
    """
    if ipStr.lower() == 'localhost':
        return True
    return _IP_REGEX.match(ipStr) is not None


def checkPort(portStr):
    """Check if a string contains a valid port number.

    """
    if portStr.isdigit():
        return int(portStr) >= MIN_PORT and int(portStr) <= MAX_PORT
    return False


def checkAdmin(adminStr):
    """Check if a string contains a valid admin password.

    """
    return len(adminStr) < ADMIN_LEN


def parseCmd(mso):
    """Parse a command string from an MSO message (e.g. '!admin DarkTimes') and 
    return it as a list.
    
    """
    if mso.__class__.__name__ == 'IS_MSO' and mso.UserType == MSO_PREFIX:
        msg = mso.Msg[mso.TextStart:]
        tokens = msg.split(' ')
        if len(tokens) > 0:
            return (tokens[0][1:], tokens[1:])
    return []

def toUnicode(lfsStr, default='L', cols=True):
    """Convert a LFS encoded string to unicode.

    """
    return _strmanip.toUnicode(lfsStr, default=default, cols=cols)

def fromUnicode(ustr, default='L'):
    """Convert unicode to a LFS encoded string.

    """
    return _strmanip.fromUnicode(ustr, default)


def stripCols(lfsStr):
    """Strip colour codes (^1, ^3 etc..) from a string.

    """
    return _COLOUR_REGEX.sub('', lfsStr)


def mps(speed):
    """Convert speed to meters per second.

    """
    return speed / 327.68


def kph(speed):
    """Convert speed to kilometers per hour.

    """
    return speed / 91.02


def mph(speed):
    """Convert speed to miles per hour.

    """
    return speed / 146.486067


def mps2kph(mps):
    """Convert meters per second to kilometers per hour.

    """
    return mps * 3.6


def mps2mph(mps):
    """Convert meters per second to miles per hour.

    """
    return mps * 2.23


def miles(meters):
    """Convert meters to miles.

    """
    return meters / 1609.344


def km(meters):
    """Convert meters to kilometers.

    """
    return meters / 1000


def degrees(direction):
    """Convert direction to degrees.

    """
    return direction / 182.04


def metres(length):
    """Convert length to meters.

    """
    return length / 65536.0


def rad2degrees(radians):
    """Convert radians to degrees.

    """
    return radians * (180.0 / math.pi);


def rad2rpm(radians):
    """Convert radians to RPM.

    """
    return radians * (60.0 / (2.0 * math.pi));


def distance(a=(0, 0, 0), b=(0, 0, 0)):
    """Determine the distance between two points.

    """
    return math.sqrt((b[0] - a[0]) * (b[0] - a[0]) + (b[1] - a[1]) *
                     (b[1] - a[1]) + (b[2] - a[2]) * (b[2] - a[2]))


def collision(a=(0, 0, 0, 0), b=(0, 0, 0, 0)):
    """Determine if two rectangles are colliding.

    """
    x1 = a[0] + a[2]
    y1 = a[1] + a[3]
    x3 = b[0] + b[2]
    y3 = b[1] + b[3]
    return not (x1 < b[0] or x3 < a[0] or y1 < b[1] or y3 < a[1])


def time(ms):
    """Convert milliseconds into hours, minutes, seconds and thousanths.

    """
    h = ms / 3600000
    m = ms / 60000 % 60
    s = ms / 1000 % 60
    t = ms % 1000
    return [h, m, s, t]


def time2ms(time=[0, 0, 0, 0]):
    """Convert hours, minutes, seconds and thousandths into milliseconds.

    """
    ms = time[0] * 3600000
    ms += time[1] * 60000
    ms += time[2] * 1000
    ms += time[3]
    return ms


def timestr(ms, hours=False):
    """Convert milliseconds into a formatted time string (e.g. h:mm:ss.ttt).

    """
    h, m, s, t = time(ms)
    if h or hours:
        return '%d.%.2d:%.2d.%.3d' % (h, m, s, t)
    else:
        return '%d:%.2d.%.3d' % (m, s, t)


def timer(callback, timeout, data=None):
    """Execute a function callback after a certain amount of time has
    elapsed.

    """
    t = None
    if data:
        t = threading.Timer(timeout, callback, [data])
    else:
        t = threading.Timer(timeout, callback)
    t.start()
    return t


def newEvent():
    """Create a new event identifier.

    """
    global _CURR_EVT
    _CURR_EVT += 1
    return _CURR_EVT


def version(reqVer=PYINSIM_VERSION, orBetter=True):
    """Determine if the correct version of pyinsim is installed.

    """
    if orBetter:
        return PYINSIM_VERSION >= reqVer
    return reqVer == PYINSIM_VERSION


def sleep(seconds):
    """Delay execution for the given number of seconds.
    
    """
    time.sleep(seconds)


def _packetSize(data):
    """Get the size of the first packet in the data."""
    try:
        return data[0]
    except IndexError:
        return 0

def _packetType(data, udp):
    """Get the type of the packet enumeration"""
    if udp:
        length = len(data)
        if length in (92, 96):
            return OUT_OUTGAUGE
        elif length in (64, 68):
            return OUT_OUTSIM
    try:                
        return data[1]
    except IndexError:
        return ISP_NONE 

def _subType(data):
    """Get the SubT of a packet (e.g. ``TINY_*`` or ``SMALL_*``)."""
    return data[3]

def _eatNullChars(str):
    """Remove trailing NULL chars from C-style strings."""
    return str.rstrip('\000')


# Pre-compiled structs
_STRUCTS = {ISP_ISI: struct.Struct('4B2HBcH15sx15sx'),
            ISP_VER: struct.Struct('4B7sx5sxH'),
            ISP_TINY: struct.Struct('4B'),
            ISP_SMALL: struct.Struct('4BI'),
            ISP_STA: struct.Struct('4BfH10B5sx2B'),
            ISP_SCH: struct.Struct('8B'),
            ISP_SFP: struct.Struct('4BH2B'),
            ISP_SCC: struct.Struct('8B'),
            ISP_CPP: struct.Struct('4B3i3H2Bf2H'),
            ISP_ISM: struct.Struct('8B31sx'),
            ISP_MSO: struct.Struct('8B127sx'),
            ISP_III: struct.Struct('8B63sx'),
            ISP_MST: struct.Struct('4B63sx'),
            ISP_MTC: struct.Struct('8B63sx'),
            ISP_MOD: struct.Struct('4B4i'),
            ISP_VTN: struct.Struct('8B'),
            ISP_RST: struct.Struct('8B5sx2B6H'),
            ISP_NCN: struct.Struct('4B23sx23sx4B'),
            ISP_CNL: struct.Struct('8B'),
            ISP_CPR: struct.Struct('4B23sx7sx'),
            ISP_NPL: struct.Struct('6BH23sx8s3sx15sx8Bi4B'),
            ISP_PLP: struct.Struct('4B'),
            ISP_PLL: struct.Struct('4B'),
            ISP_LAP: struct.Struct('4B2I2H4B'),
            ISP_SPX: struct.Struct('4B2I4B'),
            ISP_PIT: struct.Struct('4B2H8B2I'),
            ISP_PSF: struct.Struct('4B2I'),
            ISP_PLA: struct.Struct('8B'),
            ISP_CCH: struct.Struct('8B'),
            ISP_PEN: struct.Struct('8B'),
            ISP_TOC: struct.Struct('8B'),
            ISP_FLG: struct.Struct('8B'),
            ISP_PFL: struct.Struct('4B2H'),
            ISP_FIN: struct.Struct('4B2I4B2H'),
            ISP_RES: struct.Struct('4B23sx23sx7sx3sx2I4B2H2BH'),
            ISP_REO: struct.Struct('4B'), # Note PLID (32B) handled in packet.
            ISP_NLP: struct.Struct('4B'),
            ISP_MCI: struct.Struct('4B'),
            ISP_MSX: struct.Struct('4B95sx'),
            ISP_MSL: struct.Struct('4B127sx'),
            ISP_CRS: struct.Struct('4B'),
            ISP_BFN: struct.Struct('8B'),
            ISP_AXI: struct.Struct('6BH31sx'),
            ISP_AXO: struct.Struct('4B'),
            ISP_BTN: struct.Struct('12B'), #239sx'), # Text handled in packet
            ISP_BTC: struct.Struct('8B'),
            ISP_BTT: struct.Struct('8B95sx'),
            ISP_RIP: struct.Struct('8B2H63sx'),
            ISP_SSH: struct.Struct('8B31sx'),
            _ISP_NODELAP: struct.Struct('2H2B'),
            _ISP_COMPCAR: struct.Struct('2H4B3i3Hh'),
            OUT_OUTGAUGE: struct.Struct('i3sxH2B7f2i3f15sx15sx'),
            OUT_OUTSIM: struct.Struct('I12f3i'),
            IRP_ARP: struct.Struct('4B'),
            IRP_ARQ: struct.Struct('4B'),
            IRP_HLR: struct.Struct('4B'),
            IRP_HOS: struct.Struct('4B'),
            IRP_SEL: struct.Struct('4B31sx15sx15sx'), 
            IRP_ERR: struct.Struct('4B'),
            _IRP_HINFO: struct.Struct('31sx5sx2B'),}


# Packets
class IS_ISI(object):
    """InSim Init - packet to initialise the InSim system.

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'UDPPort', 'Flags', 'Sp0', 'Prefix', 'Interval', 'Admin', 'IName')
    def __init__(self, ReqI=0, UDPPort=0, Flags=0, Prefix=' ', Interval=0, Admin='', IName=''):
        """Create a new IS_ISI packet.

        Args:
            ReqI     : If non-zero LFS will send an ``IS_VER`` packet
            UDPPort  : Port for UDP replies from LFS (0 to 65535)
            Flags    : ``ISF_*`` bit flags for options
            Prefix   : Special host message prefix character
            Interval : Time in ms between ``IS_NLP`` or ``IS_MCI`` packets (0 = none)
            Admin    : Admin password (if set in LFS)
            IName    : A short name for your program

        """
        self.Size = 44
        self.Type = ISP_ISI
        self.ReqI = ReqI
        self.Zero = 0
        self.UDPPort = UDPPort
        self.Flags = Flags
        self.Sp0 = 0
        self.Prefix = Prefix
        self.Interval = Interval
        self.Admin = Admin
        self.IName = IName
    def pack(self):
        return _STRUCTS[ISP_ISI].pack(self.Size, self.Type, self.ReqI, self.Zero, self.UDPPort, self.Flags, self.Sp0, self.Prefix, self.Interval, self.Admin, self.IName)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.UDPPort, self.Flags, self.Sp0, self.Prefix, self.Interval, self.Admin, self.IName = _STRUCTS[ISP_ISI].unpack(data)
        self.Admin = _eatNullChars(self.Admin)
        self.IName = _eatNullChars(self.IName)
        return self


class IS_VER(object):
    """VERsion. It is advisable to request version information as soon as you
    have connected, to avoid problems when connecting to a host with a later or
    earlier version. You will be sent a version packet on connection if you
    set ``ReqI`` in the ``IS_ISI`` packet.

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'Version', 'Product', 'InSimVer')
    def __init__(self):
        """Initialise a new IS_VER packet.

        """
        self.Size = 20
        self.Type = ISP_VER
        self.ReqI = 0
        self.Zero = 0
        self.Version = ''
        self.Product = ''
        self.InSimVer = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.Version, self.Product, self.InSimVer = _STRUCTS[ISP_VER].unpack(data)
        self.Version = _eatNullChars(self.Version)
        self.Product = _eatNullChars(self.Product)
        return self


class IS_TINY(object):
    """General purpose packet.

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'SubT')
    def __init__(self, ReqI=0, SubT=TINY_NONE):
        """Initialise a new IS_TINY packet.

        Args:
            ReqI : zero (0) unless in response to a request.
            SubT : subtype from ``TINY_*`` enumeration (e.g. ``TINY_REN``)

        """
        self.Size = 4
        self.Type = ISP_TINY
        self.ReqI = ReqI
        self.SubT = SubT
    def pack(self):
        return _STRUCTS[ISP_TINY].pack(self.Size, self.Type, self.ReqI, self.SubT)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.SubT = _STRUCTS[ISP_TINY].unpack(data)
        return self


class IS_SMALL(object):
    """General purpose packet.

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'SubT', 'UVal')
    def __init__(self, ReqI=0, SubT=SMALL_NONE, UVal=0):
        """Initialise a new IS_SMALL packet.

        Args:
            ReqI : zero (0) unless in response to a request.
            SubT : subtype from ``SMALL_*`` enumeration (e.g. ``SMALL_SSP``)
            UVal : value (e.g. for ``SMALL_SSP`` this would be the OutSim packet rate)

        """
        self.Size = 8
        self.Type = ISP_SMALL
        self.ReqI = ReqI
        self.SubT = SubT
        self.UVal = UVal
    def pack(self):
        return _STRUCTS[ISP_SMALL].pack(self.Size, self.Type, self.ReqI, self.SubT, self.UVal)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.SubT, self.UVal = _STRUCTS[ISP_SMALL].unpack(data)
        return self


class IS_STA(object):
    """STAte packet, sent whenever the data in the packet changes. To request
    this packet send a ``IS_TINY`` with a ``ReqI`` of non-zero and a ``SubT`` of ``TINY_STA``.

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'ReplaySpeed', 'Flags', 'InGameCam', 'ViewPLID', 'NumP', 'NumConns', 'NumFinished', 'RaceInProg', 'QualMins', 'RaceLaps', 'Spare2', 'Spare3', 'Track', 'Weather', 'Wind')
    def __init__(self):
        """Initialise a new IS_STA packet.

        """
        self.Size = 28
        self.Type = ISP_STA
        self.ReqI = 0
        self.Zero = 0
        self.ReplaySpeed = 0.0
        self.Flags = 0
        self.InGameCam = 0
        self.ViewPLID = 0
        self.NumP = 0
        self.NumConns = 0
        self.NumFinished = 0
        self.RaceInProg = 0
        self.QualMins = 0
        self.RaceLaps = 0
        self.Spare2 = 0
        self.Spare3 = 0
        self.Track = ''
        self.Weather = 0
        self.Wind = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.ReplaySpeed, self.Flags, self.InGameCam, self.ViewPLID, self.NumP, self.NumConns, self.NumFinished, self.RaceInProg, self.QualMins, self.RaceLaps, self.Spare2, self.Spare3, self.Track, self.Weather, self.Wind = _STRUCTS[ISP_STA].unpack(data)
        self.Track = _eatNullChars(self.Track)
        return self


class IS_SCH(object):
    """Single CHaracter

    You can send individual key presses to LFS with the :class:`IS_SCH` packet. For
    standard keys (e.g. V and H) you should send a capital letter. This does
    not work with some keys like F keys, arrows or CTRL keys. You can also use
    :class:`IS_MST` with the /press /shift /ctrl /alt commands.

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'CharB', 'Flags', 'Spare2', 'Spare3')
    def __init__(self, ReqI=0, CharB=0, Flags=0):
        """Initialise a new IS_SCH packet.

        Args:
            ReqI  : 0
            CharB : key to press
            Flags : bit 0 : SHIFT / bit 1 : CTRL

        """
        self.Size = 8
        self.Type = ISP_SCH
        self.ReqI = ReqI
        self.Zero = 0
        self.CharB = CharB
        self.Flags = Flags
        self.Spare2 = 0
        self.Spare3 = 0
    def pack(self):
        return _STRUCTS[ISP_SCH].pack(self.Size, self.Type, self.ReqI, self.Zero, self.CharB, self.Flags, self.Spare2, self.Spare3)


class IS_SFP(object):
    """State Flags Pack. Send this packet to set the game state. Other states
    must be set by using key-presses or slash commands.

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'Flag', 'OffOn', 'Sp3')
    def __init__(self, ReqI=0, Zero=0, Flag=0, OffOn=0, Sp3=0):
        """Initialise a new IS_SFP packet.

        Args:
            ReqI  : ReqI as received in the request packet
            Flag  : ``ISS_*`` state flags
            OffOn : 0 = off / 1 = on

        """
        self.Size = 8
        self.Type = ISP_SFP
        self.ReqI = ReqI
        self.Zero = 0
        self.Flag = Flag
        self.OffOn = OffOn
        self.Sp3 = 0
    def pack(self):
        return _STRUCTS[ISP_SFP].pack(self.Size, self.Type, self.ReqI, self.Zero, self.Flag, self.OffOn, self.Sp3)


class IS_SCC(object):
    """Set Car Camera - Simplified camera packet (not SHIFT+U mode)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'ViewPLID', 'InGameCam', 'Sp2', 'Sp3')
    def __init__(self, ReqI=0, ViewPLID=0, InGameCam=0):
        """Initialise a new IS_SCC packet.

        Args:
            ReqI      : 0
            ViewPLID  : UniqueID of player to view
            InGameCam : InGameCam (as reported in StatePack)

        """
        self.Size = 8
        self.Type = ISP_SCC
        self.ReqI = ReqI
        self.Zero = 0
        self.ViewPLID = ViewPLID
        self.InGameCam = InGameCam
        self.Sp2 = 0
        self.Sp3 = 0
    def pack(self):
        return _STRUCTS[ISP_SCC].pack(self.Size, self.Type, self.ReqI, self.Zero, self.ViewPLID, self.InGameCam, self.Sp2, self.Sp3)


class IS_CPP(object):
    """Cam Pos Pack - Full camera packet (in car or SHIFT+U mode)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'Pos', 'H', 'P', 'R', 'ViewPLID', 'InGameCam', 'FOV', 'Time', 'Flags')
    def __init__(self, ReqI=0, Pos=[0,0,0], H=0, P=0, R=0, ViewPLID=0, InGameCam=0, FOV=0.0, Time=0, Flags=0):
        """Initialise a new IS_CPP packet.

        Args:
            ReqI      : instruction : 0 / or reply : ReqI as received in the ``TINY_SCP``
            Pos       : Position vector
            H         : heading - 0 points along Y axis
            P         : pitch - 0 means looking at horizon
            R         : roll - 0 means no roll
            ViewPLID  : Unique ID of viewed player (0 = none)
            InGameCam : InGameCam (as reported in StatePack)
            FOV       : FOV in degrees
            Time      : Time to get there (0 means instant + reset)
            Flags     : state flags from ``ISS_*``

        """
        self.Size = 32
        self.Type = ISP_CPP
        self.ReqI = ReqI
        self.Zero = 0
        self.Pos = Pos
        self.H = H
        self.P = P
        self.R = R
        self.ViewPLID = ViewPLID
        self.InGameCam = InGameCam
        self.FOV = FOV
        self.Time = Time
        self.Flags = Flags
    def pack(self):
        return _STRUCTS[ISP_CPP].pack(self.Size, self.Type, self.ReqI, self.Zero, self.Pos[0], self.Pos[1], self.Pos[2], self.H, self.P, self.R, self.ViewPLID, self.InGameCam, self.FOV, self.Time, self.Flags)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.Pos[0], self.Pos[1], self.Pos[2], self.H, self.P, self.R, self.ViewPLID, self.InGameCam, self.FOV, self.Time, self.Flags = _STRUCTS[ISP_CPP].unpack(data)
        return self


class IS_ISM(object):
    """InSim Multi

    LFS will send this packet when a host is started or joined.

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'Host', 'Sp1', 'Sp2', 'Sp3', 'HName')
    def __init__(self):
        """Initialise a new IS_ISM packet.

        """
        self.Size = 40
        self.Type = ISP_ISM
        self.ReqI = 0
        self.Zero = 0
        self.Host = 0
        self.Sp1 = 0
        self.Sp2 = 0
        self.Sp3 = 0
        self.HName = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.Host, self.Sp1, self.Sp2, self.Sp3, self.HName = _STRUCTS[ISP_ISM].unpack(data)
        self.HName = _eatNullChars(self.HName)
        return self


class IS_MSO(object):
    """MSg Out - system messages and user messages

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'UCID', 'PLID', 'UserType', 'TextStart', 'Msg')
    def __init__(self):
        """Initialise a new IS_MSO packet.

        """
        self.Size = 136
        self.Type = ISP_MSO
        self.ReqI = 0
        self.Zero = 0
        self.UCID =0
        self.PLID = 0
        self.UserType = 0
        self.TextStart = 0
        self.Msg = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.UCID, self.PLID, self.UserType, self.TextStart, self.Msg = _STRUCTS[ISP_MSO].unpack(data)
        self.Msg = _eatNullChars(self.Msg)
        return self


class IS_III(object):
    """InsIm Info - /i message from user to host's InSim

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'UCID', 'PLID', 'Sp2', 'Sp3', 'Msg')
    def __init__(self):
        """Initialise a new IS_III packet.

        """
        self.Size = 72
        self.Type = ISP_III
        self.ReqI = 0
        self.Zero = 0
        self.UCID = 0
        self.PLID = 0
        self.Sp2 = 0
        self.Sp3 = 0
        self.Msg = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.UCID, self.PLID, self.Sp2, self.Sp3, self.Msg = _STRUCTS[ISP_III].unpack(data)
        self.Msg = _eatNullChars(self.Msg)
        return self


class IS_MST(object):
    """MSg Type - send to LFS to type message or command

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'Msg')
    def __init__(self, ReqI=0, Msg=''):
        """Initialise a new IS_MST packet.

        Args:
            ReqI : 0
            Msg  : message (64 characters)

        """
        self.Size = 68
        self.Type = ISP_MST
        self.ReqI = ReqI
        self.Zero = 0
        self.Msg = Msg
    def pack(self):
        return _STRUCTS[ISP_MST].pack(self.Size, self.Type, self.ReqI, self.Zero, self.Msg)


class IS_MTC(object):
    """Msg To Connection - hosts only - send to a connection or a player

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'UCID', 'PLID', 'Sp2', 'Sp3', 'Msg')
    def __init__(self, ReqI=0, UCID=0, PLID=0, Msg=''):
        """Initialise a new IS_MTC packet.

        Args:
            ReqI : 0
            UCID : connection's unique id (0 = host)
            PLID : player's unique id (if zero, use :attr:`UCID`)
            Msg  : Message (64 characters)

        """
        self.Size = 72
        self.Type = ISP_MTC
        self.ReqI = ReqI
        self.Zero = 0
        self.UCID = UCID
        self.PLID = PLID
        self.Sp2 = 0
        self.Sp3 = 0
        self.Msg = Msg
    def pack(self):
        return _STRUCTS[ISP_MTC].pack(self.Size, self.Type, self.ReqI, self.Zero, self.UCID, self.PLID, self.Sp2, self.Sp3, self.Msg)


class IS_MOD(object):
    """MODe : send to LFS to change screen mode

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'Bits16', 'RR', 'Width', 'Height')
    def __init__(self, ReqI=0, Bits16=0, RR=0, Width=0, Height=0):
        """Initialise a new IS_MOD packet.

        Args:
            ReqI   : 0
            Bits16 : set to choose 16-bit
            RR     : refresh rate - zero for default
            Width  : 0 means go to window
            Height : 0 means go to window

        """
        self.Size = 20
        self.Type = ISP_MOD
        self.ReqI = ReqI
        self.Zero = 0
        self.Bits16 = Bits16
        self.RR = RR
        self.Width = Width
        self.Height = Height
    def pack(self):
        return _STRUCTS[ISP_MOD].pack(self.Size, self.Type, self.ReqI, self.Zero, self.Bits16, self.RR, self.Width, self.Height)


class IS_VTN(object):
    """VoTe Notify

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'UCID', 'Action', 'Spare2', 'Spare3')
    def __init__(self, ReqI=0, UCID=0, Action=0):
        """Initialise a new IS_VTN packet.

        Args:
            ReqI   : 0
            UCID   : connection's unique id
            Action : Vote action from ``VOTE_*``

        """
        self.Size = 8
        self.Type = ISP_VTN
        self.ReqI = ReqI
        self.Zero = 0
        self.UCID = UCID
        self.Action = Action
        self.Spare2 = 0
        self.Spare3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.UCID, self.Action, self.Spare2, self.Spare3 = _STRUCTS[ISP_VTN].unpack(data)
        return self


class IS_RST(object):
    """Race STart

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'RaceLaps', 'QualMins', 'NumP', 'Spare', 'Track', 'Weather', 'Wind', 'Flags', 'NumNodes', 'Finish', 'Split1', 'Split2', 'Split3')
    def __init__(self):
        """Initialise a new IS_RST packet.

        """
        self.Size = 28
        self.Type = ISP_RST
        self.ReqI = 0
        self.Zero = 0
        self.RaceLaps = 0
        self.QualMins = 0
        self.NumP = 0
        self.Spare = 0
        self.Track = ''
        self.Weather = 0
        self.Wind = 0
        self.Flags = 0
        self.NumNodes = 0
        self.Finish = 0
        self.Split1 = 0
        self.Split2 = 0
        self.Split3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.RaceLaps, self.QualMins, self.NumP, self.Spare, self.Track, self.Weather, self.Wind, self.Flags, self.NumNodes, self.Finish, self.Split1, self.Split2, self.Split3 = _STRUCTS[ISP_RST].unpack(data)
        self.Track = _eatNullChars(self.Track)
        return self


class IS_NCN(object):
    """New ConN

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'UCID', 'UName', 'PName', 'Admin', 'Total', 'Flags', 'Sp3')
    def __init__(self):
        """Initialise a new IS_NCN packet.

        """
        self.Size = 56
        self.Type = ISP_NCN
        self.ReqI = 0
        self.UCID = 0
        self.UName = ''
        self.PName = ''
        self.Admin = 0
        self.Total = 0
        self.Flags = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.UName, self.PName, self.Admin, self.Total, self.Flags, self.Sp3 = _STRUCTS[ISP_NCN].unpack(data)
        self.UName = _eatNullChars(self.UName)
        self.PName = _eatNullChars(self.PName)
        return self


class IS_CNL(object):
    """ConN Leave

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'UCID', 'Reason', 'Total', 'Sp2', 'Sp3')
    def __init__(self):
        """Initialise a new IS_CNL packet.

        """
        self.Size = 8
        self.Type = ISP_CNL
        self.ReqI = 0
        self.UCID = 0
        self.Reason = 0
        self.Total = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.Reason, self.Total, self.Sp2, self.Sp3 = _STRUCTS[ISP_CNL].unpack(data)
        return self


class IS_CPR(object):
    """Conn Player Rename

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'UCID', 'PName', 'Plate')
    def __init__(self):
        """Initialise a new IS_CPR packet.

        """
        self.Size = 36
        self.Type = ISP_CPR
        self.ReqI = 0
        self.UCID = 0
        self.PName = ''
        self.Plate = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.PName, self.Plate = _STRUCTS[ISP_CPR].unpack(data)
        self.PName = _eatNullChars(self.PName)
        self.Plate = _eatNullChars(self.Plate) # No trailing zero on Plate.
        return self


class IS_NPL(object):
    """New PLayer joining race (if PLID already exists, then leaving pits)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'UCID', 'PType', 'Flags', 'PName', 'Plate', 'CName', 'SName', 'Tyres', 'H_Mass', 'H_TRes', 'Model', 'Pass', 'Spare', 'SetF', 'NumP', 'Sp2', 'Sp3')
    def __init__(self):
        """Initialise a new IS_NPL packet.
        
        """
        self.Size = 76
        self.Type = ISP_NPL
        self.ReqI = 0
        self.PLID = 0
        self.UCID = 0
        self.PType = 0
        self.Flags = 0
        self.PName = ''
        self.Plate = ''
        self.CName = ''
        self.SName = ''
        self.Tyres = [0, 0, 0, 0]
        self.H_Mass = 0
        self.H_TRes = 0
        self.Model = 0
        self.Pass = 0
        self.Spare = 0
        self.SetF = 0
        self.NumP = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.UCID, self.PType, self.Flags, self.PName, self.Plate, self.CName, self.SName, self.Tyres[0], self.Tyres[1], self.Tyres[2], self.Tyres[3], self.H_Mass, self.H_TRes, self.Model, self.Pass, self.Spare, self.SetF, self.NumP, self.Sp2, self.Sp3 = _STRUCTS[ISP_NPL].unpack(data)
        self.PName = _eatNullChars(self.PName)
        self.Plate = _eatNullChars(self.Plate) # No trailing zero
        self.CName = _eatNullChars(self.CName)
        self.SName = _eatNullChars(self.SName)
        return self


class IS_PLP(object):
    """PLayer Pits (go to settings - stays in player list)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID')
    def __init__(self):
        """Initialise a new IS_PLP packet.

        """
        self.Size = 4
        self.Type = ISP_PLP
        self.ReqI = 0
        self.PLID = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID = _STRUCTS[ISP_PLP].unpack(data)
        return self


class IS_PLL(object):
    """PLayer Leave race (spectate - removed from player list)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID')
    def __init__(self):
        """Initialise a new IS_PLL packet.

        """
        self.Size = 4
        self.Type = ISP_PLL
        self.ReqI = 0
        self.PLID = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID = _STRUCTS[ISP_PLL].unpack(data)
        return self


class IS_LAP(object):
    """LAP time

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'LTime', 'ETime', 'LapsDone', 'Flags', 'Sp0', 'Penalty', 'NumStops', 'Sp3')
    def __init__(self):
        """Initialise a new IS_LAP packet.

        """
        self.Size = 20
        self.Type = ISP_LAP
        self.ReqI = 0
        self.PLID = 0
        self.LTime = 0
        self.ETime = 0
        self.LapsDone = 0
        self.Flags = 0
        self.Sp0 = 0
        self.Penalty = 0
        self.NumStops = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.LTime, self.ETime, self.LapsDone, self.Flags, self.Sp0, self.Penalty, self.NumStops, self.Sp3 = _STRUCTS[ISP_LAP].unpack(data)
        return self


class IS_SPX(object):
    """SPlit X time

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'STime', 'ETime', 'Split', 'Penalty', 'NumStops', 'Sp3')
    def __init__(self):
        """Initialise a new IS_SPX packet.

        """
        self.Size = 16
        self.Type = ISP_SPX
        self.ReqI = 0
        self.PLID = 0
        self.STime = 0
        self.ETime = 0
        self.Split = 0
        self.Penalty = 0
        self.NumStops = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.STime, self.ETime, self.Split, self.Penalty, self.NumStops, self.Sp3 = _STRUCTS[ISP_SPX].unpack(data)
        return self


class IS_PIT(object):
    """PIT stop (stop at pit garage)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'LapsDone', 'Flags', 'Sp0', 'Penalty', 'NumStops', 'Sp3', 'Tyres', 'Work', 'Spare')
    def __init__(self):
        """Initialise a new IS_PIT packet.

        """
        self.Size = 24
        self.Type = ISP_PIT
        self.ReqI = 0
        self.PLID = 0
        self.LapsDone = 0
        self.Flags = 0
        self.Sp0 = 0
        self.Penalty = 0
        self.NumStops = 0
        self.Sp3 = 0
        self.Tyres = [0, 0, 0, 0]
        self.Work = 0
        self.Spare = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.LapsDone, self.Flags, self.Sp0, self.Penalty, self.NumStops, self.Sp3, self.Tyres[0], self.Tyres[1], self.Tyres[2], self.Tyres[3], self.Work, self.Spare = _STRUCTS[ISP_PIT].unpack(data)
        return self


class IS_PSF(object):
    """Pit Stop Finished

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'STime', 'Spare')
    def __init__(self):
        """Initialise a new IS_PSF packet.

        """
        self.Size = 12
        self.Type = ISP_PSF
        self.ReqI = 0
        self.PLID = 0
        self.STime = 0
        self.Spare = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.STime, self.Spare = _STRUCTS[ISP_PSF].unpack(data)
        return self


class IS_PLA(object):
    """Pit LAne

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'Fact', 'Sp1', 'Sp2', 'Sp3')
    def __init__(self):
        """Initialise a new IS_PLA packet.

        """
        self.Size = 8
        self.Type = ISP_PLA
        self.ReqI = 0
        self.PLID = 0
        self.Fact = 0
        self.Sp1 = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.Fact, self.Sp1, self.Sp2, self.Sp3 = _STRUCTS[ISP_PLA].unpack(data)
        return self


class IS_CCH(object):
    """Camera CHange

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'Camera', 'Sp1', 'Sp2', 'Sp3')
    def __init__(self):
        """Initialise a new IS_CCH packet.

        """
        self.Size = 8
        self.Type = ISP_CCH
        self.ReqI = 0
        self.PLID = 0
        self.Camera = 0
        self.Sp1 = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.Camera, self.Sp1, self.Sp2, self.Sp3 = _STRUCTS[ISP_CCH].unpack(data)
        return self


class IS_PEN(object):
    """PENalty (given or cleared)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'OldPen', 'NewPen', 'Reason', 'Sp3')
    def __init__(self):
        """Initialise a new IS_PEN packet.

        """
        self.Size = 8
        self.Type = ISP_PEN
        self.ReqI = 0
        self.PLID = 0
        self.OldPen = 0
        self.NewPen = 0
        self.Reason = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.OldPen, self.NewPen, self.Reason, self.Sp3 = _STRUCTS[ISP_PEN].unpack(data)
        return self


class IS_TOC(object):
    """Take Over Car

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'OldUCID', 'NewUCID', 'Sp2', 'Sp3')
    def __init__(self, ReqI=0, PLID=0, OldUCID=0, NewUCID=0, Sp2=0, Sp3=0):
        """Initialise a new IS_TOC packet.

        """
        self.Size = 8
        self.Type = ISP_TOC
        self.ReqI = 0
        self.PLID = 0
        self.OldUCID = 0
        self.NewUCID = 0
        self.Sp2 = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.OldUCID, self.NewUCID, self.Sp2, self.Sp3 = _STRUCTS[ISP_TOC].unpack(data)
        return self


class IS_FLG(object):
    """FLaG (yellow or blue flag changed)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'OffOn', 'Flag', 'CarBehind', 'Sp3')
    def __init__(self):
        """Initialise a new IS_FLG packet.

        """
        self.Size = 8
        self.Type = ISP_FLG
        self.ReqI = 0
        self.PLID = 0
        self.OffOn = 0
        self.Flag = 0
        self.CarBehind = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.OffOn, self.Flag, self.CarBehind, self.Sp3 = _STRUCTS[ISP_FLG].unpack(data)
        return self


class IS_PFL(object):
    """Player FLags (help flags changed)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'Flags', 'Spare')
    def __init__(self, ReqI=0, PLID=0, Flags=0, Spare=0):
        """Initialise a new IS_PFL packet.

        """
        self.Size = 8
        self.Type = ISP_PFL
        self.ReqI = 0
        self.PLID = 0
        self.Flags = 0
        self.Spare = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.Flags, self.Spare = _STRUCTS[ISP_PFL].unpack(data)
        return self


class IS_FIN(object):
    """FINished race notification (not a final result - use :class:`IS_RES`)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'TTime', 'BTime', 'SpA', 'NumStops', 'Confirm', 'SpB', 'LapsDone', 'Flags')
    def __init__(self):
        """Initialise a new IS_FIN packet.

        """
        self.Size = 20
        self.Type = ISP_FIN
        self.ReqI = 0
        self.PLID = 0
        self.TTime = 0
        self.BTime = 0
        self.SpA = 0
        self.NumStops = 0
        self.Confirm = 0
        self.SpB = 0
        self.LapsDone = 0
        self.Flags = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.TTime, self.BTime, self.SpA, self.NumStops, self.Confirm, self.SpB, self.LapsDone, self.Flags = _STRUCTS[ISP_FIN].unpack(data)
        return self


class IS_RES(object):
    """RESult (qualify or confirmed finish)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID', 'UName', 'PName', 'Plate', 'CName', 'TTime', 'BTime', 'SpA', 'NumStops', 'Confirm', 'SpB', 'LapsDone', 'Flags', 'ResultNum', 'NumRes', 'PSeconds')
    def __init__(self):
        """Initialise a new IS_RES packet.

        """
        self.Size = 84
        self.Type = ISP_RES
        self.ReqI = 0
        self.PLID = 0
        self.UName = ''
        self.PName = ''
        self.Plate = ''
        self.CName = ''
        self.TTime = 0
        self.BTime = 0
        self.SpA = 0
        self.NumStops = 0
        self.Confirm = 0
        self.SpB = 0
        self.LapsDone = 0
        self.Flags = 0
        self.ResultNum = 0
        self.NumRes = 0
        self.PSeconds = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID, self.UName, self.PName, self.Plate, self.CName, self.TTime, self.BTime, self.SpA, self.NumStops, self.Confirm, self.SpB, self.LapsDone, self.Flags, self.ResultNum, self.NumRes, self.PSeconds = _STRUCTS[ISP_RES].unpack(data)
        self.UName = _eatNullChars(self.UName)
        self.PName = _eatNullChars(self.PName)
        self.Plate = _eatNullChars(self.Plate) # No trailing zero
        self.CName = _eatNullChars(self.CName)
        return self


class IS_REO(object):
    """REOrder (when race restarts after qualifying)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'NumP', 'PLID')
    def __init__(self, ReqI=0, NumP=0, PLID=[]):
        """Initialise a new IS_reo packet.

        Args:
            ReqI : 0 unless this is a reply to an ``TINY_REO`` request
            NumP : number of players in race
            PLID : all PLIDs in new order

        """
        self.Size = 36
        self.Type = ISP_REO
        self.ReqI = ReqI
        self.NumP = NumP
        self.PLID = PLID
    def pack(self):
        return _STRUCTS[ISP_REO].pack(self.Size, self.Type, self.ReqI, self.NumP, self.PLID)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.NumP = _STRUCTS[ISP_REO].unpack(data[:4])
        data = data[4:]
        append = self.PLID.append
        for i in xrange(self.NumP):
            append(data[i])
        return self


class IS_NLP(object):
    """Node and Lap Packet - variable size

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'NumP', 'NodeLaps')
    def __init__(self):
        """Initialise a new IS_NLP packet.

        """
        self.Size = 4
        self.Type = ISP_NLP
        self.ReqI = 0
        self.NumP = 0
        self.NodeLaps = []
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.NumP = _STRUCTS[ISP_NLP].unpack(data[:4])
        data = data[4:]
        append = self.NodeLaps.append
        nodelap = NodeLap
        for i in xrange(0, self.NumP * 6, 6):
            append(nodelap(data, i))
        return self


class NodeLap(object):
    """Car info in 6 bytes - there is an array of these in the :class:`IS_NLP`

    """
    #__slots__ = ('Node', 'Lap', 'PLID', 'Position')
    def __init__(self, data, index):
        """Initialise a new NodeLap sub-packet.

        """
        self.Node, self.Lap, self.PLID, self.Position = _STRUCTS[_ISP_NODELAP].unpack(data[index:index+6])


class IS_MCI(object):
    """Multi Car Info - if more than 8 in race then more than one of these is sent

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'NumC', 'CompCars')
    def __init__(self):
        """Initialise a new IS_MCI packet.

        """
        self.Size = 4
        self.Type = ISP_MCI
        self.ReqI = 0
        self.NumC = 0
        self.CompCars = []
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.NumC = _STRUCTS[ISP_MCI].unpack(data[:4])
        data = data[4:]
        append = self.CompCars.append
        compcar = CompCar
        for i in xrange(0, self.NumC * 28, 28):
            append(compcar(data, i))
        return self


class CompCar(object):
    """Car info in 28 bytes - there is an array of these in the :class:`IS_MCI`

    """
    #__slots__ = ('Node', 'Lap', 'PLID', 'Position', 'Info', 'Sp3', 'X', 'Y', 'Z', 'Speed', 'Direction', 'Heading', 'Angel')
    def __init__(self, data, index):
        """Initialise a new CompCar sub-packet.

        """
        self.Node, self.Lap, self.PLID, self.Position, self.Info, self.Sp3, self.X, self.Y, self.Z, self.Speed, self.Direction, self.Heading, self.Angel = _STRUCTS[_ISP_COMPCAR].unpack(data[index:index+28])


class IS_MSX(object):
    """MSg eXtended - like ``IS_MST`` but longer (not for commands)

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'Msg')
    def __init__(self, ReqI=0, Msg=''):
        """Initialise a new IS_MSX packet.

        Args:
            ReqI : 0
            Msg  : last byte must be zero
        
        """
        self.Size = 100
        self.Type = ISP_MSX
        self.ReqI = ReqI
        self.Zero = 0
        self.Msg = Msg
    def pack(self):
        return _STRUCTS[ISP_MSX].pack(self.Size, self.Type, self.ReqI, self.Zero, self.Msg)


class IS_MSL(object):
    """MSg Local - message to appear on local computer only

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Sound', 'Msg')
    def __init__(self, ReqI=0, Sound=0, Msg=''):
        """Initialise a new IS_MSL packet.

        Args:
            ReqI  : 0
            Sound : Sound from ``SND_*`` enumeration.
            Msg   : Message

        """
        self.Size = 132
        self.Type = ISP_MSL
        self.ReqI = ReqI
        self.Sound = Sound
        self.Msg = Msg
    def pack(self):
        return _STRUCTS[ISP_MSL].pack(self.Size, self.Type, self.ReqI, self.Sound, self.Msg)


class IS_CRS(object):
    """Car ReSet

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID')
    def __init__(self):
        """Initialise a new IS_CRS packet.

        """
        self.Size = 4
        self.Type = ISP_CRS
        self.ReqI = 0
        self.PLID = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID = _STRUCTS[ISP_CRS].unpack(data)
        return self


class IS_BFN(object):
    """Button FunctioN - delete buttons / receive button requests

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'SubT', 'UCID', 'ClickID', 'Inst', 'Sp3')
    def __init__(self, ReqI=0, SubT=0, UCID=0, ClickID=0, Inst=0):
        """Initialise a new IS_BFN packet.

        Args:
            ReqI    : 0
            SubT    : subtype, from ``BFN_*`` enumeration
            UCID    : connection to send to or from (0 = local / 255 = all)
            ClickID : ID of button to delete (if ``SubT`` is ``BFN_DEL_BTN``)
            Inst    : used internally by InSim

        """
        self.Size = 8
        self.Type = ISP_BFN
        self.ReqI = ReqI
        self.SubT = SubT
        self.UCID = UCID
        self.ClickID = ClickID
        self.Inst = Inst
        self.Sp3 = 0
    def pack(self):
        return _STRUCTS[ISP_BFN].pack(self.Size, self.Type, self.ReqI, self.SubT, self.UCID, self.ClickID, self.Inst, self.Sp3)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.SubT, self.UCID, self.ClickID, self.Inst, self.Sp3 = _STRUCTS[ISP_BFN].unpack(data)
        return self


class IS_AXI(object):
    """AutoX Info

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Zero', 'AXStart', 'NumCP', 'NumO', 'LName')
    def __init__(self):
        """Initialise a new IS_AXI packet.

        Args:
            ReqI    : 0 unless this is a reply to an ``TINY_AXI`` request
            AXStart : autocross start position
            NumCP   : number of checkpoints
            NumO    : number of objects
            LName   : the name of the layout last loaded (if loaded locally)

        """
        self.Size = 40
        self.Type = ISP_AXI
        self.ReqI = 0
        self.Zero = 0
        self.AXStart = 0
        self.NumCP = 0
        self.NumO = 0
        self.LName = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Zero, self.AXStart, self.NumCP, self.NumO, self.LName = _STRUCTS[ISP_AXI].unpack(data)
        self.LName = _eatNullChars(self.LName)
        return self


class IS_AXO(object):
    """AutoX Object

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'PLID')
    def __init__(self):
        """Initialise a new IS_AXO packet.

        Args:
            ReqI : 0
            PLID : player's unique id

        """
        self.Size = 4
        self.Type = ISP_AXO
        self.ReqI = 0
        self.PLID = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.PLID = _STRUCTS[ISP_AXO].unpack(data)
        return self


class IS_BTN(object):
    """BuTtoN - button header - followed by 0 to 240 characters

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'UCID', 'ClickID', 'Inst', 'BStyle', 'TypeIn', 'L', 'T', 'W', 'H', 'Text')
    def __init__(self, ReqI=0, UCID=0, ClickID=0, Inst=0, BStyle=0, TypeIn=0, L=0, T=0, W=0, H=0, Text=''):
        """Initialise a new IS_BTN packet.

        Args:
            ReqI    : non-zero (returned in ``IS_BTC`` and ``IS_BTT`` packets)
            UCID    : connection to display the button (0 = local / 255 = all)
            ClickID : button ID (0 to 239)
            Inst    : some extra flags from ``INST_*``
            BStyl   : button style flags from ``ISB_*``
            TypeIn  : max chars to type in
            L       : left: 0 - 200
            T       : top: 0 - 200
            W       : width: 0 - 200
            H       : height: 0 - 200
            Text    : 0 to 240 characters of text

        """
        self.Size = 12
        self.Type = ISP_BTN
        self.ReqI = ReqI
        self.UCID = UCID
        self.ClickID = ClickID
        self.Inst = Inst
        self.BStyle = BStyle
        self.TypeIn = TypeIn
        self.L = L
        self.T = T
        self.W = W
        self.H = H
        self.Text = Text
    def pack(self):
        TEXT_SIZE = int(math.ceil(len(self.Text) / 4.0)) * 4
        return _STRUCTS[ISP_BTN].pack(self.Size + TEXT_SIZE, self.Type, self.ReqI, self.UCID, self.ClickID, self.Inst, self.BStyle, self.TypeIn, self.L, self.T, self.W, self.H) + struct.pack('%ds' % TEXT_SIZE, self.Text)


class IS_BTC(object):
    """BuTton Click - sent back when user clicks a button

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'UCID', 'ClickID', 'Inst', 'CFlags', 'Sp3')
    def __init__(self):
        """Initialise a new IS_BTC packet.

        """
        self.Size = 8
        self.Type = ISP_BTC
        self.ReqI = 0
        self.UCID = 0
        self.ClickID = 0
        self.Inst = 0
        self.CFlags = 0
        self.Sp3 = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.ClickID, self.Inst, self.CFlags, self.Sp3 = _STRUCTS[ISP_BTC].unpack(data)
        return self


class IS_BTT(object):
    """BuTton Type - sent back when user types into a text entry button

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'UCID', 'ClickID', 'Inst', 'TypeIn', 'Sp3', 'Text')
    def __init__(self):
        """Initialise a new IS_BTT packet.

        """
        self.Size = 104
        self.Type = ISP_BTT
        self.ReqI = 0
        self.UCID = 0
        self.ClickID = 0
        self.Inst = 0
        self.TypeIn = 0
        self.Sp3 = 0
        self.Text = ''
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.UCID, self.ClickID, self.Inst, self.TypeIn, self.Sp3, self.Text = _STRUCTS[ISP_BTT].unpack(data)
        self.Text = _eatNullChars(self.Text)
        return self


class IS_RIP(object):
    """Replay Information Packet

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Error', 'MPR', 'Paused', 'Options', 'Sp3', 'CTime', 'TTime', 'RName')
    def __init__(self, ReqI=0, Error=0, MPR=0, Paused=0, Options=0, CTime=0, TTime=0, RName=''):
        """Initialise a new IS_RIP packet.

        Args:
            ReqI    : request : non-zero / reply : same value returned
            Error   : 0 or 1 = OK / options from ``RIP_*``
            MPR     : 0 = SPR / 1 = MPR
            Paused  : request : pause on arrival / reply : paused state
            Options : various options from ``RIPOPT_*``
            CTime   : (hundredths) request : destination / reply : position
            TTime   : (hundredths) request : zero / reply : replay length
            RName   : zero or replay name: last byte must be zero

        """
        self.Size = 76
        self.Type = ISP_RIP
        self.ReqI = ReqI
        self.Error = Error
        self.MPR = MPR
        self.Paused = Paused
        self.Options = Options
        self.Sp3 = 0
        self.CTime = CTime
        self.TTime = TTime
        self.RName = RName
    def pack(self):
        return _STRUCTS[ISP_RIP].pack(self.Size, self.Type, self.ReqI, self.Error, self.MPR, self.Paused, self.Options, self.Sp3, self.CTime, self.TTime, self.RName)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Error, self.MPR, self.Paused, self.Options, self.Sp3, self.CTime, self.TTime, self.RName = _STRUCTS[ISP_RIP].unpack(data)
        self.RName = _eatNullChars(self.RName)
        return self


class IS_SSH(object):
    """ScreenSHot

    """
    #__slots__ = ('Size', 'Type', 'ReqI', 'Error', 'Sp0', 'Sp1', 'Sp2', 'Sp3', 'BMP')
    def __init__(self, ReqI=0, Error=0, BMP=''):
        """Initialise a new IS_SSH packet.

        Args:
            ReqI  : request : non-zero / reply : same value returned
            Error : 0 = OK / other values from ``SSH_*``
            BMP   : name of screenshot file: last byte must be zero

        """
        self.Size = 40
        self.Type = ISP_SSH
        self.ReqI = ReqI
        self.Error = Error
        self.Sp0 = 0
        self.Sp1 = 0
        self.Sp2 = 0
        self.Sp3 = 0
        self.BMP = BMP
    def pack(self):
        return _STRUCTS[ISP_SSH].pack(self.Size, self.Type, self.ReqI, self.Error, self.Sp0, self.Sp1, self.Sp2, self.Sp3, self.BMP)
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.Error, self.Sp0, self.Sp1, self.Sp2, self.Sp3, self.BMP = _STRUCTS[ISP_SSH].unpack(data)
        self.BMP = _eatNullChars(self.BMP)
        return self


# OutGauge and OutSim packets.
class OutGaugePack(object):
    """External dashboard support.

    """
    #__slots__ = ('Time', 'Car', 'Flags', 'Gear', 'SpareB', 'Speed', 'RPM', 'Turbo', 'EngTemp', 'Fuel', 'OilPress', 'OilTemp', 'DashLights', 'ShowLights', 'Throttle', 'Brake', 'Clutch', 'Display1', 'Display2', 'ID')
    def __init__(self):
        """Initialise a new OutGaugePack packet.

        """
        self.Time = 0
        self.Car = ''
        self.Flags = 0
        self.Gear = 0
        self.SpareB = 0
        self.Speed = 0.0
        self.RPM = 0.0
        self.Turbo = 0.0
        self.EngTemp = 0.0
        self.Fuel = 0.0
        self.OilPress = 0.0
        self.OilTemp = 0.0
        self.DashLights = 0
        self.ShowLights = 0
        self.Throttle = 0.0
        self.Brake = 0.0
        self.Clutch = 0.0
        self.Display1 = ''
        self.Display2 = ''
        self.ID = 0
    def unpack(self, pdata):
        self.Time, self.Car, self.Flags, self.Gear, self.SpareB, self.Speed, self.RPM, self.Turbo, self.EngTemp,self.Fuel, self.OilPress, self.OilTemp, self.DashLights, self.ShowLights, self.Throttle, self.Brake, self.Clutch, self.Display1,self.Display2 = _STRUCTS[OUT_OUTGAUGE].unpack(pdata[:92])
        return self


class OutSimPack(object):
    """External Motion simulator support

    """
    #__slots__ = ('Time', 'AngVel', 'Heading', 'Pitch', 'Roll', 'Accel', 'Vel', 'Pos', 'ID')
    def __init__(self):
        """Initialise a new OutSimPack packet.

        """
        self.Time = 0
        self.AngVel = [0.0, 0.0, 0.0]
        self.Heading = 0.0
        self.Pitch = 0.0
        self.Roll = 0.0
        self.Accel = [0.0, 0.0, 0.0]
        self.Vel = [0.0, 0.0, 0.0]
        self.Pos = [0, 0, 0]
        self.ID = 0
    def unpack(self, pdata):
        self.Time, self.AngVel[0], self.AngVel[1], self.AngVel[2], self.Heading, self.Pitch, self.Roll, self.Accel[0], self.Accel[1], self.Accel[2], self.Vel[0], self.Vel[1], self.Vel[2], self.Pos[0], self.Pos[1], self.Pos[2] = _STRUCTS[OUT_OUTSIM].unpack(pdata[:64])
        return self

        
# InSim Relay Packets
class IR_HLR(object):
    """InSimRelay Host List Request packet

    """
    #__slots__ = ['Size', 'Type', 'ReqI', 'Sp0']
    def __init__(self, ReqI=0):
        """Initailise a new IR_HLR packet.
        
        """        
        self.Size = 4
        self.Type = IRP_HLR
        self.ReqI = ReqI
        self.Sp0 = 0
    def pack(self):
        return _STRUCTS[IRP_HLR].pack(self.Size, self.Type, self.ReqI, self.Sp0)     


class HInfo(object):
    """A sub-packet for IR_HOS containing host information
    
    """
    #__slots__ = ['HName', 'Track', 'Flags', 'NumConns']
    def __init__(self):
        """Initailise a new HInfo sub-packet.
        
        """        
        self.HName = ''
        self.Track = ''
        self.Flags = 0
        self.NumConns = 0
    def unpack(self, data, index):
        self.HName, self.Track, self.Flags, self.NumConns = _STRUCTS[_IRP_HINFO].unpack(data[index:index+40])
        self.HName = _eatNullChars(self.HName)
        self.Track = _eatNullChars(self.Track)
        return self        


class IR_HOS(object):
    """InSimRelay HOSt list packet

    """
    #__slots__ = ['Size', 'Type', 'ReqI', 'NumHosts', 'Hosts']
    def __init__(self):
        """Initailise a new IR_HOS packet.
        
        """    
        self.Size = 4
        self.Type = IRP_HOS
        self.ReqI = 0
        self.NumHosts = 0
        self.Hosts = []
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.NumHosts = _STRUCTS[IRP_HLR].unpack(data[:4])
        data = data[4:]
        for i in range(0, self.NumHosts, 40):
            self.Hosts.append(HInfo().unpack(data, i))
        return self


class IR_SEL(object):
    """InSimRelay host SELect packet, so relay starts sending data

    """
    #__slots__ = ['Size', 'Type', 'ReqI', 'Zero', 'HName', 'Admin', 'Spec']
    def __init__(self, ReqI=0, HName='', Admin='', Spec=''):
        """Initailise a new IR_SEL packet.
        
        Args:
            ReqI  : If non-zero Relay will reply with an ``IS_VER`` packet
            HName : Hostname to receive data from - may be colourcode stripped
            Admin : Admin password (to gain admin access to host)
            Spec  : Spectator password (if host requires it)        
        
        """        
        self.Size = 68
        self.Type = IRP_SEL
        self.ReqI = ReqI
        self.Zero = 0
        self.HName = HName
        self.Admin = Admin
        self.Spec = Spec
    def pack(self):
        return _STRUCTS[IRP_SEL].pack(self.Size, self.Type, self.ReqI, self.Zero, self.HName, self.Admin, self.Spec)
     

class IR_ERR(object):
    """InSimRelay ERRor packet

    """
    #__slots__ = ['Size', 'Type', 'ReqI', 'ErrNo']
    def __init__(self):
        """Initailise a new IR_ERR packet.
        
        """        
        self.Size = 4
        self.Type = IRP_ERR
        self.ReqI = 0
        self.ErrNo = 0
    def unpack(self, data):
        self.Size, self.Type, self.ReqI, self.ErrNo = _STRUCTS[IRP_ERR].unpack(data)
        return self        
        

class IR_ARQ(object):
    """InSimRelay Admin ReQuest packet.
    
    """
    #__slots__ = ['Size', 'Type', 'ReqI', 'Sp0']
    def __init__(self, ReqI=0):
        """Initailise a new IR_ARQ packet.
        
        """    
        self.Size = 4
        self.Type = IRP_ARQ
        self.ReqI = ReqI
        self.Sp0 = 0
    def pack(self):
        return _STRUCTS[IRP_ARQ].pack(self.Size, self.Type, self.ReqI, self.Sp0)      


class IR_ARP(object):
    """InSimRelay Admin ResPonse packet
    
    """
    #__slots__ = ['Size', 'Type', 'ReqI', 'Admin']
    def __init__(self):
        """Initailise a new IR_ARP packet.
        
        """
        self.Size = 4
        self.Type = IRP_ARP
        self.ReqI = 0
        self.Admin = 0
    def unpack(self, data):
        self.Size, self.Type, self.Reqi, self.Admin = _STRUCTS[IRP_ARP].unpack(data)
        return self        


# Packet look-up.
_PACKET_DEFS = {ISP_ISI: IS_ISI,
                ISP_VER: IS_VER,
                ISP_TINY: IS_TINY,
                ISP_SMALL: IS_SMALL,
                ISP_STA: IS_STA,
                ISP_SCH: IS_SCH,
                ISP_SFP: IS_SFP,
                ISP_SCC: IS_SCC,
                ISP_CPP: IS_CPP,
                ISP_ISM: IS_ISM,
                ISP_MSO: IS_MSO,
                ISP_III: IS_III,
                ISP_MST: IS_MST,
                ISP_MTC: IS_MTC,
                ISP_MOD: IS_MOD,
                ISP_VTN: IS_VTN,
                ISP_RST: IS_RST,
                ISP_NCN: IS_NCN,
                ISP_CNL: IS_CNL,
                ISP_CPR: IS_CPR,
                ISP_NPL: IS_NPL,
                ISP_PLP: IS_PLP,
                ISP_PLL: IS_PLL,
                ISP_LAP: IS_LAP,
                ISP_SPX: IS_SPX,
                ISP_PIT: IS_PIT,
                ISP_PSF: IS_PSF,
                ISP_PLA: IS_PLA,
                ISP_CCH: IS_CCH,
                ISP_PEN: IS_PEN,
                ISP_TOC: IS_TOC,
                ISP_FLG: IS_FLG,
                ISP_PFL: IS_PFL,
                ISP_FIN: IS_FIN,
                ISP_RES: IS_RES,
                ISP_REO: IS_REO,
                ISP_NLP: IS_NLP,
                ISP_MCI: IS_MCI,
                ISP_MSX: IS_MSX,
                ISP_MSL: IS_MSL,
                ISP_CRS: IS_CRS,
                ISP_BFN: IS_BFN,
                ISP_AXI: IS_AXI,
                ISP_AXO: IS_AXO,
                ISP_BTN: IS_BTN,
                ISP_BTC: IS_BTC,
                ISP_BTT: IS_BTT,
                ISP_RIP: IS_RIP,
                ISP_SSH: IS_SSH,
                OUT_OUTGAUGE: OutGaugePack,
                OUT_OUTSIM: OutSimPack,
                IRP_HLR: IR_HLR,
                IRP_HOS: IR_HOS,
                IRP_SEL: IR_SEL,
                IRP_ERR: IR_ERR,
                IRP_ARP: IR_ARP,
                IRP_ARQ: IR_ARQ,}


def lookup(ptype):
    """Determine the packet name from the packet-type.

    """
    if ptype == OUT_OUTGAUGE:
        return 'OUT_OUTGAUGE'
    elif ptype == OUT_OUTSIM:
        return 'OUT_OUTSIM'
    try:
        name = _PACKET_DEFS[ptype].__name__
        if name.startswith('IR'):
            return 'IRP_%s' % name[3:]
        return 'ISP_%s' % name[3:]
    except KeyError:
        return 'ISP_NONE'


# Exceptions
class Error(Exception):
    """Exception raised for InSim related errors.

    """
    pass


class VersionError(Error):
    """Exception raised if an incorrect version is received.

    """
    pass


class _SocketLost(Error):
    """Exception raised if a socket connection is lost.

    """
    pass


class _SocketTimeout(Error):
    """Exception raised if a socket connection is lost.

    """
    pass


# Clients
class _Buffer(object):
    """Class to manage TCP recv buffer.

    """
    def __init__(self):
        self.__buf = array.array('B')

    def __len__(self):
        return  self.__buf.buffer_info()[1]

    def __iter__(self):
        return self

    def append(self, data, len):
        self.__buf.extend(data[:len])

    def next(self):
        if len(self) > 0:
            psize = self.__buf[0]
            if len(self) >= psize:
                pdata = self.__buf[:psize]
                self.__buf = self.__buf[psize:]
                return pdata
        raise StopIteration
        

class InSim(object):
    """Class to manage InSim connection with LFS.

    """
    def __init__(self, name=None, traceback=True):
        """Initialise a new InSim object.

        """
        self.__name = name
        self.__timeout = None        
        self.__traceback = traceback
        self.__connected = False
        self.__tcp = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.__udp = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.__thread = threading.Thread(target=self.__recvThread)
        self.__hostaddr = ()
        self.__callbacks = {}
        self.__buffer = _Buffer()
        self.__data = array.array('B', [0]*_BUFFER_SIZE)

    def __str__(self):
        """Return a string representation of the object.

        """
        return self.__hostaddr

    def bind(self, evt, callback):
        """Bind a function callback that will be called when the specified 
        event occurs.

        """
        try:
            self.__callbacks[evt].append(callback)
        except KeyError:
            self.__callbacks[evt] = [callback]

    def unbind(self, evt, callback):
        """Unbind an function callback from an event.

        """
        try:
            self.__callbacks[evt].remove(callback)
            if not self.__callbacks[evt]:
                del self.__callbacks[evt]			
        except KeyError, AttributeError:
            pass

    def isbound(self, evt, callback=None):
        """Determine if an event-handler has been bound. 

        """
        if callback:
            return evt in self.__callbacks and callback in self.__callbacks[evt]
        return evt in self.__callbacks

    def event(self, evt, *args):
        """Raise an event.

        """
        if evt in self.__callbacks:
            [callback(self, *args) for callback in self.__callbacks[evt]]

    def init(self, host='localhost', port=29999, UDPPort=0, Flags=0, Prefix=' ', Interval=0, Admin='', IName='^3pyinsim'):
        """Connect to LFS and initialise the InSim system.

        """
        try:
            self.__hostaddr = (host, port)
            self.__tcp.connect(self.__hostaddr)
            self.send(ISP_ISI, ReqI=1, UDPPort=UDPPort, Flags=Flags, Prefix=Prefix, Interval=Interval, Admin=Admin, IName=IName)
            recv = self.__tcp.recv_into(self.__data, _BUFFER_SIZE)
            if recv > 0:
                self.__buffer.append(self.__data, recv)
                ver = IS_VER().unpack(self.__buffer.next())
                if ver.InSimVer == INSIM_VERSION:
                    if UDPPort > 0:
                        self.__udp.bind((host, UDPPort))
                    self.__connected = True
                    self.event(EVT_CONNECTED, self.__hostaddr)
                    self.__thread.start()
                    return ver
                else:
                    raise VersionError('Invalid InSim version')
        except socket.error as err:
            raise Error(err[1])
        raise Error('Could not init InSim')

    def close(self):
        """Close the connection with LFS.

        """
        try:
            [sock.close() for sock in self.__socks]
            self.__connected = False
            self.event(EVT_CLOSED, CLOSE_REQUEST)
        except socket.error as err:
            raise Error(err[1])

    def send(self, ptype, **args):
        """Send a packet to LFS.

        """
        self.sendb(_PACKET_DEFS[ptype](**args).pack())

    def sendp(self, *packets):
        """Send a list of packets to LFS.

        """
        [self.sendb(packet.pack()) for packet in packets]

    def sendb(self, data):
        """Send binary data to LFS.

        """
        try:
            sent = 0
            while sent < len(data):
                sent = self.__tcp.send(data[sent:])
        except socket.error as err:
            raise Error(err[1])

    def sendm(self, msg, ucid=0, plid=0):
        """Send a message to LFS.

        """
        if ucid or plid:
            self.send(ISP_MTC, Msg=msg, UCID=ucid, PLID=plid)            
        else:
            if len(msg) < 64 or msg.startswith('/'):
                self.send(ISP_MST, Msg=msg)
            else:
                self.send(ISP_MSX, Msg=msg)

    def timer(self, callback, timeout, data=None):
        """Start a timer which will execute a callback after a period has elapsed.

        """
        t = None
        if data:
            t = threading.Timer(timeout, callback, [self, data])
        else:
            t = threading.Timer(timeout, callback, [self])
        if self.__connected:
            t.start()
        return t

    def __recvThread(self):
        """Socket receive thread.

        """
        try:
            rlist = [self.__tcp, self.__udp]
            while True:
                socks = select.select(rlist, [], [], self.__timeout)[0]
                if socks:
                    [self.__recv(sock) for sock in socks]
                else: 
                    raise _SocketTimeout
        except _SocketTimeout:
            self.event(EVT_TIMEOUT, self.__timeout)
        except _SocketLost:
            self.event(EVT_CLOSED, CLOSE_LOST)
        except Exception as err:
            if self.__traceback: traceback.print_exc()
            self.event(EVT_ERROR, err)
        finally:
            self.__connected = False
            
    def __recv(self, sock):
        """Receive data from socket.
        
        """
        recv = sock.recv_into(self.__data, _BUFFER_SIZE)        
        if recv > 0:
            if sock is self.__tcp:
                self.__buffer.append(self.__data, recv)
                [self.__onPacketData(pdata) for pdata in self.__buffer]
            else: 
                self.__onPacketData(self.__data[:recv], True)
        else: 
            raise _SocketLost

    def __onPacketData(self, pdata, udp=False):
        """Function called when packet data is received.

        """
        ptype = _packetType(pdata, udp)
        if ptype == ISP_TINY and _subType(pdata) == TINY_NONE:
            self.sendb(pdata) # Keep alive
        isp = ptype in self.__callbacks
        isa = EVT_ALL in self.__callbacks
        if isp or isa:
            packet = _PACKET_DEFS[ptype]().unpack(pdata)
            if isp: self.event(ptype, packet)
            if isa: self.event(EVT_ALL, packet) 
            
    def onPacketData(self, pdata, udp):
        """Virtual function called when packet data is received.

        """        
        raise NotImplementedError                    

    name = property(lambda self: self.__name, doc='Get the name of the socket')
    connected = property(lambda self: self.__connected, doc='Get if LFS is connected')
    hostaddr = property(lambda self: self.__hostaddr, doc='Get the host address pair')       
        

class OutGauge(InSim):
    """Class to manage OutGauge UDP connection with LFS.

    """        
    def __init__(self, name=None, traceback=True):
        """Initialise a new OutSim object.

        """               
        InSim.__init__(self, name, traceback)
        
    def init(self, host='localhost', port=30000, timeout=None):
        """Initialise a connection to listen for UDP packets from the speicified host and port.      

        """             
        try:
            self._InSim__timeout = timeout
            self._InSim__hostaddr = (host, port)
            self._InSim__udp.bind(self._InSim__hostaddr)
            self._InSim__connected = True
            self.event(EVT_CONNECTED, self._InSim__hostaddr)
            self._InSim__thread.start()
        except socket.error as err:
            raise Error(err[1])
        
    def close(self):
        """Close the OutGauge connection.
        
        """        
        self._InSim__udp.close() 
        self.__connected = False
        self.event(EVT_CLOSED, CLOSE_REQUEST)   
        
        
class OutSim(InSim):
    """Class to manage OutSim UDP connection with LFS.

    """    
    def __init__(self, name=None, traceback=True):
        """Initialise a new OutSim object.

        """        
        InSim.__init__(self, name, traceback)
        
    def init(self, host='localhost', port=30000, timeout=None):
        """Initialise to listen for UDP packets from the speicified host and port.

        """        
        try:
            self._InSim__timeout = timeout            
            self._InSim__hostaddr = (host, port)
            self._InSim__udp.bind(self._InSim__hostaddr)
            self._InSim__connected = True
            self.event(EVT_CONNECTED, self._InSim__hostaddr)
            self._InSim__thread.start()
        except socket.error as err:
            raise Error(err[1])   
        
    def close(self):
        """Close the OutSim connection.
        
        """        
        self._InSim__udp.close()  
        self.__connected = False
        self.event(EVT_CLOSED, CLOSE_REQUEST)        
        

class InSimRelay(InSim):
    """Class to manage InSimRelay connection with LFSWorld.
    
    """
    def __init__(self, name=None, traceback=True):
        """Initialise a new InSimRelay object.

        """
        InSim.__init__(self, name, traceback)

    def init(self, host='isrelay.lfs.net', port=47474):
        """Initialise the InSim Relay system.

        """
        try:
            self._InSim__hostaddr = (host, port)
            self._InSim__tcp.connect((host, port))
            self._InSim__connected = True
            self.event(EVT_CONNECTED, self._InSim__hostaddr)
            self._InSim__thread.start()
        except socket.error as err:
            raise Error(err[1])     


if __name__ == '__main__':
    pass

## Uncomment to print the __all__ statement.
#    exceptions = ('all', 'exceptions', 'socket', 'struct', 'threading', 'traceback', 're', 'select', 'math', 'array', 'time')
#    all = []    
#    for d in dir():
#        if d[:1] != '_' and d not in exceptions:
#            all.append(d)
#    print '__all__ =', all  
