"""Example 7: Rolling Starts

This is a simple example program which demonstrates how to enforce a rolling 
start (or pace lap) on a host. If a player exceeds the speed limit they are
spectated. If a player overtakes another car they are given a 30 second time
penalty.

"""
import pyinsim

# Constants
FORMATION_LAPS = 1
SPEED_LIMIT = 80 # Kph

# Global variables
players = {}
formation = False

# Send a RCM (Race Control Message) to all players, then start a timer to 
# clear the message after 10 seconds
def sendRCM(insim, msg):
    insim.sendm('/rcm %s' % msg)
    insim.sendm('/rcm_all')
    insim.timer(clearRCM, 10.0)
        
# Clear the previously set RCM
def clearRCM(insim):
    insim.sendm('/rcc_all')

# Add some new instance variables to the NPL packet, to store the players
# starting position and whether they have received a penalty, then add
# to the players dict.
def playerJoined(insim, npl):
    npl.StartPosition = 0
    npl.HasPenalty = False
    players[npl.PLID] = npl

# Delete from players dict.
def playerLeft(insim, pll):
    del players[pll.PLID]
    
# Loop through each player and update their start position.
def raceReordered(insim, reo):
    position = 1
    for plid in reo.PLID:
        players[plid].StartPosition = position
        position += 1
    
# Check that the race laps have been set (otherwise practice or qual session),
# then start the formation lap and send a RCM to all players.
def raceStarted(insim, rst):
    global formation    
    if rst.RaceLaps > 0:
        formation = True
        sendRCM(insim, '^1FORMATION LAP!')

# Check if the formation laps have been completed, if so stop the formation lap
# and send the 'GO' RCM.
def lapCompleted(insim, lap):
    global formation
    if lap.LapsDone >= FORMATION_LAPS:
        formation = False
        sendRCM(insim, '^2GO GO GO!!!')
            
# Loop through each car in the race, check its speed and if it has changed
# from its starting position.
def carUpdate(insim, mci):
    if not formation: return # Formation lap over.
    for car in mci.CompCars:
        if car.PLID not in players: continue # Sometimes get MCI before NPL 
        npl = players[car.PLID]
        if pyinsim.kph(car.Speed) > SPEED_LIMIT:
            insim.sendm('/spec %s' % npl.PName)
            insim.sendm('%s ^7spectated for speeding!' % npl.PName)
        elif car.Position > 0 and car.Position < npl.StartPosition:
            if not npl.HasPenalty:
                insim.sendm('/p_30 %s' % npl.PName)
                insim.sendm('%s ^7+30 second penalty for overtaking!' % npl.PName)
                npl.HasPenalty = True
        
# Create InSim object and bind events.
insim = pyinsim.InSim()
insim.bind(pyinsim.ISP_NPL, playerJoined)
insim.bind(pyinsim.ISP_PLL, playerLeft)
insim.bind(pyinsim.ISP_RST, raceStarted)
insim.bind(pyinsim.ISP_REO, raceReordered)
insim.bind(pyinsim.ISP_LAP, lapCompleted)
insim.bind(pyinsim.ISP_MCI, carUpdate)

try:
    # Initialise InSim to send MCI packets on port 30000 every 500 milliseconds.
    insim.init('localhost', 29999, UDPPort=30000, Interval=500, Flags=pyinsim.ISF_MCI, IName='^3Example 7')
    
    # Request all NPL packets to be sent.
    insim.send(pyinsim.ISP_TINY, ReqI=1, SubT=pyinsim.TINY_NPL)
    
except pyinsim.VersionError:
    print 'Invalid InSim version'
except pyinsim.Error as err:
    print 'Error:', err