"""Example 15: Cruise

A simple cruise server. As you drive you earn cash, which can be spent on buying 
and selling cars. User data is stored as pickled Python objects in the 
data\cruise folder, and is retrieved and stored when players join and leave the
host. There are several chat commands such as !prices, !buy, !sell, !cars and
!help, plus an onscreen display which shows the players current cash and their
total kilometers driven.

"""

# Make sure correct version of pyinsim is imported.
VERSION = '1.6.3'
try:
    import pyinsim
    if not pyinsim.version(VERSION):
        raise ImportError
except ImportError:
    print 'You must install pyinsim %s or better' % VERSION
    import sys
    sys.exit(1)
    
# Dependencies.
import cPickle
import os

# Constants.
HOST = 'localhost'
PORT = 29999
PREFIX = '!'
UDPPORT = 30000
INTERVAL = 500 # Milliseconds
PROG_NAME = '^7pyCruise'
CASH_MULTIPLIER = 2 # Dollars per second
STARTING_CARS = ['UF1',]
STARTING_CASH = 1000 # Dollars
HEARTBEAT_INTERVAL = 1 # Seconds
USER_DIR = 'data\\cruise'
HOST_ID = 0
SPEED_DEADZONE = 10
CAR_PRICES = {'XFG': 4500, 'XRG': 6000, 'FBM': 150000, 'XRT': 14000, 'RB4': 12000, 
              'FXO': 12000, 'LX4': 15000, 'LX6': 25000, 'MRT': 30000,'UF1': 3000,
              'RAC': 30000, 'FZ5': 38000, 'XFR': 50000, 'UFR': 45000, 'FOX': 150000,
              'FO8': 165000, 'BF1': 350000, 'FXR': 120000, 'XRR': 120000, 'FZR': 130000}

# Global variables.
connections = {}
players = {}
insim = pyinsim.InSim()

# Class to store user info.
class UserVars:
    def __init__(self):
        self.last_pos = ()
        self.dist = 0
        self.cash = STARTING_CASH
        self.cars = STARTING_CARS
        self.on_track = False
        
# Draw on-screen display.
def draw_osd(ncn):
    insim.send(pyinsim.ISP_BTN, ReqI=1, UCID=ncn.UCID, ClickID=1, 
               BStyle=pyinsim.ISB_DARK, T=4, L=85, W=30, H=6,
               Text='Cash: $%d | Distance: %.2f Km' % (ncn.vars.cash, ncn.vars.dist)) 
        
# Called every second to update cash and OSD.
def heartbeat(insim):
    for ncn in connections.values():
        if ncn.UCID != HOST_ID:
            if ncn.vars.on_track:
                ncn.vars.cash += CASH_MULTIPLIER
            draw_osd(ncn)
    insim.timer(heartbeat, HEARTBEAT_INTERVAL)
        
# Load user info from file.
def load_user_vars(uname):
    path = os.path.join(USER_DIR, uname)
    if os.path.exists(path):
        try:
            with open(path, 'r') as f:
                return cPickle.load(f)
        except IOError as err:
            print 'Load Error:', err
    return UserVars() # Default
    
# Save user info to file.
def save_user_vars(uname, vars):
    path = os.path.join(USER_DIR, uname)
    try:
        with open(path, 'w') as f:
            cPickle.dump(vars, f)
    except IOError as err:
        print 'Save Error:', err
        
# Request all connections and players to be sent.
def req_conns():
    insim.send(pyinsim.ISP_TINY, ReqI=1, SubT=pyinsim.TINY_NCN)
    insim.send(pyinsim.ISP_TINY, ReqI=1, SubT=pyinsim.TINY_NPL)
  
# New connection joined host      
def new_conn(insim, ncn):
    if ncn.UCID != HOST_ID:
        ncn.vars = load_user_vars(ncn.UName)   
    connections[ncn.UCID] = ncn

# Connection left host.
def conn_left(insim, cnl):
    if cnl.UCID != HOST_ID:
        ncn = connections[cnl.UCID]
        save_user_vars(ncn.UName, ncn.vars)
    del connections[cnl.UCID]

# Player tries to join track.
def new_ply(insim, npl):
    players[npl.PLID] = npl
    ncn = connections[npl.UCID]
    if npl.CName in ncn.vars.cars:
        ncn.vars.on_track = True
    else:
        insim.sendm('/spec %s' % ncn.UName)
        insim.sendm('^3| ^7You do not own the %s' % npl.CName, ncn.UCID)

# Player leaves track.
def ply_left(insim, pll):
    npl = players[pll.PLID]
    ncn = connections[npl.UCID]
    ncn.vars.on_track = False
    del players[pll.PLID]
    
# Print out car prices.
def cmd_prices(ncn, args):
    insim.sendm('^3| ^7Car Prices:', ncn.UCID)
    for car, price in CAR_PRICES.iteritems():
        insim.sendm('^3| ^7%s: $%d' % (car, price), ncn.UCID)

# Buy a new car.
def cmd_buy(ncn, args):
    if args:
        for arg in args:
            car = arg.upper()
            if car in ncn.vars.cars:
                insim.sendm('^3| ^7You already own the %s' % car, ncn.UCID)
            elif car not in CAR_PRICES:
                insim.sendm('^3| ^7The %s does not exist' % car, ncn.UCID)
            elif CAR_PRICES[car] > ncn.vars.cash:
                insim.sendm('^3| ^7You do not have enough cash for the %s' % car, ncn.UCID)
            else:
                ncn.vars.cash -= CAR_PRICES[car]
                ncn.vars.cars.append(car)
                insim.sendm('^3| %s ^7bought the %s' % (ncn.PName, car))
    else:
        insim.sendm('^3| ^7No car selected', ncn.UCID)
        
# Sell an owned car.
def cmd_sell(ncn, args):
    if args:
        for arg in args:
            car = arg.upper()
            if car not in CAR_PRICES:
                insim.sendm('^3| ^7The %s does not exist' % car, ncn.UCID)
            elif car not in ncn.vars.cars:
                insim.sendm('^3| ^7You do not own the %s' % car, ncn.UCID)
            else:
                ncn.vars.cash += CAR_PRICES[car]
                ncn.vars.cars.remove(car)
                insim.sendm('^3| %s ^7sold the %s' % (ncn.PName, car))
    else:
        insim.sendm('^3| ^7No car selected', ncn.UCID)

# Print list of cars owned.
def cmd_cars(ncn, args):
    insim.sendm('^3| ^7Currently Owned Cars:', ncn.UCID)
    for car in ncn.vars.cars:
        insim.sendm('^3| ^7%s: $%d' %(car, CAR_PRICES[car]), ncn.UCID)
        
# Print usage info.
def cmd_help(ncn, args):
    insim.sendm('^3| ^7Usage Info:', ncn.UCID)
    insim.sendm('^3| !prices ^7- View car prices', ncn.UCID)
    insim.sendm('^3| !buy ^7- Buy a new car', ncn.UCID)
    insim.sendm('^3| !sell ^7- Sell an owned car', ncn.UCID)
    insim.sendm('^3| !cars ^7- See what cars you currently own', ncn.UCID)
    
CMD_LOOKUP = {'prices': cmd_prices, 'buy': cmd_buy, 'sell': cmd_sell, 
              'cars': cmd_cars, 'help': cmd_help,}
    
# Handle command message from LFS.
def message_out(insim, mso):
    cmd = pyinsim.parseCmd(mso)
    if cmd:
        ncn = connections[mso.UCID]
        cmd0 = cmd[0].lower()
        if cmd0 in CMD_LOOKUP:
            CMD_LOOKUP[cmd0](ncn, cmd[1])
        else:
            insim.sendm('^3| ^7Unknown command', ncn.UCID)
            
# Calculate distance travelled.
def add_distance(ncn, car):
    curr_pos = (car.X, car.Y, car.Z)    
    if ncn.vars.last_pos:
        dist = pyinsim.distance(ncn.vars.last_pos, curr_pos)
        ncn.vars.dist += pyinsim.metres(dist) / 1000 # Km
    ncn.vars.last_pos = curr_pos  
        
# Player MCI updates.    
def car_info(insim, mci):
    for car in mci.CompCars:
        if car.Speed < SPEED_DEADZONE: continue
        npl = players[car.PLID]
        ncn = connections[npl.UCID]
        add_distance(ncn, car)
        
# Remind to set /cruise flag.
def race_start(insim, rst):
    if not rst.Flags & pyinsim.HOSTF_CRUISE:
        insim.sendm('^3| ^1WARNING: NOT IN CRUISE MODE!')
        insim.sendm('^3| ^1WARNING: NOT IN CRUISE MODE!')
    
# Save user vars if connection is lost.
def closed(insim, reason):
    for ncn in connections.values():
        if ncn.UCID != HOST_ID:
            save_user_vars(ncn.UName, ncn)

if __name__ == '__main__':    
    # Bind event-handlers.
    insim.bind(pyinsim.ISP_NCN, new_conn)
    insim.bind(pyinsim.ISP_CNL, conn_left)
    insim.bind(pyinsim.ISP_NPL, new_ply)
    insim.bind(pyinsim.ISP_PLL, ply_left)
    insim.bind(pyinsim.ISP_MSO, message_out)
    insim.bind(pyinsim.ISP_MCI, car_info)
    insim.bind(pyinsim.ISP_RST, race_start)
    insim.bind(pyinsim.EVT_CLOSED, closed)
    
    try:
        # Initialise InSim.
        insim.init(HOST, PORT, IName=PROG_NAME, Prefix=PREFIX, UDPPort=UDPPORT, 
                   Flags=pyinsim.ISF_MCI, Interval=INTERVAL)
        
        # Request players/connections.
        req_conns()
        
        # Start heartbeat timer.
        insim.timer(heartbeat, HEARTBEAT_INTERVAL)
    except pyinsim.Error as err:
        print 'InSim Error:', err
        
        