﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Net;
using System.Net.Sockets;

namespace InSimTutorial
{
    public class Program
    {
        private const int BufferSize = 1024;
        private const int InSimVersion = 4; 

        private static Socket socket = null;
        private static RecvBuffer buffer = new RecvBuffer();
        private static byte[] bufferdata = new byte[BufferSize]; // Receive buffer.

        private static void Main(string[] args)
        {
            // Create new Tcp socket.
            socket = new Socket(
                AddressFamily.InterNetwork,
                SocketType.Stream,
                ProtocolType.Tcp);

            try
            {
                // Connect to InSim.
                socket.Connect("127.0.0.1", 29999);

                // Initailise InSim.
                IS_ISI isi = new IS_ISI()
                {
                    ReqI = 1, // Request version
                    UDPPort = 0,
                    Flags = 0,
                    Interval = 0,
                    Prefix = char.MinValue,
                    Admin = string.Empty,
                    IName = "^3Example",
                };
                Send(isi.GetPacketData());

                // Begin packet receive loop.
                BeginReceive();
            }
            catch (SocketException ex)
            {
                Console.WriteLine("Error: {0}", ex.Message);
            }

            // Keep program open.
            Console.ReadLine();
        }

        private static void Send(byte[] data)
        {
            // Send data to LFS.
            int sent = 0;
            while (sent < data.Length)
            {
                sent = socket.Send(
                    data,
                    sent,
                    data.Length - sent,
                    SocketFlags.None);
            }
        }

        private static void BeginReceive()
        {
            // Populate the bufferdata array with data from the socket.
            socket.BeginReceive(
                bufferdata, 0,
                BufferSize,
                SocketFlags.None,
                ReceiveCallback,
                null);
        }

        private static void ReceiveCallback(IAsyncResult r)
        {
            try
            {
                // End the receive call
                int count = socket.EndReceive(r);
                if (count > 0)
                {
                    // Append data to the buffer.
                    buffer.Append(bufferdata, count);

                    // Loop through completed packets in the buffer.
                    foreach (byte[] pdata in buffer.GetPackets())
                    {
                        PacketReceived(pdata);
                    }

                    // Begin next receive call.
                    BeginReceive();
                }
                else
                {
                    // If no bytes are received the socket has closed.
                    Console.WriteLine("Lost connection with LFS");
                }
            }
            catch (SocketException ex)
            {
                Console.WriteLine("Error: {0}", ex.Message);
            }
        }

        private static void PacketReceived(byte[] pdata)
        {
            // The second byte of the packet is the type.
            PacketTypes ptype = (PacketTypes)pdata[1];
            switch (ptype)
            {
                case PacketTypes.ISP_VER:
                    CheckVersion(new IS_VER(pdata));
                    break;
                case PacketTypes.ISP_TINY:
                    KeepAlive(new IS_TINY(pdata));
                    break;
                case PacketTypes.ISP_MSO:
                    MessageOut(new IS_MSO(pdata));
                    break;
            }
        }

        private static void CheckVersion(IS_VER ver)
        {
            // Make sure LFS has correct InSim version.
            if (ver.InSimVer != InSimVersion)
            {
                Console.WriteLine("Invalid InSim version!");
                Environment.Exit(1);
            }
        }

        private static void KeepAlive(IS_TINY tiny)
        {
            // Respond to the 'keep-alive' pulse.
            if ((tiny.ReqI == 0) && (tiny.SubT == TinyTypes.TINY_NONE))
            {
                Send(tiny.GetPacketData()); // Keep alive.
            }
        }

        private static void MessageOut(IS_MSO mso)
        {
            // Print out message.
            Console.WriteLine("Message Out: {0}", mso.Msg);
        }
    }
}
